package io

import (
	"io"

	"github.com/hjson/hjson-go"
)

func init() {
	registerInputHandler("hjson", func() InputHandler { return &HJSONInputHandler{} })
}

// Declare conformance with InputHandler interface.
var _ InputHandler = &HJSONInputHandler{}

// HJSONInputHandler handles parsing HJSON data.
type HJSONInputHandler struct{}

// Name implements InputHandler.Name().
func (h *HJSONInputHandler) Name() string {
	return "hjson"
}

// Parse implements InputHandler.Parse().
func (h *HJSONInputHandler) Parse(reader io.Reader) (interface{}, error) {
	inputBytes, err := io.ReadAll(reader)
	if err != nil {
		return nil, err
	}

	if len(inputBytes) == 0 {
		inputBytes = []byte("{}")
	}

	var parsed interface{}
	err = hjson.Unmarshal(inputBytes, &parsed)
	if err != nil {
		return nil, err
	}

	return parsed, nil
}

// SetOption implements InputHandler.SetOption().
func (h *HJSONInputHandler) SetOption(name string, value string) error {
	// The HJSON handler does not have any options.
	return nil
}
