# `rq` Gotchas

*[back to index](./README.md)*

## `argc` / `argv` handling

When `rq` is invoked, `argc` is at least 2 (at least 2 CLI arguments are provided), and the first argument is not a valid subcommand but is a file path that can be STAT-ed, then the subcommand is implied to be `script`, rather than `query` as it would otherwise be. This behavior is needed so as to ensure `rq` can be invoked as a shebang for script files, since the preferred way of invoking it is with `/usr/bin/env`, meaning an argument may not be able to be provided to the `rq` command on some platforms.

This does mean that if a subcommand is not provided explicitly, `rq` may attempt to call `os.Stat()` on its first argument.

This behavior is defined in [`cmd/rq/main.go`](../cmd/rq/main.go).

## Advice

`rq` includes a feature to provide advice based on the query string and other CLI arguments. This advice may be printed as an error, which will typically be prefixed with `TIP: `. This feature is intended to detect common mistakes that would cause `rq` to fail later on in its execution, typically with an unhelpful error.

This behavior is defined in [`cmd/rq/helpers.go`](../cmd/rq/helpers.go), in the `helpfulAdvice()` function.

## `NO_COLOR`

`rq` is compliant with the [`NO_COLOR` specification](http://no-color.org/). This provides a standard way for users to disable colorization of output using an environment variable. When this variable is defined, `rq` will suppress colorized output. This takes precedence over all CLI flags, including `--force-color`.

This behavior is defined in [`cmd/rq/helpers.go`](../cmd/rq/helpers.go), in the `defaultOutputOptions()` function.

## TTY Detection

In some situations, `rq` may change its behavior depending on whether it appears to be running in an interactive terminal. `rq` detects interactive terminals using the `term.IsTerminal()` function from `golang.org/x/term`. This can be disabled with the `--no-tty-check` flag. The following behaviors are influenced:

If standard input is a terminal, then the input document is assumed to be the empty object `{}`. This is useful when testing out one-off queries that don't need input data. For example:

```plain
$ rq '{"hello": "world"}'
{
	"hello": "world"
}
$ rq --no-tty-check '{"hello": "world"}'
rq: error: unexpected end of JSON input
```

NOTE: In the above example, I had to manually press `^D` (ctrl+D) to signal EOF on the input.

If standard output is a terminal, then output is automatically colorized if possible, unless a flag or `NO_COLOR` have been used.

This behavior is defined in [`cmd/rq/commands.go`](../cmd/rq/commands.go).

## Default Query

When `rq` is running in `script` mode, the default query is `data.<script package>`, which can be overridden with the `# rq: query <your query here>` directive.

When `rq` is running in `query` mode, the default query is `input`, which can be overridden with the `query` positional argument. Additionally, if the query is exactly `input`, then `rq` will skip invoking the Rego interpreter, and instead directly assign the parsed input document to the output. This is semantically equivalent, but offers a performance boost when using `rq` to convert between different formats without changing the data.

This behavior is defined in [`cmd/rq/commands.go`](../cmd/rq/commands.go).

## Marshaling Non-Map Values to XML

When `rq` is asked to produce XML output from any data other than a map with string keys (e.g. a JSON object), it will be nested under a key `object`.

```plain
$ rq -o xml '"hello"'
<doc>
        <object>hello</object>
</doc>
```

If the given data is an array, then the elements will have generated node IDs `list-element<i>` for list element `<i>`.

```plain
$ rq -o xml '[1,2,3]'
<doc>
        <list-element0>1</list-element0>
        <list-element1>2</list-element1>
        <list-element2>3</list-element2>
</doc>
$ rq -o xml '[{"foo": "bar"}, 123]'
<doc>
        <list-element0>
                <foo>bar</foo>
        </list-element0>
        <list-element1>123</list-element1>
</doc>
```

## DataSpecs With Empty File Paths

In some contexts, such as in the `rq.run()`, `rq.encode()`, and `rq.decode()` builtins, [DataSpecs](./dataspec.md) are used to specify formats and options, but not file paths. In these situations, placing non-empty values in the file path of the DataSpec will cause a fatal error. This is done to avoid situations where values intended as options end up in the file path part of the DataSpec.

To avoid ambiguity between options and file paths, the second part of 2-part DataSpecs is always assumed to be a file path. For example, the DataSpec `csv:csv.headers=true` is parsed as:

```json
{
	"format": "csv",
	"FilePath": "csv.headers=true",
	"Options": {}
}
```

However the DataSpec `csv:csv.headers=true:` is parsed as:

```json
{
	"format": "csv",
	"FilePath": "",
	"Options": {
		"csv.headers": "true"
	}
}
```

We can see this in practice with the following example:

```plain
$ rq 'rq.decode("foo\nbar\nbaz\n", "csv:csv.headers=true")'
rq: error: rq.decode options error: DataSpec has a non-empty file path 'csv.headers=true'
$ rq 'rq.decode("foo\nbar\nbaz\n", "csv:csv.headers=true:")'
[
        {
                "foo": "bar"
        },
        {
                "foo": "baz"
        }
]
```

If a non-empty file path was not an error condition, the first `rq` command would have run without error, parsed the input to `rq.decode()` as a file path (and subsequently ignored it, since `rq.decode()` does not accept a file path), and consequentially ignored the user's intended `csv.headers=true` option.

This behavior is defined in [`./builtins/builtins.go`](../builtins/builtins.go).

## Bundle Conflict Resolution

Starting with version 0.0.6, `rq` supports loading locally stored [OPA bundles](https://www.openpolicyagent.org/docs/latest/management-bundles/), via the `--bundle` flag, or via the `bundle-paths` directive for scripts. The semantics of `rq`'s bundle loading are slightly different than OPA's, especially when multiple bundles are used.

The most notable difference is that `rq` does does not throw errors when bundle roots overlap, nor even when files within the loaded bundles overlap. Instead, the bundles are silently merged, and any overlapping paths are simply overwritten, with later `--bundle` instances taking precedence over newer ones. In script mode, the script file takes precedence over any loaded bundles. Data loaded using the `--data` flag always has a higher precedence than data loaded from bundles.

This behavior is defined in [`cmd/rq/commands.go`](../cmd/rq/commands.go). The bundle merging logic is defined in [`cmd/rq/helpers.go`](../cmd/rq/helpers.go), in the `mergeBundles()` function.

Starting with version 0.0.11, `rq` supports both Rego v0 and v1, with v1 being the default. When merging bundles, the resulting bundle manifest's Rego version will be whatever rq is configured to use, either via `--v0-compatible` or via the `RQ_REGO_V0` environment variable. The declared Rego version for the input bundle's manifests is ignored. Using bundle merging with bundles of mismatched Rego versions, or with bundles which have different versions from the one `rq` is configured to use is inadvisable and may result in non-functioning bundles; `rq`'s behavior in such situations is undefined.

## Strict Mode & Dealing With Malformed Input Files

In some situations, you may find yourself needing to deal with input files that may or may not be malformed. For example, imagine using `rq.tree()` to spider a directory for JSON files, loading them with `rq.read()`, and searching them for some information. By default, `rq` will crash with an error if any input it tries to load is invalid, which is referred to as "strict mode". For some such use cases, it may be desirable to instead have the loading process fail silently. For this reason, starting with version 0.0.6, it is possible to explicitly disable this behavior setting `strict` option to `false` in the input DataSpec.

When `strict=false` is used, errors relating to parsing the input are silenced, and if one is encountered the resulting object is `null`.

Consider this (contrived) example: 

```plain
$ rq -I 'csv:csv.headers=true:./sample_data/books.csv' 'input[2]'
{
        "first author": "Alan A. A. Donovan",
        "isbn": "978-0134190440",
        "title": "The Go Programming Language",
        "year": 2015
}
$ rq -I 'csv:csv.headers=true:./sample_data/books.json' 'input[2]'
rq: error: parse error on line 3, column 9: bare " in non-quoted-field
```

Needless to say, a JSON file is not a valid CSV file, so the CSV input handler throws an error and exits. However, we can override this behavior with `strict=false` like so:

```plain
$ rq -I 'csv:csv.headers=true;strict=false:./sample_data/books.json' 'input[2]'
[]
$ rq -I 'csv:csv.headers=true;strict=false:./sample_data/books.json' 'input'
null
```

This functionality also works with `rq.decode()` and `rq.read()` as well:

```plain
$ rq 'rq.read("csv:csv.headers=true:./sample_data/books.json")'
rq: error: parse error on line 3, column 9: bare " in non-quoted-field
$ rq 'rq.read("csv:csv.headers=true;strict=false:./sample_data/books.json")'
null
$ ./rq 'rq.decode("[1,2,3", "json::")'
rq: error: ReadArrayCB: expect ] in the end, but found , error found in #6 byte of ...|[1,2,3|..., bigger context ...|[1,2,3|...
$ ./rq 'rq.decode("[1,2,3", "json:strict=false:")'
null
```

This behavior is defined in [`io/input.go`](../io/input.go).

## Input Format Inference from File Extensions

When a DataSpec is provided without a format, e.g. just a bare path like `something.json` or `data/records.csv`, the input handler to use is selected based on the file extension. The extension is lowercased, and some aliases are applied, such as `yml` being treated as `yaml` or `jsonl` being treated as `ndjson`. This is then checked against the list of available input handlers.

This behavior is defined by `ResolveDefaults()` in [`io/dataspec.go`](../io/dataspec.go).

## No Output Colors on Windows

When Chroma syntax highlighting of outputs is enabled in native Windows builds, it seems to produce broken output in the Windows terminal. I do not have the expertise to fix this, as I do not use Windows or know much about it. For this reason, syntax highlighted output is disabled on Windows. This behavior can be disabled by defining the `RQ_SUPPRESS_WINDOWS_NO_COLOR` environment variable is defined.

An example of this behavior is shown below:

```plain
PS C:\Users\user\f\src\rq> remove-item env:RQ_SUPPRESS_WINDOWS_NO_COLOR
PS C:\Users\user\f\src\rq> ./rq.exe '1+1'
2
PS C:\Users\user\f\src\rq> $env:RQ_SUPPRESS_WINDOWS_NO_COLOR = 1
PS C:\Users\user\f\src\rq> ./rq.exe '1+1'
←[36m2←[0m
```

This is tracked as a bug by [#10](https://todo.sr.ht/~charles/rq/10).

This behavior is defined in [`cmd/rq/helpers.go`](../cmd/rq/helpers.go) and [`cmd/rq/commands.go`](../cmd/rq/commands.go).
