
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


#include <stdarg.h>
#include <string.h>

#include "runtime.h"


/* Error conditions.  The tricky part here is printing a message when
 * it is the I/O subsystem that is severely wounded.  Our goal is to
 * try and print something making the fewest assumptions possible,
 * then try to clean up before actually exiting.
 *
 * The following exit conditions are defined:
 * 0    Normal program exit.
 * 1    Terminated because of operating system error.
 * 2    Error in the runtime library
 * 3    Internal error in runtime library
 * 4    Error during error processing (very bad)
 *
 * Other error returns are reserved for the STOP statement with a numeric code.
 */

/* locus variables.  These are optionally set by a caller before a
 * library subroutine is called.  They are always cleared on exit so
 * that files that report loci and those that do not can be linked
 * together without reporting an erroneous position. */

char *filename;
unsigned G95_DINT line;
frame *frame_base;

char *error_filename;






static void hex_long(unsigned long n, char *buffer) {
int s, digit;
char *p;

    if (n == 0) {
	buffer[0] = '0';
	buffer[1] = '\0';
	return;
    }

    for(s=8*sizeof(long)-4; s>=0; s-=4) {
	digit = (n >> s) & 0xF;

	if (digit > 9)
	    digit += 'a' - 10;
	else
	    digit += '0';

	*buffer++ = digit;
    }

    *buffer = '\0';
}


/* st_printf()-- simple printf() function for streams that handles the
 * formats %d, %s and %c.  This function handles printing of error
 * messages that originate within the library itself, not from a user
 * program. */

int st_printf(char *format, ...) {
char *p, *q, long_buf[50];
int count, total;
va_list arg;
stream *s;

    total = 0;
    s = init_error_stream();
    va_start(arg, format);

    for(;;) { 
	count = 0;

	while(format[count] != '%' && format[count] != '\0')
	    count++;

	if (count != 0) {
	    p = salloc_w(s, count, 0);
	    memmove(p, format, count);
	    sfree(s);
	}

	total += count;
	format += count;
	if (*format++ == '\0')
	    break;

	switch(*format) {
	case 'c':
	    count = 1;

	    p = salloc_w(s, count, 0);
	    *p = (char) va_arg(arg, int);

	    sfree(s);
	    break;

	case 'd':
	    q = int_to_a(va_arg(arg, G95_DINT));
	    count = strlen(q);

	    p = salloc_w(s, count, 0);
	    memmove(p, q, count);
	    sfree(s);
	    break;

	case 'L':
	    q = long_to_a(va_arg(arg, G95_AINT));
	    count = strlen(q);

	    p = salloc_w(s, count, 0);
	    memmove(p, q, count);
	    sfree(s);
	    break;

	case 'l':
	    q = mint_to_a(va_arg(arg, G95_MINT));
	    count = strlen(q);

	    p = salloc_w(s, count, 0);
	    memmove(p, q, count);
	    sfree(s);
	    break;

	case 'p':
	    hex_long(va_arg(arg, unsigned long), long_buf);
	    count = strlen(long_buf);

	    p = salloc_w(s, count, 0);
	    memmove(p, long_buf, count);
	    sfree(s);
	    break;

	case 'x':
	    q = hex_int(va_arg(arg, unsigned G95_DINT));
	    count = strlen(q);

	    p = salloc_w(s, count, 0);
	    memmove(p, q, count);
	    sfree(s);
	    break;

	case 's':
	    q = va_arg(arg, char *);
	    count = strlen(q);

	    p = salloc_w(s, count, 0);
	    memmove(p, q, count);
	    sfree(s);
	    break;

	case '\0':
	    return total;

	default:
	    count = 2;
	    p = salloc_w(s, count, 0);
	    p[0] = format[-1];
	    p[1] = format[0];
	    sfree(s);
	    break;
	}

	total += count;
	format++;
    }

    va_end(arg);
    return total;
}



/* st_sprintf()-- Simple sprintf() for formatting memory buffers. */

void st_sprintf(char *buffer, char *format, ...) {
va_list arg;
char c, *p;
int count;

    va_start(arg, format);

    for(;;) {
	c = *format++;
	if (c != '%') {
	    *buffer++ = c;
	    if (c == '\0')
		break;

	    continue;
	}

	c = *format++;
	switch(c) {
	case 'c':
	    *buffer++ = (char) va_arg(arg, int);
	    break;

	case 'd':
	    p = int_to_a(va_arg(arg, G95_DINT));
	    count = strlen(p);

	    memcpy(buffer, p, count);
	    buffer += count;
	    break;

	case 'x':
	    p = hex_int(va_arg(arg, unsigned G95_DINT));
	    count = strlen(p);

	    memcpy(buffer, p, count);
	    buffer += count;
	    break;

	case 'L':
	    p = long_to_a(va_arg(arg, G95_AINT));
	    count = strlen(p);

	    memcpy(buffer, p, count);
	    buffer += count;
	    break;

	case 's':
	    p = va_arg(arg, char *);
	    count = strlen(p);

	    memcpy(buffer, p, count);
	    buffer += count;
	    break;

	default:
	    *buffer++ = c;
	}
    }

    va_end(arg);
}



/* show_locus()-- Print a line number and filename describing where
 * something went wrong */

void show_locus() {
char *p, buffer[100];
frame *f;
int n;

    if (!options.locus)
	return;

    if (current_unit != NULL)
	strcpy(buffer, mint_to_a(current_unit->unit_number));

    if (filename != NULL && line != 0) {
	if (current_unit != NULL && current_unit->unit_number != -1 &&
	    current_unit->file_len > 0) {
	    st_printf("At line %d of file %s (Unit %s \"",
		      line, filename, buffer);

	    n = current_unit->file_len;
	    p = current_unit->file;

	    while(n-- > 0)
		st_printf("%c", *p++);

	    st_printf("\")\n");

	} else if (current_unit != NULL && current_unit->unit_number != -1)
	    st_printf("At line %d of file %s (Unit %s)\n",
		      line, filename, buffer);

	else {
	    st_printf("At line %d of file %s", line, filename);

	    if (error_filename != NULL) {
		st_printf(" file: \"");

		p = error_filename;
		while(*p != '\0')
		    st_printf("%c", *p++);

		st_printf("\"");
	    }

	    st_printf("\n");
	}
    }

    if (frame_base == NULL) {
	st_printf("Traceback: not available, compile with -ftrace=frame or "
		  "-ftrace=full\n");
	return;
    }

    st_printf("Traceback: (Innermost first)\n");
    for(f=frame_base; f; f=f->next) {
	p = f->f_filename;
	if (p == NULL)
	    continue;

	st_printf("   Called from line %d of file %s\n", f->f_line, p);
    }
}



/* recursion_check()-- It's possible for additional errors to occur
 * during fatal error processing.  We detect this condition here and
 * exit with code 4 immediately. */

#define MAGIC 0x20DE8101

static void recursion_check(void) {
static int magic=0;

    if (magic == MAGIC)
	sys_exit(1, 4);  /* Don't even try to print something at this point */

    magic = MAGIC;
}



/* os_error()-- Operating system error.  We get a message from the
 * operating system, show it and leave.  Some operating system errors
 * are caught and processed by the library.  If not, we come here. */

void os_error(char *message) {

    recursion_check();
    close_units();

    show_locus();
    st_printf("Operating system error: %s\n%s\n", get_oserror(), message);

    sys_exit(1, 1);
}



/* void runtime_error()-- These are errors associated with an
 * invalid fortran program. */

void runtime_error(char *message) {

    recursion_check();
    show_locus();

    close_units();
    st_printf("Fortran runtime error: %s\n", message);

    sys_exit(1, 2);
}



/* internal_error()-- These are this-can't-happen errors
 * that indicate something deeply wrong. */

void internal_error(char *message) {

    recursion_check();

    show_locus();
    st_printf("Internal Error: %s\n", message);
    sys_exit(1, 3);
}



/* translate_error()-- Given an integer error code, return a string
 * describing the error. */

char *translate_error(int code) {
char *p;

    switch(code) {
    case ERROR_EOR:
	p = "End of record";
	break;

    case ERROR_END:
	p = "End of file";
	break;

    case ERROR_OK:
	p = "Successful return";
	break;

    case ERROR_OS:
	p = "Operating system error";
	break;

    case ERROR_BAD_OPTION:
	p = "Bad statement option";
	break;

    case ERROR_MISSING_OPTION:
	p = "Missing statement option";
	break;

    case ERROR_OPTION_CONFLICT:
	p = "Conflicting statement options";
	break;

    case ERROR_ALREADY_OPEN:
	p = "File already opened in another unit";
	break;

    case ERROR_BADUNIT:
	p = "Unattached unit";
	break;

    case ERROR_FORMAT:
	p = "FORMAT error";
	break;

    case ERROR_BAD_ACTION:
	p = "Incorrect ACTION specified";
	break;

    case ERROR_ENDFILE:
	p = "Read past ENDFILE record";
	break;

    case ERROR_READ_VALUE:
	p = "Bad value during read";
	break;

    case ERROR_READ_OVERFLOW:
	p = "Numeric overflow on read";
	break;

    case ERROR_NOMEM:
	p = "Out of memory";
	break;

    case ERROR_ALREADY_ALLOC:
	p = "Array already allocated";
	break;

    case ERROR_BADFREE:
	p = "Deallocated a bad pointer";
	break;

    case ERROR_CORRUPT_DIRECT:
	p = "Bad record number in direct-access file";
	break;

    case ERROR_CORRUPT_SEQUENTIAL:
	p = "Corrupt record in unformatted sequential-access file";
	break;

    case ERROR_READ_RECL:
	p = "Reading more data than the record size (RECL)";
	break;

    case ERROR_WRITE_RECL:
	p = "Writing more data than the record size (RECL)";
	break;

    default:
	p = "Unknown error code";
	break;
    }

    return p;
}



/* generate_error()-- Come here when an error happens.  This
 * subroutine is called if it is possible to continue on after the
 * error.  If an IOSTAT variable exists, we set it.  If the IOSTAT or
 * ERR label is present, we return, otherwise we terminate the program
 * after printing a message.  The error code is always required but
 * the message parameter can be NULL, in which case a string
 * describing the most recent operating system error is used. */

void generate_error(int family, char *message) {

    if (ioparm == NULL)
	goto terminate;

    if (ioparm->iostat != NULL)
	set_integer((family == ERROR_OS) ? get_oserrno() : family,
		    ioparm->iostat, ioparm->iostat_kind);

    switch(family) {
    case ERROR_EOR:
	ioparm->library_rc = LIBRARY_EOR;
	if (ioparm->eor != 0)
	    return;

	break;

    case ERROR_END:
	ioparm->library_rc = LIBRARY_END;
	if (ioparm->end != 0)
	    return;

	break;

    default:
	ioparm->library_rc = LIBRARY_ERROR;
	break;
    }

    if (ioparm->err != 0 || ioparm->iostat != NULL)
	return;

    /* Terminate the program */

terminate:
    if (message == NULL)
	message = (family == ERROR_OS)
	    ? get_oserror()
	    : translate_error(family);

    runtime_error(message);
}



/* bad_dim()-- Report a bad DIM parameter. */

void bad_dim(void) {

    runtime_error("Bad DIM parameter during array contraction");
}
