
/* Copyright (C) 2003-2008, Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


/* power.c -- compute an integer power.  We use a simple
 * shift-multiply-square method.  This isn't always optimal but it is
 * reasonable.  See Knuth's "The Art of Computer Programming", Volume 2,
 * Section 4.6.3.  */


#if defined(SUBROUTINE) || defined(SUBROUTINE4)

#ifdef INTEGER

TYPE SUBROUTINE(TYPE base, ETYPE e) {
unsigned ETYPE exponent;
int negative;
TYPE result;

    exponent = e; 

    if ((base == 0 && exponent == 0) || (base == 1 || exponent == 0))
	return 1;

    if (base == -1)
	return (exponent & 1) ? -1 : 1;

    if (e < 0)
	return 0;

    negative = (base < 0 && ((exponent & 1) == 1));
    if (base < 0)
	base = -base;

    /* At this point, base > 0 and exponent > 0 */

    result = 1;
    for(;;) {
	if ((exponent & 1) == 1)
	    result *= base;

	exponent >>= 1;
	if (exponent == 0)
	    break;

	base = base*base;
    }

    if (negative)
	result = -result;

    return result;
}


#endif

#ifdef REAL

TYPE SUBROUTINE(TYPE base, ETYPE exponent) {
unsigned ETYPE u_exponent;
TYPE result;

    if ((base == 0.0 && exponent == 0) ||
	base == 1.0 || exponent == 0)
	return 1.0;

    if (exponent < 0) {
	exponent = -exponent;
	base = 1.0 / base;
    }

    result = 1.0;
    u_exponent = exponent;

    for(;;) {
	if ((u_exponent & 1) == 1)
	    result *= base;

	u_exponent >>= 1;
	if (u_exponent == 0)
	    break;

	base = base*base;
    }

    return result;
}


#endif

#ifdef COMPLEX

void SUBROUTINE(TYPE *result, TYPE *base, ETYPE exponent) {
unsigned ETYPE u_exponent;
TYPE z, b, t;
double det;

    if ((base->r == 0.0 && base->c == 0.0 && exponent == 0) ||
	((base->r == 1.0 && base->c == 0.0) || exponent == 0)) {
	result->r = 1.0;
	result->c = 0.0;
	return;
    }

    z.r = 1.0;
    z.c = 0.0;

    b.r = base->r;
    b.c = base->c;

    if (exponent < 0) {
	exponent = -exponent;

	det = b.r*b.r + b.c*b.c;

	b.r =  b.r / det;
	b.c = -b.c / det;
    }

    u_exponent = exponent;

    for(;;) {
	if ((u_exponent & 1) == 1) {  /* z *= b */
	    t.r = z.r*b.r - z.c*b.c;
	    t.c = z.r*b.c + z.c*b.r;

	    z = t;
	}

	u_exponent >>= 1;
	if (u_exponent == 0)
	    break;

	t.r = b.r*b.r - b.c*b.c;
	t.c = 2*b.r*b.c;

	b = t;
    }

    result->r = z.r;
    result->c = z.c;
}

#endif

#undef TYPE
#undef ETYPE
#undef SUBROUTINE

#else

#include "signal.h"
#include <math.h>
#include "runtime.h"

#define INTEGER 1

#define TYPE G95_INT1
#define ETYPE G95_INT1
#define SUBROUTINE prefix(power_i1_i1)
#include "power.c"

#define TYPE G95_INT1
#define ETYPE G95_INT2
#define SUBROUTINE prefix(power_i1_i2)
#include "power.c"

#define TYPE G95_INT1
#define ETYPE G95_INT4
#define SUBROUTINE prefix(power_i1_i4)
#include "power.c"

#define TYPE G95_INT1
#define ETYPE G95_INT8
#define SUBROUTINE prefix(power_i1_i8)
#include "power.c"

#define TYPE G95_INT2
#define ETYPE G95_INT1
#define SUBROUTINE prefix(power_i2_i1)
#include "power.c"

#define TYPE G95_INT2
#define ETYPE G95_INT2
#define SUBROUTINE prefix(power_i2_i2)
#include "power.c"

#define TYPE G95_INT2
#define ETYPE G95_INT4
#define SUBROUTINE prefix(power_i2_i4)
#include "power.c"

#define TYPE G95_INT2
#define ETYPE G95_INT8
#define SUBROUTINE prefix(power_i2_i8)
#include "power.c"

#define TYPE G95_INT4
#define ETYPE G95_INT1
#define SUBROUTINE prefix(power_i4_i1)
#include "power.c"

#define TYPE G95_INT4
#define ETYPE G95_INT2
#define SUBROUTINE prefix(power_i4_i2)
#include "power.c"

#define TYPE G95_INT4
#define ETYPE G95_INT4
#define SUBROUTINE prefix(power_i4_i4)
#include "power.c"

#define TYPE G95_INT4
#define ETYPE G95_INT8
#define SUBROUTINE prefix(power_i4_i8)
#include "power.c"

#define TYPE G95_INT8
#define ETYPE G95_INT1
#define SUBROUTINE prefix(power_i8_i1)
#include "power.c"

#define TYPE G95_INT8
#define ETYPE G95_INT2
#define SUBROUTINE prefix(power_i8_i2)
#include "power.c"

#define TYPE G95_INT8
#define ETYPE G95_INT4
#define SUBROUTINE prefix(power_i8_i4)
#include "power.c"

#define TYPE G95_INT8
#define ETYPE G95_INT8
#define SUBROUTINE prefix(power_i8_i8)
#include "power.c"


#undef INTEGER
#define REAL 1

#define TYPE float
#define ETYPE G95_INT1
#define SUBROUTINE prefix(power_r4_i1)
#include "power.c"

#define TYPE float
#define ETYPE G95_INT2
#define SUBROUTINE prefix(power_r4_i2)
#include "power.c"

#define TYPE float
#define ETYPE G95_INT4
#define SUBROUTINE prefix(power_r4_i4)
#include "power.c"

#define TYPE float
#define ETYPE G95_INT8
#define SUBROUTINE prefix(power_r4_i8)
#include "power.c"

#define TYPE double
#define ETYPE G95_INT1
#define SUBROUTINE prefix(power_r8_i1)
#include "power.c"

#define TYPE double
#define ETYPE G95_INT2
#define SUBROUTINE prefix(power_r8_i2)
#include "power.c"

#define TYPE double
#define ETYPE G95_INT4
#define SUBROUTINE prefix(power_r8_i4)
#include "power.c"

#define TYPE double
#define ETYPE G95_INT8
#define SUBROUTINE prefix(power_r8_i8)
#include "power.c"

#undef REAL
#define COMPLEX 1

#define TYPE z4
#define ETYPE G95_INT1
#define SUBROUTINE prefix(power_z4_i1)
#include "power.c"

#define TYPE z4
#define ETYPE G95_INT2
#define SUBROUTINE prefix(power_z4_i2)
#include "power.c"

#define TYPE z4
#define ETYPE G95_INT4
#define SUBROUTINE prefix(power_z4_i4)
#include "power.c"

#define TYPE z4
#define ETYPE G95_INT8
#define SUBROUTINE prefix(power_z4_i8)
#include "power.c"

#define TYPE z8
#define ETYPE G95_INT1
#define SUBROUTINE prefix(power_z8_i1)
#include "power.c"

#define TYPE z8
#define ETYPE G95_INT2
#define SUBROUTINE prefix(power_z8_i2)
#include "power.c"

#define TYPE z8
#define ETYPE G95_INT4
#define SUBROUTINE prefix(power_z8_i4)
#include "power.c"

#define TYPE z8
#define ETYPE G95_INT8
#define SUBROUTINE prefix(power_z8_i8)
#include "power.c"

#undef COMPLEX

#endif
