/* Copyright (C) 2010 Martin Linder <mali2297@users.sf.net>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/gpl-2.0.html>.
 */
#include <string.h>
#include "e6x-pref.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

static GKeyFile *preferences = NULL;

static const gchar *default_data =
  "[Commands]\n"
  "Print=xfprint4 %f\n"
  "[Zoom]\n"
  "Min=0.25\n"
  "Max=4.0\n"
  "Step=0.25\n"
  "Default=1.0\n";

gboolean
e6x_pref_load (const gchar *filename)
{
  gboolean retval = TRUE;
  gchar *file_content = NULL; 
  gchar *data = NULL;

  if (G_UNLIKELY (preferences != NULL))
    e6x_pref_free ();
  
  if (G_LIKELY (filename != NULL))
    g_file_get_contents (filename, &file_content, NULL, NULL); 
  
  if (G_LIKELY (file_content != NULL))
    data = file_content;
  else
    data = g_strdup (default_data);
  
  preferences = g_key_file_new ();
  retval = g_key_file_load_from_data (preferences, data, strlen (data),
                                          G_KEY_FILE_KEEP_COMMENTS, NULL);

  if (file_content != NULL)
  {
    if (G_LIKELY (retval))
      retval = e6x_pref_check_keyfile (preferences);
    else
      e6x_pref_load (NULL);
  }
  
  g_free (data);
  
  return retval;
}

gboolean
e6x_pref_save (const gchar *filename)
{
  gboolean retval = FALSE;
  gchar *data = NULL;
  gsize len = 0;

  if (G_LIKELY (preferences))
  {
    data = g_key_file_to_data (preferences, &len, NULL);
    if (G_LIKELY (data))
    {
      retval = g_file_set_contents (filename, data, -1, NULL);
      g_free (data);
    }
  }
  
  return retval;
}

GKeyFile *
e6x_pref_get_keyfile ()
{
  if (!preferences)
  {
    e6x_pref_load (NULL);
  }
  
  return preferences;
}

gboolean
e6x_pref_check_keyfile (GKeyFile *keyfile)
{
  gboolean retval = TRUE;  
  gdouble zoom_min, zoom_max, zoom_step, zoom_default;
  GKeyFile *defaults = NULL;
  
  defaults = g_key_file_new ();
  g_key_file_load_from_data (defaults, default_data, strlen (default_data), 
                             0, NULL);
  
  if (!g_key_file_has_key (keyfile, "Commands", "Print", NULL))
  {
    gchar *cmd = g_key_file_get_string (defaults, "Commands", "Print", NULL);
    g_key_file_set_string (keyfile, "Commands", "Print", cmd);
    g_free (cmd);
    retval = FALSE;
  }

  zoom_min = g_key_file_get_double (keyfile, "Zoom", "Min", NULL);
  zoom_max = g_key_file_get_double (keyfile, "Zoom", "Max", NULL);
  zoom_step = g_key_file_get_double (keyfile, "Zoom", "Step", NULL);
  zoom_default = g_key_file_get_double (keyfile, "Zoom", "Default", NULL); 
  
  if (zoom_min <= 0.0 || zoom_min > zoom_max)
  {
    zoom_min = g_key_file_get_double (defaults, "Zoom", "Min", NULL);
    g_key_file_set_double (keyfile, "Zoom", "Min", zoom_min);
    retval = FALSE;
  }
  if (zoom_max <= 0.0 || zoom_min > zoom_max)
  {
    zoom_max = g_key_file_get_double (defaults, "Zoom", "Max", NULL);
    g_key_file_set_double (keyfile, "Zoom", "Max", zoom_max);
    retval = FALSE;
  }
  if (zoom_step <= 0.0)
  {
    zoom_step = g_key_file_get_double (defaults, "Zoom", "Step", NULL);
    g_key_file_set_double (keyfile, "Zoom", "Step", zoom_step);
    retval = FALSE;
  }
  if (zoom_default < zoom_min || zoom_default > zoom_max)
  {
    zoom_default = CLAMP (zoom_default, zoom_min, zoom_max);
    g_key_file_set_double (keyfile, "Zoom", "Default", zoom_default);
    retval = FALSE;
  }
    
  g_key_file_free (defaults);
  
  return retval;  
}

void
e6x_pref_free ()
{
  if (G_LIKELY (preferences))
  {
    g_key_file_free (preferences);
    preferences = NULL;
  }
}

