/*----------------------------------------------------------------------------------------
  Database dv
----------------------------------------------------------------------------------------*/

#include "dvdatabase.h"

union dvTempType_ dvTemp_;
struct dvRootType_ dvRootData;
uint8 dvModuleID;
struct dvRootFields dvRoots;
struct dvModpathFields dvModpaths;
struct dvModuleFields dvModules;
struct dvLinkFields dvLinks;
struct dvSchemaFields dvSchemas;
struct dvEnumFields dvEnums;
struct dvEntryFields dvEntrys;
struct dvTypedefFields dvTypedefs;
struct dvClassFields dvClasss;
struct dvPropertyFields dvPropertys;
struct dvSparsegroupFields dvSparsegroups;
struct dvRelationshipFields dvRelationships;
struct dvKeyFields dvKeys;
struct dvUnionFields dvUnions;
struct dvCaseFields dvCases;

/*----------------------------------------------------------------------------------------
  Constructor/Destructor hooks.
----------------------------------------------------------------------------------------*/
void(*dvRootConstructorCallback)(dvRoot);
void(*dvModpathConstructorCallback)(dvModpath);
void(*dvModuleConstructorCallback)(dvModule);
void(*dvLinkConstructorCallback)(dvLink);
void(*dvSchemaConstructorCallback)(dvSchema);
void(*dvEnumConstructorCallback)(dvEnum);
void(*dvEntryConstructorCallback)(dvEntry);
void(*dvTypedefConstructorCallback)(dvTypedef);
void(*dvClassConstructorCallback)(dvClass);
void(*dvPropertyConstructorCallback)(dvProperty);
void(*dvSparsegroupConstructorCallback)(dvSparsegroup);
void(*dvSparsegroupDestructorCallback)(dvSparsegroup);
void(*dvRelationshipConstructorCallback)(dvRelationship);
void(*dvKeyConstructorCallback)(dvKey);
void(*dvKeyDestructorCallback)(dvKey);
void(*dvUnionConstructorCallback)(dvUnion);
void(*dvUnionDestructorCallback)(dvUnion);
void(*dvCaseConstructorCallback)(dvCase);
void(*dvCaseDestructorCallback)(dvCase);

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocRoot(void)
{
    dvRoot Root = dvRootAlloc();

    return dvRoot2Index(Root);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Root.
----------------------------------------------------------------------------------------*/
static void allocRoots(void)
{
    dvSetAllocatedRoot(2);
    dvSetUsedRoot(0);
    dvRoots.FirstModpath = utNewA(dvModpath, (dvAllocatedRoot()));
    dvRoots.LastModpath = utNewA(dvModpath, (dvAllocatedRoot()));
    dvRoots.ModpathTableIndex = utNewA(uint32, (dvAllocatedRoot()));
    dvRoots.NumModpathTable = utNewA(uint32, (dvAllocatedRoot()));
    dvSetUsedRootModpathTable(0);
    dvSetAllocatedRootModpathTable(2);
    dvSetFreeRootModpathTable(0);
    dvRoots.ModpathTable = utNewA(dvModpath, dvAllocatedRootModpathTable());
    dvRoots.NumModpath = utNewA(uint32, (dvAllocatedRoot()));
    dvRoots.FirstModule = utNewA(dvModule, (dvAllocatedRoot()));
    dvRoots.LastModule = utNewA(dvModule, (dvAllocatedRoot()));
    dvRoots.ModuleTableIndex = utNewA(uint32, (dvAllocatedRoot()));
    dvRoots.NumModuleTable = utNewA(uint32, (dvAllocatedRoot()));
    dvSetUsedRootModuleTable(0);
    dvSetAllocatedRootModuleTable(2);
    dvSetFreeRootModuleTable(0);
    dvRoots.ModuleTable = utNewA(dvModule, dvAllocatedRootModuleTable());
    dvRoots.NumModule = utNewA(uint32, (dvAllocatedRoot()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Root.
----------------------------------------------------------------------------------------*/
static void reallocRoots(
    uint32 newSize)
{
    utResizeArray(dvRoots.FirstModpath, (newSize));
    utResizeArray(dvRoots.LastModpath, (newSize));
    utResizeArray(dvRoots.ModpathTableIndex, (newSize));
    utResizeArray(dvRoots.NumModpathTable, (newSize));
    utResizeArray(dvRoots.NumModpath, (newSize));
    utResizeArray(dvRoots.FirstModule, (newSize));
    utResizeArray(dvRoots.LastModule, (newSize));
    utResizeArray(dvRoots.ModuleTableIndex, (newSize));
    utResizeArray(dvRoots.NumModuleTable, (newSize));
    utResizeArray(dvRoots.NumModule, (newSize));
    dvSetAllocatedRoot(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Roots.
----------------------------------------------------------------------------------------*/
void dvRootAllocMore(void)
{
    reallocRoots((uint32)(dvAllocatedRoot() + (dvAllocatedRoot() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the Root.ModpathTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactRootModpathTables(void)
{
    uint32 elementSize = sizeof(dvModpath);
    uint32 usedHeaderSize = (sizeof(dvRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvModpath *toPtr = dvRoots.ModpathTable;
    dvModpath *fromPtr = toPtr;
    dvRoot Root;
    uint32 size;

    while(fromPtr < dvRoots.ModpathTable + dvUsedRootModpathTable()) {
        Root = *(dvRoot *)(void *)fromPtr;
        if(Root != dvRootNull) {
            /* Need to move it to toPtr */
            size = utMax(dvRootGetNumModpathTable(Root) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvRootSetModpathTableIndex(Root, toPtr - dvRoots.ModpathTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvRoot *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedRootModpathTable(toPtr - dvRoots.ModpathTable);
    dvSetFreeRootModpathTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Root.ModpathTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreRootModpathTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedRootModpathTable() - dvUsedRootModpathTable();

    if((dvFreeRootModpathTable() << 2) > dvUsedRootModpathTable()) {
        dvCompactRootModpathTables();
        freeSpace = dvAllocatedRootModpathTable() - dvUsedRootModpathTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedRootModpathTable(dvAllocatedRootModpathTable() + spaceNeeded - freeSpace +
            (dvAllocatedRootModpathTable() >> 1));
        utResizeArray(dvRoots.ModpathTable, dvAllocatedRootModpathTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Root.ModpathTable array.
----------------------------------------------------------------------------------------*/
void dvRootAllocModpathTables(
    dvRoot Root,
    uint32 numModpathTables)
{
    uint32 freeSpace = dvAllocatedRootModpathTable() - dvUsedRootModpathTable();
    uint32 elementSize = sizeof(dvModpath);
    uint32 usedHeaderSize = (sizeof(dvRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numModpathTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvRootGetNumModpathTable(Root) == 0);
#endif
    if(numModpathTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreRootModpathTables(spaceNeeded);
    }
    dvRootSetModpathTableIndex(Root, dvUsedRootModpathTable() + usedHeaderSize);
    dvRootSetNumModpathTable(Root, numModpathTables);
    *(dvRoot *)(void *)(dvRoots.ModpathTable + dvUsedRootModpathTable()) = Root;
    {
        uint32 xRoot;
        for(xRoot = (uint32)(dvRootGetModpathTableIndex(Root)); xRoot < dvRootGetModpathTableIndex(Root) + numModpathTables; xRoot++) {
            dvRoots.ModpathTable[xRoot] = dvModpathNull;
        }
    }
    dvSetUsedRootModpathTable(dvUsedRootModpathTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvRootGetModpathTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getRootModpathTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvRoot Root = dvIndex2Root((uint32)objectNumber);

    *numValues = dvRootGetNumModpathTable(Root);
    return dvRootGetModpathTables(Root);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvRootAllocModpathTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocRootModpathTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvRoot Root = dvIndex2Root((uint32)objectNumber);

    dvRootSetModpathTableIndex(Root, 0);
    dvRootSetNumModpathTable(Root, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvRootAllocModpathTables(Root, numValues);
    return dvRootGetModpathTables(Root);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Root.ModpathTable array.
----------------------------------------------------------------------------------------*/
void dvRootFreeModpathTables(
    dvRoot Root)
{
    uint32 elementSize = sizeof(dvModpath);
    uint32 usedHeaderSize = (sizeof(dvRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvRootGetNumModpathTable(Root) + usedHeaderSize, freeHeaderSize);
    dvModpath *dataPtr = dvRootGetModpathTables(Root) - usedHeaderSize;

    if(dvRootGetNumModpathTable(Root) == 0) {
        return;
    }
    *(dvRoot *)(void *)(dataPtr) = dvRootNull;
    *(uint32 *)(void *)(((dvRoot *)(void *)dataPtr) + 1) = size;
    dvRootSetNumModpathTable(Root, 0);
    dvSetFreeRootModpathTable(dvFreeRootModpathTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Root.ModpathTable array.
----------------------------------------------------------------------------------------*/
void dvRootResizeModpathTables(
    dvRoot Root,
    uint32 numModpathTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvModpath);
    uint32 usedHeaderSize = (sizeof(dvRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numModpathTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvRootGetNumModpathTable(Root) + usedHeaderSize, freeHeaderSize);
    dvModpath *dataPtr;

    if(numModpathTables == 0) {
        if(dvRootGetNumModpathTable(Root) != 0) {
            dvRootFreeModpathTables(Root);
        }
        return;
    }
    if(dvRootGetNumModpathTable(Root) == 0) {
        dvRootAllocModpathTables(Root, numModpathTables);
        return;
    }
    freeSpace = dvAllocatedRootModpathTable() - dvUsedRootModpathTable();
    if(freeSpace < newSize) {
        allocMoreRootModpathTables(newSize);
    }
    dataPtr = dvRootGetModpathTables(Root) - usedHeaderSize;
    memcpy((void *)(dvRoots.ModpathTable + dvUsedRootModpathTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xRoot;
            for(xRoot = (uint32)(dvUsedRootModpathTable() + oldSize); xRoot < dvUsedRootModpathTable() + oldSize + newSize - oldSize; xRoot++) {
                dvRoots.ModpathTable[xRoot] = dvModpathNull;
            }
        }
    }
    *(dvRoot *)(void *)dataPtr = dvRootNull;
    *(uint32 *)(void *)(((dvRoot *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeRootModpathTable(dvFreeRootModpathTable() + oldSize);
    dvRootSetModpathTableIndex(Root, dvUsedRootModpathTable() + usedHeaderSize);
    dvRootSetNumModpathTable(Root, numModpathTables);
    dvSetUsedRootModpathTable(dvUsedRootModpathTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the Root.ModuleTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactRootModuleTables(void)
{
    uint32 elementSize = sizeof(dvModule);
    uint32 usedHeaderSize = (sizeof(dvRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvModule *toPtr = dvRoots.ModuleTable;
    dvModule *fromPtr = toPtr;
    dvRoot Root;
    uint32 size;

    while(fromPtr < dvRoots.ModuleTable + dvUsedRootModuleTable()) {
        Root = *(dvRoot *)(void *)fromPtr;
        if(Root != dvRootNull) {
            /* Need to move it to toPtr */
            size = utMax(dvRootGetNumModuleTable(Root) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvRootSetModuleTableIndex(Root, toPtr - dvRoots.ModuleTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvRoot *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedRootModuleTable(toPtr - dvRoots.ModuleTable);
    dvSetFreeRootModuleTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Root.ModuleTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreRootModuleTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedRootModuleTable() - dvUsedRootModuleTable();

    if((dvFreeRootModuleTable() << 2) > dvUsedRootModuleTable()) {
        dvCompactRootModuleTables();
        freeSpace = dvAllocatedRootModuleTable() - dvUsedRootModuleTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedRootModuleTable(dvAllocatedRootModuleTable() + spaceNeeded - freeSpace +
            (dvAllocatedRootModuleTable() >> 1));
        utResizeArray(dvRoots.ModuleTable, dvAllocatedRootModuleTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Root.ModuleTable array.
----------------------------------------------------------------------------------------*/
void dvRootAllocModuleTables(
    dvRoot Root,
    uint32 numModuleTables)
{
    uint32 freeSpace = dvAllocatedRootModuleTable() - dvUsedRootModuleTable();
    uint32 elementSize = sizeof(dvModule);
    uint32 usedHeaderSize = (sizeof(dvRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numModuleTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvRootGetNumModuleTable(Root) == 0);
#endif
    if(numModuleTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreRootModuleTables(spaceNeeded);
    }
    dvRootSetModuleTableIndex(Root, dvUsedRootModuleTable() + usedHeaderSize);
    dvRootSetNumModuleTable(Root, numModuleTables);
    *(dvRoot *)(void *)(dvRoots.ModuleTable + dvUsedRootModuleTable()) = Root;
    {
        uint32 xRoot;
        for(xRoot = (uint32)(dvRootGetModuleTableIndex(Root)); xRoot < dvRootGetModuleTableIndex(Root) + numModuleTables; xRoot++) {
            dvRoots.ModuleTable[xRoot] = dvModuleNull;
        }
    }
    dvSetUsedRootModuleTable(dvUsedRootModuleTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvRootGetModuleTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getRootModuleTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvRoot Root = dvIndex2Root((uint32)objectNumber);

    *numValues = dvRootGetNumModuleTable(Root);
    return dvRootGetModuleTables(Root);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvRootAllocModuleTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocRootModuleTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvRoot Root = dvIndex2Root((uint32)objectNumber);

    dvRootSetModuleTableIndex(Root, 0);
    dvRootSetNumModuleTable(Root, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvRootAllocModuleTables(Root, numValues);
    return dvRootGetModuleTables(Root);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Root.ModuleTable array.
----------------------------------------------------------------------------------------*/
void dvRootFreeModuleTables(
    dvRoot Root)
{
    uint32 elementSize = sizeof(dvModule);
    uint32 usedHeaderSize = (sizeof(dvRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvRootGetNumModuleTable(Root) + usedHeaderSize, freeHeaderSize);
    dvModule *dataPtr = dvRootGetModuleTables(Root) - usedHeaderSize;

    if(dvRootGetNumModuleTable(Root) == 0) {
        return;
    }
    *(dvRoot *)(void *)(dataPtr) = dvRootNull;
    *(uint32 *)(void *)(((dvRoot *)(void *)dataPtr) + 1) = size;
    dvRootSetNumModuleTable(Root, 0);
    dvSetFreeRootModuleTable(dvFreeRootModuleTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Root.ModuleTable array.
----------------------------------------------------------------------------------------*/
void dvRootResizeModuleTables(
    dvRoot Root,
    uint32 numModuleTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvModule);
    uint32 usedHeaderSize = (sizeof(dvRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numModuleTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvRootGetNumModuleTable(Root) + usedHeaderSize, freeHeaderSize);
    dvModule *dataPtr;

    if(numModuleTables == 0) {
        if(dvRootGetNumModuleTable(Root) != 0) {
            dvRootFreeModuleTables(Root);
        }
        return;
    }
    if(dvRootGetNumModuleTable(Root) == 0) {
        dvRootAllocModuleTables(Root, numModuleTables);
        return;
    }
    freeSpace = dvAllocatedRootModuleTable() - dvUsedRootModuleTable();
    if(freeSpace < newSize) {
        allocMoreRootModuleTables(newSize);
    }
    dataPtr = dvRootGetModuleTables(Root) - usedHeaderSize;
    memcpy((void *)(dvRoots.ModuleTable + dvUsedRootModuleTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xRoot;
            for(xRoot = (uint32)(dvUsedRootModuleTable() + oldSize); xRoot < dvUsedRootModuleTable() + oldSize + newSize - oldSize; xRoot++) {
                dvRoots.ModuleTable[xRoot] = dvModuleNull;
            }
        }
    }
    *(dvRoot *)(void *)dataPtr = dvRootNull;
    *(uint32 *)(void *)(((dvRoot *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeRootModuleTable(dvFreeRootModuleTable() + oldSize);
    dvRootSetModuleTableIndex(Root, dvUsedRootModuleTable() + usedHeaderSize);
    dvRootSetNumModuleTable(Root, numModuleTables);
    dvSetUsedRootModuleTable(dvUsedRootModuleTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Root.
----------------------------------------------------------------------------------------*/
void dvRootCopyProps(
    dvRoot oldRoot,
    dvRoot newRoot)
{
}

static void addRootModpathToHashTable(dvRoot Root, dvModpath _Modpath);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeRootModpathHashTable(
    dvRoot Root)
{
    dvModpath _Modpath;
    dvModpath *Modpaths;
    uint32 numModpaths = dvRootGetNumModpathTable(Root) << 1;

    if(numModpaths == 0) {
        numModpaths = 2;
        dvRootAllocModpathTables(Root, 2);
    } else {
        dvRootResizeModpathTables(Root, numModpaths);
    }
    Modpaths = dvRootGetModpathTables(Root);
    /* Zero out the table */
    while(numModpaths-- != 0) {
        *Modpaths++ = dvModpathNull;
    }
    dvRootSetNumModpath(Root, 0);
    dvForeachRootModpath(Root, _Modpath) {
        if(dvModpathGetSym(_Modpath) != utSymNull) {
            addRootModpathToHashTable(Root, _Modpath);
        }
    } dvEndRootModpath;
}

/*----------------------------------------------------------------------------------------
  Add the Modpath to the Root.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addRootModpathToHashTable(
    dvRoot Root,
    dvModpath _Modpath)
{
    dvModpath nextModpath;
    uint32 index;

    if(dvRootGetNumModpath(Root) >= dvRootGetNumModpathTable(Root)) {
        resizeRootModpathHashTable(Root);
        return;
    }
    index = (dvRootGetNumModpathTable(Root) - 1) & utSymGetHashValue(dvModpathGetSym(_Modpath));
    nextModpath = dvRootGetiModpathTable(Root, index);
    dvModpathSetNextTableRootModpath(_Modpath, nextModpath);
    dvRootSetiModpathTable(Root, index, _Modpath);
    dvRootSetNumModpath(Root, dvRootGetNumModpath(Root) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Modpath from the hash table.
----------------------------------------------------------------------------------------*/
static void removeRootModpathFromHashTable(
   dvRoot Root,
   dvModpath _Modpath)
{
    uint32 index = (dvRootGetNumModpathTable(Root) - 1) & utSymGetHashValue(dvModpathGetSym(_Modpath));
    dvModpath prevModpath, nextModpath;
    
    nextModpath = dvRootGetiModpathTable(Root, index);
    if(nextModpath == _Modpath) {
        dvRootSetiModpathTable(Root, index, dvModpathGetNextTableRootModpath(nextModpath));
    } else {
        do {
            prevModpath = nextModpath;
            nextModpath = dvModpathGetNextTableRootModpath(nextModpath);
        } while(nextModpath != _Modpath);
        dvModpathSetNextTableRootModpath(prevModpath, dvModpathGetNextTableRootModpath(_Modpath));
    }
    dvRootSetNumModpath(Root, dvRootGetNumModpath(Root) - 1);
    dvModpathSetNextTableRootModpath(_Modpath, dvModpathNull);
}

/*----------------------------------------------------------------------------------------
  Find the Modpath from the Root and its hash key.
----------------------------------------------------------------------------------------*/
dvModpath dvRootFindModpath(
    dvRoot Root,
    utSym Sym)
{
    uint32 mask = dvRootGetNumModpathTable(Root) - 1;
    dvModpath _Modpath;

    if(mask + 1 != 0) {
        _Modpath = dvRootGetiModpathTable(Root, utSymGetHashValue(Sym) & mask);
        while(_Modpath != dvModpathNull) {
            if(dvModpathGetSym(_Modpath) == Sym) {
                return _Modpath;
            }
            _Modpath = dvModpathGetNextTableRootModpath(_Modpath);
        }
    }
    return dvModpathNull;
}

/*----------------------------------------------------------------------------------------
  Find the Modpath from the Root and its name.
----------------------------------------------------------------------------------------*/
void dvRootRenameModpath(
    dvRoot Root,
    dvModpath _Modpath,
    utSym sym)
{
    if(dvModpathGetSym(_Modpath) != utSymNull) {
        removeRootModpathFromHashTable(Root, _Modpath);
    }
    dvModpathSetSym(_Modpath, sym);
    if(sym != utSymNull) {
        addRootModpathToHashTable(Root, _Modpath);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Modpath to the head of the list on the Root.
----------------------------------------------------------------------------------------*/
void dvRootInsertModpath(
    dvRoot Root,
    dvModpath _Modpath)
{
#if defined(DD_DEBUG)
    if(Root == dvRootNull) {
        utExit("Non-existent Root");
    }
    if(_Modpath == dvModpathNull) {
        utExit("Non-existent Modpath");
    }
    if(dvModpathGetRoot(_Modpath) != dvRootNull) {
        utExit("Attempting to add Modpath to Root twice");
    }
#endif
    dvModpathSetNextRootModpath(_Modpath, dvRootGetFirstModpath(Root));
    if(dvRootGetFirstModpath(Root) != dvModpathNull) {
        dvModpathSetPrevRootModpath(dvRootGetFirstModpath(Root), _Modpath);
    }
    dvRootSetFirstModpath(Root, _Modpath);
    dvModpathSetPrevRootModpath(_Modpath, dvModpathNull);
    if(dvRootGetLastModpath(Root) == dvModpathNull) {
        dvRootSetLastModpath(Root, _Modpath);
    }
    dvModpathSetRoot(_Modpath, Root);
    if(dvModpathGetSym(_Modpath) != utSymNull) {
        addRootModpathToHashTable(Root, _Modpath);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Modpath to the end of the list on the Root.
----------------------------------------------------------------------------------------*/
void dvRootAppendModpath(
    dvRoot Root,
    dvModpath _Modpath)
{
#if defined(DD_DEBUG)
    if(Root == dvRootNull) {
        utExit("Non-existent Root");
    }
    if(_Modpath == dvModpathNull) {
        utExit("Non-existent Modpath");
    }
    if(dvModpathGetRoot(_Modpath) != dvRootNull) {
        utExit("Attempting to add Modpath to Root twice");
    }
#endif
    dvModpathSetPrevRootModpath(_Modpath, dvRootGetLastModpath(Root));
    if(dvRootGetLastModpath(Root) != dvModpathNull) {
        dvModpathSetNextRootModpath(dvRootGetLastModpath(Root), _Modpath);
    }
    dvRootSetLastModpath(Root, _Modpath);
    dvModpathSetNextRootModpath(_Modpath, dvModpathNull);
    if(dvRootGetFirstModpath(Root) == dvModpathNull) {
        dvRootSetFirstModpath(Root, _Modpath);
    }
    dvModpathSetRoot(_Modpath, Root);
    if(dvModpathGetSym(_Modpath) != utSymNull) {
        addRootModpathToHashTable(Root, _Modpath);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Modpath to the Root after the previous Modpath.
----------------------------------------------------------------------------------------*/
void dvRootInsertAfterModpath(
    dvRoot Root,
    dvModpath prevModpath,
    dvModpath _Modpath)
{
    dvModpath nextModpath = dvModpathGetNextRootModpath(prevModpath);

#if defined(DD_DEBUG)
    if(Root == dvRootNull) {
        utExit("Non-existent Root");
    }
    if(_Modpath == dvModpathNull) {
        utExit("Non-existent Modpath");
    }
    if(dvModpathGetRoot(_Modpath) != dvRootNull) {
        utExit("Attempting to add Modpath to Root twice");
    }
#endif
    dvModpathSetNextRootModpath(_Modpath, nextModpath);
    dvModpathSetNextRootModpath(prevModpath, _Modpath);
    dvModpathSetPrevRootModpath(_Modpath, prevModpath);
    if(nextModpath != dvModpathNull) {
        dvModpathSetPrevRootModpath(nextModpath, _Modpath);
    }
    if(dvRootGetLastModpath(Root) == prevModpath) {
        dvRootSetLastModpath(Root, _Modpath);
    }
    dvModpathSetRoot(_Modpath, Root);
    if(dvModpathGetSym(_Modpath) != utSymNull) {
        addRootModpathToHashTable(Root, _Modpath);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Modpath from the Root.
----------------------------------------------------------------------------------------*/
void dvRootRemoveModpath(
    dvRoot Root,
    dvModpath _Modpath)
{
    dvModpath pModpath, nModpath;

#if defined(DD_DEBUG)
    if(_Modpath == dvModpathNull) {
        utExit("Non-existent Modpath");
    }
    if(dvModpathGetRoot(_Modpath) != dvRootNull && dvModpathGetRoot(_Modpath) != Root) {
        utExit("Delete Modpath from non-owning Root");
    }
#endif
    nModpath = dvModpathGetNextRootModpath(_Modpath);
    pModpath = dvModpathGetPrevRootModpath(_Modpath);
    if(pModpath != dvModpathNull) {
        dvModpathSetNextRootModpath(pModpath, nModpath);
    } else if(dvRootGetFirstModpath(Root) == _Modpath) {
        dvRootSetFirstModpath(Root, nModpath);
    }
    if(nModpath != dvModpathNull) {
        dvModpathSetPrevRootModpath(nModpath, pModpath);
    } else if(dvRootGetLastModpath(Root) == _Modpath) {
        dvRootSetLastModpath(Root, pModpath);
    }
    dvModpathSetNextRootModpath(_Modpath, dvModpathNull);
    dvModpathSetPrevRootModpath(_Modpath, dvModpathNull);
    dvModpathSetRoot(_Modpath, dvRootNull);
    if(dvModpathGetSym(_Modpath) != utSymNull) {
        removeRootModpathFromHashTable(Root, _Modpath);
    }
}

static void addRootModuleToHashTable(dvRoot Root, dvModule _Module);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeRootModuleHashTable(
    dvRoot Root)
{
    dvModule _Module;
    dvModule *Modules;
    uint32 numModules = dvRootGetNumModuleTable(Root) << 1;

    if(numModules == 0) {
        numModules = 2;
        dvRootAllocModuleTables(Root, 2);
    } else {
        dvRootResizeModuleTables(Root, numModules);
    }
    Modules = dvRootGetModuleTables(Root);
    /* Zero out the table */
    while(numModules-- != 0) {
        *Modules++ = dvModuleNull;
    }
    dvRootSetNumModule(Root, 0);
    dvForeachRootModule(Root, _Module) {
        if(dvModuleGetSym(_Module) != utSymNull) {
            addRootModuleToHashTable(Root, _Module);
        }
    } dvEndRootModule;
}

/*----------------------------------------------------------------------------------------
  Add the Module to the Root.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addRootModuleToHashTable(
    dvRoot Root,
    dvModule _Module)
{
    dvModule nextModule;
    uint32 index;

    if(dvRootGetNumModule(Root) >= dvRootGetNumModuleTable(Root)) {
        resizeRootModuleHashTable(Root);
        return;
    }
    index = (dvRootGetNumModuleTable(Root) - 1) & utSymGetHashValue(dvModuleGetSym(_Module));
    nextModule = dvRootGetiModuleTable(Root, index);
    dvModuleSetNextTableRootModule(_Module, nextModule);
    dvRootSetiModuleTable(Root, index, _Module);
    dvRootSetNumModule(Root, dvRootGetNumModule(Root) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Module from the hash table.
----------------------------------------------------------------------------------------*/
static void removeRootModuleFromHashTable(
   dvRoot Root,
   dvModule _Module)
{
    uint32 index = (dvRootGetNumModuleTable(Root) - 1) & utSymGetHashValue(dvModuleGetSym(_Module));
    dvModule prevModule, nextModule;
    
    nextModule = dvRootGetiModuleTable(Root, index);
    if(nextModule == _Module) {
        dvRootSetiModuleTable(Root, index, dvModuleGetNextTableRootModule(nextModule));
    } else {
        do {
            prevModule = nextModule;
            nextModule = dvModuleGetNextTableRootModule(nextModule);
        } while(nextModule != _Module);
        dvModuleSetNextTableRootModule(prevModule, dvModuleGetNextTableRootModule(_Module));
    }
    dvRootSetNumModule(Root, dvRootGetNumModule(Root) - 1);
    dvModuleSetNextTableRootModule(_Module, dvModuleNull);
}

/*----------------------------------------------------------------------------------------
  Find the Module from the Root and its hash key.
----------------------------------------------------------------------------------------*/
dvModule dvRootFindModule(
    dvRoot Root,
    utSym Sym)
{
    uint32 mask = dvRootGetNumModuleTable(Root) - 1;
    dvModule _Module;

    if(mask + 1 != 0) {
        _Module = dvRootGetiModuleTable(Root, utSymGetHashValue(Sym) & mask);
        while(_Module != dvModuleNull) {
            if(dvModuleGetSym(_Module) == Sym) {
                return _Module;
            }
            _Module = dvModuleGetNextTableRootModule(_Module);
        }
    }
    return dvModuleNull;
}

/*----------------------------------------------------------------------------------------
  Find the Module from the Root and its name.
----------------------------------------------------------------------------------------*/
void dvRootRenameModule(
    dvRoot Root,
    dvModule _Module,
    utSym sym)
{
    if(dvModuleGetSym(_Module) != utSymNull) {
        removeRootModuleFromHashTable(Root, _Module);
    }
    dvModuleSetSym(_Module, sym);
    if(sym != utSymNull) {
        addRootModuleToHashTable(Root, _Module);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Module to the head of the list on the Root.
----------------------------------------------------------------------------------------*/
void dvRootInsertModule(
    dvRoot Root,
    dvModule _Module)
{
#if defined(DD_DEBUG)
    if(Root == dvRootNull) {
        utExit("Non-existent Root");
    }
    if(_Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
#endif
    dvModuleSetNextRootModule(_Module, dvRootGetFirstModule(Root));
    if(dvRootGetFirstModule(Root) != dvModuleNull) {
        dvModuleSetPrevRootModule(dvRootGetFirstModule(Root), _Module);
    }
    dvRootSetFirstModule(Root, _Module);
    dvModuleSetPrevRootModule(_Module, dvModuleNull);
    if(dvRootGetLastModule(Root) == dvModuleNull) {
        dvRootSetLastModule(Root, _Module);
    }
    if(dvModuleGetSym(_Module) != utSymNull) {
        addRootModuleToHashTable(Root, _Module);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Module to the end of the list on the Root.
----------------------------------------------------------------------------------------*/
void dvRootAppendModule(
    dvRoot Root,
    dvModule _Module)
{
#if defined(DD_DEBUG)
    if(Root == dvRootNull) {
        utExit("Non-existent Root");
    }
    if(_Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
#endif
    dvModuleSetPrevRootModule(_Module, dvRootGetLastModule(Root));
    if(dvRootGetLastModule(Root) != dvModuleNull) {
        dvModuleSetNextRootModule(dvRootGetLastModule(Root), _Module);
    }
    dvRootSetLastModule(Root, _Module);
    dvModuleSetNextRootModule(_Module, dvModuleNull);
    if(dvRootGetFirstModule(Root) == dvModuleNull) {
        dvRootSetFirstModule(Root, _Module);
    }
    if(dvModuleGetSym(_Module) != utSymNull) {
        addRootModuleToHashTable(Root, _Module);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Module to the Root after the previous Module.
----------------------------------------------------------------------------------------*/
void dvRootInsertAfterModule(
    dvRoot Root,
    dvModule prevModule,
    dvModule _Module)
{
    dvModule nextModule = dvModuleGetNextRootModule(prevModule);

#if defined(DD_DEBUG)
    if(Root == dvRootNull) {
        utExit("Non-existent Root");
    }
    if(_Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
#endif
    dvModuleSetNextRootModule(_Module, nextModule);
    dvModuleSetNextRootModule(prevModule, _Module);
    dvModuleSetPrevRootModule(_Module, prevModule);
    if(nextModule != dvModuleNull) {
        dvModuleSetPrevRootModule(nextModule, _Module);
    }
    if(dvRootGetLastModule(Root) == prevModule) {
        dvRootSetLastModule(Root, _Module);
    }
    if(dvModuleGetSym(_Module) != utSymNull) {
        addRootModuleToHashTable(Root, _Module);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Module from the Root.
----------------------------------------------------------------------------------------*/
void dvRootRemoveModule(
    dvRoot Root,
    dvModule _Module)
{
    dvModule pModule, nModule;

#if defined(DD_DEBUG)
    if(_Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
#endif
    nModule = dvModuleGetNextRootModule(_Module);
    pModule = dvModuleGetPrevRootModule(_Module);
    if(pModule != dvModuleNull) {
        dvModuleSetNextRootModule(pModule, nModule);
    } else if(dvRootGetFirstModule(Root) == _Module) {
        dvRootSetFirstModule(Root, nModule);
    }
    if(nModule != dvModuleNull) {
        dvModuleSetPrevRootModule(nModule, pModule);
    } else if(dvRootGetLastModule(Root) == _Module) {
        dvRootSetLastModule(Root, pModule);
    }
    dvModuleSetNextRootModule(_Module, dvModuleNull);
    dvModuleSetPrevRootModule(_Module, dvModuleNull);
    if(dvModuleGetSym(_Module) != utSymNull) {
        removeRootModuleFromHashTable(Root, _Module);
    }
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowRoot(
    dvRoot Root)
{
    utDatabaseShowObject("dv", "Root", dvRoot2Index(Root));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocModpath(void)
{
    dvModpath Modpath = dvModpathAlloc();

    return dvModpath2Index(Modpath);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Modpath.
----------------------------------------------------------------------------------------*/
static void allocModpaths(void)
{
    dvSetAllocatedModpath(2);
    dvSetUsedModpath(0);
    dvModpaths.Sym = utNewA(utSym, (dvAllocatedModpath()));
    dvModpaths.Root = utNewA(dvRoot, (dvAllocatedModpath()));
    dvModpaths.NextRootModpath = utNewA(dvModpath, (dvAllocatedModpath()));
    dvModpaths.PrevRootModpath = utNewA(dvModpath, (dvAllocatedModpath()));
    dvModpaths.NextTableRootModpath = utNewA(dvModpath, (dvAllocatedModpath()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Modpath.
----------------------------------------------------------------------------------------*/
static void reallocModpaths(
    uint32 newSize)
{
    utResizeArray(dvModpaths.Sym, (newSize));
    utResizeArray(dvModpaths.Root, (newSize));
    utResizeArray(dvModpaths.NextRootModpath, (newSize));
    utResizeArray(dvModpaths.PrevRootModpath, (newSize));
    utResizeArray(dvModpaths.NextTableRootModpath, (newSize));
    dvSetAllocatedModpath(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Modpaths.
----------------------------------------------------------------------------------------*/
void dvModpathAllocMore(void)
{
    reallocModpaths((uint32)(dvAllocatedModpath() + (dvAllocatedModpath() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Modpath.
----------------------------------------------------------------------------------------*/
void dvModpathCopyProps(
    dvModpath oldModpath,
    dvModpath newModpath)
{
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowModpath(
    dvModpath Modpath)
{
    utDatabaseShowObject("dv", "Modpath", dvModpath2Index(Modpath));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocModule(void)
{
    dvModule Module = dvModuleAlloc();

    return dvModule2Index(Module);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Module.
----------------------------------------------------------------------------------------*/
static void allocModules(void)
{
    dvSetAllocatedModule(2);
    dvSetUsedModule(0);
    dvModules.Sym = utNewA(utSym, (dvAllocatedModule()));
    dvModules.PrefixSym = utNewA(utSym, (dvAllocatedModule()));
    dvModules.Persistent = utNewA(uint8, (dvAllocatedModule() + 7) >> 3);
    dvModules.UndoRedo = utNewA(uint8, (dvAllocatedModule() + 7) >> 3);
    dvModules.HasSparseData = utNewA(uint8, (dvAllocatedModule() + 7) >> 3);
    dvModules.NumFields = utNewA(uint16, (dvAllocatedModule()));
    dvModules.NumClasses = utNewA(uint32, (dvAllocatedModule()));
    dvModules.NumEnums = utNewA(uint32, (dvAllocatedModule()));
    dvModules.NextRootModule = utNewA(dvModule, (dvAllocatedModule()));
    dvModules.PrevRootModule = utNewA(dvModule, (dvAllocatedModule()));
    dvModules.NextTableRootModule = utNewA(dvModule, (dvAllocatedModule()));
    dvModules.FirstClass = utNewA(dvClass, (dvAllocatedModule()));
    dvModules.LastClass = utNewA(dvClass, (dvAllocatedModule()));
    dvModules.ClassTableIndex = utNewA(uint32, (dvAllocatedModule()));
    dvModules.NumClassTable = utNewA(uint32, (dvAllocatedModule()));
    dvSetUsedModuleClassTable(0);
    dvSetAllocatedModuleClassTable(2);
    dvSetFreeModuleClassTable(0);
    dvModules.ClassTable = utNewA(dvClass, dvAllocatedModuleClassTable());
    dvModules.NumClass = utNewA(uint32, (dvAllocatedModule()));
    dvModules.FirstEnum = utNewA(dvEnum, (dvAllocatedModule()));
    dvModules.LastEnum = utNewA(dvEnum, (dvAllocatedModule()));
    dvModules.EnumTableIndex = utNewA(uint32, (dvAllocatedModule()));
    dvModules.NumEnumTable = utNewA(uint32, (dvAllocatedModule()));
    dvSetUsedModuleEnumTable(0);
    dvSetAllocatedModuleEnumTable(2);
    dvSetFreeModuleEnumTable(0);
    dvModules.EnumTable = utNewA(dvEnum, dvAllocatedModuleEnumTable());
    dvModules.NumEnum = utNewA(uint32, (dvAllocatedModule()));
    dvModules.FirstTypedef = utNewA(dvTypedef, (dvAllocatedModule()));
    dvModules.LastTypedef = utNewA(dvTypedef, (dvAllocatedModule()));
    dvModules.TypedefTableIndex = utNewA(uint32, (dvAllocatedModule()));
    dvModules.NumTypedefTable = utNewA(uint32, (dvAllocatedModule()));
    dvSetUsedModuleTypedefTable(0);
    dvSetAllocatedModuleTypedefTable(2);
    dvSetFreeModuleTypedefTable(0);
    dvModules.TypedefTable = utNewA(dvTypedef, dvAllocatedModuleTypedefTable());
    dvModules.NumTypedef = utNewA(uint32, (dvAllocatedModule()));
    dvModules.FirstSchema = utNewA(dvSchema, (dvAllocatedModule()));
    dvModules.LastSchema = utNewA(dvSchema, (dvAllocatedModule()));
    dvModules.SchemaTableIndex = utNewA(uint32, (dvAllocatedModule()));
    dvModules.NumSchemaTable = utNewA(uint32, (dvAllocatedModule()));
    dvSetUsedModuleSchemaTable(0);
    dvSetAllocatedModuleSchemaTable(2);
    dvSetFreeModuleSchemaTable(0);
    dvModules.SchemaTable = utNewA(dvSchema, dvAllocatedModuleSchemaTable());
    dvModules.NumSchema = utNewA(uint32, (dvAllocatedModule()));
    dvModules.FirstImportLink = utNewA(dvLink, (dvAllocatedModule()));
    dvModules.LastImportLink = utNewA(dvLink, (dvAllocatedModule()));
    dvModules.FirstExportLink = utNewA(dvLink, (dvAllocatedModule()));
    dvModules.LastExportLink = utNewA(dvLink, (dvAllocatedModule()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Module.
----------------------------------------------------------------------------------------*/
static void reallocModules(
    uint32 newSize)
{
    utResizeArray(dvModules.Sym, (newSize));
    utResizeArray(dvModules.PrefixSym, (newSize));
    utResizeArray(dvModules.Persistent, (newSize + 7) >> 3);
    utResizeArray(dvModules.UndoRedo, (newSize + 7) >> 3);
    utResizeArray(dvModules.HasSparseData, (newSize + 7) >> 3);
    utResizeArray(dvModules.NumFields, (newSize));
    utResizeArray(dvModules.NumClasses, (newSize));
    utResizeArray(dvModules.NumEnums, (newSize));
    utResizeArray(dvModules.NextRootModule, (newSize));
    utResizeArray(dvModules.PrevRootModule, (newSize));
    utResizeArray(dvModules.NextTableRootModule, (newSize));
    utResizeArray(dvModules.FirstClass, (newSize));
    utResizeArray(dvModules.LastClass, (newSize));
    utResizeArray(dvModules.ClassTableIndex, (newSize));
    utResizeArray(dvModules.NumClassTable, (newSize));
    utResizeArray(dvModules.NumClass, (newSize));
    utResizeArray(dvModules.FirstEnum, (newSize));
    utResizeArray(dvModules.LastEnum, (newSize));
    utResizeArray(dvModules.EnumTableIndex, (newSize));
    utResizeArray(dvModules.NumEnumTable, (newSize));
    utResizeArray(dvModules.NumEnum, (newSize));
    utResizeArray(dvModules.FirstTypedef, (newSize));
    utResizeArray(dvModules.LastTypedef, (newSize));
    utResizeArray(dvModules.TypedefTableIndex, (newSize));
    utResizeArray(dvModules.NumTypedefTable, (newSize));
    utResizeArray(dvModules.NumTypedef, (newSize));
    utResizeArray(dvModules.FirstSchema, (newSize));
    utResizeArray(dvModules.LastSchema, (newSize));
    utResizeArray(dvModules.SchemaTableIndex, (newSize));
    utResizeArray(dvModules.NumSchemaTable, (newSize));
    utResizeArray(dvModules.NumSchema, (newSize));
    utResizeArray(dvModules.FirstImportLink, (newSize));
    utResizeArray(dvModules.LastImportLink, (newSize));
    utResizeArray(dvModules.FirstExportLink, (newSize));
    utResizeArray(dvModules.LastExportLink, (newSize));
    dvSetAllocatedModule(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Modules.
----------------------------------------------------------------------------------------*/
void dvModuleAllocMore(void)
{
    reallocModules((uint32)(dvAllocatedModule() + (dvAllocatedModule() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the Module.ClassTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactModuleClassTables(void)
{
    uint32 elementSize = sizeof(dvClass);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvClass *toPtr = dvModules.ClassTable;
    dvClass *fromPtr = toPtr;
    dvModule Module;
    uint32 size;

    while(fromPtr < dvModules.ClassTable + dvUsedModuleClassTable()) {
        Module = *(dvModule *)(void *)fromPtr;
        if(Module != dvModuleNull) {
            /* Need to move it to toPtr */
            size = utMax(dvModuleGetNumClassTable(Module) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvModuleSetClassTableIndex(Module, toPtr - dvModules.ClassTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvModule *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedModuleClassTable(toPtr - dvModules.ClassTable);
    dvSetFreeModuleClassTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Module.ClassTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreModuleClassTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedModuleClassTable() - dvUsedModuleClassTable();

    if((dvFreeModuleClassTable() << 2) > dvUsedModuleClassTable()) {
        dvCompactModuleClassTables();
        freeSpace = dvAllocatedModuleClassTable() - dvUsedModuleClassTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedModuleClassTable(dvAllocatedModuleClassTable() + spaceNeeded - freeSpace +
            (dvAllocatedModuleClassTable() >> 1));
        utResizeArray(dvModules.ClassTable, dvAllocatedModuleClassTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Module.ClassTable array.
----------------------------------------------------------------------------------------*/
void dvModuleAllocClassTables(
    dvModule Module,
    uint32 numClassTables)
{
    uint32 freeSpace = dvAllocatedModuleClassTable() - dvUsedModuleClassTable();
    uint32 elementSize = sizeof(dvClass);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numClassTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvModuleGetNumClassTable(Module) == 0);
#endif
    if(numClassTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreModuleClassTables(spaceNeeded);
    }
    dvModuleSetClassTableIndex(Module, dvUsedModuleClassTable() + usedHeaderSize);
    dvModuleSetNumClassTable(Module, numClassTables);
    *(dvModule *)(void *)(dvModules.ClassTable + dvUsedModuleClassTable()) = Module;
    {
        uint32 xModule;
        for(xModule = (uint32)(dvModuleGetClassTableIndex(Module)); xModule < dvModuleGetClassTableIndex(Module) + numClassTables; xModule++) {
            dvModules.ClassTable[xModule] = dvClassNull;
        }
    }
    dvSetUsedModuleClassTable(dvUsedModuleClassTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvModuleGetClassTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getModuleClassTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvModule Module = dvIndex2Module((uint32)objectNumber);

    *numValues = dvModuleGetNumClassTable(Module);
    return dvModuleGetClassTables(Module);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvModuleAllocClassTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocModuleClassTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvModule Module = dvIndex2Module((uint32)objectNumber);

    dvModuleSetClassTableIndex(Module, 0);
    dvModuleSetNumClassTable(Module, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvModuleAllocClassTables(Module, numValues);
    return dvModuleGetClassTables(Module);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Module.ClassTable array.
----------------------------------------------------------------------------------------*/
void dvModuleFreeClassTables(
    dvModule Module)
{
    uint32 elementSize = sizeof(dvClass);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvModuleGetNumClassTable(Module) + usedHeaderSize, freeHeaderSize);
    dvClass *dataPtr = dvModuleGetClassTables(Module) - usedHeaderSize;

    if(dvModuleGetNumClassTable(Module) == 0) {
        return;
    }
    *(dvModule *)(void *)(dataPtr) = dvModuleNull;
    *(uint32 *)(void *)(((dvModule *)(void *)dataPtr) + 1) = size;
    dvModuleSetNumClassTable(Module, 0);
    dvSetFreeModuleClassTable(dvFreeModuleClassTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Module.ClassTable array.
----------------------------------------------------------------------------------------*/
void dvModuleResizeClassTables(
    dvModule Module,
    uint32 numClassTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvClass);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numClassTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvModuleGetNumClassTable(Module) + usedHeaderSize, freeHeaderSize);
    dvClass *dataPtr;

    if(numClassTables == 0) {
        if(dvModuleGetNumClassTable(Module) != 0) {
            dvModuleFreeClassTables(Module);
        }
        return;
    }
    if(dvModuleGetNumClassTable(Module) == 0) {
        dvModuleAllocClassTables(Module, numClassTables);
        return;
    }
    freeSpace = dvAllocatedModuleClassTable() - dvUsedModuleClassTable();
    if(freeSpace < newSize) {
        allocMoreModuleClassTables(newSize);
    }
    dataPtr = dvModuleGetClassTables(Module) - usedHeaderSize;
    memcpy((void *)(dvModules.ClassTable + dvUsedModuleClassTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xModule;
            for(xModule = (uint32)(dvUsedModuleClassTable() + oldSize); xModule < dvUsedModuleClassTable() + oldSize + newSize - oldSize; xModule++) {
                dvModules.ClassTable[xModule] = dvClassNull;
            }
        }
    }
    *(dvModule *)(void *)dataPtr = dvModuleNull;
    *(uint32 *)(void *)(((dvModule *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeModuleClassTable(dvFreeModuleClassTable() + oldSize);
    dvModuleSetClassTableIndex(Module, dvUsedModuleClassTable() + usedHeaderSize);
    dvModuleSetNumClassTable(Module, numClassTables);
    dvSetUsedModuleClassTable(dvUsedModuleClassTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the Module.EnumTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactModuleEnumTables(void)
{
    uint32 elementSize = sizeof(dvEnum);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvEnum *toPtr = dvModules.EnumTable;
    dvEnum *fromPtr = toPtr;
    dvModule Module;
    uint32 size;

    while(fromPtr < dvModules.EnumTable + dvUsedModuleEnumTable()) {
        Module = *(dvModule *)(void *)fromPtr;
        if(Module != dvModuleNull) {
            /* Need to move it to toPtr */
            size = utMax(dvModuleGetNumEnumTable(Module) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvModuleSetEnumTableIndex(Module, toPtr - dvModules.EnumTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvModule *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedModuleEnumTable(toPtr - dvModules.EnumTable);
    dvSetFreeModuleEnumTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Module.EnumTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreModuleEnumTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedModuleEnumTable() - dvUsedModuleEnumTable();

    if((dvFreeModuleEnumTable() << 2) > dvUsedModuleEnumTable()) {
        dvCompactModuleEnumTables();
        freeSpace = dvAllocatedModuleEnumTable() - dvUsedModuleEnumTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedModuleEnumTable(dvAllocatedModuleEnumTable() + spaceNeeded - freeSpace +
            (dvAllocatedModuleEnumTable() >> 1));
        utResizeArray(dvModules.EnumTable, dvAllocatedModuleEnumTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Module.EnumTable array.
----------------------------------------------------------------------------------------*/
void dvModuleAllocEnumTables(
    dvModule Module,
    uint32 numEnumTables)
{
    uint32 freeSpace = dvAllocatedModuleEnumTable() - dvUsedModuleEnumTable();
    uint32 elementSize = sizeof(dvEnum);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numEnumTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvModuleGetNumEnumTable(Module) == 0);
#endif
    if(numEnumTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreModuleEnumTables(spaceNeeded);
    }
    dvModuleSetEnumTableIndex(Module, dvUsedModuleEnumTable() + usedHeaderSize);
    dvModuleSetNumEnumTable(Module, numEnumTables);
    *(dvModule *)(void *)(dvModules.EnumTable + dvUsedModuleEnumTable()) = Module;
    {
        uint32 xModule;
        for(xModule = (uint32)(dvModuleGetEnumTableIndex(Module)); xModule < dvModuleGetEnumTableIndex(Module) + numEnumTables; xModule++) {
            dvModules.EnumTable[xModule] = dvEnumNull;
        }
    }
    dvSetUsedModuleEnumTable(dvUsedModuleEnumTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvModuleGetEnumTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getModuleEnumTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvModule Module = dvIndex2Module((uint32)objectNumber);

    *numValues = dvModuleGetNumEnumTable(Module);
    return dvModuleGetEnumTables(Module);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvModuleAllocEnumTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocModuleEnumTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvModule Module = dvIndex2Module((uint32)objectNumber);

    dvModuleSetEnumTableIndex(Module, 0);
    dvModuleSetNumEnumTable(Module, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvModuleAllocEnumTables(Module, numValues);
    return dvModuleGetEnumTables(Module);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Module.EnumTable array.
----------------------------------------------------------------------------------------*/
void dvModuleFreeEnumTables(
    dvModule Module)
{
    uint32 elementSize = sizeof(dvEnum);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvModuleGetNumEnumTable(Module) + usedHeaderSize, freeHeaderSize);
    dvEnum *dataPtr = dvModuleGetEnumTables(Module) - usedHeaderSize;

    if(dvModuleGetNumEnumTable(Module) == 0) {
        return;
    }
    *(dvModule *)(void *)(dataPtr) = dvModuleNull;
    *(uint32 *)(void *)(((dvModule *)(void *)dataPtr) + 1) = size;
    dvModuleSetNumEnumTable(Module, 0);
    dvSetFreeModuleEnumTable(dvFreeModuleEnumTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Module.EnumTable array.
----------------------------------------------------------------------------------------*/
void dvModuleResizeEnumTables(
    dvModule Module,
    uint32 numEnumTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvEnum);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numEnumTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvModuleGetNumEnumTable(Module) + usedHeaderSize, freeHeaderSize);
    dvEnum *dataPtr;

    if(numEnumTables == 0) {
        if(dvModuleGetNumEnumTable(Module) != 0) {
            dvModuleFreeEnumTables(Module);
        }
        return;
    }
    if(dvModuleGetNumEnumTable(Module) == 0) {
        dvModuleAllocEnumTables(Module, numEnumTables);
        return;
    }
    freeSpace = dvAllocatedModuleEnumTable() - dvUsedModuleEnumTable();
    if(freeSpace < newSize) {
        allocMoreModuleEnumTables(newSize);
    }
    dataPtr = dvModuleGetEnumTables(Module) - usedHeaderSize;
    memcpy((void *)(dvModules.EnumTable + dvUsedModuleEnumTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xModule;
            for(xModule = (uint32)(dvUsedModuleEnumTable() + oldSize); xModule < dvUsedModuleEnumTable() + oldSize + newSize - oldSize; xModule++) {
                dvModules.EnumTable[xModule] = dvEnumNull;
            }
        }
    }
    *(dvModule *)(void *)dataPtr = dvModuleNull;
    *(uint32 *)(void *)(((dvModule *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeModuleEnumTable(dvFreeModuleEnumTable() + oldSize);
    dvModuleSetEnumTableIndex(Module, dvUsedModuleEnumTable() + usedHeaderSize);
    dvModuleSetNumEnumTable(Module, numEnumTables);
    dvSetUsedModuleEnumTable(dvUsedModuleEnumTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the Module.TypedefTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactModuleTypedefTables(void)
{
    uint32 elementSize = sizeof(dvTypedef);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvTypedef *toPtr = dvModules.TypedefTable;
    dvTypedef *fromPtr = toPtr;
    dvModule Module;
    uint32 size;

    while(fromPtr < dvModules.TypedefTable + dvUsedModuleTypedefTable()) {
        Module = *(dvModule *)(void *)fromPtr;
        if(Module != dvModuleNull) {
            /* Need to move it to toPtr */
            size = utMax(dvModuleGetNumTypedefTable(Module) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvModuleSetTypedefTableIndex(Module, toPtr - dvModules.TypedefTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvModule *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedModuleTypedefTable(toPtr - dvModules.TypedefTable);
    dvSetFreeModuleTypedefTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Module.TypedefTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreModuleTypedefTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedModuleTypedefTable() - dvUsedModuleTypedefTable();

    if((dvFreeModuleTypedefTable() << 2) > dvUsedModuleTypedefTable()) {
        dvCompactModuleTypedefTables();
        freeSpace = dvAllocatedModuleTypedefTable() - dvUsedModuleTypedefTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedModuleTypedefTable(dvAllocatedModuleTypedefTable() + spaceNeeded - freeSpace +
            (dvAllocatedModuleTypedefTable() >> 1));
        utResizeArray(dvModules.TypedefTable, dvAllocatedModuleTypedefTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Module.TypedefTable array.
----------------------------------------------------------------------------------------*/
void dvModuleAllocTypedefTables(
    dvModule Module,
    uint32 numTypedefTables)
{
    uint32 freeSpace = dvAllocatedModuleTypedefTable() - dvUsedModuleTypedefTable();
    uint32 elementSize = sizeof(dvTypedef);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numTypedefTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvModuleGetNumTypedefTable(Module) == 0);
#endif
    if(numTypedefTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreModuleTypedefTables(spaceNeeded);
    }
    dvModuleSetTypedefTableIndex(Module, dvUsedModuleTypedefTable() + usedHeaderSize);
    dvModuleSetNumTypedefTable(Module, numTypedefTables);
    *(dvModule *)(void *)(dvModules.TypedefTable + dvUsedModuleTypedefTable()) = Module;
    {
        uint32 xModule;
        for(xModule = (uint32)(dvModuleGetTypedefTableIndex(Module)); xModule < dvModuleGetTypedefTableIndex(Module) + numTypedefTables; xModule++) {
            dvModules.TypedefTable[xModule] = dvTypedefNull;
        }
    }
    dvSetUsedModuleTypedefTable(dvUsedModuleTypedefTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvModuleGetTypedefTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getModuleTypedefTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvModule Module = dvIndex2Module((uint32)objectNumber);

    *numValues = dvModuleGetNumTypedefTable(Module);
    return dvModuleGetTypedefTables(Module);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvModuleAllocTypedefTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocModuleTypedefTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvModule Module = dvIndex2Module((uint32)objectNumber);

    dvModuleSetTypedefTableIndex(Module, 0);
    dvModuleSetNumTypedefTable(Module, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvModuleAllocTypedefTables(Module, numValues);
    return dvModuleGetTypedefTables(Module);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Module.TypedefTable array.
----------------------------------------------------------------------------------------*/
void dvModuleFreeTypedefTables(
    dvModule Module)
{
    uint32 elementSize = sizeof(dvTypedef);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvModuleGetNumTypedefTable(Module) + usedHeaderSize, freeHeaderSize);
    dvTypedef *dataPtr = dvModuleGetTypedefTables(Module) - usedHeaderSize;

    if(dvModuleGetNumTypedefTable(Module) == 0) {
        return;
    }
    *(dvModule *)(void *)(dataPtr) = dvModuleNull;
    *(uint32 *)(void *)(((dvModule *)(void *)dataPtr) + 1) = size;
    dvModuleSetNumTypedefTable(Module, 0);
    dvSetFreeModuleTypedefTable(dvFreeModuleTypedefTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Module.TypedefTable array.
----------------------------------------------------------------------------------------*/
void dvModuleResizeTypedefTables(
    dvModule Module,
    uint32 numTypedefTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvTypedef);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numTypedefTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvModuleGetNumTypedefTable(Module) + usedHeaderSize, freeHeaderSize);
    dvTypedef *dataPtr;

    if(numTypedefTables == 0) {
        if(dvModuleGetNumTypedefTable(Module) != 0) {
            dvModuleFreeTypedefTables(Module);
        }
        return;
    }
    if(dvModuleGetNumTypedefTable(Module) == 0) {
        dvModuleAllocTypedefTables(Module, numTypedefTables);
        return;
    }
    freeSpace = dvAllocatedModuleTypedefTable() - dvUsedModuleTypedefTable();
    if(freeSpace < newSize) {
        allocMoreModuleTypedefTables(newSize);
    }
    dataPtr = dvModuleGetTypedefTables(Module) - usedHeaderSize;
    memcpy((void *)(dvModules.TypedefTable + dvUsedModuleTypedefTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xModule;
            for(xModule = (uint32)(dvUsedModuleTypedefTable() + oldSize); xModule < dvUsedModuleTypedefTable() + oldSize + newSize - oldSize; xModule++) {
                dvModules.TypedefTable[xModule] = dvTypedefNull;
            }
        }
    }
    *(dvModule *)(void *)dataPtr = dvModuleNull;
    *(uint32 *)(void *)(((dvModule *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeModuleTypedefTable(dvFreeModuleTypedefTable() + oldSize);
    dvModuleSetTypedefTableIndex(Module, dvUsedModuleTypedefTable() + usedHeaderSize);
    dvModuleSetNumTypedefTable(Module, numTypedefTables);
    dvSetUsedModuleTypedefTable(dvUsedModuleTypedefTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the Module.SchemaTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactModuleSchemaTables(void)
{
    uint32 elementSize = sizeof(dvSchema);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvSchema *toPtr = dvModules.SchemaTable;
    dvSchema *fromPtr = toPtr;
    dvModule Module;
    uint32 size;

    while(fromPtr < dvModules.SchemaTable + dvUsedModuleSchemaTable()) {
        Module = *(dvModule *)(void *)fromPtr;
        if(Module != dvModuleNull) {
            /* Need to move it to toPtr */
            size = utMax(dvModuleGetNumSchemaTable(Module) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvModuleSetSchemaTableIndex(Module, toPtr - dvModules.SchemaTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvModule *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedModuleSchemaTable(toPtr - dvModules.SchemaTable);
    dvSetFreeModuleSchemaTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Module.SchemaTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreModuleSchemaTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedModuleSchemaTable() - dvUsedModuleSchemaTable();

    if((dvFreeModuleSchemaTable() << 2) > dvUsedModuleSchemaTable()) {
        dvCompactModuleSchemaTables();
        freeSpace = dvAllocatedModuleSchemaTable() - dvUsedModuleSchemaTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedModuleSchemaTable(dvAllocatedModuleSchemaTable() + spaceNeeded - freeSpace +
            (dvAllocatedModuleSchemaTable() >> 1));
        utResizeArray(dvModules.SchemaTable, dvAllocatedModuleSchemaTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Module.SchemaTable array.
----------------------------------------------------------------------------------------*/
void dvModuleAllocSchemaTables(
    dvModule Module,
    uint32 numSchemaTables)
{
    uint32 freeSpace = dvAllocatedModuleSchemaTable() - dvUsedModuleSchemaTable();
    uint32 elementSize = sizeof(dvSchema);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numSchemaTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvModuleGetNumSchemaTable(Module) == 0);
#endif
    if(numSchemaTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreModuleSchemaTables(spaceNeeded);
    }
    dvModuleSetSchemaTableIndex(Module, dvUsedModuleSchemaTable() + usedHeaderSize);
    dvModuleSetNumSchemaTable(Module, numSchemaTables);
    *(dvModule *)(void *)(dvModules.SchemaTable + dvUsedModuleSchemaTable()) = Module;
    {
        uint32 xModule;
        for(xModule = (uint32)(dvModuleGetSchemaTableIndex(Module)); xModule < dvModuleGetSchemaTableIndex(Module) + numSchemaTables; xModule++) {
            dvModules.SchemaTable[xModule] = dvSchemaNull;
        }
    }
    dvSetUsedModuleSchemaTable(dvUsedModuleSchemaTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvModuleGetSchemaTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getModuleSchemaTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvModule Module = dvIndex2Module((uint32)objectNumber);

    *numValues = dvModuleGetNumSchemaTable(Module);
    return dvModuleGetSchemaTables(Module);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvModuleAllocSchemaTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocModuleSchemaTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvModule Module = dvIndex2Module((uint32)objectNumber);

    dvModuleSetSchemaTableIndex(Module, 0);
    dvModuleSetNumSchemaTable(Module, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvModuleAllocSchemaTables(Module, numValues);
    return dvModuleGetSchemaTables(Module);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Module.SchemaTable array.
----------------------------------------------------------------------------------------*/
void dvModuleFreeSchemaTables(
    dvModule Module)
{
    uint32 elementSize = sizeof(dvSchema);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvModuleGetNumSchemaTable(Module) + usedHeaderSize, freeHeaderSize);
    dvSchema *dataPtr = dvModuleGetSchemaTables(Module) - usedHeaderSize;

    if(dvModuleGetNumSchemaTable(Module) == 0) {
        return;
    }
    *(dvModule *)(void *)(dataPtr) = dvModuleNull;
    *(uint32 *)(void *)(((dvModule *)(void *)dataPtr) + 1) = size;
    dvModuleSetNumSchemaTable(Module, 0);
    dvSetFreeModuleSchemaTable(dvFreeModuleSchemaTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Module.SchemaTable array.
----------------------------------------------------------------------------------------*/
void dvModuleResizeSchemaTables(
    dvModule Module,
    uint32 numSchemaTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvSchema);
    uint32 usedHeaderSize = (sizeof(dvModule) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvModule) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numSchemaTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvModuleGetNumSchemaTable(Module) + usedHeaderSize, freeHeaderSize);
    dvSchema *dataPtr;

    if(numSchemaTables == 0) {
        if(dvModuleGetNumSchemaTable(Module) != 0) {
            dvModuleFreeSchemaTables(Module);
        }
        return;
    }
    if(dvModuleGetNumSchemaTable(Module) == 0) {
        dvModuleAllocSchemaTables(Module, numSchemaTables);
        return;
    }
    freeSpace = dvAllocatedModuleSchemaTable() - dvUsedModuleSchemaTable();
    if(freeSpace < newSize) {
        allocMoreModuleSchemaTables(newSize);
    }
    dataPtr = dvModuleGetSchemaTables(Module) - usedHeaderSize;
    memcpy((void *)(dvModules.SchemaTable + dvUsedModuleSchemaTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xModule;
            for(xModule = (uint32)(dvUsedModuleSchemaTable() + oldSize); xModule < dvUsedModuleSchemaTable() + oldSize + newSize - oldSize; xModule++) {
                dvModules.SchemaTable[xModule] = dvSchemaNull;
            }
        }
    }
    *(dvModule *)(void *)dataPtr = dvModuleNull;
    *(uint32 *)(void *)(((dvModule *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeModuleSchemaTable(dvFreeModuleSchemaTable() + oldSize);
    dvModuleSetSchemaTableIndex(Module, dvUsedModuleSchemaTable() + usedHeaderSize);
    dvModuleSetNumSchemaTable(Module, numSchemaTables);
    dvSetUsedModuleSchemaTable(dvUsedModuleSchemaTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Module.
----------------------------------------------------------------------------------------*/
void dvModuleCopyProps(
    dvModule oldModule,
    dvModule newModule)
{
    dvModuleSetPrefixSym(newModule, dvModuleGetPrefixSym(oldModule));
    dvModuleSetPersistent(newModule, dvModulePersistent(oldModule));
    dvModuleSetUndoRedo(newModule, dvModuleUndoRedo(oldModule));
    dvModuleSetHasSparseData(newModule, dvModuleHasSparseData(oldModule));
    dvModuleSetNumFields(newModule, dvModuleGetNumFields(oldModule));
    dvModuleSetNumClasses(newModule, dvModuleGetNumClasses(oldModule));
    dvModuleSetNumEnums(newModule, dvModuleGetNumEnums(oldModule));
}

/*----------------------------------------------------------------------------------------
  Return the integer equivalent for the bit fields in Module.
----------------------------------------------------------------------------------------*/
uint32 dvModuleGetBitfield(
    dvModule _Module)
{
    uint32 bitfield = 0;
    uint8 xLevel = 0;

    bitfield |= dvModulePersistent(_Module) << xLevel++;
    bitfield |= dvModuleUndoRedo(_Module) << xLevel++;
    bitfield |= dvModuleHasSparseData(_Module) << xLevel++;
    return bitfield;
}

/*----------------------------------------------------------------------------------------
  Set bit fields in Module using bitfield.
----------------------------------------------------------------------------------------*/
void dvModuleSetBitfield(
    dvModule _Module,
     uint32 bitfield)
{
    dvModuleSetPersistent(_Module, bitfield & 1);
    bitfield >>= 1;
    dvModuleSetUndoRedo(_Module, bitfield & 1);
    bitfield >>= 1;
    dvModuleSetHasSparseData(_Module, bitfield & 1);
    bitfield >>= 1;
}

static void addModuleClassToHashTable(dvModule Module, dvClass _Class);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeModuleClassHashTable(
    dvModule Module)
{
    dvClass _Class;
    dvClass *Classs;
    uint32 numClasss = dvModuleGetNumClassTable(Module) << 1;

    if(numClasss == 0) {
        numClasss = 2;
        dvModuleAllocClassTables(Module, 2);
    } else {
        dvModuleResizeClassTables(Module, numClasss);
    }
    Classs = dvModuleGetClassTables(Module);
    /* Zero out the table */
    while(numClasss-- != 0) {
        *Classs++ = dvClassNull;
    }
    dvModuleSetNumClass(Module, 0);
    dvForeachModuleClass(Module, _Class) {
        if(dvClassGetSym(_Class) != utSymNull) {
            addModuleClassToHashTable(Module, _Class);
        }
    } dvEndModuleClass;
}

/*----------------------------------------------------------------------------------------
  Add the Class to the Module.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addModuleClassToHashTable(
    dvModule Module,
    dvClass _Class)
{
    dvClass nextClass;
    uint32 index;

    if(dvModuleGetNumClass(Module) >= dvModuleGetNumClassTable(Module)) {
        resizeModuleClassHashTable(Module);
        return;
    }
    index = (dvModuleGetNumClassTable(Module) - 1) & utSymGetHashValue(dvClassGetSym(_Class));
    nextClass = dvModuleGetiClassTable(Module, index);
    dvClassSetNextTableModuleClass(_Class, nextClass);
    dvModuleSetiClassTable(Module, index, _Class);
    dvModuleSetNumClass(Module, dvModuleGetNumClass(Module) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Class from the hash table.
----------------------------------------------------------------------------------------*/
static void removeModuleClassFromHashTable(
   dvModule Module,
   dvClass _Class)
{
    uint32 index = (dvModuleGetNumClassTable(Module) - 1) & utSymGetHashValue(dvClassGetSym(_Class));
    dvClass prevClass, nextClass;
    
    nextClass = dvModuleGetiClassTable(Module, index);
    if(nextClass == _Class) {
        dvModuleSetiClassTable(Module, index, dvClassGetNextTableModuleClass(nextClass));
    } else {
        do {
            prevClass = nextClass;
            nextClass = dvClassGetNextTableModuleClass(nextClass);
        } while(nextClass != _Class);
        dvClassSetNextTableModuleClass(prevClass, dvClassGetNextTableModuleClass(_Class));
    }
    dvModuleSetNumClass(Module, dvModuleGetNumClass(Module) - 1);
    dvClassSetNextTableModuleClass(_Class, dvClassNull);
}

/*----------------------------------------------------------------------------------------
  Find the Class from the Module and its hash key.
----------------------------------------------------------------------------------------*/
dvClass dvModuleFindClass(
    dvModule Module,
    utSym Sym)
{
    uint32 mask = dvModuleGetNumClassTable(Module) - 1;
    dvClass _Class;

    if(mask + 1 != 0) {
        _Class = dvModuleGetiClassTable(Module, utSymGetHashValue(Sym) & mask);
        while(_Class != dvClassNull) {
            if(dvClassGetSym(_Class) == Sym) {
                return _Class;
            }
            _Class = dvClassGetNextTableModuleClass(_Class);
        }
    }
    return dvClassNull;
}

/*----------------------------------------------------------------------------------------
  Find the Class from the Module and its name.
----------------------------------------------------------------------------------------*/
void dvModuleRenameClass(
    dvModule Module,
    dvClass _Class,
    utSym sym)
{
    if(dvClassGetSym(_Class) != utSymNull) {
        removeModuleClassFromHashTable(Module, _Class);
    }
    dvClassSetSym(_Class, sym);
    if(sym != utSymNull) {
        addModuleClassToHashTable(Module, _Class);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Class to the head of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleInsertClass(
    dvModule Module,
    dvClass _Class)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(dvClassGetModule(_Class) != dvModuleNull) {
        utExit("Attempting to add Class to Module twice");
    }
#endif
    dvClassSetNextModuleClass(_Class, dvModuleGetFirstClass(Module));
    if(dvModuleGetFirstClass(Module) != dvClassNull) {
        dvClassSetPrevModuleClass(dvModuleGetFirstClass(Module), _Class);
    }
    dvModuleSetFirstClass(Module, _Class);
    dvClassSetPrevModuleClass(_Class, dvClassNull);
    if(dvModuleGetLastClass(Module) == dvClassNull) {
        dvModuleSetLastClass(Module, _Class);
    }
    dvClassSetModule(_Class, Module);
    if(dvClassGetSym(_Class) != utSymNull) {
        addModuleClassToHashTable(Module, _Class);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Class to the end of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleAppendClass(
    dvModule Module,
    dvClass _Class)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(dvClassGetModule(_Class) != dvModuleNull) {
        utExit("Attempting to add Class to Module twice");
    }
#endif
    dvClassSetPrevModuleClass(_Class, dvModuleGetLastClass(Module));
    if(dvModuleGetLastClass(Module) != dvClassNull) {
        dvClassSetNextModuleClass(dvModuleGetLastClass(Module), _Class);
    }
    dvModuleSetLastClass(Module, _Class);
    dvClassSetNextModuleClass(_Class, dvClassNull);
    if(dvModuleGetFirstClass(Module) == dvClassNull) {
        dvModuleSetFirstClass(Module, _Class);
    }
    dvClassSetModule(_Class, Module);
    if(dvClassGetSym(_Class) != utSymNull) {
        addModuleClassToHashTable(Module, _Class);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Class to the Module after the previous Class.
----------------------------------------------------------------------------------------*/
void dvModuleInsertAfterClass(
    dvModule Module,
    dvClass prevClass,
    dvClass _Class)
{
    dvClass nextClass = dvClassGetNextModuleClass(prevClass);

#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(dvClassGetModule(_Class) != dvModuleNull) {
        utExit("Attempting to add Class to Module twice");
    }
#endif
    dvClassSetNextModuleClass(_Class, nextClass);
    dvClassSetNextModuleClass(prevClass, _Class);
    dvClassSetPrevModuleClass(_Class, prevClass);
    if(nextClass != dvClassNull) {
        dvClassSetPrevModuleClass(nextClass, _Class);
    }
    if(dvModuleGetLastClass(Module) == prevClass) {
        dvModuleSetLastClass(Module, _Class);
    }
    dvClassSetModule(_Class, Module);
    if(dvClassGetSym(_Class) != utSymNull) {
        addModuleClassToHashTable(Module, _Class);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Class from the Module.
----------------------------------------------------------------------------------------*/
void dvModuleRemoveClass(
    dvModule Module,
    dvClass _Class)
{
    dvClass pClass, nClass;

#if defined(DD_DEBUG)
    if(_Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(dvClassGetModule(_Class) != dvModuleNull && dvClassGetModule(_Class) != Module) {
        utExit("Delete Class from non-owning Module");
    }
#endif
    nClass = dvClassGetNextModuleClass(_Class);
    pClass = dvClassGetPrevModuleClass(_Class);
    if(pClass != dvClassNull) {
        dvClassSetNextModuleClass(pClass, nClass);
    } else if(dvModuleGetFirstClass(Module) == _Class) {
        dvModuleSetFirstClass(Module, nClass);
    }
    if(nClass != dvClassNull) {
        dvClassSetPrevModuleClass(nClass, pClass);
    } else if(dvModuleGetLastClass(Module) == _Class) {
        dvModuleSetLastClass(Module, pClass);
    }
    dvClassSetNextModuleClass(_Class, dvClassNull);
    dvClassSetPrevModuleClass(_Class, dvClassNull);
    dvClassSetModule(_Class, dvModuleNull);
    if(dvClassGetSym(_Class) != utSymNull) {
        removeModuleClassFromHashTable(Module, _Class);
    }
}

static void addModuleEnumToHashTable(dvModule Module, dvEnum _Enum);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeModuleEnumHashTable(
    dvModule Module)
{
    dvEnum _Enum;
    dvEnum *Enums;
    uint32 numEnums = dvModuleGetNumEnumTable(Module) << 1;

    if(numEnums == 0) {
        numEnums = 2;
        dvModuleAllocEnumTables(Module, 2);
    } else {
        dvModuleResizeEnumTables(Module, numEnums);
    }
    Enums = dvModuleGetEnumTables(Module);
    /* Zero out the table */
    while(numEnums-- != 0) {
        *Enums++ = dvEnumNull;
    }
    dvModuleSetNumEnum(Module, 0);
    dvForeachModuleEnum(Module, _Enum) {
        if(dvEnumGetSym(_Enum) != utSymNull) {
            addModuleEnumToHashTable(Module, _Enum);
        }
    } dvEndModuleEnum;
}

/*----------------------------------------------------------------------------------------
  Add the Enum to the Module.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addModuleEnumToHashTable(
    dvModule Module,
    dvEnum _Enum)
{
    dvEnum nextEnum;
    uint32 index;

    if(dvModuleGetNumEnum(Module) >= dvModuleGetNumEnumTable(Module)) {
        resizeModuleEnumHashTable(Module);
        return;
    }
    index = (dvModuleGetNumEnumTable(Module) - 1) & utSymGetHashValue(dvEnumGetSym(_Enum));
    nextEnum = dvModuleGetiEnumTable(Module, index);
    dvEnumSetNextTableModuleEnum(_Enum, nextEnum);
    dvModuleSetiEnumTable(Module, index, _Enum);
    dvModuleSetNumEnum(Module, dvModuleGetNumEnum(Module) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Enum from the hash table.
----------------------------------------------------------------------------------------*/
static void removeModuleEnumFromHashTable(
   dvModule Module,
   dvEnum _Enum)
{
    uint32 index = (dvModuleGetNumEnumTable(Module) - 1) & utSymGetHashValue(dvEnumGetSym(_Enum));
    dvEnum prevEnum, nextEnum;
    
    nextEnum = dvModuleGetiEnumTable(Module, index);
    if(nextEnum == _Enum) {
        dvModuleSetiEnumTable(Module, index, dvEnumGetNextTableModuleEnum(nextEnum));
    } else {
        do {
            prevEnum = nextEnum;
            nextEnum = dvEnumGetNextTableModuleEnum(nextEnum);
        } while(nextEnum != _Enum);
        dvEnumSetNextTableModuleEnum(prevEnum, dvEnumGetNextTableModuleEnum(_Enum));
    }
    dvModuleSetNumEnum(Module, dvModuleGetNumEnum(Module) - 1);
    dvEnumSetNextTableModuleEnum(_Enum, dvEnumNull);
}

/*----------------------------------------------------------------------------------------
  Find the Enum from the Module and its hash key.
----------------------------------------------------------------------------------------*/
dvEnum dvModuleFindEnum(
    dvModule Module,
    utSym Sym)
{
    uint32 mask = dvModuleGetNumEnumTable(Module) - 1;
    dvEnum _Enum;

    if(mask + 1 != 0) {
        _Enum = dvModuleGetiEnumTable(Module, utSymGetHashValue(Sym) & mask);
        while(_Enum != dvEnumNull) {
            if(dvEnumGetSym(_Enum) == Sym) {
                return _Enum;
            }
            _Enum = dvEnumGetNextTableModuleEnum(_Enum);
        }
    }
    return dvEnumNull;
}

/*----------------------------------------------------------------------------------------
  Find the Enum from the Module and its name.
----------------------------------------------------------------------------------------*/
void dvModuleRenameEnum(
    dvModule Module,
    dvEnum _Enum,
    utSym sym)
{
    if(dvEnumGetSym(_Enum) != utSymNull) {
        removeModuleEnumFromHashTable(Module, _Enum);
    }
    dvEnumSetSym(_Enum, sym);
    if(sym != utSymNull) {
        addModuleEnumToHashTable(Module, _Enum);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Enum to the head of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleInsertEnum(
    dvModule Module,
    dvEnum _Enum)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Enum == dvEnumNull) {
        utExit("Non-existent Enum");
    }
    if(dvEnumGetModule(_Enum) != dvModuleNull) {
        utExit("Attempting to add Enum to Module twice");
    }
#endif
    dvEnumSetNextModuleEnum(_Enum, dvModuleGetFirstEnum(Module));
    if(dvModuleGetFirstEnum(Module) != dvEnumNull) {
        dvEnumSetPrevModuleEnum(dvModuleGetFirstEnum(Module), _Enum);
    }
    dvModuleSetFirstEnum(Module, _Enum);
    dvEnumSetPrevModuleEnum(_Enum, dvEnumNull);
    if(dvModuleGetLastEnum(Module) == dvEnumNull) {
        dvModuleSetLastEnum(Module, _Enum);
    }
    dvEnumSetModule(_Enum, Module);
    if(dvEnumGetSym(_Enum) != utSymNull) {
        addModuleEnumToHashTable(Module, _Enum);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Enum to the end of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleAppendEnum(
    dvModule Module,
    dvEnum _Enum)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Enum == dvEnumNull) {
        utExit("Non-existent Enum");
    }
    if(dvEnumGetModule(_Enum) != dvModuleNull) {
        utExit("Attempting to add Enum to Module twice");
    }
#endif
    dvEnumSetPrevModuleEnum(_Enum, dvModuleGetLastEnum(Module));
    if(dvModuleGetLastEnum(Module) != dvEnumNull) {
        dvEnumSetNextModuleEnum(dvModuleGetLastEnum(Module), _Enum);
    }
    dvModuleSetLastEnum(Module, _Enum);
    dvEnumSetNextModuleEnum(_Enum, dvEnumNull);
    if(dvModuleGetFirstEnum(Module) == dvEnumNull) {
        dvModuleSetFirstEnum(Module, _Enum);
    }
    dvEnumSetModule(_Enum, Module);
    if(dvEnumGetSym(_Enum) != utSymNull) {
        addModuleEnumToHashTable(Module, _Enum);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Enum to the Module after the previous Enum.
----------------------------------------------------------------------------------------*/
void dvModuleInsertAfterEnum(
    dvModule Module,
    dvEnum prevEnum,
    dvEnum _Enum)
{
    dvEnum nextEnum = dvEnumGetNextModuleEnum(prevEnum);

#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Enum == dvEnumNull) {
        utExit("Non-existent Enum");
    }
    if(dvEnumGetModule(_Enum) != dvModuleNull) {
        utExit("Attempting to add Enum to Module twice");
    }
#endif
    dvEnumSetNextModuleEnum(_Enum, nextEnum);
    dvEnumSetNextModuleEnum(prevEnum, _Enum);
    dvEnumSetPrevModuleEnum(_Enum, prevEnum);
    if(nextEnum != dvEnumNull) {
        dvEnumSetPrevModuleEnum(nextEnum, _Enum);
    }
    if(dvModuleGetLastEnum(Module) == prevEnum) {
        dvModuleSetLastEnum(Module, _Enum);
    }
    dvEnumSetModule(_Enum, Module);
    if(dvEnumGetSym(_Enum) != utSymNull) {
        addModuleEnumToHashTable(Module, _Enum);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Enum from the Module.
----------------------------------------------------------------------------------------*/
void dvModuleRemoveEnum(
    dvModule Module,
    dvEnum _Enum)
{
    dvEnum pEnum, nEnum;

#if defined(DD_DEBUG)
    if(_Enum == dvEnumNull) {
        utExit("Non-existent Enum");
    }
    if(dvEnumGetModule(_Enum) != dvModuleNull && dvEnumGetModule(_Enum) != Module) {
        utExit("Delete Enum from non-owning Module");
    }
#endif
    nEnum = dvEnumGetNextModuleEnum(_Enum);
    pEnum = dvEnumGetPrevModuleEnum(_Enum);
    if(pEnum != dvEnumNull) {
        dvEnumSetNextModuleEnum(pEnum, nEnum);
    } else if(dvModuleGetFirstEnum(Module) == _Enum) {
        dvModuleSetFirstEnum(Module, nEnum);
    }
    if(nEnum != dvEnumNull) {
        dvEnumSetPrevModuleEnum(nEnum, pEnum);
    } else if(dvModuleGetLastEnum(Module) == _Enum) {
        dvModuleSetLastEnum(Module, pEnum);
    }
    dvEnumSetNextModuleEnum(_Enum, dvEnumNull);
    dvEnumSetPrevModuleEnum(_Enum, dvEnumNull);
    dvEnumSetModule(_Enum, dvModuleNull);
    if(dvEnumGetSym(_Enum) != utSymNull) {
        removeModuleEnumFromHashTable(Module, _Enum);
    }
}

static void addModuleTypedefToHashTable(dvModule Module, dvTypedef _Typedef);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeModuleTypedefHashTable(
    dvModule Module)
{
    dvTypedef _Typedef;
    dvTypedef *Typedefs;
    uint32 numTypedefs = dvModuleGetNumTypedefTable(Module) << 1;

    if(numTypedefs == 0) {
        numTypedefs = 2;
        dvModuleAllocTypedefTables(Module, 2);
    } else {
        dvModuleResizeTypedefTables(Module, numTypedefs);
    }
    Typedefs = dvModuleGetTypedefTables(Module);
    /* Zero out the table */
    while(numTypedefs-- != 0) {
        *Typedefs++ = dvTypedefNull;
    }
    dvModuleSetNumTypedef(Module, 0);
    dvForeachModuleTypedef(Module, _Typedef) {
        if(dvTypedefGetSym(_Typedef) != utSymNull) {
            addModuleTypedefToHashTable(Module, _Typedef);
        }
    } dvEndModuleTypedef;
}

/*----------------------------------------------------------------------------------------
  Add the Typedef to the Module.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addModuleTypedefToHashTable(
    dvModule Module,
    dvTypedef _Typedef)
{
    dvTypedef nextTypedef;
    uint32 index;

    if(dvModuleGetNumTypedef(Module) >= dvModuleGetNumTypedefTable(Module)) {
        resizeModuleTypedefHashTable(Module);
        return;
    }
    index = (dvModuleGetNumTypedefTable(Module) - 1) & utSymGetHashValue(dvTypedefGetSym(_Typedef));
    nextTypedef = dvModuleGetiTypedefTable(Module, index);
    dvTypedefSetNextTableModuleTypedef(_Typedef, nextTypedef);
    dvModuleSetiTypedefTable(Module, index, _Typedef);
    dvModuleSetNumTypedef(Module, dvModuleGetNumTypedef(Module) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Typedef from the hash table.
----------------------------------------------------------------------------------------*/
static void removeModuleTypedefFromHashTable(
   dvModule Module,
   dvTypedef _Typedef)
{
    uint32 index = (dvModuleGetNumTypedefTable(Module) - 1) & utSymGetHashValue(dvTypedefGetSym(_Typedef));
    dvTypedef prevTypedef, nextTypedef;
    
    nextTypedef = dvModuleGetiTypedefTable(Module, index);
    if(nextTypedef == _Typedef) {
        dvModuleSetiTypedefTable(Module, index, dvTypedefGetNextTableModuleTypedef(nextTypedef));
    } else {
        do {
            prevTypedef = nextTypedef;
            nextTypedef = dvTypedefGetNextTableModuleTypedef(nextTypedef);
        } while(nextTypedef != _Typedef);
        dvTypedefSetNextTableModuleTypedef(prevTypedef, dvTypedefGetNextTableModuleTypedef(_Typedef));
    }
    dvModuleSetNumTypedef(Module, dvModuleGetNumTypedef(Module) - 1);
    dvTypedefSetNextTableModuleTypedef(_Typedef, dvTypedefNull);
}

/*----------------------------------------------------------------------------------------
  Find the Typedef from the Module and its hash key.
----------------------------------------------------------------------------------------*/
dvTypedef dvModuleFindTypedef(
    dvModule Module,
    utSym Sym)
{
    uint32 mask = dvModuleGetNumTypedefTable(Module) - 1;
    dvTypedef _Typedef;

    if(mask + 1 != 0) {
        _Typedef = dvModuleGetiTypedefTable(Module, utSymGetHashValue(Sym) & mask);
        while(_Typedef != dvTypedefNull) {
            if(dvTypedefGetSym(_Typedef) == Sym) {
                return _Typedef;
            }
            _Typedef = dvTypedefGetNextTableModuleTypedef(_Typedef);
        }
    }
    return dvTypedefNull;
}

/*----------------------------------------------------------------------------------------
  Find the Typedef from the Module and its name.
----------------------------------------------------------------------------------------*/
void dvModuleRenameTypedef(
    dvModule Module,
    dvTypedef _Typedef,
    utSym sym)
{
    if(dvTypedefGetSym(_Typedef) != utSymNull) {
        removeModuleTypedefFromHashTable(Module, _Typedef);
    }
    dvTypedefSetSym(_Typedef, sym);
    if(sym != utSymNull) {
        addModuleTypedefToHashTable(Module, _Typedef);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Typedef to the head of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleInsertTypedef(
    dvModule Module,
    dvTypedef _Typedef)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Typedef == dvTypedefNull) {
        utExit("Non-existent Typedef");
    }
    if(dvTypedefGetModule(_Typedef) != dvModuleNull) {
        utExit("Attempting to add Typedef to Module twice");
    }
#endif
    dvTypedefSetNextModuleTypedef(_Typedef, dvModuleGetFirstTypedef(Module));
    if(dvModuleGetFirstTypedef(Module) != dvTypedefNull) {
        dvTypedefSetPrevModuleTypedef(dvModuleGetFirstTypedef(Module), _Typedef);
    }
    dvModuleSetFirstTypedef(Module, _Typedef);
    dvTypedefSetPrevModuleTypedef(_Typedef, dvTypedefNull);
    if(dvModuleGetLastTypedef(Module) == dvTypedefNull) {
        dvModuleSetLastTypedef(Module, _Typedef);
    }
    dvTypedefSetModule(_Typedef, Module);
    if(dvTypedefGetSym(_Typedef) != utSymNull) {
        addModuleTypedefToHashTable(Module, _Typedef);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Typedef to the end of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleAppendTypedef(
    dvModule Module,
    dvTypedef _Typedef)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Typedef == dvTypedefNull) {
        utExit("Non-existent Typedef");
    }
    if(dvTypedefGetModule(_Typedef) != dvModuleNull) {
        utExit("Attempting to add Typedef to Module twice");
    }
#endif
    dvTypedefSetPrevModuleTypedef(_Typedef, dvModuleGetLastTypedef(Module));
    if(dvModuleGetLastTypedef(Module) != dvTypedefNull) {
        dvTypedefSetNextModuleTypedef(dvModuleGetLastTypedef(Module), _Typedef);
    }
    dvModuleSetLastTypedef(Module, _Typedef);
    dvTypedefSetNextModuleTypedef(_Typedef, dvTypedefNull);
    if(dvModuleGetFirstTypedef(Module) == dvTypedefNull) {
        dvModuleSetFirstTypedef(Module, _Typedef);
    }
    dvTypedefSetModule(_Typedef, Module);
    if(dvTypedefGetSym(_Typedef) != utSymNull) {
        addModuleTypedefToHashTable(Module, _Typedef);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Typedef to the Module after the previous Typedef.
----------------------------------------------------------------------------------------*/
void dvModuleInsertAfterTypedef(
    dvModule Module,
    dvTypedef prevTypedef,
    dvTypedef _Typedef)
{
    dvTypedef nextTypedef = dvTypedefGetNextModuleTypedef(prevTypedef);

#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Typedef == dvTypedefNull) {
        utExit("Non-existent Typedef");
    }
    if(dvTypedefGetModule(_Typedef) != dvModuleNull) {
        utExit("Attempting to add Typedef to Module twice");
    }
#endif
    dvTypedefSetNextModuleTypedef(_Typedef, nextTypedef);
    dvTypedefSetNextModuleTypedef(prevTypedef, _Typedef);
    dvTypedefSetPrevModuleTypedef(_Typedef, prevTypedef);
    if(nextTypedef != dvTypedefNull) {
        dvTypedefSetPrevModuleTypedef(nextTypedef, _Typedef);
    }
    if(dvModuleGetLastTypedef(Module) == prevTypedef) {
        dvModuleSetLastTypedef(Module, _Typedef);
    }
    dvTypedefSetModule(_Typedef, Module);
    if(dvTypedefGetSym(_Typedef) != utSymNull) {
        addModuleTypedefToHashTable(Module, _Typedef);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Typedef from the Module.
----------------------------------------------------------------------------------------*/
void dvModuleRemoveTypedef(
    dvModule Module,
    dvTypedef _Typedef)
{
    dvTypedef pTypedef, nTypedef;

#if defined(DD_DEBUG)
    if(_Typedef == dvTypedefNull) {
        utExit("Non-existent Typedef");
    }
    if(dvTypedefGetModule(_Typedef) != dvModuleNull && dvTypedefGetModule(_Typedef) != Module) {
        utExit("Delete Typedef from non-owning Module");
    }
#endif
    nTypedef = dvTypedefGetNextModuleTypedef(_Typedef);
    pTypedef = dvTypedefGetPrevModuleTypedef(_Typedef);
    if(pTypedef != dvTypedefNull) {
        dvTypedefSetNextModuleTypedef(pTypedef, nTypedef);
    } else if(dvModuleGetFirstTypedef(Module) == _Typedef) {
        dvModuleSetFirstTypedef(Module, nTypedef);
    }
    if(nTypedef != dvTypedefNull) {
        dvTypedefSetPrevModuleTypedef(nTypedef, pTypedef);
    } else if(dvModuleGetLastTypedef(Module) == _Typedef) {
        dvModuleSetLastTypedef(Module, pTypedef);
    }
    dvTypedefSetNextModuleTypedef(_Typedef, dvTypedefNull);
    dvTypedefSetPrevModuleTypedef(_Typedef, dvTypedefNull);
    dvTypedefSetModule(_Typedef, dvModuleNull);
    if(dvTypedefGetSym(_Typedef) != utSymNull) {
        removeModuleTypedefFromHashTable(Module, _Typedef);
    }
}

static void addModuleSchemaToHashTable(dvModule Module, dvSchema _Schema);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeModuleSchemaHashTable(
    dvModule Module)
{
    dvSchema _Schema;
    dvSchema *Schemas;
    uint32 numSchemas = dvModuleGetNumSchemaTable(Module) << 1;

    if(numSchemas == 0) {
        numSchemas = 2;
        dvModuleAllocSchemaTables(Module, 2);
    } else {
        dvModuleResizeSchemaTables(Module, numSchemas);
    }
    Schemas = dvModuleGetSchemaTables(Module);
    /* Zero out the table */
    while(numSchemas-- != 0) {
        *Schemas++ = dvSchemaNull;
    }
    dvModuleSetNumSchema(Module, 0);
    dvForeachModuleSchema(Module, _Schema) {
        if(dvSchemaGetSym(_Schema) != utSymNull) {
            addModuleSchemaToHashTable(Module, _Schema);
        }
    } dvEndModuleSchema;
}

/*----------------------------------------------------------------------------------------
  Add the Schema to the Module.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addModuleSchemaToHashTable(
    dvModule Module,
    dvSchema _Schema)
{
    dvSchema nextSchema;
    uint32 index;

    if(dvModuleGetNumSchema(Module) >= dvModuleGetNumSchemaTable(Module)) {
        resizeModuleSchemaHashTable(Module);
        return;
    }
    index = (dvModuleGetNumSchemaTable(Module) - 1) & utSymGetHashValue(dvSchemaGetSym(_Schema));
    nextSchema = dvModuleGetiSchemaTable(Module, index);
    dvSchemaSetNextTableModuleSchema(_Schema, nextSchema);
    dvModuleSetiSchemaTable(Module, index, _Schema);
    dvModuleSetNumSchema(Module, dvModuleGetNumSchema(Module) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Schema from the hash table.
----------------------------------------------------------------------------------------*/
static void removeModuleSchemaFromHashTable(
   dvModule Module,
   dvSchema _Schema)
{
    uint32 index = (dvModuleGetNumSchemaTable(Module) - 1) & utSymGetHashValue(dvSchemaGetSym(_Schema));
    dvSchema prevSchema, nextSchema;
    
    nextSchema = dvModuleGetiSchemaTable(Module, index);
    if(nextSchema == _Schema) {
        dvModuleSetiSchemaTable(Module, index, dvSchemaGetNextTableModuleSchema(nextSchema));
    } else {
        do {
            prevSchema = nextSchema;
            nextSchema = dvSchemaGetNextTableModuleSchema(nextSchema);
        } while(nextSchema != _Schema);
        dvSchemaSetNextTableModuleSchema(prevSchema, dvSchemaGetNextTableModuleSchema(_Schema));
    }
    dvModuleSetNumSchema(Module, dvModuleGetNumSchema(Module) - 1);
    dvSchemaSetNextTableModuleSchema(_Schema, dvSchemaNull);
}

/*----------------------------------------------------------------------------------------
  Find the Schema from the Module and its hash key.
----------------------------------------------------------------------------------------*/
dvSchema dvModuleFindSchema(
    dvModule Module,
    utSym Sym)
{
    uint32 mask = dvModuleGetNumSchemaTable(Module) - 1;
    dvSchema _Schema;

    if(mask + 1 != 0) {
        _Schema = dvModuleGetiSchemaTable(Module, utSymGetHashValue(Sym) & mask);
        while(_Schema != dvSchemaNull) {
            if(dvSchemaGetSym(_Schema) == Sym) {
                return _Schema;
            }
            _Schema = dvSchemaGetNextTableModuleSchema(_Schema);
        }
    }
    return dvSchemaNull;
}

/*----------------------------------------------------------------------------------------
  Find the Schema from the Module and its name.
----------------------------------------------------------------------------------------*/
void dvModuleRenameSchema(
    dvModule Module,
    dvSchema _Schema,
    utSym sym)
{
    if(dvSchemaGetSym(_Schema) != utSymNull) {
        removeModuleSchemaFromHashTable(Module, _Schema);
    }
    dvSchemaSetSym(_Schema, sym);
    if(sym != utSymNull) {
        addModuleSchemaToHashTable(Module, _Schema);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Schema to the head of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleInsertSchema(
    dvModule Module,
    dvSchema _Schema)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Schema == dvSchemaNull) {
        utExit("Non-existent Schema");
    }
    if(dvSchemaGetModule(_Schema) != dvModuleNull) {
        utExit("Attempting to add Schema to Module twice");
    }
#endif
    dvSchemaSetNextModuleSchema(_Schema, dvModuleGetFirstSchema(Module));
    if(dvModuleGetFirstSchema(Module) != dvSchemaNull) {
        dvSchemaSetPrevModuleSchema(dvModuleGetFirstSchema(Module), _Schema);
    }
    dvModuleSetFirstSchema(Module, _Schema);
    dvSchemaSetPrevModuleSchema(_Schema, dvSchemaNull);
    if(dvModuleGetLastSchema(Module) == dvSchemaNull) {
        dvModuleSetLastSchema(Module, _Schema);
    }
    dvSchemaSetModule(_Schema, Module);
    if(dvSchemaGetSym(_Schema) != utSymNull) {
        addModuleSchemaToHashTable(Module, _Schema);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Schema to the end of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleAppendSchema(
    dvModule Module,
    dvSchema _Schema)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Schema == dvSchemaNull) {
        utExit("Non-existent Schema");
    }
    if(dvSchemaGetModule(_Schema) != dvModuleNull) {
        utExit("Attempting to add Schema to Module twice");
    }
#endif
    dvSchemaSetPrevModuleSchema(_Schema, dvModuleGetLastSchema(Module));
    if(dvModuleGetLastSchema(Module) != dvSchemaNull) {
        dvSchemaSetNextModuleSchema(dvModuleGetLastSchema(Module), _Schema);
    }
    dvModuleSetLastSchema(Module, _Schema);
    dvSchemaSetNextModuleSchema(_Schema, dvSchemaNull);
    if(dvModuleGetFirstSchema(Module) == dvSchemaNull) {
        dvModuleSetFirstSchema(Module, _Schema);
    }
    dvSchemaSetModule(_Schema, Module);
    if(dvSchemaGetSym(_Schema) != utSymNull) {
        addModuleSchemaToHashTable(Module, _Schema);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Schema to the Module after the previous Schema.
----------------------------------------------------------------------------------------*/
void dvModuleInsertAfterSchema(
    dvModule Module,
    dvSchema prevSchema,
    dvSchema _Schema)
{
    dvSchema nextSchema = dvSchemaGetNextModuleSchema(prevSchema);

#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Schema == dvSchemaNull) {
        utExit("Non-existent Schema");
    }
    if(dvSchemaGetModule(_Schema) != dvModuleNull) {
        utExit("Attempting to add Schema to Module twice");
    }
#endif
    dvSchemaSetNextModuleSchema(_Schema, nextSchema);
    dvSchemaSetNextModuleSchema(prevSchema, _Schema);
    dvSchemaSetPrevModuleSchema(_Schema, prevSchema);
    if(nextSchema != dvSchemaNull) {
        dvSchemaSetPrevModuleSchema(nextSchema, _Schema);
    }
    if(dvModuleGetLastSchema(Module) == prevSchema) {
        dvModuleSetLastSchema(Module, _Schema);
    }
    dvSchemaSetModule(_Schema, Module);
    if(dvSchemaGetSym(_Schema) != utSymNull) {
        addModuleSchemaToHashTable(Module, _Schema);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Schema from the Module.
----------------------------------------------------------------------------------------*/
void dvModuleRemoveSchema(
    dvModule Module,
    dvSchema _Schema)
{
    dvSchema pSchema, nSchema;

#if defined(DD_DEBUG)
    if(_Schema == dvSchemaNull) {
        utExit("Non-existent Schema");
    }
    if(dvSchemaGetModule(_Schema) != dvModuleNull && dvSchemaGetModule(_Schema) != Module) {
        utExit("Delete Schema from non-owning Module");
    }
#endif
    nSchema = dvSchemaGetNextModuleSchema(_Schema);
    pSchema = dvSchemaGetPrevModuleSchema(_Schema);
    if(pSchema != dvSchemaNull) {
        dvSchemaSetNextModuleSchema(pSchema, nSchema);
    } else if(dvModuleGetFirstSchema(Module) == _Schema) {
        dvModuleSetFirstSchema(Module, nSchema);
    }
    if(nSchema != dvSchemaNull) {
        dvSchemaSetPrevModuleSchema(nSchema, pSchema);
    } else if(dvModuleGetLastSchema(Module) == _Schema) {
        dvModuleSetLastSchema(Module, pSchema);
    }
    dvSchemaSetNextModuleSchema(_Schema, dvSchemaNull);
    dvSchemaSetPrevModuleSchema(_Schema, dvSchemaNull);
    dvSchemaSetModule(_Schema, dvModuleNull);
    if(dvSchemaGetSym(_Schema) != utSymNull) {
        removeModuleSchemaFromHashTable(Module, _Schema);
    }
}

/*----------------------------------------------------------------------------------------
  Add the ImportLink to the head of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleInsertImportLink(
    dvModule Module,
    dvLink _Link)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Link == dvLinkNull) {
        utExit("Non-existent Link");
    }
    if(dvLinkGetImportModule(_Link) != dvModuleNull) {
        utExit("Attempting to add Link to Module twice");
    }
#endif
    dvLinkSetNextModuleImportLink(_Link, dvModuleGetFirstImportLink(Module));
    dvModuleSetFirstImportLink(Module, _Link);
    if(dvModuleGetLastImportLink(Module) == dvLinkNull) {
        dvModuleSetLastImportLink(Module, _Link);
    }
    dvLinkSetImportModule(_Link, Module);
}

/*----------------------------------------------------------------------------------------
  Add the ImportLink to the end of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleAppendImportLink(
    dvModule Module,
    dvLink _Link)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Link == dvLinkNull) {
        utExit("Non-existent Link");
    }
    if(dvLinkGetImportModule(_Link) != dvModuleNull) {
        utExit("Attempting to add Link to Module twice");
    }
#endif
    if(dvModuleGetLastImportLink(Module) != dvLinkNull) {
        dvLinkSetNextModuleImportLink(dvModuleGetLastImportLink(Module), _Link);
    } else {
        dvModuleSetFirstImportLink(Module, _Link);
    }
    dvModuleSetLastImportLink(Module, _Link);
    dvLinkSetNextModuleImportLink(_Link, dvLinkNull);
    dvLinkSetImportModule(_Link, Module);
}

/*----------------------------------------------------------------------------------------
  Insert the ImportLink to the Module after the previous ImportLink.
----------------------------------------------------------------------------------------*/
void dvModuleInsertAfterImportLink(
    dvModule Module,
    dvLink prevLink,
    dvLink _Link)
{
    dvLink nextLink = dvLinkGetNextModuleImportLink(prevLink);

#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Link == dvLinkNull) {
        utExit("Non-existent Link");
    }
    if(dvLinkGetImportModule(_Link) != dvModuleNull) {
        utExit("Attempting to add Link to Module twice");
    }
#endif
    dvLinkSetNextModuleImportLink(_Link, nextLink);
    dvLinkSetNextModuleImportLink(prevLink, _Link);
    if(dvModuleGetLastImportLink(Module) == prevLink) {
        dvModuleSetLastImportLink(Module, _Link);
    }
    dvLinkSetImportModule(_Link, Module);
}

/*----------------------------------------------------------------------------------------
 Remove the ImportLink from the Module.
----------------------------------------------------------------------------------------*/
void dvModuleRemoveImportLink(
    dvModule Module,
    dvLink _Link)
{
    dvLink pLink, nLink;

#if defined(DD_DEBUG)
    if(_Link == dvLinkNull) {
        utExit("Non-existent Link");
    }
    if(dvLinkGetImportModule(_Link) != dvModuleNull && dvLinkGetImportModule(_Link) != Module) {
        utExit("Delete Link from non-owning Module");
    }
#endif
    pLink = dvLinkNull;
    for(nLink = dvModuleGetFirstImportLink(Module); nLink != dvLinkNull && nLink != _Link;
            nLink = dvLinkGetNextModuleImportLink(nLink)) {
        pLink = nLink;
    }
    if(pLink != dvLinkNull) {
        dvLinkSetNextModuleImportLink(pLink, dvLinkGetNextModuleImportLink(_Link));
    } else {
        dvModuleSetFirstImportLink(Module, dvLinkGetNextModuleImportLink(_Link));
    }
    dvLinkSetNextModuleImportLink(_Link, dvLinkNull);
    if(dvModuleGetLastImportLink(Module) == _Link) {
        dvModuleSetLastImportLink(Module, pLink);
    }
    dvLinkSetImportModule(_Link, dvModuleNull);
}

/*----------------------------------------------------------------------------------------
  Add the ExportLink to the head of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleInsertExportLink(
    dvModule Module,
    dvLink _Link)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Link == dvLinkNull) {
        utExit("Non-existent Link");
    }
    if(dvLinkGetExportModule(_Link) != dvModuleNull) {
        utExit("Attempting to add Link to Module twice");
    }
#endif
    dvLinkSetNextModuleExportLink(_Link, dvModuleGetFirstExportLink(Module));
    dvModuleSetFirstExportLink(Module, _Link);
    if(dvModuleGetLastExportLink(Module) == dvLinkNull) {
        dvModuleSetLastExportLink(Module, _Link);
    }
    dvLinkSetExportModule(_Link, Module);
}

/*----------------------------------------------------------------------------------------
  Add the ExportLink to the end of the list on the Module.
----------------------------------------------------------------------------------------*/
void dvModuleAppendExportLink(
    dvModule Module,
    dvLink _Link)
{
#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Link == dvLinkNull) {
        utExit("Non-existent Link");
    }
    if(dvLinkGetExportModule(_Link) != dvModuleNull) {
        utExit("Attempting to add Link to Module twice");
    }
#endif
    if(dvModuleGetLastExportLink(Module) != dvLinkNull) {
        dvLinkSetNextModuleExportLink(dvModuleGetLastExportLink(Module), _Link);
    } else {
        dvModuleSetFirstExportLink(Module, _Link);
    }
    dvModuleSetLastExportLink(Module, _Link);
    dvLinkSetNextModuleExportLink(_Link, dvLinkNull);
    dvLinkSetExportModule(_Link, Module);
}

/*----------------------------------------------------------------------------------------
  Insert the ExportLink to the Module after the previous ExportLink.
----------------------------------------------------------------------------------------*/
void dvModuleInsertAfterExportLink(
    dvModule Module,
    dvLink prevLink,
    dvLink _Link)
{
    dvLink nextLink = dvLinkGetNextModuleExportLink(prevLink);

#if defined(DD_DEBUG)
    if(Module == dvModuleNull) {
        utExit("Non-existent Module");
    }
    if(_Link == dvLinkNull) {
        utExit("Non-existent Link");
    }
    if(dvLinkGetExportModule(_Link) != dvModuleNull) {
        utExit("Attempting to add Link to Module twice");
    }
#endif
    dvLinkSetNextModuleExportLink(_Link, nextLink);
    dvLinkSetNextModuleExportLink(prevLink, _Link);
    if(dvModuleGetLastExportLink(Module) == prevLink) {
        dvModuleSetLastExportLink(Module, _Link);
    }
    dvLinkSetExportModule(_Link, Module);
}

/*----------------------------------------------------------------------------------------
 Remove the ExportLink from the Module.
----------------------------------------------------------------------------------------*/
void dvModuleRemoveExportLink(
    dvModule Module,
    dvLink _Link)
{
    dvLink pLink, nLink;

#if defined(DD_DEBUG)
    if(_Link == dvLinkNull) {
        utExit("Non-existent Link");
    }
    if(dvLinkGetExportModule(_Link) != dvModuleNull && dvLinkGetExportModule(_Link) != Module) {
        utExit("Delete Link from non-owning Module");
    }
#endif
    pLink = dvLinkNull;
    for(nLink = dvModuleGetFirstExportLink(Module); nLink != dvLinkNull && nLink != _Link;
            nLink = dvLinkGetNextModuleExportLink(nLink)) {
        pLink = nLink;
    }
    if(pLink != dvLinkNull) {
        dvLinkSetNextModuleExportLink(pLink, dvLinkGetNextModuleExportLink(_Link));
    } else {
        dvModuleSetFirstExportLink(Module, dvLinkGetNextModuleExportLink(_Link));
    }
    dvLinkSetNextModuleExportLink(_Link, dvLinkNull);
    if(dvModuleGetLastExportLink(Module) == _Link) {
        dvModuleSetLastExportLink(Module, pLink);
    }
    dvLinkSetExportModule(_Link, dvModuleNull);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowModule(
    dvModule Module)
{
    utDatabaseShowObject("dv", "Module", dvModule2Index(Module));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocLink(void)
{
    dvLink Link = dvLinkAlloc();

    return dvLink2Index(Link);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Link.
----------------------------------------------------------------------------------------*/
static void allocLinks(void)
{
    dvSetAllocatedLink(2);
    dvSetUsedLink(0);
    dvLinks.ImportModule = utNewA(dvModule, (dvAllocatedLink()));
    dvLinks.NextModuleImportLink = utNewA(dvLink, (dvAllocatedLink()));
    dvLinks.ExportModule = utNewA(dvModule, (dvAllocatedLink()));
    dvLinks.NextModuleExportLink = utNewA(dvLink, (dvAllocatedLink()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Link.
----------------------------------------------------------------------------------------*/
static void reallocLinks(
    uint32 newSize)
{
    utResizeArray(dvLinks.ImportModule, (newSize));
    utResizeArray(dvLinks.NextModuleImportLink, (newSize));
    utResizeArray(dvLinks.ExportModule, (newSize));
    utResizeArray(dvLinks.NextModuleExportLink, (newSize));
    dvSetAllocatedLink(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Links.
----------------------------------------------------------------------------------------*/
void dvLinkAllocMore(void)
{
    reallocLinks((uint32)(dvAllocatedLink() + (dvAllocatedLink() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Link.
----------------------------------------------------------------------------------------*/
void dvLinkCopyProps(
    dvLink oldLink,
    dvLink newLink)
{
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowLink(
    dvLink Link)
{
    utDatabaseShowObject("dv", "Link", dvLink2Index(Link));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocSchema(void)
{
    dvSchema Schema = dvSchemaAlloc();

    return dvSchema2Index(Schema);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Schema.
----------------------------------------------------------------------------------------*/
static void allocSchemas(void)
{
    dvSetAllocatedSchema(2);
    dvSetUsedSchema(0);
    dvSchemas.Sym = utNewA(utSym, (dvAllocatedSchema()));
    dvSchemas.Module = utNewA(dvModule, (dvAllocatedSchema()));
    dvSchemas.NextModuleSchema = utNewA(dvSchema, (dvAllocatedSchema()));
    dvSchemas.PrevModuleSchema = utNewA(dvSchema, (dvAllocatedSchema()));
    dvSchemas.NextTableModuleSchema = utNewA(dvSchema, (dvAllocatedSchema()));
    dvSchemas.FirstRelationship = utNewA(dvRelationship, (dvAllocatedSchema()));
    dvSchemas.LastRelationship = utNewA(dvRelationship, (dvAllocatedSchema()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Schema.
----------------------------------------------------------------------------------------*/
static void reallocSchemas(
    uint32 newSize)
{
    utResizeArray(dvSchemas.Sym, (newSize));
    utResizeArray(dvSchemas.Module, (newSize));
    utResizeArray(dvSchemas.NextModuleSchema, (newSize));
    utResizeArray(dvSchemas.PrevModuleSchema, (newSize));
    utResizeArray(dvSchemas.NextTableModuleSchema, (newSize));
    utResizeArray(dvSchemas.FirstRelationship, (newSize));
    utResizeArray(dvSchemas.LastRelationship, (newSize));
    dvSetAllocatedSchema(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Schemas.
----------------------------------------------------------------------------------------*/
void dvSchemaAllocMore(void)
{
    reallocSchemas((uint32)(dvAllocatedSchema() + (dvAllocatedSchema() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Schema.
----------------------------------------------------------------------------------------*/
void dvSchemaCopyProps(
    dvSchema oldSchema,
    dvSchema newSchema)
{
}

/*----------------------------------------------------------------------------------------
  Add the Relationship to the head of the list on the Schema.
----------------------------------------------------------------------------------------*/
void dvSchemaInsertRelationship(
    dvSchema Schema,
    dvRelationship _Relationship)
{
#if defined(DD_DEBUG)
    if(Schema == dvSchemaNull) {
        utExit("Non-existent Schema");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetSchema(_Relationship) != dvSchemaNull) {
        utExit("Attempting to add Relationship to Schema twice");
    }
#endif
    dvRelationshipSetNextSchemaRelationship(_Relationship, dvSchemaGetFirstRelationship(Schema));
    dvSchemaSetFirstRelationship(Schema, _Relationship);
    if(dvSchemaGetLastRelationship(Schema) == dvRelationshipNull) {
        dvSchemaSetLastRelationship(Schema, _Relationship);
    }
    dvRelationshipSetSchema(_Relationship, Schema);
}

/*----------------------------------------------------------------------------------------
  Add the Relationship to the end of the list on the Schema.
----------------------------------------------------------------------------------------*/
void dvSchemaAppendRelationship(
    dvSchema Schema,
    dvRelationship _Relationship)
{
#if defined(DD_DEBUG)
    if(Schema == dvSchemaNull) {
        utExit("Non-existent Schema");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetSchema(_Relationship) != dvSchemaNull) {
        utExit("Attempting to add Relationship to Schema twice");
    }
#endif
    if(dvSchemaGetLastRelationship(Schema) != dvRelationshipNull) {
        dvRelationshipSetNextSchemaRelationship(dvSchemaGetLastRelationship(Schema), _Relationship);
    } else {
        dvSchemaSetFirstRelationship(Schema, _Relationship);
    }
    dvSchemaSetLastRelationship(Schema, _Relationship);
    dvRelationshipSetNextSchemaRelationship(_Relationship, dvRelationshipNull);
    dvRelationshipSetSchema(_Relationship, Schema);
}

/*----------------------------------------------------------------------------------------
  Insert the Relationship to the Schema after the previous Relationship.
----------------------------------------------------------------------------------------*/
void dvSchemaInsertAfterRelationship(
    dvSchema Schema,
    dvRelationship prevRelationship,
    dvRelationship _Relationship)
{
    dvRelationship nextRelationship = dvRelationshipGetNextSchemaRelationship(prevRelationship);

#if defined(DD_DEBUG)
    if(Schema == dvSchemaNull) {
        utExit("Non-existent Schema");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetSchema(_Relationship) != dvSchemaNull) {
        utExit("Attempting to add Relationship to Schema twice");
    }
#endif
    dvRelationshipSetNextSchemaRelationship(_Relationship, nextRelationship);
    dvRelationshipSetNextSchemaRelationship(prevRelationship, _Relationship);
    if(dvSchemaGetLastRelationship(Schema) == prevRelationship) {
        dvSchemaSetLastRelationship(Schema, _Relationship);
    }
    dvRelationshipSetSchema(_Relationship, Schema);
}

/*----------------------------------------------------------------------------------------
 Remove the Relationship from the Schema.
----------------------------------------------------------------------------------------*/
void dvSchemaRemoveRelationship(
    dvSchema Schema,
    dvRelationship _Relationship)
{
    dvRelationship pRelationship, nRelationship;

#if defined(DD_DEBUG)
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetSchema(_Relationship) != dvSchemaNull && dvRelationshipGetSchema(_Relationship) != Schema) {
        utExit("Delete Relationship from non-owning Schema");
    }
#endif
    pRelationship = dvRelationshipNull;
    for(nRelationship = dvSchemaGetFirstRelationship(Schema); nRelationship != dvRelationshipNull && nRelationship != _Relationship;
            nRelationship = dvRelationshipGetNextSchemaRelationship(nRelationship)) {
        pRelationship = nRelationship;
    }
    if(pRelationship != dvRelationshipNull) {
        dvRelationshipSetNextSchemaRelationship(pRelationship, dvRelationshipGetNextSchemaRelationship(_Relationship));
    } else {
        dvSchemaSetFirstRelationship(Schema, dvRelationshipGetNextSchemaRelationship(_Relationship));
    }
    dvRelationshipSetNextSchemaRelationship(_Relationship, dvRelationshipNull);
    if(dvSchemaGetLastRelationship(Schema) == _Relationship) {
        dvSchemaSetLastRelationship(Schema, pRelationship);
    }
    dvRelationshipSetSchema(_Relationship, dvSchemaNull);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowSchema(
    dvSchema Schema)
{
    utDatabaseShowObject("dv", "Schema", dvSchema2Index(Schema));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocEnum(void)
{
    dvEnum Enum = dvEnumAlloc();

    return dvEnum2Index(Enum);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Enum.
----------------------------------------------------------------------------------------*/
static void allocEnums(void)
{
    dvSetAllocatedEnum(2);
    dvSetUsedEnum(0);
    dvEnums.Sym = utNewA(utSym, (dvAllocatedEnum()));
    dvEnums.PrefixSym = utNewA(utSym, (dvAllocatedEnum()));
    dvEnums.NumEntries = utNewA(uint16, (dvAllocatedEnum()));
    dvEnums.Module = utNewA(dvModule, (dvAllocatedEnum()));
    dvEnums.NextModuleEnum = utNewA(dvEnum, (dvAllocatedEnum()));
    dvEnums.PrevModuleEnum = utNewA(dvEnum, (dvAllocatedEnum()));
    dvEnums.NextTableModuleEnum = utNewA(dvEnum, (dvAllocatedEnum()));
    dvEnums.FirstEntry = utNewA(dvEntry, (dvAllocatedEnum()));
    dvEnums.LastEntry = utNewA(dvEntry, (dvAllocatedEnum()));
    dvEnums.EntryTableIndex = utNewA(uint32, (dvAllocatedEnum()));
    dvEnums.NumEntryTable = utNewA(uint32, (dvAllocatedEnum()));
    dvSetUsedEnumEntryTable(0);
    dvSetAllocatedEnumEntryTable(2);
    dvSetFreeEnumEntryTable(0);
    dvEnums.EntryTable = utNewA(dvEntry, dvAllocatedEnumEntryTable());
    dvEnums.NumEntry = utNewA(uint32, (dvAllocatedEnum()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Enum.
----------------------------------------------------------------------------------------*/
static void reallocEnums(
    uint32 newSize)
{
    utResizeArray(dvEnums.Sym, (newSize));
    utResizeArray(dvEnums.PrefixSym, (newSize));
    utResizeArray(dvEnums.NumEntries, (newSize));
    utResizeArray(dvEnums.Module, (newSize));
    utResizeArray(dvEnums.NextModuleEnum, (newSize));
    utResizeArray(dvEnums.PrevModuleEnum, (newSize));
    utResizeArray(dvEnums.NextTableModuleEnum, (newSize));
    utResizeArray(dvEnums.FirstEntry, (newSize));
    utResizeArray(dvEnums.LastEntry, (newSize));
    utResizeArray(dvEnums.EntryTableIndex, (newSize));
    utResizeArray(dvEnums.NumEntryTable, (newSize));
    utResizeArray(dvEnums.NumEntry, (newSize));
    dvSetAllocatedEnum(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Enums.
----------------------------------------------------------------------------------------*/
void dvEnumAllocMore(void)
{
    reallocEnums((uint32)(dvAllocatedEnum() + (dvAllocatedEnum() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the Enum.EntryTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactEnumEntryTables(void)
{
    uint32 elementSize = sizeof(dvEntry);
    uint32 usedHeaderSize = (sizeof(dvEnum) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvEnum) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvEntry *toPtr = dvEnums.EntryTable;
    dvEntry *fromPtr = toPtr;
    dvEnum Enum;
    uint32 size;

    while(fromPtr < dvEnums.EntryTable + dvUsedEnumEntryTable()) {
        Enum = *(dvEnum *)(void *)fromPtr;
        if(Enum != dvEnumNull) {
            /* Need to move it to toPtr */
            size = utMax(dvEnumGetNumEntryTable(Enum) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvEnumSetEntryTableIndex(Enum, toPtr - dvEnums.EntryTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvEnum *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedEnumEntryTable(toPtr - dvEnums.EntryTable);
    dvSetFreeEnumEntryTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Enum.EntryTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreEnumEntryTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedEnumEntryTable() - dvUsedEnumEntryTable();

    if((dvFreeEnumEntryTable() << 2) > dvUsedEnumEntryTable()) {
        dvCompactEnumEntryTables();
        freeSpace = dvAllocatedEnumEntryTable() - dvUsedEnumEntryTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedEnumEntryTable(dvAllocatedEnumEntryTable() + spaceNeeded - freeSpace +
            (dvAllocatedEnumEntryTable() >> 1));
        utResizeArray(dvEnums.EntryTable, dvAllocatedEnumEntryTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Enum.EntryTable array.
----------------------------------------------------------------------------------------*/
void dvEnumAllocEntryTables(
    dvEnum Enum,
    uint32 numEntryTables)
{
    uint32 freeSpace = dvAllocatedEnumEntryTable() - dvUsedEnumEntryTable();
    uint32 elementSize = sizeof(dvEntry);
    uint32 usedHeaderSize = (sizeof(dvEnum) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvEnum) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numEntryTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvEnumGetNumEntryTable(Enum) == 0);
#endif
    if(numEntryTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreEnumEntryTables(spaceNeeded);
    }
    dvEnumSetEntryTableIndex(Enum, dvUsedEnumEntryTable() + usedHeaderSize);
    dvEnumSetNumEntryTable(Enum, numEntryTables);
    *(dvEnum *)(void *)(dvEnums.EntryTable + dvUsedEnumEntryTable()) = Enum;
    {
        uint32 xEnum;
        for(xEnum = (uint32)(dvEnumGetEntryTableIndex(Enum)); xEnum < dvEnumGetEntryTableIndex(Enum) + numEntryTables; xEnum++) {
            dvEnums.EntryTable[xEnum] = dvEntryNull;
        }
    }
    dvSetUsedEnumEntryTable(dvUsedEnumEntryTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvEnumGetEntryTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getEnumEntryTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvEnum Enum = dvIndex2Enum((uint32)objectNumber);

    *numValues = dvEnumGetNumEntryTable(Enum);
    return dvEnumGetEntryTables(Enum);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvEnumAllocEntryTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocEnumEntryTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvEnum Enum = dvIndex2Enum((uint32)objectNumber);

    dvEnumSetEntryTableIndex(Enum, 0);
    dvEnumSetNumEntryTable(Enum, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvEnumAllocEntryTables(Enum, numValues);
    return dvEnumGetEntryTables(Enum);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Enum.EntryTable array.
----------------------------------------------------------------------------------------*/
void dvEnumFreeEntryTables(
    dvEnum Enum)
{
    uint32 elementSize = sizeof(dvEntry);
    uint32 usedHeaderSize = (sizeof(dvEnum) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvEnum) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvEnumGetNumEntryTable(Enum) + usedHeaderSize, freeHeaderSize);
    dvEntry *dataPtr = dvEnumGetEntryTables(Enum) - usedHeaderSize;

    if(dvEnumGetNumEntryTable(Enum) == 0) {
        return;
    }
    *(dvEnum *)(void *)(dataPtr) = dvEnumNull;
    *(uint32 *)(void *)(((dvEnum *)(void *)dataPtr) + 1) = size;
    dvEnumSetNumEntryTable(Enum, 0);
    dvSetFreeEnumEntryTable(dvFreeEnumEntryTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Enum.EntryTable array.
----------------------------------------------------------------------------------------*/
void dvEnumResizeEntryTables(
    dvEnum Enum,
    uint32 numEntryTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvEntry);
    uint32 usedHeaderSize = (sizeof(dvEnum) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvEnum) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numEntryTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvEnumGetNumEntryTable(Enum) + usedHeaderSize, freeHeaderSize);
    dvEntry *dataPtr;

    if(numEntryTables == 0) {
        if(dvEnumGetNumEntryTable(Enum) != 0) {
            dvEnumFreeEntryTables(Enum);
        }
        return;
    }
    if(dvEnumGetNumEntryTable(Enum) == 0) {
        dvEnumAllocEntryTables(Enum, numEntryTables);
        return;
    }
    freeSpace = dvAllocatedEnumEntryTable() - dvUsedEnumEntryTable();
    if(freeSpace < newSize) {
        allocMoreEnumEntryTables(newSize);
    }
    dataPtr = dvEnumGetEntryTables(Enum) - usedHeaderSize;
    memcpy((void *)(dvEnums.EntryTable + dvUsedEnumEntryTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xEnum;
            for(xEnum = (uint32)(dvUsedEnumEntryTable() + oldSize); xEnum < dvUsedEnumEntryTable() + oldSize + newSize - oldSize; xEnum++) {
                dvEnums.EntryTable[xEnum] = dvEntryNull;
            }
        }
    }
    *(dvEnum *)(void *)dataPtr = dvEnumNull;
    *(uint32 *)(void *)(((dvEnum *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeEnumEntryTable(dvFreeEnumEntryTable() + oldSize);
    dvEnumSetEntryTableIndex(Enum, dvUsedEnumEntryTable() + usedHeaderSize);
    dvEnumSetNumEntryTable(Enum, numEntryTables);
    dvSetUsedEnumEntryTable(dvUsedEnumEntryTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Enum.
----------------------------------------------------------------------------------------*/
void dvEnumCopyProps(
    dvEnum oldEnum,
    dvEnum newEnum)
{
    dvEnumSetPrefixSym(newEnum, dvEnumGetPrefixSym(oldEnum));
    dvEnumSetNumEntries(newEnum, dvEnumGetNumEntries(oldEnum));
}

static void addEnumEntryToHashTable(dvEnum Enum, dvEntry _Entry);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeEnumEntryHashTable(
    dvEnum Enum)
{
    dvEntry _Entry;
    dvEntry *Entrys;
    uint32 numEntrys = dvEnumGetNumEntryTable(Enum) << 1;

    if(numEntrys == 0) {
        numEntrys = 2;
        dvEnumAllocEntryTables(Enum, 2);
    } else {
        dvEnumResizeEntryTables(Enum, numEntrys);
    }
    Entrys = dvEnumGetEntryTables(Enum);
    /* Zero out the table */
    while(numEntrys-- != 0) {
        *Entrys++ = dvEntryNull;
    }
    dvEnumSetNumEntry(Enum, 0);
    dvForeachEnumEntry(Enum, _Entry) {
        if(dvEntryGetSym(_Entry) != utSymNull) {
            addEnumEntryToHashTable(Enum, _Entry);
        }
    } dvEndEnumEntry;
}

/*----------------------------------------------------------------------------------------
  Add the Entry to the Enum.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addEnumEntryToHashTable(
    dvEnum Enum,
    dvEntry _Entry)
{
    dvEntry nextEntry;
    uint32 index;

    if(dvEnumGetNumEntry(Enum) >= dvEnumGetNumEntryTable(Enum)) {
        resizeEnumEntryHashTable(Enum);
        return;
    }
    index = (dvEnumGetNumEntryTable(Enum) - 1) & utSymGetHashValue(dvEntryGetSym(_Entry));
    nextEntry = dvEnumGetiEntryTable(Enum, index);
    dvEntrySetNextTableEnumEntry(_Entry, nextEntry);
    dvEnumSetiEntryTable(Enum, index, _Entry);
    dvEnumSetNumEntry(Enum, dvEnumGetNumEntry(Enum) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Entry from the hash table.
----------------------------------------------------------------------------------------*/
static void removeEnumEntryFromHashTable(
   dvEnum Enum,
   dvEntry _Entry)
{
    uint32 index = (dvEnumGetNumEntryTable(Enum) - 1) & utSymGetHashValue(dvEntryGetSym(_Entry));
    dvEntry prevEntry, nextEntry;
    
    nextEntry = dvEnumGetiEntryTable(Enum, index);
    if(nextEntry == _Entry) {
        dvEnumSetiEntryTable(Enum, index, dvEntryGetNextTableEnumEntry(nextEntry));
    } else {
        do {
            prevEntry = nextEntry;
            nextEntry = dvEntryGetNextTableEnumEntry(nextEntry);
        } while(nextEntry != _Entry);
        dvEntrySetNextTableEnumEntry(prevEntry, dvEntryGetNextTableEnumEntry(_Entry));
    }
    dvEnumSetNumEntry(Enum, dvEnumGetNumEntry(Enum) - 1);
    dvEntrySetNextTableEnumEntry(_Entry, dvEntryNull);
}

/*----------------------------------------------------------------------------------------
  Find the Entry from the Enum and its hash key.
----------------------------------------------------------------------------------------*/
dvEntry dvEnumFindEntry(
    dvEnum Enum,
    utSym Sym)
{
    uint32 mask = dvEnumGetNumEntryTable(Enum) - 1;
    dvEntry _Entry;

    if(mask + 1 != 0) {
        _Entry = dvEnumGetiEntryTable(Enum, utSymGetHashValue(Sym) & mask);
        while(_Entry != dvEntryNull) {
            if(dvEntryGetSym(_Entry) == Sym) {
                return _Entry;
            }
            _Entry = dvEntryGetNextTableEnumEntry(_Entry);
        }
    }
    return dvEntryNull;
}

/*----------------------------------------------------------------------------------------
  Find the Entry from the Enum and its name.
----------------------------------------------------------------------------------------*/
void dvEnumRenameEntry(
    dvEnum Enum,
    dvEntry _Entry,
    utSym sym)
{
    if(dvEntryGetSym(_Entry) != utSymNull) {
        removeEnumEntryFromHashTable(Enum, _Entry);
    }
    dvEntrySetSym(_Entry, sym);
    if(sym != utSymNull) {
        addEnumEntryToHashTable(Enum, _Entry);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Entry to the head of the list on the Enum.
----------------------------------------------------------------------------------------*/
void dvEnumInsertEntry(
    dvEnum Enum,
    dvEntry _Entry)
{
#if defined(DD_DEBUG)
    if(Enum == dvEnumNull) {
        utExit("Non-existent Enum");
    }
    if(_Entry == dvEntryNull) {
        utExit("Non-existent Entry");
    }
    if(dvEntryGetEnum(_Entry) != dvEnumNull) {
        utExit("Attempting to add Entry to Enum twice");
    }
#endif
    dvEntrySetNextEnumEntry(_Entry, dvEnumGetFirstEntry(Enum));
    if(dvEnumGetFirstEntry(Enum) != dvEntryNull) {
        dvEntrySetPrevEnumEntry(dvEnumGetFirstEntry(Enum), _Entry);
    }
    dvEnumSetFirstEntry(Enum, _Entry);
    dvEntrySetPrevEnumEntry(_Entry, dvEntryNull);
    if(dvEnumGetLastEntry(Enum) == dvEntryNull) {
        dvEnumSetLastEntry(Enum, _Entry);
    }
    dvEntrySetEnum(_Entry, Enum);
    if(dvEntryGetSym(_Entry) != utSymNull) {
        addEnumEntryToHashTable(Enum, _Entry);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Entry to the end of the list on the Enum.
----------------------------------------------------------------------------------------*/
void dvEnumAppendEntry(
    dvEnum Enum,
    dvEntry _Entry)
{
#if defined(DD_DEBUG)
    if(Enum == dvEnumNull) {
        utExit("Non-existent Enum");
    }
    if(_Entry == dvEntryNull) {
        utExit("Non-existent Entry");
    }
    if(dvEntryGetEnum(_Entry) != dvEnumNull) {
        utExit("Attempting to add Entry to Enum twice");
    }
#endif
    dvEntrySetPrevEnumEntry(_Entry, dvEnumGetLastEntry(Enum));
    if(dvEnumGetLastEntry(Enum) != dvEntryNull) {
        dvEntrySetNextEnumEntry(dvEnumGetLastEntry(Enum), _Entry);
    }
    dvEnumSetLastEntry(Enum, _Entry);
    dvEntrySetNextEnumEntry(_Entry, dvEntryNull);
    if(dvEnumGetFirstEntry(Enum) == dvEntryNull) {
        dvEnumSetFirstEntry(Enum, _Entry);
    }
    dvEntrySetEnum(_Entry, Enum);
    if(dvEntryGetSym(_Entry) != utSymNull) {
        addEnumEntryToHashTable(Enum, _Entry);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Entry to the Enum after the previous Entry.
----------------------------------------------------------------------------------------*/
void dvEnumInsertAfterEntry(
    dvEnum Enum,
    dvEntry prevEntry,
    dvEntry _Entry)
{
    dvEntry nextEntry = dvEntryGetNextEnumEntry(prevEntry);

#if defined(DD_DEBUG)
    if(Enum == dvEnumNull) {
        utExit("Non-existent Enum");
    }
    if(_Entry == dvEntryNull) {
        utExit("Non-existent Entry");
    }
    if(dvEntryGetEnum(_Entry) != dvEnumNull) {
        utExit("Attempting to add Entry to Enum twice");
    }
#endif
    dvEntrySetNextEnumEntry(_Entry, nextEntry);
    dvEntrySetNextEnumEntry(prevEntry, _Entry);
    dvEntrySetPrevEnumEntry(_Entry, prevEntry);
    if(nextEntry != dvEntryNull) {
        dvEntrySetPrevEnumEntry(nextEntry, _Entry);
    }
    if(dvEnumGetLastEntry(Enum) == prevEntry) {
        dvEnumSetLastEntry(Enum, _Entry);
    }
    dvEntrySetEnum(_Entry, Enum);
    if(dvEntryGetSym(_Entry) != utSymNull) {
        addEnumEntryToHashTable(Enum, _Entry);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Entry from the Enum.
----------------------------------------------------------------------------------------*/
void dvEnumRemoveEntry(
    dvEnum Enum,
    dvEntry _Entry)
{
    dvEntry pEntry, nEntry;

#if defined(DD_DEBUG)
    if(_Entry == dvEntryNull) {
        utExit("Non-existent Entry");
    }
    if(dvEntryGetEnum(_Entry) != dvEnumNull && dvEntryGetEnum(_Entry) != Enum) {
        utExit("Delete Entry from non-owning Enum");
    }
#endif
    nEntry = dvEntryGetNextEnumEntry(_Entry);
    pEntry = dvEntryGetPrevEnumEntry(_Entry);
    if(pEntry != dvEntryNull) {
        dvEntrySetNextEnumEntry(pEntry, nEntry);
    } else if(dvEnumGetFirstEntry(Enum) == _Entry) {
        dvEnumSetFirstEntry(Enum, nEntry);
    }
    if(nEntry != dvEntryNull) {
        dvEntrySetPrevEnumEntry(nEntry, pEntry);
    } else if(dvEnumGetLastEntry(Enum) == _Entry) {
        dvEnumSetLastEntry(Enum, pEntry);
    }
    dvEntrySetNextEnumEntry(_Entry, dvEntryNull);
    dvEntrySetPrevEnumEntry(_Entry, dvEntryNull);
    dvEntrySetEnum(_Entry, dvEnumNull);
    if(dvEntryGetSym(_Entry) != utSymNull) {
        removeEnumEntryFromHashTable(Enum, _Entry);
    }
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowEnum(
    dvEnum Enum)
{
    utDatabaseShowObject("dv", "Enum", dvEnum2Index(Enum));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocEntry(void)
{
    dvEntry Entry = dvEntryAlloc();

    return dvEntry2Index(Entry);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Entry.
----------------------------------------------------------------------------------------*/
static void allocEntrys(void)
{
    dvSetAllocatedEntry(2);
    dvSetUsedEntry(0);
    dvEntrys.Sym = utNewA(utSym, (dvAllocatedEntry()));
    dvEntrys.Value = utNewA(uint32, (dvAllocatedEntry()));
    dvEntrys.Enum = utNewA(dvEnum, (dvAllocatedEntry()));
    dvEntrys.NextEnumEntry = utNewA(dvEntry, (dvAllocatedEntry()));
    dvEntrys.PrevEnumEntry = utNewA(dvEntry, (dvAllocatedEntry()));
    dvEntrys.NextTableEnumEntry = utNewA(dvEntry, (dvAllocatedEntry()));
    dvEntrys.FirstCase = utNewA(dvCase, (dvAllocatedEntry()));
    dvEntrys.LastCase = utNewA(dvCase, (dvAllocatedEntry()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Entry.
----------------------------------------------------------------------------------------*/
static void reallocEntrys(
    uint32 newSize)
{
    utResizeArray(dvEntrys.Sym, (newSize));
    utResizeArray(dvEntrys.Value, (newSize));
    utResizeArray(dvEntrys.Enum, (newSize));
    utResizeArray(dvEntrys.NextEnumEntry, (newSize));
    utResizeArray(dvEntrys.PrevEnumEntry, (newSize));
    utResizeArray(dvEntrys.NextTableEnumEntry, (newSize));
    utResizeArray(dvEntrys.FirstCase, (newSize));
    utResizeArray(dvEntrys.LastCase, (newSize));
    dvSetAllocatedEntry(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Entrys.
----------------------------------------------------------------------------------------*/
void dvEntryAllocMore(void)
{
    reallocEntrys((uint32)(dvAllocatedEntry() + (dvAllocatedEntry() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Entry.
----------------------------------------------------------------------------------------*/
void dvEntryCopyProps(
    dvEntry oldEntry,
    dvEntry newEntry)
{
    dvEntrySetValue(newEntry, dvEntryGetValue(oldEntry));
}

/*----------------------------------------------------------------------------------------
  Add the Case to the head of the list on the Entry.
----------------------------------------------------------------------------------------*/
void dvEntryInsertCase(
    dvEntry Entry,
    dvCase _Case)
{
#if defined(DD_DEBUG)
    if(Entry == dvEntryNull) {
        utExit("Non-existent Entry");
    }
    if(_Case == dvCaseNull) {
        utExit("Non-existent Case");
    }
    if(dvCaseGetEntry(_Case) != dvEntryNull) {
        utExit("Attempting to add Case to Entry twice");
    }
#endif
    dvCaseSetNextEntryCase(_Case, dvEntryGetFirstCase(Entry));
    dvEntrySetFirstCase(Entry, _Case);
    if(dvEntryGetLastCase(Entry) == dvCaseNull) {
        dvEntrySetLastCase(Entry, _Case);
    }
    dvCaseSetEntry(_Case, Entry);
}

/*----------------------------------------------------------------------------------------
  Add the Case to the end of the list on the Entry.
----------------------------------------------------------------------------------------*/
void dvEntryAppendCase(
    dvEntry Entry,
    dvCase _Case)
{
#if defined(DD_DEBUG)
    if(Entry == dvEntryNull) {
        utExit("Non-existent Entry");
    }
    if(_Case == dvCaseNull) {
        utExit("Non-existent Case");
    }
    if(dvCaseGetEntry(_Case) != dvEntryNull) {
        utExit("Attempting to add Case to Entry twice");
    }
#endif
    if(dvEntryGetLastCase(Entry) != dvCaseNull) {
        dvCaseSetNextEntryCase(dvEntryGetLastCase(Entry), _Case);
    } else {
        dvEntrySetFirstCase(Entry, _Case);
    }
    dvEntrySetLastCase(Entry, _Case);
    dvCaseSetNextEntryCase(_Case, dvCaseNull);
    dvCaseSetEntry(_Case, Entry);
}

/*----------------------------------------------------------------------------------------
  Insert the Case to the Entry after the previous Case.
----------------------------------------------------------------------------------------*/
void dvEntryInsertAfterCase(
    dvEntry Entry,
    dvCase prevCase,
    dvCase _Case)
{
    dvCase nextCase = dvCaseGetNextEntryCase(prevCase);

#if defined(DD_DEBUG)
    if(Entry == dvEntryNull) {
        utExit("Non-existent Entry");
    }
    if(_Case == dvCaseNull) {
        utExit("Non-existent Case");
    }
    if(dvCaseGetEntry(_Case) != dvEntryNull) {
        utExit("Attempting to add Case to Entry twice");
    }
#endif
    dvCaseSetNextEntryCase(_Case, nextCase);
    dvCaseSetNextEntryCase(prevCase, _Case);
    if(dvEntryGetLastCase(Entry) == prevCase) {
        dvEntrySetLastCase(Entry, _Case);
    }
    dvCaseSetEntry(_Case, Entry);
}

/*----------------------------------------------------------------------------------------
 Remove the Case from the Entry.
----------------------------------------------------------------------------------------*/
void dvEntryRemoveCase(
    dvEntry Entry,
    dvCase _Case)
{
    dvCase pCase, nCase;

#if defined(DD_DEBUG)
    if(_Case == dvCaseNull) {
        utExit("Non-existent Case");
    }
    if(dvCaseGetEntry(_Case) != dvEntryNull && dvCaseGetEntry(_Case) != Entry) {
        utExit("Delete Case from non-owning Entry");
    }
#endif
    pCase = dvCaseNull;
    for(nCase = dvEntryGetFirstCase(Entry); nCase != dvCaseNull && nCase != _Case;
            nCase = dvCaseGetNextEntryCase(nCase)) {
        pCase = nCase;
    }
    if(pCase != dvCaseNull) {
        dvCaseSetNextEntryCase(pCase, dvCaseGetNextEntryCase(_Case));
    } else {
        dvEntrySetFirstCase(Entry, dvCaseGetNextEntryCase(_Case));
    }
    dvCaseSetNextEntryCase(_Case, dvCaseNull);
    if(dvEntryGetLastCase(Entry) == _Case) {
        dvEntrySetLastCase(Entry, pCase);
    }
    dvCaseSetEntry(_Case, dvEntryNull);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowEntry(
    dvEntry Entry)
{
    utDatabaseShowObject("dv", "Entry", dvEntry2Index(Entry));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocTypedef(void)
{
    dvTypedef Typedef = dvTypedefAlloc();

    return dvTypedef2Index(Typedef);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Typedef.
----------------------------------------------------------------------------------------*/
static void allocTypedefs(void)
{
    dvSetAllocatedTypedef(2);
    dvSetUsedTypedef(0);
    dvTypedefs.Sym = utNewA(utSym, (dvAllocatedTypedef()));
    dvTypedefs.InitializerIndex = utNewA(uint32, (dvAllocatedTypedef()));
    dvTypedefs.NumInitializer = utNewA(uint32, (dvAllocatedTypedef()));
    dvSetUsedTypedefInitializer(0);
    dvSetAllocatedTypedefInitializer(2);
    dvSetFreeTypedefInitializer(0);
    dvTypedefs.Initializer = utNewA(char, dvAllocatedTypedefInitializer());
    dvTypedefs.Module = utNewA(dvModule, (dvAllocatedTypedef()));
    dvTypedefs.NextModuleTypedef = utNewA(dvTypedef, (dvAllocatedTypedef()));
    dvTypedefs.PrevModuleTypedef = utNewA(dvTypedef, (dvAllocatedTypedef()));
    dvTypedefs.NextTableModuleTypedef = utNewA(dvTypedef, (dvAllocatedTypedef()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Typedef.
----------------------------------------------------------------------------------------*/
static void reallocTypedefs(
    uint32 newSize)
{
    utResizeArray(dvTypedefs.Sym, (newSize));
    utResizeArray(dvTypedefs.InitializerIndex, (newSize));
    utResizeArray(dvTypedefs.NumInitializer, (newSize));
    utResizeArray(dvTypedefs.Module, (newSize));
    utResizeArray(dvTypedefs.NextModuleTypedef, (newSize));
    utResizeArray(dvTypedefs.PrevModuleTypedef, (newSize));
    utResizeArray(dvTypedefs.NextTableModuleTypedef, (newSize));
    dvSetAllocatedTypedef(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Typedefs.
----------------------------------------------------------------------------------------*/
void dvTypedefAllocMore(void)
{
    reallocTypedefs((uint32)(dvAllocatedTypedef() + (dvAllocatedTypedef() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the Typedef.Initializer heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactTypedefInitializers(void)
{
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvTypedef) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvTypedef) + sizeof(uint32) + elementSize - 1)/elementSize;
    char *toPtr = dvTypedefs.Initializer;
    char *fromPtr = toPtr;
    dvTypedef Typedef;
    uint32 size;

    while(fromPtr < dvTypedefs.Initializer + dvUsedTypedefInitializer()) {
        Typedef = *(dvTypedef *)(void *)fromPtr;
        if(Typedef != dvTypedefNull) {
            /* Need to move it to toPtr */
            size = utMax(dvTypedefGetNumInitializer(Typedef) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvTypedefSetInitializerIndex(Typedef, toPtr - dvTypedefs.Initializer + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvTypedef *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedTypedefInitializer(toPtr - dvTypedefs.Initializer);
    dvSetFreeTypedefInitializer(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Typedef.Initializer heap.
----------------------------------------------------------------------------------------*/
static void allocMoreTypedefInitializers(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedTypedefInitializer() - dvUsedTypedefInitializer();

    if((dvFreeTypedefInitializer() << 2) > dvUsedTypedefInitializer()) {
        dvCompactTypedefInitializers();
        freeSpace = dvAllocatedTypedefInitializer() - dvUsedTypedefInitializer();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedTypedefInitializer(dvAllocatedTypedefInitializer() + spaceNeeded - freeSpace +
            (dvAllocatedTypedefInitializer() >> 1));
        utResizeArray(dvTypedefs.Initializer, dvAllocatedTypedefInitializer());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Typedef.Initializer array.
----------------------------------------------------------------------------------------*/
void dvTypedefAllocInitializers(
    dvTypedef Typedef,
    uint32 numInitializers)
{
    uint32 freeSpace = dvAllocatedTypedefInitializer() - dvUsedTypedefInitializer();
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvTypedef) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvTypedef) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numInitializers + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvTypedefGetNumInitializer(Typedef) == 0);
#endif
    if(numInitializers == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreTypedefInitializers(spaceNeeded);
    }
    dvTypedefSetInitializerIndex(Typedef, dvUsedTypedefInitializer() + usedHeaderSize);
    dvTypedefSetNumInitializer(Typedef, numInitializers);
    *(dvTypedef *)(void *)(dvTypedefs.Initializer + dvUsedTypedefInitializer()) = Typedef;
    memset(dvTypedefs.Initializer + dvTypedefGetInitializerIndex(Typedef), 0, ((numInitializers))*sizeof(char));
    dvSetUsedTypedefInitializer(dvUsedTypedefInitializer() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvTypedefGetInitializers for the database manager.
----------------------------------------------------------------------------------------*/
static void *getTypedefInitializers(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvTypedef Typedef = dvIndex2Typedef((uint32)objectNumber);

    *numValues = dvTypedefGetNumInitializer(Typedef);
    return dvTypedefGetInitializers(Typedef);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvTypedefAllocInitializers for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocTypedefInitializers(
    uint64 objectNumber,
    uint32 numValues)
{
    dvTypedef Typedef = dvIndex2Typedef((uint32)objectNumber);

    dvTypedefSetInitializerIndex(Typedef, 0);
    dvTypedefSetNumInitializer(Typedef, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvTypedefAllocInitializers(Typedef, numValues);
    return dvTypedefGetInitializers(Typedef);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Typedef.Initializer array.
----------------------------------------------------------------------------------------*/
void dvTypedefFreeInitializers(
    dvTypedef Typedef)
{
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvTypedef) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvTypedef) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvTypedefGetNumInitializer(Typedef) + usedHeaderSize, freeHeaderSize);
    char *dataPtr = dvTypedefGetInitializers(Typedef) - usedHeaderSize;

    if(dvTypedefGetNumInitializer(Typedef) == 0) {
        return;
    }
    *(dvTypedef *)(void *)(dataPtr) = dvTypedefNull;
    *(uint32 *)(void *)(((dvTypedef *)(void *)dataPtr) + 1) = size;
    dvTypedefSetNumInitializer(Typedef, 0);
    dvSetFreeTypedefInitializer(dvFreeTypedefInitializer() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Typedef.Initializer array.
----------------------------------------------------------------------------------------*/
void dvTypedefResizeInitializers(
    dvTypedef Typedef,
    uint32 numInitializers)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvTypedef) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvTypedef) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numInitializers + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvTypedefGetNumInitializer(Typedef) + usedHeaderSize, freeHeaderSize);
    char *dataPtr;

    if(numInitializers == 0) {
        if(dvTypedefGetNumInitializer(Typedef) != 0) {
            dvTypedefFreeInitializers(Typedef);
        }
        return;
    }
    if(dvTypedefGetNumInitializer(Typedef) == 0) {
        dvTypedefAllocInitializers(Typedef, numInitializers);
        return;
    }
    freeSpace = dvAllocatedTypedefInitializer() - dvUsedTypedefInitializer();
    if(freeSpace < newSize) {
        allocMoreTypedefInitializers(newSize);
    }
    dataPtr = dvTypedefGetInitializers(Typedef) - usedHeaderSize;
    memcpy((void *)(dvTypedefs.Initializer + dvUsedTypedefInitializer()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        memset(dvTypedefs.Initializer + dvUsedTypedefInitializer() + oldSize, 0, ((newSize - oldSize))*sizeof(char));
    }
    *(dvTypedef *)(void *)dataPtr = dvTypedefNull;
    *(uint32 *)(void *)(((dvTypedef *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeTypedefInitializer(dvFreeTypedefInitializer() + oldSize);
    dvTypedefSetInitializerIndex(Typedef, dvUsedTypedefInitializer() + usedHeaderSize);
    dvTypedefSetNumInitializer(Typedef, numInitializers);
    dvSetUsedTypedefInitializer(dvUsedTypedefInitializer() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Typedef.
----------------------------------------------------------------------------------------*/
void dvTypedefCopyProps(
    dvTypedef oldTypedef,
    dvTypedef newTypedef)
{
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowTypedef(
    dvTypedef Typedef)
{
    utDatabaseShowObject("dv", "Typedef", dvTypedef2Index(Typedef));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocClass(void)
{
    dvClass Class = dvClassAlloc();

    return dvClass2Index(Class);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Class.
----------------------------------------------------------------------------------------*/
static void allocClasss(void)
{
    dvSetAllocatedClass(2);
    dvSetUsedClass(0);
    dvClasss.Sym = utNewA(utSym, (dvAllocatedClass()));
    dvClasss.MemoryStyle = utNewA(dvMemoryStyle, (dvAllocatedClass()));
    dvClasss.ReferenceSize = utNewA(uint8, (dvAllocatedClass()));
    dvClasss.GenerateArrayClass = utNewA(uint8, (dvAllocatedClass() + 7) >> 3);
    dvClasss.GenerateAttributes = utNewA(uint8, (dvAllocatedClass() + 7) >> 3);
    dvClasss.Sparse = utNewA(uint8, (dvAllocatedClass() + 7) >> 3);
    dvClasss.NumFields = utNewA(uint16, (dvAllocatedClass()));
    dvClasss.Number = utNewA(uint16, (dvAllocatedClass()));
    dvClasss.Module = utNewA(dvModule, (dvAllocatedClass()));
    dvClasss.NextModuleClass = utNewA(dvClass, (dvAllocatedClass()));
    dvClasss.PrevModuleClass = utNewA(dvClass, (dvAllocatedClass()));
    dvClasss.NextTableModuleClass = utNewA(dvClass, (dvAllocatedClass()));
    dvClasss.FirstProperty = utNewA(dvProperty, (dvAllocatedClass()));
    dvClasss.LastProperty = utNewA(dvProperty, (dvAllocatedClass()));
    dvClasss.PropertyTableIndex = utNewA(uint32, (dvAllocatedClass()));
    dvClasss.NumPropertyTable = utNewA(uint32, (dvAllocatedClass()));
    dvSetUsedClassPropertyTable(0);
    dvSetAllocatedClassPropertyTable(2);
    dvSetFreeClassPropertyTable(0);
    dvClasss.PropertyTable = utNewA(dvProperty, dvAllocatedClassPropertyTable());
    dvClasss.NumProperty = utNewA(uint32, (dvAllocatedClass()));
    dvClasss.FreeListProperty = utNewA(dvProperty, (dvAllocatedClass()));
    dvClasss.FirstSparsegroup = utNewA(dvSparsegroup, (dvAllocatedClass()));
    dvClasss.LastSparsegroup = utNewA(dvSparsegroup, (dvAllocatedClass()));
    dvClasss.SparsegroupTableIndex = utNewA(uint32, (dvAllocatedClass()));
    dvClasss.NumSparsegroupTable = utNewA(uint32, (dvAllocatedClass()));
    dvSetUsedClassSparsegroupTable(0);
    dvSetAllocatedClassSparsegroupTable(2);
    dvSetFreeClassSparsegroupTable(0);
    dvClasss.SparsegroupTable = utNewA(dvSparsegroup, dvAllocatedClassSparsegroupTable());
    dvClasss.NumSparsegroup = utNewA(uint32, (dvAllocatedClass()));
    dvClasss.BaseClass = utNewA(dvClass, (dvAllocatedClass()));
    dvClasss.FirstDerivedClass = utNewA(dvClass, (dvAllocatedClass()));
    dvClasss.NextClassDerivedClass = utNewA(dvClass, (dvAllocatedClass()));
    dvClasss.LastDerivedClass = utNewA(dvClass, (dvAllocatedClass()));
    dvClasss.FirstChildRelationship = utNewA(dvRelationship, (dvAllocatedClass()));
    dvClasss.LastChildRelationship = utNewA(dvRelationship, (dvAllocatedClass()));
    dvClasss.FirstParentRelationship = utNewA(dvRelationship, (dvAllocatedClass()));
    dvClasss.LastParentRelationship = utNewA(dvRelationship, (dvAllocatedClass()));
    dvClasss.FirstUnion = utNewA(dvUnion, (dvAllocatedClass()));
    dvClasss.LastUnion = utNewA(dvUnion, (dvAllocatedClass()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Class.
----------------------------------------------------------------------------------------*/
static void reallocClasss(
    uint32 newSize)
{
    utResizeArray(dvClasss.Sym, (newSize));
    utResizeArray(dvClasss.MemoryStyle, (newSize));
    utResizeArray(dvClasss.ReferenceSize, (newSize));
    utResizeArray(dvClasss.GenerateArrayClass, (newSize + 7) >> 3);
    utResizeArray(dvClasss.GenerateAttributes, (newSize + 7) >> 3);
    utResizeArray(dvClasss.Sparse, (newSize + 7) >> 3);
    utResizeArray(dvClasss.NumFields, (newSize));
    utResizeArray(dvClasss.Number, (newSize));
    utResizeArray(dvClasss.Module, (newSize));
    utResizeArray(dvClasss.NextModuleClass, (newSize));
    utResizeArray(dvClasss.PrevModuleClass, (newSize));
    utResizeArray(dvClasss.NextTableModuleClass, (newSize));
    utResizeArray(dvClasss.FirstProperty, (newSize));
    utResizeArray(dvClasss.LastProperty, (newSize));
    utResizeArray(dvClasss.PropertyTableIndex, (newSize));
    utResizeArray(dvClasss.NumPropertyTable, (newSize));
    utResizeArray(dvClasss.NumProperty, (newSize));
    utResizeArray(dvClasss.FreeListProperty, (newSize));
    utResizeArray(dvClasss.FirstSparsegroup, (newSize));
    utResizeArray(dvClasss.LastSparsegroup, (newSize));
    utResizeArray(dvClasss.SparsegroupTableIndex, (newSize));
    utResizeArray(dvClasss.NumSparsegroupTable, (newSize));
    utResizeArray(dvClasss.NumSparsegroup, (newSize));
    utResizeArray(dvClasss.BaseClass, (newSize));
    utResizeArray(dvClasss.FirstDerivedClass, (newSize));
    utResizeArray(dvClasss.NextClassDerivedClass, (newSize));
    utResizeArray(dvClasss.LastDerivedClass, (newSize));
    utResizeArray(dvClasss.FirstChildRelationship, (newSize));
    utResizeArray(dvClasss.LastChildRelationship, (newSize));
    utResizeArray(dvClasss.FirstParentRelationship, (newSize));
    utResizeArray(dvClasss.LastParentRelationship, (newSize));
    utResizeArray(dvClasss.FirstUnion, (newSize));
    utResizeArray(dvClasss.LastUnion, (newSize));
    dvSetAllocatedClass(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Classs.
----------------------------------------------------------------------------------------*/
void dvClassAllocMore(void)
{
    reallocClasss((uint32)(dvAllocatedClass() + (dvAllocatedClass() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the Class.PropertyTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactClassPropertyTables(void)
{
    uint32 elementSize = sizeof(dvProperty);
    uint32 usedHeaderSize = (sizeof(dvClass) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvClass) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvProperty *toPtr = dvClasss.PropertyTable;
    dvProperty *fromPtr = toPtr;
    dvClass Class;
    uint32 size;

    while(fromPtr < dvClasss.PropertyTable + dvUsedClassPropertyTable()) {
        Class = *(dvClass *)(void *)fromPtr;
        if(Class != dvClassNull) {
            /* Need to move it to toPtr */
            size = utMax(dvClassGetNumPropertyTable(Class) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvClassSetPropertyTableIndex(Class, toPtr - dvClasss.PropertyTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvClass *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedClassPropertyTable(toPtr - dvClasss.PropertyTable);
    dvSetFreeClassPropertyTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Class.PropertyTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreClassPropertyTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedClassPropertyTable() - dvUsedClassPropertyTable();

    if((dvFreeClassPropertyTable() << 2) > dvUsedClassPropertyTable()) {
        dvCompactClassPropertyTables();
        freeSpace = dvAllocatedClassPropertyTable() - dvUsedClassPropertyTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedClassPropertyTable(dvAllocatedClassPropertyTable() + spaceNeeded - freeSpace +
            (dvAllocatedClassPropertyTable() >> 1));
        utResizeArray(dvClasss.PropertyTable, dvAllocatedClassPropertyTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Class.PropertyTable array.
----------------------------------------------------------------------------------------*/
void dvClassAllocPropertyTables(
    dvClass Class,
    uint32 numPropertyTables)
{
    uint32 freeSpace = dvAllocatedClassPropertyTable() - dvUsedClassPropertyTable();
    uint32 elementSize = sizeof(dvProperty);
    uint32 usedHeaderSize = (sizeof(dvClass) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvClass) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numPropertyTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvClassGetNumPropertyTable(Class) == 0);
#endif
    if(numPropertyTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreClassPropertyTables(spaceNeeded);
    }
    dvClassSetPropertyTableIndex(Class, dvUsedClassPropertyTable() + usedHeaderSize);
    dvClassSetNumPropertyTable(Class, numPropertyTables);
    *(dvClass *)(void *)(dvClasss.PropertyTable + dvUsedClassPropertyTable()) = Class;
    {
        uint32 xClass;
        for(xClass = (uint32)(dvClassGetPropertyTableIndex(Class)); xClass < dvClassGetPropertyTableIndex(Class) + numPropertyTables; xClass++) {
            dvClasss.PropertyTable[xClass] = dvPropertyNull;
        }
    }
    dvSetUsedClassPropertyTable(dvUsedClassPropertyTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvClassGetPropertyTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getClassPropertyTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvClass Class = dvIndex2Class((uint32)objectNumber);

    *numValues = dvClassGetNumPropertyTable(Class);
    return dvClassGetPropertyTables(Class);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvClassAllocPropertyTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocClassPropertyTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvClass Class = dvIndex2Class((uint32)objectNumber);

    dvClassSetPropertyTableIndex(Class, 0);
    dvClassSetNumPropertyTable(Class, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvClassAllocPropertyTables(Class, numValues);
    return dvClassGetPropertyTables(Class);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Class.PropertyTable array.
----------------------------------------------------------------------------------------*/
void dvClassFreePropertyTables(
    dvClass Class)
{
    uint32 elementSize = sizeof(dvProperty);
    uint32 usedHeaderSize = (sizeof(dvClass) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvClass) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvClassGetNumPropertyTable(Class) + usedHeaderSize, freeHeaderSize);
    dvProperty *dataPtr = dvClassGetPropertyTables(Class) - usedHeaderSize;

    if(dvClassGetNumPropertyTable(Class) == 0) {
        return;
    }
    *(dvClass *)(void *)(dataPtr) = dvClassNull;
    *(uint32 *)(void *)(((dvClass *)(void *)dataPtr) + 1) = size;
    dvClassSetNumPropertyTable(Class, 0);
    dvSetFreeClassPropertyTable(dvFreeClassPropertyTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Class.PropertyTable array.
----------------------------------------------------------------------------------------*/
void dvClassResizePropertyTables(
    dvClass Class,
    uint32 numPropertyTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvProperty);
    uint32 usedHeaderSize = (sizeof(dvClass) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvClass) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numPropertyTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvClassGetNumPropertyTable(Class) + usedHeaderSize, freeHeaderSize);
    dvProperty *dataPtr;

    if(numPropertyTables == 0) {
        if(dvClassGetNumPropertyTable(Class) != 0) {
            dvClassFreePropertyTables(Class);
        }
        return;
    }
    if(dvClassGetNumPropertyTable(Class) == 0) {
        dvClassAllocPropertyTables(Class, numPropertyTables);
        return;
    }
    freeSpace = dvAllocatedClassPropertyTable() - dvUsedClassPropertyTable();
    if(freeSpace < newSize) {
        allocMoreClassPropertyTables(newSize);
    }
    dataPtr = dvClassGetPropertyTables(Class) - usedHeaderSize;
    memcpy((void *)(dvClasss.PropertyTable + dvUsedClassPropertyTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xClass;
            for(xClass = (uint32)(dvUsedClassPropertyTable() + oldSize); xClass < dvUsedClassPropertyTable() + oldSize + newSize - oldSize; xClass++) {
                dvClasss.PropertyTable[xClass] = dvPropertyNull;
            }
        }
    }
    *(dvClass *)(void *)dataPtr = dvClassNull;
    *(uint32 *)(void *)(((dvClass *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeClassPropertyTable(dvFreeClassPropertyTable() + oldSize);
    dvClassSetPropertyTableIndex(Class, dvUsedClassPropertyTable() + usedHeaderSize);
    dvClassSetNumPropertyTable(Class, numPropertyTables);
    dvSetUsedClassPropertyTable(dvUsedClassPropertyTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the Class.SparsegroupTable heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactClassSparsegroupTables(void)
{
    uint32 elementSize = sizeof(dvSparsegroup);
    uint32 usedHeaderSize = (sizeof(dvClass) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvClass) + sizeof(uint32) + elementSize - 1)/elementSize;
    dvSparsegroup *toPtr = dvClasss.SparsegroupTable;
    dvSparsegroup *fromPtr = toPtr;
    dvClass Class;
    uint32 size;

    while(fromPtr < dvClasss.SparsegroupTable + dvUsedClassSparsegroupTable()) {
        Class = *(dvClass *)(void *)fromPtr;
        if(Class != dvClassNull) {
            /* Need to move it to toPtr */
            size = utMax(dvClassGetNumSparsegroupTable(Class) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvClassSetSparsegroupTableIndex(Class, toPtr - dvClasss.SparsegroupTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvClass *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedClassSparsegroupTable(toPtr - dvClasss.SparsegroupTable);
    dvSetFreeClassSparsegroupTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Class.SparsegroupTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreClassSparsegroupTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedClassSparsegroupTable() - dvUsedClassSparsegroupTable();

    if((dvFreeClassSparsegroupTable() << 2) > dvUsedClassSparsegroupTable()) {
        dvCompactClassSparsegroupTables();
        freeSpace = dvAllocatedClassSparsegroupTable() - dvUsedClassSparsegroupTable();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedClassSparsegroupTable(dvAllocatedClassSparsegroupTable() + spaceNeeded - freeSpace +
            (dvAllocatedClassSparsegroupTable() >> 1));
        utResizeArray(dvClasss.SparsegroupTable, dvAllocatedClassSparsegroupTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Class.SparsegroupTable array.
----------------------------------------------------------------------------------------*/
void dvClassAllocSparsegroupTables(
    dvClass Class,
    uint32 numSparsegroupTables)
{
    uint32 freeSpace = dvAllocatedClassSparsegroupTable() - dvUsedClassSparsegroupTable();
    uint32 elementSize = sizeof(dvSparsegroup);
    uint32 usedHeaderSize = (sizeof(dvClass) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvClass) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numSparsegroupTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvClassGetNumSparsegroupTable(Class) == 0);
#endif
    if(numSparsegroupTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreClassSparsegroupTables(spaceNeeded);
    }
    dvClassSetSparsegroupTableIndex(Class, dvUsedClassSparsegroupTable() + usedHeaderSize);
    dvClassSetNumSparsegroupTable(Class, numSparsegroupTables);
    *(dvClass *)(void *)(dvClasss.SparsegroupTable + dvUsedClassSparsegroupTable()) = Class;
    {
        uint32 xClass;
        for(xClass = (uint32)(dvClassGetSparsegroupTableIndex(Class)); xClass < dvClassGetSparsegroupTableIndex(Class) + numSparsegroupTables; xClass++) {
            dvClasss.SparsegroupTable[xClass] = dvSparsegroupNull;
        }
    }
    dvSetUsedClassSparsegroupTable(dvUsedClassSparsegroupTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvClassGetSparsegroupTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getClassSparsegroupTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvClass Class = dvIndex2Class((uint32)objectNumber);

    *numValues = dvClassGetNumSparsegroupTable(Class);
    return dvClassGetSparsegroupTables(Class);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvClassAllocSparsegroupTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocClassSparsegroupTables(
    uint64 objectNumber,
    uint32 numValues)
{
    dvClass Class = dvIndex2Class((uint32)objectNumber);

    dvClassSetSparsegroupTableIndex(Class, 0);
    dvClassSetNumSparsegroupTable(Class, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvClassAllocSparsegroupTables(Class, numValues);
    return dvClassGetSparsegroupTables(Class);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Class.SparsegroupTable array.
----------------------------------------------------------------------------------------*/
void dvClassFreeSparsegroupTables(
    dvClass Class)
{
    uint32 elementSize = sizeof(dvSparsegroup);
    uint32 usedHeaderSize = (sizeof(dvClass) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvClass) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvClassGetNumSparsegroupTable(Class) + usedHeaderSize, freeHeaderSize);
    dvSparsegroup *dataPtr = dvClassGetSparsegroupTables(Class) - usedHeaderSize;

    if(dvClassGetNumSparsegroupTable(Class) == 0) {
        return;
    }
    *(dvClass *)(void *)(dataPtr) = dvClassNull;
    *(uint32 *)(void *)(((dvClass *)(void *)dataPtr) + 1) = size;
    dvClassSetNumSparsegroupTable(Class, 0);
    dvSetFreeClassSparsegroupTable(dvFreeClassSparsegroupTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Class.SparsegroupTable array.
----------------------------------------------------------------------------------------*/
void dvClassResizeSparsegroupTables(
    dvClass Class,
    uint32 numSparsegroupTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(dvSparsegroup);
    uint32 usedHeaderSize = (sizeof(dvClass) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvClass) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numSparsegroupTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvClassGetNumSparsegroupTable(Class) + usedHeaderSize, freeHeaderSize);
    dvSparsegroup *dataPtr;

    if(numSparsegroupTables == 0) {
        if(dvClassGetNumSparsegroupTable(Class) != 0) {
            dvClassFreeSparsegroupTables(Class);
        }
        return;
    }
    if(dvClassGetNumSparsegroupTable(Class) == 0) {
        dvClassAllocSparsegroupTables(Class, numSparsegroupTables);
        return;
    }
    freeSpace = dvAllocatedClassSparsegroupTable() - dvUsedClassSparsegroupTable();
    if(freeSpace < newSize) {
        allocMoreClassSparsegroupTables(newSize);
    }
    dataPtr = dvClassGetSparsegroupTables(Class) - usedHeaderSize;
    memcpy((void *)(dvClasss.SparsegroupTable + dvUsedClassSparsegroupTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xClass;
            for(xClass = (uint32)(dvUsedClassSparsegroupTable() + oldSize); xClass < dvUsedClassSparsegroupTable() + oldSize + newSize - oldSize; xClass++) {
                dvClasss.SparsegroupTable[xClass] = dvSparsegroupNull;
            }
        }
    }
    *(dvClass *)(void *)dataPtr = dvClassNull;
    *(uint32 *)(void *)(((dvClass *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreeClassSparsegroupTable(dvFreeClassSparsegroupTable() + oldSize);
    dvClassSetSparsegroupTableIndex(Class, dvUsedClassSparsegroupTable() + usedHeaderSize);
    dvClassSetNumSparsegroupTable(Class, numSparsegroupTables);
    dvSetUsedClassSparsegroupTable(dvUsedClassSparsegroupTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Class.
----------------------------------------------------------------------------------------*/
void dvClassCopyProps(
    dvClass oldClass,
    dvClass newClass)
{
    dvClassSetMemoryStyle(newClass, dvClassGetMemoryStyle(oldClass));
    dvClassSetReferenceSize(newClass, dvClassGetReferenceSize(oldClass));
    dvClassSetGenerateArrayClass(newClass, dvClassGenerateArrayClass(oldClass));
    dvClassSetGenerateAttributes(newClass, dvClassGenerateAttributes(oldClass));
    dvClassSetSparse(newClass, dvClassSparse(oldClass));
    dvClassSetNumFields(newClass, dvClassGetNumFields(oldClass));
    dvClassSetNumber(newClass, dvClassGetNumber(oldClass));
}

/*----------------------------------------------------------------------------------------
  Return the integer equivalent for the bit fields in Class.
----------------------------------------------------------------------------------------*/
uint32 dvClassGetBitfield(
    dvClass _Class)
{
    uint32 bitfield = 0;
    uint8 xLevel = 0;

    bitfield |= dvClassGenerateArrayClass(_Class) << xLevel++;
    bitfield |= dvClassGenerateAttributes(_Class) << xLevel++;
    bitfield |= dvClassSparse(_Class) << xLevel++;
    return bitfield;
}

/*----------------------------------------------------------------------------------------
  Set bit fields in Class using bitfield.
----------------------------------------------------------------------------------------*/
void dvClassSetBitfield(
    dvClass _Class,
     uint32 bitfield)
{
    dvClassSetGenerateArrayClass(_Class, bitfield & 1);
    bitfield >>= 1;
    dvClassSetGenerateAttributes(_Class, bitfield & 1);
    bitfield >>= 1;
    dvClassSetSparse(_Class, bitfield & 1);
    bitfield >>= 1;
}

static void addClassPropertyToHashTable(dvClass Class, dvProperty _Property);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeClassPropertyHashTable(
    dvClass Class)
{
    dvProperty _Property;
    dvProperty *Propertys;
    uint32 numPropertys = dvClassGetNumPropertyTable(Class) << 1;

    if(numPropertys == 0) {
        numPropertys = 2;
        dvClassAllocPropertyTables(Class, 2);
    } else {
        dvClassResizePropertyTables(Class, numPropertys);
    }
    Propertys = dvClassGetPropertyTables(Class);
    /* Zero out the table */
    while(numPropertys-- != 0) {
        *Propertys++ = dvPropertyNull;
    }
    dvClassSetNumProperty(Class, 0);
    dvForeachClassProperty(Class, _Property) {
        if(dvPropertyGetSym(_Property) != utSymNull) {
            addClassPropertyToHashTable(Class, _Property);
        }
    } dvEndClassProperty;
}

/*----------------------------------------------------------------------------------------
  Add the Property to the Class.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addClassPropertyToHashTable(
    dvClass Class,
    dvProperty _Property)
{
    dvProperty nextProperty;
    uint32 index;

    if(dvClassGetNumProperty(Class) >= dvClassGetNumPropertyTable(Class)) {
        resizeClassPropertyHashTable(Class);
        return;
    }
    index = (dvClassGetNumPropertyTable(Class) - 1) & utSymGetHashValue(dvPropertyGetSym(_Property));
    nextProperty = dvClassGetiPropertyTable(Class, index);
    dvPropertySetNextTableClassProperty(_Property, nextProperty);
    dvClassSetiPropertyTable(Class, index, _Property);
    dvClassSetNumProperty(Class, dvClassGetNumProperty(Class) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Property from the hash table.
----------------------------------------------------------------------------------------*/
static void removeClassPropertyFromHashTable(
   dvClass Class,
   dvProperty _Property)
{
    uint32 index = (dvClassGetNumPropertyTable(Class) - 1) & utSymGetHashValue(dvPropertyGetSym(_Property));
    dvProperty prevProperty, nextProperty;
    
    nextProperty = dvClassGetiPropertyTable(Class, index);
    if(nextProperty == _Property) {
        dvClassSetiPropertyTable(Class, index, dvPropertyGetNextTableClassProperty(nextProperty));
    } else {
        do {
            prevProperty = nextProperty;
            nextProperty = dvPropertyGetNextTableClassProperty(nextProperty);
        } while(nextProperty != _Property);
        dvPropertySetNextTableClassProperty(prevProperty, dvPropertyGetNextTableClassProperty(_Property));
    }
    dvClassSetNumProperty(Class, dvClassGetNumProperty(Class) - 1);
    dvPropertySetNextTableClassProperty(_Property, dvPropertyNull);
}

/*----------------------------------------------------------------------------------------
  Find the Property from the Class and its hash key.
----------------------------------------------------------------------------------------*/
dvProperty dvClassFindProperty(
    dvClass Class,
    utSym Sym)
{
    uint32 mask = dvClassGetNumPropertyTable(Class) - 1;
    dvProperty _Property;

    if(mask + 1 != 0) {
        _Property = dvClassGetiPropertyTable(Class, utSymGetHashValue(Sym) & mask);
        while(_Property != dvPropertyNull) {
            if(dvPropertyGetSym(_Property) == Sym) {
                return _Property;
            }
            _Property = dvPropertyGetNextTableClassProperty(_Property);
        }
    }
    return dvPropertyNull;
}

/*----------------------------------------------------------------------------------------
  Find the Property from the Class and its name.
----------------------------------------------------------------------------------------*/
void dvClassRenameProperty(
    dvClass Class,
    dvProperty _Property,
    utSym sym)
{
    if(dvPropertyGetSym(_Property) != utSymNull) {
        removeClassPropertyFromHashTable(Class, _Property);
    }
    dvPropertySetSym(_Property, sym);
    if(sym != utSymNull) {
        addClassPropertyToHashTable(Class, _Property);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Property to the head of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassInsertProperty(
    dvClass Class,
    dvProperty _Property)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetClass(_Property) != dvClassNull) {
        utExit("Attempting to add Property to Class twice");
    }
#endif
    dvPropertySetNextClassProperty(_Property, dvClassGetFirstProperty(Class));
    if(dvClassGetFirstProperty(Class) != dvPropertyNull) {
        dvPropertySetPrevClassProperty(dvClassGetFirstProperty(Class), _Property);
    }
    dvClassSetFirstProperty(Class, _Property);
    dvPropertySetPrevClassProperty(_Property, dvPropertyNull);
    if(dvClassGetLastProperty(Class) == dvPropertyNull) {
        dvClassSetLastProperty(Class, _Property);
    }
    dvPropertySetClass(_Property, Class);
    if(dvPropertyGetSym(_Property) != utSymNull) {
        addClassPropertyToHashTable(Class, _Property);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Property to the end of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassAppendProperty(
    dvClass Class,
    dvProperty _Property)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetClass(_Property) != dvClassNull) {
        utExit("Attempting to add Property to Class twice");
    }
#endif
    dvPropertySetPrevClassProperty(_Property, dvClassGetLastProperty(Class));
    if(dvClassGetLastProperty(Class) != dvPropertyNull) {
        dvPropertySetNextClassProperty(dvClassGetLastProperty(Class), _Property);
    }
    dvClassSetLastProperty(Class, _Property);
    dvPropertySetNextClassProperty(_Property, dvPropertyNull);
    if(dvClassGetFirstProperty(Class) == dvPropertyNull) {
        dvClassSetFirstProperty(Class, _Property);
    }
    dvPropertySetClass(_Property, Class);
    if(dvPropertyGetSym(_Property) != utSymNull) {
        addClassPropertyToHashTable(Class, _Property);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Property to the Class after the previous Property.
----------------------------------------------------------------------------------------*/
void dvClassInsertAfterProperty(
    dvClass Class,
    dvProperty prevProperty,
    dvProperty _Property)
{
    dvProperty nextProperty = dvPropertyGetNextClassProperty(prevProperty);

#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetClass(_Property) != dvClassNull) {
        utExit("Attempting to add Property to Class twice");
    }
#endif
    dvPropertySetNextClassProperty(_Property, nextProperty);
    dvPropertySetNextClassProperty(prevProperty, _Property);
    dvPropertySetPrevClassProperty(_Property, prevProperty);
    if(nextProperty != dvPropertyNull) {
        dvPropertySetPrevClassProperty(nextProperty, _Property);
    }
    if(dvClassGetLastProperty(Class) == prevProperty) {
        dvClassSetLastProperty(Class, _Property);
    }
    dvPropertySetClass(_Property, Class);
    if(dvPropertyGetSym(_Property) != utSymNull) {
        addClassPropertyToHashTable(Class, _Property);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Property from the Class.
----------------------------------------------------------------------------------------*/
void dvClassRemoveProperty(
    dvClass Class,
    dvProperty _Property)
{
    dvProperty pProperty, nProperty;

#if defined(DD_DEBUG)
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetClass(_Property) != dvClassNull && dvPropertyGetClass(_Property) != Class) {
        utExit("Delete Property from non-owning Class");
    }
#endif
    nProperty = dvPropertyGetNextClassProperty(_Property);
    pProperty = dvPropertyGetPrevClassProperty(_Property);
    if(pProperty != dvPropertyNull) {
        dvPropertySetNextClassProperty(pProperty, nProperty);
    } else if(dvClassGetFirstProperty(Class) == _Property) {
        dvClassSetFirstProperty(Class, nProperty);
    }
    if(nProperty != dvPropertyNull) {
        dvPropertySetPrevClassProperty(nProperty, pProperty);
    } else if(dvClassGetLastProperty(Class) == _Property) {
        dvClassSetLastProperty(Class, pProperty);
    }
    dvPropertySetNextClassProperty(_Property, dvPropertyNull);
    dvPropertySetPrevClassProperty(_Property, dvPropertyNull);
    dvPropertySetClass(_Property, dvClassNull);
    if(dvPropertyGetSym(_Property) != utSymNull) {
        removeClassPropertyFromHashTable(Class, _Property);
    }
}

static void addClassSparsegroupToHashTable(dvClass Class, dvSparsegroup _Sparsegroup);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeClassSparsegroupHashTable(
    dvClass Class)
{
    dvSparsegroup _Sparsegroup;
    dvSparsegroup *Sparsegroups;
    uint32 numSparsegroups = dvClassGetNumSparsegroupTable(Class) << 1;

    if(numSparsegroups == 0) {
        numSparsegroups = 2;
        dvClassAllocSparsegroupTables(Class, 2);
    } else {
        dvClassResizeSparsegroupTables(Class, numSparsegroups);
    }
    Sparsegroups = dvClassGetSparsegroupTables(Class);
    /* Zero out the table */
    while(numSparsegroups-- != 0) {
        *Sparsegroups++ = dvSparsegroupNull;
    }
    dvClassSetNumSparsegroup(Class, 0);
    dvForeachClassSparsegroup(Class, _Sparsegroup) {
        if(dvSparsegroupGetSym(_Sparsegroup) != utSymNull) {
            addClassSparsegroupToHashTable(Class, _Sparsegroup);
        }
    } dvEndClassSparsegroup;
}

/*----------------------------------------------------------------------------------------
  Add the Sparsegroup to the Class.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addClassSparsegroupToHashTable(
    dvClass Class,
    dvSparsegroup _Sparsegroup)
{
    dvSparsegroup nextSparsegroup;
    uint32 index;

    if(dvClassGetNumSparsegroup(Class) >= dvClassGetNumSparsegroupTable(Class)) {
        resizeClassSparsegroupHashTable(Class);
        return;
    }
    index = (dvClassGetNumSparsegroupTable(Class) - 1) & utSymGetHashValue(dvSparsegroupGetSym(_Sparsegroup));
    nextSparsegroup = dvClassGetiSparsegroupTable(Class, index);
    dvSparsegroupSetNextTableClassSparsegroup(_Sparsegroup, nextSparsegroup);
    dvClassSetiSparsegroupTable(Class, index, _Sparsegroup);
    dvClassSetNumSparsegroup(Class, dvClassGetNumSparsegroup(Class) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Sparsegroup from the hash table.
----------------------------------------------------------------------------------------*/
static void removeClassSparsegroupFromHashTable(
   dvClass Class,
   dvSparsegroup _Sparsegroup)
{
    uint32 index = (dvClassGetNumSparsegroupTable(Class) - 1) & utSymGetHashValue(dvSparsegroupGetSym(_Sparsegroup));
    dvSparsegroup prevSparsegroup, nextSparsegroup;
    
    nextSparsegroup = dvClassGetiSparsegroupTable(Class, index);
    if(nextSparsegroup == _Sparsegroup) {
        dvClassSetiSparsegroupTable(Class, index, dvSparsegroupGetNextTableClassSparsegroup(nextSparsegroup));
    } else {
        do {
            prevSparsegroup = nextSparsegroup;
            nextSparsegroup = dvSparsegroupGetNextTableClassSparsegroup(nextSparsegroup);
        } while(nextSparsegroup != _Sparsegroup);
        dvSparsegroupSetNextTableClassSparsegroup(prevSparsegroup, dvSparsegroupGetNextTableClassSparsegroup(_Sparsegroup));
    }
    dvClassSetNumSparsegroup(Class, dvClassGetNumSparsegroup(Class) - 1);
    dvSparsegroupSetNextTableClassSparsegroup(_Sparsegroup, dvSparsegroupNull);
}

/*----------------------------------------------------------------------------------------
  Find the Sparsegroup from the Class and its hash key.
----------------------------------------------------------------------------------------*/
dvSparsegroup dvClassFindSparsegroup(
    dvClass Class,
    utSym Sym)
{
    uint32 mask = dvClassGetNumSparsegroupTable(Class) - 1;
    dvSparsegroup _Sparsegroup;

    if(mask + 1 != 0) {
        _Sparsegroup = dvClassGetiSparsegroupTable(Class, utSymGetHashValue(Sym) & mask);
        while(_Sparsegroup != dvSparsegroupNull) {
            if(dvSparsegroupGetSym(_Sparsegroup) == Sym) {
                return _Sparsegroup;
            }
            _Sparsegroup = dvSparsegroupGetNextTableClassSparsegroup(_Sparsegroup);
        }
    }
    return dvSparsegroupNull;
}

/*----------------------------------------------------------------------------------------
  Find the Sparsegroup from the Class and its name.
----------------------------------------------------------------------------------------*/
void dvClassRenameSparsegroup(
    dvClass Class,
    dvSparsegroup _Sparsegroup,
    utSym sym)
{
    if(dvSparsegroupGetSym(_Sparsegroup) != utSymNull) {
        removeClassSparsegroupFromHashTable(Class, _Sparsegroup);
    }
    dvSparsegroupSetSym(_Sparsegroup, sym);
    if(sym != utSymNull) {
        addClassSparsegroupToHashTable(Class, _Sparsegroup);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Sparsegroup to the head of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassInsertSparsegroup(
    dvClass Class,
    dvSparsegroup _Sparsegroup)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Sparsegroup == dvSparsegroupNull) {
        utExit("Non-existent Sparsegroup");
    }
    if(dvSparsegroupGetClass(_Sparsegroup) != dvClassNull) {
        utExit("Attempting to add Sparsegroup to Class twice");
    }
#endif
    dvSparsegroupSetNextClassSparsegroup(_Sparsegroup, dvClassGetFirstSparsegroup(Class));
    if(dvClassGetFirstSparsegroup(Class) != dvSparsegroupNull) {
        dvSparsegroupSetPrevClassSparsegroup(dvClassGetFirstSparsegroup(Class), _Sparsegroup);
    }
    dvClassSetFirstSparsegroup(Class, _Sparsegroup);
    dvSparsegroupSetPrevClassSparsegroup(_Sparsegroup, dvSparsegroupNull);
    if(dvClassGetLastSparsegroup(Class) == dvSparsegroupNull) {
        dvClassSetLastSparsegroup(Class, _Sparsegroup);
    }
    dvSparsegroupSetClass(_Sparsegroup, Class);
    if(dvSparsegroupGetSym(_Sparsegroup) != utSymNull) {
        addClassSparsegroupToHashTable(Class, _Sparsegroup);
    }
}

/*----------------------------------------------------------------------------------------
  Add the Sparsegroup to the end of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassAppendSparsegroup(
    dvClass Class,
    dvSparsegroup _Sparsegroup)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Sparsegroup == dvSparsegroupNull) {
        utExit("Non-existent Sparsegroup");
    }
    if(dvSparsegroupGetClass(_Sparsegroup) != dvClassNull) {
        utExit("Attempting to add Sparsegroup to Class twice");
    }
#endif
    dvSparsegroupSetPrevClassSparsegroup(_Sparsegroup, dvClassGetLastSparsegroup(Class));
    if(dvClassGetLastSparsegroup(Class) != dvSparsegroupNull) {
        dvSparsegroupSetNextClassSparsegroup(dvClassGetLastSparsegroup(Class), _Sparsegroup);
    }
    dvClassSetLastSparsegroup(Class, _Sparsegroup);
    dvSparsegroupSetNextClassSparsegroup(_Sparsegroup, dvSparsegroupNull);
    if(dvClassGetFirstSparsegroup(Class) == dvSparsegroupNull) {
        dvClassSetFirstSparsegroup(Class, _Sparsegroup);
    }
    dvSparsegroupSetClass(_Sparsegroup, Class);
    if(dvSparsegroupGetSym(_Sparsegroup) != utSymNull) {
        addClassSparsegroupToHashTable(Class, _Sparsegroup);
    }
}

/*----------------------------------------------------------------------------------------
  Insert the Sparsegroup to the Class after the previous Sparsegroup.
----------------------------------------------------------------------------------------*/
void dvClassInsertAfterSparsegroup(
    dvClass Class,
    dvSparsegroup prevSparsegroup,
    dvSparsegroup _Sparsegroup)
{
    dvSparsegroup nextSparsegroup = dvSparsegroupGetNextClassSparsegroup(prevSparsegroup);

#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Sparsegroup == dvSparsegroupNull) {
        utExit("Non-existent Sparsegroup");
    }
    if(dvSparsegroupGetClass(_Sparsegroup) != dvClassNull) {
        utExit("Attempting to add Sparsegroup to Class twice");
    }
#endif
    dvSparsegroupSetNextClassSparsegroup(_Sparsegroup, nextSparsegroup);
    dvSparsegroupSetNextClassSparsegroup(prevSparsegroup, _Sparsegroup);
    dvSparsegroupSetPrevClassSparsegroup(_Sparsegroup, prevSparsegroup);
    if(nextSparsegroup != dvSparsegroupNull) {
        dvSparsegroupSetPrevClassSparsegroup(nextSparsegroup, _Sparsegroup);
    }
    if(dvClassGetLastSparsegroup(Class) == prevSparsegroup) {
        dvClassSetLastSparsegroup(Class, _Sparsegroup);
    }
    dvSparsegroupSetClass(_Sparsegroup, Class);
    if(dvSparsegroupGetSym(_Sparsegroup) != utSymNull) {
        addClassSparsegroupToHashTable(Class, _Sparsegroup);
    }
}

/*----------------------------------------------------------------------------------------
 Remove the Sparsegroup from the Class.
----------------------------------------------------------------------------------------*/
void dvClassRemoveSparsegroup(
    dvClass Class,
    dvSparsegroup _Sparsegroup)
{
    dvSparsegroup pSparsegroup, nSparsegroup;

#if defined(DD_DEBUG)
    if(_Sparsegroup == dvSparsegroupNull) {
        utExit("Non-existent Sparsegroup");
    }
    if(dvSparsegroupGetClass(_Sparsegroup) != dvClassNull && dvSparsegroupGetClass(_Sparsegroup) != Class) {
        utExit("Delete Sparsegroup from non-owning Class");
    }
#endif
    nSparsegroup = dvSparsegroupGetNextClassSparsegroup(_Sparsegroup);
    pSparsegroup = dvSparsegroupGetPrevClassSparsegroup(_Sparsegroup);
    if(pSparsegroup != dvSparsegroupNull) {
        dvSparsegroupSetNextClassSparsegroup(pSparsegroup, nSparsegroup);
    } else if(dvClassGetFirstSparsegroup(Class) == _Sparsegroup) {
        dvClassSetFirstSparsegroup(Class, nSparsegroup);
    }
    if(nSparsegroup != dvSparsegroupNull) {
        dvSparsegroupSetPrevClassSparsegroup(nSparsegroup, pSparsegroup);
    } else if(dvClassGetLastSparsegroup(Class) == _Sparsegroup) {
        dvClassSetLastSparsegroup(Class, pSparsegroup);
    }
    dvSparsegroupSetNextClassSparsegroup(_Sparsegroup, dvSparsegroupNull);
    dvSparsegroupSetPrevClassSparsegroup(_Sparsegroup, dvSparsegroupNull);
    dvSparsegroupSetClass(_Sparsegroup, dvClassNull);
    if(dvSparsegroupGetSym(_Sparsegroup) != utSymNull) {
        removeClassSparsegroupFromHashTable(Class, _Sparsegroup);
    }
}

/*----------------------------------------------------------------------------------------
  Add the DerivedClass to the head of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassInsertDerivedClass(
    dvClass Class,
    dvClass _Class)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(dvClassGetBaseClass(_Class) != dvClassNull) {
        utExit("Attempting to add Class to Class twice");
    }
#endif
    dvClassSetNextClassDerivedClass(_Class, dvClassGetFirstDerivedClass(Class));
    dvClassSetFirstDerivedClass(Class, _Class);
    if(dvClassGetLastDerivedClass(Class) == dvClassNull) {
        dvClassSetLastDerivedClass(Class, _Class);
    }
    dvClassSetBaseClass(_Class, Class);
}

/*----------------------------------------------------------------------------------------
  Add the DerivedClass to the end of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassAppendDerivedClass(
    dvClass Class,
    dvClass _Class)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(dvClassGetBaseClass(_Class) != dvClassNull) {
        utExit("Attempting to add Class to Class twice");
    }
#endif
    if(dvClassGetLastDerivedClass(Class) != dvClassNull) {
        dvClassSetNextClassDerivedClass(dvClassGetLastDerivedClass(Class), _Class);
    } else {
        dvClassSetFirstDerivedClass(Class, _Class);
    }
    dvClassSetLastDerivedClass(Class, _Class);
    dvClassSetNextClassDerivedClass(_Class, dvClassNull);
    dvClassSetBaseClass(_Class, Class);
}

/*----------------------------------------------------------------------------------------
  Insert the DerivedClass to the Class after the previous DerivedClass.
----------------------------------------------------------------------------------------*/
void dvClassInsertAfterDerivedClass(
    dvClass Class,
    dvClass prevClass,
    dvClass _Class)
{
    dvClass nextClass = dvClassGetNextClassDerivedClass(prevClass);

#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(dvClassGetBaseClass(_Class) != dvClassNull) {
        utExit("Attempting to add Class to Class twice");
    }
#endif
    dvClassSetNextClassDerivedClass(_Class, nextClass);
    dvClassSetNextClassDerivedClass(prevClass, _Class);
    if(dvClassGetLastDerivedClass(Class) == prevClass) {
        dvClassSetLastDerivedClass(Class, _Class);
    }
    dvClassSetBaseClass(_Class, Class);
}

/*----------------------------------------------------------------------------------------
 Remove the DerivedClass from the Class.
----------------------------------------------------------------------------------------*/
void dvClassRemoveDerivedClass(
    dvClass Class,
    dvClass _Class)
{
    dvClass pClass, nClass;

#if defined(DD_DEBUG)
    if(_Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(dvClassGetBaseClass(_Class) != dvClassNull && dvClassGetBaseClass(_Class) != Class) {
        utExit("Delete Class from non-owning Class");
    }
#endif
    pClass = dvClassNull;
    for(nClass = dvClassGetFirstDerivedClass(Class); nClass != dvClassNull && nClass != _Class;
            nClass = dvClassGetNextClassDerivedClass(nClass)) {
        pClass = nClass;
    }
    if(pClass != dvClassNull) {
        dvClassSetNextClassDerivedClass(pClass, dvClassGetNextClassDerivedClass(_Class));
    } else {
        dvClassSetFirstDerivedClass(Class, dvClassGetNextClassDerivedClass(_Class));
    }
    dvClassSetNextClassDerivedClass(_Class, dvClassNull);
    if(dvClassGetLastDerivedClass(Class) == _Class) {
        dvClassSetLastDerivedClass(Class, pClass);
    }
    dvClassSetBaseClass(_Class, dvClassNull);
}

/*----------------------------------------------------------------------------------------
  Add the ChildRelationship to the head of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassInsertChildRelationship(
    dvClass Class,
    dvRelationship _Relationship)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetParentClass(_Relationship) != dvClassNull) {
        utExit("Attempting to add Relationship to Class twice");
    }
#endif
    dvRelationshipSetNextClassChildRelationship(_Relationship, dvClassGetFirstChildRelationship(Class));
    dvClassSetFirstChildRelationship(Class, _Relationship);
    if(dvClassGetLastChildRelationship(Class) == dvRelationshipNull) {
        dvClassSetLastChildRelationship(Class, _Relationship);
    }
    dvRelationshipSetParentClass(_Relationship, Class);
}

/*----------------------------------------------------------------------------------------
  Add the ChildRelationship to the end of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassAppendChildRelationship(
    dvClass Class,
    dvRelationship _Relationship)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetParentClass(_Relationship) != dvClassNull) {
        utExit("Attempting to add Relationship to Class twice");
    }
#endif
    if(dvClassGetLastChildRelationship(Class) != dvRelationshipNull) {
        dvRelationshipSetNextClassChildRelationship(dvClassGetLastChildRelationship(Class), _Relationship);
    } else {
        dvClassSetFirstChildRelationship(Class, _Relationship);
    }
    dvClassSetLastChildRelationship(Class, _Relationship);
    dvRelationshipSetNextClassChildRelationship(_Relationship, dvRelationshipNull);
    dvRelationshipSetParentClass(_Relationship, Class);
}

/*----------------------------------------------------------------------------------------
  Insert the ChildRelationship to the Class after the previous ChildRelationship.
----------------------------------------------------------------------------------------*/
void dvClassInsertAfterChildRelationship(
    dvClass Class,
    dvRelationship prevRelationship,
    dvRelationship _Relationship)
{
    dvRelationship nextRelationship = dvRelationshipGetNextClassChildRelationship(prevRelationship);

#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetParentClass(_Relationship) != dvClassNull) {
        utExit("Attempting to add Relationship to Class twice");
    }
#endif
    dvRelationshipSetNextClassChildRelationship(_Relationship, nextRelationship);
    dvRelationshipSetNextClassChildRelationship(prevRelationship, _Relationship);
    if(dvClassGetLastChildRelationship(Class) == prevRelationship) {
        dvClassSetLastChildRelationship(Class, _Relationship);
    }
    dvRelationshipSetParentClass(_Relationship, Class);
}

/*----------------------------------------------------------------------------------------
 Remove the ChildRelationship from the Class.
----------------------------------------------------------------------------------------*/
void dvClassRemoveChildRelationship(
    dvClass Class,
    dvRelationship _Relationship)
{
    dvRelationship pRelationship, nRelationship;

#if defined(DD_DEBUG)
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetParentClass(_Relationship) != dvClassNull && dvRelationshipGetParentClass(_Relationship) != Class) {
        utExit("Delete Relationship from non-owning Class");
    }
#endif
    pRelationship = dvRelationshipNull;
    for(nRelationship = dvClassGetFirstChildRelationship(Class); nRelationship != dvRelationshipNull && nRelationship != _Relationship;
            nRelationship = dvRelationshipGetNextClassChildRelationship(nRelationship)) {
        pRelationship = nRelationship;
    }
    if(pRelationship != dvRelationshipNull) {
        dvRelationshipSetNextClassChildRelationship(pRelationship, dvRelationshipGetNextClassChildRelationship(_Relationship));
    } else {
        dvClassSetFirstChildRelationship(Class, dvRelationshipGetNextClassChildRelationship(_Relationship));
    }
    dvRelationshipSetNextClassChildRelationship(_Relationship, dvRelationshipNull);
    if(dvClassGetLastChildRelationship(Class) == _Relationship) {
        dvClassSetLastChildRelationship(Class, pRelationship);
    }
    dvRelationshipSetParentClass(_Relationship, dvClassNull);
}

/*----------------------------------------------------------------------------------------
  Add the ParentRelationship to the head of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassInsertParentRelationship(
    dvClass Class,
    dvRelationship _Relationship)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetChildClass(_Relationship) != dvClassNull) {
        utExit("Attempting to add Relationship to Class twice");
    }
#endif
    dvRelationshipSetNextClassParentRelationship(_Relationship, dvClassGetFirstParentRelationship(Class));
    dvClassSetFirstParentRelationship(Class, _Relationship);
    if(dvClassGetLastParentRelationship(Class) == dvRelationshipNull) {
        dvClassSetLastParentRelationship(Class, _Relationship);
    }
    dvRelationshipSetChildClass(_Relationship, Class);
}

/*----------------------------------------------------------------------------------------
  Add the ParentRelationship to the end of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassAppendParentRelationship(
    dvClass Class,
    dvRelationship _Relationship)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetChildClass(_Relationship) != dvClassNull) {
        utExit("Attempting to add Relationship to Class twice");
    }
#endif
    if(dvClassGetLastParentRelationship(Class) != dvRelationshipNull) {
        dvRelationshipSetNextClassParentRelationship(dvClassGetLastParentRelationship(Class), _Relationship);
    } else {
        dvClassSetFirstParentRelationship(Class, _Relationship);
    }
    dvClassSetLastParentRelationship(Class, _Relationship);
    dvRelationshipSetNextClassParentRelationship(_Relationship, dvRelationshipNull);
    dvRelationshipSetChildClass(_Relationship, Class);
}

/*----------------------------------------------------------------------------------------
  Insert the ParentRelationship to the Class after the previous ParentRelationship.
----------------------------------------------------------------------------------------*/
void dvClassInsertAfterParentRelationship(
    dvClass Class,
    dvRelationship prevRelationship,
    dvRelationship _Relationship)
{
    dvRelationship nextRelationship = dvRelationshipGetNextClassParentRelationship(prevRelationship);

#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetChildClass(_Relationship) != dvClassNull) {
        utExit("Attempting to add Relationship to Class twice");
    }
#endif
    dvRelationshipSetNextClassParentRelationship(_Relationship, nextRelationship);
    dvRelationshipSetNextClassParentRelationship(prevRelationship, _Relationship);
    if(dvClassGetLastParentRelationship(Class) == prevRelationship) {
        dvClassSetLastParentRelationship(Class, _Relationship);
    }
    dvRelationshipSetChildClass(_Relationship, Class);
}

/*----------------------------------------------------------------------------------------
 Remove the ParentRelationship from the Class.
----------------------------------------------------------------------------------------*/
void dvClassRemoveParentRelationship(
    dvClass Class,
    dvRelationship _Relationship)
{
    dvRelationship pRelationship, nRelationship;

#if defined(DD_DEBUG)
    if(_Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(dvRelationshipGetChildClass(_Relationship) != dvClassNull && dvRelationshipGetChildClass(_Relationship) != Class) {
        utExit("Delete Relationship from non-owning Class");
    }
#endif
    pRelationship = dvRelationshipNull;
    for(nRelationship = dvClassGetFirstParentRelationship(Class); nRelationship != dvRelationshipNull && nRelationship != _Relationship;
            nRelationship = dvRelationshipGetNextClassParentRelationship(nRelationship)) {
        pRelationship = nRelationship;
    }
    if(pRelationship != dvRelationshipNull) {
        dvRelationshipSetNextClassParentRelationship(pRelationship, dvRelationshipGetNextClassParentRelationship(_Relationship));
    } else {
        dvClassSetFirstParentRelationship(Class, dvRelationshipGetNextClassParentRelationship(_Relationship));
    }
    dvRelationshipSetNextClassParentRelationship(_Relationship, dvRelationshipNull);
    if(dvClassGetLastParentRelationship(Class) == _Relationship) {
        dvClassSetLastParentRelationship(Class, pRelationship);
    }
    dvRelationshipSetChildClass(_Relationship, dvClassNull);
}

/*----------------------------------------------------------------------------------------
  Add the Union to the head of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassInsertUnion(
    dvClass Class,
    dvUnion _Union)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Union == dvUnionNull) {
        utExit("Non-existent Union");
    }
    if(dvUnionGetClass(_Union) != dvClassNull) {
        utExit("Attempting to add Union to Class twice");
    }
#endif
    dvUnionSetNextClassUnion(_Union, dvClassGetFirstUnion(Class));
    dvClassSetFirstUnion(Class, _Union);
    if(dvClassGetLastUnion(Class) == dvUnionNull) {
        dvClassSetLastUnion(Class, _Union);
    }
    dvUnionSetClass(_Union, Class);
}

/*----------------------------------------------------------------------------------------
  Add the Union to the end of the list on the Class.
----------------------------------------------------------------------------------------*/
void dvClassAppendUnion(
    dvClass Class,
    dvUnion _Union)
{
#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Union == dvUnionNull) {
        utExit("Non-existent Union");
    }
    if(dvUnionGetClass(_Union) != dvClassNull) {
        utExit("Attempting to add Union to Class twice");
    }
#endif
    if(dvClassGetLastUnion(Class) != dvUnionNull) {
        dvUnionSetNextClassUnion(dvClassGetLastUnion(Class), _Union);
    } else {
        dvClassSetFirstUnion(Class, _Union);
    }
    dvClassSetLastUnion(Class, _Union);
    dvUnionSetNextClassUnion(_Union, dvUnionNull);
    dvUnionSetClass(_Union, Class);
}

/*----------------------------------------------------------------------------------------
  Insert the Union to the Class after the previous Union.
----------------------------------------------------------------------------------------*/
void dvClassInsertAfterUnion(
    dvClass Class,
    dvUnion prevUnion,
    dvUnion _Union)
{
    dvUnion nextUnion = dvUnionGetNextClassUnion(prevUnion);

#if defined(DD_DEBUG)
    if(Class == dvClassNull) {
        utExit("Non-existent Class");
    }
    if(_Union == dvUnionNull) {
        utExit("Non-existent Union");
    }
    if(dvUnionGetClass(_Union) != dvClassNull) {
        utExit("Attempting to add Union to Class twice");
    }
#endif
    dvUnionSetNextClassUnion(_Union, nextUnion);
    dvUnionSetNextClassUnion(prevUnion, _Union);
    if(dvClassGetLastUnion(Class) == prevUnion) {
        dvClassSetLastUnion(Class, _Union);
    }
    dvUnionSetClass(_Union, Class);
}

/*----------------------------------------------------------------------------------------
 Remove the Union from the Class.
----------------------------------------------------------------------------------------*/
void dvClassRemoveUnion(
    dvClass Class,
    dvUnion _Union)
{
    dvUnion pUnion, nUnion;

#if defined(DD_DEBUG)
    if(_Union == dvUnionNull) {
        utExit("Non-existent Union");
    }
    if(dvUnionGetClass(_Union) != dvClassNull && dvUnionGetClass(_Union) != Class) {
        utExit("Delete Union from non-owning Class");
    }
#endif
    pUnion = dvUnionNull;
    for(nUnion = dvClassGetFirstUnion(Class); nUnion != dvUnionNull && nUnion != _Union;
            nUnion = dvUnionGetNextClassUnion(nUnion)) {
        pUnion = nUnion;
    }
    if(pUnion != dvUnionNull) {
        dvUnionSetNextClassUnion(pUnion, dvUnionGetNextClassUnion(_Union));
    } else {
        dvClassSetFirstUnion(Class, dvUnionGetNextClassUnion(_Union));
    }
    dvUnionSetNextClassUnion(_Union, dvUnionNull);
    if(dvClassGetLastUnion(Class) == _Union) {
        dvClassSetLastUnion(Class, pUnion);
    }
    dvUnionSetClass(_Union, dvClassNull);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowClass(
    dvClass Class)
{
    utDatabaseShowObject("dv", "Class", dvClass2Index(Class));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocProperty(void)
{
    dvProperty Property = dvPropertyAlloc();

    return dvProperty2Index(Property);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Property.
----------------------------------------------------------------------------------------*/
static void allocPropertys(void)
{
    dvSetAllocatedProperty(2);
    dvSetUsedProperty(0);
    dvPropertys.Sym = utNewA(utSym, (dvAllocatedProperty()));
    dvPropertys.Type = utNewA(dvPropertyType, (dvAllocatedProperty()));
    dvPropertys.Array = utNewA(uint8, (dvAllocatedProperty() + 7) >> 3);
    dvPropertys.Cascade = utNewA(uint8, (dvAllocatedProperty() + 7) >> 3);
    dvPropertys.Sparse = utNewA(uint8, (dvAllocatedProperty() + 7) >> 3);
    dvPropertys.Expanded = utNewA(uint8, (dvAllocatedProperty() + 7) >> 3);
    dvPropertys.FieldNumber = utNewA(uint32, (dvAllocatedProperty()));
    dvPropertys.FirstElementProp = utNewA(dvProperty, (dvAllocatedProperty()));
    dvPropertys.NumElementsProp = utNewA(dvProperty, (dvAllocatedProperty()));
    dvPropertys.Hidden = utNewA(uint8, (dvAllocatedProperty() + 7) >> 3);
    dvPropertys.InitializerIndex = utNewA(uint32, (dvAllocatedProperty()));
    dvPropertys.NumInitializer = utNewA(uint32, (dvAllocatedProperty()));
    dvSetUsedPropertyInitializer(0);
    dvSetAllocatedPropertyInitializer(2);
    dvSetFreePropertyInitializer(0);
    dvPropertys.Initializer = utNewA(char, dvAllocatedPropertyInitializer());
    dvPropertys.FixedSize = utNewA(uint8, (dvAllocatedProperty() + 7) >> 3);
    dvPropertys.IndexIndex = utNewA(uint32, (dvAllocatedProperty()));
    dvPropertys.NumIndex = utNewA(uint32, (dvAllocatedProperty()));
    dvSetUsedPropertyIndex(0);
    dvSetAllocatedPropertyIndex(2);
    dvSetFreePropertyIndex(0);
    dvPropertys.Index = utNewA(char, dvAllocatedPropertyIndex());
    dvPropertys.Line = utNewA(uint32, (dvAllocatedProperty()));
    dvPropertys.Class = utNewA(dvClass, (dvAllocatedProperty()));
    dvPropertys.NextClassProperty = utNewA(dvProperty, (dvAllocatedProperty()));
    dvPropertys.PrevClassProperty = utNewA(dvProperty, (dvAllocatedProperty()));
    dvPropertys.NextTableClassProperty = utNewA(dvProperty, (dvAllocatedProperty()));
    dvPropertys.FirstCase = utNewA(dvCase, (dvAllocatedProperty()));
    dvPropertys.LastCase = utNewA(dvCase, (dvAllocatedProperty()));
    dvPropertys.FirstKey = utNewA(dvKey, (dvAllocatedProperty()));
    dvPropertys.LastKey = utNewA(dvKey, (dvAllocatedProperty()));
    dvPropertys.Sparsegroup = utNewA(dvSparsegroup, (dvAllocatedProperty()));
    dvPropertys.NextSparsegroupProperty = utNewA(dvProperty, (dvAllocatedProperty()));
    dvPropertys.Relationship = utNewA(dvRelationship, (dvAllocatedProperty()));
    dvPropertys.NextRelationshipProperty = utNewA(dvProperty, (dvAllocatedProperty()));
    dvPropertys.Union = utNewA(dvUnion, (dvAllocatedProperty()));
    dvPropertys.NextUnionProperty = utNewA(dvProperty, (dvAllocatedProperty()));
    dvPropertys.union1 = utNewA(dvPropertyUnion1, dvAllocatedProperty());
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Property.
----------------------------------------------------------------------------------------*/
static void reallocPropertys(
    uint32 newSize)
{
    utResizeArray(dvPropertys.Sym, (newSize));
    utResizeArray(dvPropertys.Type, (newSize));
    utResizeArray(dvPropertys.Array, (newSize + 7) >> 3);
    utResizeArray(dvPropertys.Cascade, (newSize + 7) >> 3);
    utResizeArray(dvPropertys.Sparse, (newSize + 7) >> 3);
    utResizeArray(dvPropertys.Expanded, (newSize + 7) >> 3);
    utResizeArray(dvPropertys.FieldNumber, (newSize));
    utResizeArray(dvPropertys.FirstElementProp, (newSize));
    utResizeArray(dvPropertys.NumElementsProp, (newSize));
    utResizeArray(dvPropertys.Hidden, (newSize + 7) >> 3);
    utResizeArray(dvPropertys.InitializerIndex, (newSize));
    utResizeArray(dvPropertys.NumInitializer, (newSize));
    utResizeArray(dvPropertys.FixedSize, (newSize + 7) >> 3);
    utResizeArray(dvPropertys.IndexIndex, (newSize));
    utResizeArray(dvPropertys.NumIndex, (newSize));
    utResizeArray(dvPropertys.Line, (newSize));
    utResizeArray(dvPropertys.Class, (newSize));
    utResizeArray(dvPropertys.NextClassProperty, (newSize));
    utResizeArray(dvPropertys.PrevClassProperty, (newSize));
    utResizeArray(dvPropertys.NextTableClassProperty, (newSize));
    utResizeArray(dvPropertys.FirstCase, (newSize));
    utResizeArray(dvPropertys.LastCase, (newSize));
    utResizeArray(dvPropertys.FirstKey, (newSize));
    utResizeArray(dvPropertys.LastKey, (newSize));
    utResizeArray(dvPropertys.Sparsegroup, (newSize));
    utResizeArray(dvPropertys.NextSparsegroupProperty, (newSize));
    utResizeArray(dvPropertys.Relationship, (newSize));
    utResizeArray(dvPropertys.NextRelationshipProperty, (newSize));
    utResizeArray(dvPropertys.Union, (newSize));
    utResizeArray(dvPropertys.NextUnionProperty, (newSize));
    utResizeArray(dvPropertys.union1, newSize);
    dvSetAllocatedProperty(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Propertys.
----------------------------------------------------------------------------------------*/
void dvPropertyAllocMore(void)
{
    reallocPropertys((uint32)(dvAllocatedProperty() + (dvAllocatedProperty() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the Property.Initializer heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactPropertyInitializers(void)
{
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvProperty) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvProperty) + sizeof(uint32) + elementSize - 1)/elementSize;
    char *toPtr = dvPropertys.Initializer;
    char *fromPtr = toPtr;
    dvProperty Property;
    uint32 size;

    while(fromPtr < dvPropertys.Initializer + dvUsedPropertyInitializer()) {
        Property = *(dvProperty *)(void *)fromPtr;
        if(Property != dvPropertyNull) {
            /* Need to move it to toPtr */
            size = utMax(dvPropertyGetNumInitializer(Property) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvPropertySetInitializerIndex(Property, toPtr - dvPropertys.Initializer + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvProperty *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedPropertyInitializer(toPtr - dvPropertys.Initializer);
    dvSetFreePropertyInitializer(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Property.Initializer heap.
----------------------------------------------------------------------------------------*/
static void allocMorePropertyInitializers(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedPropertyInitializer() - dvUsedPropertyInitializer();

    if((dvFreePropertyInitializer() << 2) > dvUsedPropertyInitializer()) {
        dvCompactPropertyInitializers();
        freeSpace = dvAllocatedPropertyInitializer() - dvUsedPropertyInitializer();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedPropertyInitializer(dvAllocatedPropertyInitializer() + spaceNeeded - freeSpace +
            (dvAllocatedPropertyInitializer() >> 1));
        utResizeArray(dvPropertys.Initializer, dvAllocatedPropertyInitializer());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Property.Initializer array.
----------------------------------------------------------------------------------------*/
void dvPropertyAllocInitializers(
    dvProperty Property,
    uint32 numInitializers)
{
    uint32 freeSpace = dvAllocatedPropertyInitializer() - dvUsedPropertyInitializer();
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvProperty) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvProperty) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numInitializers + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvPropertyGetNumInitializer(Property) == 0);
#endif
    if(numInitializers == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMorePropertyInitializers(spaceNeeded);
    }
    dvPropertySetInitializerIndex(Property, dvUsedPropertyInitializer() + usedHeaderSize);
    dvPropertySetNumInitializer(Property, numInitializers);
    *(dvProperty *)(void *)(dvPropertys.Initializer + dvUsedPropertyInitializer()) = Property;
    memset(dvPropertys.Initializer + dvPropertyGetInitializerIndex(Property), 0, ((numInitializers))*sizeof(char));
    dvSetUsedPropertyInitializer(dvUsedPropertyInitializer() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvPropertyGetInitializers for the database manager.
----------------------------------------------------------------------------------------*/
static void *getPropertyInitializers(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvProperty Property = dvIndex2Property((uint32)objectNumber);

    *numValues = dvPropertyGetNumInitializer(Property);
    return dvPropertyGetInitializers(Property);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvPropertyAllocInitializers for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocPropertyInitializers(
    uint64 objectNumber,
    uint32 numValues)
{
    dvProperty Property = dvIndex2Property((uint32)objectNumber);

    dvPropertySetInitializerIndex(Property, 0);
    dvPropertySetNumInitializer(Property, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvPropertyAllocInitializers(Property, numValues);
    return dvPropertyGetInitializers(Property);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Property.Initializer array.
----------------------------------------------------------------------------------------*/
void dvPropertyFreeInitializers(
    dvProperty Property)
{
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvProperty) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvProperty) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvPropertyGetNumInitializer(Property) + usedHeaderSize, freeHeaderSize);
    char *dataPtr = dvPropertyGetInitializers(Property) - usedHeaderSize;

    if(dvPropertyGetNumInitializer(Property) == 0) {
        return;
    }
    *(dvProperty *)(void *)(dataPtr) = dvPropertyNull;
    *(uint32 *)(void *)(((dvProperty *)(void *)dataPtr) + 1) = size;
    dvPropertySetNumInitializer(Property, 0);
    dvSetFreePropertyInitializer(dvFreePropertyInitializer() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Property.Initializer array.
----------------------------------------------------------------------------------------*/
void dvPropertyResizeInitializers(
    dvProperty Property,
    uint32 numInitializers)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvProperty) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvProperty) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numInitializers + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvPropertyGetNumInitializer(Property) + usedHeaderSize, freeHeaderSize);
    char *dataPtr;

    if(numInitializers == 0) {
        if(dvPropertyGetNumInitializer(Property) != 0) {
            dvPropertyFreeInitializers(Property);
        }
        return;
    }
    if(dvPropertyGetNumInitializer(Property) == 0) {
        dvPropertyAllocInitializers(Property, numInitializers);
        return;
    }
    freeSpace = dvAllocatedPropertyInitializer() - dvUsedPropertyInitializer();
    if(freeSpace < newSize) {
        allocMorePropertyInitializers(newSize);
    }
    dataPtr = dvPropertyGetInitializers(Property) - usedHeaderSize;
    memcpy((void *)(dvPropertys.Initializer + dvUsedPropertyInitializer()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        memset(dvPropertys.Initializer + dvUsedPropertyInitializer() + oldSize, 0, ((newSize - oldSize))*sizeof(char));
    }
    *(dvProperty *)(void *)dataPtr = dvPropertyNull;
    *(uint32 *)(void *)(((dvProperty *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreePropertyInitializer(dvFreePropertyInitializer() + oldSize);
    dvPropertySetInitializerIndex(Property, dvUsedPropertyInitializer() + usedHeaderSize);
    dvPropertySetNumInitializer(Property, numInitializers);
    dvSetUsedPropertyInitializer(dvUsedPropertyInitializer() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the Property.Index heap to free memory.
----------------------------------------------------------------------------------------*/
void dvCompactPropertyIndexs(void)
{
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvProperty) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvProperty) + sizeof(uint32) + elementSize - 1)/elementSize;
    char *toPtr = dvPropertys.Index;
    char *fromPtr = toPtr;
    dvProperty Property;
    uint32 size;

    while(fromPtr < dvPropertys.Index + dvUsedPropertyIndex()) {
        Property = *(dvProperty *)(void *)fromPtr;
        if(Property != dvPropertyNull) {
            /* Need to move it to toPtr */
            size = utMax(dvPropertyGetNumIndex(Property) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            dvPropertySetIndexIndex(Property, toPtr - dvPropertys.Index + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((dvProperty *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    dvSetUsedPropertyIndex(toPtr - dvPropertys.Index);
    dvSetFreePropertyIndex(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Property.Index heap.
----------------------------------------------------------------------------------------*/
static void allocMorePropertyIndexs(
    uint32 spaceNeeded)
{
    uint32 freeSpace = dvAllocatedPropertyIndex() - dvUsedPropertyIndex();

    if((dvFreePropertyIndex() << 2) > dvUsedPropertyIndex()) {
        dvCompactPropertyIndexs();
        freeSpace = dvAllocatedPropertyIndex() - dvUsedPropertyIndex();
    }
    if(freeSpace < spaceNeeded) {
        dvSetAllocatedPropertyIndex(dvAllocatedPropertyIndex() + spaceNeeded - freeSpace +
            (dvAllocatedPropertyIndex() >> 1));
        utResizeArray(dvPropertys.Index, dvAllocatedPropertyIndex());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Property.Index array.
----------------------------------------------------------------------------------------*/
void dvPropertyAllocIndexs(
    dvProperty Property,
    uint32 numIndexs)
{
    uint32 freeSpace = dvAllocatedPropertyIndex() - dvUsedPropertyIndex();
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvProperty) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvProperty) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numIndexs + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(dvPropertyGetNumIndex(Property) == 0);
#endif
    if(numIndexs == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMorePropertyIndexs(spaceNeeded);
    }
    dvPropertySetIndexIndex(Property, dvUsedPropertyIndex() + usedHeaderSize);
    dvPropertySetNumIndex(Property, numIndexs);
    *(dvProperty *)(void *)(dvPropertys.Index + dvUsedPropertyIndex()) = Property;
    memset(dvPropertys.Index + dvPropertyGetIndexIndex(Property), 0, ((numIndexs))*sizeof(char));
    dvSetUsedPropertyIndex(dvUsedPropertyIndex() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvPropertyGetIndexs for the database manager.
----------------------------------------------------------------------------------------*/
static void *getPropertyIndexs(
    uint64 objectNumber,
    uint32 *numValues)
{
    dvProperty Property = dvIndex2Property((uint32)objectNumber);

    *numValues = dvPropertyGetNumIndex(Property);
    return dvPropertyGetIndexs(Property);
}

/*----------------------------------------------------------------------------------------
  Wrapper around dvPropertyAllocIndexs for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocPropertyIndexs(
    uint64 objectNumber,
    uint32 numValues)
{
    dvProperty Property = dvIndex2Property((uint32)objectNumber);

    dvPropertySetIndexIndex(Property, 0);
    dvPropertySetNumIndex(Property, 0);
    if(numValues == 0) {
        return NULL;
    }
    dvPropertyAllocIndexs(Property, numValues);
    return dvPropertyGetIndexs(Property);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Property.Index array.
----------------------------------------------------------------------------------------*/
void dvPropertyFreeIndexs(
    dvProperty Property)
{
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvProperty) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvProperty) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(dvPropertyGetNumIndex(Property) + usedHeaderSize, freeHeaderSize);
    char *dataPtr = dvPropertyGetIndexs(Property) - usedHeaderSize;

    if(dvPropertyGetNumIndex(Property) == 0) {
        return;
    }
    *(dvProperty *)(void *)(dataPtr) = dvPropertyNull;
    *(uint32 *)(void *)(((dvProperty *)(void *)dataPtr) + 1) = size;
    dvPropertySetNumIndex(Property, 0);
    dvSetFreePropertyIndex(dvFreePropertyIndex() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Property.Index array.
----------------------------------------------------------------------------------------*/
void dvPropertyResizeIndexs(
    dvProperty Property,
    uint32 numIndexs)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(dvProperty) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(dvProperty) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numIndexs + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(dvPropertyGetNumIndex(Property) + usedHeaderSize, freeHeaderSize);
    char *dataPtr;

    if(numIndexs == 0) {
        if(dvPropertyGetNumIndex(Property) != 0) {
            dvPropertyFreeIndexs(Property);
        }
        return;
    }
    if(dvPropertyGetNumIndex(Property) == 0) {
        dvPropertyAllocIndexs(Property, numIndexs);
        return;
    }
    freeSpace = dvAllocatedPropertyIndex() - dvUsedPropertyIndex();
    if(freeSpace < newSize) {
        allocMorePropertyIndexs(newSize);
    }
    dataPtr = dvPropertyGetIndexs(Property) - usedHeaderSize;
    memcpy((void *)(dvPropertys.Index + dvUsedPropertyIndex()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        memset(dvPropertys.Index + dvUsedPropertyIndex() + oldSize, 0, ((newSize - oldSize))*sizeof(char));
    }
    *(dvProperty *)(void *)dataPtr = dvPropertyNull;
    *(uint32 *)(void *)(((dvProperty *)(void *)dataPtr) + 1) = oldSize;
    dvSetFreePropertyIndex(dvFreePropertyIndex() + oldSize);
    dvPropertySetIndexIndex(Property, dvUsedPropertyIndex() + usedHeaderSize);
    dvPropertySetNumIndex(Property, numIndexs);
    dvSetUsedPropertyIndex(dvUsedPropertyIndex() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Property.
----------------------------------------------------------------------------------------*/
void dvPropertyCopyProps(
    dvProperty oldProperty,
    dvProperty newProperty)
{
    dvPropertySetType(newProperty, dvPropertyGetType(oldProperty));
    dvPropertySetArray(newProperty, dvPropertyArray(oldProperty));
    dvPropertySetCascade(newProperty, dvPropertyCascade(oldProperty));
    dvPropertySetSparse(newProperty, dvPropertySparse(oldProperty));
    dvPropertySetExpanded(newProperty, dvPropertyExpanded(oldProperty));
    dvPropertySetFieldNumber(newProperty, dvPropertyGetFieldNumber(oldProperty));
    dvPropertySetHidden(newProperty, dvPropertyHidden(oldProperty));
    dvPropertySetFixedSize(newProperty, dvPropertyFixedSize(oldProperty));
    dvPropertySetLine(newProperty, dvPropertyGetLine(oldProperty));
}

/*----------------------------------------------------------------------------------------
  Return the integer equivalent for the bit fields in Property.
----------------------------------------------------------------------------------------*/
uint32 dvPropertyGetBitfield(
    dvProperty _Property)
{
    uint32 bitfield = 0;
    uint8 xLevel = 0;

    bitfield |= dvPropertyArray(_Property) << xLevel++;
    bitfield |= dvPropertyCascade(_Property) << xLevel++;
    bitfield |= dvPropertySparse(_Property) << xLevel++;
    bitfield |= dvPropertyExpanded(_Property) << xLevel++;
    bitfield |= dvPropertyHidden(_Property) << xLevel++;
    bitfield |= dvPropertyFixedSize(_Property) << xLevel++;
    return bitfield;
}

/*----------------------------------------------------------------------------------------
  Set bit fields in Property using bitfield.
----------------------------------------------------------------------------------------*/
void dvPropertySetBitfield(
    dvProperty _Property,
     uint32 bitfield)
{
    dvPropertySetArray(_Property, bitfield & 1);
    bitfield >>= 1;
    dvPropertySetCascade(_Property, bitfield & 1);
    bitfield >>= 1;
    dvPropertySetSparse(_Property, bitfield & 1);
    bitfield >>= 1;
    dvPropertySetExpanded(_Property, bitfield & 1);
    bitfield >>= 1;
    dvPropertySetHidden(_Property, bitfield & 1);
    bitfield >>= 1;
    dvPropertySetFixedSize(_Property, bitfield & 1);
    bitfield >>= 1;
}

/*----------------------------------------------------------------------------------------
  Add the Case to the head of the list on the Property.
----------------------------------------------------------------------------------------*/
void dvPropertyInsertCase(
    dvProperty Property,
    dvCase _Case)
{
#if defined(DD_DEBUG)
    if(Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(_Case == dvCaseNull) {
        utExit("Non-existent Case");
    }
    if(dvCaseGetProperty(_Case) != dvPropertyNull) {
        utExit("Attempting to add Case to Property twice");
    }
#endif
    dvCaseSetNextPropertyCase(_Case, dvPropertyGetFirstCase(Property));
    dvPropertySetFirstCase(Property, _Case);
    if(dvPropertyGetLastCase(Property) == dvCaseNull) {
        dvPropertySetLastCase(Property, _Case);
    }
    dvCaseSetProperty(_Case, Property);
}

/*----------------------------------------------------------------------------------------
  Add the Case to the end of the list on the Property.
----------------------------------------------------------------------------------------*/
void dvPropertyAppendCase(
    dvProperty Property,
    dvCase _Case)
{
#if defined(DD_DEBUG)
    if(Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(_Case == dvCaseNull) {
        utExit("Non-existent Case");
    }
    if(dvCaseGetProperty(_Case) != dvPropertyNull) {
        utExit("Attempting to add Case to Property twice");
    }
#endif
    if(dvPropertyGetLastCase(Property) != dvCaseNull) {
        dvCaseSetNextPropertyCase(dvPropertyGetLastCase(Property), _Case);
    } else {
        dvPropertySetFirstCase(Property, _Case);
    }
    dvPropertySetLastCase(Property, _Case);
    dvCaseSetNextPropertyCase(_Case, dvCaseNull);
    dvCaseSetProperty(_Case, Property);
}

/*----------------------------------------------------------------------------------------
  Insert the Case to the Property after the previous Case.
----------------------------------------------------------------------------------------*/
void dvPropertyInsertAfterCase(
    dvProperty Property,
    dvCase prevCase,
    dvCase _Case)
{
    dvCase nextCase = dvCaseGetNextPropertyCase(prevCase);

#if defined(DD_DEBUG)
    if(Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(_Case == dvCaseNull) {
        utExit("Non-existent Case");
    }
    if(dvCaseGetProperty(_Case) != dvPropertyNull) {
        utExit("Attempting to add Case to Property twice");
    }
#endif
    dvCaseSetNextPropertyCase(_Case, nextCase);
    dvCaseSetNextPropertyCase(prevCase, _Case);
    if(dvPropertyGetLastCase(Property) == prevCase) {
        dvPropertySetLastCase(Property, _Case);
    }
    dvCaseSetProperty(_Case, Property);
}

/*----------------------------------------------------------------------------------------
 Remove the Case from the Property.
----------------------------------------------------------------------------------------*/
void dvPropertyRemoveCase(
    dvProperty Property,
    dvCase _Case)
{
    dvCase pCase, nCase;

#if defined(DD_DEBUG)
    if(_Case == dvCaseNull) {
        utExit("Non-existent Case");
    }
    if(dvCaseGetProperty(_Case) != dvPropertyNull && dvCaseGetProperty(_Case) != Property) {
        utExit("Delete Case from non-owning Property");
    }
#endif
    pCase = dvCaseNull;
    for(nCase = dvPropertyGetFirstCase(Property); nCase != dvCaseNull && nCase != _Case;
            nCase = dvCaseGetNextPropertyCase(nCase)) {
        pCase = nCase;
    }
    if(pCase != dvCaseNull) {
        dvCaseSetNextPropertyCase(pCase, dvCaseGetNextPropertyCase(_Case));
    } else {
        dvPropertySetFirstCase(Property, dvCaseGetNextPropertyCase(_Case));
    }
    dvCaseSetNextPropertyCase(_Case, dvCaseNull);
    if(dvPropertyGetLastCase(Property) == _Case) {
        dvPropertySetLastCase(Property, pCase);
    }
    dvCaseSetProperty(_Case, dvPropertyNull);
}

/*----------------------------------------------------------------------------------------
  Add the Key to the head of the list on the Property.
----------------------------------------------------------------------------------------*/
void dvPropertyInsertKey(
    dvProperty Property,
    dvKey _Key)
{
#if defined(DD_DEBUG)
    if(Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(_Key == dvKeyNull) {
        utExit("Non-existent Key");
    }
    if(dvKeyGetProperty(_Key) != dvPropertyNull) {
        utExit("Attempting to add Key to Property twice");
    }
#endif
    dvKeySetNextPropertyKey(_Key, dvPropertyGetFirstKey(Property));
    dvPropertySetFirstKey(Property, _Key);
    if(dvPropertyGetLastKey(Property) == dvKeyNull) {
        dvPropertySetLastKey(Property, _Key);
    }
    dvKeySetProperty(_Key, Property);
}

/*----------------------------------------------------------------------------------------
  Add the Key to the end of the list on the Property.
----------------------------------------------------------------------------------------*/
void dvPropertyAppendKey(
    dvProperty Property,
    dvKey _Key)
{
#if defined(DD_DEBUG)
    if(Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(_Key == dvKeyNull) {
        utExit("Non-existent Key");
    }
    if(dvKeyGetProperty(_Key) != dvPropertyNull) {
        utExit("Attempting to add Key to Property twice");
    }
#endif
    if(dvPropertyGetLastKey(Property) != dvKeyNull) {
        dvKeySetNextPropertyKey(dvPropertyGetLastKey(Property), _Key);
    } else {
        dvPropertySetFirstKey(Property, _Key);
    }
    dvPropertySetLastKey(Property, _Key);
    dvKeySetNextPropertyKey(_Key, dvKeyNull);
    dvKeySetProperty(_Key, Property);
}

/*----------------------------------------------------------------------------------------
  Insert the Key to the Property after the previous Key.
----------------------------------------------------------------------------------------*/
void dvPropertyInsertAfterKey(
    dvProperty Property,
    dvKey prevKey,
    dvKey _Key)
{
    dvKey nextKey = dvKeyGetNextPropertyKey(prevKey);

#if defined(DD_DEBUG)
    if(Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(_Key == dvKeyNull) {
        utExit("Non-existent Key");
    }
    if(dvKeyGetProperty(_Key) != dvPropertyNull) {
        utExit("Attempting to add Key to Property twice");
    }
#endif
    dvKeySetNextPropertyKey(_Key, nextKey);
    dvKeySetNextPropertyKey(prevKey, _Key);
    if(dvPropertyGetLastKey(Property) == prevKey) {
        dvPropertySetLastKey(Property, _Key);
    }
    dvKeySetProperty(_Key, Property);
}

/*----------------------------------------------------------------------------------------
 Remove the Key from the Property.
----------------------------------------------------------------------------------------*/
void dvPropertyRemoveKey(
    dvProperty Property,
    dvKey _Key)
{
    dvKey pKey, nKey;

#if defined(DD_DEBUG)
    if(_Key == dvKeyNull) {
        utExit("Non-existent Key");
    }
    if(dvKeyGetProperty(_Key) != dvPropertyNull && dvKeyGetProperty(_Key) != Property) {
        utExit("Delete Key from non-owning Property");
    }
#endif
    pKey = dvKeyNull;
    for(nKey = dvPropertyGetFirstKey(Property); nKey != dvKeyNull && nKey != _Key;
            nKey = dvKeyGetNextPropertyKey(nKey)) {
        pKey = nKey;
    }
    if(pKey != dvKeyNull) {
        dvKeySetNextPropertyKey(pKey, dvKeyGetNextPropertyKey(_Key));
    } else {
        dvPropertySetFirstKey(Property, dvKeyGetNextPropertyKey(_Key));
    }
    dvKeySetNextPropertyKey(_Key, dvKeyNull);
    if(dvPropertyGetLastKey(Property) == _Key) {
        dvPropertySetLastKey(Property, pKey);
    }
    dvKeySetProperty(_Key, dvPropertyNull);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowProperty(
    dvProperty Property)
{
    utDatabaseShowObject("dv", "Property", dvProperty2Index(Property));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy Sparsegroup including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void dvSparsegroupDestroy(
    dvSparsegroup Sparsegroup)
{
    dvClass owningClass = dvSparsegroupGetClass(Sparsegroup);
    dvRelationship owningRelationship = dvSparsegroupGetRelationship(Sparsegroup);

    if(dvSparsegroupDestructorCallback != NULL) {
        dvSparsegroupDestructorCallback(Sparsegroup);
    }
    if(owningClass != dvClassNull) {
        dvClassRemoveSparsegroup(owningClass, Sparsegroup);
#if defined(DD_DEBUG)
    } else {
        utExit("Sparsegroup without owning Class");
#endif
    }
    if(owningRelationship != dvRelationshipNull) {
        dvRelationshipSetParentSparsegroup(owningRelationship, dvSparsegroupNull);
    }
    if(owningRelationship != dvRelationshipNull) {
        dvRelationshipSetChildSparsegroup(owningRelationship, dvSparsegroupNull);
    }
    dvSparsegroupFree(Sparsegroup);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocSparsegroup(void)
{
    dvSparsegroup Sparsegroup = dvSparsegroupAlloc();

    return dvSparsegroup2Index(Sparsegroup);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroySparsegroup(
    uint64 objectIndex)
{
    dvSparsegroupDestroy(dvIndex2Sparsegroup((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Sparsegroup.
----------------------------------------------------------------------------------------*/
static void allocSparsegroups(void)
{
    dvSetAllocatedSparsegroup(2);
    dvSetUsedSparsegroup(0);
    dvSetFirstFreeSparsegroup(dvSparsegroupNull);
    dvSparsegroups.Sym = utNewA(utSym, (dvAllocatedSparsegroup()));
    dvSparsegroups.Class = utNewA(dvClass, (dvAllocatedSparsegroup()));
    dvSparsegroups.NextClassSparsegroup = utNewA(dvSparsegroup, (dvAllocatedSparsegroup()));
    dvSparsegroups.PrevClassSparsegroup = utNewA(dvSparsegroup, (dvAllocatedSparsegroup()));
    dvSparsegroups.NextTableClassSparsegroup = utNewA(dvSparsegroup, (dvAllocatedSparsegroup()));
    dvSparsegroups.FirstProperty = utNewA(dvProperty, (dvAllocatedSparsegroup()));
    dvSparsegroups.LastProperty = utNewA(dvProperty, (dvAllocatedSparsegroup()));
    dvSparsegroups.Relationship = utNewA(dvRelationship, (dvAllocatedSparsegroup()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Sparsegroup.
----------------------------------------------------------------------------------------*/
static void reallocSparsegroups(
    uint32 newSize)
{
    utResizeArray(dvSparsegroups.Sym, (newSize));
    utResizeArray(dvSparsegroups.Class, (newSize));
    utResizeArray(dvSparsegroups.NextClassSparsegroup, (newSize));
    utResizeArray(dvSparsegroups.PrevClassSparsegroup, (newSize));
    utResizeArray(dvSparsegroups.NextTableClassSparsegroup, (newSize));
    utResizeArray(dvSparsegroups.FirstProperty, (newSize));
    utResizeArray(dvSparsegroups.LastProperty, (newSize));
    utResizeArray(dvSparsegroups.Relationship, (newSize));
    dvSetAllocatedSparsegroup(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Sparsegroups.
----------------------------------------------------------------------------------------*/
void dvSparsegroupAllocMore(void)
{
    reallocSparsegroups((uint32)(dvAllocatedSparsegroup() + (dvAllocatedSparsegroup() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Sparsegroup.
----------------------------------------------------------------------------------------*/
void dvSparsegroupCopyProps(
    dvSparsegroup oldSparsegroup,
    dvSparsegroup newSparsegroup)
{
}

/*----------------------------------------------------------------------------------------
  Add the Property to the head of the list on the Sparsegroup.
----------------------------------------------------------------------------------------*/
void dvSparsegroupInsertProperty(
    dvSparsegroup Sparsegroup,
    dvProperty _Property)
{
#if defined(DD_DEBUG)
    if(Sparsegroup == dvSparsegroupNull) {
        utExit("Non-existent Sparsegroup");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetSparsegroup(_Property) != dvSparsegroupNull) {
        utExit("Attempting to add Property to Sparsegroup twice");
    }
#endif
    dvPropertySetNextSparsegroupProperty(_Property, dvSparsegroupGetFirstProperty(Sparsegroup));
    dvSparsegroupSetFirstProperty(Sparsegroup, _Property);
    if(dvSparsegroupGetLastProperty(Sparsegroup) == dvPropertyNull) {
        dvSparsegroupSetLastProperty(Sparsegroup, _Property);
    }
    dvPropertySetSparsegroup(_Property, Sparsegroup);
}

/*----------------------------------------------------------------------------------------
  Add the Property to the end of the list on the Sparsegroup.
----------------------------------------------------------------------------------------*/
void dvSparsegroupAppendProperty(
    dvSparsegroup Sparsegroup,
    dvProperty _Property)
{
#if defined(DD_DEBUG)
    if(Sparsegroup == dvSparsegroupNull) {
        utExit("Non-existent Sparsegroup");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetSparsegroup(_Property) != dvSparsegroupNull) {
        utExit("Attempting to add Property to Sparsegroup twice");
    }
#endif
    if(dvSparsegroupGetLastProperty(Sparsegroup) != dvPropertyNull) {
        dvPropertySetNextSparsegroupProperty(dvSparsegroupGetLastProperty(Sparsegroup), _Property);
    } else {
        dvSparsegroupSetFirstProperty(Sparsegroup, _Property);
    }
    dvSparsegroupSetLastProperty(Sparsegroup, _Property);
    dvPropertySetNextSparsegroupProperty(_Property, dvPropertyNull);
    dvPropertySetSparsegroup(_Property, Sparsegroup);
}

/*----------------------------------------------------------------------------------------
  Insert the Property to the Sparsegroup after the previous Property.
----------------------------------------------------------------------------------------*/
void dvSparsegroupInsertAfterProperty(
    dvSparsegroup Sparsegroup,
    dvProperty prevProperty,
    dvProperty _Property)
{
    dvProperty nextProperty = dvPropertyGetNextSparsegroupProperty(prevProperty);

#if defined(DD_DEBUG)
    if(Sparsegroup == dvSparsegroupNull) {
        utExit("Non-existent Sparsegroup");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetSparsegroup(_Property) != dvSparsegroupNull) {
        utExit("Attempting to add Property to Sparsegroup twice");
    }
#endif
    dvPropertySetNextSparsegroupProperty(_Property, nextProperty);
    dvPropertySetNextSparsegroupProperty(prevProperty, _Property);
    if(dvSparsegroupGetLastProperty(Sparsegroup) == prevProperty) {
        dvSparsegroupSetLastProperty(Sparsegroup, _Property);
    }
    dvPropertySetSparsegroup(_Property, Sparsegroup);
}

/*----------------------------------------------------------------------------------------
 Remove the Property from the Sparsegroup.
----------------------------------------------------------------------------------------*/
void dvSparsegroupRemoveProperty(
    dvSparsegroup Sparsegroup,
    dvProperty _Property)
{
    dvProperty pProperty, nProperty;

#if defined(DD_DEBUG)
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetSparsegroup(_Property) != dvSparsegroupNull && dvPropertyGetSparsegroup(_Property) != Sparsegroup) {
        utExit("Delete Property from non-owning Sparsegroup");
    }
#endif
    pProperty = dvPropertyNull;
    for(nProperty = dvSparsegroupGetFirstProperty(Sparsegroup); nProperty != dvPropertyNull && nProperty != _Property;
            nProperty = dvPropertyGetNextSparsegroupProperty(nProperty)) {
        pProperty = nProperty;
    }
    if(pProperty != dvPropertyNull) {
        dvPropertySetNextSparsegroupProperty(pProperty, dvPropertyGetNextSparsegroupProperty(_Property));
    } else {
        dvSparsegroupSetFirstProperty(Sparsegroup, dvPropertyGetNextSparsegroupProperty(_Property));
    }
    dvPropertySetNextSparsegroupProperty(_Property, dvPropertyNull);
    if(dvSparsegroupGetLastProperty(Sparsegroup) == _Property) {
        dvSparsegroupSetLastProperty(Sparsegroup, pProperty);
    }
    dvPropertySetSparsegroup(_Property, dvSparsegroupNull);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowSparsegroup(
    dvSparsegroup Sparsegroup)
{
    utDatabaseShowObject("dv", "Sparsegroup", dvSparsegroup2Index(Sparsegroup));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocRelationship(void)
{
    dvRelationship Relationship = dvRelationshipAlloc();

    return dvRelationship2Index(Relationship);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Relationship.
----------------------------------------------------------------------------------------*/
static void allocRelationships(void)
{
    dvSetAllocatedRelationship(2);
    dvSetUsedRelationship(0);
    dvRelationships.Type = utNewA(dvRelationshipType, (dvAllocatedRelationship()));
    dvRelationships.ParentLabelSym = utNewA(utSym, (dvAllocatedRelationship()));
    dvRelationships.ChildLabelSym = utNewA(utSym, (dvAllocatedRelationship()));
    dvRelationships.Mandatory = utNewA(uint8, (dvAllocatedRelationship() + 7) >> 3);
    dvRelationships.Cascade = utNewA(uint8, (dvAllocatedRelationship() + 7) >> 3);
    dvRelationships.AccessChild = utNewA(uint8, (dvAllocatedRelationship() + 7) >> 3);
    dvRelationships.AccessParent = utNewA(uint8, (dvAllocatedRelationship() + 7) >> 3);
    dvRelationships.SharedParent = utNewA(uint8, (dvAllocatedRelationship() + 7) >> 3);
    dvRelationships.Sparse = utNewA(uint8, (dvAllocatedRelationship() + 7) >> 3);
    dvRelationships.Expanded = utNewA(uint8, (dvAllocatedRelationship() + 7) >> 3);
    dvRelationships.Schema = utNewA(dvSchema, (dvAllocatedRelationship()));
    dvRelationships.NextSchemaRelationship = utNewA(dvRelationship, (dvAllocatedRelationship()));
    dvRelationships.ParentClass = utNewA(dvClass, (dvAllocatedRelationship()));
    dvRelationships.NextClassChildRelationship = utNewA(dvRelationship, (dvAllocatedRelationship()));
    dvRelationships.ChildClass = utNewA(dvClass, (dvAllocatedRelationship()));
    dvRelationships.NextClassParentRelationship = utNewA(dvRelationship, (dvAllocatedRelationship()));
    dvRelationships.FirstProperty = utNewA(dvProperty, (dvAllocatedRelationship()));
    dvRelationships.LastProperty = utNewA(dvProperty, (dvAllocatedRelationship()));
    dvRelationships.FirstKey = utNewA(dvKey, (dvAllocatedRelationship()));
    dvRelationships.LastKey = utNewA(dvKey, (dvAllocatedRelationship()));
    dvRelationships.ParentSparsegroup = utNewA(dvSparsegroup, (dvAllocatedRelationship()));
    dvRelationships.ChildSparsegroup = utNewA(dvSparsegroup, (dvAllocatedRelationship()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Relationship.
----------------------------------------------------------------------------------------*/
static void reallocRelationships(
    uint32 newSize)
{
    utResizeArray(dvRelationships.Type, (newSize));
    utResizeArray(dvRelationships.ParentLabelSym, (newSize));
    utResizeArray(dvRelationships.ChildLabelSym, (newSize));
    utResizeArray(dvRelationships.Mandatory, (newSize + 7) >> 3);
    utResizeArray(dvRelationships.Cascade, (newSize + 7) >> 3);
    utResizeArray(dvRelationships.AccessChild, (newSize + 7) >> 3);
    utResizeArray(dvRelationships.AccessParent, (newSize + 7) >> 3);
    utResizeArray(dvRelationships.SharedParent, (newSize + 7) >> 3);
    utResizeArray(dvRelationships.Sparse, (newSize + 7) >> 3);
    utResizeArray(dvRelationships.Expanded, (newSize + 7) >> 3);
    utResizeArray(dvRelationships.Schema, (newSize));
    utResizeArray(dvRelationships.NextSchemaRelationship, (newSize));
    utResizeArray(dvRelationships.ParentClass, (newSize));
    utResizeArray(dvRelationships.NextClassChildRelationship, (newSize));
    utResizeArray(dvRelationships.ChildClass, (newSize));
    utResizeArray(dvRelationships.NextClassParentRelationship, (newSize));
    utResizeArray(dvRelationships.FirstProperty, (newSize));
    utResizeArray(dvRelationships.LastProperty, (newSize));
    utResizeArray(dvRelationships.FirstKey, (newSize));
    utResizeArray(dvRelationships.LastKey, (newSize));
    utResizeArray(dvRelationships.ParentSparsegroup, (newSize));
    utResizeArray(dvRelationships.ChildSparsegroup, (newSize));
    dvSetAllocatedRelationship(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Relationships.
----------------------------------------------------------------------------------------*/
void dvRelationshipAllocMore(void)
{
    reallocRelationships((uint32)(dvAllocatedRelationship() + (dvAllocatedRelationship() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Relationship.
----------------------------------------------------------------------------------------*/
void dvRelationshipCopyProps(
    dvRelationship oldRelationship,
    dvRelationship newRelationship)
{
    dvRelationshipSetType(newRelationship, dvRelationshipGetType(oldRelationship));
    dvRelationshipSetParentLabelSym(newRelationship, dvRelationshipGetParentLabelSym(oldRelationship));
    dvRelationshipSetChildLabelSym(newRelationship, dvRelationshipGetChildLabelSym(oldRelationship));
    dvRelationshipSetMandatory(newRelationship, dvRelationshipMandatory(oldRelationship));
    dvRelationshipSetCascade(newRelationship, dvRelationshipCascade(oldRelationship));
    dvRelationshipSetAccessChild(newRelationship, dvRelationshipAccessChild(oldRelationship));
    dvRelationshipSetAccessParent(newRelationship, dvRelationshipAccessParent(oldRelationship));
    dvRelationshipSetSharedParent(newRelationship, dvRelationshipSharedParent(oldRelationship));
    dvRelationshipSetSparse(newRelationship, dvRelationshipSparse(oldRelationship));
    dvRelationshipSetExpanded(newRelationship, dvRelationshipExpanded(oldRelationship));
}

/*----------------------------------------------------------------------------------------
  Return the integer equivalent for the bit fields in Relationship.
----------------------------------------------------------------------------------------*/
uint32 dvRelationshipGetBitfield(
    dvRelationship _Relationship)
{
    uint32 bitfield = 0;
    uint8 xLevel = 0;

    bitfield |= dvRelationshipMandatory(_Relationship) << xLevel++;
    bitfield |= dvRelationshipCascade(_Relationship) << xLevel++;
    bitfield |= dvRelationshipAccessChild(_Relationship) << xLevel++;
    bitfield |= dvRelationshipAccessParent(_Relationship) << xLevel++;
    bitfield |= dvRelationshipSharedParent(_Relationship) << xLevel++;
    bitfield |= dvRelationshipSparse(_Relationship) << xLevel++;
    bitfield |= dvRelationshipExpanded(_Relationship) << xLevel++;
    return bitfield;
}

/*----------------------------------------------------------------------------------------
  Set bit fields in Relationship using bitfield.
----------------------------------------------------------------------------------------*/
void dvRelationshipSetBitfield(
    dvRelationship _Relationship,
     uint32 bitfield)
{
    dvRelationshipSetMandatory(_Relationship, bitfield & 1);
    bitfield >>= 1;
    dvRelationshipSetCascade(_Relationship, bitfield & 1);
    bitfield >>= 1;
    dvRelationshipSetAccessChild(_Relationship, bitfield & 1);
    bitfield >>= 1;
    dvRelationshipSetAccessParent(_Relationship, bitfield & 1);
    bitfield >>= 1;
    dvRelationshipSetSharedParent(_Relationship, bitfield & 1);
    bitfield >>= 1;
    dvRelationshipSetSparse(_Relationship, bitfield & 1);
    bitfield >>= 1;
    dvRelationshipSetExpanded(_Relationship, bitfield & 1);
    bitfield >>= 1;
}

/*----------------------------------------------------------------------------------------
  Add the Property to the head of the list on the Relationship.
----------------------------------------------------------------------------------------*/
void dvRelationshipInsertProperty(
    dvRelationship Relationship,
    dvProperty _Property)
{
#if defined(DD_DEBUG)
    if(Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetRelationship(_Property) != dvRelationshipNull) {
        utExit("Attempting to add Property to Relationship twice");
    }
#endif
    dvPropertySetNextRelationshipProperty(_Property, dvRelationshipGetFirstProperty(Relationship));
    dvRelationshipSetFirstProperty(Relationship, _Property);
    if(dvRelationshipGetLastProperty(Relationship) == dvPropertyNull) {
        dvRelationshipSetLastProperty(Relationship, _Property);
    }
    dvPropertySetRelationship(_Property, Relationship);
}

/*----------------------------------------------------------------------------------------
  Add the Property to the end of the list on the Relationship.
----------------------------------------------------------------------------------------*/
void dvRelationshipAppendProperty(
    dvRelationship Relationship,
    dvProperty _Property)
{
#if defined(DD_DEBUG)
    if(Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetRelationship(_Property) != dvRelationshipNull) {
        utExit("Attempting to add Property to Relationship twice");
    }
#endif
    if(dvRelationshipGetLastProperty(Relationship) != dvPropertyNull) {
        dvPropertySetNextRelationshipProperty(dvRelationshipGetLastProperty(Relationship), _Property);
    } else {
        dvRelationshipSetFirstProperty(Relationship, _Property);
    }
    dvRelationshipSetLastProperty(Relationship, _Property);
    dvPropertySetNextRelationshipProperty(_Property, dvPropertyNull);
    dvPropertySetRelationship(_Property, Relationship);
}

/*----------------------------------------------------------------------------------------
  Insert the Property to the Relationship after the previous Property.
----------------------------------------------------------------------------------------*/
void dvRelationshipInsertAfterProperty(
    dvRelationship Relationship,
    dvProperty prevProperty,
    dvProperty _Property)
{
    dvProperty nextProperty = dvPropertyGetNextRelationshipProperty(prevProperty);

#if defined(DD_DEBUG)
    if(Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetRelationship(_Property) != dvRelationshipNull) {
        utExit("Attempting to add Property to Relationship twice");
    }
#endif
    dvPropertySetNextRelationshipProperty(_Property, nextProperty);
    dvPropertySetNextRelationshipProperty(prevProperty, _Property);
    if(dvRelationshipGetLastProperty(Relationship) == prevProperty) {
        dvRelationshipSetLastProperty(Relationship, _Property);
    }
    dvPropertySetRelationship(_Property, Relationship);
}

/*----------------------------------------------------------------------------------------
 Remove the Property from the Relationship.
----------------------------------------------------------------------------------------*/
void dvRelationshipRemoveProperty(
    dvRelationship Relationship,
    dvProperty _Property)
{
    dvProperty pProperty, nProperty;

#if defined(DD_DEBUG)
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetRelationship(_Property) != dvRelationshipNull && dvPropertyGetRelationship(_Property) != Relationship) {
        utExit("Delete Property from non-owning Relationship");
    }
#endif
    pProperty = dvPropertyNull;
    for(nProperty = dvRelationshipGetFirstProperty(Relationship); nProperty != dvPropertyNull && nProperty != _Property;
            nProperty = dvPropertyGetNextRelationshipProperty(nProperty)) {
        pProperty = nProperty;
    }
    if(pProperty != dvPropertyNull) {
        dvPropertySetNextRelationshipProperty(pProperty, dvPropertyGetNextRelationshipProperty(_Property));
    } else {
        dvRelationshipSetFirstProperty(Relationship, dvPropertyGetNextRelationshipProperty(_Property));
    }
    dvPropertySetNextRelationshipProperty(_Property, dvPropertyNull);
    if(dvRelationshipGetLastProperty(Relationship) == _Property) {
        dvRelationshipSetLastProperty(Relationship, pProperty);
    }
    dvPropertySetRelationship(_Property, dvRelationshipNull);
}

/*----------------------------------------------------------------------------------------
  Add the Key to the head of the list on the Relationship.
----------------------------------------------------------------------------------------*/
void dvRelationshipInsertKey(
    dvRelationship Relationship,
    dvKey _Key)
{
#if defined(DD_DEBUG)
    if(Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(_Key == dvKeyNull) {
        utExit("Non-existent Key");
    }
    if(dvKeyGetRelationship(_Key) != dvRelationshipNull) {
        utExit("Attempting to add Key to Relationship twice");
    }
#endif
    dvKeySetNextRelationshipKey(_Key, dvRelationshipGetFirstKey(Relationship));
    dvRelationshipSetFirstKey(Relationship, _Key);
    if(dvRelationshipGetLastKey(Relationship) == dvKeyNull) {
        dvRelationshipSetLastKey(Relationship, _Key);
    }
    dvKeySetRelationship(_Key, Relationship);
}

/*----------------------------------------------------------------------------------------
  Add the Key to the end of the list on the Relationship.
----------------------------------------------------------------------------------------*/
void dvRelationshipAppendKey(
    dvRelationship Relationship,
    dvKey _Key)
{
#if defined(DD_DEBUG)
    if(Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(_Key == dvKeyNull) {
        utExit("Non-existent Key");
    }
    if(dvKeyGetRelationship(_Key) != dvRelationshipNull) {
        utExit("Attempting to add Key to Relationship twice");
    }
#endif
    if(dvRelationshipGetLastKey(Relationship) != dvKeyNull) {
        dvKeySetNextRelationshipKey(dvRelationshipGetLastKey(Relationship), _Key);
    } else {
        dvRelationshipSetFirstKey(Relationship, _Key);
    }
    dvRelationshipSetLastKey(Relationship, _Key);
    dvKeySetNextRelationshipKey(_Key, dvKeyNull);
    dvKeySetRelationship(_Key, Relationship);
}

/*----------------------------------------------------------------------------------------
  Insert the Key to the Relationship after the previous Key.
----------------------------------------------------------------------------------------*/
void dvRelationshipInsertAfterKey(
    dvRelationship Relationship,
    dvKey prevKey,
    dvKey _Key)
{
    dvKey nextKey = dvKeyGetNextRelationshipKey(prevKey);

#if defined(DD_DEBUG)
    if(Relationship == dvRelationshipNull) {
        utExit("Non-existent Relationship");
    }
    if(_Key == dvKeyNull) {
        utExit("Non-existent Key");
    }
    if(dvKeyGetRelationship(_Key) != dvRelationshipNull) {
        utExit("Attempting to add Key to Relationship twice");
    }
#endif
    dvKeySetNextRelationshipKey(_Key, nextKey);
    dvKeySetNextRelationshipKey(prevKey, _Key);
    if(dvRelationshipGetLastKey(Relationship) == prevKey) {
        dvRelationshipSetLastKey(Relationship, _Key);
    }
    dvKeySetRelationship(_Key, Relationship);
}

/*----------------------------------------------------------------------------------------
 Remove the Key from the Relationship.
----------------------------------------------------------------------------------------*/
void dvRelationshipRemoveKey(
    dvRelationship Relationship,
    dvKey _Key)
{
    dvKey pKey, nKey;

#if defined(DD_DEBUG)
    if(_Key == dvKeyNull) {
        utExit("Non-existent Key");
    }
    if(dvKeyGetRelationship(_Key) != dvRelationshipNull && dvKeyGetRelationship(_Key) != Relationship) {
        utExit("Delete Key from non-owning Relationship");
    }
#endif
    pKey = dvKeyNull;
    for(nKey = dvRelationshipGetFirstKey(Relationship); nKey != dvKeyNull && nKey != _Key;
            nKey = dvKeyGetNextRelationshipKey(nKey)) {
        pKey = nKey;
    }
    if(pKey != dvKeyNull) {
        dvKeySetNextRelationshipKey(pKey, dvKeyGetNextRelationshipKey(_Key));
    } else {
        dvRelationshipSetFirstKey(Relationship, dvKeyGetNextRelationshipKey(_Key));
    }
    dvKeySetNextRelationshipKey(_Key, dvKeyNull);
    if(dvRelationshipGetLastKey(Relationship) == _Key) {
        dvRelationshipSetLastKey(Relationship, pKey);
    }
    dvKeySetRelationship(_Key, dvRelationshipNull);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowRelationship(
    dvRelationship Relationship)
{
    utDatabaseShowObject("dv", "Relationship", dvRelationship2Index(Relationship));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy Key including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void dvKeyDestroy(
    dvKey Key)
{
    dvRelationship owningRelationship = dvKeyGetRelationship(Key);
    dvProperty owningProperty = dvKeyGetProperty(Key);

    if(dvKeyDestructorCallback != NULL) {
        dvKeyDestructorCallback(Key);
    }
    if(owningRelationship != dvRelationshipNull) {
        dvRelationshipRemoveKey(owningRelationship, Key);
#if defined(DD_DEBUG)
    } else {
        utExit("Key without owning Relationship");
#endif
    }
    if(owningProperty != dvPropertyNull) {
        dvPropertyRemoveKey(owningProperty, Key);
#if defined(DD_DEBUG)
    } else {
        utExit("Key without owning Property");
#endif
    }
    dvKeyFree(Key);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocKey(void)
{
    dvKey Key = dvKeyAlloc();

    return dvKey2Index(Key);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroyKey(
    uint64 objectIndex)
{
    dvKeyDestroy(dvIndex2Key((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Key.
----------------------------------------------------------------------------------------*/
static void allocKeys(void)
{
    dvSetAllocatedKey(2);
    dvSetUsedKey(0);
    dvSetFirstFreeKey(dvKeyNull);
    dvKeys.PropertySym = utNewA(utSym, (dvAllocatedKey()));
    dvKeys.LineNum = utNewA(uint32, (dvAllocatedKey()));
    dvKeys.Property = utNewA(dvProperty, (dvAllocatedKey()));
    dvKeys.NextPropertyKey = utNewA(dvKey, (dvAllocatedKey()));
    dvKeys.Relationship = utNewA(dvRelationship, (dvAllocatedKey()));
    dvKeys.NextRelationshipKey = utNewA(dvKey, (dvAllocatedKey()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Key.
----------------------------------------------------------------------------------------*/
static void reallocKeys(
    uint32 newSize)
{
    utResizeArray(dvKeys.PropertySym, (newSize));
    utResizeArray(dvKeys.LineNum, (newSize));
    utResizeArray(dvKeys.Property, (newSize));
    utResizeArray(dvKeys.NextPropertyKey, (newSize));
    utResizeArray(dvKeys.Relationship, (newSize));
    utResizeArray(dvKeys.NextRelationshipKey, (newSize));
    dvSetAllocatedKey(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Keys.
----------------------------------------------------------------------------------------*/
void dvKeyAllocMore(void)
{
    reallocKeys((uint32)(dvAllocatedKey() + (dvAllocatedKey() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Key.
----------------------------------------------------------------------------------------*/
void dvKeyCopyProps(
    dvKey oldKey,
    dvKey newKey)
{
    dvKeySetPropertySym(newKey, dvKeyGetPropertySym(oldKey));
    dvKeySetLineNum(newKey, dvKeyGetLineNum(oldKey));
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowKey(
    dvKey Key)
{
    utDatabaseShowObject("dv", "Key", dvKey2Index(Key));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy Union including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void dvUnionDestroy(
    dvUnion Union)
{
    dvClass owningClass = dvUnionGetClass(Union);

    if(dvUnionDestructorCallback != NULL) {
        dvUnionDestructorCallback(Union);
    }
    if(owningClass != dvClassNull) {
        dvClassRemoveUnion(owningClass, Union);
#if defined(DD_DEBUG)
    } else {
        utExit("Union without owning Class");
#endif
    }
    dvUnionFree(Union);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocUnion(void)
{
    dvUnion Union = dvUnionAlloc();

    return dvUnion2Index(Union);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroyUnion(
    uint64 objectIndex)
{
    dvUnionDestroy(dvIndex2Union((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Union.
----------------------------------------------------------------------------------------*/
static void allocUnions(void)
{
    dvSetAllocatedUnion(2);
    dvSetUsedUnion(0);
    dvSetFirstFreeUnion(dvUnionNull);
    dvUnions.PropertySym = utNewA(utSym, (dvAllocatedUnion()));
    dvUnions.TypeProperty = utNewA(dvProperty, (dvAllocatedUnion()));
    dvUnions.Line = utNewA(uint32, (dvAllocatedUnion()));
    dvUnions.Number = utNewA(uint16, (dvAllocatedUnion()));
    dvUnions.FieldNumber = utNewA(uint32, (dvAllocatedUnion()));
    dvUnions.NumCases = utNewA(uint16, (dvAllocatedUnion()));
    dvUnions.Class = utNewA(dvClass, (dvAllocatedUnion()));
    dvUnions.NextClassUnion = utNewA(dvUnion, (dvAllocatedUnion()));
    dvUnions.FirstProperty = utNewA(dvProperty, (dvAllocatedUnion()));
    dvUnions.LastProperty = utNewA(dvProperty, (dvAllocatedUnion()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Union.
----------------------------------------------------------------------------------------*/
static void reallocUnions(
    uint32 newSize)
{
    utResizeArray(dvUnions.PropertySym, (newSize));
    utResizeArray(dvUnions.TypeProperty, (newSize));
    utResizeArray(dvUnions.Line, (newSize));
    utResizeArray(dvUnions.Number, (newSize));
    utResizeArray(dvUnions.FieldNumber, (newSize));
    utResizeArray(dvUnions.NumCases, (newSize));
    utResizeArray(dvUnions.Class, (newSize));
    utResizeArray(dvUnions.NextClassUnion, (newSize));
    utResizeArray(dvUnions.FirstProperty, (newSize));
    utResizeArray(dvUnions.LastProperty, (newSize));
    dvSetAllocatedUnion(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Unions.
----------------------------------------------------------------------------------------*/
void dvUnionAllocMore(void)
{
    reallocUnions((uint32)(dvAllocatedUnion() + (dvAllocatedUnion() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Union.
----------------------------------------------------------------------------------------*/
void dvUnionCopyProps(
    dvUnion oldUnion,
    dvUnion newUnion)
{
    dvUnionSetPropertySym(newUnion, dvUnionGetPropertySym(oldUnion));
    dvUnionSetLine(newUnion, dvUnionGetLine(oldUnion));
    dvUnionSetNumber(newUnion, dvUnionGetNumber(oldUnion));
    dvUnionSetFieldNumber(newUnion, dvUnionGetFieldNumber(oldUnion));
    dvUnionSetNumCases(newUnion, dvUnionGetNumCases(oldUnion));
}

/*----------------------------------------------------------------------------------------
  Add the Property to the head of the list on the Union.
----------------------------------------------------------------------------------------*/
void dvUnionInsertProperty(
    dvUnion Union,
    dvProperty _Property)
{
#if defined(DD_DEBUG)
    if(Union == dvUnionNull) {
        utExit("Non-existent Union");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetUnion(_Property) != dvUnionNull) {
        utExit("Attempting to add Property to Union twice");
    }
#endif
    dvPropertySetNextUnionProperty(_Property, dvUnionGetFirstProperty(Union));
    dvUnionSetFirstProperty(Union, _Property);
    if(dvUnionGetLastProperty(Union) == dvPropertyNull) {
        dvUnionSetLastProperty(Union, _Property);
    }
    dvPropertySetUnion(_Property, Union);
}

/*----------------------------------------------------------------------------------------
  Add the Property to the end of the list on the Union.
----------------------------------------------------------------------------------------*/
void dvUnionAppendProperty(
    dvUnion Union,
    dvProperty _Property)
{
#if defined(DD_DEBUG)
    if(Union == dvUnionNull) {
        utExit("Non-existent Union");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetUnion(_Property) != dvUnionNull) {
        utExit("Attempting to add Property to Union twice");
    }
#endif
    if(dvUnionGetLastProperty(Union) != dvPropertyNull) {
        dvPropertySetNextUnionProperty(dvUnionGetLastProperty(Union), _Property);
    } else {
        dvUnionSetFirstProperty(Union, _Property);
    }
    dvUnionSetLastProperty(Union, _Property);
    dvPropertySetNextUnionProperty(_Property, dvPropertyNull);
    dvPropertySetUnion(_Property, Union);
}

/*----------------------------------------------------------------------------------------
  Insert the Property to the Union after the previous Property.
----------------------------------------------------------------------------------------*/
void dvUnionInsertAfterProperty(
    dvUnion Union,
    dvProperty prevProperty,
    dvProperty _Property)
{
    dvProperty nextProperty = dvPropertyGetNextUnionProperty(prevProperty);

#if defined(DD_DEBUG)
    if(Union == dvUnionNull) {
        utExit("Non-existent Union");
    }
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetUnion(_Property) != dvUnionNull) {
        utExit("Attempting to add Property to Union twice");
    }
#endif
    dvPropertySetNextUnionProperty(_Property, nextProperty);
    dvPropertySetNextUnionProperty(prevProperty, _Property);
    if(dvUnionGetLastProperty(Union) == prevProperty) {
        dvUnionSetLastProperty(Union, _Property);
    }
    dvPropertySetUnion(_Property, Union);
}

/*----------------------------------------------------------------------------------------
 Remove the Property from the Union.
----------------------------------------------------------------------------------------*/
void dvUnionRemoveProperty(
    dvUnion Union,
    dvProperty _Property)
{
    dvProperty pProperty, nProperty;

#if defined(DD_DEBUG)
    if(_Property == dvPropertyNull) {
        utExit("Non-existent Property");
    }
    if(dvPropertyGetUnion(_Property) != dvUnionNull && dvPropertyGetUnion(_Property) != Union) {
        utExit("Delete Property from non-owning Union");
    }
#endif
    pProperty = dvPropertyNull;
    for(nProperty = dvUnionGetFirstProperty(Union); nProperty != dvPropertyNull && nProperty != _Property;
            nProperty = dvPropertyGetNextUnionProperty(nProperty)) {
        pProperty = nProperty;
    }
    if(pProperty != dvPropertyNull) {
        dvPropertySetNextUnionProperty(pProperty, dvPropertyGetNextUnionProperty(_Property));
    } else {
        dvUnionSetFirstProperty(Union, dvPropertyGetNextUnionProperty(_Property));
    }
    dvPropertySetNextUnionProperty(_Property, dvPropertyNull);
    if(dvUnionGetLastProperty(Union) == _Property) {
        dvUnionSetLastProperty(Union, pProperty);
    }
    dvPropertySetUnion(_Property, dvUnionNull);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowUnion(
    dvUnion Union)
{
    utDatabaseShowObject("dv", "Union", dvUnion2Index(Union));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy Case including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void dvCaseDestroy(
    dvCase Case)
{
    dvEntry owningEntry = dvCaseGetEntry(Case);
    dvProperty owningProperty = dvCaseGetProperty(Case);

    if(dvCaseDestructorCallback != NULL) {
        dvCaseDestructorCallback(Case);
    }
    if(owningEntry != dvEntryNull) {
        dvEntryRemoveCase(owningEntry, Case);
#if defined(DD_DEBUG)
    } else {
        utExit("Case without owning Entry");
#endif
    }
    if(owningProperty != dvPropertyNull) {
        dvPropertyRemoveCase(owningProperty, Case);
#if defined(DD_DEBUG)
    } else {
        utExit("Case without owning Property");
#endif
    }
    dvCaseFree(Case);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocCase(void)
{
    dvCase Case = dvCaseAlloc();

    return dvCase2Index(Case);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroyCase(
    uint64 objectIndex)
{
    dvCaseDestroy(dvIndex2Case((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Case.
----------------------------------------------------------------------------------------*/
static void allocCases(void)
{
    dvSetAllocatedCase(2);
    dvSetUsedCase(0);
    dvSetFirstFreeCase(dvCaseNull);
    dvCases.EntrySym = utNewA(utSym, (dvAllocatedCase()));
    dvCases.Entry = utNewA(dvEntry, (dvAllocatedCase()));
    dvCases.NextEntryCase = utNewA(dvCase, (dvAllocatedCase()));
    dvCases.Property = utNewA(dvProperty, (dvAllocatedCase()));
    dvCases.NextPropertyCase = utNewA(dvCase, (dvAllocatedCase()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Case.
----------------------------------------------------------------------------------------*/
static void reallocCases(
    uint32 newSize)
{
    utResizeArray(dvCases.EntrySym, (newSize));
    utResizeArray(dvCases.Entry, (newSize));
    utResizeArray(dvCases.NextEntryCase, (newSize));
    utResizeArray(dvCases.Property, (newSize));
    utResizeArray(dvCases.NextPropertyCase, (newSize));
    dvSetAllocatedCase(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Cases.
----------------------------------------------------------------------------------------*/
void dvCaseAllocMore(void)
{
    reallocCases((uint32)(dvAllocatedCase() + (dvAllocatedCase() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Case.
----------------------------------------------------------------------------------------*/
void dvCaseCopyProps(
    dvCase oldCase,
    dvCase newCase)
{
    dvCaseSetEntrySym(newCase, dvCaseGetEntrySym(oldCase));
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void dvShowCase(
    dvCase Case)
{
    utDatabaseShowObject("dv", "Case", dvCase2Index(Case));
}
#endif

/*----------------------------------------------------------------------------------------
  Free memory used by the dv database.
----------------------------------------------------------------------------------------*/
void dvDatabaseStop(void)
{
    utFree(dvRoots.FirstModpath);
    utFree(dvRoots.LastModpath);
    utFree(dvRoots.ModpathTableIndex);
    utFree(dvRoots.NumModpathTable);
    utFree(dvRoots.ModpathTable);
    utFree(dvRoots.NumModpath);
    utFree(dvRoots.FirstModule);
    utFree(dvRoots.LastModule);
    utFree(dvRoots.ModuleTableIndex);
    utFree(dvRoots.NumModuleTable);
    utFree(dvRoots.ModuleTable);
    utFree(dvRoots.NumModule);
    utFree(dvModpaths.Sym);
    utFree(dvModpaths.Root);
    utFree(dvModpaths.NextRootModpath);
    utFree(dvModpaths.PrevRootModpath);
    utFree(dvModpaths.NextTableRootModpath);
    utFree(dvModules.Sym);
    utFree(dvModules.PrefixSym);
    utFree(dvModules.Persistent);
    utFree(dvModules.UndoRedo);
    utFree(dvModules.HasSparseData);
    utFree(dvModules.NumFields);
    utFree(dvModules.NumClasses);
    utFree(dvModules.NumEnums);
    utFree(dvModules.NextRootModule);
    utFree(dvModules.PrevRootModule);
    utFree(dvModules.NextTableRootModule);
    utFree(dvModules.FirstClass);
    utFree(dvModules.LastClass);
    utFree(dvModules.ClassTableIndex);
    utFree(dvModules.NumClassTable);
    utFree(dvModules.ClassTable);
    utFree(dvModules.NumClass);
    utFree(dvModules.FirstEnum);
    utFree(dvModules.LastEnum);
    utFree(dvModules.EnumTableIndex);
    utFree(dvModules.NumEnumTable);
    utFree(dvModules.EnumTable);
    utFree(dvModules.NumEnum);
    utFree(dvModules.FirstTypedef);
    utFree(dvModules.LastTypedef);
    utFree(dvModules.TypedefTableIndex);
    utFree(dvModules.NumTypedefTable);
    utFree(dvModules.TypedefTable);
    utFree(dvModules.NumTypedef);
    utFree(dvModules.FirstSchema);
    utFree(dvModules.LastSchema);
    utFree(dvModules.SchemaTableIndex);
    utFree(dvModules.NumSchemaTable);
    utFree(dvModules.SchemaTable);
    utFree(dvModules.NumSchema);
    utFree(dvModules.FirstImportLink);
    utFree(dvModules.LastImportLink);
    utFree(dvModules.FirstExportLink);
    utFree(dvModules.LastExportLink);
    utFree(dvLinks.ImportModule);
    utFree(dvLinks.NextModuleImportLink);
    utFree(dvLinks.ExportModule);
    utFree(dvLinks.NextModuleExportLink);
    utFree(dvSchemas.Sym);
    utFree(dvSchemas.Module);
    utFree(dvSchemas.NextModuleSchema);
    utFree(dvSchemas.PrevModuleSchema);
    utFree(dvSchemas.NextTableModuleSchema);
    utFree(dvSchemas.FirstRelationship);
    utFree(dvSchemas.LastRelationship);
    utFree(dvEnums.Sym);
    utFree(dvEnums.PrefixSym);
    utFree(dvEnums.NumEntries);
    utFree(dvEnums.Module);
    utFree(dvEnums.NextModuleEnum);
    utFree(dvEnums.PrevModuleEnum);
    utFree(dvEnums.NextTableModuleEnum);
    utFree(dvEnums.FirstEntry);
    utFree(dvEnums.LastEntry);
    utFree(dvEnums.EntryTableIndex);
    utFree(dvEnums.NumEntryTable);
    utFree(dvEnums.EntryTable);
    utFree(dvEnums.NumEntry);
    utFree(dvEntrys.Sym);
    utFree(dvEntrys.Value);
    utFree(dvEntrys.Enum);
    utFree(dvEntrys.NextEnumEntry);
    utFree(dvEntrys.PrevEnumEntry);
    utFree(dvEntrys.NextTableEnumEntry);
    utFree(dvEntrys.FirstCase);
    utFree(dvEntrys.LastCase);
    utFree(dvTypedefs.Sym);
    utFree(dvTypedefs.InitializerIndex);
    utFree(dvTypedefs.NumInitializer);
    utFree(dvTypedefs.Initializer);
    utFree(dvTypedefs.Module);
    utFree(dvTypedefs.NextModuleTypedef);
    utFree(dvTypedefs.PrevModuleTypedef);
    utFree(dvTypedefs.NextTableModuleTypedef);
    utFree(dvClasss.Sym);
    utFree(dvClasss.MemoryStyle);
    utFree(dvClasss.ReferenceSize);
    utFree(dvClasss.GenerateArrayClass);
    utFree(dvClasss.GenerateAttributes);
    utFree(dvClasss.Sparse);
    utFree(dvClasss.NumFields);
    utFree(dvClasss.Number);
    utFree(dvClasss.Module);
    utFree(dvClasss.NextModuleClass);
    utFree(dvClasss.PrevModuleClass);
    utFree(dvClasss.NextTableModuleClass);
    utFree(dvClasss.FirstProperty);
    utFree(dvClasss.LastProperty);
    utFree(dvClasss.PropertyTableIndex);
    utFree(dvClasss.NumPropertyTable);
    utFree(dvClasss.PropertyTable);
    utFree(dvClasss.NumProperty);
    utFree(dvClasss.FreeListProperty);
    utFree(dvClasss.FirstSparsegroup);
    utFree(dvClasss.LastSparsegroup);
    utFree(dvClasss.SparsegroupTableIndex);
    utFree(dvClasss.NumSparsegroupTable);
    utFree(dvClasss.SparsegroupTable);
    utFree(dvClasss.NumSparsegroup);
    utFree(dvClasss.BaseClass);
    utFree(dvClasss.FirstDerivedClass);
    utFree(dvClasss.NextClassDerivedClass);
    utFree(dvClasss.LastDerivedClass);
    utFree(dvClasss.FirstChildRelationship);
    utFree(dvClasss.LastChildRelationship);
    utFree(dvClasss.FirstParentRelationship);
    utFree(dvClasss.LastParentRelationship);
    utFree(dvClasss.FirstUnion);
    utFree(dvClasss.LastUnion);
    utFree(dvPropertys.Sym);
    utFree(dvPropertys.Type);
    utFree(dvPropertys.Array);
    utFree(dvPropertys.Cascade);
    utFree(dvPropertys.Sparse);
    utFree(dvPropertys.Expanded);
    utFree(dvPropertys.FieldNumber);
    utFree(dvPropertys.FirstElementProp);
    utFree(dvPropertys.NumElementsProp);
    utFree(dvPropertys.Hidden);
    utFree(dvPropertys.InitializerIndex);
    utFree(dvPropertys.NumInitializer);
    utFree(dvPropertys.Initializer);
    utFree(dvPropertys.FixedSize);
    utFree(dvPropertys.IndexIndex);
    utFree(dvPropertys.NumIndex);
    utFree(dvPropertys.Index);
    utFree(dvPropertys.Line);
    utFree(dvPropertys.Class);
    utFree(dvPropertys.NextClassProperty);
    utFree(dvPropertys.PrevClassProperty);
    utFree(dvPropertys.NextTableClassProperty);
    utFree(dvPropertys.FirstCase);
    utFree(dvPropertys.LastCase);
    utFree(dvPropertys.FirstKey);
    utFree(dvPropertys.LastKey);
    utFree(dvPropertys.Sparsegroup);
    utFree(dvPropertys.NextSparsegroupProperty);
    utFree(dvPropertys.Relationship);
    utFree(dvPropertys.NextRelationshipProperty);
    utFree(dvPropertys.Union);
    utFree(dvPropertys.NextUnionProperty);
    utFree(dvPropertys.union1);
    utFree(dvSparsegroups.Sym);
    utFree(dvSparsegroups.Class);
    utFree(dvSparsegroups.NextClassSparsegroup);
    utFree(dvSparsegroups.PrevClassSparsegroup);
    utFree(dvSparsegroups.NextTableClassSparsegroup);
    utFree(dvSparsegroups.FirstProperty);
    utFree(dvSparsegroups.LastProperty);
    utFree(dvSparsegroups.Relationship);
    utFree(dvRelationships.Type);
    utFree(dvRelationships.ParentLabelSym);
    utFree(dvRelationships.ChildLabelSym);
    utFree(dvRelationships.Mandatory);
    utFree(dvRelationships.Cascade);
    utFree(dvRelationships.AccessChild);
    utFree(dvRelationships.AccessParent);
    utFree(dvRelationships.SharedParent);
    utFree(dvRelationships.Sparse);
    utFree(dvRelationships.Expanded);
    utFree(dvRelationships.Schema);
    utFree(dvRelationships.NextSchemaRelationship);
    utFree(dvRelationships.ParentClass);
    utFree(dvRelationships.NextClassChildRelationship);
    utFree(dvRelationships.ChildClass);
    utFree(dvRelationships.NextClassParentRelationship);
    utFree(dvRelationships.FirstProperty);
    utFree(dvRelationships.LastProperty);
    utFree(dvRelationships.FirstKey);
    utFree(dvRelationships.LastKey);
    utFree(dvRelationships.ParentSparsegroup);
    utFree(dvRelationships.ChildSparsegroup);
    utFree(dvKeys.PropertySym);
    utFree(dvKeys.LineNum);
    utFree(dvKeys.Property);
    utFree(dvKeys.NextPropertyKey);
    utFree(dvKeys.Relationship);
    utFree(dvKeys.NextRelationshipKey);
    utFree(dvUnions.PropertySym);
    utFree(dvUnions.TypeProperty);
    utFree(dvUnions.Line);
    utFree(dvUnions.Number);
    utFree(dvUnions.FieldNumber);
    utFree(dvUnions.NumCases);
    utFree(dvUnions.Class);
    utFree(dvUnions.NextClassUnion);
    utFree(dvUnions.FirstProperty);
    utFree(dvUnions.LastProperty);
    utFree(dvCases.EntrySym);
    utFree(dvCases.Entry);
    utFree(dvCases.NextEntryCase);
    utFree(dvCases.Property);
    utFree(dvCases.NextPropertyCase);
    utUnregisterModule(dvModuleID);
}

/*----------------------------------------------------------------------------------------
  Allocate memory used by the dv database.
----------------------------------------------------------------------------------------*/
void dvDatabaseStart(void)
{
    if(!utInitialized()) {
        utStart();
    }
    dvRootData.hash = 0x85ebc91d;
    dvModuleID = utRegisterModule("dv", false, dvHash(), 15, 215, 3, sizeof(struct dvRootType_),
        &dvRootData, dvDatabaseStart, dvDatabaseStop);
    utRegisterEnum("RelationshipType", 8);
    utRegisterEntry("LINKED_LIST", 0);
    utRegisterEntry("DOUBLY_LINKED", 1);
    utRegisterEntry("TAIL_LINKED", 2);
    utRegisterEntry("POINTER", 3);
    utRegisterEntry("ARRAY", 4);
    utRegisterEntry("HEAP", 5);
    utRegisterEntry("HASHED", 6);
    utRegisterEntry("UNBOUND", 7);
    utRegisterEnum("PropertyType", 12);
    utRegisterEntry("INT", 0);
    utRegisterEntry("UINT", 1);
    utRegisterEntry("FLOAT", 2);
    utRegisterEntry("DOUBLE", 3);
    utRegisterEntry("BIT", 4);
    utRegisterEntry("BOOL", 5);
    utRegisterEntry("CHAR", 6);
    utRegisterEntry("ENUM", 7);
    utRegisterEntry("TYPEDEF", 8);
    utRegisterEntry("POINTER", 9);
    utRegisterEntry("SYM", 10);
    utRegisterEntry("UNBOUND", 11);
    utRegisterEnum("MemoryStyle", 2);
    utRegisterEntry("CREATE_ONLY", 0);
    utRegisterEntry("FREE_LIST", 1);
    utRegisterClass("Root", 12, &dvRootData.usedRoot, &dvRootData.allocatedRoot,
        NULL, 65535, 4, allocRoot, NULL);
    utRegisterField("FirstModpath", &dvRoots.FirstModpath, sizeof(dvModpath), UT_POINTER, "Modpath");
    utRegisterField("LastModpath", &dvRoots.LastModpath, sizeof(dvModpath), UT_POINTER, "Modpath");
    utRegisterField("ModpathTableIndex", &dvRoots.ModpathTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumModpathTable", &dvRoots.NumModpathTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("ModpathTable", &dvRoots.ModpathTable, sizeof(dvModpath), UT_POINTER, "Modpath");
    utRegisterArray(&dvRootData.usedRootModpathTable, &dvRootData.allocatedRootModpathTable,
        getRootModpathTables, allocRootModpathTables, dvCompactRootModpathTables);
    utRegisterField("NumModpath", &dvRoots.NumModpath, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstModule", &dvRoots.FirstModule, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("LastModule", &dvRoots.LastModule, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("ModuleTableIndex", &dvRoots.ModuleTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumModuleTable", &dvRoots.NumModuleTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("ModuleTable", &dvRoots.ModuleTable, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterArray(&dvRootData.usedRootModuleTable, &dvRootData.allocatedRootModuleTable,
        getRootModuleTables, allocRootModuleTables, dvCompactRootModuleTables);
    utRegisterField("NumModule", &dvRoots.NumModule, sizeof(uint32), UT_UINT, NULL);
    utRegisterClass("Modpath", 5, &dvRootData.usedModpath, &dvRootData.allocatedModpath,
        NULL, 65535, 4, allocModpath, NULL);
    utRegisterField("Sym", &dvModpaths.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("Root", &dvModpaths.Root, sizeof(dvRoot), UT_POINTER, "Root");
    utRegisterField("NextRootModpath", &dvModpaths.NextRootModpath, sizeof(dvModpath), UT_POINTER, "Modpath");
    utRegisterField("PrevRootModpath", &dvModpaths.PrevRootModpath, sizeof(dvModpath), UT_POINTER, "Modpath");
    utRegisterField("NextTableRootModpath", &dvModpaths.NextTableRootModpath, sizeof(dvModpath), UT_POINTER, "Modpath");
    utRegisterClass("Module", 39, &dvRootData.usedModule, &dvRootData.allocatedModule,
        NULL, 65535, 4, allocModule, NULL);
    utRegisterField("Sym", &dvModules.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("PrefixSym", &dvModules.PrefixSym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("Persistent", &dvModules.Persistent, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("UndoRedo", &dvModules.UndoRedo, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("HasSparseData", &dvModules.HasSparseData, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("NumFields", &dvModules.NumFields, sizeof(uint16), UT_UINT, NULL);
    utRegisterField("NumClasses", &dvModules.NumClasses, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("NumEnums", &dvModules.NumEnums, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("NextRootModule", &dvModules.NextRootModule, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("PrevRootModule", &dvModules.PrevRootModule, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("NextTableRootModule", &dvModules.NextTableRootModule, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("FirstClass", &dvModules.FirstClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("LastClass", &dvModules.LastClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("ClassTableIndex", &dvModules.ClassTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumClassTable", &dvModules.NumClassTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("ClassTable", &dvModules.ClassTable, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterArray(&dvRootData.usedModuleClassTable, &dvRootData.allocatedModuleClassTable,
        getModuleClassTables, allocModuleClassTables, dvCompactModuleClassTables);
    utRegisterField("NumClass", &dvModules.NumClass, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstEnum", &dvModules.FirstEnum, sizeof(dvEnum), UT_POINTER, "Enum");
    utRegisterField("LastEnum", &dvModules.LastEnum, sizeof(dvEnum), UT_POINTER, "Enum");
    utRegisterField("EnumTableIndex", &dvModules.EnumTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumEnumTable", &dvModules.NumEnumTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("EnumTable", &dvModules.EnumTable, sizeof(dvEnum), UT_POINTER, "Enum");
    utRegisterArray(&dvRootData.usedModuleEnumTable, &dvRootData.allocatedModuleEnumTable,
        getModuleEnumTables, allocModuleEnumTables, dvCompactModuleEnumTables);
    utRegisterField("NumEnum", &dvModules.NumEnum, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstTypedef", &dvModules.FirstTypedef, sizeof(dvTypedef), UT_POINTER, "Typedef");
    utRegisterField("LastTypedef", &dvModules.LastTypedef, sizeof(dvTypedef), UT_POINTER, "Typedef");
    utRegisterField("TypedefTableIndex", &dvModules.TypedefTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumTypedefTable", &dvModules.NumTypedefTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("TypedefTable", &dvModules.TypedefTable, sizeof(dvTypedef), UT_POINTER, "Typedef");
    utRegisterArray(&dvRootData.usedModuleTypedefTable, &dvRootData.allocatedModuleTypedefTable,
        getModuleTypedefTables, allocModuleTypedefTables, dvCompactModuleTypedefTables);
    utRegisterField("NumTypedef", &dvModules.NumTypedef, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstSchema", &dvModules.FirstSchema, sizeof(dvSchema), UT_POINTER, "Schema");
    utRegisterField("LastSchema", &dvModules.LastSchema, sizeof(dvSchema), UT_POINTER, "Schema");
    utRegisterField("SchemaTableIndex", &dvModules.SchemaTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumSchemaTable", &dvModules.NumSchemaTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("SchemaTable", &dvModules.SchemaTable, sizeof(dvSchema), UT_POINTER, "Schema");
    utRegisterArray(&dvRootData.usedModuleSchemaTable, &dvRootData.allocatedModuleSchemaTable,
        getModuleSchemaTables, allocModuleSchemaTables, dvCompactModuleSchemaTables);
    utRegisterField("NumSchema", &dvModules.NumSchema, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstImportLink", &dvModules.FirstImportLink, sizeof(dvLink), UT_POINTER, "Link");
    utRegisterField("LastImportLink", &dvModules.LastImportLink, sizeof(dvLink), UT_POINTER, "Link");
    utRegisterField("FirstExportLink", &dvModules.FirstExportLink, sizeof(dvLink), UT_POINTER, "Link");
    utRegisterField("LastExportLink", &dvModules.LastExportLink, sizeof(dvLink), UT_POINTER, "Link");
    utRegisterClass("Link", 4, &dvRootData.usedLink, &dvRootData.allocatedLink,
        NULL, 65535, 4, allocLink, NULL);
    utRegisterField("ImportModule", &dvLinks.ImportModule, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("NextModuleImportLink", &dvLinks.NextModuleImportLink, sizeof(dvLink), UT_POINTER, "Link");
    utRegisterField("ExportModule", &dvLinks.ExportModule, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("NextModuleExportLink", &dvLinks.NextModuleExportLink, sizeof(dvLink), UT_POINTER, "Link");
    utRegisterClass("Schema", 7, &dvRootData.usedSchema, &dvRootData.allocatedSchema,
        NULL, 65535, 4, allocSchema, NULL);
    utRegisterField("Sym", &dvSchemas.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("Module", &dvSchemas.Module, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("NextModuleSchema", &dvSchemas.NextModuleSchema, sizeof(dvSchema), UT_POINTER, "Schema");
    utRegisterField("PrevModuleSchema", &dvSchemas.PrevModuleSchema, sizeof(dvSchema), UT_POINTER, "Schema");
    utRegisterField("NextTableModuleSchema", &dvSchemas.NextTableModuleSchema, sizeof(dvSchema), UT_POINTER, "Schema");
    utRegisterField("FirstRelationship", &dvSchemas.FirstRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("LastRelationship", &dvSchemas.LastRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterClass("Enum", 13, &dvRootData.usedEnum, &dvRootData.allocatedEnum,
        NULL, 65535, 4, allocEnum, NULL);
    utRegisterField("Sym", &dvEnums.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("PrefixSym", &dvEnums.PrefixSym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("NumEntries", &dvEnums.NumEntries, sizeof(uint16), UT_UINT, NULL);
    utRegisterField("Module", &dvEnums.Module, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("NextModuleEnum", &dvEnums.NextModuleEnum, sizeof(dvEnum), UT_POINTER, "Enum");
    utRegisterField("PrevModuleEnum", &dvEnums.PrevModuleEnum, sizeof(dvEnum), UT_POINTER, "Enum");
    utRegisterField("NextTableModuleEnum", &dvEnums.NextTableModuleEnum, sizeof(dvEnum), UT_POINTER, "Enum");
    utRegisterField("FirstEntry", &dvEnums.FirstEntry, sizeof(dvEntry), UT_POINTER, "Entry");
    utRegisterField("LastEntry", &dvEnums.LastEntry, sizeof(dvEntry), UT_POINTER, "Entry");
    utRegisterField("EntryTableIndex", &dvEnums.EntryTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumEntryTable", &dvEnums.NumEntryTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("EntryTable", &dvEnums.EntryTable, sizeof(dvEntry), UT_POINTER, "Entry");
    utRegisterArray(&dvRootData.usedEnumEntryTable, &dvRootData.allocatedEnumEntryTable,
        getEnumEntryTables, allocEnumEntryTables, dvCompactEnumEntryTables);
    utRegisterField("NumEntry", &dvEnums.NumEntry, sizeof(uint32), UT_UINT, NULL);
    utRegisterClass("Entry", 8, &dvRootData.usedEntry, &dvRootData.allocatedEntry,
        NULL, 65535, 4, allocEntry, NULL);
    utRegisterField("Sym", &dvEntrys.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("Value", &dvEntrys.Value, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("Enum", &dvEntrys.Enum, sizeof(dvEnum), UT_POINTER, "Enum");
    utRegisterField("NextEnumEntry", &dvEntrys.NextEnumEntry, sizeof(dvEntry), UT_POINTER, "Entry");
    utRegisterField("PrevEnumEntry", &dvEntrys.PrevEnumEntry, sizeof(dvEntry), UT_POINTER, "Entry");
    utRegisterField("NextTableEnumEntry", &dvEntrys.NextTableEnumEntry, sizeof(dvEntry), UT_POINTER, "Entry");
    utRegisterField("FirstCase", &dvEntrys.FirstCase, sizeof(dvCase), UT_POINTER, "Case");
    utRegisterField("LastCase", &dvEntrys.LastCase, sizeof(dvCase), UT_POINTER, "Case");
    utRegisterClass("Typedef", 8, &dvRootData.usedTypedef, &dvRootData.allocatedTypedef,
        NULL, 65535, 4, allocTypedef, NULL);
    utRegisterField("Sym", &dvTypedefs.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("InitializerIndex", &dvTypedefs.InitializerIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumInitializer", &dvTypedefs.NumInitializer, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("Initializer", &dvTypedefs.Initializer, sizeof(char), UT_CHAR, NULL);
    utRegisterArray(&dvRootData.usedTypedefInitializer, &dvRootData.allocatedTypedefInitializer,
        getTypedefInitializers, allocTypedefInitializers, dvCompactTypedefInitializers);
    utRegisterField("Module", &dvTypedefs.Module, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("NextModuleTypedef", &dvTypedefs.NextModuleTypedef, sizeof(dvTypedef), UT_POINTER, "Typedef");
    utRegisterField("PrevModuleTypedef", &dvTypedefs.PrevModuleTypedef, sizeof(dvTypedef), UT_POINTER, "Typedef");
    utRegisterField("NextTableModuleTypedef", &dvTypedefs.NextTableModuleTypedef, sizeof(dvTypedef), UT_POINTER, "Typedef");
    utRegisterClass("Class", 35, &dvRootData.usedClass, &dvRootData.allocatedClass,
        NULL, 65535, 4, allocClass, NULL);
    utRegisterField("Sym", &dvClasss.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("MemoryStyle", &dvClasss.MemoryStyle, sizeof(dvMemoryStyle), UT_ENUM, "MemoryStyle");
    utRegisterField("ReferenceSize", &dvClasss.ReferenceSize, sizeof(uint8), UT_UINT, NULL);
    utRegisterField("GenerateArrayClass", &dvClasss.GenerateArrayClass, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("GenerateAttributes", &dvClasss.GenerateAttributes, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("Sparse", &dvClasss.Sparse, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("NumFields", &dvClasss.NumFields, sizeof(uint16), UT_UINT, NULL);
    utRegisterField("Number", &dvClasss.Number, sizeof(uint16), UT_UINT, NULL);
    utRegisterField("Module", &dvClasss.Module, sizeof(dvModule), UT_POINTER, "Module");
    utRegisterField("NextModuleClass", &dvClasss.NextModuleClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("PrevModuleClass", &dvClasss.PrevModuleClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("NextTableModuleClass", &dvClasss.NextTableModuleClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("FirstProperty", &dvClasss.FirstProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("LastProperty", &dvClasss.LastProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("PropertyTableIndex", &dvClasss.PropertyTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumPropertyTable", &dvClasss.NumPropertyTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("PropertyTable", &dvClasss.PropertyTable, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterArray(&dvRootData.usedClassPropertyTable, &dvRootData.allocatedClassPropertyTable,
        getClassPropertyTables, allocClassPropertyTables, dvCompactClassPropertyTables);
    utRegisterField("NumProperty", &dvClasss.NumProperty, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FreeListProperty", &dvClasss.FreeListProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("FirstSparsegroup", &dvClasss.FirstSparsegroup, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterField("LastSparsegroup", &dvClasss.LastSparsegroup, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterField("SparsegroupTableIndex", &dvClasss.SparsegroupTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumSparsegroupTable", &dvClasss.NumSparsegroupTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("SparsegroupTable", &dvClasss.SparsegroupTable, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterArray(&dvRootData.usedClassSparsegroupTable, &dvRootData.allocatedClassSparsegroupTable,
        getClassSparsegroupTables, allocClassSparsegroupTables, dvCompactClassSparsegroupTables);
    utRegisterField("NumSparsegroup", &dvClasss.NumSparsegroup, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("BaseClass", &dvClasss.BaseClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("FirstDerivedClass", &dvClasss.FirstDerivedClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("NextClassDerivedClass", &dvClasss.NextClassDerivedClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("LastDerivedClass", &dvClasss.LastDerivedClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("FirstChildRelationship", &dvClasss.FirstChildRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("LastChildRelationship", &dvClasss.LastChildRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("FirstParentRelationship", &dvClasss.FirstParentRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("LastParentRelationship", &dvClasss.LastParentRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("FirstUnion", &dvClasss.FirstUnion, sizeof(dvUnion), UT_POINTER, "Union");
    utRegisterField("LastUnion", &dvClasss.LastUnion, sizeof(dvUnion), UT_POINTER, "Union");
    utRegisterClass("Property", 33, &dvRootData.usedProperty, &dvRootData.allocatedProperty,
        NULL, 65535, 4, allocProperty, NULL);
    utRegisterField("Sym", &dvPropertys.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("Type", &dvPropertys.Type, sizeof(dvPropertyType), UT_ENUM, "PropertyType");
    utRegisterField("Array", &dvPropertys.Array, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("Cascade", &dvPropertys.Cascade, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("Sparse", &dvPropertys.Sparse, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("Expanded", &dvPropertys.Expanded, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("FieldNumber", &dvPropertys.FieldNumber, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstElementProp", &dvPropertys.FirstElementProp, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("NumElementsProp", &dvPropertys.NumElementsProp, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("Hidden", &dvPropertys.Hidden, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("InitializerIndex", &dvPropertys.InitializerIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumInitializer", &dvPropertys.NumInitializer, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("Initializer", &dvPropertys.Initializer, sizeof(char), UT_CHAR, NULL);
    utRegisterArray(&dvRootData.usedPropertyInitializer, &dvRootData.allocatedPropertyInitializer,
        getPropertyInitializers, allocPropertyInitializers, dvCompactPropertyInitializers);
    utRegisterField("FixedSize", &dvPropertys.FixedSize, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("IndexIndex", &dvPropertys.IndexIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumIndex", &dvPropertys.NumIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("Index", &dvPropertys.Index, sizeof(char), UT_CHAR, NULL);
    utRegisterArray(&dvRootData.usedPropertyIndex, &dvRootData.allocatedPropertyIndex,
        getPropertyIndexs, allocPropertyIndexs, dvCompactPropertyIndexs);
    utRegisterField("Line", &dvPropertys.Line, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("Class", &dvPropertys.Class, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("NextClassProperty", &dvPropertys.NextClassProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("PrevClassProperty", &dvPropertys.PrevClassProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("NextTableClassProperty", &dvPropertys.NextTableClassProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("FirstCase", &dvPropertys.FirstCase, sizeof(dvCase), UT_POINTER, "Case");
    utRegisterField("LastCase", &dvPropertys.LastCase, sizeof(dvCase), UT_POINTER, "Case");
    utRegisterField("FirstKey", &dvPropertys.FirstKey, sizeof(dvKey), UT_POINTER, "Key");
    utRegisterField("LastKey", &dvPropertys.LastKey, sizeof(dvKey), UT_POINTER, "Key");
    utRegisterField("Sparsegroup", &dvPropertys.Sparsegroup, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterField("NextSparsegroupProperty", &dvPropertys.NextSparsegroupProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("Relationship", &dvPropertys.Relationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("NextRelationshipProperty", &dvPropertys.NextRelationshipProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("Union", &dvPropertys.Union, sizeof(dvUnion), UT_POINTER, "Union");
    utRegisterField("NextUnionProperty", &dvPropertys.NextUnionProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("union1", &dvPropertys.union1, sizeof(dvPropertyUnion1), UT_UNION, "Type");
    utRegisterUnion("Type", 6);
    utRegisterUnionCase(7, UT_POINTER, sizeof(dvEnum));
    utRegisterUnionCase(8, UT_POINTER, sizeof(dvTypedef));
    utRegisterUnionCase(9, UT_POINTER, sizeof(dvClass));
    utRegisterUnionCase(10, UT_SYM, sizeof(utSym));
    utRegisterUnionCase(0, UT_UINT, sizeof(uint8));
    utRegisterUnionCase(1, UT_UINT, sizeof(uint8));
    utRegisterClass("Sparsegroup", 8, &dvRootData.usedSparsegroup, &dvRootData.allocatedSparsegroup,
        &dvRootData.firstFreeSparsegroup, 165, 4, allocSparsegroup, destroySparsegroup);
    utRegisterField("Sym", &dvSparsegroups.Sym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("Class", &dvSparsegroups.Class, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("NextClassSparsegroup", &dvSparsegroups.NextClassSparsegroup, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterField("PrevClassSparsegroup", &dvSparsegroups.PrevClassSparsegroup, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterField("NextTableClassSparsegroup", &dvSparsegroups.NextTableClassSparsegroup, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterField("FirstProperty", &dvSparsegroups.FirstProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("LastProperty", &dvSparsegroups.LastProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("Relationship", &dvSparsegroups.Relationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterClass("Relationship", 22, &dvRootData.usedRelationship, &dvRootData.allocatedRelationship,
        NULL, 65535, 4, allocRelationship, NULL);
    utRegisterField("Type", &dvRelationships.Type, sizeof(dvRelationshipType), UT_ENUM, "RelationshipType");
    utRegisterField("ParentLabelSym", &dvRelationships.ParentLabelSym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("ChildLabelSym", &dvRelationships.ChildLabelSym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("Mandatory", &dvRelationships.Mandatory, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("Cascade", &dvRelationships.Cascade, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("AccessChild", &dvRelationships.AccessChild, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("AccessParent", &dvRelationships.AccessParent, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("SharedParent", &dvRelationships.SharedParent, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("Sparse", &dvRelationships.Sparse, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("Expanded", &dvRelationships.Expanded, sizeof(uint8), UT_BIT, NULL);
    utRegisterField("Schema", &dvRelationships.Schema, sizeof(dvSchema), UT_POINTER, "Schema");
    utRegisterField("NextSchemaRelationship", &dvRelationships.NextSchemaRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("ParentClass", &dvRelationships.ParentClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("NextClassChildRelationship", &dvRelationships.NextClassChildRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("ChildClass", &dvRelationships.ChildClass, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("NextClassParentRelationship", &dvRelationships.NextClassParentRelationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("FirstProperty", &dvRelationships.FirstProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("LastProperty", &dvRelationships.LastProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("FirstKey", &dvRelationships.FirstKey, sizeof(dvKey), UT_POINTER, "Key");
    utRegisterField("LastKey", &dvRelationships.LastKey, sizeof(dvKey), UT_POINTER, "Key");
    utRegisterField("ParentSparsegroup", &dvRelationships.ParentSparsegroup, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterField("ChildSparsegroup", &dvRelationships.ChildSparsegroup, sizeof(dvSparsegroup), UT_POINTER, "Sparsegroup");
    utRegisterClass("Key", 6, &dvRootData.usedKey, &dvRootData.allocatedKey,
        &dvRootData.firstFreeKey, 196, 4, allocKey, destroyKey);
    utRegisterField("PropertySym", &dvKeys.PropertySym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("LineNum", &dvKeys.LineNum, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("Property", &dvKeys.Property, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("NextPropertyKey", &dvKeys.NextPropertyKey, sizeof(dvKey), UT_POINTER, "Key");
    utRegisterField("Relationship", &dvKeys.Relationship, sizeof(dvRelationship), UT_POINTER, "Relationship");
    utRegisterField("NextRelationshipKey", &dvKeys.NextRelationshipKey, sizeof(dvKey), UT_POINTER, "Key");
    utRegisterClass("Union", 10, &dvRootData.usedUnion, &dvRootData.allocatedUnion,
        &dvRootData.firstFreeUnion, 201, 4, allocUnion, destroyUnion);
    utRegisterField("PropertySym", &dvUnions.PropertySym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("TypeProperty", &dvUnions.TypeProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("Line", &dvUnions.Line, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("Number", &dvUnions.Number, sizeof(uint16), UT_UINT, NULL);
    utRegisterField("FieldNumber", &dvUnions.FieldNumber, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("NumCases", &dvUnions.NumCases, sizeof(uint16), UT_UINT, NULL);
    utRegisterField("Class", &dvUnions.Class, sizeof(dvClass), UT_POINTER, "Class");
    utRegisterField("NextClassUnion", &dvUnions.NextClassUnion, sizeof(dvUnion), UT_POINTER, "Union");
    utRegisterField("FirstProperty", &dvUnions.FirstProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("LastProperty", &dvUnions.LastProperty, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterClass("Case", 5, &dvRootData.usedCase, &dvRootData.allocatedCase,
        &dvRootData.firstFreeCase, 211, 4, allocCase, destroyCase);
    utRegisterField("EntrySym", &dvCases.EntrySym, sizeof(utSym), UT_SYM, NULL);
    utRegisterField("Entry", &dvCases.Entry, sizeof(dvEntry), UT_POINTER, "Entry");
    utRegisterField("NextEntryCase", &dvCases.NextEntryCase, sizeof(dvCase), UT_POINTER, "Case");
    utRegisterField("Property", &dvCases.Property, sizeof(dvProperty), UT_POINTER, "Property");
    utRegisterField("NextPropertyCase", &dvCases.NextPropertyCase, sizeof(dvCase), UT_POINTER, "Case");
    allocRoots();
    allocModpaths();
    allocModules();
    allocLinks();
    allocSchemas();
    allocEnums();
    allocEntrys();
    allocTypedefs();
    allocClasss();
    allocPropertys();
    allocSparsegroups();
    allocRelationships();
    allocKeys();
    allocUnions();
    allocCases();
}

#if defined(DD_DEBUG)
#undef dvRootGetFirstModpath
dvModpath dvRootGetFirstModpath(
    dvRoot _Root)
{
    return dvRoots.FirstModpath[dvRoot2Index(_Root)];
}

#undef dvRootSetFirstModpath
void dvRootSetFirstModpath(
    dvRoot _Root,
    dvModpath value)
{
    dvRoots.FirstModpath[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetLastModpath
dvModpath dvRootGetLastModpath(
    dvRoot _Root)
{
    return dvRoots.LastModpath[dvRoot2Index(_Root)];
}

#undef dvRootSetLastModpath
void dvRootSetLastModpath(
    dvRoot _Root,
    dvModpath value)
{
    dvRoots.LastModpath[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetModpathTableIndex
uint32 dvRootGetModpathTableIndex(
    dvRoot _Root)
{
    return dvRoots.ModpathTableIndex[dvRoot2Index(_Root)];
}

#undef dvRootSetModpathTableIndex
void dvRootSetModpathTableIndex(
    dvRoot _Root,
    uint32 value)
{
    dvRoots.ModpathTableIndex[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetNumModpathTable
uint32 dvRootGetNumModpathTable(
    dvRoot _Root)
{
    return dvRoots.NumModpathTable[dvRoot2Index(_Root)];
}

#undef dvRootSetNumModpathTable
void dvRootSetNumModpathTable(
    dvRoot _Root,
    uint32 value)
{
    dvRoots.NumModpathTable[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetiModpathTable
dvModpath dvRootGetiModpathTable(
    dvRoot _Root,
    uint32 x)
{
    return (dvRoots.ModpathTable)[dvRootGetModpathTableIndex(_Root) + x];
}

#undef dvRootSetiModpathTable
void dvRootSetiModpathTable(
    dvRoot Root,
    uint32 x,
    dvModpath value)
{
    dvRoots.ModpathTable[dvRootGetModpathTableIndex(Root) + x] = value;
}

#undef dvRootGetModpathTable
dvModpath *dvRootGetModpathTable(
    dvRoot Root)
{
    return dvRoots.ModpathTable + dvRootGetModpathTableIndex(Root);
}

#undef dvRootGetNumModpath
uint32 dvRootGetNumModpath(
    dvRoot _Root)
{
    return dvRoots.NumModpath[dvRoot2Index(_Root)];
}

#undef dvRootSetNumModpath
void dvRootSetNumModpath(
    dvRoot _Root,
    uint32 value)
{
    dvRoots.NumModpath[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetFirstModule
dvModule dvRootGetFirstModule(
    dvRoot _Root)
{
    return dvRoots.FirstModule[dvRoot2Index(_Root)];
}

#undef dvRootSetFirstModule
void dvRootSetFirstModule(
    dvRoot _Root,
    dvModule value)
{
    dvRoots.FirstModule[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetLastModule
dvModule dvRootGetLastModule(
    dvRoot _Root)
{
    return dvRoots.LastModule[dvRoot2Index(_Root)];
}

#undef dvRootSetLastModule
void dvRootSetLastModule(
    dvRoot _Root,
    dvModule value)
{
    dvRoots.LastModule[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetModuleTableIndex
uint32 dvRootGetModuleTableIndex(
    dvRoot _Root)
{
    return dvRoots.ModuleTableIndex[dvRoot2Index(_Root)];
}

#undef dvRootSetModuleTableIndex
void dvRootSetModuleTableIndex(
    dvRoot _Root,
    uint32 value)
{
    dvRoots.ModuleTableIndex[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetNumModuleTable
uint32 dvRootGetNumModuleTable(
    dvRoot _Root)
{
    return dvRoots.NumModuleTable[dvRoot2Index(_Root)];
}

#undef dvRootSetNumModuleTable
void dvRootSetNumModuleTable(
    dvRoot _Root,
    uint32 value)
{
    dvRoots.NumModuleTable[dvRoot2Index(_Root)] = value;
}

#undef dvRootGetiModuleTable
dvModule dvRootGetiModuleTable(
    dvRoot _Root,
    uint32 x)
{
    return (dvRoots.ModuleTable)[dvRootGetModuleTableIndex(_Root) + x];
}

#undef dvRootSetiModuleTable
void dvRootSetiModuleTable(
    dvRoot Root,
    uint32 x,
    dvModule value)
{
    dvRoots.ModuleTable[dvRootGetModuleTableIndex(Root) + x] = value;
}

#undef dvRootGetModuleTable
dvModule *dvRootGetModuleTable(
    dvRoot Root)
{
    return dvRoots.ModuleTable + dvRootGetModuleTableIndex(Root);
}

#undef dvRootGetNumModule
uint32 dvRootGetNumModule(
    dvRoot _Root)
{
    return dvRoots.NumModule[dvRoot2Index(_Root)];
}

#undef dvRootSetNumModule
void dvRootSetNumModule(
    dvRoot _Root,
    uint32 value)
{
    dvRoots.NumModule[dvRoot2Index(_Root)] = value;
}

#undef dvModpathGetName
char *dvModpathGetName(
    dvModpath Modpath)
{
    return utSymGetName(dvModpathGetSym(Modpath));
}

#undef dvModuleGetName
char *dvModuleGetName(
    dvModule Module)
{
    return utSymGetName(dvModuleGetSym(Module));
}

#undef dvModpathGetSym
utSym dvModpathGetSym(
    dvModpath _Modpath)
{
    return dvModpaths.Sym[dvModpath2Index(_Modpath)];
}

#undef dvModpathSetSym
void dvModpathSetSym(
    dvModpath _Modpath,
    utSym value)
{
    dvModpaths.Sym[dvModpath2Index(_Modpath)] = value;
}

#undef dvModpathGetRoot
dvRoot dvModpathGetRoot(
    dvModpath _Modpath)
{
    return dvModpaths.Root[dvModpath2Index(_Modpath)];
}

#undef dvModpathSetRoot
void dvModpathSetRoot(
    dvModpath _Modpath,
    dvRoot value)
{
    dvModpaths.Root[dvModpath2Index(_Modpath)] = value;
}

#undef dvModpathGetNextRootModpath
dvModpath dvModpathGetNextRootModpath(
    dvModpath _Modpath)
{
    return dvModpaths.NextRootModpath[dvModpath2Index(_Modpath)];
}

#undef dvModpathSetNextRootModpath
void dvModpathSetNextRootModpath(
    dvModpath _Modpath,
    dvModpath value)
{
    dvModpaths.NextRootModpath[dvModpath2Index(_Modpath)] = value;
}

#undef dvModpathGetPrevRootModpath
dvModpath dvModpathGetPrevRootModpath(
    dvModpath _Modpath)
{
    return dvModpaths.PrevRootModpath[dvModpath2Index(_Modpath)];
}

#undef dvModpathSetPrevRootModpath
void dvModpathSetPrevRootModpath(
    dvModpath _Modpath,
    dvModpath value)
{
    dvModpaths.PrevRootModpath[dvModpath2Index(_Modpath)] = value;
}

#undef dvModpathGetNextTableRootModpath
dvModpath dvModpathGetNextTableRootModpath(
    dvModpath _Modpath)
{
    return dvModpaths.NextTableRootModpath[dvModpath2Index(_Modpath)];
}

#undef dvModpathSetNextTableRootModpath
void dvModpathSetNextTableRootModpath(
    dvModpath _Modpath,
    dvModpath value)
{
    dvModpaths.NextTableRootModpath[dvModpath2Index(_Modpath)] = value;
}

#undef dvModuleGetSym
utSym dvModuleGetSym(
    dvModule _Module)
{
    return dvModules.Sym[dvModule2Index(_Module)];
}

#undef dvModuleSetSym
void dvModuleSetSym(
    dvModule _Module,
    utSym value)
{
    dvModules.Sym[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetPrefixSym
utSym dvModuleGetPrefixSym(
    dvModule _Module)
{
    return dvModules.PrefixSym[dvModule2Index(_Module)];
}

#undef dvModuleSetPrefixSym
void dvModuleSetPrefixSym(
    dvModule _Module,
    utSym value)
{
    dvModules.PrefixSym[dvModule2Index(_Module)] = value;
}

#undef dvModulePersistent
bool dvModulePersistent(
    dvModule _Module)
{
    return (dvModules.Persistent[dvModule2Index(_Module) >> 3] >> (dvModule2Index(_Module) & 7)) & 1;
}

#undef dvModuleSetPersistent
void dvModuleSetPersistent(
    dvModule _Module,
    bool value)
{
    uint32 xModule = dvModule2Index(_Module);

    dvModules.Persistent[xModule >> 3] = (dvModules.Persistent[xModule >> 3] & ~(1 << (xModule & 7))) |
        ((value != 0) << (xModule & 7));
}

#undef dvModuleUndoRedo
bool dvModuleUndoRedo(
    dvModule _Module)
{
    return (dvModules.UndoRedo[dvModule2Index(_Module) >> 3] >> (dvModule2Index(_Module) & 7)) & 1;
}

#undef dvModuleSetUndoRedo
void dvModuleSetUndoRedo(
    dvModule _Module,
    bool value)
{
    uint32 xModule = dvModule2Index(_Module);

    dvModules.UndoRedo[xModule >> 3] = (dvModules.UndoRedo[xModule >> 3] & ~(1 << (xModule & 7))) |
        ((value != 0) << (xModule & 7));
}

#undef dvModuleHasSparseData
bool dvModuleHasSparseData(
    dvModule _Module)
{
    return (dvModules.HasSparseData[dvModule2Index(_Module) >> 3] >> (dvModule2Index(_Module) & 7)) & 1;
}

#undef dvModuleSetHasSparseData
void dvModuleSetHasSparseData(
    dvModule _Module,
    bool value)
{
    uint32 xModule = dvModule2Index(_Module);

    dvModules.HasSparseData[xModule >> 3] = (dvModules.HasSparseData[xModule >> 3] & ~(1 << (xModule & 7))) |
        ((value != 0) << (xModule & 7));
}

#undef dvModuleGetNumFields
uint16 dvModuleGetNumFields(
    dvModule _Module)
{
    return dvModules.NumFields[dvModule2Index(_Module)];
}

#undef dvModuleSetNumFields
void dvModuleSetNumFields(
    dvModule _Module,
    uint16 value)
{
    dvModules.NumFields[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetNumClasses
uint32 dvModuleGetNumClasses(
    dvModule _Module)
{
    return dvModules.NumClasses[dvModule2Index(_Module)];
}

#undef dvModuleSetNumClasses
void dvModuleSetNumClasses(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumClasses[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetNumEnums
uint32 dvModuleGetNumEnums(
    dvModule _Module)
{
    return dvModules.NumEnums[dvModule2Index(_Module)];
}

#undef dvModuleSetNumEnums
void dvModuleSetNumEnums(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumEnums[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetNextRootModule
dvModule dvModuleGetNextRootModule(
    dvModule _Module)
{
    return dvModules.NextRootModule[dvModule2Index(_Module)];
}

#undef dvModuleSetNextRootModule
void dvModuleSetNextRootModule(
    dvModule _Module,
    dvModule value)
{
    dvModules.NextRootModule[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetPrevRootModule
dvModule dvModuleGetPrevRootModule(
    dvModule _Module)
{
    return dvModules.PrevRootModule[dvModule2Index(_Module)];
}

#undef dvModuleSetPrevRootModule
void dvModuleSetPrevRootModule(
    dvModule _Module,
    dvModule value)
{
    dvModules.PrevRootModule[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetNextTableRootModule
dvModule dvModuleGetNextTableRootModule(
    dvModule _Module)
{
    return dvModules.NextTableRootModule[dvModule2Index(_Module)];
}

#undef dvModuleSetNextTableRootModule
void dvModuleSetNextTableRootModule(
    dvModule _Module,
    dvModule value)
{
    dvModules.NextTableRootModule[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetFirstClass
dvClass dvModuleGetFirstClass(
    dvModule _Module)
{
    return dvModules.FirstClass[dvModule2Index(_Module)];
}

#undef dvModuleSetFirstClass
void dvModuleSetFirstClass(
    dvModule _Module,
    dvClass value)
{
    dvModules.FirstClass[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetLastClass
dvClass dvModuleGetLastClass(
    dvModule _Module)
{
    return dvModules.LastClass[dvModule2Index(_Module)];
}

#undef dvModuleSetLastClass
void dvModuleSetLastClass(
    dvModule _Module,
    dvClass value)
{
    dvModules.LastClass[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetClassTableIndex
uint32 dvModuleGetClassTableIndex(
    dvModule _Module)
{
    return dvModules.ClassTableIndex[dvModule2Index(_Module)];
}

#undef dvModuleSetClassTableIndex
void dvModuleSetClassTableIndex(
    dvModule _Module,
    uint32 value)
{
    dvModules.ClassTableIndex[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetNumClassTable
uint32 dvModuleGetNumClassTable(
    dvModule _Module)
{
    return dvModules.NumClassTable[dvModule2Index(_Module)];
}

#undef dvModuleSetNumClassTable
void dvModuleSetNumClassTable(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumClassTable[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetiClassTable
dvClass dvModuleGetiClassTable(
    dvModule _Module,
    uint32 x)
{
    return (dvModules.ClassTable)[dvModuleGetClassTableIndex(_Module) + x];
}

#undef dvModuleSetiClassTable
void dvModuleSetiClassTable(
    dvModule Module,
    uint32 x,
    dvClass value)
{
    dvModules.ClassTable[dvModuleGetClassTableIndex(Module) + x] = value;
}

#undef dvModuleGetClassTable
dvClass *dvModuleGetClassTable(
    dvModule Module)
{
    return dvModules.ClassTable + dvModuleGetClassTableIndex(Module);
}

#undef dvModuleGetNumClass
uint32 dvModuleGetNumClass(
    dvModule _Module)
{
    return dvModules.NumClass[dvModule2Index(_Module)];
}

#undef dvModuleSetNumClass
void dvModuleSetNumClass(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumClass[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetFirstEnum
dvEnum dvModuleGetFirstEnum(
    dvModule _Module)
{
    return dvModules.FirstEnum[dvModule2Index(_Module)];
}

#undef dvModuleSetFirstEnum
void dvModuleSetFirstEnum(
    dvModule _Module,
    dvEnum value)
{
    dvModules.FirstEnum[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetLastEnum
dvEnum dvModuleGetLastEnum(
    dvModule _Module)
{
    return dvModules.LastEnum[dvModule2Index(_Module)];
}

#undef dvModuleSetLastEnum
void dvModuleSetLastEnum(
    dvModule _Module,
    dvEnum value)
{
    dvModules.LastEnum[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetEnumTableIndex
uint32 dvModuleGetEnumTableIndex(
    dvModule _Module)
{
    return dvModules.EnumTableIndex[dvModule2Index(_Module)];
}

#undef dvModuleSetEnumTableIndex
void dvModuleSetEnumTableIndex(
    dvModule _Module,
    uint32 value)
{
    dvModules.EnumTableIndex[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetNumEnumTable
uint32 dvModuleGetNumEnumTable(
    dvModule _Module)
{
    return dvModules.NumEnumTable[dvModule2Index(_Module)];
}

#undef dvModuleSetNumEnumTable
void dvModuleSetNumEnumTable(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumEnumTable[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetiEnumTable
dvEnum dvModuleGetiEnumTable(
    dvModule _Module,
    uint32 x)
{
    return (dvModules.EnumTable)[dvModuleGetEnumTableIndex(_Module) + x];
}

#undef dvModuleSetiEnumTable
void dvModuleSetiEnumTable(
    dvModule Module,
    uint32 x,
    dvEnum value)
{
    dvModules.EnumTable[dvModuleGetEnumTableIndex(Module) + x] = value;
}

#undef dvModuleGetEnumTable
dvEnum *dvModuleGetEnumTable(
    dvModule Module)
{
    return dvModules.EnumTable + dvModuleGetEnumTableIndex(Module);
}

#undef dvModuleGetNumEnum
uint32 dvModuleGetNumEnum(
    dvModule _Module)
{
    return dvModules.NumEnum[dvModule2Index(_Module)];
}

#undef dvModuleSetNumEnum
void dvModuleSetNumEnum(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumEnum[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetFirstTypedef
dvTypedef dvModuleGetFirstTypedef(
    dvModule _Module)
{
    return dvModules.FirstTypedef[dvModule2Index(_Module)];
}

#undef dvModuleSetFirstTypedef
void dvModuleSetFirstTypedef(
    dvModule _Module,
    dvTypedef value)
{
    dvModules.FirstTypedef[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetLastTypedef
dvTypedef dvModuleGetLastTypedef(
    dvModule _Module)
{
    return dvModules.LastTypedef[dvModule2Index(_Module)];
}

#undef dvModuleSetLastTypedef
void dvModuleSetLastTypedef(
    dvModule _Module,
    dvTypedef value)
{
    dvModules.LastTypedef[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetTypedefTableIndex
uint32 dvModuleGetTypedefTableIndex(
    dvModule _Module)
{
    return dvModules.TypedefTableIndex[dvModule2Index(_Module)];
}

#undef dvModuleSetTypedefTableIndex
void dvModuleSetTypedefTableIndex(
    dvModule _Module,
    uint32 value)
{
    dvModules.TypedefTableIndex[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetNumTypedefTable
uint32 dvModuleGetNumTypedefTable(
    dvModule _Module)
{
    return dvModules.NumTypedefTable[dvModule2Index(_Module)];
}

#undef dvModuleSetNumTypedefTable
void dvModuleSetNumTypedefTable(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumTypedefTable[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetiTypedefTable
dvTypedef dvModuleGetiTypedefTable(
    dvModule _Module,
    uint32 x)
{
    return (dvModules.TypedefTable)[dvModuleGetTypedefTableIndex(_Module) + x];
}

#undef dvModuleSetiTypedefTable
void dvModuleSetiTypedefTable(
    dvModule Module,
    uint32 x,
    dvTypedef value)
{
    dvModules.TypedefTable[dvModuleGetTypedefTableIndex(Module) + x] = value;
}

#undef dvModuleGetTypedefTable
dvTypedef *dvModuleGetTypedefTable(
    dvModule Module)
{
    return dvModules.TypedefTable + dvModuleGetTypedefTableIndex(Module);
}

#undef dvModuleGetNumTypedef
uint32 dvModuleGetNumTypedef(
    dvModule _Module)
{
    return dvModules.NumTypedef[dvModule2Index(_Module)];
}

#undef dvModuleSetNumTypedef
void dvModuleSetNumTypedef(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumTypedef[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetFirstSchema
dvSchema dvModuleGetFirstSchema(
    dvModule _Module)
{
    return dvModules.FirstSchema[dvModule2Index(_Module)];
}

#undef dvModuleSetFirstSchema
void dvModuleSetFirstSchema(
    dvModule _Module,
    dvSchema value)
{
    dvModules.FirstSchema[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetLastSchema
dvSchema dvModuleGetLastSchema(
    dvModule _Module)
{
    return dvModules.LastSchema[dvModule2Index(_Module)];
}

#undef dvModuleSetLastSchema
void dvModuleSetLastSchema(
    dvModule _Module,
    dvSchema value)
{
    dvModules.LastSchema[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetSchemaTableIndex
uint32 dvModuleGetSchemaTableIndex(
    dvModule _Module)
{
    return dvModules.SchemaTableIndex[dvModule2Index(_Module)];
}

#undef dvModuleSetSchemaTableIndex
void dvModuleSetSchemaTableIndex(
    dvModule _Module,
    uint32 value)
{
    dvModules.SchemaTableIndex[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetNumSchemaTable
uint32 dvModuleGetNumSchemaTable(
    dvModule _Module)
{
    return dvModules.NumSchemaTable[dvModule2Index(_Module)];
}

#undef dvModuleSetNumSchemaTable
void dvModuleSetNumSchemaTable(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumSchemaTable[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetiSchemaTable
dvSchema dvModuleGetiSchemaTable(
    dvModule _Module,
    uint32 x)
{
    return (dvModules.SchemaTable)[dvModuleGetSchemaTableIndex(_Module) + x];
}

#undef dvModuleSetiSchemaTable
void dvModuleSetiSchemaTable(
    dvModule Module,
    uint32 x,
    dvSchema value)
{
    dvModules.SchemaTable[dvModuleGetSchemaTableIndex(Module) + x] = value;
}

#undef dvModuleGetSchemaTable
dvSchema *dvModuleGetSchemaTable(
    dvModule Module)
{
    return dvModules.SchemaTable + dvModuleGetSchemaTableIndex(Module);
}

#undef dvModuleGetNumSchema
uint32 dvModuleGetNumSchema(
    dvModule _Module)
{
    return dvModules.NumSchema[dvModule2Index(_Module)];
}

#undef dvModuleSetNumSchema
void dvModuleSetNumSchema(
    dvModule _Module,
    uint32 value)
{
    dvModules.NumSchema[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetFirstImportLink
dvLink dvModuleGetFirstImportLink(
    dvModule _Module)
{
    return dvModules.FirstImportLink[dvModule2Index(_Module)];
}

#undef dvModuleSetFirstImportLink
void dvModuleSetFirstImportLink(
    dvModule _Module,
    dvLink value)
{
    dvModules.FirstImportLink[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetLastImportLink
dvLink dvModuleGetLastImportLink(
    dvModule _Module)
{
    return dvModules.LastImportLink[dvModule2Index(_Module)];
}

#undef dvModuleSetLastImportLink
void dvModuleSetLastImportLink(
    dvModule _Module,
    dvLink value)
{
    dvModules.LastImportLink[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetFirstExportLink
dvLink dvModuleGetFirstExportLink(
    dvModule _Module)
{
    return dvModules.FirstExportLink[dvModule2Index(_Module)];
}

#undef dvModuleSetFirstExportLink
void dvModuleSetFirstExportLink(
    dvModule _Module,
    dvLink value)
{
    dvModules.FirstExportLink[dvModule2Index(_Module)] = value;
}

#undef dvModuleGetLastExportLink
dvLink dvModuleGetLastExportLink(
    dvModule _Module)
{
    return dvModules.LastExportLink[dvModule2Index(_Module)];
}

#undef dvModuleSetLastExportLink
void dvModuleSetLastExportLink(
    dvModule _Module,
    dvLink value)
{
    dvModules.LastExportLink[dvModule2Index(_Module)] = value;
}

#undef dvClassGetName
char *dvClassGetName(
    dvClass Class)
{
    return utSymGetName(dvClassGetSym(Class));
}

#undef dvEnumGetName
char *dvEnumGetName(
    dvEnum Enum)
{
    return utSymGetName(dvEnumGetSym(Enum));
}

#undef dvTypedefGetName
char *dvTypedefGetName(
    dvTypedef Typedef)
{
    return utSymGetName(dvTypedefGetSym(Typedef));
}

#undef dvSchemaGetName
char *dvSchemaGetName(
    dvSchema Schema)
{
    return utSymGetName(dvSchemaGetSym(Schema));
}

#undef dvLinkGetImportModule
dvModule dvLinkGetImportModule(
    dvLink _Link)
{
    return dvLinks.ImportModule[dvLink2Index(_Link)];
}

#undef dvLinkSetImportModule
void dvLinkSetImportModule(
    dvLink _Link,
    dvModule value)
{
    dvLinks.ImportModule[dvLink2Index(_Link)] = value;
}

#undef dvLinkGetNextModuleImportLink
dvLink dvLinkGetNextModuleImportLink(
    dvLink _Link)
{
    return dvLinks.NextModuleImportLink[dvLink2Index(_Link)];
}

#undef dvLinkSetNextModuleImportLink
void dvLinkSetNextModuleImportLink(
    dvLink _Link,
    dvLink value)
{
    dvLinks.NextModuleImportLink[dvLink2Index(_Link)] = value;
}

#undef dvLinkGetExportModule
dvModule dvLinkGetExportModule(
    dvLink _Link)
{
    return dvLinks.ExportModule[dvLink2Index(_Link)];
}

#undef dvLinkSetExportModule
void dvLinkSetExportModule(
    dvLink _Link,
    dvModule value)
{
    dvLinks.ExportModule[dvLink2Index(_Link)] = value;
}

#undef dvLinkGetNextModuleExportLink
dvLink dvLinkGetNextModuleExportLink(
    dvLink _Link)
{
    return dvLinks.NextModuleExportLink[dvLink2Index(_Link)];
}

#undef dvLinkSetNextModuleExportLink
void dvLinkSetNextModuleExportLink(
    dvLink _Link,
    dvLink value)
{
    dvLinks.NextModuleExportLink[dvLink2Index(_Link)] = value;
}

#undef dvSchemaGetSym
utSym dvSchemaGetSym(
    dvSchema _Schema)
{
    return dvSchemas.Sym[dvSchema2Index(_Schema)];
}

#undef dvSchemaSetSym
void dvSchemaSetSym(
    dvSchema _Schema,
    utSym value)
{
    dvSchemas.Sym[dvSchema2Index(_Schema)] = value;
}

#undef dvSchemaGetModule
dvModule dvSchemaGetModule(
    dvSchema _Schema)
{
    return dvSchemas.Module[dvSchema2Index(_Schema)];
}

#undef dvSchemaSetModule
void dvSchemaSetModule(
    dvSchema _Schema,
    dvModule value)
{
    dvSchemas.Module[dvSchema2Index(_Schema)] = value;
}

#undef dvSchemaGetNextModuleSchema
dvSchema dvSchemaGetNextModuleSchema(
    dvSchema _Schema)
{
    return dvSchemas.NextModuleSchema[dvSchema2Index(_Schema)];
}

#undef dvSchemaSetNextModuleSchema
void dvSchemaSetNextModuleSchema(
    dvSchema _Schema,
    dvSchema value)
{
    dvSchemas.NextModuleSchema[dvSchema2Index(_Schema)] = value;
}

#undef dvSchemaGetPrevModuleSchema
dvSchema dvSchemaGetPrevModuleSchema(
    dvSchema _Schema)
{
    return dvSchemas.PrevModuleSchema[dvSchema2Index(_Schema)];
}

#undef dvSchemaSetPrevModuleSchema
void dvSchemaSetPrevModuleSchema(
    dvSchema _Schema,
    dvSchema value)
{
    dvSchemas.PrevModuleSchema[dvSchema2Index(_Schema)] = value;
}

#undef dvSchemaGetNextTableModuleSchema
dvSchema dvSchemaGetNextTableModuleSchema(
    dvSchema _Schema)
{
    return dvSchemas.NextTableModuleSchema[dvSchema2Index(_Schema)];
}

#undef dvSchemaSetNextTableModuleSchema
void dvSchemaSetNextTableModuleSchema(
    dvSchema _Schema,
    dvSchema value)
{
    dvSchemas.NextTableModuleSchema[dvSchema2Index(_Schema)] = value;
}

#undef dvSchemaGetFirstRelationship
dvRelationship dvSchemaGetFirstRelationship(
    dvSchema _Schema)
{
    return dvSchemas.FirstRelationship[dvSchema2Index(_Schema)];
}

#undef dvSchemaSetFirstRelationship
void dvSchemaSetFirstRelationship(
    dvSchema _Schema,
    dvRelationship value)
{
    dvSchemas.FirstRelationship[dvSchema2Index(_Schema)] = value;
}

#undef dvSchemaGetLastRelationship
dvRelationship dvSchemaGetLastRelationship(
    dvSchema _Schema)
{
    return dvSchemas.LastRelationship[dvSchema2Index(_Schema)];
}

#undef dvSchemaSetLastRelationship
void dvSchemaSetLastRelationship(
    dvSchema _Schema,
    dvRelationship value)
{
    dvSchemas.LastRelationship[dvSchema2Index(_Schema)] = value;
}

#undef dvEnumGetSym
utSym dvEnumGetSym(
    dvEnum _Enum)
{
    return dvEnums.Sym[dvEnum2Index(_Enum)];
}

#undef dvEnumSetSym
void dvEnumSetSym(
    dvEnum _Enum,
    utSym value)
{
    dvEnums.Sym[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetPrefixSym
utSym dvEnumGetPrefixSym(
    dvEnum _Enum)
{
    return dvEnums.PrefixSym[dvEnum2Index(_Enum)];
}

#undef dvEnumSetPrefixSym
void dvEnumSetPrefixSym(
    dvEnum _Enum,
    utSym value)
{
    dvEnums.PrefixSym[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetNumEntries
uint16 dvEnumGetNumEntries(
    dvEnum _Enum)
{
    return dvEnums.NumEntries[dvEnum2Index(_Enum)];
}

#undef dvEnumSetNumEntries
void dvEnumSetNumEntries(
    dvEnum _Enum,
    uint16 value)
{
    dvEnums.NumEntries[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetModule
dvModule dvEnumGetModule(
    dvEnum _Enum)
{
    return dvEnums.Module[dvEnum2Index(_Enum)];
}

#undef dvEnumSetModule
void dvEnumSetModule(
    dvEnum _Enum,
    dvModule value)
{
    dvEnums.Module[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetNextModuleEnum
dvEnum dvEnumGetNextModuleEnum(
    dvEnum _Enum)
{
    return dvEnums.NextModuleEnum[dvEnum2Index(_Enum)];
}

#undef dvEnumSetNextModuleEnum
void dvEnumSetNextModuleEnum(
    dvEnum _Enum,
    dvEnum value)
{
    dvEnums.NextModuleEnum[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetPrevModuleEnum
dvEnum dvEnumGetPrevModuleEnum(
    dvEnum _Enum)
{
    return dvEnums.PrevModuleEnum[dvEnum2Index(_Enum)];
}

#undef dvEnumSetPrevModuleEnum
void dvEnumSetPrevModuleEnum(
    dvEnum _Enum,
    dvEnum value)
{
    dvEnums.PrevModuleEnum[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetNextTableModuleEnum
dvEnum dvEnumGetNextTableModuleEnum(
    dvEnum _Enum)
{
    return dvEnums.NextTableModuleEnum[dvEnum2Index(_Enum)];
}

#undef dvEnumSetNextTableModuleEnum
void dvEnumSetNextTableModuleEnum(
    dvEnum _Enum,
    dvEnum value)
{
    dvEnums.NextTableModuleEnum[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetFirstEntry
dvEntry dvEnumGetFirstEntry(
    dvEnum _Enum)
{
    return dvEnums.FirstEntry[dvEnum2Index(_Enum)];
}

#undef dvEnumSetFirstEntry
void dvEnumSetFirstEntry(
    dvEnum _Enum,
    dvEntry value)
{
    dvEnums.FirstEntry[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetLastEntry
dvEntry dvEnumGetLastEntry(
    dvEnum _Enum)
{
    return dvEnums.LastEntry[dvEnum2Index(_Enum)];
}

#undef dvEnumSetLastEntry
void dvEnumSetLastEntry(
    dvEnum _Enum,
    dvEntry value)
{
    dvEnums.LastEntry[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetEntryTableIndex
uint32 dvEnumGetEntryTableIndex(
    dvEnum _Enum)
{
    return dvEnums.EntryTableIndex[dvEnum2Index(_Enum)];
}

#undef dvEnumSetEntryTableIndex
void dvEnumSetEntryTableIndex(
    dvEnum _Enum,
    uint32 value)
{
    dvEnums.EntryTableIndex[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetNumEntryTable
uint32 dvEnumGetNumEntryTable(
    dvEnum _Enum)
{
    return dvEnums.NumEntryTable[dvEnum2Index(_Enum)];
}

#undef dvEnumSetNumEntryTable
void dvEnumSetNumEntryTable(
    dvEnum _Enum,
    uint32 value)
{
    dvEnums.NumEntryTable[dvEnum2Index(_Enum)] = value;
}

#undef dvEnumGetiEntryTable
dvEntry dvEnumGetiEntryTable(
    dvEnum _Enum,
    uint32 x)
{
    return (dvEnums.EntryTable)[dvEnumGetEntryTableIndex(_Enum) + x];
}

#undef dvEnumSetiEntryTable
void dvEnumSetiEntryTable(
    dvEnum Enum,
    uint32 x,
    dvEntry value)
{
    dvEnums.EntryTable[dvEnumGetEntryTableIndex(Enum) + x] = value;
}

#undef dvEnumGetEntryTable
dvEntry *dvEnumGetEntryTable(
    dvEnum Enum)
{
    return dvEnums.EntryTable + dvEnumGetEntryTableIndex(Enum);
}

#undef dvEnumGetNumEntry
uint32 dvEnumGetNumEntry(
    dvEnum _Enum)
{
    return dvEnums.NumEntry[dvEnum2Index(_Enum)];
}

#undef dvEnumSetNumEntry
void dvEnumSetNumEntry(
    dvEnum _Enum,
    uint32 value)
{
    dvEnums.NumEntry[dvEnum2Index(_Enum)] = value;
}

#undef dvEntryGetName
char *dvEntryGetName(
    dvEntry Entry)
{
    return utSymGetName(dvEntryGetSym(Entry));
}

#undef dvEntryGetSym
utSym dvEntryGetSym(
    dvEntry _Entry)
{
    return dvEntrys.Sym[dvEntry2Index(_Entry)];
}

#undef dvEntrySetSym
void dvEntrySetSym(
    dvEntry _Entry,
    utSym value)
{
    dvEntrys.Sym[dvEntry2Index(_Entry)] = value;
}

#undef dvEntryGetValue
uint32 dvEntryGetValue(
    dvEntry _Entry)
{
    return dvEntrys.Value[dvEntry2Index(_Entry)];
}

#undef dvEntrySetValue
void dvEntrySetValue(
    dvEntry _Entry,
    uint32 value)
{
    dvEntrys.Value[dvEntry2Index(_Entry)] = value;
}

#undef dvEntryGetEnum
dvEnum dvEntryGetEnum(
    dvEntry _Entry)
{
    return dvEntrys.Enum[dvEntry2Index(_Entry)];
}

#undef dvEntrySetEnum
void dvEntrySetEnum(
    dvEntry _Entry,
    dvEnum value)
{
    dvEntrys.Enum[dvEntry2Index(_Entry)] = value;
}

#undef dvEntryGetNextEnumEntry
dvEntry dvEntryGetNextEnumEntry(
    dvEntry _Entry)
{
    return dvEntrys.NextEnumEntry[dvEntry2Index(_Entry)];
}

#undef dvEntrySetNextEnumEntry
void dvEntrySetNextEnumEntry(
    dvEntry _Entry,
    dvEntry value)
{
    dvEntrys.NextEnumEntry[dvEntry2Index(_Entry)] = value;
}

#undef dvEntryGetPrevEnumEntry
dvEntry dvEntryGetPrevEnumEntry(
    dvEntry _Entry)
{
    return dvEntrys.PrevEnumEntry[dvEntry2Index(_Entry)];
}

#undef dvEntrySetPrevEnumEntry
void dvEntrySetPrevEnumEntry(
    dvEntry _Entry,
    dvEntry value)
{
    dvEntrys.PrevEnumEntry[dvEntry2Index(_Entry)] = value;
}

#undef dvEntryGetNextTableEnumEntry
dvEntry dvEntryGetNextTableEnumEntry(
    dvEntry _Entry)
{
    return dvEntrys.NextTableEnumEntry[dvEntry2Index(_Entry)];
}

#undef dvEntrySetNextTableEnumEntry
void dvEntrySetNextTableEnumEntry(
    dvEntry _Entry,
    dvEntry value)
{
    dvEntrys.NextTableEnumEntry[dvEntry2Index(_Entry)] = value;
}

#undef dvEntryGetFirstCase
dvCase dvEntryGetFirstCase(
    dvEntry _Entry)
{
    return dvEntrys.FirstCase[dvEntry2Index(_Entry)];
}

#undef dvEntrySetFirstCase
void dvEntrySetFirstCase(
    dvEntry _Entry,
    dvCase value)
{
    dvEntrys.FirstCase[dvEntry2Index(_Entry)] = value;
}

#undef dvEntryGetLastCase
dvCase dvEntryGetLastCase(
    dvEntry _Entry)
{
    return dvEntrys.LastCase[dvEntry2Index(_Entry)];
}

#undef dvEntrySetLastCase
void dvEntrySetLastCase(
    dvEntry _Entry,
    dvCase value)
{
    dvEntrys.LastCase[dvEntry2Index(_Entry)] = value;
}

#undef dvTypedefGetSym
utSym dvTypedefGetSym(
    dvTypedef _Typedef)
{
    return dvTypedefs.Sym[dvTypedef2Index(_Typedef)];
}

#undef dvTypedefSetSym
void dvTypedefSetSym(
    dvTypedef _Typedef,
    utSym value)
{
    dvTypedefs.Sym[dvTypedef2Index(_Typedef)] = value;
}

#undef dvTypedefGetInitializerIndex
uint32 dvTypedefGetInitializerIndex(
    dvTypedef _Typedef)
{
    return dvTypedefs.InitializerIndex[dvTypedef2Index(_Typedef)];
}

#undef dvTypedefSetInitializerIndex
void dvTypedefSetInitializerIndex(
    dvTypedef _Typedef,
    uint32 value)
{
    dvTypedefs.InitializerIndex[dvTypedef2Index(_Typedef)] = value;
}

#undef dvTypedefGetNumInitializer
uint32 dvTypedefGetNumInitializer(
    dvTypedef _Typedef)
{
    return dvTypedefs.NumInitializer[dvTypedef2Index(_Typedef)];
}

#undef dvTypedefSetNumInitializer
void dvTypedefSetNumInitializer(
    dvTypedef _Typedef,
    uint32 value)
{
    dvTypedefs.NumInitializer[dvTypedef2Index(_Typedef)] = value;
}

#undef dvTypedefGetiInitializer
char dvTypedefGetiInitializer(
    dvTypedef _Typedef,
    uint32 x)
{
    return (dvTypedefs.Initializer)[dvTypedefGetInitializerIndex(_Typedef) + x];
}

#undef dvTypedefSetiInitializer
void dvTypedefSetiInitializer(
    dvTypedef Typedef,
    uint32 x,
    char value)
{
    dvTypedefs.Initializer[dvTypedefGetInitializerIndex(Typedef) + x] = value;
}

#undef dvTypedefGetInitializer
char *dvTypedefGetInitializer(
    dvTypedef Typedef)
{
    return dvTypedefs.Initializer + dvTypedefGetInitializerIndex(Typedef);
}

#undef dvTypedefGetModule
dvModule dvTypedefGetModule(
    dvTypedef _Typedef)
{
    return dvTypedefs.Module[dvTypedef2Index(_Typedef)];
}

#undef dvTypedefSetModule
void dvTypedefSetModule(
    dvTypedef _Typedef,
    dvModule value)
{
    dvTypedefs.Module[dvTypedef2Index(_Typedef)] = value;
}

#undef dvTypedefGetNextModuleTypedef
dvTypedef dvTypedefGetNextModuleTypedef(
    dvTypedef _Typedef)
{
    return dvTypedefs.NextModuleTypedef[dvTypedef2Index(_Typedef)];
}

#undef dvTypedefSetNextModuleTypedef
void dvTypedefSetNextModuleTypedef(
    dvTypedef _Typedef,
    dvTypedef value)
{
    dvTypedefs.NextModuleTypedef[dvTypedef2Index(_Typedef)] = value;
}

#undef dvTypedefGetPrevModuleTypedef
dvTypedef dvTypedefGetPrevModuleTypedef(
    dvTypedef _Typedef)
{
    return dvTypedefs.PrevModuleTypedef[dvTypedef2Index(_Typedef)];
}

#undef dvTypedefSetPrevModuleTypedef
void dvTypedefSetPrevModuleTypedef(
    dvTypedef _Typedef,
    dvTypedef value)
{
    dvTypedefs.PrevModuleTypedef[dvTypedef2Index(_Typedef)] = value;
}

#undef dvTypedefGetNextTableModuleTypedef
dvTypedef dvTypedefGetNextTableModuleTypedef(
    dvTypedef _Typedef)
{
    return dvTypedefs.NextTableModuleTypedef[dvTypedef2Index(_Typedef)];
}

#undef dvTypedefSetNextTableModuleTypedef
void dvTypedefSetNextTableModuleTypedef(
    dvTypedef _Typedef,
    dvTypedef value)
{
    dvTypedefs.NextTableModuleTypedef[dvTypedef2Index(_Typedef)] = value;
}

#undef dvClassGetSym
utSym dvClassGetSym(
    dvClass _Class)
{
    return dvClasss.Sym[dvClass2Index(_Class)];
}

#undef dvClassSetSym
void dvClassSetSym(
    dvClass _Class,
    utSym value)
{
    dvClasss.Sym[dvClass2Index(_Class)] = value;
}

#undef dvClassGetMemoryStyle
dvMemoryStyle dvClassGetMemoryStyle(
    dvClass _Class)
{
    return dvClasss.MemoryStyle[dvClass2Index(_Class)];
}

#undef dvClassSetMemoryStyle
void dvClassSetMemoryStyle(
    dvClass _Class,
    dvMemoryStyle value)
{
    dvClasss.MemoryStyle[dvClass2Index(_Class)] = value;
}

#undef dvClassGetReferenceSize
uint8 dvClassGetReferenceSize(
    dvClass _Class)
{
    return dvClasss.ReferenceSize[dvClass2Index(_Class)];
}

#undef dvClassSetReferenceSize
void dvClassSetReferenceSize(
    dvClass _Class,
    uint8 value)
{
    dvClasss.ReferenceSize[dvClass2Index(_Class)] = value;
}

#undef dvClassGenerateArrayClass
bool dvClassGenerateArrayClass(
    dvClass _Class)
{
    return (dvClasss.GenerateArrayClass[dvClass2Index(_Class) >> 3] >> (dvClass2Index(_Class) & 7)) & 1;
}

#undef dvClassSetGenerateArrayClass
void dvClassSetGenerateArrayClass(
    dvClass _Class,
    bool value)
{
    uint32 xClass = dvClass2Index(_Class);

    dvClasss.GenerateArrayClass[xClass >> 3] = (dvClasss.GenerateArrayClass[xClass >> 3] & ~(1 << (xClass & 7))) |
        ((value != 0) << (xClass & 7));
}

#undef dvClassGenerateAttributes
bool dvClassGenerateAttributes(
    dvClass _Class)
{
    return (dvClasss.GenerateAttributes[dvClass2Index(_Class) >> 3] >> (dvClass2Index(_Class) & 7)) & 1;
}

#undef dvClassSetGenerateAttributes
void dvClassSetGenerateAttributes(
    dvClass _Class,
    bool value)
{
    uint32 xClass = dvClass2Index(_Class);

    dvClasss.GenerateAttributes[xClass >> 3] = (dvClasss.GenerateAttributes[xClass >> 3] & ~(1 << (xClass & 7))) |
        ((value != 0) << (xClass & 7));
}

#undef dvClassSparse
bool dvClassSparse(
    dvClass _Class)
{
    return (dvClasss.Sparse[dvClass2Index(_Class) >> 3] >> (dvClass2Index(_Class) & 7)) & 1;
}

#undef dvClassSetSparse
void dvClassSetSparse(
    dvClass _Class,
    bool value)
{
    uint32 xClass = dvClass2Index(_Class);

    dvClasss.Sparse[xClass >> 3] = (dvClasss.Sparse[xClass >> 3] & ~(1 << (xClass & 7))) |
        ((value != 0) << (xClass & 7));
}

#undef dvClassGetNumFields
uint16 dvClassGetNumFields(
    dvClass _Class)
{
    return dvClasss.NumFields[dvClass2Index(_Class)];
}

#undef dvClassSetNumFields
void dvClassSetNumFields(
    dvClass _Class,
    uint16 value)
{
    dvClasss.NumFields[dvClass2Index(_Class)] = value;
}

#undef dvClassGetNumber
uint16 dvClassGetNumber(
    dvClass _Class)
{
    return dvClasss.Number[dvClass2Index(_Class)];
}

#undef dvClassSetNumber
void dvClassSetNumber(
    dvClass _Class,
    uint16 value)
{
    dvClasss.Number[dvClass2Index(_Class)] = value;
}

#undef dvClassGetModule
dvModule dvClassGetModule(
    dvClass _Class)
{
    return dvClasss.Module[dvClass2Index(_Class)];
}

#undef dvClassSetModule
void dvClassSetModule(
    dvClass _Class,
    dvModule value)
{
    dvClasss.Module[dvClass2Index(_Class)] = value;
}

#undef dvClassGetNextModuleClass
dvClass dvClassGetNextModuleClass(
    dvClass _Class)
{
    return dvClasss.NextModuleClass[dvClass2Index(_Class)];
}

#undef dvClassSetNextModuleClass
void dvClassSetNextModuleClass(
    dvClass _Class,
    dvClass value)
{
    dvClasss.NextModuleClass[dvClass2Index(_Class)] = value;
}

#undef dvClassGetPrevModuleClass
dvClass dvClassGetPrevModuleClass(
    dvClass _Class)
{
    return dvClasss.PrevModuleClass[dvClass2Index(_Class)];
}

#undef dvClassSetPrevModuleClass
void dvClassSetPrevModuleClass(
    dvClass _Class,
    dvClass value)
{
    dvClasss.PrevModuleClass[dvClass2Index(_Class)] = value;
}

#undef dvClassGetNextTableModuleClass
dvClass dvClassGetNextTableModuleClass(
    dvClass _Class)
{
    return dvClasss.NextTableModuleClass[dvClass2Index(_Class)];
}

#undef dvClassSetNextTableModuleClass
void dvClassSetNextTableModuleClass(
    dvClass _Class,
    dvClass value)
{
    dvClasss.NextTableModuleClass[dvClass2Index(_Class)] = value;
}

#undef dvClassGetFirstProperty
dvProperty dvClassGetFirstProperty(
    dvClass _Class)
{
    return dvClasss.FirstProperty[dvClass2Index(_Class)];
}

#undef dvClassSetFirstProperty
void dvClassSetFirstProperty(
    dvClass _Class,
    dvProperty value)
{
    dvClasss.FirstProperty[dvClass2Index(_Class)] = value;
}

#undef dvClassGetLastProperty
dvProperty dvClassGetLastProperty(
    dvClass _Class)
{
    return dvClasss.LastProperty[dvClass2Index(_Class)];
}

#undef dvClassSetLastProperty
void dvClassSetLastProperty(
    dvClass _Class,
    dvProperty value)
{
    dvClasss.LastProperty[dvClass2Index(_Class)] = value;
}

#undef dvClassGetPropertyTableIndex
uint32 dvClassGetPropertyTableIndex(
    dvClass _Class)
{
    return dvClasss.PropertyTableIndex[dvClass2Index(_Class)];
}

#undef dvClassSetPropertyTableIndex
void dvClassSetPropertyTableIndex(
    dvClass _Class,
    uint32 value)
{
    dvClasss.PropertyTableIndex[dvClass2Index(_Class)] = value;
}

#undef dvClassGetNumPropertyTable
uint32 dvClassGetNumPropertyTable(
    dvClass _Class)
{
    return dvClasss.NumPropertyTable[dvClass2Index(_Class)];
}

#undef dvClassSetNumPropertyTable
void dvClassSetNumPropertyTable(
    dvClass _Class,
    uint32 value)
{
    dvClasss.NumPropertyTable[dvClass2Index(_Class)] = value;
}

#undef dvClassGetiPropertyTable
dvProperty dvClassGetiPropertyTable(
    dvClass _Class,
    uint32 x)
{
    return (dvClasss.PropertyTable)[dvClassGetPropertyTableIndex(_Class) + x];
}

#undef dvClassSetiPropertyTable
void dvClassSetiPropertyTable(
    dvClass Class,
    uint32 x,
    dvProperty value)
{
    dvClasss.PropertyTable[dvClassGetPropertyTableIndex(Class) + x] = value;
}

#undef dvClassGetPropertyTable
dvProperty *dvClassGetPropertyTable(
    dvClass Class)
{
    return dvClasss.PropertyTable + dvClassGetPropertyTableIndex(Class);
}

#undef dvClassGetNumProperty
uint32 dvClassGetNumProperty(
    dvClass _Class)
{
    return dvClasss.NumProperty[dvClass2Index(_Class)];
}

#undef dvClassSetNumProperty
void dvClassSetNumProperty(
    dvClass _Class,
    uint32 value)
{
    dvClasss.NumProperty[dvClass2Index(_Class)] = value;
}

#undef dvClassGetFreeListProperty
dvProperty dvClassGetFreeListProperty(
    dvClass _Class)
{
    return dvClasss.FreeListProperty[dvClass2Index(_Class)];
}

#undef dvClassSetFreeListProperty
void dvClassSetFreeListProperty(
    dvClass _Class,
    dvProperty value)
{
    dvClasss.FreeListProperty[dvClass2Index(_Class)] = value;
}

#undef dvClassGetFirstSparsegroup
dvSparsegroup dvClassGetFirstSparsegroup(
    dvClass _Class)
{
    return dvClasss.FirstSparsegroup[dvClass2Index(_Class)];
}

#undef dvClassSetFirstSparsegroup
void dvClassSetFirstSparsegroup(
    dvClass _Class,
    dvSparsegroup value)
{
    dvClasss.FirstSparsegroup[dvClass2Index(_Class)] = value;
}

#undef dvClassGetLastSparsegroup
dvSparsegroup dvClassGetLastSparsegroup(
    dvClass _Class)
{
    return dvClasss.LastSparsegroup[dvClass2Index(_Class)];
}

#undef dvClassSetLastSparsegroup
void dvClassSetLastSparsegroup(
    dvClass _Class,
    dvSparsegroup value)
{
    dvClasss.LastSparsegroup[dvClass2Index(_Class)] = value;
}

#undef dvClassGetSparsegroupTableIndex
uint32 dvClassGetSparsegroupTableIndex(
    dvClass _Class)
{
    return dvClasss.SparsegroupTableIndex[dvClass2Index(_Class)];
}

#undef dvClassSetSparsegroupTableIndex
void dvClassSetSparsegroupTableIndex(
    dvClass _Class,
    uint32 value)
{
    dvClasss.SparsegroupTableIndex[dvClass2Index(_Class)] = value;
}

#undef dvClassGetNumSparsegroupTable
uint32 dvClassGetNumSparsegroupTable(
    dvClass _Class)
{
    return dvClasss.NumSparsegroupTable[dvClass2Index(_Class)];
}

#undef dvClassSetNumSparsegroupTable
void dvClassSetNumSparsegroupTable(
    dvClass _Class,
    uint32 value)
{
    dvClasss.NumSparsegroupTable[dvClass2Index(_Class)] = value;
}

#undef dvClassGetiSparsegroupTable
dvSparsegroup dvClassGetiSparsegroupTable(
    dvClass _Class,
    uint32 x)
{
    return (dvClasss.SparsegroupTable)[dvClassGetSparsegroupTableIndex(_Class) + x];
}

#undef dvClassSetiSparsegroupTable
void dvClassSetiSparsegroupTable(
    dvClass Class,
    uint32 x,
    dvSparsegroup value)
{
    dvClasss.SparsegroupTable[dvClassGetSparsegroupTableIndex(Class) + x] = value;
}

#undef dvClassGetSparsegroupTable
dvSparsegroup *dvClassGetSparsegroupTable(
    dvClass Class)
{
    return dvClasss.SparsegroupTable + dvClassGetSparsegroupTableIndex(Class);
}

#undef dvClassGetNumSparsegroup
uint32 dvClassGetNumSparsegroup(
    dvClass _Class)
{
    return dvClasss.NumSparsegroup[dvClass2Index(_Class)];
}

#undef dvClassSetNumSparsegroup
void dvClassSetNumSparsegroup(
    dvClass _Class,
    uint32 value)
{
    dvClasss.NumSparsegroup[dvClass2Index(_Class)] = value;
}

#undef dvClassGetBaseClass
dvClass dvClassGetBaseClass(
    dvClass _Class)
{
    return dvClasss.BaseClass[dvClass2Index(_Class)];
}

#undef dvClassSetBaseClass
void dvClassSetBaseClass(
    dvClass _Class,
    dvClass value)
{
    dvClasss.BaseClass[dvClass2Index(_Class)] = value;
}

#undef dvClassGetFirstDerivedClass
dvClass dvClassGetFirstDerivedClass(
    dvClass _Class)
{
    return dvClasss.FirstDerivedClass[dvClass2Index(_Class)];
}

#undef dvClassSetFirstDerivedClass
void dvClassSetFirstDerivedClass(
    dvClass _Class,
    dvClass value)
{
    dvClasss.FirstDerivedClass[dvClass2Index(_Class)] = value;
}

#undef dvClassGetNextClassDerivedClass
dvClass dvClassGetNextClassDerivedClass(
    dvClass _Class)
{
    return dvClasss.NextClassDerivedClass[dvClass2Index(_Class)];
}

#undef dvClassSetNextClassDerivedClass
void dvClassSetNextClassDerivedClass(
    dvClass _Class,
    dvClass value)
{
    dvClasss.NextClassDerivedClass[dvClass2Index(_Class)] = value;
}

#undef dvClassGetLastDerivedClass
dvClass dvClassGetLastDerivedClass(
    dvClass _Class)
{
    return dvClasss.LastDerivedClass[dvClass2Index(_Class)];
}

#undef dvClassSetLastDerivedClass
void dvClassSetLastDerivedClass(
    dvClass _Class,
    dvClass value)
{
    dvClasss.LastDerivedClass[dvClass2Index(_Class)] = value;
}

#undef dvClassGetFirstChildRelationship
dvRelationship dvClassGetFirstChildRelationship(
    dvClass _Class)
{
    return dvClasss.FirstChildRelationship[dvClass2Index(_Class)];
}

#undef dvClassSetFirstChildRelationship
void dvClassSetFirstChildRelationship(
    dvClass _Class,
    dvRelationship value)
{
    dvClasss.FirstChildRelationship[dvClass2Index(_Class)] = value;
}

#undef dvClassGetLastChildRelationship
dvRelationship dvClassGetLastChildRelationship(
    dvClass _Class)
{
    return dvClasss.LastChildRelationship[dvClass2Index(_Class)];
}

#undef dvClassSetLastChildRelationship
void dvClassSetLastChildRelationship(
    dvClass _Class,
    dvRelationship value)
{
    dvClasss.LastChildRelationship[dvClass2Index(_Class)] = value;
}

#undef dvClassGetFirstParentRelationship
dvRelationship dvClassGetFirstParentRelationship(
    dvClass _Class)
{
    return dvClasss.FirstParentRelationship[dvClass2Index(_Class)];
}

#undef dvClassSetFirstParentRelationship
void dvClassSetFirstParentRelationship(
    dvClass _Class,
    dvRelationship value)
{
    dvClasss.FirstParentRelationship[dvClass2Index(_Class)] = value;
}

#undef dvClassGetLastParentRelationship
dvRelationship dvClassGetLastParentRelationship(
    dvClass _Class)
{
    return dvClasss.LastParentRelationship[dvClass2Index(_Class)];
}

#undef dvClassSetLastParentRelationship
void dvClassSetLastParentRelationship(
    dvClass _Class,
    dvRelationship value)
{
    dvClasss.LastParentRelationship[dvClass2Index(_Class)] = value;
}

#undef dvClassGetFirstUnion
dvUnion dvClassGetFirstUnion(
    dvClass _Class)
{
    return dvClasss.FirstUnion[dvClass2Index(_Class)];
}

#undef dvClassSetFirstUnion
void dvClassSetFirstUnion(
    dvClass _Class,
    dvUnion value)
{
    dvClasss.FirstUnion[dvClass2Index(_Class)] = value;
}

#undef dvClassGetLastUnion
dvUnion dvClassGetLastUnion(
    dvClass _Class)
{
    return dvClasss.LastUnion[dvClass2Index(_Class)];
}

#undef dvClassSetLastUnion
void dvClassSetLastUnion(
    dvClass _Class,
    dvUnion value)
{
    dvClasss.LastUnion[dvClass2Index(_Class)] = value;
}

#undef dvPropertyGetName
char *dvPropertyGetName(
    dvProperty Property)
{
    return utSymGetName(dvPropertyGetSym(Property));
}

#undef dvSparsegroupGetName
char *dvSparsegroupGetName(
    dvSparsegroup Sparsegroup)
{
    return utSymGetName(dvSparsegroupGetSym(Sparsegroup));
}

#undef dvPropertyGetSym
utSym dvPropertyGetSym(
    dvProperty _Property)
{
    return dvPropertys.Sym[dvProperty2Index(_Property)];
}

#undef dvPropertySetSym
void dvPropertySetSym(
    dvProperty _Property,
    utSym value)
{
    dvPropertys.Sym[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetType
dvPropertyType dvPropertyGetType(
    dvProperty _Property)
{
    return dvPropertys.Type[dvProperty2Index(_Property)];
}

#undef dvPropertySetType
void dvPropertySetType(
    dvProperty _Property,
    dvPropertyType value)
{
    dvPropertys.Type[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyArray
bool dvPropertyArray(
    dvProperty _Property)
{
    return (dvPropertys.Array[dvProperty2Index(_Property) >> 3] >> (dvProperty2Index(_Property) & 7)) & 1;
}

#undef dvPropertySetArray
void dvPropertySetArray(
    dvProperty _Property,
    bool value)
{
    uint32 xProperty = dvProperty2Index(_Property);

    dvPropertys.Array[xProperty >> 3] = (dvPropertys.Array[xProperty >> 3] & ~(1 << (xProperty & 7))) |
        ((value != 0) << (xProperty & 7));
}

#undef dvPropertyCascade
bool dvPropertyCascade(
    dvProperty _Property)
{
    return (dvPropertys.Cascade[dvProperty2Index(_Property) >> 3] >> (dvProperty2Index(_Property) & 7)) & 1;
}

#undef dvPropertySetCascade
void dvPropertySetCascade(
    dvProperty _Property,
    bool value)
{
    uint32 xProperty = dvProperty2Index(_Property);

    dvPropertys.Cascade[xProperty >> 3] = (dvPropertys.Cascade[xProperty >> 3] & ~(1 << (xProperty & 7))) |
        ((value != 0) << (xProperty & 7));
}

#undef dvPropertySparse
bool dvPropertySparse(
    dvProperty _Property)
{
    return (dvPropertys.Sparse[dvProperty2Index(_Property) >> 3] >> (dvProperty2Index(_Property) & 7)) & 1;
}

#undef dvPropertySetSparse
void dvPropertySetSparse(
    dvProperty _Property,
    bool value)
{
    uint32 xProperty = dvProperty2Index(_Property);

    dvPropertys.Sparse[xProperty >> 3] = (dvPropertys.Sparse[xProperty >> 3] & ~(1 << (xProperty & 7))) |
        ((value != 0) << (xProperty & 7));
}

#undef dvPropertyExpanded
bool dvPropertyExpanded(
    dvProperty _Property)
{
    return (dvPropertys.Expanded[dvProperty2Index(_Property) >> 3] >> (dvProperty2Index(_Property) & 7)) & 1;
}

#undef dvPropertySetExpanded
void dvPropertySetExpanded(
    dvProperty _Property,
    bool value)
{
    uint32 xProperty = dvProperty2Index(_Property);

    dvPropertys.Expanded[xProperty >> 3] = (dvPropertys.Expanded[xProperty >> 3] & ~(1 << (xProperty & 7))) |
        ((value != 0) << (xProperty & 7));
}

#undef dvPropertyGetFieldNumber
uint32 dvPropertyGetFieldNumber(
    dvProperty _Property)
{
    return dvPropertys.FieldNumber[dvProperty2Index(_Property)];
}

#undef dvPropertySetFieldNumber
void dvPropertySetFieldNumber(
    dvProperty _Property,
    uint32 value)
{
    dvPropertys.FieldNumber[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetFirstElementProp
dvProperty dvPropertyGetFirstElementProp(
    dvProperty _Property)
{
    return dvPropertys.FirstElementProp[dvProperty2Index(_Property)];
}

#undef dvPropertySetFirstElementProp
void dvPropertySetFirstElementProp(
    dvProperty _Property,
    dvProperty value)
{
    dvPropertys.FirstElementProp[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetNumElementsProp
dvProperty dvPropertyGetNumElementsProp(
    dvProperty _Property)
{
    return dvPropertys.NumElementsProp[dvProperty2Index(_Property)];
}

#undef dvPropertySetNumElementsProp
void dvPropertySetNumElementsProp(
    dvProperty _Property,
    dvProperty value)
{
    dvPropertys.NumElementsProp[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyHidden
bool dvPropertyHidden(
    dvProperty _Property)
{
    return (dvPropertys.Hidden[dvProperty2Index(_Property) >> 3] >> (dvProperty2Index(_Property) & 7)) & 1;
}

#undef dvPropertySetHidden
void dvPropertySetHidden(
    dvProperty _Property,
    bool value)
{
    uint32 xProperty = dvProperty2Index(_Property);

    dvPropertys.Hidden[xProperty >> 3] = (dvPropertys.Hidden[xProperty >> 3] & ~(1 << (xProperty & 7))) |
        ((value != 0) << (xProperty & 7));
}

#undef dvPropertyGetInitializerIndex
uint32 dvPropertyGetInitializerIndex(
    dvProperty _Property)
{
    return dvPropertys.InitializerIndex[dvProperty2Index(_Property)];
}

#undef dvPropertySetInitializerIndex
void dvPropertySetInitializerIndex(
    dvProperty _Property,
    uint32 value)
{
    dvPropertys.InitializerIndex[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetNumInitializer
uint32 dvPropertyGetNumInitializer(
    dvProperty _Property)
{
    return dvPropertys.NumInitializer[dvProperty2Index(_Property)];
}

#undef dvPropertySetNumInitializer
void dvPropertySetNumInitializer(
    dvProperty _Property,
    uint32 value)
{
    dvPropertys.NumInitializer[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetiInitializer
char dvPropertyGetiInitializer(
    dvProperty _Property,
    uint32 x)
{
    return (dvPropertys.Initializer)[dvPropertyGetInitializerIndex(_Property) + x];
}

#undef dvPropertySetiInitializer
void dvPropertySetiInitializer(
    dvProperty Property,
    uint32 x,
    char value)
{
    dvPropertys.Initializer[dvPropertyGetInitializerIndex(Property) + x] = value;
}

#undef dvPropertyGetInitializer
char *dvPropertyGetInitializer(
    dvProperty Property)
{
    return dvPropertys.Initializer + dvPropertyGetInitializerIndex(Property);
}

#undef dvPropertyFixedSize
bool dvPropertyFixedSize(
    dvProperty _Property)
{
    return (dvPropertys.FixedSize[dvProperty2Index(_Property) >> 3] >> (dvProperty2Index(_Property) & 7)) & 1;
}

#undef dvPropertySetFixedSize
void dvPropertySetFixedSize(
    dvProperty _Property,
    bool value)
{
    uint32 xProperty = dvProperty2Index(_Property);

    dvPropertys.FixedSize[xProperty >> 3] = (dvPropertys.FixedSize[xProperty >> 3] & ~(1 << (xProperty & 7))) |
        ((value != 0) << (xProperty & 7));
}

#undef dvPropertyGetIndexIndex
uint32 dvPropertyGetIndexIndex(
    dvProperty _Property)
{
    return dvPropertys.IndexIndex[dvProperty2Index(_Property)];
}

#undef dvPropertySetIndexIndex
void dvPropertySetIndexIndex(
    dvProperty _Property,
    uint32 value)
{
    dvPropertys.IndexIndex[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetNumIndex
uint32 dvPropertyGetNumIndex(
    dvProperty _Property)
{
    return dvPropertys.NumIndex[dvProperty2Index(_Property)];
}

#undef dvPropertySetNumIndex
void dvPropertySetNumIndex(
    dvProperty _Property,
    uint32 value)
{
    dvPropertys.NumIndex[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetiIndex
char dvPropertyGetiIndex(
    dvProperty _Property,
    uint32 x)
{
    return (dvPropertys.Index)[dvPropertyGetIndexIndex(_Property) + x];
}

#undef dvPropertySetiIndex
void dvPropertySetiIndex(
    dvProperty Property,
    uint32 x,
    char value)
{
    dvPropertys.Index[dvPropertyGetIndexIndex(Property) + x] = value;
}

#undef dvPropertyGetIndex
char *dvPropertyGetIndex(
    dvProperty Property)
{
    return dvPropertys.Index + dvPropertyGetIndexIndex(Property);
}

#undef dvPropertyGetEnumProp
dvEnum dvPropertyGetEnumProp(
    dvProperty _Property)
{
    return dvPropertys.union1[dvProperty2Index(_Property)].EnumProp;
}

#undef dvPropertySetEnumProp
void dvPropertySetEnumProp(
    dvProperty _Property,
    dvEnum value)
{
    dvPropertys.union1[dvProperty2Index(_Property)].EnumProp = value;
}

#undef dvPropertyGetTypedefProp
dvTypedef dvPropertyGetTypedefProp(
    dvProperty _Property)
{
    return dvPropertys.union1[dvProperty2Index(_Property)].TypedefProp;
}

#undef dvPropertySetTypedefProp
void dvPropertySetTypedefProp(
    dvProperty _Property,
    dvTypedef value)
{
    dvPropertys.union1[dvProperty2Index(_Property)].TypedefProp = value;
}

#undef dvPropertyGetClassProp
dvClass dvPropertyGetClassProp(
    dvProperty _Property)
{
    return dvPropertys.union1[dvProperty2Index(_Property)].ClassProp;
}

#undef dvPropertySetClassProp
void dvPropertySetClassProp(
    dvProperty _Property,
    dvClass value)
{
    dvPropertys.union1[dvProperty2Index(_Property)].ClassProp = value;
}

#undef dvPropertyGetTypeSym
utSym dvPropertyGetTypeSym(
    dvProperty _Property)
{
    return dvPropertys.union1[dvProperty2Index(_Property)].TypeSym;
}

#undef dvPropertySetTypeSym
void dvPropertySetTypeSym(
    dvProperty _Property,
    utSym value)
{
    dvPropertys.union1[dvProperty2Index(_Property)].TypeSym = value;
}

#undef dvPropertyGetWidth
uint8 dvPropertyGetWidth(
    dvProperty _Property)
{
    return dvPropertys.union1[dvProperty2Index(_Property)].Width;
}

#undef dvPropertySetWidth
void dvPropertySetWidth(
    dvProperty _Property,
    uint8 value)
{
    dvPropertys.union1[dvProperty2Index(_Property)].Width = value;
}

#undef dvPropertyGetLine
uint32 dvPropertyGetLine(
    dvProperty _Property)
{
    return dvPropertys.Line[dvProperty2Index(_Property)];
}

#undef dvPropertySetLine
void dvPropertySetLine(
    dvProperty _Property,
    uint32 value)
{
    dvPropertys.Line[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetClass
dvClass dvPropertyGetClass(
    dvProperty _Property)
{
    return dvPropertys.Class[dvProperty2Index(_Property)];
}

#undef dvPropertySetClass
void dvPropertySetClass(
    dvProperty _Property,
    dvClass value)
{
    dvPropertys.Class[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetNextClassProperty
dvProperty dvPropertyGetNextClassProperty(
    dvProperty _Property)
{
    return dvPropertys.NextClassProperty[dvProperty2Index(_Property)];
}

#undef dvPropertySetNextClassProperty
void dvPropertySetNextClassProperty(
    dvProperty _Property,
    dvProperty value)
{
    dvPropertys.NextClassProperty[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetPrevClassProperty
dvProperty dvPropertyGetPrevClassProperty(
    dvProperty _Property)
{
    return dvPropertys.PrevClassProperty[dvProperty2Index(_Property)];
}

#undef dvPropertySetPrevClassProperty
void dvPropertySetPrevClassProperty(
    dvProperty _Property,
    dvProperty value)
{
    dvPropertys.PrevClassProperty[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetNextTableClassProperty
dvProperty dvPropertyGetNextTableClassProperty(
    dvProperty _Property)
{
    return dvPropertys.NextTableClassProperty[dvProperty2Index(_Property)];
}

#undef dvPropertySetNextTableClassProperty
void dvPropertySetNextTableClassProperty(
    dvProperty _Property,
    dvProperty value)
{
    dvPropertys.NextTableClassProperty[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetFirstCase
dvCase dvPropertyGetFirstCase(
    dvProperty _Property)
{
    return dvPropertys.FirstCase[dvProperty2Index(_Property)];
}

#undef dvPropertySetFirstCase
void dvPropertySetFirstCase(
    dvProperty _Property,
    dvCase value)
{
    dvPropertys.FirstCase[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetLastCase
dvCase dvPropertyGetLastCase(
    dvProperty _Property)
{
    return dvPropertys.LastCase[dvProperty2Index(_Property)];
}

#undef dvPropertySetLastCase
void dvPropertySetLastCase(
    dvProperty _Property,
    dvCase value)
{
    dvPropertys.LastCase[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetFirstKey
dvKey dvPropertyGetFirstKey(
    dvProperty _Property)
{
    return dvPropertys.FirstKey[dvProperty2Index(_Property)];
}

#undef dvPropertySetFirstKey
void dvPropertySetFirstKey(
    dvProperty _Property,
    dvKey value)
{
    dvPropertys.FirstKey[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetLastKey
dvKey dvPropertyGetLastKey(
    dvProperty _Property)
{
    return dvPropertys.LastKey[dvProperty2Index(_Property)];
}

#undef dvPropertySetLastKey
void dvPropertySetLastKey(
    dvProperty _Property,
    dvKey value)
{
    dvPropertys.LastKey[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetSparsegroup
dvSparsegroup dvPropertyGetSparsegroup(
    dvProperty _Property)
{
    return dvPropertys.Sparsegroup[dvProperty2Index(_Property)];
}

#undef dvPropertySetSparsegroup
void dvPropertySetSparsegroup(
    dvProperty _Property,
    dvSparsegroup value)
{
    dvPropertys.Sparsegroup[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetNextSparsegroupProperty
dvProperty dvPropertyGetNextSparsegroupProperty(
    dvProperty _Property)
{
    return dvPropertys.NextSparsegroupProperty[dvProperty2Index(_Property)];
}

#undef dvPropertySetNextSparsegroupProperty
void dvPropertySetNextSparsegroupProperty(
    dvProperty _Property,
    dvProperty value)
{
    dvPropertys.NextSparsegroupProperty[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetRelationship
dvRelationship dvPropertyGetRelationship(
    dvProperty _Property)
{
    return dvPropertys.Relationship[dvProperty2Index(_Property)];
}

#undef dvPropertySetRelationship
void dvPropertySetRelationship(
    dvProperty _Property,
    dvRelationship value)
{
    dvPropertys.Relationship[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetNextRelationshipProperty
dvProperty dvPropertyGetNextRelationshipProperty(
    dvProperty _Property)
{
    return dvPropertys.NextRelationshipProperty[dvProperty2Index(_Property)];
}

#undef dvPropertySetNextRelationshipProperty
void dvPropertySetNextRelationshipProperty(
    dvProperty _Property,
    dvProperty value)
{
    dvPropertys.NextRelationshipProperty[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetUnion
dvUnion dvPropertyGetUnion(
    dvProperty _Property)
{
    return dvPropertys.Union[dvProperty2Index(_Property)];
}

#undef dvPropertySetUnion
void dvPropertySetUnion(
    dvProperty _Property,
    dvUnion value)
{
    dvPropertys.Union[dvProperty2Index(_Property)] = value;
}

#undef dvPropertyGetNextUnionProperty
dvProperty dvPropertyGetNextUnionProperty(
    dvProperty _Property)
{
    return dvPropertys.NextUnionProperty[dvProperty2Index(_Property)];
}

#undef dvPropertySetNextUnionProperty
void dvPropertySetNextUnionProperty(
    dvProperty _Property,
    dvProperty value)
{
    dvPropertys.NextUnionProperty[dvProperty2Index(_Property)] = value;
}

#undef dvSparsegroupGetSym
utSym dvSparsegroupGetSym(
    dvSparsegroup _Sparsegroup)
{
    return dvSparsegroups.Sym[dvSparsegroup2Index(_Sparsegroup)];
}

#undef dvSparsegroupSetSym
void dvSparsegroupSetSym(
    dvSparsegroup _Sparsegroup,
    utSym value)
{
    dvSparsegroups.Sym[dvSparsegroup2Index(_Sparsegroup)] = value;
}

#undef dvSparsegroupGetClass
dvClass dvSparsegroupGetClass(
    dvSparsegroup _Sparsegroup)
{
    return dvSparsegroups.Class[dvSparsegroup2Index(_Sparsegroup)];
}

#undef dvSparsegroupSetClass
void dvSparsegroupSetClass(
    dvSparsegroup _Sparsegroup,
    dvClass value)
{
    dvSparsegroups.Class[dvSparsegroup2Index(_Sparsegroup)] = value;
}

#undef dvSparsegroupGetNextClassSparsegroup
dvSparsegroup dvSparsegroupGetNextClassSparsegroup(
    dvSparsegroup _Sparsegroup)
{
    return dvSparsegroups.NextClassSparsegroup[dvSparsegroup2Index(_Sparsegroup)];
}

#undef dvSparsegroupSetNextClassSparsegroup
void dvSparsegroupSetNextClassSparsegroup(
    dvSparsegroup _Sparsegroup,
    dvSparsegroup value)
{
    dvSparsegroups.NextClassSparsegroup[dvSparsegroup2Index(_Sparsegroup)] = value;
}

#undef dvSparsegroupGetPrevClassSparsegroup
dvSparsegroup dvSparsegroupGetPrevClassSparsegroup(
    dvSparsegroup _Sparsegroup)
{
    return dvSparsegroups.PrevClassSparsegroup[dvSparsegroup2Index(_Sparsegroup)];
}

#undef dvSparsegroupSetPrevClassSparsegroup
void dvSparsegroupSetPrevClassSparsegroup(
    dvSparsegroup _Sparsegroup,
    dvSparsegroup value)
{
    dvSparsegroups.PrevClassSparsegroup[dvSparsegroup2Index(_Sparsegroup)] = value;
}

#undef dvSparsegroupGetNextTableClassSparsegroup
dvSparsegroup dvSparsegroupGetNextTableClassSparsegroup(
    dvSparsegroup _Sparsegroup)
{
    return dvSparsegroups.NextTableClassSparsegroup[dvSparsegroup2Index(_Sparsegroup)];
}

#undef dvSparsegroupSetNextTableClassSparsegroup
void dvSparsegroupSetNextTableClassSparsegroup(
    dvSparsegroup _Sparsegroup,
    dvSparsegroup value)
{
    dvSparsegroups.NextTableClassSparsegroup[dvSparsegroup2Index(_Sparsegroup)] = value;
}

#undef dvSparsegroupGetFirstProperty
dvProperty dvSparsegroupGetFirstProperty(
    dvSparsegroup _Sparsegroup)
{
    return dvSparsegroups.FirstProperty[dvSparsegroup2Index(_Sparsegroup)];
}

#undef dvSparsegroupSetFirstProperty
void dvSparsegroupSetFirstProperty(
    dvSparsegroup _Sparsegroup,
    dvProperty value)
{
    dvSparsegroups.FirstProperty[dvSparsegroup2Index(_Sparsegroup)] = value;
}

#undef dvSparsegroupGetLastProperty
dvProperty dvSparsegroupGetLastProperty(
    dvSparsegroup _Sparsegroup)
{
    return dvSparsegroups.LastProperty[dvSparsegroup2Index(_Sparsegroup)];
}

#undef dvSparsegroupSetLastProperty
void dvSparsegroupSetLastProperty(
    dvSparsegroup _Sparsegroup,
    dvProperty value)
{
    dvSparsegroups.LastProperty[dvSparsegroup2Index(_Sparsegroup)] = value;
}

#undef dvSparsegroupGetRelationship
dvRelationship dvSparsegroupGetRelationship(
    dvSparsegroup _Sparsegroup)
{
    return dvSparsegroups.Relationship[dvSparsegroup2Index(_Sparsegroup)];
}

#undef dvSparsegroupSetRelationship
void dvSparsegroupSetRelationship(
    dvSparsegroup _Sparsegroup,
    dvRelationship value)
{
    dvSparsegroups.Relationship[dvSparsegroup2Index(_Sparsegroup)] = value;
}

#undef dvRelationshipGetType
dvRelationshipType dvRelationshipGetType(
    dvRelationship _Relationship)
{
    return dvRelationships.Type[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetType
void dvRelationshipSetType(
    dvRelationship _Relationship,
    dvRelationshipType value)
{
    dvRelationships.Type[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetParentLabelSym
utSym dvRelationshipGetParentLabelSym(
    dvRelationship _Relationship)
{
    return dvRelationships.ParentLabelSym[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetParentLabelSym
void dvRelationshipSetParentLabelSym(
    dvRelationship _Relationship,
    utSym value)
{
    dvRelationships.ParentLabelSym[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetChildLabelSym
utSym dvRelationshipGetChildLabelSym(
    dvRelationship _Relationship)
{
    return dvRelationships.ChildLabelSym[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetChildLabelSym
void dvRelationshipSetChildLabelSym(
    dvRelationship _Relationship,
    utSym value)
{
    dvRelationships.ChildLabelSym[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipMandatory
bool dvRelationshipMandatory(
    dvRelationship _Relationship)
{
    return (dvRelationships.Mandatory[dvRelationship2Index(_Relationship) >> 3] >> (dvRelationship2Index(_Relationship) & 7)) & 1;
}

#undef dvRelationshipSetMandatory
void dvRelationshipSetMandatory(
    dvRelationship _Relationship,
    bool value)
{
    uint32 xRelationship = dvRelationship2Index(_Relationship);

    dvRelationships.Mandatory[xRelationship >> 3] = (dvRelationships.Mandatory[xRelationship >> 3] & ~(1 << (xRelationship & 7))) |
        ((value != 0) << (xRelationship & 7));
}

#undef dvRelationshipCascade
bool dvRelationshipCascade(
    dvRelationship _Relationship)
{
    return (dvRelationships.Cascade[dvRelationship2Index(_Relationship) >> 3] >> (dvRelationship2Index(_Relationship) & 7)) & 1;
}

#undef dvRelationshipSetCascade
void dvRelationshipSetCascade(
    dvRelationship _Relationship,
    bool value)
{
    uint32 xRelationship = dvRelationship2Index(_Relationship);

    dvRelationships.Cascade[xRelationship >> 3] = (dvRelationships.Cascade[xRelationship >> 3] & ~(1 << (xRelationship & 7))) |
        ((value != 0) << (xRelationship & 7));
}

#undef dvRelationshipAccessChild
bool dvRelationshipAccessChild(
    dvRelationship _Relationship)
{
    return (dvRelationships.AccessChild[dvRelationship2Index(_Relationship) >> 3] >> (dvRelationship2Index(_Relationship) & 7)) & 1;
}

#undef dvRelationshipSetAccessChild
void dvRelationshipSetAccessChild(
    dvRelationship _Relationship,
    bool value)
{
    uint32 xRelationship = dvRelationship2Index(_Relationship);

    dvRelationships.AccessChild[xRelationship >> 3] = (dvRelationships.AccessChild[xRelationship >> 3] & ~(1 << (xRelationship & 7))) |
        ((value != 0) << (xRelationship & 7));
}

#undef dvRelationshipAccessParent
bool dvRelationshipAccessParent(
    dvRelationship _Relationship)
{
    return (dvRelationships.AccessParent[dvRelationship2Index(_Relationship) >> 3] >> (dvRelationship2Index(_Relationship) & 7)) & 1;
}

#undef dvRelationshipSetAccessParent
void dvRelationshipSetAccessParent(
    dvRelationship _Relationship,
    bool value)
{
    uint32 xRelationship = dvRelationship2Index(_Relationship);

    dvRelationships.AccessParent[xRelationship >> 3] = (dvRelationships.AccessParent[xRelationship >> 3] & ~(1 << (xRelationship & 7))) |
        ((value != 0) << (xRelationship & 7));
}

#undef dvRelationshipSharedParent
bool dvRelationshipSharedParent(
    dvRelationship _Relationship)
{
    return (dvRelationships.SharedParent[dvRelationship2Index(_Relationship) >> 3] >> (dvRelationship2Index(_Relationship) & 7)) & 1;
}

#undef dvRelationshipSetSharedParent
void dvRelationshipSetSharedParent(
    dvRelationship _Relationship,
    bool value)
{
    uint32 xRelationship = dvRelationship2Index(_Relationship);

    dvRelationships.SharedParent[xRelationship >> 3] = (dvRelationships.SharedParent[xRelationship >> 3] & ~(1 << (xRelationship & 7))) |
        ((value != 0) << (xRelationship & 7));
}

#undef dvRelationshipSparse
bool dvRelationshipSparse(
    dvRelationship _Relationship)
{
    return (dvRelationships.Sparse[dvRelationship2Index(_Relationship) >> 3] >> (dvRelationship2Index(_Relationship) & 7)) & 1;
}

#undef dvRelationshipSetSparse
void dvRelationshipSetSparse(
    dvRelationship _Relationship,
    bool value)
{
    uint32 xRelationship = dvRelationship2Index(_Relationship);

    dvRelationships.Sparse[xRelationship >> 3] = (dvRelationships.Sparse[xRelationship >> 3] & ~(1 << (xRelationship & 7))) |
        ((value != 0) << (xRelationship & 7));
}

#undef dvRelationshipExpanded
bool dvRelationshipExpanded(
    dvRelationship _Relationship)
{
    return (dvRelationships.Expanded[dvRelationship2Index(_Relationship) >> 3] >> (dvRelationship2Index(_Relationship) & 7)) & 1;
}

#undef dvRelationshipSetExpanded
void dvRelationshipSetExpanded(
    dvRelationship _Relationship,
    bool value)
{
    uint32 xRelationship = dvRelationship2Index(_Relationship);

    dvRelationships.Expanded[xRelationship >> 3] = (dvRelationships.Expanded[xRelationship >> 3] & ~(1 << (xRelationship & 7))) |
        ((value != 0) << (xRelationship & 7));
}

#undef dvRelationshipGetSchema
dvSchema dvRelationshipGetSchema(
    dvRelationship _Relationship)
{
    return dvRelationships.Schema[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetSchema
void dvRelationshipSetSchema(
    dvRelationship _Relationship,
    dvSchema value)
{
    dvRelationships.Schema[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetNextSchemaRelationship
dvRelationship dvRelationshipGetNextSchemaRelationship(
    dvRelationship _Relationship)
{
    return dvRelationships.NextSchemaRelationship[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetNextSchemaRelationship
void dvRelationshipSetNextSchemaRelationship(
    dvRelationship _Relationship,
    dvRelationship value)
{
    dvRelationships.NextSchemaRelationship[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetParentClass
dvClass dvRelationshipGetParentClass(
    dvRelationship _Relationship)
{
    return dvRelationships.ParentClass[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetParentClass
void dvRelationshipSetParentClass(
    dvRelationship _Relationship,
    dvClass value)
{
    dvRelationships.ParentClass[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetNextClassChildRelationship
dvRelationship dvRelationshipGetNextClassChildRelationship(
    dvRelationship _Relationship)
{
    return dvRelationships.NextClassChildRelationship[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetNextClassChildRelationship
void dvRelationshipSetNextClassChildRelationship(
    dvRelationship _Relationship,
    dvRelationship value)
{
    dvRelationships.NextClassChildRelationship[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetChildClass
dvClass dvRelationshipGetChildClass(
    dvRelationship _Relationship)
{
    return dvRelationships.ChildClass[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetChildClass
void dvRelationshipSetChildClass(
    dvRelationship _Relationship,
    dvClass value)
{
    dvRelationships.ChildClass[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetNextClassParentRelationship
dvRelationship dvRelationshipGetNextClassParentRelationship(
    dvRelationship _Relationship)
{
    return dvRelationships.NextClassParentRelationship[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetNextClassParentRelationship
void dvRelationshipSetNextClassParentRelationship(
    dvRelationship _Relationship,
    dvRelationship value)
{
    dvRelationships.NextClassParentRelationship[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetFirstProperty
dvProperty dvRelationshipGetFirstProperty(
    dvRelationship _Relationship)
{
    return dvRelationships.FirstProperty[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetFirstProperty
void dvRelationshipSetFirstProperty(
    dvRelationship _Relationship,
    dvProperty value)
{
    dvRelationships.FirstProperty[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetLastProperty
dvProperty dvRelationshipGetLastProperty(
    dvRelationship _Relationship)
{
    return dvRelationships.LastProperty[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetLastProperty
void dvRelationshipSetLastProperty(
    dvRelationship _Relationship,
    dvProperty value)
{
    dvRelationships.LastProperty[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetFirstKey
dvKey dvRelationshipGetFirstKey(
    dvRelationship _Relationship)
{
    return dvRelationships.FirstKey[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetFirstKey
void dvRelationshipSetFirstKey(
    dvRelationship _Relationship,
    dvKey value)
{
    dvRelationships.FirstKey[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetLastKey
dvKey dvRelationshipGetLastKey(
    dvRelationship _Relationship)
{
    return dvRelationships.LastKey[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetLastKey
void dvRelationshipSetLastKey(
    dvRelationship _Relationship,
    dvKey value)
{
    dvRelationships.LastKey[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetParentSparsegroup
dvSparsegroup dvRelationshipGetParentSparsegroup(
    dvRelationship _Relationship)
{
    return dvRelationships.ParentSparsegroup[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetParentSparsegroup
void dvRelationshipSetParentSparsegroup(
    dvRelationship _Relationship,
    dvSparsegroup value)
{
    dvRelationships.ParentSparsegroup[dvRelationship2Index(_Relationship)] = value;
}

#undef dvRelationshipGetChildSparsegroup
dvSparsegroup dvRelationshipGetChildSparsegroup(
    dvRelationship _Relationship)
{
    return dvRelationships.ChildSparsegroup[dvRelationship2Index(_Relationship)];
}

#undef dvRelationshipSetChildSparsegroup
void dvRelationshipSetChildSparsegroup(
    dvRelationship _Relationship,
    dvSparsegroup value)
{
    dvRelationships.ChildSparsegroup[dvRelationship2Index(_Relationship)] = value;
}

#undef dvKeyGetPropertySym
utSym dvKeyGetPropertySym(
    dvKey _Key)
{
    return dvKeys.PropertySym[dvKey2Index(_Key)];
}

#undef dvKeySetPropertySym
void dvKeySetPropertySym(
    dvKey _Key,
    utSym value)
{
    dvKeys.PropertySym[dvKey2Index(_Key)] = value;
}

#undef dvKeyGetLineNum
uint32 dvKeyGetLineNum(
    dvKey _Key)
{
    return dvKeys.LineNum[dvKey2Index(_Key)];
}

#undef dvKeySetLineNum
void dvKeySetLineNum(
    dvKey _Key,
    uint32 value)
{
    dvKeys.LineNum[dvKey2Index(_Key)] = value;
}

#undef dvKeyGetProperty
dvProperty dvKeyGetProperty(
    dvKey _Key)
{
    return dvKeys.Property[dvKey2Index(_Key)];
}

#undef dvKeySetProperty
void dvKeySetProperty(
    dvKey _Key,
    dvProperty value)
{
    dvKeys.Property[dvKey2Index(_Key)] = value;
}

#undef dvKeyGetNextPropertyKey
dvKey dvKeyGetNextPropertyKey(
    dvKey _Key)
{
    return dvKeys.NextPropertyKey[dvKey2Index(_Key)];
}

#undef dvKeySetNextPropertyKey
void dvKeySetNextPropertyKey(
    dvKey _Key,
    dvKey value)
{
    dvKeys.NextPropertyKey[dvKey2Index(_Key)] = value;
}

#undef dvKeyGetRelationship
dvRelationship dvKeyGetRelationship(
    dvKey _Key)
{
    return dvKeys.Relationship[dvKey2Index(_Key)];
}

#undef dvKeySetRelationship
void dvKeySetRelationship(
    dvKey _Key,
    dvRelationship value)
{
    dvKeys.Relationship[dvKey2Index(_Key)] = value;
}

#undef dvKeyGetNextRelationshipKey
dvKey dvKeyGetNextRelationshipKey(
    dvKey _Key)
{
    return dvKeys.NextRelationshipKey[dvKey2Index(_Key)];
}

#undef dvKeySetNextRelationshipKey
void dvKeySetNextRelationshipKey(
    dvKey _Key,
    dvKey value)
{
    dvKeys.NextRelationshipKey[dvKey2Index(_Key)] = value;
}

#undef dvUnionGetPropertySym
utSym dvUnionGetPropertySym(
    dvUnion _Union)
{
    return dvUnions.PropertySym[dvUnion2Index(_Union)];
}

#undef dvUnionSetPropertySym
void dvUnionSetPropertySym(
    dvUnion _Union,
    utSym value)
{
    dvUnions.PropertySym[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetTypeProperty
dvProperty dvUnionGetTypeProperty(
    dvUnion _Union)
{
    return dvUnions.TypeProperty[dvUnion2Index(_Union)];
}

#undef dvUnionSetTypeProperty
void dvUnionSetTypeProperty(
    dvUnion _Union,
    dvProperty value)
{
    dvUnions.TypeProperty[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetLine
uint32 dvUnionGetLine(
    dvUnion _Union)
{
    return dvUnions.Line[dvUnion2Index(_Union)];
}

#undef dvUnionSetLine
void dvUnionSetLine(
    dvUnion _Union,
    uint32 value)
{
    dvUnions.Line[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetNumber
uint16 dvUnionGetNumber(
    dvUnion _Union)
{
    return dvUnions.Number[dvUnion2Index(_Union)];
}

#undef dvUnionSetNumber
void dvUnionSetNumber(
    dvUnion _Union,
    uint16 value)
{
    dvUnions.Number[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetFieldNumber
uint32 dvUnionGetFieldNumber(
    dvUnion _Union)
{
    return dvUnions.FieldNumber[dvUnion2Index(_Union)];
}

#undef dvUnionSetFieldNumber
void dvUnionSetFieldNumber(
    dvUnion _Union,
    uint32 value)
{
    dvUnions.FieldNumber[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetNumCases
uint16 dvUnionGetNumCases(
    dvUnion _Union)
{
    return dvUnions.NumCases[dvUnion2Index(_Union)];
}

#undef dvUnionSetNumCases
void dvUnionSetNumCases(
    dvUnion _Union,
    uint16 value)
{
    dvUnions.NumCases[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetClass
dvClass dvUnionGetClass(
    dvUnion _Union)
{
    return dvUnions.Class[dvUnion2Index(_Union)];
}

#undef dvUnionSetClass
void dvUnionSetClass(
    dvUnion _Union,
    dvClass value)
{
    dvUnions.Class[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetNextClassUnion
dvUnion dvUnionGetNextClassUnion(
    dvUnion _Union)
{
    return dvUnions.NextClassUnion[dvUnion2Index(_Union)];
}

#undef dvUnionSetNextClassUnion
void dvUnionSetNextClassUnion(
    dvUnion _Union,
    dvUnion value)
{
    dvUnions.NextClassUnion[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetFirstProperty
dvProperty dvUnionGetFirstProperty(
    dvUnion _Union)
{
    return dvUnions.FirstProperty[dvUnion2Index(_Union)];
}

#undef dvUnionSetFirstProperty
void dvUnionSetFirstProperty(
    dvUnion _Union,
    dvProperty value)
{
    dvUnions.FirstProperty[dvUnion2Index(_Union)] = value;
}

#undef dvUnionGetLastProperty
dvProperty dvUnionGetLastProperty(
    dvUnion _Union)
{
    return dvUnions.LastProperty[dvUnion2Index(_Union)];
}

#undef dvUnionSetLastProperty
void dvUnionSetLastProperty(
    dvUnion _Union,
    dvProperty value)
{
    dvUnions.LastProperty[dvUnion2Index(_Union)] = value;
}

#undef dvCaseGetEntrySym
utSym dvCaseGetEntrySym(
    dvCase _Case)
{
    return dvCases.EntrySym[dvCase2Index(_Case)];
}

#undef dvCaseSetEntrySym
void dvCaseSetEntrySym(
    dvCase _Case,
    utSym value)
{
    dvCases.EntrySym[dvCase2Index(_Case)] = value;
}

#undef dvCaseGetEntry
dvEntry dvCaseGetEntry(
    dvCase _Case)
{
    return dvCases.Entry[dvCase2Index(_Case)];
}

#undef dvCaseSetEntry
void dvCaseSetEntry(
    dvCase _Case,
    dvEntry value)
{
    dvCases.Entry[dvCase2Index(_Case)] = value;
}

#undef dvCaseGetNextEntryCase
dvCase dvCaseGetNextEntryCase(
    dvCase _Case)
{
    return dvCases.NextEntryCase[dvCase2Index(_Case)];
}

#undef dvCaseSetNextEntryCase
void dvCaseSetNextEntryCase(
    dvCase _Case,
    dvCase value)
{
    dvCases.NextEntryCase[dvCase2Index(_Case)] = value;
}

#undef dvCaseGetProperty
dvProperty dvCaseGetProperty(
    dvCase _Case)
{
    return dvCases.Property[dvCase2Index(_Case)];
}

#undef dvCaseSetProperty
void dvCaseSetProperty(
    dvCase _Case,
    dvProperty value)
{
    dvCases.Property[dvCase2Index(_Case)] = value;
}

#undef dvCaseGetNextPropertyCase
dvCase dvCaseGetNextPropertyCase(
    dvCase _Case)
{
    return dvCases.NextPropertyCase[dvCase2Index(_Case)];
}

#undef dvCaseSetNextPropertyCase
void dvCaseSetNextPropertyCase(
    dvCase _Case,
    dvCase value)
{
    dvCases.NextPropertyCase[dvCase2Index(_Case)] = value;
}

#endif
