package ffmpreg

import (
	"context"
	"errors"
	"sync"

	"codeberg.org/gruf/go-ffmpreg/embed"
	"codeberg.org/gruf/go-ffmpreg/internal"
	"codeberg.org/gruf/go-ffmpreg/wasm"
	"github.com/tetratelabs/wazero"
)

var (
	// concurrency protected
	// WASM module and runtime.
	wzrt wazero.Runtime
	cmod wazero.CompiledModule
	once sync.Once
)

// Initialize ensures that the ffmpeg WebAssembly
// module is decoded and pre-compiled in memory to
// a format that the WebAssembly runtime can instantiate.
//
// If the module is already pre-compiled, this is a noop.
// This is called within Run(), but you can opt to call
// this yourself as the pre-compilation stage can be slow.
//
// Note that this is safe to call concurrently.
func Initialize() (err error) {
	once.Do(func() {

		// Initialize WASM runtime.
		wzrt = internal.Runtime()

		// Compile the ffmpeg WASM.
		cmod, err = wzrt.CompileModule(
			context.Background(),
			embed.B,
		)

		// Drop compiled
		// from memory.
		embed.B = nil
	})

	// Check for a previous fail.
	if cmod == nil && err == nil {
		err = errors.New("failed compilation")
	}

	return
}

// Ffmpeg will run ffmpeg with the given arguments, returning exit code or error.
// The first call to Run() may be slow if Initialize() has not already been called.
func Ffmpeg(ctx context.Context, args wasm.Args) (uint32, error) {

	// Ensure the module is initialized.
	if err := Initialize(); err != nil {
		return 0, err
	}

	// Call ffprobe program.
	args.Name = "ffmpeg"

	// Run the compiled module with args.
	return wasm.Run(ctx, wzrt, cmod, args)
}

// Ffprobe will run ffprobe with the given arguments, returning exit code or error.
// The first call to Run() may be slow if Initialize() has not already been called.
func Ffprobe(ctx context.Context, args wasm.Args) (uint32, error) {

	// Ensure the module is initialized.
	if err := Initialize(); err != nil {
		return 0, err
	}

	// Call ffprobe program.
	args.Name = "ffprobe"

	// Run the compiled module with args.
	return wasm.Run(ctx, wzrt, cmod, args)
}

// Run will run ffmpreg with the given arguments, returning exit code or error.
// The first call to Run() may be slow if Initialize() has not already been called.
func Run(ctx context.Context, args wasm.Args) (uint32, error) {
	switch args.Name {
	case "ffmpeg", "ffprobe":
	default:
		return 0, errors.New("unexpected program name, want [ffmpeg|ffprobe]")
	}

	// Ensure the module is initialized.
	if err := Initialize(); err != nil {
		return 0, err
	}

	// Run the compiled module with args.
	return wasm.Run(ctx, wzrt, cmod, args)
}
