use std::ops::{Deref, DerefMut};

use derive_new::new;
use miette::{Context, IntoDiagnostic};

use crate::paths::token_path;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash, new)]
pub struct Token(pub String);

impl Deref for Token {
    type Target = String;
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl DerefMut for Token {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.0
    }
}

impl Token {
    pub fn read_from_data_dir(instance: impl AsRef<str>) -> miette::Result<Self> {
        let token_path = token_path(instance)?;
        let token = std::fs::read_to_string(token_path)
            .into_diagnostic()
            .context("Reading token from '{token_path}' failed!")?;
        Ok(Self(token))
    }

    pub fn read_from_env() -> miette::Result<Self> {
        std::env::var("BERG_TOKEN").into_diagnostic().map(Token)
    }
}
