use forgejo_api::structs::{Release, RepoListReleasesQuery};
use miette::IntoDiagnostic;

use crate::actions::GlobalArgs;
use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;

use clap::Parser;

/// List all releases in the current repository
#[derive(Parser, Debug)]
pub struct ListReleaseArgs;

impl ListReleaseArgs {
    pub async fn run(self, global_args: GlobalArgs) -> miette::Result<()> {
        let ctx = BergContext::new(self, global_args).await?;
        let OwnerRepo { repo, owner } = ctx.owner_repo()?;
        let (_, releases_list) = spin_until_ready(
            ctx.client
                .repo_list_releases(
                    owner.as_str(),
                    repo.as_str(),
                    RepoListReleasesQuery {
                        ..Default::default()
                    },
                )
                .send(),
        )
        .await
        .into_diagnostic()?;

        match ctx.global_args.output_mode {
            crate::types::output::OutputMode::Pretty => {
                present_releases_list(&ctx, releases_list);
            }
            crate::types::output::OutputMode::Json => {
                releases_list.print_json()?;
            }
        }

        Ok(())
    }
}

fn present_releases_list(ctx: &BergContext<ListReleaseArgs>, releases: Vec<Release>) {
    let releases_empty = releases.is_empty();

    let table = ctx
        .make_table()
        .set_header(vec!["ID", "Name", "Tag", "Published At"])
        .add_rows(releases.into_iter().map(|release| {
            let Release {
                id,
                name,
                tag_name,
                published_at,
                ..
            } = release;
            vec![
                option_display(&id),
                option_display(&name),
                option_display(&tag_name),
                option_display(&published_at),
            ]
        }));

    let header = format!(
        "Releases{}",
        if releases_empty {
            " (empty)"
        } else {
            Default::default()
        }
    );

    let underscore = "=".repeat(header.len());
    let out = [header, underscore, table.show()].join("\n");

    println!("{out}");
}
