/*
 * Copyright (c) 2012 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <err.h>
#include <sysexits.h>
#include <ctype.h>
#include <util.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/disklabel.h>

#include <dev/raidframe/raidframevar.h>
#include <dev/raidframe/raidframeio.h>

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif  

void
panic(msg)
	char *msg;
{
	printf("%s\n", msg);
	exit(STATE_UNKNOWN);
}

char **
get_devs(void)
{
	size_t oldlen;
	char *oldp;
	int count;
	char *cp;
	char **result;
	char *last;
	
	if (sysctlbyname("hw.disknames", NULL, &oldlen, NULL, 0) != 0)
		panic("sysctlbyname(\"hw.disknames\") failed");

	if ((oldp = malloc(oldlen)) == NULL)
		panic("malloc failed");

	if (sysctlbyname("hw.disknames", oldp, &oldlen, NULL, 0) != 0)
		panic("sysctlbyname(\"hw.disknames\") failed");

	count = 1;
	for (cp = oldp; *cp; cp++) {
		if (isblank((int)*cp))
			count++;
	}

	if ((result = malloc(count * sizeof(*result))) == NULL)
		panic("malloc failed");
	
	result[count--] = NULL;
	result[count--] = oldp;

	for (cp = strtok_r(oldp, " ", &last);
	     (cp = strtok_r(NULL, " ", &last));
	     count--) {
		result[count] = cp;
	}

	return result;
}

int
get_status(dev)
	char *dev;
{
	int status = STATE_OK;
	int fd;
	char devname[MAXPATHLEN];
	RF_DeviceConfig_t rfdc;
	void *rfdcp = &rfdc;
	int i;

	fd = opendisk(dev, O_RDONLY, devname, sizeof(devname), 0);
	if (fd == -1)
		panic("opendisk failed");

	if (ioctl(fd, RAIDFRAME_GET_INFO, &rfdcp) != 0)
		panic("ioctl RAIDFRAME_GET_INFO failed");

	(void)close(fd);

	for (i = 0; i < rfdc.ndevs; i++) {
		switch (rfdc.devs[i].status) {
		case rf_ds_failed:
			printf("FAILED %s:%s ", dev, rfdc.devs[i].devname);
			status = STATE_CRITICAL;
			break;
		case rf_ds_reconstructing:
			printf("WARNING reconstructing %s:%s ", 
			       dev, rfdc.devs[i].devname);
			if (status != STATE_CRITICAL)
				status = STATE_WARNING;
			break;
		default:
			break;
		}
	}
	
	return status;
}

int
main(void)
{
	char **devs;
	char output[4096];
	int status = STATE_OK;
	int stmp;

	devs = get_devs();

	while (*devs != NULL) {
		if (strncmp(*devs, "raid", 4) == 0) {
			snprintf(output, sizeof(output), "%s %s",
				 output, *devs);
			stmp = get_status(*devs);

			if (stmp == STATE_CRITICAL)
				status = STATE_CRITICAL;

			if ((stmp == STATE_WARNING) && 
			    (status != STATE_CRITICAL))
				status = STATE_WARNING;
		}
		devs++;
	}

	if (status == STATE_OK) {
		printf("OK %s\n", output);
		return STATE_OK;
	} 

	printf("\n");
	return status;
}
