/*
 * Copyright (c) 2023 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <dirent.h>
#include <err.h>
#include <fcntl.h>
#include <paths.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/param.h>

#define NAGIOS_OK	0
#define NAGIOS_WARN	1
#define NAGIOS_CRIT	2
#define NAGIOS_UNKNOWN	3

#define DEFAULT_WARN 3
#define DEFAULT_CRIT 10

int
main(int argc, char **argv)
{
	int i;
	char ok_str[512] = "";
	size_t ok_strlen = 0;
	char bad_str[512] = "";
	size_t bad_strlen = 0;
	int retval = NAGIOS_OK;

	if (argc < 2)
		errx(NAGIOS_UNKNOWN, 
	  	     "usage: check_mqueue queudir,warn,crit ...");
	/* skip progname */
	argc--;
	argv++;


	for (i = 0; i < argc; i++) {
		char *last;
		char *queuedir;
		char *warnstr;
		char *critstr;
		int warn;
		int crit;
		DIR *queue;
		struct dirent *dp;
		int queuelen = 0;

		queuedir = strtok_r(argv[i], ",", &last);
		warnstr = strtok_r(NULL, ",", &last);
		critstr = strtok_r(NULL, ",", &last);

		if (queuedir == NULL || warnstr == NULL || critstr == NULL)
			errx(NAGIOS_UNKNOWN, 
		  	     "usage: check_mqueue queuedir,warn,crit ...");

		warn = atoi(warnstr);
		crit = atoi(critstr);
	

		if ((queue = opendir(queuedir)) == NULL) 
			err(NAGIOS_UNKNOWN, "opendir %s failed", queuedir);

		while ((dp = readdir(queue)) != NULL) {
			if (dp->d_namlen < 1)
				continue;
			if (dp->d_name[0] == 'd')
				queuelen++;
		}

		if (queuelen >= crit && retval < NAGIOS_CRIT)
			retval = NAGIOS_CRIT;
		else if (queuelen >= warn && retval < NAGIOS_WARN)
			retval = NAGIOS_WARN;

		if (queuelen >= warn || queuelen >= crit) {
			bad_strlen += snprintf(bad_str + bad_strlen,
					       sizeof(bad_str) - bad_strlen,
					       "%s%s has %d jobs pending",
					       bad_strlen ? ", " : "",
					       queuedir, queuelen);
		} else {
			ok_strlen += snprintf(ok_str + ok_strlen,
					       sizeof(ok_str) - ok_strlen,
					       "%s%s has %d jobs pending",
					       ok_strlen ? ", " : "",
					       queuedir, queuelen);
		}

		if (queue != NULL)
			closedir(queue);
	
	}

	if (retval == NAGIOS_OK) {
		printf("OK: %s\n", ok_str); 
	} else {
		printf("%s: %s\n", 
		       retval == NAGIOS_CRIT ? "CRITICAL" : "WARNING",
		       bad_str);
	}

	return retval;
}
