"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.DebugLine = void 0;

var _react = require("devtools/client/shared/vendor/react");

var _reactPropTypes = _interopRequireDefault(require("devtools/client/shared/vendor/react-prop-types"));

loader.lazyRequireGetter(this, "_index", "devtools/client/debugger/src/utils/editor/index");
loader.lazyRequireGetter(this, "_index2", "devtools/client/debugger/src/utils/pause/index");
loader.lazyRequireGetter(this, "_indentation", "devtools/client/debugger/src/utils/indentation");

var _reactRedux = require("devtools/client/shared/vendor/react-redux");

loader.lazyRequireGetter(this, "_constants", "devtools/client/debugger/src/constants");
loader.lazyRequireGetter(this, "_index3", "devtools/client/debugger/src/selectors/index");
loader.lazyRequireGetter(this, "_prefs", "devtools/client/debugger/src/utils/prefs");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class DebugLine extends _react.PureComponent {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "debugExpression", void 0);
  }

  static get propTypes() {
    return {
      editor: _reactPropTypes.default.object,
      selectedSource: _reactPropTypes.default.object,
      location: _reactPropTypes.default.object,
      why: _reactPropTypes.default.object,
      sourceTextContent: _reactPropTypes.default.object
    };
  }

  componentDidMount() {
    this.setDebugLine();
  }

  componentWillUnmount() {
    this.clearDebugLine(this.props);
  }

  componentDidUpdate(prevProps) {
    if (!_prefs.features.codemirrorNext) {
      (0, _index.startOperation)();
    }

    this.clearDebugLine(prevProps);
    this.setDebugLine();

    if (!_prefs.features.codemirrorNext) {
      (0, _index.endOperation)();
    }
  }

  setDebugLine() {
    const {
      why,
      location,
      editor,
      selectedSource
    } = this.props;

    if (!location) {
      return;
    }

    if (_prefs.features.codemirrorNext) {
      if (!selectedSource || location.source.id !== selectedSource.id) {
        return;
      }

      const {
        lineClass,
        markTextClass
      } = this.getTextClasses(why);
      const editorLocation = (0, _index.toEditorPosition)(location); // Show the paused "caret", to highlight on which particular line **and column** we are paused.
      //
      // Using only a `positionClassName` wouldn't only be applied to the immediate
      // token after the position and force to use ::before to show the paused location.
      // Using ::before prevents using :hover to be able to hide the icon on mouse hovering.
      //
      // So we have to use `createPositionElementNode`, similarly to column breakpoints
      // to have a new dedicated DOM element for the paused location.

      editor.setPositionContentMarker({
        id: _constants.markerTypes.PAUSED_LOCATION_MARKER,
        // Ensure displaying the marker after all the other markers and especially the column breakpoint markers
        displayLast: true,
        positions: [editorLocation],

        createPositionElementNode(_line, _column, isFirstNonSpaceColumn) {
          const pausedLocation = document.createElement("span");
          pausedLocation.className = `paused-location${isFirstNonSpaceColumn ? " first-column" : ""}`;
          const bar = document.createElement("span");
          bar.className = `vertical-bar`;
          pausedLocation.appendChild(bar);
          return pausedLocation;
        }

      });
      editor.setLineContentMarker({
        id: _constants.markerTypes.DEBUG_LINE_MARKER,
        lineClassName: lineClass,
        lines: [{
          line: editorLocation.line
        }]
      });
      editor.setPositionContentMarker({
        id: _constants.markerTypes.DEBUG_POSITION_MARKER,
        positionClassName: markTextClass,
        positions: [editorLocation]
      });
    } else {
      const doc = (0, _index.getDocument)(location.source.id);
      let {
        line,
        column
      } = (0, _index.toEditorPosition)(location);
      let {
        markTextClass,
        lineClass
      } = this.getTextClasses(why);
      doc.addLineClass(line, "wrap", lineClass);
      const lineText = doc.getLine(line);
      column = Math.max(column, (0, _indentation.getIndentation)(lineText)); // If component updates because user clicks on
      // another source tab, codeMirror will be null.

      const columnEnd = doc.cm ? (0, _index.getTokenEnd)(doc.cm, line, column) : null;

      if (columnEnd === null) {
        markTextClass += " to-line-end";
      }

      this.debugExpression = doc.markText({
        ch: column,
        line
      }, {
        ch: columnEnd,
        line
      }, {
        className: markTextClass
      });
    }
  }

  clearDebugLine(otherProps = {}) {
    if (_prefs.features.codemirrorNext) {
      const {
        location,
        editor,
        selectedSource
      } = this.props; // Remove the debug line marker when no longer paused, or the selected source
      // is no longer the source where the pause occured.

      if (!location || location.source.id !== selectedSource.id || otherProps?.location !== location || otherProps?.selectedSource?.id !== selectedSource.id) {
        editor.removeLineContentMarker(_constants.markerTypes.DEBUG_LINE_MARKER);
        editor.removePositionContentMarker(_constants.markerTypes.DEBUG_POSITION_MARKER);
        editor.removePositionContentMarker(_constants.markerTypes.PAUSED_LOCATION_MARKER);
      }
    } else {
      const {
        why,
        location
      } = otherProps; // Avoid clearing the line if we didn't set a debug line before,
      // or, if the document is no longer available

      if (!location || !(0, _index.hasDocument)(location.source.id)) {
        return;
      }

      if (this.debugExpression) {
        this.debugExpression.clear();
      }

      const {
        line
      } = (0, _index.toEditorPosition)(location);
      const doc = (0, _index.getDocument)(location.source.id);
      const {
        lineClass
      } = this.getTextClasses(why);
      doc.removeLineClass(line, "wrap", lineClass);
    }
  }

  getTextClasses(why) {
    if (why && (0, _index2.isException)(why)) {
      return {
        markTextClass: "debug-expression-error",
        lineClass: "new-debug-line-error"
      };
    } // In CM6, we no longer highlight the next token via debug-expression
    // and only highlight the line via paused-line.


    return {
      markTextClass: _prefs.features.codemirrorNext ? null : "debug-expression",
      lineClass: "paused-line"
    };
  }

  render() {
    return null;
  }

}

exports.DebugLine = DebugLine;

function isDocumentReady(location, sourceTextContent) {
  const contentAvailable = location && sourceTextContent; // With CM6, the codemirror document is no longer cached
  // so no need to check if its available

  if (_prefs.features.codemirrorNext) {
    return contentAvailable;
  }

  return contentAvailable && (0, _index.hasDocument)(location.source.id);
}

const mapStateToProps = state => {
  // Avoid unecessary intermediate updates when there is no location
  // or the source text content isn't yet fully loaded
  const frame = (0, _index3.getVisibleSelectedFrame)(state);
  const location = frame?.location;

  if (!location) {
    return {};
  } // For CM6, also check if we have a valid viewport.
  // This is a way to know if the actual source is displayed
  // and we are no longer on the "loading..." message


  if (_prefs.features.codemirrorNext) {
    const viewport = (0, _index3.getViewport)(state);

    if (!viewport) {
      return {};
    }
  }

  const sourceTextContent = (0, _index3.getSourceTextContent)(state, location);

  if (!isDocumentReady(location, sourceTextContent)) {
    return {};
  }

  return {
    location,
    why: (0, _index3.getPauseReason)(state, (0, _index3.getCurrentThread)(state)),
    sourceTextContent
  };
};

var _default = (0, _reactRedux.connect)(mapStateToProps)(DebugLine);

exports.default = _default;