/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.protocol.session;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.LongConsumer;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.commons.conf.CommonDescriptor;
import org.apache.iotdb.commons.conf.IoTDBConstant;
import org.apache.iotdb.commons.exception.IoTDBRuntimeException;
import org.apache.iotdb.commons.service.JMXService;
import org.apache.iotdb.commons.service.ServiceType;
import org.apache.iotdb.commons.service.metric.MetricService;
import org.apache.iotdb.commons.service.metric.enums.Metric;
import org.apache.iotdb.commons.service.metric.enums.Tag;
import org.apache.iotdb.commons.utils.AuthUtils;
import org.apache.iotdb.commons.utils.CommonDateTimeUtils;
import org.apache.iotdb.db.audit.AuditLogger;
import org.apache.iotdb.db.auth.AuthorityChecker;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.protocol.basic.BasicOpenSessionResp;
import org.apache.iotdb.db.protocol.session.IClientSession;
import org.apache.iotdb.db.protocol.session.MqttClientSession;
import org.apache.iotdb.db.protocol.session.SessionManagerMBean;
import org.apache.iotdb.db.protocol.thrift.OperationType;
import org.apache.iotdb.db.queryengine.common.SessionInfo;
import org.apache.iotdb.db.queryengine.plan.Coordinator;
import org.apache.iotdb.db.queryengine.plan.analyze.ClusterPartitionFetcher;
import org.apache.iotdb.db.queryengine.plan.analyze.schema.ClusterSchemaFetcher;
import org.apache.iotdb.db.queryengine.plan.execution.ExecutionResult;
import org.apache.iotdb.db.queryengine.plan.execution.IQueryExecution;
import org.apache.iotdb.db.queryengine.plan.parser.StatementGenerator;
import org.apache.iotdb.db.queryengine.plan.statement.Statement;
import org.apache.iotdb.db.queryengine.plan.statement.StatementType;
import org.apache.iotdb.db.queryengine.plan.statement.sys.AuthorStatement;
import org.apache.iotdb.db.storageengine.dataregion.read.control.QueryResourceManager;
import org.apache.iotdb.db.utils.DataNodeAuthUtils;
import org.apache.iotdb.db.utils.ErrorHandlingUtils;
import org.apache.iotdb.metrics.utils.MetricLevel;
import org.apache.iotdb.metrics.utils.MetricType;
import org.apache.iotdb.rpc.RpcUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.iotdb.service.rpc.thrift.TSConnectionInfo;
import org.apache.iotdb.service.rpc.thrift.TSConnectionInfoResp;
import org.apache.iotdb.service.rpc.thrift.TSLastDataQueryReq;
import org.apache.iotdb.service.rpc.thrift.TSProtocolVersion;
import org.apache.tsfile.read.common.block.TsBlock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManager
implements SessionManagerMBean {
    private static final Logger LOGGER = LoggerFactory.getLogger(SessionManager.class);
    private final ThreadLocal<IClientSession> currSession = new ThreadLocal();
    private final ThreadLocal<Long> currSessionIdleTime = new ThreadLocal();
    private final Map<IClientSession, Object> sessions = new ConcurrentHashMap<IClientSession, Object>();
    private final Object placeHolder = new Object();
    private final AtomicLong sessionIdGenerator = new AtomicLong();
    private final AtomicLong statementIdGenerator = new AtomicLong();
    private static final AuthorStatement AUTHOR_STATEMENT = new AuthorStatement(StatementType.AUTHOR);
    public static final TSProtocolVersion CURRENT_RPC_VERSION = TSProtocolVersion.IOTDB_SERVICE_PROTOCOL_V3;
    private static final boolean ENABLE_AUDIT_LOG = IoTDBDescriptor.getInstance().getConfig().isEnableAuditLog();

    protected SessionManager() {
        String mbeanName = String.format("%s:%s=%s", "org.apache.iotdb.service", "type", ServiceType.SESSION_MANAGER.getJmxName());
        JMXService.registerMBean((Object)this, (String)mbeanName);
    }

    public BasicOpenSessionResp login(IClientSession session, String username, String password, String zoneId, TSProtocolVersion tsProtocolVersion, IoTDBConstant.ClientVersion clientVersion) {
        return this.login(session, username, password, zoneId, tsProtocolVersion, clientVersion, IClientSession.SqlDialect.TREE);
    }

    public Long checkPasswordExpiration(String username, String password) {
        long passwordExpirationDays = CommonDescriptor.getInstance().getConfig().getPasswordExpirationDays();
        boolean mayBypassPasswordCheckInException = CommonDescriptor.getInstance().getConfig().isMayBypassPasswordCheckInException();
        TSLastDataQueryReq lastDataQueryReq = new TSLastDataQueryReq();
        lastDataQueryReq.setSessionId(0L);
        lastDataQueryReq.setPaths(Collections.singletonList("root.__system.password_history.`_" + username + "`.password"));
        long queryId = -1L;
        try {
            Statement statement = StatementGenerator.createStatement(lastDataQueryReq);
            SessionInfo sessionInfo = new SessionInfo(0L, AuthorityChecker.SUPER_USER, ZoneId.systemDefault());
            queryId = this.requestQueryId();
            ExecutionResult result = Coordinator.getInstance().executeForTreeModel(statement, queryId, sessionInfo, "", ClusterPartitionFetcher.getInstance(), ClusterSchemaFetcher.getInstance());
            if (result.status.getCode() != TSStatusCode.SUCCESS_STATUS.getStatusCode()) {
                LOGGER.warn("Fail to check password expiration: {}", (Object)result.status);
                throw new IoTDBRuntimeException("Cannot query password history because: " + result + ", please log in later or disable password expiration.", result.status.getCode());
            }
            IQueryExecution queryExecution = Coordinator.getInstance().getQueryExecution(queryId);
            Optional<TsBlock> batchResult = queryExecution.getBatchResult();
            if (batchResult.isPresent()) {
                TsBlock tsBlock = batchResult.get();
                if (tsBlock.getPositionCount() <= 0) {
                    Long l = null;
                    return l;
                }
                long lastPasswordTime = CommonDateTimeUtils.convertIoTDBTimeToMillis((long)tsBlock.getTimeByIndex(0));
                String oldPassword = tsBlock.getColumn(1).getBinary(0).toString();
                if (oldPassword.equals(AuthUtils.encryptPassword((String)password))) {
                    if (lastPasswordTime + passwordExpirationDays * 1000L * 86400L <= lastPasswordTime) {
                        Long l = Long.MAX_VALUE;
                        return l;
                    }
                    Long l = lastPasswordTime + passwordExpirationDays * 1000L * 86400L;
                    return l;
                }
                Long l = null;
                return l;
            }
            Long l = null;
            return l;
        }
        catch (Throwable e) {
            LOGGER.error("Fail to check password expiration", e);
            if (mayBypassPasswordCheckInException) {
                Long l = Long.MAX_VALUE;
                return l;
            }
            throw new IoTDBRuntimeException("Internal server error , please log in later or disable password expiration.", TSStatusCode.INTERNAL_SERVER_ERROR.getStatusCode());
        }
        finally {
            if (queryId != -1L) {
                Coordinator.getInstance().cleanupQueryExecution(queryId);
            }
        }
    }

    public BasicOpenSessionResp login(IClientSession session, String username, String password, String zoneId, TSProtocolVersion tsProtocolVersion, IoTDBConstant.ClientVersion clientVersion, IClientSession.SqlDialect sqlDialect) {
        BasicOpenSessionResp openSessionResp = new BasicOpenSessionResp();
        Long timeToExpire = this.checkPasswordExpiration(username, password);
        if (timeToExpire != null && timeToExpire <= System.currentTimeMillis()) {
            openSessionResp.sessionId(-1L).setCode(TSStatusCode.ILLEGAL_PASSWORD.getStatusCode()).setMessage("Password has expired, please use \"ALTER USER\" to change to a new one");
            return openSessionResp;
        }
        TSStatus loginStatus = AuthorityChecker.checkUser(username, password);
        if (loginStatus.getCode() == TSStatusCode.SUCCESS_STATUS.getStatusCode()) {
            if (!tsProtocolVersion.equals((Object)CURRENT_RPC_VERSION)) {
                openSessionResp.sessionId(-1L).setCode(TSStatusCode.INCOMPATIBLE_VERSION.getStatusCode()).setMessage("The version is incompatible, please upgrade to " + IoTDBConstant.VERSION);
            } else {
                session.setSqlDialect(sqlDialect);
                this.supplySession(session, username, ZoneId.of(zoneId), clientVersion);
                String logInMessage = "Login successfully";
                if (timeToExpire != null && timeToExpire != Long.MAX_VALUE) {
                    DateTimeFormatter dateFormat = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
                    logInMessage = logInMessage + ". Your password will expire at " + dateFormat.format(LocalDateTime.ofInstant(Instant.ofEpochMilli(timeToExpire), ZoneId.systemDefault()));
                } else if (timeToExpire == null) {
                    LOGGER.info("No password history for user {}, using the current time to create a new one", (Object)username);
                    long currentTime = CommonDateTimeUtils.currentTime();
                    TSStatus tsStatus = DataNodeAuthUtils.recordPassword(username, password, null, currentTime);
                    if (tsStatus.getCode() != TSStatusCode.SUCCESS_STATUS.getStatusCode()) {
                        openSessionResp.sessionId(-1L).setCode(tsStatus.getCode()).setMessage(tsStatus.getMessage());
                        return openSessionResp;
                    }
                    timeToExpire = CommonDateTimeUtils.convertIoTDBTimeToMillis((long)currentTime) + CommonDescriptor.getInstance().getConfig().getPasswordExpirationDays() * 1000L * 86400L;
                    if (timeToExpire > System.currentTimeMillis()) {
                        DateTimeFormatter dateFormat = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
                        logInMessage = logInMessage + ". Your password will expire at " + dateFormat.format(LocalDateTime.ofInstant(Instant.ofEpochMilli(timeToExpire), ZoneId.systemDefault()));
                    }
                }
                openSessionResp.sessionId(session.getId()).setCode(TSStatusCode.SUCCESS_STATUS.getStatusCode()).setMessage(logInMessage);
                LOGGER.info("{}: Login status: {}. User : {}, opens Session-{}", new Object[]{"IoTDB", openSessionResp.getMessage(), username, session});
                if (ENABLE_AUDIT_LOG) {
                    AuditLogger.log(String.format("%s: Login status: %s. User : %s, opens Session-%s", "IoTDB", openSessionResp.getMessage(), username, session), AUTHOR_STATEMENT);
                }
            }
        } else {
            if (ENABLE_AUDIT_LOG) {
                AuditLogger.log(String.format("User %s opens Session failed with an incorrect password", username), AUTHOR_STATEMENT);
            }
            openSessionResp.sessionId(-1L).setMessage(loginStatus.message).setCode(loginStatus.code);
        }
        return openSessionResp;
    }

    public boolean closeSession(IClientSession session, LongConsumer releaseByQueryId) {
        this.releaseSessionResource(session, releaseByQueryId);
        MetricService.getInstance().remove(MetricType.HISTOGRAM, Metric.SESSION_IDLE_TIME.toString(), new String[]{Tag.NAME.toString(), String.valueOf(session.getId())});
        IClientSession session1 = this.currSession.get();
        if (session1 != null && session != session1) {
            if (ENABLE_AUDIT_LOG) {
                AuditLogger.log(String.format("The client-%s is trying to close another session %s, pls check if it's a bug", session, session1), AUTHOR_STATEMENT);
            }
            return false;
        }
        if (ENABLE_AUDIT_LOG) {
            AuditLogger.log(String.format("Session-%s is closing", session), AUTHOR_STATEMENT);
        }
        return true;
    }

    private void releaseSessionResource(IClientSession session, LongConsumer releaseQueryResource) {
        Iterable<Long> statementIds = session.getStatementIds();
        if (statementIds != null) {
            for (Long statementId : statementIds) {
                Set<Long> queryIdSet = session.removeStatementId(statementId);
                if (queryIdSet == null) continue;
                for (Long queryId : queryIdSet) {
                    releaseQueryResource.accept(queryId);
                }
            }
        }
    }

    public TSStatus closeOperation(IClientSession session, long queryId, long statementId, boolean haveStatementId, boolean haveSetQueryId, LongConsumer releaseByQueryId) {
        if (!this.checkLogin(session)) {
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.NOT_LOGIN, (String)"Log in failed. Either you are not authorized or the session has timed out.");
        }
        try {
            if (haveStatementId) {
                if (haveSetQueryId) {
                    this.closeDataset(session, statementId, queryId, releaseByQueryId);
                } else {
                    this.closeStatement(session, statementId, releaseByQueryId);
                }
                return RpcUtils.getStatus((TSStatusCode)TSStatusCode.SUCCESS_STATUS);
            }
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.CLOSE_OPERATION_ERROR, (String)"statement id not set by client.");
        }
        catch (Exception e) {
            return ErrorHandlingUtils.onNpeOrUnexpectedException(e, OperationType.CLOSE_OPERATION, TSStatusCode.CLOSE_OPERATION_ERROR);
        }
    }

    public boolean checkLogin(IClientSession session) {
        boolean isLoggedIn;
        boolean bl = isLoggedIn = session != null && session.isLogin();
        if (!isLoggedIn) {
            LOGGER.info("{}: Not login. ", (Object)"IoTDB");
        }
        return isLoggedIn;
    }

    public long requestStatementId(IClientSession session) {
        long statementId = this.statementIdGenerator.incrementAndGet();
        session.addStatementId(statementId);
        return statementId;
    }

    public void closeStatement(IClientSession session, long statementId, LongConsumer releaseByQueryId) {
        Set<Long> queryIdSet = session.removeStatementId(statementId);
        if (queryIdSet != null) {
            for (Long queryId : queryIdSet) {
                releaseByQueryId.accept(queryId);
            }
        }
        session.removeStatementId(statementId);
    }

    public long requestQueryId(IClientSession session, Long statementId) {
        long queryId = this.requestQueryId();
        session.addQueryId(statementId, queryId);
        return queryId;
    }

    public long requestQueryId() {
        return QueryResourceManager.getInstance().assignQueryId();
    }

    public IClientSession getCurrSession() {
        return this.currSession.get();
    }

    public IClientSession getCurrSessionAndUpdateIdleTime() {
        IClientSession clientSession = this.getCurrSession();
        Long idleTime = this.currSessionIdleTime.get();
        if (idleTime == null) {
            this.currSessionIdleTime.set(System.nanoTime());
        } else {
            MetricService.getInstance().getOrCreateHistogram(Metric.SESSION_IDLE_TIME.toString(), MetricLevel.CORE, new String[]{Tag.NAME.toString(), String.valueOf(clientSession.getId())}).update(System.nanoTime() - idleTime);
        }
        return clientSession;
    }

    public void updateIdleTime() {
        this.currSessionIdleTime.set(System.nanoTime());
    }

    public TimeZone getSessionTimeZone() {
        IClientSession session = this.currSession.get();
        if (session != null) {
            return session.getTimeZone();
        }
        return TimeZone.getTimeZone(ZoneId.systemDefault());
    }

    public void removeCurrSession() {
        IClientSession session = this.currSession.get();
        if (session != null) {
            this.sessions.remove(session);
        }
        this.currSession.remove();
        this.currSessionIdleTime.remove();
    }

    public void removeCurrSessionForMqtt(MqttClientSession mqttClientSession) {
        if (mqttClientSession != null) {
            this.sessions.remove(mqttClientSession);
        }
    }

    public boolean registerSession(IClientSession session) {
        if (this.currSession.get() != null) {
            LOGGER.error("the client session is registered repeatedly, pls check whether this is a bug.");
            return false;
        }
        this.currSession.set(session);
        this.currSessionIdleTime.set(System.nanoTime());
        this.sessions.put(session, this.placeHolder);
        return true;
    }

    public void registerSessionForMqtt(IClientSession session) {
        this.sessions.put(session, this.placeHolder);
    }

    public void supplySession(IClientSession session, String username, ZoneId zoneId, IoTDBConstant.ClientVersion clientVersion) {
        session.setId(this.sessionIdGenerator.incrementAndGet());
        session.setUsername(username);
        session.setZoneId(zoneId);
        session.setClientVersion(clientVersion);
        session.setLogin(true);
        session.setLogInTime(System.currentTimeMillis());
    }

    public void closeDataset(IClientSession session, Long statementId, Long queryId, LongConsumer releaseByQueryId) {
        releaseByQueryId.accept(queryId);
        session.removeQueryId(statementId, queryId);
    }

    public static SessionManager getInstance() {
        return SessionManagerHelper.INSTANCE;
    }

    public SessionInfo getSessionInfo(IClientSession session) {
        return new SessionInfo(session.getId(), session.getUsername(), session.getZoneId(), session.getClientVersion(), session.getDatabaseName(), session.getSqlDialect());
    }

    public SessionInfo copySessionInfoForTreeModel(SessionInfo sessionInfo) {
        return new SessionInfo(sessionInfo.getSessionId(), sessionInfo.getUserName(), ZoneId.systemDefault(), sessionInfo.getVersion(), sessionInfo.getDatabaseName().orElse(null), IClientSession.SqlDialect.TREE);
    }

    public SessionInfo getSessionInfoOfTreeModel(IClientSession session) {
        return new SessionInfo(session.getId(), session.getUsername(), ZoneId.systemDefault(), session.getClientVersion(), session.getDatabaseName(), IClientSession.SqlDialect.TREE);
    }

    public SessionInfo getSessionInfoOfTableModel(IClientSession session) {
        return new SessionInfo(session.getId(), session.getUsername(), ZoneId.systemDefault(), session.getClientVersion(), session.getDatabaseName(), IClientSession.SqlDialect.TABLE);
    }

    public SessionInfo getSessionInfoOfPipeReceiver(IClientSession session, String databaseName) {
        return new SessionInfo(session.getId(), session.getUsername(), ZoneId.systemDefault(), session.getClientVersion(), databaseName, IClientSession.SqlDialect.TABLE);
    }

    @Override
    public Set<String> getAllRpcClients() {
        return this.sessions.keySet().stream().map(IClientSession::toString).collect(Collectors.toSet());
    }

    public TSConnectionInfoResp getAllConnectionInfo() {
        return new TSConnectionInfoResp(this.sessions.keySet().stream().filter(s -> StringUtils.isNotEmpty((CharSequence)s.getUsername())).map(IClientSession::convertToTSConnectionInfo).sorted(Comparator.comparingLong(TSConnectionInfo::getLogInTime)).collect(Collectors.toList()));
    }

    private static class SessionManagerHelper {
        private static final SessionManager INSTANCE = new SessionManager();

        private SessionManagerHelper() {
        }
    }
}

