/*
* @license Apache-2.0
*
* Copyright (c) 2019 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/**
* Evaluates the logarithm of the cumulative distribution function (CDF) for an arcsine distribution.
*
* @param x - input value
* @returns evaluated logCDF
*/
type Unary = ( x: number ) => number;

/**
* Interface for the logarithm of the cumulative distribution function (CDF) of an arcsine distribution.
*/
interface LogCDF {
	/**
	* Evaluates the logarithm of the cumulative distribution function (CDF) for an arcsine distribution with minimum support `a` and maximum support `b` at a value `x`.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated logCDF
	*
	* @example
	* var y = logcdf( 9.0, 0.0, 10.0 );
	* // returns ~-0.23
	*
	* @example
	* var y = logcdf( 0.5, 0.0, 2.0 );
	* // returns ~-1.1
	*
	* @example
	* var y = logcdf( +Infinity, 2.0, 4.0 );
	* // returns 0.0
	*
	* @example
	* var y = logcdf( -Infinity, 2.0, 4.0 );
	* // returns -Infinity
	*
	* @example
	* var y = logcdf( NaN, 0.0, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = logcdf( 0.0, NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = logcdf( 0.0, 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = logcdf( 2.0, 1.0, 0.0 );
	* // returns NaN
	*/
	( x: number, a: number, b: number ): number;

	/**
	* Returns a function for evaluating the logarithm of the cumulative distribution function (CDF) for an arcsine distribution with minimum support `a` and maximum support `b`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns logCDF
	*
	* @example
	* var mylogcdf = logcdf.factory( 0.0, 10.0 );
	* var y = mylogcdf( 0.5 );
	* // returns ~-1.941
	*
	* y = mylogcdf( 8.0 );
	* // returns ~-0.35
	*/
	factory( a: number, b: number ): Unary;
}

/**
* Arcsine distribution logarithm of cumulative distribution function (CDF).
*
* @param x - input value
* @param a - minimum support
* @param b - maximum support
* @returns evaluated logCDF
*
* @example
* var y = logcdf( 5.0, 0.0, 4.0 );
* // returns 0.0
*
* var mylogcdf = logcdf.factory( 0.0, 10.0 );
* y = mylogcdf( 0.5 );
* // returns ~-1.938
*
* y = mylogcdf( 8.0 );
* // returns ~-0.35
*/
declare var logCDF: LogCDF;


// EXPORTS //

export = logCDF;
