/*************************************************************************\
*   Copyright (C) 2009 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA            *
\*************************************************************************/

//--- LOCAL CLASSES ---
#include "yawpday.h"
#include "utils.h"

#include "logger/streamlogger.h"

//--- KDE4 CLASSES ---
#include <KLocalizedString>
#include <KSystemTimeZone>


YawpWeather::YawpWeather()
{
	clear();
}

YawpWeather::~YawpWeather()
{
}

void
YawpWeather::clear()
{
	m_sCurrIconName.clear();
	m_sIconName                 = QLatin1String("unknown");
	m_sDescription.clear();

	m_iWindSpeed                = SHRT_MAX;
	m_sWindDirection.clear();
	m_sWindShortText.clear();
	m_iWindChill                = SHRT_MAX;
	m_iWindGust                 = SHRT_MAX;

	m_iHumidity                 = SHRT_MAX;

	m_iCurrentTemperature       = SHRT_MAX;
	m_iCurrentRealFeelTemperature = SHRT_MAX;
	m_iTemperatureHigh          = SHRT_MAX;
	m_iTemperatureLow           = SHRT_MAX;
	m_iTemperatureRealFeelHigh  = SHRT_MAX;
	m_iTemperatureRealFeelLow   = SHRT_MAX;

	m_iDewpoint                 = SHRT_MAX;
	m_iPressure                 = SHRT_MAX;
	m_sPressureTendency.clear();
	m_sPressureShortText.clear();
	m_iUVIndex                  = SHRT_MAX;
	m_sUVRating.clear();
	m_iVisibility               = SHRT_MAX;

	m_bDayTime                  = true;

	m_vPropertyLines.clear();
}

const QString &
YawpWeather::currentIconName() const { return m_sCurrIconName; }

void
YawpWeather::setCurrentIconName(const QString & name) { m_sCurrIconName = name.toLower().replace(" ","-"); }

const QString &
YawpWeather::iconName() const { return m_sIconName; }

void
YawpWeather::setIconName(const QString & name) { m_sIconName = name.toLower().replace(" ","-"); }

const QString &
YawpWeather::description() const { return m_sDescription; }

void
YawpWeather::setDescription(const QString & desc) { m_sDescription = desc; }

const QString &
YawpWeather::windShortText() const {return m_sWindShortText;}

void
YawpWeather::setWindShortText(const QString& text) { m_sWindShortText = text; }

short int
YawpWeather::windSpeed() const { return m_iWindSpeed; }

void
YawpWeather::setWindSpeed(short speed) { m_iWindSpeed = speed; }

const QString &
YawpWeather::windDirection() const { return m_sWindDirection; }

void
YawpWeather::setWindDirection(const QString & direction) { m_sWindDirection = direction; }

short
YawpWeather::windChill() const { return m_iWindChill; }

void
YawpWeather::setWindChill(short windChill) { m_iWindChill = windChill; }

short
YawpWeather::windGust() const { return m_iWindGust; }

void
YawpWeather::setWindGust(short windGust) { m_iWindGust = windGust; }

short
YawpWeather::humidity() const { return m_iHumidity; }

void
YawpWeather::setHumidity(short humidity) { m_iHumidity = humidity; }

short
YawpWeather::currentTemperature() const { return m_iCurrentTemperature; }

void
YawpWeather::setCurrentTemperature(short current) { m_iCurrentTemperature = current; }

short
YawpWeather::currentRealFeelTemperature() const { return m_iCurrentRealFeelTemperature; }

void
YawpWeather::setCurrentRealFeelTemperature(short current) { m_iCurrentRealFeelTemperature = current; }

short
YawpWeather::lowTemperature() const { return m_iTemperatureLow; }

void
YawpWeather::setLowTemperature(short low) { m_iTemperatureLow = low; }

short
YawpWeather::highTemperature() const { return m_iTemperatureHigh; }

void
YawpWeather::setHighTemperature(short high) { m_iTemperatureHigh = high; }

short
YawpWeather::temperatureRealFeelLow() const { return m_iTemperatureRealFeelLow; }

void
YawpWeather::setTemperatureRealFeelLow(short low) { m_iTemperatureRealFeelLow = low; }

short
YawpWeather::temperatureRealFeelHigh() const { return m_iTemperatureRealFeelHigh; }

void
YawpWeather::setTemperatureRealFeelHigh(short high) { m_iTemperatureRealFeelHigh = high; }

short
YawpWeather::dewpoint() const { return m_iDewpoint; }

void
YawpWeather::setDewpoint(short dewpoint) { m_iDewpoint = dewpoint; }

short
YawpWeather::pressure() const { return m_iPressure; }

void
YawpWeather::setPressure(short pressure) { m_iPressure = pressure; }

const QString &
YawpWeather::pressureTendency() const { return m_sPressureTendency; }

void
YawpWeather::setPressureTendency(const QString & sTendency) { m_sPressureTendency = sTendency; }

const QString &
YawpWeather::pressureShortText() const { return m_sPressureShortText; }

void
YawpWeather::setPressureShortText(const QString& text) { m_sPressureShortText = text; }

short
YawpWeather::uvIndex() const { return m_iUVIndex; }

void
YawpWeather::setUVIndex(short uvIndex) { m_iUVIndex = uvIndex; }

const QString &
YawpWeather::uvRating() const { return m_sUVRating; }

void
YawpWeather::setUVRating(const QString & sUVRating) { m_sUVRating = sUVRating; }

short
YawpWeather::visibility() const { return m_iVisibility; }

void
YawpWeather::setVisibility(short visibility) { m_iVisibility = visibility; }

bool
YawpWeather::dayTime() const { return m_bDayTime; }

void
YawpWeather::setDayTime( bool bDayTime ) { m_bDayTime = bDayTime; }

QStringList &
YawpWeather::propertyTextLines() { return m_vPropertyLines; }

const QStringList &
YawpWeather::propertyTextLines() const { return m_vPropertyLines; }



YawpDay::YawpDay()
{
	clear();
}

YawpDay::~YawpDay()
{
}

void
YawpDay::clear()
{
	m_date			= QDate();
	m_sunrise		= QTime();
	m_sunset		= QTime();

	m_bHasNightValues	= false;
	m_vWeather[0].clear();
	m_vWeather[1].clear();
}

CityWeather::CityWeather()
{
	clear();
}

CityWeather::CityWeather( const CityWeather & other )
{
	*this = other;
}

CityWeather::~CityWeather()
{
	deleteAllDays();
}

void
CityWeather::setCity( const QString & city )
{
	m_sCity = city;
	createLocalizedCityString();
}

void
CityWeather::setCountry(const QString & country)
{
	m_sCountry = country;
	createLocalizedCityString();
}

CityWeather &
CityWeather::copy(const CityWeather & other)
{
	if (this == &other)
		return *this;

	m_sCity           = other.m_sCity;
	m_sCountry        = other.m_sCountry;
	m_sCountryCode    = other.m_sCountryCode;
	m_sProvider       = other.m_sProvider;
	m_sExtraData      = other.m_sExtraData;
	m_sLocalizedCity  = other.m_sLocalizedCity;

	m_lastUpdate      = other.m_lastUpdate;
	m_satelliteImage  = other.m_satelliteImage;
	
	m_timeZone        = other.m_timeZone;
	
	return *this;
}

CityWeather &
CityWeather::operator=(const CityWeather & other)
{
	return copy( other );
}

/*   Some providers do not provide all informations when searching for a city.
 *   Therefore we check on dataupdate if we can add country and countrycode.
 *   This is the reason, why we can not include sCountryCode and sCountry to compare two cities.
 */
bool
CityWeather::isEqual( const CityWeather & other ) const
{
	if( (quintptr)this == (quintptr)(&other) )
		return true;
	if( m_sProvider.compare(other.m_sProvider) != 0 )
		return false;
	if( !m_sExtraData.isEmpty() && !other.m_sExtraData.isEmpty() )
		return (m_sExtraData.compare(other.m_sExtraData) == 0);
	return (m_sCity.compare(other.m_sCity) == 0 && m_sCountry.compare(other.m_sCountry) == 0);
}

bool
CityWeather::operator== (const CityWeather & other) const
{
	return isEqual(other);
}

const QString &
CityWeather::countryCode() const
{
	return m_sCountryCode;
}

void
CityWeather::setCountryCode(const QString & cc)
{
	m_sCountryCode = cc;
}

const KTimeZone &
CityWeather::timeZone() const
{
	return m_timeZone;
}

bool
CityWeather::setTimeZone( const QString & timezone )
{
	if( !m_timeZone.isValid() || m_timeZone.name().compare(timezone, Qt::CaseInsensitive) != 0 )
		m_timeZone = KSystemTimeZones::zone(timezone);
	
	if( m_timeZone.isValid() && !m_timeZone.countryCode().isEmpty() )
		m_sCountryCode = m_timeZone.countryCode().toLower();

	return m_timeZone.isValid();
}


bool
CityWeather::isValid() const
{
	return (!m_sCity.isEmpty() && !m_sProvider.isEmpty());
}

void
CityWeather::clear()
{
	m_sCity.clear();
	m_sCountry.clear();
	m_sCountryCode.clear();
	m_sExtraData.clear();
	m_sProvider.clear();
	m_sLocalizedCity.clear();
	m_lastUpdate = QDateTime();
	m_satelliteImage = QImage();
	deleteAllDays();

	m_sCredit.clear();
	m_sCreditUrl.clear();
	
	m_timeZone = KTimeZone();
}

void
CityWeather::deleteAllDays()
{
	qDeleteAll( m_vDays.begin(), m_vDays.end() );
	m_vDays.clear();
}

void
CityWeather::createLocalizedCityString()
{
	QString sCity, sDistrict, sCountry;
	Utils::ExtractLocationInfo( m_sCity, sCity, sDistrict, sCountry );

	/*** Sometimes a provider returns more information to a city, when we request weather datas.
	*    Therefore the country does not has to be a part of the m_sCity string.
	*/
	if( sCountry.isEmpty() )
		sCountry = m_sCountry;
	else if( !m_sCountry.isEmpty() && sCountry.compare(m_sCountry) != 0 )
	{
		sDistrict = sCountry;
		sCountry = m_sCountry;
	}

	/*** Create the string.
	*/
	m_sLocalizedCity = sCity;
	if( !sCountry.isEmpty() )
		m_sLocalizedCity += ", " + i18nc("Country or state", sCountry.toUtf8().constData());
	if( !sDistrict.isEmpty() )
		m_sLocalizedCity += " (" + sDistrict + ")";
}

QDateTime
CityWeather::localTime() const
{
	return toLocalTime(QDateTime::currentDateTime());
}

QDateTime
CityWeather::toLocalTime(const QDateTime & dateTime) const
{
	KTimeZone local = KSystemTimeZones::local();
	if( m_timeZone.isValid() && local.name() != m_timeZone.name() )
		return KSystemTimeZones::local().convert(m_timeZone, dateTime);
	return dateTime;
}

QDateTime
CityWeather::fromLocalTime(const QDateTime & dateTime) const
{
	KTimeZone local = KSystemTimeZones::local();
	if( m_timeZone.isValid() && local.name() != m_timeZone.name() )
		return m_timeZone.convert(local, dateTime);
	return dateTime;
}

bool
CityWeather::isDayTime( const YawpDay * day ) const
{
	if( !day || !day->sunset().isValid() || !day->sunrise().isValid() )
		return true;

	QDateTime currentTime;
	if( observationPeriode().isValid() )
		currentTime = observationPeriode();
	else
		currentTime = localTime();
	
	QDateTime sunrise( currentTime.date(), day->sunrise() );
	QDateTime sunset( currentTime.date(), day->sunset() );
	
	bool bReturn = (currentTime > sunrise && currentTime  < sunset);

	dTracing() << "City: " << m_sLocalizedCity <<"   current Time:" << currentTime
	           << "   sunrise:" << sunrise << "   sunset:" << sunset
	           << "   daytime:" << bReturn;

	return bReturn;
}

void
CityWeather::debug_PrintCityWeather( const CityWeather & cityInfo )
{
	dDebug() << QString("City <%1> has weatherinfos for %2 day(s)")
		.arg(cityInfo.city()).arg(cityInfo.days().count());

	foreach( const YawpDay * pDay, cityInfo.days() )
	{
		dDebug();
		dDebug() << "Date: " << pDay->date();
		dDebug() << "   date          = " << pDay->date();
		dDebug() << "   sunrise       = " << pDay->sunrise();
		dDebug() << "   sunset        = " << pDay->sunset();
		dDebug();

		int iMax = (pDay->hasNightValues() ? 2 : 1);
		for( int i=0; i<iMax; ++i )
		{
			const YawpWeather & weather = (i == 0 ? pDay->weather() : pDay->nightWeather());
			const QString sPrefix(i==0 ? "" : "night ");

			dDebug() << QString("   %1current temp  = %2")
				.arg(sPrefix).arg(weather.currentTemperature()).toUtf8().constData();
			dDebug() << QString("   %1current real feel temp = %2")
				 .arg(sPrefix).arg(weather.currentRealFeelTemperature()).toUtf8().constData();
			dDebug() << QString("   %1current icon  = %2")
				.arg(sPrefix).arg(weather.currentIconName()).toUtf8().constData();

			dDebug() << QString("   %1iconName      = %2")
				.arg(sPrefix).arg(weather.iconName()).toUtf8().constData();
			dDebug() << QString("   %1low           = %2")
				.arg(sPrefix).arg(weather.lowTemperature()).toUtf8().constData();
			dDebug() << QString("   %1realfeellow   = %2")
				.arg(sPrefix).arg(weather.temperatureRealFeelLow()).toUtf8().constData();
			dDebug() << QString("   %1high          = %2")
				.arg(sPrefix).arg(weather.highTemperature()).toUtf8().constData();
			dDebug() << QString("   %1realfeelhigh  = %2")
				.arg(sPrefix).arg(weather.temperatureRealFeelHigh()).toUtf8().constData();
			dDebug() << QString("   %1description   = %2")
				.arg(sPrefix).arg(weather.description()).toUtf8().constData();
			dDebug() << QString("   %1humidity      = %2")
				.arg(sPrefix).arg(weather.humidity()).toUtf8().constData();
			dDebug() << QString("   %1windSpeed     = %2")
				.arg(sPrefix).arg(weather.windSpeed()).toUtf8().constData();
			dDebug() << QString("   %1windDirection = %2")
				.arg(sPrefix).arg(weather.windDirection()).toUtf8().constData();
			dDebug() << QString("   %1wind chill    = %2")
				.arg(sPrefix).arg(weather.windChill()).toUtf8().constData();
			dDebug() << QString("   %1wind gust    = %2")
				.arg(sPrefix).arg(weather.windGust()).toUtf8().constData();

			dDebug() << QString("  %1property text lines = %2")
				.arg(sPrefix).arg(weather.propertyTextLines().count()).toUtf8().constData();
			  
			dDebug();
		}
	}
}
