/*************************************************************************\
*   Copyright (C) 2011 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA            *
\*************************************************************************/

//--- LOCAL CLASSES ---
#include "paneldesktopinterface.h"

//--- QT4 CLASSES ---
#include <QGraphicsSceneResizeEvent>
#include <QGraphicsLinearLayout>


PanelDesktopInterface::PanelDesktopInterface(
	BaseDesktopPainter * painter,
	QGraphicsItem * parent,
	Qt::WindowFlags wFlags)
    : QGraphicsWidget(parent, wFlags)
{
	m_painter = painter;
	if (m_painter != NULL)
		m_painter->setWidget(this);
	
	m_busyWidget = new Plasma::BusyWidget();
	m_busyWidget->setAcceptHoverEvents(false);
	m_busyWidget->setAcceptedMouseButtons(Qt::NoButton);

	QGraphicsLinearLayout * mainLayout = new QGraphicsLinearLayout(this);
	mainLayout->addStretch();
	mainLayout->addItem(m_busyWidget);
	mainLayout->addStretch();

	setBusy(false);
}

PanelDesktopInterface::~PanelDesktopInterface()
{
	if (m_painter != NULL)
		delete m_painter;
	m_busyWidget->deleteLater();
}

BaseDesktopPainter *
PanelDesktopInterface::desktopPainter() const
{
	return (BaseDesktopPainter *)m_painter;
}

void
PanelDesktopInterface::setDesktopPainter(BaseDesktopPainter * painter)
{
	if (m_painter != NULL)
		delete m_painter;

	m_painter = painter;
	
	if (m_painter != NULL)
	{
		m_painter->setWidget(this);
		setupPainterSize();
	}
}

void
PanelDesktopInterface::resizeEvent(QGraphicsSceneResizeEvent * event)
{
	if (m_painter == NULL)
		return;

	setupPainterSize();
	
	event->accept();
}

void
PanelDesktopInterface::setupPainterSize()
{
	QSizeF contentsSize = size();
	QRect painterRect;

	bool keepAspectRatio = m_painter->aspectRatioMode() == Plasma::KeepAspectRatio;
	int heightForWidth = m_painter->heightForWidth(contentsSize.width());
	
	if (heightForWidth <= contentsSize.height())
		painterRect = QRect(0, 0, contentsSize.width(), keepAspectRatio ? heightForWidth : contentsSize.height());
	else
	{
		int widthForHeight = m_painter->widthForHeight(contentsSize.height());
		if (widthForHeight <= contentsSize.width())
			painterRect = QRect(0, 0, keepAspectRatio ? widthForHeight : contentsSize.width() , contentsSize.height());
	}
	
	if (painterRect != m_painterRect)
	{
		m_painterRect = painterRect;
		m_painter->update();
	}
}

void
PanelDesktopInterface::mousePressEvent(QGraphicsSceneMouseEvent * event)
{
	if (m_painter == NULL)
		return;

	const CityWeather * const pCity = m_painter->stateMachine()->currentCity();

	if (pCity == NULL ||
	    m_painter->timeLine()->state() == QTimeLine::Running ||
	    event->button() != Qt::LeftButton )
	{
		return;
	}

	m_painter->mousePressEvent(event);
}

void
PanelDesktopInterface::paint(QPainter * painter, const QStyleOptionGraphicsItem * option, QWidget * widget)
{
	Q_UNUSED(widget);

	if (m_painter != NULL)
		m_painter->paintApplet(painter, option, m_painterRect);
}

void
PanelDesktopInterface::setBusy(bool busy)
{
	m_busyWidget->setRunning(busy);
	m_busyWidget->setVisible(busy);
}
