/***************************************************************************
@author:    Fabian Wuertz <xadras@sidux.com>
@copyright: Fabian Wuertz <xadras@sidux.com>
@license:   GPL2, GPL3

@project:   Waheela
@file:      dateTime.cpp    
***************************************************************************/


#include "dateTime.h"


#include <KColorDialog>
#include <klocale.h>
#include <KPageDialog>
#include <QDesktopWidget>
#include <QApplication>
#include <QTimer>

DateTime::DateTime (QWidget* parent, Qt::WFlags flags): QWidget (parent, flags)
{
	settingsWidget = new QWidget();
	dateTimeUi.setupUi(settingsWidget);

	QDesktopWidget* desktopWidget = QApplication::desktop();
	QRect screensize = desktopWidget->screenGeometry();
	
	setMinimumSize(QSize(screensize.width(),60));
	show();
	setAutoFillBackground(TRUE);
	
	colorButton  = new KPushButton(KIcon("applications-graphics"), "");
	colorButton -> setIconSize(QSize(24,24));
	colorButton -> setGeometry(10, 10, 30, 30);
	colorButton -> hide();
	colorButton -> setStyleSheet("border: 0");
	connect(colorButton, SIGNAL(clicked()), this, SLOT(showColorDialog()));

	configButton  = new KPushButton(KIcon("xclock"), "", this);
	configButton -> setIconSize(QSize(24,24));
	configButton -> setGeometry(50, 10, 30, 30);
	configButton -> hide();
	configButton -> setStyleSheet("border: 0");
	connect(configButton, SIGNAL(clicked()), this , SLOT( showSettingsDialog() ) );

	dateLabel     = new QLabel();
	dateLabel    -> show();
	timeLabel     = new QLabel();
	timeLabel    -> show();
	
	QWidget *spacerWidget2 = new QWidget();
	spacerWidget2->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

	QHBoxLayout *layout = new QHBoxLayout;
	layout->addWidget(colorButton);
	layout->addWidget(spacerWidget2);
	layout->addWidget(configButton);
	layout->addWidget(dateLabel);
	layout->addWidget(timeLabel);
	setLayout(layout);
	
	updateClock();
}

//------------------------------------------------------------------------------
//--- clock --------------------------------------------------------------------
//------------------------------------------------------------------------------

void DateTime::updateClock() {
	QDateTime time = QDateTime::currentDateTime();
	dateLabel->setText(time.toString(dateFormat));
	timeLabel->setText(time.toString(timeFormat));
	QTimer::singleShot(10000, this, SLOT(updateClock()));
}

//------------------------------------------------------------------------------
//--- control elements ---------------------------------------------------------
//------------------------------------------------------------------------------

void DateTime::hideControlElements() {
	colorButton->hide();
	configButton->hide();
}

void DateTime::showControlElements() {
	colorButton->show();
	configButton->show();
}

//------------------------------------------------------------------------------
//--- settings -----------------------------------------------------------------
//------------------------------------------------------------------------------


void DateTime::showColorDialog() {
	KColorDialog dataTimeColorDialog(this);
	QColor currentBackgroundColor = palette().color(QPalette::Background);
	dataTimeColorDialog.setColor( currentBackgroundColor );
	connect(&dataTimeColorDialog, SIGNAL(colorSelected(const QColor &)), this, SLOT(changeBackgroundColor(const QColor &)));
	dataTimeColorDialog.exec();
}


void DateTime::changeBackgroundColor(const QColor& backgroundColor) {
	KConfig themeConfig("waheela/themes/"+theme+".config");
	KConfigGroup dateTimeSettings( &themeConfig, "DateTime" );
	dateTimeSettings.writeEntry( "BackgroundColor", backgroundColor );
	QColor backgroundColor2 = QColor(backgroundColor);
	backgroundColor2.setAlpha(127);
	QPalette p = palette();
	p.setBrush(QPalette::Background, QColor(backgroundColor2) );
	setPalette(p);
}


void DateTime::showSettingsDialog()
{
  	dateTimeUi.dateFormatLineEdit->setText( dateFormat );
	dateTimeUi.timeFormatLineEdit->setText( timeFormat );
	
	dateTimeUi.showDateCheckBox->setChecked( dateLabel->isVisible() );
	dateTimeUi.showTimeCheckBox->setChecked( timeLabel->isVisible() );

	dateTimeUi.fontChooser->setFont(dateLabel->font());
	dateTimeUi.fontKcolorbutton->setColor(dateLabel->palette().color(QPalette::WindowText));
	
	KPageDialog *dateTimeDialog = new KPageDialog(this);
	connect(dateTimeDialog, SIGNAL(okClicked()), this, SLOT(apply()));
	dateTimeDialog->setButtons( KDialog::Ok | KDialog::Cancel);
	dateTimeDialog->addPage(settingsWidget, i18n("Date&Time"));
	dateTimeDialog->exec();
}


void DateTime::apply()
{
	KConfig config("waheela/waheela.config");
	KConfig themeConfig("waheela/themes/"+theme+".config");
	KConfigGroup settings( &themeConfig, "DateTime" );

	// format
	dateFormat = dateTimeUi.dateFormatLineEdit->text();
	timeFormat = dateTimeUi.timeFormatLineEdit->text();
	KConfigGroup formatSettings( &config, "Format" );
	formatSettings.writeEntry( "DateFormat", dateFormat );
	formatSettings.writeEntry( "TimeFormat", timeFormat );
	
	// visible
	settings.writeEntry( "DateIsVisible", dateTimeUi.showDateCheckBox->isChecked() );
	settings.writeEntry( "TimeIsVisible", dateTimeUi.showTimeCheckBox->isChecked() );

	// font
	settings.writeEntry( "Font", dateTimeUi.fontChooser->font() );
	settings.writeEntry( "FontColor", dateTimeUi.fontKcolorbutton->color() );
	
	setup(settings, formatSettings, theme);
}



void DateTime::setup(KConfigGroup settings, KConfigGroup formatSettings, QString inputTheme) {
	theme = inputTheme;
  
	// color
	QColor backgroundColor = settings.readEntry( "BackgroundColor", QColor( 0, 0, 255, 127) );
	backgroundColor.setAlpha(127);
	QPalette p = palette();
	p.setBrush(QPalette::Background, QColor(backgroundColor) );
	setPalette(p);
	setWindowOpacity(0.1);
  
	// date format
	dateFormat  = formatSettings.readEntry( "DateFormat", "dddd d. MMMM yyyy" );
	timeFormat  = formatSettings.readEntry( "TimeFormat", "h:mm" );
	QDateTime time = QDateTime::currentDateTime();
	dateLabel->setText(time.toString(dateFormat));
	timeLabel->setText(time.toString(timeFormat));
	
	// visible
	bool dateIsVisible = settings.readEntry( "DateIsVisible", true );
	dateLabel    -> setVisible( dateIsVisible );
	bool timeIsVisible = settings.readEntry( "TimeIsVisible", true );
	timeLabel    -> setVisible( timeIsVisible );
	if(dateLabel->isHidden() and timeLabel->isHidden()) {
		setHidden(TRUE);
	}
	
	// font
	QFont dateTimeFont = settings.readEntry( "Font", QFont("Sans Serif", 13, QFont::Normal) );
	dateLabel->setFont(dateTimeFont);
	dateTimeFont.setBold(true);
	timeLabel->setFont(dateTimeFont);
	// font color
	QColor fontColor = settings.readEntry( "FontColor", QColor( 0, 0, 255, 127) );
	QPalette q = palette();
	q.setBrush(QPalette::WindowText, QColor(fontColor) );
	dateLabel->setPalette(q);
	timeLabel->setPalette(q);
}


//------------------------------------------------------------------------------
//--- show/hide ----------------------------------------------------------------
//------------------------------------------------------------------------------

void DateTime::show2() {
  	show();
	if( dateLabel->isVisible() or timeLabel->isVisible() ) {
		return;
	}
	KConfig waheelaConfig("waheela/waheela.config");
	KConfigGroup settings( &waheelaConfig, "DateTime" );
	dateLabel->show();
	settings.writeEntry( "DateIsVisible", TRUE );
	timeLabel->show();
	settings.writeEntry( "TimeIsVisible", TRUE );
}

void DateTime::hide2() {
	hide();
	KConfig waheelaConfig("waheela/waheela.config");
	KConfigGroup settings( &waheelaConfig, "DateTime" );
	settings.writeEntry( "DateIsVisible", FALSE );
	settings.writeEntry( "TimeIsVisible", FALSE );
}