// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#include <UnitTest.hpp>

#include "Private.hpp"

#include <algorithm>
#include <cstdlib>
#include <stdexcept>
#include <typeinfo>

#include <cxxabi.h>

// This file contains implementations of methods for UnitTest and
// AssertionFailure.

namespace UnitTest {

  namespace Private {

    class MessageAssertionFailure : public BasicAssertionFailure {
    public:
      MessageAssertionFailure(const std::string &message,
			      const char *filename,
			      int line, const char *function)
	: BasicAssertionFailure(filename, line, function),
	  message_(message) {
      }

      void printMe(std::ostream &stream, const std::string &testName) const {
        BasicAssertionFailure::printMe(stream, testName);

        stream << message_ << std::endl;
      }

    private:
      const std::string message_;
    };

    std::string demangledName(const std::type_info &typeinfo) {
      int status;

      char *name = abi::__cxa_demangle(typeinfo.name(), 0,
				       0, &status);
      std::string retval(name ? name : typeinfo.name());

      std::free(name);
  
      return retval;
    }

    void printFailure(std::ostream &stream,
		      const char *filename, int line,
		      const std::string &message,
		      const std::string &testName,
		      const char *function,
		      const std::string &trailer) {
      // NOTE (2004-01-19, carlton): The space after the colon after the
      // line number is there to make this match the appropriate
      // compilation error regexp with XEmacs 21.4p14 (and possibly
      // other Emacsen.)
      std::ostringstream oss;
      oss << std::endl
	  << filename << ":"
	  << line << ": " << std::endl
	  << message << " in test "
	  << testName << std::endl
	  << "within function "
	  << function << trailer;
      stream << oss.str() << std::endl;
    }

    BasicAssertionFailure::BasicAssertionFailure (const char *filename,
						  int line,
						  const char *function)
      : filename_(filename), line_(line), function_(function) {
    }

    void BasicAssertionFailure::printMe(std::ostream &stream,
					const std::string &testName)
      const {
      printFailure(stream, filename_, line_, "Assertion failure",
		   testName, function_, "");
    }

    Radix radix(DECIMAL);

  }

  using namespace Private;

  void assertTrue(bool val, const char *filename,
		  int line, const char *function) {
    if (!val)
      throw BasicAssertionFailure(filename, line, function);
  }

  void assertEquals(const std::string &expected, const char *actual,
		    const char *filename,
		    int line, const char *function) {
    if (expected != actual)
      throw EqualityAssertionFailure<std::string>
	(expected, actual, filename, line, function);
  }

  void assertEquals(const char *expected, const std::string &actual,
		    const char *filename,
		    int line, const char *function) {
    if (expected != actual)
      throw EqualityAssertionFailure<std::string>
	(expected, actual, filename, line, function);
  }

  void assertMessage(const std::string &message, const char *filename,
		     int line, const char *function) {
    throw MessageAssertionFailure(message, filename, line, function);
  }

  void assertFailure(const std::string &message, const char *filename,
		     int line, const char *function) {
    failureRequests_.pop_front();
    failureRequests_.push_front(FailureRequest(message, filename,
					       line, function));
  }

  Radix radix() {
    return Private::radix;
  }

  void radix(Radix radix) {
    Private::radix = radix;
  }

  int Test::passes() const {
    return 0;
  }

  int Test::failures() const {
    return 0;
  }

  void Test::setUp() {
    // Default is to do nothing.
  }

  void Test::tearDown() {
    // Default is to do nothing.
  }

  std::string Test::name() const {
    return demangledName(typeid(*this));
  }

  TestPtr::TestPtr(Test *test)
    : ref_(new int(1)),
      test_(test) {
  }

  TestPtr::TestPtr(const TestPtr &other)
    : ref_(other.ref_),
      test_(other.test_) {
    if (ref_)
      ++*ref_;
  }

  TestPtr::TestPtr()
    : ref_(0),
      test_(0) {
  }

  TestPtr::~TestPtr() {
    if (!ref_)
      return;

    --*ref_;

    if (!*ref_) {
      delete ref_;
      delete test_;
    }
  }

  TestPtr &TestPtr::operator=(const TestPtr &other) {
    TestPtr copy(other);

    std::swap(ref_, copy.ref_);
    std::swap(test_, copy.test_);

    return *this;
  }

  Test *TestPtr::operator->() const {
    return test_;
  }

  Test &TestPtr::operator*() const {
    return *test_;
  }

  AssertionFailure::AssertionFailure() {
    triggered();
  }

  AssertionFailure::~AssertionFailure() {
  }

}
