// The original version of this code (revision 2 of the Subversion
// archive) was released subject to the following license:

//   Copyright (c) 2006, Sun Microsystems, Inc.  All rights reserved.
//   Redistribution and use in source and binary forms, with or
//   without modification, are permitted provided that the following
//   conditions are met:

//   * Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above
//     copyright notice, this list of conditions and the following
//     disclaimer in the documentation and/or other materials provided
//     with the distribution.
//   * Neither the name of Sun Microsystems or the names of
//     contributors may be used to endorse or promote products derived
//     from this software without specific prior written permission.
 
//   THIS SOFTWARE IS PROVIDED BY SUN AND ITS LICENSORS ``AS IS'' AND
//   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
//   THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
//   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SUN OR ITS
//   LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
//   AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
//   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//   ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
//   POSSIBILITY OF SUCH DAMAGE.

// Subsequent additions and modifications are Copyright (c) 2006 David
// Carlton, and may be used subject to the same conditions.

#include <Registry.hpp>

#include <cstring>
#include <iostream>
#include <list>
#include <stdexcept>
#include <utility>

#include "Private.hpp"

namespace UnitTest {

  namespace Private {

    bool Debug::debug_ = false;

    class Okay {};

    class BadTest {
    public:
      BadTest(const std::string &message) : message_(message) {}

      std::string message() const { return message_; }

    private:
      std::string message_;
    };

  }

  using namespace Private;

  Registry::Registry()
    : stream_(&std::cout) {
  }

  Registry::~Registry() {
  }

  void Registry::add(TestPtr test,
		     const char *name) {
    tests_.push_back(Registration(test, name));
  }

  int Registry::run(int argc, char **argv) {
    try {
      bool debug = pruneArgs(argc, argv);
      Suite suite;
      addTests(argc, argv, suite);
      return (run(debug, suite) ? 0 : 1);
    } catch (const Okay &) {
      return 0;
    } catch (const BadTest &oops) {
      *stream_ << oops.message() << std::endl;;
      return 1;
    } catch (std::exception &except) {
      *stream_ << "Tests aborted because of unexpected exception:\n"
	       << except.what() << std::endl;
      return 1;
    } catch (...) {
      *stream_ << "Tests aborted because of unexpected exception."
	       << std::endl;
      return 1;
    }
  }

  void Registry::stream(std::ostream &stream) {
    stream_ = &stream;
  }

  void Registry::addTests(int argc, char **argv, Suite &suite) {
    if (argc == 1) {
      for (std::list<Registry::Registration>::const_iterator i =
	     tests_.begin();
	   i != tests_.end(); ++i) {
	suite.add(i->first);
      }
    } else {
      for (int j = 1; j < argc; ++j)
	suite.add(test(argv[j]));
    }
  }

  bool Registry::pruneArgs(int &argc, char **&argv) {
    if (tests_.empty())
      throw BadTest("No tests available.");

    if ((argc >= 2) && !std::strcmp(argv[1], "-debug")) {
      --argc;
      ++argv;
      return true;
    }

    if ((argc >= 2) && !std::strcmp(argv[1], "-list")) {
      for (std::list<Registry::Registration>::const_iterator reg =
	     tests_.begin();
	   reg != tests_.end(); ++reg)
	*stream_ << reg->second << std::endl;
      throw Okay();
    }

    return false;
  }

  TestPtr Registry::test(const std::string &name ) {
    for (std::list<Registry::Registration>::const_iterator i =
	   tests_.begin();
	 i != tests_.end(); ++i) {
      if (name == i->second)
	return i->first;
    }

    throw BadTest
      ("Bad test name: " + name + ".  Use -list for valid names.");
  }

  bool Registry::run(bool debug, Suite &suite) {
    Debug guard(debug);

    suite.stream(*stream_);
    suite.run();

    std::ostringstream oss;
    oss << std::endl << "Tests finished with "
	<< suite.passes() << " passes and "
	<< suite.failures() << " failures.";

    *stream_ << oss.str() << std::endl;

    return (suite.failures() == 0);
  }

}
