#!/bin/sh -efu
#
# Copyright (c) 2017 Dmitry V. Levin <ldv@altlinux.org>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

usage()
{
	cat >&2 <<EOF
Usage: $0 [<input> [<output>]]

Generate test script(s) from <input> table.
If a <output> is specified, generate the script for this test only,
otherwise generate scripts for all tests specified in <input> table.
EOF
	exit 1
}

input="${0%/*}/gen_tests.in"
[ $# -eq 0 ] || { input="$1"; shift; }
output=
[ $# -eq 0 ] || { output="$1"; shift; }
[ $# -eq 0 ] || usage

if [ -n "$output" ]; then
	match="${output##*/}"
	match="${match%.gen.test}"
	[ -n "$match" ] || usage
else
	match=
	dir="$(dirname "$input")"
fi

names=

while read -r name args; do {
	[ -n "${name###*}" ] || continue
	if [ -z "$match" ]; then
		names="$names $name"
		output="$dir/$name.gen.test"
	else
		[ "$match" = "$name" ] || continue
	fi

	hdr="\
#!/bin/sh -efu
# Generated by $0 from $input ($name $args); do not edit."

	case "$args" in
		+*)
		cat <<-EOF
		$hdr
		. "\${srcdir=.}/${args#+}"
		EOF
		;;

		''|-*)
		cat <<-EOF
		$hdr
		. "\${srcdir=.}/init.sh"
		run_strace_match_diff $args
		EOF
		;;

		*)
		cat <<-EOF
		$hdr
		. "\${srcdir=.}/init.sh"
		$args
		EOF
		;;
	esac > "$output"

	chmod a+x "$output"
} < /dev/null; done < "$input"

if [ -n "$names" ]; then
	{
		printf '# Generated by %s from %s; do not edit.\n' "$0" "$input"
		printf 'GEN_TESTS ='
		printf ' %s.gen.test' $names
		echo
		target='$(srcdir)/%s.gen.test'
		dep1='$(abs_srcdir)/gen_tests.sh'
		dep2='$(srcdir)/gen_tests.in'
		recipe='$(AM_V_GEN) $^ $@'
		printf "\\n$target: $dep1 $dep2\\n\\t$recipe\\n" $names
	} > "$dir/gen_tests.am"
fi
