/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#ifndef RAST_H
#define RAST_H

/**
 * @file rast.h rast
 */

#include <apr.h>

#include "rast/error.h"
#include "rast/types.h"
#include "rast/macros.h"

RAST_EXTERN_C_BEGIN

/**
 * @defgroup rast rast
 * @{
 */

RAST_EXTERN_C_BEGIN

#define RAST_RDWR       0       /**< This is OBSOLETE.  Please replace to RAST_DB_RDWR */
#define RAST_RDONLY     1       /**< This is OBSOLETE.  Please replace to RAST_DB_RDONLY */

#define RAST_BUFSIZ 1024

#define RAST_RESULT_ALL_ITEMS -1

#define RAST_CALC_SCORE_STANDALONE -1

/** Enum for sort methods */
typedef enum {
    RAST_SORT_METHOD_SCORE,      /**< Sort by score */
    RAST_SORT_METHOD_PROPERTY    /**< Sort by property */
} rast_sort_method_e;

/** Enum for sort order */
typedef enum {
    /**
     * descending if sort method is RAST_SORT_METHOD_SCORE,
     * ascending otherwise
     */
    RAST_SORT_ORDER_DEFAULT,
    RAST_SORT_ORDER_ASCENDING,   /**< ascending order */
    RAST_SORT_ORDER_DESCENDING   /**< descending order */
} rast_sort_order_e;

/** Enum for sort methods */
typedef enum {
    RAST_SCORE_METHOD_NONE,      /**< Do not calculate scores */
    RAST_SCORE_METHOD_TFIDF,     /**< Use TF-IDF method */
} rast_score_method_e;

/** Options for search */
typedef struct {
    /** The start number of requested items.  start_no starts from 0. */
    rast_doc_id_t start_no;
    /** The number of requested items */
    int num_items;
    /** Whether summary is needed or not */
    int need_summary;
    /** The number of characters of summary */
    rast_size_t summary_nchars;
    /** The sort method */
    rast_sort_method_e sort_method;
    /** The sort property */
    const char *sort_property;
    /** The sort order */
    rast_sort_order_e sort_order;
    /** The score method */
    rast_score_method_e score_method;
    /** The requested properties */
    const char **properties;
    /** The number of requested properties */
    int num_properties;
    /** The number of documents in all of search nodes */
    int all_num_docs;
    /** The number of documents that contains the term */
    int *terms;
    /** The number of terms */
    int num_terms;
} rast_search_option_t;

/** Enum for byte order */
typedef enum {
    RAST_LITTLE_ENDIAN = 0,  /**< Little endian */
    RAST_BIG_ENDIAN = 1,     /**< Big endian */
    RAST_NATIVE_ENDIAN = 2,  /**< Native endian */

    RAST_UNKNOWN_ENDIAN = -1   /**< Not implemented database */
} rast_byte_order_e;

static inline rast_byte_order_e
rast_check_byte_order()
{
    union {
        long long_value;
        char char_value[sizeof(long)];
    } data;

    data.long_value = 1;
    return data.char_value[0] == 1 ? RAST_LITTLE_ENDIAN : RAST_BIG_ENDIAN;
}

static inline apr_uint32_t
rast_swap32(apr_uint32_t x)
{
    return ((x & 0x000000FF) << 24) |
        ((x & 0x0000FF00) << 8) |
        ((x & 0x00FF0000) >> 8) |
        ((x & 0xFF000000) >> 24);
}

static inline rast_size_t
rast_fix_byte_order(rast_size_t n, int is_native)
{
    return is_native ? n : rast_swap32(n);
}

/**
 * Set the function to be called when an allocation failure occurs.
 */
void rast_set_abort_func(apr_abortfunc_t func);

/**
 * Create a pool which calls the function specified by rast_set_abort_func()
 * when an allocation failure occurs.
 * @param parent The parent pool
 * @return The created pool
 */
apr_pool_t *rast_pool_create(apr_pool_t *parent);

/**
 * Setup any Rast internal data structures.  This MUST be the first function
 * called for any Rast library.
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_initialize();

/**
 * Tear down any Rast internal data structures.
 */
void rast_finalize();

/**
 * Get global pool.
 * @return global pool if initialized, NULL otherwise
 */
apr_pool_t *rast_get_global_pool();

/**
 * Create options for search.
 * @param pool The pool to allocate the memory out of
 * @return The created options
 */
rast_search_option_t *rast_search_option_create(apr_pool_t *pool);

/** @} */

RAST_EXTERN_C_END

#endif /* RAST_H */

/* vim: set filetype=c sw=4 expandtab : */
