/*
** PHREL
** $Id: capture.c,v 1.24 2006/04/06 02:24:53 sella Exp $
** Copyright (c) 2005,2006 James M. Sella. All Rights Reserved.
** Released under the GPL Version 2 License.
** http://www.digitalgenesis.com
*/

static const char rcsid[] = "$Id: capture.c,v 1.24 2006/04/06 02:24:53 sella Exp $";

#include "capture.h"

#include "util.h"
#include "phrel.h"
#include "prefs.h"
#include "data_hash.h"

#include <pcap.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <pthread.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>

#ifndef __USE_BSD
	#define __USE_BSD
#endif
#include <netinet/ip.h>
#ifndef __FAVOR_BSD
	#define __FAVOR_BSD
#endif
#include <netinet/udp.h>
#include <netinet/tcp.h>
#include <netinet/if_ether.h>

#include <errno.h>
#include <assert.h>

#ifndef PCAP_ERRBUF_SIZE
	#error "You must have libpcap v0.8 or better installed."
#endif

/* Global variables */
extern struct args_t args;
extern int running, restart;
extern pthread_mutex_t data_mutex;
extern int errno;

int capture() {
	int c, fd, val, ret_val = 1;
	char *dev = NULL; 
	char errbuf[PCAP_ERRBUF_SIZE];
	struct bpf_program fp;      /* hold compiled program     */
	bpf_u_int32 netp = 1;       /* ip  */
	pcap_t* pcap = NULL;
	struct pcap_pkthdr *pkt_header = NULL;
	const u_char *pkt_data = NULL;
	fd_set fds;
	struct timeval timeout;
     
	if (args.interface == NULL) {
		if ((dev = pcap_lookupdev(errbuf)) == NULL) {
			syslog(LOG_INFO, "%s", errbuf);

			goto error;
		}
	} else {
		dev = args.interface;
	}

	if ((pcap = pcap_open_live(dev, BUFSIZ, args.promisc, -1, errbuf)) == NULL) {
		syslog(LOG_INFO, "pcap_open_live(): %s", errbuf);

		goto error;
	}

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "setting up filter: %s", (args.expression == NULL) ? "none" : args.expression);
	}

	/* Compile the pcap expression (with optimization) */
	if (pcap_compile(pcap, &fp, args.expression, 1, netp) == -1) {
		syslog(LOG_INFO, "invalid expression: %s", args.expression);

		goto error;
	}

	if (pcap_setfilter(pcap, &fp) == -1) {
		syslog(LOG_INFO, "error setting filter");

		goto error;
	}

	pcap_freecode(&fp); /* Release our filter program */

	if ((fd = pcap_fileno(pcap)) < 0) {
		syslog(LOG_INFO, "error getting pcap fileno");

		goto error;
	}

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "packet capture starting");
	}

	/* Set successful return value. */
	ret_val = 0;

	/* Initialize the file descriptor set. */
	FD_ZERO(&fds);
	FD_SET(fd, &fds);

	/* Capture packets and run callback() for each. */
	while (running) {
		/* Reset timeout. */
		timeout.tv_sec = SELECT_TIMEOUT_SEC;
		timeout.tv_usec = 0;

		if ((c = select(fd + 1, &fds, NULL, NULL, NULL)) > 0) {
			if (FD_ISSET(fd, &fds)) {
				if ((val = pcap_next_ex(pcap, &pkt_header, &pkt_data)) >= 1) {
					callback(pkt_header, pkt_data);
				} else if (val == -1) {
					running = 0;
					ret_val = 1; /* Failure */

					syslog(LOG_INFO, "pcap_next_ex() returned in error - shutting down");
				} else {
					syslog(LOG_INFO, "pcap_next_ex() returned with 0");
				}
			}
		} else if (c == -1) {
			running = 0;

			syslog(LOG_INFO, "select() - %s", strerror(errno));
		} else {
			if (args.debug >= DEBUG_DUMP1) {
				syslog(LOG_INFO, "select() - timeout (%u sec)", SELECT_TIMEOUT_SEC);
			}
		}
	}

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "packet capture ended");
	}

error:
	if (pcap) {
		pcap_close(pcap);
		pcap = NULL;
	}

	return ret_val;
}

void callback(const struct pcap_pkthdr* pkthdr, const u_char* packet) {
	struct data_t d, *d_ptr;
	int index;
	time_t cur;
	char buf[32];

	const struct ether_header *ethernet; /* The ethernet header */
	const struct ip *ip; /* The IP header */

	int size_ethernet = sizeof(struct ether_header);

	ethernet = (struct ether_header*)(packet);
	ip = (struct ip*)(packet + size_ethernet);

	cur = time(NULL);
	index = cur % args.max_samples;

	pthread_mutex_lock(&data_mutex);

	if ((d_ptr = hash_find((struct in_addr*) &(ip->ip_src))) == NULL) {
		/* Check Filters - Do we track this sources? */
		if (check_filter(ip) != 0) {
			pthread_mutex_unlock(&data_mutex);

			return; /* skip source */
		}

		/* Build new data_t structure */
		d_ptr = &d;
		memset(d_ptr, 0, sizeof(struct data_t));
		d_ptr->ip = ip->ip_src;

		/* Dynamically allocate samples */
		d_ptr->samples = malloc((args.interval + 1) * sizeof(unsigned int));
		memset(d_ptr->samples, 0, (args.interval + 1) * sizeof(unsigned int));
		d_ptr->samples[index] = 1;

		/* Dynamically allocate last_update */
		d_ptr->last_update = malloc((args.interval + 1) * sizeof(time_t));
		memset(d_ptr->last_update, 0, (args.interval + 1) * sizeof(time_t));
		d_ptr->last_update[index] = cur;

		if (args.debug >= DEBUG_DUMP1) {
			inet_ntop(AF_INET, &(ip->ip_src), buf, sizeof(buf));
			syslog(LOG_INFO, "adding %s to hash", buf);
		}

		hash_add(&d_ptr->ip, d_ptr);
	} else {
		if (d_ptr->last_update[index] == cur) {
			d_ptr->samples[index]++;
		} else {
			d_ptr->samples[index] = 1;
			d_ptr->last_update[index] = cur;
		}

	}

	if (args.debug >= DEBUG_DUMP2) {
		inet_ntop(AF_INET, &(ip->ip_src), buf, sizeof(buf));
		syslog(LOG_INFO, "%16s sample[%d] = %d", buf, index, d_ptr->samples[index]);
	}

	pthread_mutex_unlock(&data_mutex);
}

int check_filter(const struct ip *ip) {
	int include;
	char buf[32], buf2[32];
	struct prefix_t *p;

	/* Ignore packets from exclude list. */
	p = args.exclude;
	while (p != NULL) {
		if (p->ip.s_addr == (ip->ip_src.s_addr & p->mask.s_addr)) {
			if (args.debug >= DEBUG_DUMP2) {
				inet_ntop(AF_INET, &(ip->ip_src), buf, sizeof(buf));
				inet_ntop(AF_INET, &(p->ip), buf2, sizeof(buf2));
				syslog(LOG_INFO, "excluding packet from %s%s (matches: %s/%u)", buf, p->interface ? "*" : "", buf2, p->cidr);
			}

			return 1; /* Skip - Within exclude prefix */
		}
		p = p->next;
	}

	/* Accept packets within include list */
	if (args.include == NULL) { /* No include list, include all (0.0.0.0/0). */
		if (args.debug >= DEBUG_DUMP2) {
			inet_ntop(AF_INET, &(ip->ip_src), buf, sizeof(buf));
			syslog(LOG_INFO, "including packet from %s (matches: 0.0.0.0/0)", buf);
		}

		return 0; /* Accept - Within 0.0.0.0/0 */
	} else { /* Include specified prefixes */
		include = 0;
		p = args.include;
		while (p != NULL) {
			if (p->ip.s_addr == (ip->ip_src.s_addr & p->mask.s_addr)) {
				if (args.debug >= DEBUG_DUMP2) {
					inet_ntop(AF_INET, &(ip->ip_src), buf, sizeof(buf));
					inet_ntop(AF_INET, &(p->ip), buf2, sizeof(buf2));
					syslog(LOG_INFO, "including packet from %s (matches: %s/%u)", buf, buf2, p->cidr);
				}

				include = 1;
				break;
			}
			p = p->next;
		}

		if (include == 1) {
			return 0; /* Accept - Within include prefix */
		}
	}

	if (args.debug >= DEBUG_DUMP2) {
		inet_ntop(AF_INET, &(ip->ip_src), buf, sizeof(buf));
		syslog(LOG_INFO, "excluding packet from %s (non-included prefix)", buf);
	}

	return 1; /* Default exclude */
}

/*
** Local Variables:
** c-basic-offset: 3
** tab-width: 3
** End:
** vim: noet ts=3 sw=3
*/
