/**
 * $Id: mod_estraier_cache.c,v 1.4 2006/04/22 19:28:48 shinh Exp $
 *
 * Copyright (C) shinichiro.h <hamaji@nii.ac.jp>
 *  http://shinh.skr.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
 * The present copyright holders of this program have given permission,
 * as a special exception, to link this program with Apache Portable
 * Runtime (APR) and to include header files for APR components when
 * those header files are covered by the Apache licenses, as long as
 * the GNU GPL is followed for this program in all other ways. 
 */

#include <estraier.h>
#include <estnode.h>
#include <cabin.h>
#include <stdlib.h>

#include <assert.h>

/* apache */
#include <http_protocol.h>
#include <http_config.h>
#include <http_log.h>
#include <apr_strings.h>

module AP_MODULE_DECLARE_DATA estraier_cache_module;

typedef struct {
  char *node;
  char *user;
  char *pass;

  char *proxy_host;
  int proxy_port;
  int timeout;
} estcache_dir_config;

typedef struct {
  char *phrase;
  int max;
  int page;
} estcache_args;

#define PTR_OR(a, b) (((a) != NULL) ? (a) : (b))
#define INT_OR(a, b) (((a) != -1) ? (a) : (b))

/* for debugging */
static void printfinfo(const char *format, ...){
  FILE* fp = fopen("/tmp/log", "a");
  va_list ap;
  va_start(ap, format);
  fprintf(fp, "%s: INFO: ", "mod_estraier_cache");
  vfprintf(fp, format, ap);
  fputc('\n', fp);
  fflush(fp);
  va_end(ap);
  fclose(fp);
}

static int check_estcache_config(request_rec *r, estcache_dir_config *conf) {
  if (conf->node == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "EstcacheNode is not set");
    return TRUE;
  }
  if (conf->user == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "EstcacheUser is not set");
    return TRUE;
  }
  if (conf->pass == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "EstcachePass is not set");
    return TRUE;
  }

  return FALSE;
}

static char *get_full_uri(request_rec *r, apr_pool_t *pool) {
  char *uri;

  uri = r->uri;

  /* add hostname */
  if (!strstr(uri, "://")) {
    uri = apr_pstrcat(pool, "http://", r->hostname, uri, NULL);
  }

  /* add query string */
  if (r->args && !strstr(uri, r->args)) {
    uri = apr_pstrcat(pool, uri, "?", r->args, NULL);
  }

  return uri;
}

static void get_link_pos(const CBLIST *l, int i, int *ll, int *ls) {
  if (i >= cblistnum(l)) {
    *ll = -1;
  }
  else {
    const char *s;
    char *p;
    char buf[32];
    s = cblistval(l, i, NULL);
    strcpy(buf, s);
    p = strchr(buf, ',');
    *p = '\0';
    p++;
    *ll = atoi(buf);
    *ls = atoi(p);
  }
}

static int estcache_handler(request_rec *r) {
  apr_bucket_brigade *bb;
  apr_bucket *b;
  conn_rec *c = r->connection;
  estcache_dir_config *conf;

  ESTNODE *node;
  ESTDOC *doc;

  if (strcmp(r->handler, "estraier_cache")) {
    return DECLINED;
  }

  /* get directory config */
  conf = (estcache_dir_config *)ap_get_module_config(
    r->per_dir_config, &estraier_cache_module);

  /* check conf */
  if (check_estcache_config(r, conf)) {
    return DECLINED;
  }

  ap_set_content_type(r, "text/html; charset=UTF-8");

  bb = apr_brigade_create(r->pool, c->bucket_alloc);

  /* get document */
  node = est_node_new(conf->node);
  if (conf->proxy_host) {
    est_node_set_proxy(node, conf->proxy_host, conf->proxy_port);
  }
  est_node_set_timeout(node, conf->timeout);
  est_node_set_auth(node, conf->user, conf->pass);

  doc = est_node_get_doc_by_uri(node, get_full_uri(r, r->pool));

  /* display document */
  if (!doc) {
    apr_brigade_printf(bb, NULL, NULL, "cached document not found");
    goto display_end;
  }

  {
    const CBLIST *texts;
    CBLIST *link, *linkpos;
    const char *linkstr;
    int i, li, ll, ls;

    texts = est_doc_texts(doc);
    linkstr = est_doc_attr(doc, "linkstr");
    if (!linkstr) {
      apr_brigade_printf(bb, NULL, NULL, "cached document is old format");
      goto display_end;
    }
    link = cbsplit(est_doc_attr(doc, "link"), -1, " ");
    linkpos = cbsplit(est_doc_attr(doc, "linkpos"), -1, " ");

    li = 0;
    get_link_pos(linkpos, li, &ll, &ls);

    for (i = 0; i < cblistnum(texts); i++) {
      const char *t = cblistval(texts, i, NULL);
      apr_brigade_printf(bb, NULL, NULL, "%s\n", t);

      if (i == ll) {
        apr_brigade_printf(bb, NULL, NULL, " => ");
        while (i == ll) {
          apr_brigade_printf(bb, NULL, NULL, "(<a href=\"%s\">%s</a>) ",
                             cblistval(link, li, NULL),
                             apr_pstrndup(r->pool, linkstr, ls));

          linkstr += ls + 1;

          get_link_pos(linkpos, ++li, &ll, &ls);
        }
      }

      apr_brigade_printf(bb, NULL, NULL, "<br>\n");
    }

    cblistclose(link);
    cblistclose(linkpos);
    est_doc_delete(doc);
  }

display_end:

  est_node_delete(node);

  b = apr_bucket_eos_create(c->bucket_alloc);
  APR_BRIGADE_INSERT_TAIL(bb, b);

  ap_pass_brigade(r->output_filters, bb);

  return OK;
}

static apr_status_t cleanup_estcache(void *p) {
  est_free_net_env();
  return OK;
}

static int estcache_post_config(apr_pool_t* p, apr_pool_t* p1, apr_pool_t* p2,
                                 server_rec* s) {
  if (!est_init_net_env()) {
    ap_log_error(APLOG_MARK, APLOG_ERR, 0, s, "est_init_net_env failed");
    return !OK;
  }
  apr_pool_cleanup_register(p, NULL, cleanup_estcache, apr_pool_cleanup_null);

  return OK;
}

static void register_estcache_hooks(apr_pool_t* p) {
  ap_hook_handler(estcache_handler, NULL, NULL, APR_HOOK_MIDDLE);
  ap_hook_post_config(estcache_post_config, NULL, NULL, APR_HOOK_MIDDLE);
}

static void *create_estcache_dir_config(apr_pool_t *p, char *path) {
  estcache_dir_config *conf;

  conf = (estcache_dir_config *)apr_pcalloc(p, sizeof(estcache_dir_config));

  conf->node = NULL;
  conf->user = NULL;
  conf->pass = NULL;

  conf->proxy_host = NULL;
  conf->proxy_port = 80;
  conf->timeout = 5;

  return (void *)conf;
}

static void *merge_estcache_dir_configs(apr_pool_t *p, void *bp, void *ap) {
  estcache_dir_config *base = (estcache_dir_config *)bp;
  estcache_dir_config *add = (estcache_dir_config *)ap;
  estcache_dir_config *conf;

  conf = (estcache_dir_config *)apr_pcalloc(p, sizeof(estcache_dir_config));

  conf->node = PTR_OR(add->node, base->node);
  conf->user = PTR_OR(add->user, base->user);
  conf->pass = PTR_OR(add->pass, base->pass);

  conf->proxy_host = PTR_OR(add->proxy_host, base->proxy_host);
  conf->proxy_port = INT_OR(add->proxy_port, base->proxy_port);
  conf->timeout = INT_OR(add->timeout, base->timeout);

  return conf;
}

static const char *set_estcache_node(cmd_parms *cmd,
                                      void *tmp, const char *arg)
{
  estcache_dir_config *conf = (estcache_dir_config *)tmp;
  conf->node = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estcache_user(cmd_parms *cmd,
                                      void *tmp, const char *arg)
{
  estcache_dir_config *conf = (estcache_dir_config *)tmp;
  conf->user = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estcache_pass(cmd_parms *cmd,
                                      void *tmp, const char *arg)
{
  estcache_dir_config *conf = (estcache_dir_config *)tmp;
  conf->pass = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estcache_proxy_host(cmd_parms *cmd,
                                            void *tmp, const char *arg)
{
  estcache_dir_config *conf = (estcache_dir_config *)tmp;
  conf->proxy_host = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estcache_proxy_port(cmd_parms *cmd,
                                            void *tmp, const char *arg)
{
  int port = atoi(arg);
  if (port < 1 || port > 65535) {
    return "EstcachePort must be valid port";
  }

  estcache_dir_config *conf = (estcache_dir_config *)tmp;
  conf->proxy_port = port;

  return NULL;
}

static const char *set_estcache_timeout(cmd_parms *cmd,
                                         void *tmp, const char *arg)
{
  int timeout = atoi(arg);
  if (timeout < 1 || timeout > 30) {
    return "EstcacheTimeout must be between 1 and 30";
  }

  estcache_dir_config *conf = (estcache_dir_config *)tmp;
  conf->timeout = timeout;

  return NULL;
}

static const command_rec estcache_cmds[] = {
  AP_INIT_TAKE1("EstcacheNode", set_estcache_node, NULL, OR_OPTIONS,
                "url of node"),
  AP_INIT_TAKE1("EstcacheUser", set_estcache_user, NULL, OR_OPTIONS,
                "user of node"),
  AP_INIT_TAKE1("EstcachePass", set_estcache_pass, NULL, OR_OPTIONS,
                "password of node"),
  AP_INIT_TAKE1("EstcacheProxyHost", set_estcache_proxy_host, NULL,
                OR_OPTIONS, "proxy host for node"),
  AP_INIT_TAKE1("EstcacheProxyPort", set_estcache_proxy_port, NULL,
                OR_OPTIONS, "proxy port for node"),
  AP_INIT_TAKE1("EstcacheTimeout", set_estcache_timeout, NULL, OR_OPTIONS,
                "timeout seconds, integer number"),
  {NULL}
};

module AP_MODULE_DECLARE_DATA estraier_cache_module = {
  STANDARD20_MODULE_STUFF,
  create_estcache_dir_config,
  merge_estcache_dir_configs,
  NULL,
  NULL,
  estcache_cmds,
  register_estcache_hooks
};
