/* Lyrics Plasmoid.
 * Copyright (C) 2010  Mauro E. Bender <maurobender@gmail.com>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "lyrics_getter.h"

#include <QDebug>
#include <QFile>
#include <QTextStream>

#include <QMessageBox>
#include <QRegExp>

//KDE Localization
#include <KLocale>

LyricsGetter::LyricsGetter(QObject *parent, LyricSource *source) : QObject(parent)
{
	m_networkAccess = new QNetworkAccessManager(this);
	m_source = source;
	
	connect(m_networkAccess, SIGNAL(finished(QNetworkReply *)),
			  this, SLOT(replyFinished(QNetworkReply *)));
}

void LyricsGetter::setLyricSource(LyricSource *source) {
	qDebug() << ":::LyricsGetter::: Setting source '" << source->name << "'";
	m_source = source;
}

LyricSource* LyricsGetter::currentLyricSource() {
	return m_source;
}

void LyricsGetter::getLyric(const Song &song) {
	getLyric(song.artist, song.title);
}

void LyricsGetter::getLyric(QString artist, QString title) {
	qDebug() << ":::LyricsGetter::: Getting lyrics for Song(Artist: " << artist << ", Title: " << title << ")";
	// If there is not source to fecth the lyrics from, don't do anything.
	if(m_source == 0)
		return;
	
	QString url = m_source->url;
	artist = prepareString(artist, m_source->url_replace, m_source->start_case);
	title = prepareString(title, m_source->url_replace, m_source->start_case);
	
	url.replace("{artist}", artist);
	url.replace("{title}", title);
	
	//qDebug() << "* Url builded: " << url << endl;

	m_networkAccess->get(QNetworkRequest(QUrl(url)));
}

void LyricsGetter::replyFinished(QNetworkReply *reply) {
	QString lyrics;
	
	QString page = reply->readAll();
	if( page.isNull() || page.isEmpty() ) {
		qDebug() << ":::LyricsGetter::: Lyrics fetch complete. [Lyrics not found (Empty response)].";
		emit success(false, "<p align=\"center\">Lyrics not found.</p>");
		return;
	}
	
	QPair <QString, QString> extract_pattern;
	foreach(extract_pattern, m_source->extract) {
		// Generamos la expresin regular que se usar
		// para encontrar las letras dentro de la pgina.
		QRegExp re(extract_pattern.first + "(.*)" + extract_pattern.second);
		//QRegExp re("<div class='lyricbox'>(.*)<!--");
		//qDebug() <<  "RegExp: " << re.pattern() << endl;
		
		// Buscamos las letras
		re.indexIn(page, 0);
		lyrics += re.cap(1);
	}
	
	QPair <QString, QString> exclude_pattern;
	foreach(exclude_pattern, m_source->exclude) {
		// Generamos la expresin regular que se usar
		// para encontrar las letras dentro de la pgina.
		QRegExp re(exclude_pattern.first + ".*" + exclude_pattern.second);
		
		//Removemos los comentarios
		lyrics.remove(re);
	}
	
	if( lyrics.isEmpty() ) {
		emit success(false, i18n("Lyrics not found."));
		qDebug() << ":::LyricsGetter::: Lyrics fetch complete. [Can't extract the lyrics from the page].";
		return;
	}
	
	emit success(true, lyrics);
	qDebug() << ":::LyricsGetter::: Lyrics fetch complete. [Lyrics found].";
}

//------------------ Utility Functions -------------------//
QString LyricsGetter::prepareString(QString string, QList < QPair <QString, QString> > &rules, bool start_case) {
	QString res;
	
	if(start_case) {
		QStringList res_parts = string.split(' ', QString::SkipEmptyParts);
		for(int i = 0; i < res_parts.size(); i++) {
		res_parts[i][0] = res_parts[i][0].toUpper();
		}
		res = res_parts.join(" ");
	} else {
		res = string;
	}
	
	QPair <QString, QString> url_replacement;
	foreach(url_replacement, rules) {
		res.replace(url_replacement.first, url_replacement.second);
	}

	return res;
}
