/*
 * device_conf.h: device XML handling entry points
 *
 * Copyright (C) 2006-2012, 2014-2016 Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Author: Daniel P. Berrange <berrange@redhat.com>
 */

#ifndef __DEVICE_CONF_H__
# define __DEVICE_CONF_H__

# include <libxml/parser.h>
# include <libxml/tree.h>
# include <libxml/xpath.h>

# include "internal.h"
# include "virutil.h"
# include "virthread.h"
# include "virbuffer.h"
# include "virpci.h"

typedef enum {
    VIR_INTERFACE_STATE_UNKNOWN = 1,
    VIR_INTERFACE_STATE_NOT_PRESENT,
    VIR_INTERFACE_STATE_DOWN,
    VIR_INTERFACE_STATE_LOWER_LAYER_DOWN,
    VIR_INTERFACE_STATE_TESTING,
    VIR_INTERFACE_STATE_DORMANT,
    VIR_INTERFACE_STATE_UP,
    VIR_INTERFACE_STATE_LAST
} virInterfaceState;

VIR_ENUM_DECL(virInterfaceState)

typedef struct _virInterfaceLink virInterfaceLink;
typedef virInterfaceLink *virInterfaceLinkPtr;
struct _virInterfaceLink {
    virInterfaceState state; /* link state */
    unsigned int speed;      /* link speed in Mbits per second */
};

typedef enum {
    VIR_NET_DEV_FEAT_GRXCSUM,
    VIR_NET_DEV_FEAT_GTXCSUM,
    VIR_NET_DEV_FEAT_GSG,
    VIR_NET_DEV_FEAT_GTSO,
    VIR_NET_DEV_FEAT_GGSO,
    VIR_NET_DEV_FEAT_GGRO,
    VIR_NET_DEV_FEAT_LRO,
    VIR_NET_DEV_FEAT_RXVLAN,
    VIR_NET_DEV_FEAT_TXVLAN,
    VIR_NET_DEV_FEAT_NTUPLE,
    VIR_NET_DEV_FEAT_RXHASH,
    VIR_NET_DEV_FEAT_RDMA,
    VIR_NET_DEV_FEAT_TXUDPTNL,
    VIR_NET_DEV_FEAT_LAST
} virNetDevFeature;

VIR_ENUM_DECL(virNetDevFeature)

typedef enum {
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_NONE,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_PCI,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_DRIVE,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_VIRTIO_SERIAL,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_CCID,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_USB,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_SPAPRVIO,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_VIRTIO_S390,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_CCW,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_VIRTIO_MMIO,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_ISA,
    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_DIMM,

    VIR_DOMAIN_DEVICE_ADDRESS_TYPE_LAST
} virDomainDeviceAddressType;

typedef struct _virDomainDeviceDriveAddress {
    unsigned int controller;
    unsigned int bus;
    unsigned int target;
    unsigned int unit;
} virDomainDeviceDriveAddress, *virDomainDeviceDriveAddressPtr;

typedef struct _virDomainDeviceVirtioSerialAddress {
    unsigned int controller;
    unsigned int bus;
    unsigned int port;
} virDomainDeviceVirtioSerialAddress, *virDomainDeviceVirtioSerialAddressPtr;

# define VIR_DOMAIN_DEVICE_CCW_MAX_CSSID    254
# define VIR_DOMAIN_DEVICE_CCW_MAX_SSID       3
# define VIR_DOMAIN_DEVICE_CCW_MAX_DEVNO  65535

typedef struct _virDomainDeviceCCWAddress {
    unsigned int cssid;
    unsigned int ssid;
    unsigned int devno;
    bool         assigned;
} virDomainDeviceCCWAddress, *virDomainDeviceCCWAddressPtr;

typedef struct _virDomainDeviceCcidAddress {
    unsigned int controller;
    unsigned int slot;
} virDomainDeviceCcidAddress, *virDomainDeviceCcidAddressPtr;

typedef struct _virDomainDeviceUSBAddress {
    unsigned int bus;
    char *port;
} virDomainDeviceUSBAddress, *virDomainDeviceUSBAddressPtr;

typedef struct _virDomainDeviceSpaprVioAddress {
    unsigned long long reg;
    bool has_reg;
} virDomainDeviceSpaprVioAddress, *virDomainDeviceSpaprVioAddressPtr;

typedef enum {
    VIR_DOMAIN_CONTROLLER_MASTER_NONE,
    VIR_DOMAIN_CONTROLLER_MASTER_USB,

    VIR_DOMAIN_CONTROLLER_MASTER_LAST
} virDomainControllerMaster;

typedef struct _virDomainDeviceUSBMaster {
    unsigned int startport;
} virDomainDeviceUSBMaster, *virDomainDeviceUSBMasterPtr;

typedef struct _virDomainDeviceISAAddress {
    unsigned int iobase;
    unsigned int irq;
} virDomainDeviceISAAddress, *virDomainDeviceISAAddressPtr;

typedef struct _virDomainDeviceDimmAddress {
    unsigned int slot;
    unsigned long long base;
} virDomainDeviceDimmAddress, *virDomainDeviceDimmAddressPtr;

typedef struct _virDomainDeviceInfo {
    /* If adding to this struct, ensure that
     * virDomainDeviceInfoIsSet() is updated
     * to consider the new fields
     */
    char *alias;
    int type; /* virDomainDeviceAddressType */
    union {
        virPCIDeviceAddress pci;
        virDomainDeviceDriveAddress drive;
        virDomainDeviceVirtioSerialAddress vioserial;
        virDomainDeviceCcidAddress ccid;
        virDomainDeviceUSBAddress usb;
        virDomainDeviceSpaprVioAddress spaprvio;
        virDomainDeviceCCWAddress ccw;
        virDomainDeviceISAAddress isa;
        virDomainDeviceDimmAddress dimm;
    } addr;
    int mastertype;
    union {
        virDomainDeviceUSBMaster usb;
    } master;
    /* rombar and romfile are only used for pci hostdev and network
     * devices. */
    int rombar;         /* enum virTristateSwitch */
    char *romfile;
    /* bootIndex is only used for disk, network interface, hostdev
     * and redirdev devices */
    unsigned int bootIndex;
} virDomainDeviceInfo, *virDomainDeviceInfoPtr;


int virPCIDeviceAddressIsValid(virPCIDeviceAddressPtr addr,
                               bool report);

static inline bool
virPCIDeviceAddressIsEmpty(const virPCIDeviceAddress *addr)
{
    return !(addr->domain || addr->bus || addr->slot);
}

static inline bool
virDeviceInfoPCIAddressWanted(const virDomainDeviceInfo *info)
{
    return info->type == VIR_DOMAIN_DEVICE_ADDRESS_TYPE_NONE ||
        (info->type == VIR_DOMAIN_DEVICE_ADDRESS_TYPE_PCI &&
         virPCIDeviceAddressIsEmpty(&info->addr.pci));
}

static inline bool
virDeviceInfoPCIAddressPresent(const virDomainDeviceInfo *info)
{
    return info->type == VIR_DOMAIN_DEVICE_ADDRESS_TYPE_PCI &&
       !virPCIDeviceAddressIsEmpty(&info->addr.pci);
}

int virPCIDeviceAddressParseXML(xmlNodePtr node,
                                virPCIDeviceAddressPtr addr);

int virPCIDeviceAddressFormat(virBufferPtr buf,
                              virPCIDeviceAddress addr,
                              bool includeTypeInAddr);

bool virPCIDeviceAddressEqual(virPCIDeviceAddress *addr1,
                              virPCIDeviceAddress *addr2);

int virInterfaceLinkParseXML(xmlNodePtr node,
                             virInterfaceLinkPtr lnk);

int virInterfaceLinkFormat(virBufferPtr buf,
                           const virInterfaceLink *lnk);

#endif /* __DEVICE_CONF_H__ */
