/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qdragobject.h>
#include <kprinter.h>
#include <qmsgbox.h>
#include <qpopupmenu.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <kfiledialog.h>
#include <qdir.h>

#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdeversion.h>
#include <kstatusbar.h>
#include <ktoolbar.h>
#include <kmenubar.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <kurldrag.h>
#include <kurlrequesterdlg.h>
#include <kshortcut.h>

#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>

#include "KYumPreferences.h"
#include "DlgEditRepo.h"
#include "DlgGenerateRSS.h"
#include "DlgInfo.h"
#include "KYumConf.h"
#include "RepoView.h"
#include "ActionsView.h"
#include "GroupsView.h"
#include "kyum.h"

enum
  {
    c_ID_ABORT = 500
  };


/***************************************************************************/
/**
 *  Default constructor
 *
 ***************************************************************************/

 KYum::KYum()

  : KMainWindow         (0, "KYum"),
    m_strCurrentProfile ("/etc/kyum-current.kypro"),
    m_pPrinter          (0)

{
    KYumPreferences::readConfig(kapp->config());
    m_pView = new KYumView(this);

    setAcceptDrops(true);
    setCentralWidget(m_pView);

    setupActions();
    statusBar()->show();

    //
    // Create the main window and ask the mainwindow to automatically save
    // settings if changed: window size, toolbar position, icon size, etc.
    // Also to add actions for the statusbar, toolbar, and keybindings if
    // necessary.
    //
    setupGUI();

    connect(m_pView->getActionsView(),
            SIGNAL(actionStarted(Process *)),
            this,
            SLOT(slotActionStarted(Process *)));

    connect(m_pView->getGroupsView(),
            SIGNAL(actionStarted(Process *)),
            this,
            SLOT(slotActionStarted(Process *)));

    connect(m_pView->getActionsView(),
            SIGNAL(actionTerminated(Process *)),
            this,
            SLOT(slotActionTerminated(Process *)));

    connect(m_pView->getGroupsView(),
            SIGNAL(actionTerminated(Process *)),
            this,
            SLOT(slotActionTerminated(Process *)));

    //
    // allow the view to change the statusbar and caption
    //
    connect(m_pView, SIGNAL(signalChangeStatusbar(const QString&)),
            this,   SLOT(changeStatusbar(const QString&)));

    connect(m_pView, SIGNAL(signalChangeCaption(const QString&)),
            this,   SLOT(changeCaption(const QString&)));

    connect(m_pView, SIGNAL(signalContextAddRepo()),
            this,    SLOT(fileNew()));
}


/***************************************************************************/
/**
 *  Default destructor
 *
 ***************************************************************************/

KYum::~KYum()
{
    saveCurrentProfile();
    KYumPreferences::writeConfig(kapp->config());
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYum::setupActions()
{
    KStdAction::openNew(this, SLOT(fileNew()), actionCollection());
    KStdAction::save(this, SLOT(fileSave()), actionCollection());
    KStdAction::quit(kapp, SLOT(quit()), actionCollection());
    KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

    m_pActionAbort = new KAction(i18n("Abort running process"),
                                 "abort",
                                 KShortcut("Ctrl+Shift+B"),
                                 this,
                                 SLOT(slotAbort()),
                                 actionCollection(),
                                 "Abort");
/*
    m_pActionLocalInstall = new KAction(i18n("Install local rpm files"),
                                 "localinst",
                                 KShortcut("Ctrl+Shift+I"),
                                 this,
                                 SLOT(slotLocalInstall()),
                                 actionCollection(),
                                 "LocalInst");
*/
    m_pActionClean = new KAction(i18n("Clean yum's cache"),
                                 "eraser",
                                 KShortcut("Ctrl+Shift+C"),
                                 this,
                                 SLOT(slotCleanCache()),
                                 actionCollection(),
                                 "Clean");

    m_pActionGenerateRSS = new KAction(i18n("Generate RSS"),
                                 "generate_rss",
                                 KShortcut("Ctrl+Shift+R"),
                                 this,
                                 SLOT(slotGenerateRSS()),
                                 actionCollection(),
                                 "GenerateRSS");

    m_pActionLoadProfile = new KAction(i18n("Load Profile"),
                                 "load_profile",
                                 KShortcut("Ctrl+Shift+L"),
                                 this,
                                 SLOT(slotLoadProfile()),
                                 actionCollection(),
                                 "file_load_profile");

    m_pActionSaveProfile = new KAction(i18n("Save Profile"),
                                 "save_profile",
                                 KShortcut("Ctrl+Shift+S"),
                                 this,
                                 SLOT(slotSaveProfile()),
                                 actionCollection(),
                                 "file_save_profile");
    //
    // Not process running yet...
    //
    enableAbort(false);
}


/***************************************************************************/
/**
 * the 'config' object points to the session managed
 * config file.  anything you write here will be available
 * later when this app is restored
 *
 ***************************************************************************/

void KYum::saveProperties(KConfig * pConfig)
{
    KYumPreferences::writeConfig(pConfig);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::readProperties(KConfig * pConfig)
{
    KYumPreferences::readConfig(pConfig);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::dragEnterEvent(QDragEnterEvent *event)
{
    // accept uri drops only
    event->accept(KURLDrag::canDecode(event));
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::dropEvent(QDropEvent * /*event*/)
{
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::fileNew()
{
    // this slot is called whenever the File->New menu is selected,
    // the New shortcut is pressed (usually CTRL+N) or the New toolbar
    // button is clicked
    fileAddRepository();
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::fileOpen()
{
    // this slot is called whenever the File->Open menu is selected,
    // the Open shortcut is pressed (usually CTRL+O) or the Open toolbar
    // button is clicked
/*
    // this brings up the generic open dialog
    KURL url = KURLRequesterDlg::getURL(QString::null, this, i18n("Open Location") );
*/
    // standard filedialog
/*
    KURL url = KFileDialog::getOpenURL(QString::null, QString::null, this, i18n("Open Location"));
    if (!url.isEmpty())
        m_pView->openURL(url);
*/
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::fileSave()
{
    if ( m_pView->saveAll() )
    {
        if ( DlgInfo::isMsgEnabled(DlgInfo::c_YumConfAndReposSaved) )
            DlgInfo(this, DlgInfo::c_YumConfAndReposSaved).exec();
    }
    else
    {
        QMessageBox::warning(this,
                             "KYum",
                             i18n("Could not save all repository files."));
    }
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::fileSaveAs()
{
    // this slot is called whenever the File->Save As menu is selected,
    KURL file_url = KFileDialog::getSaveURL();

    if (!file_url.isEmpty() && file_url.isValid())
    {
        // save your info, here
    }
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::filePrint()
{
    // this slot is called whenever the File->Print menu is selected,
    // the Print shortcut is pressed (usually CTRL+P) or the Print toolbar
    // button is clicked
    if (!m_pPrinter) m_pPrinter = new KPrinter;
    if (m_pPrinter->setup(this))
    {
        // setup the printer.  with Qt, you always "print" to a
        // QPainter.. whether the output medium is a pixmap, a screen,
        // or paper
        QPainter p;
        p.begin(m_pPrinter);

        // we let our view do the actual printing
        QPaintDeviceMetrics metrics(m_pPrinter);
        m_pView->print(&p, metrics.height(), metrics.width());

        // and send the result to the printer
        p.end();
    }
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::optionsPreferences()
{
    KYumPreferencesDlg().exec();
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::fileAddRepository()
{
    Repo::Ptr   ptrRepo = new Repo;
    DlgEditRepo dlg(this, "AddRepository", *ptrRepo.getPtr());

    dlg.setCaption(i18n("Add Repository"));

    if ( QDialog::Accepted == dlg.exec() )
    {
        m_pView->addRepo(ptrRepo);
    }
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::changeStatusbar(const QString& text)
{
    // display the text on the statusbar
    statusBar()->message(text);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::changeCaption(const QString& text)
{
    // display the text on the caption
    setCaption(text);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::enableAbort(bool bEnable)
{
    m_pActionAbort->setEnabled(bEnable);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::enableClean(bool bEnable)
{
    m_pActionClean->setEnabled(bEnable);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::enableRSS(bool bEnable)
{
    m_pActionGenerateRSS->setEnabled(bEnable);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotAbort()
{
    //
    // stopAction() won't do any harm if no action, that is controlled
    // by the respective view, is currently running. So we need not
    // care which if our views is actually affected - they will do that
    // themselves.
    //
    m_pView->getActionsView()->stopAction();
    m_pView->getGroupsView()->stopAction();
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotCleanCache()
{
    m_pView->getActionsView()->cleanCache();
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotLocalUpdate()
{
    m_pView->getActionsView()->slotLocalUpdate();
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotLocalInstall()
{
    m_pView->getActionsView()->slotLocalInstall();
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotGenerateRSS()
{
    DlgGenerateRSS dlg(this);

    if ( dlg.exec() == DlgGenerateRSS::Accepted )
    {
    }
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotActionStarted(Process * /*pProcess*/)
{
    enableAbort(true);
    enableClean(false);
    enableRSS(false);

    //
    // Forbid starting other actions while the current one is still running..
    //
    m_pView->getActionsView()->enableActions(false);
    m_pView->getGroupsView()->enableActions(false);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotActionTerminated(Process * /*pProcess*/)
{
    enableAbort(false);
    enableClean(true);
    enableRSS(true);

    //
    // Permit starting actions again...
    //
    m_pView->getActionsView()->enableActions(true);
    m_pView->getGroupsView()->enableActions(true);
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotLoadProfile()
{
    static QString  strDir,
                    strFile;
    QString         strFilter("*.kypro|KYum Profiles (*.kypro)\n*|All Files (*)"),
                    strSelection;

    strSelection = KFileDialog::getOpenFileName(strDir,
                                                strFilter,
                                                this,
                                                i18n("Select profile to load"));

    if ( !strSelection.isEmpty() )
    {
        bool bContinue = true;

/*
        if ( 0 == QMessageBox::question(this,
                                        "KYum",
                                        i18n("Do want to save the current configuration"
                                             "before overwriting it?"),
                                        i18n("Yes"),
                                        i18n("No")) )
        {
            //TODO ...
        }
*/
        strFile = strSelection;
        strDir  = QFileInfo(strFile).filePath();

        if ( bContinue )
        {
            m_pView->getYumConf().loadProfile(strFile);

/*
            This is now done automatically, when m_pView->m_yumConf fires
            reposLoaded().

            m_pView->getRepoView()->redrawList();
*/

            //m_pView->getYumConfView()->configLoaded();
        }
    }
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

void KYum::slotSaveProfile()
{
    static QString  strDir,
                    strFile;
    QString         strFilter("*.kypro|KYum Profiles (*.kypro)\n*|All Files (*)"),
                    strSelection;

    strSelection = KFileDialog::getSaveFileName(strDir,
                                                strFilter,
                                                this,
                                                i18n("Choose a filename to save profile under"));

    if ( !strSelection.isEmpty() )
    {
        strFile = strSelection;
        strDir  = QFileInfo(strFile).filePath();

        //
        // Save yum.conf and repos to the specified profile...
        //
        m_pView->getYumConf().saveProfile(strFile);
    }
}


/***************************************************************************/
/**
 *
 *
 *
 ***************************************************************************/

bool KYum::saveCurrentProfile()
{
    return m_pView->getYumConf().saveProfile(m_strCurrentProfile);
}


#include "kyum.moc"
