/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qapp.h>
#include <qmessagebox.h>
#include <qfile.h>

#include <klocale.h>

#include "UserEvents.h"
#include "StrTok.h"
#include "KYumPreferences.h"
#include "KYumConf.h"
#include "KYumView.h"
#include "RepoView.h"
#include "YumConfView.h"


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

YumConfView::YumConfView(QWidget * pParent)

  : KTextEdit       (pParent, "YumConfView"),
    m_bExitRequested(false),
    m_bDirty        (false),
    m_cfgFileInfo   (KYumPreferences::getYumConfPath())

{
    setReadOnly(true);

    KYumConf & yumConf = KYumView::getKYumView()->getYumConf();

    connect(&yumConf, SIGNAL(mainCfgLoaded()), this, SLOT(configLoaded()));
    connect(&yumConf, SIGNAL(mainCfgSaved()),  this, SLOT(configSaved()));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

YumConfView::~YumConfView()
{
    stopThread();
    wait();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool YumConfView::close()
{
    stopThread();
    wait();

    return KTextEdit::close();
}


/***************************************************************************/
/**
 *  The contents of yum.conf have just been reloaded.
 *
 ***************************************************************************/

void YumConfView::configLoaded()
{
    configChanged(false);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void YumConfView::configSaved()
{
    configChanged(false);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void YumConfView::configChanged(bool bDirty)
{
    QMutexLocker  locker(&m_mutex);
    KYumConf &    yumConf = KYumView::getKYumView()->getYumConf();
    QString       strText = yumConf.toString();

    if ( bDirty )
    {
        setColor(KYumPreferences::c_clrDirty);
        strText.replace("[main]", "[*main]");
    }
    else
        setColor(QColor(0, 0, 0));

    m_bDirty = bDirty;
    setText(strText);
}


/***************************************************************************/
/**
 *  This method is called from the monitoring thread when the file yum.conf
 *  has changed.
 *
 ***************************************************************************/

void YumConfView::cfgFileChanged()
{
    QMutexLocker  locker(&m_mutex);
    QString       strMsg;

    strMsg.sprintf(i18n("The file %s has been changed on your disk. Do you "
                        "want it to be reloaded?"),
                        KYumPreferences::getYumConfPath().ascii());

    if ( 0 == QMessageBox::question(this, "KYum", strMsg, i18n("&Yes"), i18n("&No")) )
    {
        KYumConf  & yumConf  = KYumView::getKYumView()->getYumConf();

        //
        // If yumConf.loadMainCfg() succeeds, it will fire mainCfgLoaded,
        // and configLoaded() will be called.
        //
        if ( !yumConf.loadMainCfg() )
            QMessageBox::warning(this, "KYum", i18n("The file could not be loaded!"));
    }
}


/***************************************************************************/
/**
 *  Starts the file monitoring thread.
 *
 ***************************************************************************/

bool YumConfView::startThread()
{
    bool bSuccess = false;

    if ( !running() )
    {
        m_bExitRequested = false;
        start();
        bSuccess = true;
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void YumConfView::run()
{
    int  count   = 4;
    bool bFirst = true;

    do
    {
        if ( ++count == 5 ) // Every 5 seconds
        {
            if ( KYumPreferences::getMonitorYumConf() )
            {
                QMutexLocker  locker(&m_mutex);
                QFileInfo     fileInfoNew(KYumPreferences::getYumConfPath());

                if ( bFirst ||
                     m_cfgFileInfo.size()         != fileInfoNew.size() ||
                     m_cfgFileInfo.lastModified() != fileInfoNew.lastModified() )
                {
                    QApplication::postEvent(
                          this,
                          new FileChangedEvent(KYumPreferences::getYumConfPath()));

                    m_cfgFileInfo = fileInfoNew;
                    bFirst        = false;
                }

            }

            count = 0;
        }

        sleep(1);
    }
    while ( !exitRequested() );
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool YumConfView::event(QEvent * pEvent)
{
    bool bProcessed = false;

    if ( pEvent->type() == QEvent::Type(c_evt_FileChanged) )
    {
        QMutexLocker locker(&m_mutex);
        cfgFileChanged();
    }
    else
        bProcessed = KTextEdit::event(pEvent);

    return bProcessed;
}

#include "YumConfView.moc"
