/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include "klocale.h"

#include "System.h"
#include "KYumView.h"
#include "OutputView.h"
#include "RepoView.h"

enum
  {
    c_idxTag,
    c_idxName,
    c_idxBaseURL,
    c_idxFile
  };

static struct
  {
    const char  * m_pTitle;
    int           m_cx;
  }
  g_columns[] =
    {
      { /*i18n*/("Tag"),      -1 },
      { /*i18n*/("Name"),     -1 },
      { /*i18n*/("Base URL"), -1 },
      { /*i18n*/("File"),     -1 }
    };


#define ArraySize(a) (sizeof(a) / sizeof((a)[0]))


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

RepoView::RepoView(QWidget * pParent)

  : KListView(pParent, "RepoView")

{
    for ( int idx=0; idx < (int) ArraySize(g_columns); idx++ )
        addColumn(g_columns[idx].m_pTitle, g_columns[idx].m_cx);

    setAllColumnsShowFocus(true);
    setSelectionMode(QListView::Single);

    KYumConf & yumConf = KYumView::getKYumView()->getYumConf();

    connect(&yumConf, SIGNAL(reposLoaded()), this, SLOT(slotReposLoaded()));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoView::redrawList()
{
    KYumView *                      pKYumView = KYumView::getKYumView();
    KYumConf &                      yumConf   = pKYumView->getYumConf();
    const RepoFile::List &          repoFiles = yumConf.getRepoFiles();
    RepoFile::List::const_iterator  itFiles;

    clear();

    for ( itFiles = repoFiles.begin(); itFiles != repoFiles.end(); itFiles++ )
    {
        RepoFile::Ptr               ptrFile  = *itFiles;
        const Repo::List &          repoList = ptrFile->getRepos();
        Repo::List::const_iterator  itRepos;

        for ( itRepos = repoList.begin(); itRepos != repoList.end(); itRepos++ )
        {
            addToList(*itRepos, ptrFile->getFileName());
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoView::slotReposLoaded()
{
    redrawList();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoView::changeItem(QCheckListItem * pItem, Repo::Ptr ptrRepo)
{
    if ( pItem )
    {
        pItem->setText(c_idxTag, ptrRepo->getTag());
        pItem->setText(c_idxName, System::expandString(ptrRepo->getName()));
        pItem->setText(c_idxBaseURL, System::expandString(ptrRepo->getBaseURL().url()));
        pItem->setText(c_idxFile, ptrRepo->getFileName());
        pItem->setOn(ptrRepo->isEnabled());
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoView::addToList(Repo::Ptr ptrRepo, QString strFileName)
{
    if ( !strFileName.isNull() )
        ptrRepo->setFileName(strFileName);

    QCheckListItem * pNewItem = new QCheckListItem(this,
                                                   ptrRepo->getTag(),
                                                   QCheckListItem::CheckBox);

    if ( pNewItem )
        changeItem(pNewItem, ptrRepo);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

Repo::Ptr RepoView::getRepoAtItem(const QListViewItem * pItem) const
{
    return KYumView::getKYumView()->getYumConf().findRepo(pItem->text(c_idxTag));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoView::deleteItem(QListViewItem * pItem)
{
    Repo::Ptr ptrRepo = getRepoAtItem(pItem);

    if ( ptrRepo )
        KYumView::getKYumView()->getYumConf().deleteRepo(ptrRepo);

    delete pItem;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoView::repoUpdated(QString strTag, Repo::Ptr ptrRepo)
{
    QListViewItemIterator it(this);

    while ( it.current() )
    {
        QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

        if ( pItem->text(c_idxTag) == strTag )
        {
            changeItem(pItem, ptrRepo);
            break;
        }

        ++it;
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void RepoView::addRepo(Repo::Ptr ptrRepo, QString strFileName)
{
    if ( KYumView::getKYumView()->getYumConf().addRepo(ptrRepo, strFileName) )
    {
        addToList(ptrRepo, strFileName);
    }
}


#include "RepoView.moc"
