/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "prefs.h"

#include <klocale.h>
#include <kfontdialog.h>
#include <kcolordialog.h>
#include <kpushbutton.h>
#include <ktextedit.h>
#include <kcolorbutton.h>
#include <kurllabel.h>
#include <kmessagebox.h>
#include <kapplication.h>
#include <kiconloader.h>

#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qcheckbox.h>
#include <qspinbox.h>
#include <qbuttongroup.h>
#include <qvbox.h>

#include "libkita/kitaconfig.h"
#include "libkita/signalcollection.h"
#include "libkita/boardmanager.h"
#include "debugprefbase.h"
#include "aboneprefpage.h"
#include "loginprefpage.h"
#include "writeprefpage.h"

using namespace Kita;

KitaPreferences::KitaPreferences()
        : KDialogBase( IconList, i18n( "Kita Preferences" ),
                       Help | Default | Ok | Apply | Cancel, Ok )
{
    enableButtonApply( false );
    enableButton( Help, false );
    // this is the base class for your preferences dialog.  it is now
    // a Treelist dialog.. but there are a number of other
    // possibilities (including Tab, Swallow, and just Plain)

    QVBox * faceFrame = addVBoxPage( i18n( "Face" ), i18n( "Face" ), BarIcon( "view_detailed", KIcon::SizeMedium ) );
    m_facePage = new Kita::FacePrefPage( faceFrame );

    connect( m_facePage, SIGNAL( fontChanged( const QFont& ) ),
             SIGNAL( fontChanged( const QFont& ) ) );

    QVBox * asciiArtFrame = addVBoxPage( i18n( "AsciiArt" ), i18n( "AsciiArt" ), BarIcon( "kita", KIcon::SizeMedium ) );
    m_asciiArtPage = new Kita::AsciiArtPrefPage( asciiArtFrame );

    QVBox * uiFrame = addVBoxPage( i18n( "User Interface" ), i18n( "User Interface" ), BarIcon( "configure", KIcon::SizeMedium ) );
    m_uiPage = new Kita::UIPrefPage( uiFrame );

    QVBox * aboneFrame = addVBoxPage( i18n( "Abone" ), i18n( "Abone" ), BarIcon( "kita", KIcon::SizeMedium ) );
    m_abonePage = new Kita::AbonePrefPage( aboneFrame );

    QVBox * loginFrame = addVBoxPage( i18n( "Login" ), i18n( "Login" ), BarIcon( "connect_established", KIcon::SizeMedium ) );
    m_loginPage = new Kita::LoginPrefPage( loginFrame );

    QVBox * writeFrame = addVBoxPage( i18n( "Write" ), i18n( "Write" ), BarIcon( "edit", KIcon::SizeMedium ) );
    m_writePage = new Kita::WritePrefPage( writeFrame );

    //    QVBox * debugFrame = addVBoxPage( i18n( "Debug" ), i18n( "Debug" ) );
    //    m_debugPage = new DebugPrefPage( debugFrame );

    connect( m_facePage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_asciiArtPage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_uiPage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_abonePage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_loginPage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_writePage, SIGNAL( changed() ), SLOT( slotChanged() ) );

    connect( this, SIGNAL( aboutToShowPage( QWidget* ) ), SLOT( slotAboutToShowPage( QWidget* ) ) );
}

void KitaPreferences::slotApply()
{
    qDebug( "activePageIndex = %d", activePageIndex() );
    switch ( activePageIndex() ) {
        // TODO: refactoring.
    case 0:
        // face
        m_facePage->apply();
        break;
    case 1:
        // asciiart
        m_asciiArtPage->apply();
        break;
    case 2:
        // user interface
        m_uiPage->apply();
        break;
    case 3:
        // abone
        m_abonePage->apply();
        break;
    case 4:
        // login
        m_loginPage->apply();
        break;
    case 5:
        // write
        m_writePage->apply();
        break;
    case 6:
        // debug
        break;
    }
    enableButtonApply( false );
}

void KitaPreferences::slotDefault()
{
    switch ( activePageIndex() ) {
    case 0:
        // face
        m_facePage->reset();
        break;
    case 1:
        // asciiart
        m_asciiArtPage->reset();
        break;
    case 2:
        // user
        m_uiPage->reset();
        break;
    case 3:
        // abone
        break;
    case 4:
        // login
        break;
    case 5:
        // write
        break;
    case 6:
        // debug
        break;
    }
    enableButtonApply( true );
}

void KitaPreferences::slotChanged()
{
    enableButtonApply( true );
}

void KitaPreferences::slotOk()
{
    m_facePage->apply();
    m_asciiArtPage->apply();
    m_uiPage->apply();
    m_abonePage->apply();
    m_loginPage->apply();
    m_writePage->apply();

    KDialogBase::slotOk();
}

void KitaPreferences::slotAboutToShowPage( QWidget* page )
{
    if ( activePageIndex() == 1 ) {
        // ascii art
        m_asciiArtPage->init();
    }
}

DebugPrefPage::DebugPrefPage( QWidget* parent )
        : DebugPrefBase( parent )
{
    connect( replaceButton, SIGNAL( clicked() ), SLOT( replace() ) );
}

void DebugPrefPage::replace()
{
    QString fromURL = fromURLLine->text();
    QString toURL = toURLLine->text();

    if ( toURL.find( fromURL ) == 0 ) {
        QMessageBox::critical( this, i18n( "replace" ),
                               i18n( "'To' URL must not contain 'From' URL." ) );;
        return ;
    }

    if ( QMessageBox::warning( this, i18n( "replace" ),
                               i18n( "This tool may possibly break thread's info.\n"
                                     "Please create backups of '~/.kde/share/apps/kita/cache'\n"
                                     "and '~/.kde/share/apps/kita/favorites.xml' before performing.\n"
                                     "Are you ready?" ), QMessageBox::Ok,
                               QMessageBox::Cancel | QMessageBox::Default | QMessageBox::Escape ) == QMessageBox::Ok ) {

        if ( BoardManager::moveBoard( fromURL, toURL ) )
            QMessageBox::information( this, i18n( "replace" ), i18n( "Complete." ) );
        else KMessageBox::error( this, i18n( "replace" ), i18n( "Failed.." ) );
    }
}

AsciiArtPrefPage::AsciiArtPrefPage( QWidget* parent )
        : AsciiArtPrefBase( parent )
{
    init();

    connect( asciiArtText, SIGNAL( textChanged() ), SIGNAL( changed() ) );
    connect( showAACheckBox, SIGNAL( clicked() ), SIGNAL( changed() ) );
}

void AsciiArtPrefPage::init()
{
    asciiArtText->setText( KitaConfig::asciiArtList().join( "\n" ) );
    asciiArtText->setFont( KitaConfig::threadFont() );
    showAACheckBox->setChecked( KitaConfig::showAA() );
}

void AsciiArtPrefPage::apply()
{
    QString text = asciiArtText->text();
    QStringList list = QStringList::split( '\n', text );

    KitaConfig::setAsciiArtList( list );
    KitaConfig::setShowAA( showAACheckBox->isChecked() );
}

void AsciiArtPrefPage::reset()
{
    asciiArtText->setText( KitaConfig::defaultAsciiArtList().join( "\n" ) );
    showAACheckBox->setChecked( KitaConfig::defaultShowAA() );
}

UIPrefPage::UIPrefPage( QWidget* parent )
        : UIPrefBase( parent )
{
    useTabCheckBox->setChecked( KitaConfig::alwaysUseTab() );
    MarkTimespinBox->setValue( KitaConfig::MarkTime() );
    showMailCheckBox->setChecked( KitaConfig::showMailAddress() );
    orderGroup->setButton( KitaConfig::sortOrder() );
    showNumSpin->setValue( KitaConfig::showNum() );
    useExtPartCheck->setChecked( KitaConfig::usePart() );
    partMimeList->setText( KitaConfig::partMimeList().join( "," ) );

    connect( useTabCheckBox, SIGNAL( clicked() ), SIGNAL( changed() ) );
    connect( MarkTimespinBox, SIGNAL( valueChanged( int ) ), SIGNAL( changed() ) );
    connect( showMailCheckBox, SIGNAL( clicked() ), SIGNAL( changed() ) );
    connect( orderGroup, SIGNAL( clicked( int ) ), SIGNAL( changed() ) );
    connect( showNumSpin, SIGNAL( valueChanged( int ) ), SIGNAL( changed() ) );
#define NCC(ctl, signal) \
    connect( ctl, SIGNAL( signal ), SIGNAL( changed() ) )
    //    NCC( extPartGroup, toggled( bool ) );
    NCC( useExtPartCheck, toggled( bool ) );
    NCC( partMimeList, textChanged( const QString& ) );

    connect( editFileAssociation, SIGNAL( leftClickedURL() ), SLOT( slotEditFileAssociation() ) );
}

void UIPrefPage::apply()
{
    KitaConfig::setAlwaysUseTab( useTabCheckBox->isChecked() );
    KitaConfig::setMarkTime( MarkTimespinBox->value() );
    KitaConfig::setShowMailAddress( showMailCheckBox->isChecked() );
    KitaConfig::setSortOrder( orderGroup->id( orderGroup->selected() ) );
    KitaConfig::setShowNum( showNumSpin->value() );
    KitaConfig::setUsePart( useExtPartCheck->isChecked() );
    KitaConfig::setPartMimeList( partMimeList->text() );
}

void UIPrefPage::reset()
{
    useTabCheckBox->setChecked( KitaConfig::defaultAlwaysUseTab() );
    MarkTimespinBox->setValue( KitaConfig::defaultMarkTime() );
    showMailCheckBox->setChecked( KitaConfig::defaultShowMailAddress() );
    orderGroup->setButton( KitaConfig::defaultSortOrder() );
    showNumSpin->setValue( KitaConfig::defaultShowNum() );
    useExtPartCheck->setChecked( KitaConfig::defaultUsePart() );
    partMimeList->setText( KitaConfig::defaultPartMimeList().join( "," ) );
}

void UIPrefPage::slotEditFileAssociation()
{
    KApplication::kdeinitExec( "kcmshell", "filetypes" );
}

FacePrefPage::FacePrefPage( QWidget* parent )
        : FacePrefBase( parent )
{
    // font
    connect( listFontButton, SIGNAL( clicked() ), SLOT( slotFontButtonClicked() ) );

    connect( threadFontButton, SIGNAL( clicked() ),
             SLOT( slotThreadFontButtonClicked() ) );

    connect( popupFontButton, SIGNAL( clicked() ),
             SLOT( slotPopupFontButtonClicked() ) );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( this, SIGNAL( threadFaceChanged() ),
             signalCollection, SIGNAL( threadFaceChanged() ) );

    updateButtons();

    m_threadFontchanged = FALSE;
    m_threadColorChanged = FALSE;

    // color
    threadColorButton->setColor( KitaConfig::threadColor() );
    threadBackgroundColorButton->setColor( KitaConfig::threadBackgroundColor() );
    popupColorButton->setColor( KitaConfig::popupColor() );
    popupBackgroundColorButton->setColor( KitaConfig::popupBackgroundColor() );

    connect( threadColorButton, SIGNAL( changed( const QColor& ) ), SIGNAL( changed() ) );
    connect( threadBackgroundColorButton, SIGNAL( changed( const QColor& ) ), SIGNAL( changed() ) );
    connect( threadColorButton, SIGNAL( changed( const QColor& ) ), SLOT( slotColorChanged() ) );
    connect( threadBackgroundColorButton, SIGNAL( changed( const QColor& ) ), SLOT( slotColorChanged() ) );
    connect( popupColorButton, SIGNAL( changed( const QColor& ) ), SIGNAL( changed() ) );
    connect( popupBackgroundColorButton, SIGNAL( changed( const QColor& ) ), SIGNAL( changed() ) );

    // stylesheet
    styleSheetText->setText( KitaConfig::styleSheetText() );
    useStyleSheetCheckBox->setChecked( KitaConfig::useStyleSheet() );

    connect( styleSheetText, SIGNAL( textChanged() ), SIGNAL( changed() ) );
    connect( useStyleSheetCheckBox, SIGNAL( toggled( bool ) ), SIGNAL( changed() ) );

    connect( this, SIGNAL( setStyleSheetOfHTMLPart() ),
             signalCollection, SIGNAL( setStyleSheetOfHTMLPart() ) );
}

void FacePrefPage::apply()
{
    // font
    QFont font = listFontButton->font();
    KitaConfig::setListFont( font );
    emit fontChanged( font );

    if ( m_threadFontchanged ) {
        QFont threadFont = threadFontButton->font();
        KitaConfig::setThreadFont( threadFont );
        emit threadFaceChanged();
    }
    m_threadFontchanged = FALSE;

    QFont popupFont = popupFontButton->font();
    KitaConfig::setPopupFont( popupFont );

    // color
    if ( m_threadColorChanged ) {
        KitaConfig::setThreadColor( threadColorButton->color() );
        KitaConfig::setThreadBackgroundColor( threadBackgroundColorButton->color() );
        // TODO: rename signal.
        emit threadFaceChanged();
    }
    m_threadColorChanged = FALSE;
    KitaConfig::setPopupColor( popupColorButton->color() );
    KitaConfig::setPopupBackgroundColor( popupBackgroundColorButton->color() );

    // stylesheet
    if ( KitaConfig::styleSheetText() != styleSheetText->text() ||
            KitaConfig::useStyleSheet() != useStyleSheetCheckBox->isChecked() ) {
        KitaConfig::setStyleSheetText( styleSheetText->text() );
        KitaConfig::setUseStyleSheet( useStyleSheetCheckBox->isChecked() );
        emit setStyleSheetOfHTMLPart();
    }
}

void FacePrefPage::reset()
{
    // font
    QFont font;
    font = KitaConfig::defaultListFont();
    listFontButton->setText( font.family() + " " + QString::number( font.pointSize() ) );
    listFontButton->setFont( font );

    font = KitaConfig::defaultThreadFont();
    threadFontButton->setText( font.family() + " " + QString::number( font.pointSize() ) );
    threadFontButton->setFont( font );
    m_threadFontchanged = TRUE;

    font = KitaConfig::defaultPopupFont();
    popupFontButton->setText( font.family() + " " + QString::number( font.pointSize() ) );
    popupFontButton->setFont( font );

    // color
    threadColorButton->setColor( KitaConfig::defaultThreadColor() );
    threadBackgroundColorButton->setColor( KitaConfig::defaultThreadBackgroundColor() );
    popupColorButton->setColor( KitaConfig::defaultPopupColor() );
    popupBackgroundColorButton->setColor( KitaConfig::defaultPopupBackgroundColor() );
    m_threadColorChanged = TRUE;

    // reset
    useStyleSheetCheckBox->setChecked( KitaConfig::defaultUseStyleSheet() );
    styleSheetText->setText( KitaConfig::defaultStyleSheetText() );
}

void FacePrefPage::updateButtons()
{
    QFont font = KitaConfig::listFont();
    listFontButton->setText( font.family() + " " + QString::number( font.pointSize() ) );
    listFontButton->setFont( font );

    QFont threadFont = KitaConfig::threadFont();
    threadFontButton->setText( threadFont.family() + " " + QString::number( threadFont.pointSize() ) );
    threadFontButton->setFont( threadFont );

    QFont popupFont = KitaConfig::popupFont();
    popupFontButton->setText( popupFont.family() + " " + QString::number( popupFont.pointSize() ) );
    popupFontButton->setFont( popupFont );
}

void FacePrefPage::slotThreadFontButtonClicked()
{
    QFont threadFont = threadFontButton->font();

    if ( KFontDialog::getFont( threadFont, false, this ) == QDialog::Accepted ) {
        threadFontButton->setText( threadFont.family() + " " + QString::number( threadFont.pointSize() ) );
        threadFontButton->setFont( threadFont );
        emit changed();
        m_threadFontchanged = TRUE;
    }
}

void FacePrefPage::slotFontButtonClicked()
{
    QFont font = listFontButton->font();

    if ( KFontDialog::getFont( font, false, this ) == QDialog::Accepted ) {
        listFontButton->setText( font.family() + " " + QString::number( font.pointSize() ) );
        listFontButton->setFont( font );
        emit changed();
    }
}

void FacePrefPage::slotPopupFontButtonClicked()
{
    QFont font = popupFontButton->font();

    if ( KFontDialog::getFont( font, false, this ) == QDialog::Accepted ) {
        popupFontButton->setText( font.family() + " " + QString::number( font.pointSize() ) );
        popupFontButton->setFont( font );
        emit changed();
    }
}

void FacePrefPage::slotColorChanged()
{
    m_threadColorChanged = TRUE;
}
