#!/usr/bin/env python

import network, util
from util import exceptions
import hashlib, mx.DateTime, time
from gettext import lgettext as _

PROTOCOL_INFO = {
  "name": "Facebook",
  "version": "1.0",
  
  "config": [
    "color",
    "receive_enabled",
    "send_enabled",
    "username",
    "session_key",
    "private:secret_key"
  ],

  "authtype": "facebook",
  "color": "#64006C",

  "features": [
    "send",
    "reply",
    "receive",
    "responses",
    "thread",
    "delete",
    "send_thread",
    "like",
    "images",
  ],

  "default_streams": [
    "receive",
    "responses",
    "images",
  ]
}

APP_KEY = "71b85c6d8cb5bbb9f1a3f8bbdcdd4b05"
URL_PREFIX = "https://api.facebook.com/restserver.php"
POST_URL = "http://www.facebook.com/profile.php?id=%s&v=feed&story_fbid=%s&ref=mf"

class Client:
  def __init__(self, acct):
    self.account = acct
    self.user_id = acct["session_key"].split("-")[1]

  def _get(self, operation, post=False, single=False, **args):
    args.update({
      "v": "1.0",
      "format": "json",
      "method": "facebook." + operation,
      "api_key": APP_KEY,
      "session_key": self.account["session_key"],
      "call_id": str(int(time.time()) * 1000),
    })

    sig = "".join("%s=%s" % (k, v) for k, v in sorted(args.items()))
    args["sig"] = hashlib.md5(sig + self.account["secret_key"]).hexdigest()
    data = network.Download(URL_PREFIX, args, post).get_json()
    if "error_msg" in data:
      if data.has_key("error_msg"):
        if "permission" in data["error_msg"]:
          raise exceptions.GwibberProtocolError("auth", self.account["protocol"], self.account["username"], data["error_msg"])

    return data

  def _sender(self, user):
    sender = {
      "name": user["name"],
      "id": str(user["id"]),
      "is_me": str(user["id"]) == self.user_id,
      "url": user["url"],
      "image": user["pic_square"],
    }
    
    if not "?" in user["url"]:
      sender["nick"] = user["url"].rsplit("/", 1)[-1]
    return sender
  
  def _message(self, data, profiles):
    m = {}
    m["id"] = str(data["post_id"])
    m["protocol"] = "facebook"
    m["account"] = self.account["_id"]
    m["time"] = int(mx.DateTime.DateTimeFrom(data['created_time']).gmtime())
    m["url"] = data["permalink"]

    if data.get("attribution", 0):
      m["source"] = util.strip_urls(data["attribution"]).replace("via ", "")
    
    if data.get("message", "").strip():
      m["text"] = data["message"]
      m["html"] = util.linkify(data["message"])
      m["content"] = m["html"]
    else:
      m["text"] = ""
      m["html"] = ""
      m["content"] = ""

    if data.get("actor_id", 0) in profiles:
      m["sender"] = self._sender(profiles[data["actor_id"]])

    if data.get("likes", {}).get("count", None):
      m["likes"] = {
        "count": data["likes"]["count"],
        "url": data["likes"]["href"],
      }

    if data.get("comments", 0):
      m["comments"] = []
      for item in data["comments"]["comment_list"]:
        m["comments"].append({
          "text": item["text"],
          "time": int(mx.DateTime.DateTimeFrom(item["time"]).gmtime()),
          "sender": self._sender(profiles[item["fromid"]]),
        })

    if data.get("attachment", 0):
      if data["attachment"].get("name", 0):
        m["content"] += "<p><b>%s</b></p>" % data["attachment"]["name"]

      if data["attachment"].get("description", 0):
        m["content"] += "<p>%s</p>" % data["attachment"]["description"]

      m["images"] = []
      for a in data["attachment"].get("media", []):
        if a["type"] in ["photo", "video", "link"]:
          if a.get("src", 0):
            if a["src"].startswith("/"):
              a["src"] = "http://facebook.com" + a["src"]
            m["images"].append({"src": a["src"], "url": a["href"]})

    return m

  def _comment(self, data, profiles):
    user = profiles[data["fromid"]]
    return {
      "id": str(data["id"]),
      "protocol": "facebook",
      "account": self.account["_id"],
      "time": int(mx.DateTime.DateTimeFrom(data['time']).gmtime()),
      "text": "@%s: %s" % (self.account["username"], data["text"]),
      "content": "@%s: %s" % (self.account["username"], data["text"]),
      "html": "@%s: %s" % (self.account["username"], data["text"]),
      "reply": {
        "id": data["post_id"],
        "nick": self.account["username"],
        "url": POST_URL % (self.user_id, data["object_id"]),
      },
      "sender": {
        "nick": user["username"] or str(user["uid"]),
        "name": user["name"],
        "id": str(user["uid"]),
        "url": user["profile_url"],
        "image": user["pic_square"],
      }
    }

  def _image(self, data, profiles):
    user = profiles[data["owner"]]
    return {
      "id": str(data["object_id"]),
      "protocol": "facebook",
      "account": self.account["_id"],
      "time": int(mx.DateTime.DateTimeFrom(data['created']).gmtime()),
      "content": data["caption"],
      "text": data["caption"],
      "html": data["caption"],
      "images": [{
        "full": data["src_big"],
        "src": data["src_big"],
        "thumb": data["src_small"],
        "url": data["link"],
      }],
      "sender": {
        "nick": user["username"] or str(user["uid"]),
        "name": user["name"],
        "id": str(user["uid"]),
        "url": user["profile_url"],
        "image": user["pic_square"],
      }
    }

  def _friends(self):
    friends = self._get("fql.query", query="""
      SELECT name, profile_url, pic_square, username, uid
        FROM user WHERE uid in (SELECT uid2 FROM friend WHERE uid1=%s)
      """ % self.user_id)

    return dict((p["uid"], p) for p in friends)

  def __call__(self, opname, **args):
    return getattr(self, opname)(**args)

  def thread(self, id):
    query = "SELECT name, profile_url, pic_square, username, uid FROM user WHERE uid in \
      (SELECT fromid FROM comment WHERE post_id = '%s')" % id
    
    profiles = dict((p["uid"], p) for p in self._get("fql.query", query=query))
    comments = self._get("stream.getComments", post_id=id)
    return [self._comment(comment, profiles) for comment in comments]

  def receive(self):
    data = self._get("stream.get", viewer_id=self.user_id, limit=80)
    profiles = dict((p["id"], p) for p in data["profiles"])
    return [self._message(post, profiles) for post in data["posts"]]

  def responses(self, limit=100):
    data = self._get("fql.query", query="""
      SELECT id, post_id, time, fromid, text, object_id FROM comment WHERE post_id IN
        (SELECT post_id FROM stream WHERE source_id = %s) AND
        fromid <> %s ORDER BY time DESC LIMIT 1,%s
      """ % (self.user_id, self.user_id, limit))
    
    profiles = self._friends()
    return [self._comment(comment, profiles) for comment in data]

  def images(self, limit=100):
    data = self._get("fql.query", query="""
      SELECT owner, object_id, created, src_small, src_big, link, caption
        FROM photo WHERE aid in
        (SELECT aid FROM album WHERE owner IN
        (SELECT uid2 FROM friend WHERE uid1=%s))
        ORDER BY created DESC LIMIT 1,%s
      """ % (self.user_id, limit))
    
    profiles = self._friends()
    return [self._image(post, profiles) for post in data]

  def delete(self, message):
    self._get("stream.remove", post_id=message["id"])
    return []

  def like(self, message):
    self._get("stream.addLike", post_id=message["id"])
    return []

  def send(self, message):
    self._get("users.setStatus", status=message, status_includes_verb=False)
    return []

  def send_thread(self, message, target):
    self._get("stream.addComment", post_id=target["id"], comment=message)
    return []
