/*
 * Gnome Attacks
 *
 * Copyright (c) 2003 by Andy Balaam
 *
 * Released under the GNU General Public License
 * with ABSOLUTELY NO WARRANTY.
 *
 * See the file COPYING for more information.
 *
 */

#include "levelset.h"

#include "main.h"
#include "graphics.h"

/**
 * Gets all the directories that contain levels and stores the results in
 * our levelset globals: levelset_paths, levelset_names, num_levelsets.
 */
void levelset_setup()
{ TRACE(levelset_setup)
  
  /* ---declare variables------------------------------------------------- */
  
  GError* err;  /* An error signal from opening system dir */
  
  GString* system_level_path_mine;  /* Path to the system dir */
  GString* user_level_path_mine;    /* Path to the user dir */
  
  GDir* system_level_dir_mine;
    /* The system directory containing levels
      e.g. /usr/local/share/pixmaps/gnomeattacks */
    
  GDir* user_level_dir_mine;
    /* The user directory containing levels
      i.e. ~/.gnomeattacks */
  
  gint i; /* Just a loop counter */
  
  /* ---init variables----------------------------------------------------- */
  
  err = NULL;
  
  system_level_path_mine = g_string_new( PIXMAPS_DIR );
  
  user_level_path_mine = g_string_new( g_get_home_dir() );
  g_string_append( user_level_path_mine, "/.gnomeattacks" );
  
  system_level_dir_mine = g_dir_open( system_level_path_mine->str, 0, &err );
  user_level_dir_mine = g_dir_open( user_level_path_mine->str, 0, NULL );
  
  /* ---reset the global variables we are setting up------------------------*/
  
  num_levelsets = 0;
  current_levelset = 0;
  levelset_paths = g_ptr_array_new();
  levelset_names = g_ptr_array_new();
  
  /* ---count the level sets----------------------------------------------- */
  
  /* Check that the system dir exists - it must or we fail */
  if( system_level_dir_mine == NULL ) {
    
    fprintf( stderr, "gnomeattacks: Unable to open levels directory: %s\n",
      err->message );
    g_error_free( err );
    exit( 4 );
    
  }
  
  levelset_setup_process_dir( system_level_dir_mine, system_level_path_mine );
  
  /* Then the user dir (ignore if it doesn't exist) */
  if( user_level_dir_mine != NULL ) {
    
    levelset_setup_process_dir( user_level_dir_mine, user_level_path_mine );
    
  }
  
  /* If there are none, complain and quit */
  if( num_levelsets == 0 ) {
    
    fprintf( stderr,
      "gnomeattacks: no level sets found in either '%s' or '%s'!\n",
      system_level_path_mine, user_level_path_mine );
    exit( 5 );
    
  }
  
  g_dir_close( system_level_dir_mine );
  
  if( user_level_dir_mine != NULL ) {
    g_dir_close( user_level_dir_mine );
  }
  
  g_string_free( system_level_path_mine, TRUE );
  g_string_free( user_level_path_mine, TRUE );
  
  
}

/**
 * Read through the supplied directory, adding any subdirectories to our list
 * of level sets.
 *
 * Alters the global variables levelset_paths, levelset_names, num_levelsets.
 *
 * @param dir the GDir to read through.
 */
void levelset_setup_process_dir( GDir* dir, GString* path )
{ TRACE(levelset_setup_process_dir)
  
  G_CONST_RETURN gchar* tmp_file_name;
    /* Holds a filename from g_dir_read_name */
    
  GString* tmp_file_path_mine; /* Temporary file path */
  
  /* Initialise our GString */
  tmp_file_path_mine = g_string_new("");
  
  /* Step through the dir finding directories and remembering their names. */
  while( tmp_file_name = g_dir_read_name( dir ) )
  {
    
    /* Construct the full file path */
    g_string_assign( tmp_file_path_mine, path->str );
    g_string_append( tmp_file_path_mine, "/" );
    g_string_append( tmp_file_path_mine, tmp_file_name );
    
    /* If it's a directory, remember it */
    if( g_file_test( tmp_file_path_mine->str, G_FILE_TEST_IS_DIR ) ) {
      
      /* Increment the number of levelsets */
      ++num_levelsets;
      
      /* Create 2 GStrings and add pointers to them to the arrays */
      g_ptr_array_add( levelset_paths, g_string_new(
        tmp_file_path_mine->str ) );
      
      g_ptr_array_add( levelset_names, g_string_new( tmp_file_name ) );
      
    }
    
  }
  
  g_string_free( tmp_file_path_mine, TRUE );
  
}

/**
 * Create a menu listing all the channel sets stored in levelset_names and
 * add it as a submenu to the menu item levels1.
 */
void levelset_create_menu()
{ TRACE(levelset_create_menu)
  
  GtkMenu* levels_menu;
  
  gint i;
  GSList *group;
  GString* levelset_name;
  
  group = NULL;
  
  levels_menu = GTK_MENU( gtk_menu_new() );
  
  for( i=0; i<num_levelsets; i++ )
  {
      
    GtkWidget* menuitem;
    
    levelset_name = g_ptr_array_index( levelset_names, i );
    
    menuitem = gtk_radio_menu_item_new_with_label( group, levelset_name->str );
    
    group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM( menuitem ) );
    
    g_signal_connect( (gpointer) menuitem, "activate",
                    G_CALLBACK( levelset_on_menuitem_chosen ),
                    (gpointer)i );
    
    gtk_menu_append( levels_menu, menuitem );
    
  }
  
  gtk_menu_item_set_submenu( levels1, GTK_WIDGET( levels_menu ) );
  
  gtk_widget_show_all( GTK_WIDGET( levels1 ) );
  
  
}

/**
 * A callback for when the user chooses a channel set.
 *
 * Assumes it won't get called in-game.
 *
 * FIXME: Uses the pointer passed to it as an int, which is probably dodgy...?
 */
void levelset_on_menuitem_chosen( GtkWidget* menu_item,
  gpointer which_levelset_ptr )
{ TRACE(levelset_on_menuitem_chosen)
  
  gint which_levelset;
  
  which_levelset = (gint)which_levelset_ptr;
  
  if( current_levelset != which_levelset ) {
  
    current_levelset = which_levelset;

    graphics_reload_nonlevel_images();
    
  }
  
}

/**
 * Find the filename of a given file from the standard locations:
 *
 * e.g. /usr/local/share/pixmaps/gnomeattacks/LevelSet
 *   or ~/.gnomeattacks/LevelSet
 *
 * and test whether it exists.
 *
 * If it doesn't exist in any location, it returns NULL
 *
 * @param file_name the beginning of the file name to be found eg "level_1_fg"
 *                  _or_ "level_2_heights.txt".
 *
 * Returns a new GString* that must be g_string_free'd
 *
 */
GString* levelset_get_file_name( gchar* file_name )
{ TRACE(levelset_get_file_name)
  
  GString* ans;
  GString* levelset_path;
  GDir* search_dir;
  
  ans = NULL;
  levelset_path = g_ptr_array_index( levelset_paths, current_levelset );
  
  /* First look in the current levelset location */
  search_dir = g_dir_open( levelset_path->str, 0, NULL );
  ans = _search_dir( search_dir, file_name );
  
  if( ans == NULL ) {
    
    g_dir_close( search_dir );
    /* but if not found, look in the default location */
    search_dir = g_dir_open( PIXMAPS_DIR, 0, NULL );
    ans = _search_dir( search_dir, file_name );
    g_string_prepend( ans, "/" );
    g_string_prepend( ans, PIXMAPS_DIR );
    
  } else {
    
    g_string_prepend( ans, "/" );
    g_string_prepend( ans, levelset_path->str );
    
  }
  
  g_dir_close( search_dir );
  
  return ans;
  
}

/**
 * Finds the first file in the supplied directory whose name begins with the
 * characters stored in file_name.
 *
 */
GString* _search_dir( GDir* search_dir, gchar* file_name )
{
  
  G_CONST_RETURN gchar* search_file_name;
  gsize file_name_length;
  gchar* truncated_search_file_name;
  GString* ans;
  
  search_file_name = NULL;
  file_name_length = strlen( file_name ) + 1;
  truncated_search_file_name = g_malloc( file_name_length );
  
  while( search_file_name = g_dir_read_name( search_dir ) )
  {
    
    g_strlcpy( truncated_search_file_name, search_file_name, file_name_length );
    
    if( strcmp( truncated_search_file_name, file_name ) == 0 ) {
      
      TRACESTR(search_file_name);
      
      g_free( truncated_search_file_name );
      return g_string_new( search_file_name );
      
    }
    
  }
  
  g_free( truncated_search_file_name );
  return NULL;
  
}

/**
 * Load a pixbuf from one of the standard locations:
 *
 * e.g. /usr/local/share/pixmaps/gnomeattacks/LevelSet
 *   or ~/.gnomeattacks/LevelSet
 *
 * If there is an error opening the file, it dies if must_load
 * is set to TRUE, and returns NULL otherwise.
 *
 * @param file_name the beginning of the file name to be loaded eg "level_1_fg"
 *                  _or_ "level_2_bg".
 *
 * Returns a new GdkPixbuf that must be g_free'd
 *
 */
GdkPixbuf* levelset_get_pixbuf( gchar* file_name, gboolean must_load )
{ TRACE(levelset_get_pixbuf)
  
  GError* err;
  GdkPixbuf* ans;
  GString* file_path_mine;
  
  err = NULL;
  ans = NULL;
  
  file_path_mine = levelset_get_file_name( file_name );
  
  TRACEGSTR(file_path_mine)
  
  if( file_path_mine != NULL ) {
    
    ans = gdk_pixbuf_new_from_file( file_path_mine->str, &err );
  
    if( ans == NULL ) {
      fprintf( stderr, "Error loading file: %s: %s\n", file_path_mine->str,
        err->message );
      g_error_free( err );
      g_string_free( file_path_mine, TRUE );
      exit( 6 );
    }
    
    g_string_free( file_path_mine, TRUE );
    
  } else {
    
    if( must_load ) {
      fprintf( stderr, "gnomeattacks: Unable to locate file '%s' in any of the"
        "possible locations\n", file_name );
      exit( 7 );
    }
    
  }
  
  TRACEPTR(ans)
  
  return GDK_PIXBUF( ans );
  
}

/**
 * Clean up after ourselves: delete the levelset arrays.
 */
void levelset_quit()
{ TRACE(levelset_quit)
  
  /* TRUE here frees the GStrings contained in the arrays. I think this
     means we are ok with just this. */
  g_ptr_array_free( levelset_paths, TRUE );
  g_ptr_array_free( levelset_names, TRUE );
  
  num_levelsets = 0;
  current_levelset = 0;
  
}

