#!/usr/bin/perl

use Getopt::Std;
use Net::Telnet ();

my $ME = $0;

END {
  defined fileno STDOUT or return;
  close STDOUT and return;
  warn "$ME: failed to close standard output: $!\n";
  $? ||= 1;
}

# Get the program name from $0 and strip directory names
$_=$0;
s/.*\///;
my $pname = $_;

# WARNING!! Do not add code bewteen "#BEGIN_VERSION_GENERATION" and
# "#END_VERSION_GENERATION"  It is generated by the Makefile

#BEGIN_VERSION_GENERATION
$RELEASE_VERSION="";
$REDHAT_COPYRIGHT="";
$BUILD_DATE="";
#END_VERSION_GENERATION


sub usage 
{
    print "Usage:\n\n"; 
    print "$pname [options]\n\n";
    print "Options:\n";
    print "  -a <ip>          IP address or hostname of switch\n";
    print "  -h               Usage\n";
    print "  -o metadata      print XML metadata for fence agent\n";
    print "  -n <num>         Port number to disable\n";
    print "  -p <string>      Password for login\n";
    print "  -S <path>        Script to run to retrieve login password\n";
    print "  -V               version\n\n";

    exit 0;
}

sub fail
{
  ($msg) = @_;
  print $msg."\n" unless defined $opt_q;
  $t->close if defined $t;
  exit 1;
}

sub fail_usage
{
  ($msg) = @_;
  print STDERR $msg."\n" if $msg;
  print STDERR "Please use '-h' for usage.\n";
  exit 1;
}

sub version
{
  print "$pname $RELEASE_VERSION $BUILD_DATE\n";
  print "$REDHAT_COPYRIGHT\n" if ( $REDHAT_COPYRIGHT );

  exit 0;
}

sub print_metadata
{
print '<?xml version="1.0" ?>
<resource-agent name="fence_vixel" shortdesc="I/O Fencing agent for Vixel FC switches" >
<longdesc>
fence_vixel is an I/O Fencing agent which can be used with Vixel FC switches. It logs into a Vixel switch via telnet and removes the specified port from the zone. Removing the zone access from the port disables the port from being able to access the storage.

After a fence operation has taken place the fenced machine can no longer connect to the Vixel FC switch. When the fenced machine is ready to be brought back into the GFS cluster (after reboot) the port on the Vixel FC switch needs to be enabled. In order to do this, log into the Vixel FC switch. Then go to:

config->zones->config &lt;port&gt; &lt;comma-separated-list-of-ports-in-the-zone&gt;

Then apply. Consult the Vixel manual for details.
</longdesc>
<vendor-url>http://www.emulex.com</vendor-url>
<parameters>
        <parameter name="ipaddr" unique="1" required="1">
                <getopt mixed="-a &lt;ip&gt;" />
                <content type="string"  />
                <shortdesc lang="en">IP Address or Hostname</shortdesc>
        </parameter>
        <parameter name="passwd" unique="1" required="0">
                <getopt mixed="-p &lt;password&gt;" />
                <content type="string"  />
                <shortdesc lang="en">Login password or passphrase</shortdesc>
        </parameter>
        <parameter name="passwd_script" unique="1" required="0">
                <getopt mixed="-S &lt;script&gt;" />
                <content type="string"  />
                <shortdesc lang="en">Script to retrieve password</shortdesc>
        </parameter>
        <parameter name="port" unique="1" required="1">
                <getopt mixed="-n &lt;id&gt;" />
                <content type="string"  />
                <shortdesc lang="en">Physical plug number or name of virtual machine</shortdesc>
        </parameter>
        <parameter name="help" unique="1" required="0">
                <getopt mixed="-h" />           
                <content type="string"  />
                <shortdesc lang="en">Display help and exit</shortdesc>                    
        </parameter>
</parameters>
<actions>
        <action name="metadata" />
</actions>
</resource-agent>
';
}


if (@ARGV > 0) {
    getopts("a:hn:p:S:Vo:") || fail_usage ;

    usage if defined $opt_h;
    version if defined $opt_V;

    fail_usage "Unknown parameter." if (@ARGV > 0);

    if ((defined $opt_o) && ($opt_o =~ /metadata/i)) {
        print_metadata();
        exit 0;
    }

    fail_usage "No '-a' flag specified." unless defined $opt_a;

	if (defined $opt_S) {
		$pwd_script_out = `$opt_S`;
		chomp($pwd_script_out);
		if ($pwd_script_out) {
			$opt_p = $pwd_script_out;
		}
	}

    fail_usage "No '-p' or '-S' flag specified." unless defined $opt_p;
    fail_usage "No '-n' flag specified." unless defined $opt_n;

} else {
    get_options_stdin();

    fail "failed: no IP address for the Vixel." unless defined $opt_a;

	if (defined $opt_S) {
		$pwd_script_out = `$opt_S`;
		chomp($pwd_script_out);
		if ($pwd_script_out) {
			$opt_p = $pwd_script_out;
		}
	}

    fail "failed: no password provided." unless defined $opt_p;
    fail "failed: no port number specified." unless defined $opt_n;
}

#
# Set up and log in
#

$t = new Net::Telnet;

$t->open($opt_a);

$t->waitfor('/assword:/');

$t->print($opt_p);

($out, $match)= $t->waitfor(Match => '/\>/', Match => '/assword:/');

if ($match =~ /assword:/) {
  fail "failed: incorrect password\n";
} elsif ( $match !~ />/ ) {
  fail "failed: timed out waiting for prompt\n";
}
 
$t->print("config");

$t->waitfor('/\(config\)\>/');

$t->print("zone");

$t->waitfor('/\(config\/zone\)\>/');

#
# Do the command
#

$cmd = "config $opt_n \"\"";
$t->print($cmd);

$t->waitfor('/\(config\/zone\)\>/');

$t->print("apply");

($text, $match) = $t->waitfor('/\>/');
if ($text !~ /[Oo][Kk]/) {
	fail "failed: error from switch\n";
}

$t->print("exit");

print "success: zonedisable $opt_n\n";
exit 0;


sub get_options_stdin
{
	my $opt;
	my $line = 0;

	while( defined($in = <>) )
	{
		$_ = $in;
		chomp;

		# strip leading and trailing whitespace
		s/^\s*//;
		s/\s*$//;

		# skip comments
		next if /^#/;

		$line+=1;
		$opt=$_;
		next unless $opt;

		($name,$val)=split /\s*=\s*/, $opt;

		if ( $name eq "" ) {
			print("parse error: illegal name in option $line\n");
			exit 2;
		} 

		# DO NOTHING -- this field is used by fenced
		elsif ($name eq "agent" ) { }

		# FIXME -- depricated.  use "port" instead.
		elsif ($name eq "fm" ) {
			(my $dummy,$opt_n) = split /\s+/,$val;
			print STDERR "Depricated \"fm\" entry detected. Refer to man page.\n";
		} 
		elsif ($name eq "ipaddr" ) 
		{
			$opt_a = $val;
		} 
		elsif ($name eq "name" ) { }
		elsif ($name eq "passwd" ) 
		{
			$opt_p = $val;
		}
		elsif ($name eq "passwd_script" )
		{
			$opt_S = $val;
		} 
		elsif ($name eq "port" ) 
		{
			$opt_n = $val;
		} 
	}
}

