/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimsignsession.c 690 2009-03-02 10:27:22Z tsuruda $
 */

#include "rcsid.h"
RCSID("$Id: dkimsignsession.c 690 2009-03-02 10:27:22Z tsuruda $");

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <limits.h>
#include <errno.h>
#include <time.h>
#include <sys/param.h>

#include "dkimlogger.h"
#include "strarray.h"
#include "mailheaders.h"
#include "ptrop.h"
#include "inetdomain.h"
#include "inetmailbox.h"
#include "dkim.h"
#include "dkimdigest.h"
#include "dkimheaders.h"
#include "dkimsignpolicy.h"

struct DkimSignSession {
    const DkimSignPolicy *spolicy;
    dkim_stat_t signstat;
    const MailHeaders *headers;

    DkimDigest *digest;
    DkimSignature *signature;

    // author
    InetMailbox *author;
    size_t authoridx;
    const char *refRawAuthorField;  // リファレンスを保持するだけなので解放はしないこと
    const char *refRawAuthorValue;  // リファレンスを保持するだけなので解放はしないこと
};

/**
 * DkimSignSession オブジェクトの構築
 * @return 空の DkimSignSession オブジェクト
 */
DkimSignSession *
DkimSignSession_new(const DkimSignPolicy *spolicy)
{
    assert(NULL != spolicy);

    DkimSignSession *self = (DkimSignSession *) malloc(sizeof(DkimSignSession));
    if (NULL == self) {
        return NULL;
    }   // end if
    memset(self, 0, sizeof(DkimSignSession));

    // 最低限必要なオブジェクトの初期化
    self->signature = DkimSignature_new((const DkimPolicy *) spolicy);
    if (NULL == self->signature) {
        goto cleanup;
    }   // end if

    self->spolicy = spolicy;
    return self;

  cleanup:
    DkimSignSession_free(self);
    return NULL;
}   // end function : DkimSignSession_new

/**
 * DkimSignSession オブジェクトの解放
 * @param self 解放する DkimSignSession オブジェクト
 */
void
DkimSignSession_free(DkimSignSession *self)
{
    assert(NULL != self);

    if (NULL != self->signature) {
        DkimSignature_free(self->signature);
    }   // end if
    if (NULL != self->digest) {
        DkimDigest_free(self->digest);
    }   // end if
    if (NULL != self->author) {
        InetMailbox_free(self->author);
    }   // end if
//    free(self->signhdrbuf);
    free(self);
}   // end function : DkimSignSession_free

/**
 * DKIM の署名対象とするメールのヘッダを登録する.
 * @param headers ヘッダを格納する MailHeaders オブジェクト.
 *                キーはヘッダ名. ": " の左側の部分. ただし ":" は含まない.
 *                値はヘッダの値. ": " の右側の部分. ただし ":" の直後のスペースは含まない (sendmail 8.13 まで, 8.14 以降では仕様が変わる).
 */
dkim_stat_t
DkimSignSession_setHeaders(DkimSignSession *self, const MailHeaders *headers)
{
    assert(NULL != self);
    assert(NULL != headers);
    assert(NULL == self->headers);

    self->headers = headers;

    // author の抽出
    dkim_stat_t ext_stat =
        DkimHeaders_extractAuthor((const DkimPolicy *) self->spolicy, self->headers,
                                  self->spolicy->author_priority,
                                  &(self->authoridx),
                                  &(self->refRawAuthorField),
                                  &(self->refRawAuthorValue), &(self->author));
    if (DSTAT_OK != ext_stat) {
        self->signstat = ext_stat;
    }   // end if

    return self->signstat;
}   // end function : DkimSignSession_eoh

dkim_stat_t
DkimSignSession_startBody(DkimSignSession *self)
{
    assert(NULL != self);

    if (DSTAT_OK != self->signstat) {
        // 何もしない
        return DSTAT_OK;
    }   // end if

    assert(NULL != self->author);

    // メッセージの署名処理
    // sig-t-tag に使用する現在時刻の取得
    time_t epoch;
    if (0 > time(&epoch)) {
        DkimLogSysError(self->spolicy, "time(2) failed: err=%s", strerror(errno));
        self->signstat = DSTAT_SYSERR_IMPLERROR;
        return self->signstat;
    }   // end if

    // DkimSignature オブジェクトの構築と各種パラメーターの設定
    const DkimSignPolicy *spolicy = self->spolicy;
    DkimSignature_setDigestAlg(self->signature, spolicy->digestalg);
    DkimSignature_setPubKeyAlg(self->signature, spolicy->pubkeyalg);
    DkimSignature_setHeaderCanonAlg(self->signature, spolicy->canon_method_header);
    DkimSignature_setBodyCanonAlg(self->signature, spolicy->canon_method_body);
    DkimSignature_setBodyLengthLimit(self->signature, -1LL);    // body length limit は使用しない

    // sig-d-tag はいつでも From ヘッダから取得する.
    dkim_stat_t ret =
        DkimSignature_setSigningDomain(self->signature, InetMailbox_getDomain(self->author));
    if (DSTAT_OK != ret) {
        self->signstat = ret;
        return self->signstat;
    }   // end if

    // Author を sig-i-tag として使用する.
    ret = DkimSignature_setIdentity(self->signature, self->author);
    if (DSTAT_OK != ret) {
        self->signstat = ret;
        return self->signstat;
    }   // end if

    DkimSignature_setGeneratedTime(self->signature, (long long) epoch);
    DkimSignature_setSignatureTTL(self->signature, spolicy->signature_ttl);

    size_t headernum = MailHeaders_getCount(self->headers);
    for (size_t headeridx = 0; headeridx < headernum; ++headeridx) {
        const char *headerf, *headerv;
        MailHeaders_get(self->headers, headeridx, &headerf, &headerv);
        if (NULL == headerf || NULL == headerv) {
            DkimLogWarning(self->spolicy, "ignore an invalid header: no=%d, name=%s, value=%s",
                           headeridx, NNSTR(headerf), NNSTR(headerv));
            continue;
        }   // end if

        // headers が含む全てのヘッダを署名対象として登録する.
        dkim_stat_t select_stat = DkimSignature_addSelectHeaderName(self->signature, headerf);
        if (DSTAT_OK != select_stat) {
            self->signstat = select_stat;
            return self->signstat;
        }   // end if
    }   // end for

    self->digest =
        DkimDigest_newWithSignature((const DkimPolicy *) self->spolicy, self->signature, &ret);
    if (NULL == self->digest) {
        self->signstat = ret;
        return self->signstat;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimSignSession_startBody

dkim_stat_t
DkimSignSession_updateBody(DkimSignSession *self, const unsigned char *bodyp, size_t len)
{
    assert(NULL != self);

    if (DSTAT_OK != self->signstat) {
        // 何もしない
        return DSTAT_OK;
    }   // end if

    self->signstat = DkimDigest_updateBody(self->digest, bodyp, len);
    return self->signstat;
}   // end function : DkimSignSession_updateBody

/**
 * メッセージ本文の終了を確定し, DKIM-Signature ヘッダを生成する.
 * @param self DkimSignSession オブジェクト
 * @param headerf 生成した DKIM-Signature ヘッダのヘッダフィールド名を受け取る変数へのポインタ.
 *                通常は "DKIM-Signature" という文字列が返る.
 * @param headerv 生成した DKIM-Signature ヘッダのヘッダフィールド値を受け取る変数へのポインタ.
 */
dkim_stat_t
DkimSignSession_finishBody(DkimSignSession *self, const char *selector, EVP_PKEY *pkey,
                           const char **headerf, const char **headerv)
{
    assert(NULL != self);

    if (DSTAT_OK != self->signstat) {
        return self->signstat;  // 最終結果として保存してあるステータスコードを返す
    }   // end if

    dkim_stat_t ret = DkimSignature_setSelector(self->signature, selector);
    if (DSTAT_OK != ret) {
        self->signstat = ret;
        return self->signstat;
    }   // end if

    ret = DkimDigest_signMessage(self->digest, self->headers, self->signature, pkey);
    if (DSTAT_OK != ret) {
        self->signstat = ret;
        return self->signstat;
    }   // end if
    self->signstat = DkimSignature_buildRawHeader(self->signature, false, headerf, headerv);
    return self->signstat;
}   // end function : DkimSignSession_finishBody

/**
 * DkimSignSession_finishBody() 成功後に呼び出す必要がある.
 */
dkim_stat_t
DkimSignSession_setCanonDump(DkimSignSession *self, const char *basedir, const char *prefix)
{
    assert(NULL != self);

    if (DSTAT_OK != self->signstat) {
        // do nothing
        return DSTAT_OK;
    }   // end if

    char fnHeader[MAXPATHLEN];
    char fnBody[MAXPATHLEN];

    snprintf(fnHeader, MAXPATHLEN, "%s/%s.header", basedir, prefix);
    snprintf(fnBody, MAXPATHLEN, "%s/%s.body", basedir, prefix);
    return DkimDigest_openCanonDump(self->digest, fnHeader, fnBody);
}   // end function : DkimSignSession_setCanonDump

////////////////////////////////////////////////////////////////////////
// accessor

const char *
DkimSignSession_getAuthorHeaderName(const DkimSignSession *self)
{
    return self->refRawAuthorField;
}   // end function : DkimSignSession_getAuthorHeaderName

/**
 * Author として使用されるアドレスを返す.
 * DkimSignSession_setHeaders() 以降のみ有効.
 */
const InetMailbox *
DkimSignSession_getAuthorMailbox(const DkimSignSession *self)
{
    return self->author;
}   // end function : DkimSignSession_getAuthorMailbox
