/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include "tabdisc.h"

#include "disc.h"
#include "log.h"
#include "i18n.h"

#include <QtGui/QFontMetrics>
#include <QtGui/QHBoxLayout>
#include <QtGui/QLineEdit>
#include <QtGui/QListWidget>
#include <QtGui/QListWidgetItem>
#include <QtGui/QPushButton>
#include <QtGui/QTextEdit>
#include <QtGui/QVBoxLayout>


TabDisc::TabDisc(KoverArtist::Project* aProject, QWidget* aParent)
:Inherited(aProject, aParent)
,mDisc(0)
{
   createCaption(i18n("Disc Title (optional)"));

   QFontMetrics fontMetrics(font());
   int edtHeight = int(fontMetrics.lineSpacing()*2.5);

   mEdtTitle = new QTextEdit(this);
   mEdtTitle->setAcceptRichText(false);
   mEdtTitle->setWordWrapMode(QTextOption::NoWrap);
   mEdtTitle->setFixedHeight(edtHeight + mEdtTitle->height() - mEdtTitle->contentsRect().height());
   mEdtTitle->setMinimumWidth(mEdtTitle->sizeHint().width());
   connect(mEdtTitle, SIGNAL(textChanged()), SLOT(titleChanged()));
   mMainBox->addWidget(mEdtTitle);

   createCaption(i18n("Disc Contents"));

   mEdtEntry = new QLineEdit(this);
   connect(mEdtEntry, SIGNAL(textEdited(const QString&)), SLOT(entryTextEdited(const QString&)));
   connect(mEdtEntry, SIGNAL(editingFinished()), SLOT(entryTextEditFinished()));
   connect(mEdtEntry, SIGNAL(returnPressed()), SLOT(newEntry()));
   mMainBox->addWidget(mEdtEntry);

   QHBoxLayout *hbox = new QHBoxLayout;
   hbox->setSpacing(2);
   mMainBox->addLayout(hbox, 1000);

   mLstEntries = new QListWidget(this);
   connect(mLstEntries, SIGNAL(itemClicked(QListWidgetItem*)),
           SLOT(itemSelected(QListWidgetItem*)));
   hbox->addWidget(mLstEntries, 100);

   QVBoxLayout *btnBox = new QVBoxLayout;
   btnBox->setSpacing(2);
   hbox->addLayout(btnBox, 0);

   mBtnNew = new QPushButton(i18n("New"), this);
   mBtnNew->setFixedHeight(mBtnNew->sizeHint().height());
   mBtnNew->setMinimumWidth(mBtnNew->sizeHint().width());
   mBtnNew->setToolTip(i18n("Create a new entry\nat the end of the list"));
   connect(mBtnNew, SIGNAL(clicked()), SLOT(newEntry()));
   btnBox->addWidget(mBtnNew);

   mBtnRemove = new QPushButton(i18n("Remove"), this);
   mBtnRemove->setFixedHeight(mBtnRemove->sizeHint().height());
   mBtnRemove->setMinimumWidth(mBtnRemove->sizeHint().width());
   mBtnRemove->setToolTip(i18n("Remove the current entry"));
   connect(mBtnRemove, SIGNAL(clicked()), SLOT(removeEntry()));
   btnBox->addWidget(mBtnRemove);

   mBtnUp = new QPushButton(i18n("Move Up"), this);
   mBtnUp->setFixedHeight(mBtnUp->sizeHint().height());
   mBtnUp->setMinimumWidth(mBtnUp->sizeHint().width());
   mBtnUp->setToolTip(i18n("Move the current entry\nupwards in the list"));
   connect(mBtnUp, SIGNAL(clicked()), SLOT(upEntry()));
   btnBox->addWidget(mBtnUp);

   mBtnDown = new QPushButton(i18n("Move Down"), this);
   mBtnDown->setFixedHeight(mBtnDown->sizeHint().height());
   mBtnDown->setMinimumWidth(mBtnDown->sizeHint().width());
   mBtnDown->setToolTip(i18n("Move the current entry\ndownwards in the list"));
   connect(mBtnDown, SIGNAL(clicked()), SLOT(downEntry()));
   btnBox->addWidget(mBtnDown);

   btnBox->addStretch(100);
}


TabDisc::~TabDisc()
{
}


void TabDisc::setDisc(KoverArtist::Disc* aDisc)
{
   logDebug("setDisc %1").arg(aDisc? aDisc->title() : QString("<none>"));
   mDisc = aDisc;
   updateContents();
}


void TabDisc::updateContents()
{
   logDebug("updateContents");

   mLstEntries->clear();
   mEdtEntry->setText(QString::null);
   mBtnNew->setEnabled(false);
   mBtnRemove->setEnabled(false);
   mBtnUp->setEnabled(false);
   mBtnDown->setEnabled(false);

   if (!mDisc)
   {
      mEdtTitle->setText(QString::null);
      return;
   }

   mEdtTitle->setText(mDisc->title());

   int i, num = mDisc->count();
   for (i=0; i<num; ++i)
      mLstEntries->addItem(mDisc->entry(i));
}


void TabDisc::titleChanged()
{
   if (!mDisc) return;

   mDisc->setTitle(mEdtTitle->toPlainText());
   if (updatesEnabled()) emit changed();
}


void TabDisc::itemSelected(QListWidgetItem* aItem)
{
   mEdtEntry->setText(aItem ? aItem->text() : QString::null);

   mBtnNew->setEnabled(aItem!=0);
   mBtnRemove->setEnabled(aItem!=0);

   const int num = mLstEntries->count();
   mBtnUp->setEnabled(num>1 && aItem!=mLstEntries->item(0));
   mBtnDown->setEnabled(num>1 && aItem!=mLstEntries->item(num-1));
}


void TabDisc::entryTextEdited(const QString& aText)
{
   const bool hasText = !aText.isEmpty();
   int idx = mLstEntries->currentRow();

   if (idx>=0)
   {
      mLstEntries->currentItem()->setText(aText);
      mDisc->setEntry(idx, aText);
   }
   else if (hasText)
   {
      mDisc->entries().push_back(aText);
      mLstEntries->addItem(aText);
      mLstEntries->setCurrentRow(mLstEntries->count()-1);
   }

   mBtnNew->setEnabled(hasText);
   mBtnRemove->setEnabled(hasText || idx>=0);
   mBtnUp->setEnabled(false);
   mBtnDown->setEnabled(false);

   if (updatesEnabled()) emit changed();
}


void TabDisc::entryTextEditFinished()
{
   if (!mEdtEntry->text().isEmpty())
      return;

   int idx = mLstEntries->currentRow();
   mDisc->entries().removeAt(idx);
   delete mLstEntries->item(idx);

   mLstEntries->setCurrentRow(-1);

   mBtnNew->setEnabled(false);
   mBtnRemove->setEnabled(false);
   mBtnUp->setEnabled(false);
   mBtnDown->setEnabled(false);
}


void TabDisc::newEntry()
{
   mEdtEntry->setText(QString::null);
   mEdtEntry->setFocus(Qt::MouseFocusReason);
   mLstEntries->setCurrentRow(-1);

   if (updatesEnabled()) emit changed();
}


void TabDisc::removeEntry()
{
   int idx = mLstEntries->currentRow();
   if (idx<0) return;

   int num = mLstEntries->count();
   mDisc->entries().removeAt(idx);
   delete mLstEntries->item(idx);

   --num;
   if (idx>=num) --idx;
   mLstEntries->setCurrentRow(idx);
   itemSelected(mLstEntries->currentItem());

   mBtnRemove->setEnabled(num>0);
   mBtnUp->setEnabled(idx>0);
   mBtnDown->setEnabled(idx<num-1);

   if (updatesEnabled()) emit changed();
}


void TabDisc::upEntry()
{
   int idx = mLstEntries->currentRow();
   if (idx<1) return;

   QString entry = mDisc->entry(idx);
   QString otherEntry = mDisc->entry(idx-1);

   mDisc->setEntry(idx, otherEntry);
   mLstEntries->item(idx)->setText(otherEntry);

   mDisc->setEntry(idx-1, entry);
   mLstEntries->item(idx-1)->setText(entry);

   --idx;
   mLstEntries->setCurrentRow(idx);
   mBtnUp->setEnabled(idx>0);
   mBtnDown->setEnabled(true);

   if (updatesEnabled()) emit changed();
}


void TabDisc::downEntry()
{
   int idx = mLstEntries->currentRow();
   const int num = mLstEntries->count();
   if (idx>=num-1) return;

   QString entry = mDisc->entry(idx);
   QString otherEntry = mDisc->entry(idx+1);

   mDisc->setEntry(idx, otherEntry);
   mLstEntries->item(idx)->setText(otherEntry);

   mDisc->setEntry(idx+1, entry);
   mLstEntries->item(idx+1)->setText(entry);

   ++idx;
   mLstEntries->setCurrentRow(idx);
   mBtnUp->setEnabled(true);
   mBtnDown->setEnabled(idx<num-1);

   if (updatesEnabled()) emit changed();
}
