/*
   Copyright (C) 2006-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// renderimage.cpp

#include "renderimage.h"

#include <assert.h>


static bool sAlphaTabNeedInit = true;
static int sAlphaTab[65536];


RenderImage::RenderImage()
:Inherited()
{
}


RenderImage::RenderImage(int aWidth, int aHeight)
:Inherited(aWidth, aHeight, QImage::Format_ARGB32)
{
}


RenderImage::RenderImage(const RenderImage& o)
:Inherited(o)
{
}


RenderImage::RenderImage(const QImage& o)
:Inherited(o)
{
}


RenderImage& RenderImage::operator=(const RenderImage& o)
{
   Inherited::operator=(o);
   return *this;
}


RenderImage& RenderImage::operator=(const QImage& o)
{
   Inherited::operator=(o);
   return *this;
}


RenderImage::~RenderImage()
{
}


void RenderImage::initAlphaTab()
{
   int idx, x, alpha;

   for (x=0; x<=255; ++x)
   {
      for (alpha=0; alpha<=255; ++alpha)
      {
         idx = (alpha<<8)+x;
         assert(idx<65536);
         sAlphaTab[idx] = alpha*x;
      }
   }

   sAlphaTabNeedInit = false;
}


void RenderImage::setImage(int aX0, int aY0, const QImage& aImg,
                           int aSX0, int aSY0, int aSW, int aSH)
{
   int w=width(), h=height(), iw=aImg.width(), ih=aImg.height();
   int x, y, sx, sx1, sy, sy1, ialpha, alpha, r, g, b;
   QRgb pix, ipix;

   if (sAlphaTabNeedInit)
      initAlphaTab();

   if (aSW<0) aSW = iw;
   if (aSH<0) aSH = ih;

   sx1 = aSX0+aSW;
   sy1 = aSY0+aSH;

#define RMIX(a,b) ((sAlphaTab[(ialpha<<8)+(a)] + sAlphaTab[(alpha<<8)+(b)]) >> 8)

   for (y=aY0,sy=aSY0; y<h && sy<sy1; ++y,++sy)
   {
      for (x=aX0,sx=aSX0; x<w && sx<sx1; ++x,++sx)
      {
         ipix = aImg.pixel(sx, sy);
         ialpha = qAlpha(ipix);
         if (ialpha>=255)
         {
            setPixel(x, y, ipix);
         }
         else
         {
            pix = pixel(x, y);
            alpha = 255-ialpha;
            r = RMIX(qRed(ipix), qRed(pix));
            g = RMIX(qGreen(ipix), qGreen(pix));
            b = RMIX(qBlue(ipix), qBlue(pix));
            setPixel(x, y, qRgb(r, g, b));
         }
      }
   }

#undef RMIX
}


void RenderImage::threshold(int aLimit, QRgb aPixClear, QRgb aPixSet)
{
   int x, y, w=width(), h=height();
   QRgb pix;

   for (y=0; y<h; ++y)
   {
      for (x=0; x<w; ++x)
      {
         pix = pixel(x, y);
         if (qRed(pix)<aLimit && qBlue(pix)<aLimit && qGreen(pix)<aLimit)
            setPixel(x, y, aPixClear);
         else setPixel(x, y, aPixSet);
      }
   }
}
