/*
   Copyright (C) 2006-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* rendereffect.h
 */
#ifndef rendereffect_h
#define rendereffect_h

#include "renderimage.h"
#include "effect.h"

#include <qpainter.h>
#include <qimage.h>
#include <qstring.h>
#include <qvector.h>


namespace KoverArtist
{

class RendererBase;


/** Renders a text effect.
 */
class RenderEffect: public RenderImage
{
   typedef RenderImage Inherited;

public:
   RenderEffect(const RendererBase* parent, const Effect& eff, int w, int h);

   virtual ~RenderEffect() {}

   /** Returns the bounding box. The bounding box may be modified by
    * @ref render.
    */
   const QRect& boundingBox() const { return mBB; }
   QRect& boundingBox() { return mBB; }

   /** Prepare painter and pixmap. Optional. */
   virtual void setup(QPainter& p, QPixmap& pix);

   /** Set the bounding box for rendering. */
   virtual void setBoundingBox(const QRect& bb);

   /** Render the effect. */
   virtual void render(QPixmap textPix) = 0;

   RenderEffect& operator=(const QImage&);

   /** Creates a render-effect object for effect-type t. Returns 0 if
    * the effect is invalid. The caller must delete the returned object.
    */
    static RenderEffect* create(const RendererBase* parent, const Effect& eff,
                                int width, int height);

   /** Returns a color that is a mix between color1 and color2. The factor
    * f controls the mixing: f=0 means 100% color1, f=1.0 means 100% color2.
    * f=0.5 is 50% color1 and 50% color2, etc.
    */
   static QRgb mixColors(QRgb color1, QRgb color2, double f, int alpha=-1);

protected:
   /** Enlarge bounding box mBB by d in all directions. Ensures that the
    * bounding box stays valid, i.e. all edges within (0,0,mWidth,mHeight).
    */
   virtual void growBoundingBox(unsigned int d);

   /** Set the alpha value of the pixels in the bounding box to indicate the
    * distance to the next full (alpha==255) pixel. The maximum distance is
    * given by maxDist.
    */
   virtual void makeDistanceMap(int maxDist);

   /** Setup color table mColors to span from rgb color c1 with transparency tr1
    * to rgb color c2 with transparency tr2, using the transparency factor tf.
    */
   virtual void setupColorTable(QRgb c1, QRgb c2, int tr1, int tr2, double tf);

   /** Create a QRgb with RGB+Alpha from a QColor c and an alpha value. */
   static QRgb rgba(const QColor& c, int alpha);

private:
   RenderEffect(const RenderEffect&);
   RenderEffect& operator=(const RenderEffect&);

   bool makeDistanceMapPixel(int x, int y, int alpha);

protected:
   const RendererBase *mParent;
   const Effect mEff;
   int mWidth, mHeight;
   QRect mBB;
   QVector<QRgb> mColors;
};



class RenderEffectBox: public RenderEffect
{
   typedef RenderEffect Inherited;
public:
   RenderEffectBox(const RendererBase* parent, const Effect& eff, int w, int h);
   RenderEffectBox& operator=(const QImage& o) { Inherited::operator=(o); return *this; }
   virtual ~RenderEffectBox();
   virtual void render(QPixmap textPix);
};



class RenderEffectGlow: public RenderEffect
{
   typedef RenderEffect Inherited;
public:
   RenderEffectGlow(const RendererBase* parent, const Effect& eff, int w, int h);
   RenderEffectGlow& operator=(const QImage& o) { Inherited::operator=(o); return *this; }
   virtual ~RenderEffectGlow();
   virtual void setup(QPainter& p, QPixmap& pix);
   virtual void render(QPixmap textPix);
};



class RenderEffectShadow: public RenderEffectGlow
{
   typedef RenderEffectGlow Inherited;
public:
   RenderEffectShadow(const RendererBase* parent, const Effect& eff, int w, int h);
   RenderEffectShadow& operator=(const QImage& o) { Inherited::operator=(o); return *this; }
   virtual ~RenderEffectShadow();
   virtual void setup(QPainter& p, QPixmap& pix);
};



class RenderEffectOutline: public RenderEffectGlow
{
   typedef RenderEffectGlow Inherited;
public:
   RenderEffectOutline(const RendererBase* parent, const Effect& eff, int w, int h);
   RenderEffectOutline& operator=(const QImage& o) { Inherited::operator=(o); return *this; }
   virtual ~RenderEffectOutline();
   virtual void setup(QPainter& p, QPixmap& pix);
};



#ifdef BROKEN
class RenderEffectRoundedBox: public RenderEffect
{
public:
   RenderEffectRoundedBox(const Renderer* parent, double scale, const QFont& f, const QRect& bb,
                Qt::AlignmentFlags f, const QString& txt, const Effect& eff);
   virtual ~RenderEffectRoundedBox() {}
   virtual void setupHelper(QPainter& phlp, int w, int h);

protected:
   virtual bool renderIt(QPainter& p, int step, QImage& helper);

   int mRndX, mRndY;
};

#endif /*BROKEN*/


} //namespace

#endif /*rendereffect_h*/
