/*
   Copyright (C) 2006-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "prefcddb.h"

#include "cddbchooser.h"
#include "cdinfo.h"
#include "settings.h"

#include <klocale.h>
#include <keditlistbox.h>
#include <kurlrequester.h>

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qboxlayout.h>



KoverArtistPrefCddb::KoverArtistPrefCddb(QWidget *parent)
:QFrame(parent)
{
   QVBoxLayout *vbox = new QVBoxLayout(this);
   QGridLayout *grpGrid;
   QBoxLayout *grpBox;
   QGroupBox *grp;
   QRadioButton *rbtn;
   QPushButton *btn;
   QLabel *lbl;
   int i;

#ifdef ENABLE_AUDIOCD

   // Options
   mCbxAutoSetTitle = new QCheckBox(i18n("Set project title from CDDB information "
      "if the project title is empty."), this);
   mCbxAutoSetTitle->setFixedSize(mCbxAutoSetTitle->sizeHint());
   vbox->addWidget(mCbxAutoSetTitle, Qt::AlignLeft);

   vbox->addSpacing(8);

   //
   // CD-Drive
   //
   grp = new QGroupBox(i18n("Audio-CD Device"), this);
   vbox->addWidget(grp);
   grpBox = new QHBoxLayout(grp);

   lbl = new QLabel(i18n("Device")+": ", grp);
   lbl->setFixedSize(lbl->sizeHint());
   grpBox->addWidget(lbl);

   mEdtCdDrive = new QLineEdit(grp);
   mEdtCdDrive->setFixedHeight(mEdtCdDrive->sizeHint().height());
   grpBox->addWidget(mEdtCdDrive);

   vbox->addSpacing(8);

   //
   // CDDB Mode
   //
   grp = new QGroupBox(i18n("CDDB Access Mode"), this);
   vbox->addWidget(grp);
   grpBox = new QHBoxLayout(grp);

   mBgrCddbAccess = new QButtonGroup(grp); //3, Qt::Horizontal, i18n("CDDB Access Mode"), this);
   mBgrCddbAccess->setExclusive(true);

   static struct
   {
      const char* label;
      KoverArtist::Settings::CddbAccess mode;
   } choiceTab[] =
   { { I18N_NOOP("Only local"), KoverArtist::Settings::CDDB_LOCAL },
     { I18N_NOOP("Only remote"), KoverArtist::Settings::CDDB_REMOTE },
     { I18N_NOOP("Local and remote"), KoverArtist::Settings::CDDB_ANY },
     { 0, KoverArtist::Settings::CDDB_LOCAL }
   };
   for (i=0; choiceTab[i].label; ++i)
   {
      rbtn = new QRadioButton(i18n(choiceTab[i].label), grp);
      rbtn->setMinimumSize(rbtn->sizeHint());
      mBgrCddbAccess->addButton(rbtn, int(choiceTab[i].mode));
      grpBox->addWidget(rbtn);
   }
   vbox->addSpacing(8);

   //
   // CDDB-Server
   //
   grp = new QGroupBox(i18n("Server For Remote CDDB Queries"), this);
   vbox->addWidget(grp);
   grpGrid = new QGridLayout(grp);

   lbl = new QLabel(i18n("CDDB-Server")+":", grp);
   lbl->setFixedSize(lbl->sizeHint());
   grpGrid->addWidget(lbl, 0, 0);

   mEdtCddbServer = new QLineEdit(grp);
   mEdtCddbServer->setFixedHeight(mEdtCddbServer->sizeHint().height());
   grpGrid->addWidget(mEdtCddbServer, 0, 1);

   lbl = new QLabel(i18n("Port")+":", grp);
   lbl->setFixedSize(lbl->sizeHint());
   grpGrid->addWidget(lbl, 1, 0);

   mEdtCddbPort = new QLineEdit("99999", grp);
   mEdtCddbPort->setFixedSize(mEdtCddbPort->sizeHint());
   grpGrid->addWidget(mEdtCddbPort, 1, 1);

   lbl = new QLabel(i18n("Mirrors")+":", grp);
   lbl->setFixedSize(lbl->sizeHint());
   grpGrid->addWidget(lbl, 2, 0);

   btn = new QPushButton(i18n("Select a mirror..."), grp);
   connect(btn, SIGNAL(clicked()), this, SLOT(selectMirror()));
   btn->setFixedSize(btn->sizeHint());
   grpGrid->addWidget(btn, 2, 1);

   vbox->addSpacing(8);


   //
   // CDDB local directories
   //
   KUrlRequester* urq = new KUrlRequester(this);
//    urq->setShowLocalProtocol(false);
   urq->setMode(KFile::Directory|KFile::ExistingOnly|KFile::LocalOnly);
   mElbCddbDirs = new KEditListBox(i18n("Local CDDB Cache Directories"),
      KEditListBox::CustomEditor(urq, urq->lineEdit()), this);
   vbox->addWidget(mElbCddbDirs, Qt::AlignLeft);


#else /*not ENABLE_AUDIOCD*/

   lbl = new QLabel(i18n("Audio-CD and CDDB support was disabled during compile time.\n"
      "Either manually during configure or because your system is not supported."),
      this);
   lbl->setFixedSize(lbl->sizeHint());
   vbox->addWidget(lbl);

#endif /*ENABLE_AUDIOCD*/

   vbox->addStretch(1000);
   vbox->activate();
}


KoverArtistPrefCddb::~KoverArtistPrefCddb()
{
}


void KoverArtistPrefCddb::fromConfig(const KoverArtist::Settings* aStg)
{
#ifdef ENABLE_AUDIOCD
   QString str;

   mEdtCdDrive->setText(aStg->cdDevice);
   mEdtCddbServer->setText(aStg->cddbServer);
   str.sprintf("%d", aStg->cddbPort);
   mEdtCddbPort->setText(str);

   // TODO button lookup is most probably broken:
   QAbstractButton *btn = mBgrCddbAccess->button(aStg->cddbAccess);
   if (btn) btn->setChecked(true);

   mCbxAutoSetTitle->setChecked(aStg->cddbAutoSetTitle);

   mElbCddbDirs->clear();
   mElbCddbDirs->insertStringList(aStg->cddbDirs);
#endif /*ENABLE_AUDIOCD*/
}


void KoverArtistPrefCddb::toConfig(KoverArtist::Settings* aStg) const
{
#ifdef ENABLE_AUDIOCD
   aStg->cdDevice = mEdtCdDrive->text();
   aStg->cddbServer = mEdtCddbServer->text();
   aStg->cddbPort = mEdtCddbPort->text().toInt();
   aStg->cddbAutoSetTitle = mCbxAutoSetTitle->isChecked();
   aStg->cddbAccess = (KoverArtist::Settings::CddbAccess) mBgrCddbAccess->checkedId();
   aStg->cddbDirs = mElbCddbDirs->items();
#endif /*ENABLE_AUDIOCD*/
}


bool KoverArtistPrefCddb::isValid() const
{
   return true;
}


void KoverArtistPrefCddb::selectMirror()
{
   QStringList lst;
   lst.append("http://at.freedb.org");
   lst.append("http://au.freedb.org");
   lst.append("http://ca.freedb.org");
   lst.append("http://es.freedb.org");
   lst.append("http://fi.freedb.org");
   lst.append("http://fr.freedb.org");
   lst.append("http://jp.freedb.org");
   lst.append("http://jp2.freedb.org");
   lst.append("http://ru.freedb.org");
   lst.append("http://uk.freedb.org");
   lst.append("http://uk2.freedb.org");
   lst.append("http://us.freedb.org");

   CddbChooser dlg(lst, i18n("Select a CDDB Mirror"),
         i18n("Please select a CDDB mirror from\nthe list below:"), this);
   if (dlg.exec()!=QDialog::Accepted) return;

   QString srv = dlg.currentItem();

   if (!srv.isEmpty())
   {
      mEdtCddbServer->setText(srv);
      mEdtCddbPort->setText("80");
   }
}
