/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include "image.h"

#include "i18n.h"
#include "log.h"
#include "settings.h"

#include <QtXml/QDomElement>
#include <QtGui/QImage>
#include <QtGui/QMessageBox>
#include <QtGui/QPainter>

#include <errno.h>
#include <string.h>


namespace KoverArtist
{


Image::Image()
:mUrl()
,mPosition(KoverArtist::ScaleCrop)
,mPixmap()
,mOnSides(true)
,mTransparency(255)
{
}


Image::Image(const Image& o)
:mUrl(o.mUrl)
,mPosition(o.mPosition)
,mPixmap(o.mPixmap)
,mOnSides(o.mOnSides)
,mTransparency(o.mTransparency)
{
}


Image& Image::operator=(const Image& o)
{
   if (&o!=this)
   {
      mUrl = o.mUrl;
      mPosition = o.mPosition;
      mPixmap = o.mPixmap;
      mOnSides = o.mOnSides;
      mTransparency = o.mTransparency;
   }
   return *this;
}


void Image::clear()
{
   mUrl = QString::null;
   mPosition = KoverArtist::Scaled;
   mPixmap = QPixmap();
   mOnSides = false;
   mTransparency = 255;
}


QPixmap Image::imageToPixmapBordersRemoved(QImage& aImg)
{
   int x, y, w=aImg.width(), h=aImg.height();
   int x0=0, x1=w-1;
   bool cont;

   if (w<=0 && h<=0) return QPixmap(0, 0);

   for (x=0,cont=true; x<w && cont; ++x)
   {
      for (y=0; y<h && cont; y+=32)
         if (aImg.pixel(x,y)!=0) cont = false;
   }
   x0 = x;

   for (x=w-1,cont=true; x>x0 && cont; --x)
   {
      for (y=0; y<h && cont; y+=32)
         if (aImg.pixel(x,y)!=0) cont = false;
   }
   x1 = x;

   QPixmap pix(x1-x0, h);
   QPainter p(&pix);
   p.drawImage(0, 0, aImg, x0, 0, x1-x0, h);
   p.end();

   return pix;
}


bool Image::load(QString aUrl)
{
   mPixmap = QPixmap(1, 1);
   bool ok = false;

   logDebug("Loading image %1").arg(aUrl);
   if (aUrl.isEmpty())
   {
      mUrl = aUrl;
      ok = true;
   }
   else if (Settings::instance()->removeImageBorders)
   {
      QImage img;
      ok = img.load(aUrl);
      if (ok) mPixmap = imageToPixmapBordersRemoved(img);
   }
   else ok = mPixmap.load(aUrl);

   if (ok) mUrl = aUrl;
   else
   {
      mUrl = QString::null;
      QMessageBox::critical(0, i18n("KoverArtist Error"),
         i18n("Cannot load image\n%1\n\n%2").arg(aUrl)
         .arg(errno ? QString(strerror(errno)) : i18n("broken image or unknown format")));
   }

   return ok;
}


void Image::toDom(QDomDocument& aDoc, QDomElement& aElem, const QString& aProjectDir) const
{
   aElem.setAttribute("position", imagePositionAsString(mPosition));
   aElem.setAttribute("onsides", mOnSides ? "true" : "false");
   aElem.setAttribute("transparency", mTransparency);

   logDebug("Image::toDom mUrl=%1, aProjectDir=%2").arg(mUrl).arg(aProjectDir);
   if (mUrl.startsWith(aProjectDir+'/'))
      aElem.appendChild(aDoc.createTextNode(mUrl.mid(aProjectDir.length()+1)));
   else aElem.appendChild(aDoc.createTextNode(mUrl));
}


bool Image::fromDom(const QDomElement& aElem, const QString& aProjectDir)
{
   mPosition = imagePositionFromString(aElem.attribute("position"));
   mOnSides = (aElem.attribute("onsides") == "true");

   if (aElem.hasAttribute("transparency"))
      mTransparency = aElem.attribute("transparency").toInt() & 255;
   else mTransparency = 255;

   QString fname = aElem.text().trimmed();
   if (fname.indexOf('/')>0) mUrl = fname;
   else if (!fname.isEmpty()) mUrl = aProjectDir+'/'+fname;
   else mUrl = QString::null;
//    std::cout<<"Image::fromDom: "<<fname<<" =("<<aProjectDir<<")=> "<<mUrl.prettyURL()<<std::endl;

   if (!mUrl.isEmpty()) load(mUrl);
   return true;
}


const char* Image::imagePositionAsString(ImagePosition ip)
{
   switch (ip)
   {
   case Centered: return "centered";
   case Tiled: return "tiled";
   case CenteredTiled: return "centered-tiled";
   case CenteredMaxpect: return "maxpect";
   case Scaled: return "scaled";
   case ScaleCrop: return "scale-crop";
   default: return "";
   }
}


ImagePosition Image::imagePositionFromString(const QString& aStr)
{
   if (aStr=="centered") return KoverArtist::Centered;
   if (aStr=="tiled") return KoverArtist::Tiled;
   if (aStr=="centered-tiled") return KoverArtist::CenteredTiled;
   if (aStr=="maxpect") return KoverArtist::CenteredMaxpect;
   if (aStr=="scaled") return KoverArtist::Scaled;
   if (aStr=="scale-crop") return KoverArtist::ScaleCrop;
   return KoverArtist::Scaled;
}


} //namespace
