/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include "effectwidget.h"

#include "effect.h"
#include "i18n.h"

#include <kcolorbutton.h>

#include <QComboBox>
#include <QHBoxLayout>
#include <QLabel>
#include <QSlider>
#include <QVBoxLayout>


EffectWidget::EffectWidget(KoverArtist::Effect* aEffect, QWidget* aParent)
:Inherited(aParent)
,mEffect(aEffect)
{
   QHBoxLayout *box = 0;

   QVBoxLayout *vbox = new QVBoxLayout(this);
   vbox->setMargin(0);
   vbox->setSpacing(2);

   box = new QHBoxLayout;
   box->setSpacing(2);
   vbox->addLayout(box);

   mCboType = new QComboBox(this);
   mCboType->addItem(i18n("No Effect"), int(KoverArtist::Effect::NONE));
   mCboType->addItem(i18n("Outline"), int(KoverArtist::Effect::OUTLINE));
   mCboType->addItem(i18n("Glow"), int(KoverArtist::Effect::GLOW));
   mCboType->addItem(i18n("Shadow"), int(KoverArtist::Effect::SHADOW));
   mCboType->addItem(i18n("Box"), int(KoverArtist::Effect::BOX));
//    mCboType->addItem(i18n("Rounded Box"), int(KoverArtist::Effect::ROUNDED_BOX));
   mCboType->setFixedHeight(mCboType->sizeHint().height());
   connect(mCboType, SIGNAL(activated(int)), this, SLOT(setType(int)));
   box->addWidget(mCboType);

   mBtnColor1 = new KColorButton(Qt::white, this);
   mBtnColor1->setFixedSize(mBtnColor1->sizeHint());
   connect(mBtnColor1, SIGNAL(changed(const QColor&)), SLOT(setColor1(const QColor&)));
   mBtnColor1->setToolTip(i18n("Change the primary effect color"));
   box->addWidget(mBtnColor1);

   mBtnColor2 = new KColorButton(Qt::white, this);
   mBtnColor2->setFixedSize(mBtnColor2->sizeHint());
   connect(mBtnColor2, SIGNAL(changed(const QColor&)), SLOT(setColor2(const QColor&)));
   mBtnColor2->setToolTip(i18n("Change the secondary effect color"));
   box->addWidget(mBtnColor2);

   box = new QHBoxLayout;
   box->setSpacing(2);
   vbox->addLayout(box);

   mLblSize = new QLabel(i18n("Size")+":", this);
   mLblSize->setFixedSize(mLblSize->sizeHint());
   box->addWidget(mLblSize);

   mLblSizeVal = new QLabel("99", this);
   mLblSizeVal->setAlignment(Qt::AlignRight);
   mLblSizeVal->setFixedSize(mLblSizeVal->sizeHint());
   box->addWidget(mLblSizeVal);

   box->addSpacing(4);

   mSldSize = new QSlider(Qt::Horizontal, this);
   mSldSize->setRange(0, 40);
   mSldSize->setPageStep(5);
   mSldSize->setTickInterval(10);
//    mSldSize = new QSlider(0, 40, 5, 0, Qt::Horizontal, this);
   connect(mSldSize, SIGNAL(valueChanged(int)), this, SLOT(setSize(int)));
   connect(mSldSize, SIGNAL(sliderReleased()), this, SLOT(sendChanged()));
   box->addWidget(mSldSize);

   box = new QHBoxLayout;
   box->setSpacing(2);
   vbox->addLayout(box);

   mLblTrans = new QLabel(i18n("Transparency") + ':', this);
   mLblTrans->setFixedSize(mLblTrans->sizeHint());
   box->addWidget(mLblTrans);

   mLblTransVal = new QLabel("999", this);
   mLblTransVal->setAlignment(Qt::AlignRight);
   mLblTransVal->setFixedSize(mLblTransVal->sizeHint());
   box->addWidget(mLblTransVal);

   box->addSpacing(4);

   mSldTrans = new QSlider(Qt::Horizontal, this);
   mSldTrans->setRange(0, 255);
   mSldTrans->setPageStep(32);
   mSldTrans->setTickInterval(64);
   connect(mSldTrans, SIGNAL(valueChanged(int)), this, SLOT(setTrans(int)));
   connect(mSldTrans, SIGNAL(sliderReleased()), this, SLOT(sendChanged()));
   box->addWidget(mSldTrans);

   vbox->addStretch(100);

   updateContents();
}


EffectWidget::~EffectWidget()
{
}


void EffectWidget::updateContents()
{
   bool upd = updatesEnabled();
   setUpdatesEnabled(false);

   mCboType->setCurrentIndex(int(mEffect->type()));
   setType(int(mEffect->type()));

   mBtnColor1->setColor(mEffect->color1());
   mBtnColor2->setColor(mEffect->color2());
   mSldSize->setValue(mEffect->size());
   mLblSizeVal->setText(QString("%1").arg(mEffect->size()));
   mSldTrans->setValue(mEffect->transparency());
   mLblTransVal->setText(QString("%1").arg(mEffect->transparency()));

   setUpdatesEnabled(upd);
}


void EffectWidget::setType(int aType)
{
   bool act = (aType!=0);
   mBtnColor1->setEnabled(act);
   mBtnColor2->setEnabled(act);
   mLblSize->setEnabled(act);
   mLblSizeVal->setEnabled(act);
   mSldSize->setEnabled(act);
   mLblTrans->setEnabled(act);
   mLblTransVal->setEnabled(act);
   mSldTrans->setEnabled(act);

   if (aType!=mEffect->type())
   {
      mEffect->setType((KoverArtist::Effect::Type)aType);
      if (updatesEnabled()) emit changed();
   }
}


void EffectWidget::setColor1(const QColor& c)
{
   if (mEffect->color1()==c) return;
   mEffect->setColor1(c);
   if (updatesEnabled()) emit changed();
}


void EffectWidget::setColor2(const QColor& c)
{
   if (mEffect->color2()==c) return;
   mEffect->setColor2(c);
   if (updatesEnabled()) emit changed();
}


void EffectWidget::setSize(int x)
{
   if (x==mEffect->size()) return;
   mEffect->setSize(x);
   mLblSizeVal->setText(QString("%1").arg(x));
}


void EffectWidget::setTrans(int x)
{
   if (x==mEffect->transparency()) return;
   mEffect->setTransparency(x);
   mLblTransVal->setText(QString("%1").arg(x));
}


void EffectWidget::sendChanged()
{
   if (updatesEnabled()) emit changed();
}
