﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/TrialComponentSource.h>
#include <aws/sagemaker/model/TrialComponentStatus.h>
#include <aws/sagemaker/model/UserContext.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SageMaker {
namespace Model {

/**
 * <p>A summary of the properties of a trial component. To get all the properties,
 * call the <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeTrialComponent.html">DescribeTrialComponent</a>
 * API and provide the <code>TrialComponentName</code>.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sagemaker-2017-07-24/TrialComponentSummary">AWS
 * API Reference</a></p>
 */
class TrialComponentSummary {
 public:
  AWS_SAGEMAKER_API TrialComponentSummary() = default;
  AWS_SAGEMAKER_API TrialComponentSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API TrialComponentSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the trial component.</p>
   */
  inline const Aws::String& GetTrialComponentName() const { return m_trialComponentName; }
  inline bool TrialComponentNameHasBeenSet() const { return m_trialComponentNameHasBeenSet; }
  template <typename TrialComponentNameT = Aws::String>
  void SetTrialComponentName(TrialComponentNameT&& value) {
    m_trialComponentNameHasBeenSet = true;
    m_trialComponentName = std::forward<TrialComponentNameT>(value);
  }
  template <typename TrialComponentNameT = Aws::String>
  TrialComponentSummary& WithTrialComponentName(TrialComponentNameT&& value) {
    SetTrialComponentName(std::forward<TrialComponentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the trial component.</p>
   */
  inline const Aws::String& GetTrialComponentArn() const { return m_trialComponentArn; }
  inline bool TrialComponentArnHasBeenSet() const { return m_trialComponentArnHasBeenSet; }
  template <typename TrialComponentArnT = Aws::String>
  void SetTrialComponentArn(TrialComponentArnT&& value) {
    m_trialComponentArnHasBeenSet = true;
    m_trialComponentArn = std::forward<TrialComponentArnT>(value);
  }
  template <typename TrialComponentArnT = Aws::String>
  TrialComponentSummary& WithTrialComponentArn(TrialComponentArnT&& value) {
    SetTrialComponentArn(std::forward<TrialComponentArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the component as displayed. If <code>DisplayName</code> isn't
   * specified, <code>TrialComponentName</code> is displayed.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  TrialComponentSummary& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const TrialComponentSource& GetTrialComponentSource() const { return m_trialComponentSource; }
  inline bool TrialComponentSourceHasBeenSet() const { return m_trialComponentSourceHasBeenSet; }
  template <typename TrialComponentSourceT = TrialComponentSource>
  void SetTrialComponentSource(TrialComponentSourceT&& value) {
    m_trialComponentSourceHasBeenSet = true;
    m_trialComponentSource = std::forward<TrialComponentSourceT>(value);
  }
  template <typename TrialComponentSourceT = TrialComponentSource>
  TrialComponentSummary& WithTrialComponentSource(TrialComponentSourceT&& value) {
    SetTrialComponentSource(std::forward<TrialComponentSourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the component. States include:</p> <ul> <li> <p>InProgress</p>
   * </li> <li> <p>Completed</p> </li> <li> <p>Failed</p> </li> </ul>
   */
  inline const TrialComponentStatus& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = TrialComponentStatus>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = TrialComponentStatus>
  TrialComponentSummary& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the component started.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  TrialComponentSummary& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the component ended.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  TrialComponentSummary& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the component was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  TrialComponentSummary& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Who created the trial component.</p>
   */
  inline const UserContext& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = UserContext>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = UserContext>
  TrialComponentSummary& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the component was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  inline bool LastModifiedTimeHasBeenSet() const { return m_lastModifiedTimeHasBeenSet; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  TrialComponentSummary& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Who last modified the component.</p>
   */
  inline const UserContext& GetLastModifiedBy() const { return m_lastModifiedBy; }
  inline bool LastModifiedByHasBeenSet() const { return m_lastModifiedByHasBeenSet; }
  template <typename LastModifiedByT = UserContext>
  void SetLastModifiedBy(LastModifiedByT&& value) {
    m_lastModifiedByHasBeenSet = true;
    m_lastModifiedBy = std::forward<LastModifiedByT>(value);
  }
  template <typename LastModifiedByT = UserContext>
  TrialComponentSummary& WithLastModifiedBy(LastModifiedByT&& value) {
    SetLastModifiedBy(std::forward<LastModifiedByT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_trialComponentName;

  Aws::String m_trialComponentArn;

  Aws::String m_displayName;

  TrialComponentSource m_trialComponentSource;

  TrialComponentStatus m_status;

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  Aws::Utils::DateTime m_creationTime{};

  UserContext m_createdBy;

  Aws::Utils::DateTime m_lastModifiedTime{};

  UserContext m_lastModifiedBy;
  bool m_trialComponentNameHasBeenSet = false;
  bool m_trialComponentArnHasBeenSet = false;
  bool m_displayNameHasBeenSet = false;
  bool m_trialComponentSourceHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_createdByHasBeenSet = false;
  bool m_lastModifiedTimeHasBeenSet = false;
  bool m_lastModifiedByHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
