/**
 * @file
 * @brief Read in the .route file generated by VPR.
 *
 * This is done in order to perform timing analysis only using the
 * --analysis option. This file set up the routing tree. It also checks that
 * routing resource nodes and placement is consistent with the
 * .net, .place, or the routing resource graph files. It is assumed
 * that the .route file's formatting matches those as described in the
 * vtr documentation. For example, the coordinates is assumed to be
 * in (x,y) format. Appropriate error messages are displayed when
 * formats are incorrect or when the routing file does not match
 * other file's information
 */

#include <iostream>
#include <fstream>
#include <cstdio>
#include <cstring>
#include <ctime>
#include <sstream>
#include <string>

#include "atom_netlist.h"
#include "atom_netlist_utils.h"
#include "rr_graph.h"
#include "vtr_assert.h"
#include "vtr_digest.h"
#include "vtr_util.h"
#include "tatum/echo_writer.hpp"
#include "vtr_log.h"
#include "check_route.h"
#include "route_common.h"
#include "vpr_types.h"
#include "globals.h"
#include "vpr_api.h"
#include "read_place.h"
#include "vpr_types.h"
#include "vpr_utils.h"
#include "vpr_error.h"
#include "place_and_route.h"
#include "timing_place.h"
#include "route_export.h"
#include "echo_files.h"
#include "route_common.h"
#include "route_tree.h"
#include "read_route.h"

#include "old_traceback.h"

/*************Functions local to this module*************/
static void process_route(const Netlist<>& net_list, std::ifstream& fp, const char* filename, int& lineno, bool is_flat);
static void process_nodes(const Netlist<>& net_list, std::ifstream& fp, ClusterNetId inet, const char* filename, int& lineno);
static void process_nets(const Netlist<>& net_list, std::ifstream& fp, ClusterNetId inet, std::string name, std::vector<std::string> input_tokens, const char* filename, int& lineno, bool is_flat);
static void process_global_blocks(const Netlist<>& net_list, std::ifstream& fp, ClusterNetId inet, const char* filename, int& lineno, bool is_flat);
static void format_coordinates(int& layer_num, int& x, int& y, std::string coord, ClusterNetId net, const char* filename, const int lineno);
static void format_pin_info(std::string& pb_name, std::string& port_name, int& pb_pin_num, std::string input);
static std::string format_name(std::string name);
static bool check_rr_graph_connectivity(RRNodeId prev_node, RRNodeId node);
void print_route(const Netlist<>& net_list, FILE* fp, bool is_flat);

/*************Global Functions****************************/

/**
 * @brief Reads in the routing file to fill in the trace.head and t_clb_opins_used data structure.
 *
 * Perform a series of verification tests to ensure the netlist,
 * placement, and routing files match
 */
bool read_route(const char* route_file, const t_router_opts& router_opts, bool verify_file_digests, bool is_flat) {
    auto& device_ctx = g_vpr_ctx.mutable_device();
    auto& place_ctx = g_vpr_ctx.placement();
    bool flat_router = router_opts.flat_routing;
    /* Begin parsing the file */
    VTR_LOG("Begin loading FPGA routing file.\n");

    std::string header_str;

    std::ifstream fp;
    fp.open(route_file);

    int lineno = 0;

    if (!fp.is_open()) {
        vpr_throw(VPR_ERROR_ROUTE, route_file, lineno,
                  "Cannot open %s routing file", route_file);
    }

    std::getline(fp, header_str);
    ++lineno;

    std::vector<std::string> header = vtr::split(header_str);
    if (header[0] == "Placement_File:" && header[2] == "Placement_ID:" && header[3] != place_ctx.placement_id) {
        auto msg = vtr::string_fmt(
            "Placement file %s specified in the routing file"
            " does not match the loaded placement (ID %s != %s)",
            header[1].c_str(), header[3].c_str(), place_ctx.placement_id.c_str());
        if (verify_file_digests) {
            vpr_throw(VPR_ERROR_ROUTE, route_file, lineno, msg.c_str());
        } else {
            VTR_LOGF_WARN(route_file, lineno, "%s\n", msg.c_str());
        }
    }

    /*Allocate necessary routing structures*/
    alloc_and_load_rr_node_route_structs();
    const Netlist<>& router_net_list = (flat_router) ? (const Netlist<>&)g_vpr_ctx.atom().nlist : (const Netlist<>&)g_vpr_ctx.clustering().clb_nlist;
    init_route_structs(router_net_list,
                       router_opts.bb_factor,
                       router_opts.has_choke_point,
                       flat_router);

    /*Check dimensions*/
    std::getline(fp, header_str);
    ++lineno;
    header.clear();
    header = vtr::split(header_str);
    if (header[0] == "Array" && header[1] == "size:" && (vtr::atou(header[2].c_str()) != device_ctx.grid.width() || vtr::atou(header[4].c_str()) != device_ctx.grid.height())) {
        vpr_throw(VPR_ERROR_ROUTE, route_file, lineno,
                  "Device dimensions %sx%s specified in the routing file does not match given %dx%d ",
                  header[2].c_str(), header[4].c_str(), device_ctx.grid.width(), device_ctx.grid.height());
    }

    /* Read in every net */
    process_route(router_net_list, fp, route_file, lineno, is_flat);

    fp.close();

    /*Correctly set up the clb opins*/
    FourAryHeap small_heap;
    small_heap.init_heap(device_ctx.grid);
    if (!flat_router) {
        reserve_locally_used_opins(&small_heap, router_opts.initial_pres_fac,
                                   router_opts.acc_fac, false, flat_router);
    }
    recompute_occupancy_from_scratch(router_net_list,
                                     flat_router);

    /* Note: This pres_fac is not necessarily correct since it isn't the first routing iteration*/
    OveruseInfo overuse_info(device_ctx.rr_graph.num_nodes());
    pathfinder_update_acc_cost_and_overuse_info(router_opts.acc_fac, overuse_info);
    if (!flat_router) {
        reserve_locally_used_opins(&small_heap, router_opts.initial_pres_fac,
                                   router_opts.acc_fac, true, flat_router);
    }

    /* Finished loading in the routing, now check it*/
    recompute_occupancy_from_scratch(router_net_list,
                                     flat_router);
    bool is_feasible = feasible_routing();

    VTR_LOG("Finished loading route file\n");

    return is_feasible;
}

///@brief Walks through every net and add the routing appropriately
static void process_route(const Netlist<>& net_list, std::ifstream& fp, const char* filename, int& lineno, bool is_flat) {
    std::string input;
    std::vector<std::string> tokens;
    while (std::getline(fp, input)) {
        ++lineno;
        tokens.clear();
        tokens = vtr::split(input);
        if (tokens.empty()) {
            continue; //Skip blank lines
        } else if (tokens[0][0] == '#') {
            continue; //Skip commented lines
        } else if (tokens[0] == "Net") {
            ClusterNetId inet(atoi(tokens[1].c_str()));
            process_nets(net_list, fp, inet, tokens[2], tokens, filename, lineno, is_flat);
        }
    }

    tokens.clear();
}

///@brief Check if the net is global or not, and process appropriately
static void process_nets(const Netlist<>& net_list, std::ifstream& fp, ClusterNetId inet, std::string name, std::vector<std::string> input_tokens, const char* filename, int& lineno, bool is_flat) {
    if (input_tokens.size() > 3 && input_tokens[3] == "global"
        && input_tokens[4] == "net" && input_tokens[5] == "connecting:") {
        /* Global net.  Never routed. */
        if (!net_list.net_is_ignored(inet)) {
            vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                      "Net %lu should be a global net", size_t(inet));
        }
        /*erase an extra colon for global nets*/
        name.erase(name.end() - 1);
        name = format_name(name);

        if (0 != net_list.net_name(inet).compare(name)) {
            vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                      "Net name %s for net number %lu specified in the routing file does not match given %s. Running without flat routing; if this file was created with flat routing, re-run vpr with the --flat_routing option",
                      name.c_str(), size_t(inet), net_list.net_name(inet).c_str());
        }

        process_global_blocks(net_list, fp, inet, filename, lineno, is_flat);
    } else {
        /* Not a global net */
        if (net_list.net_is_ignored(inet)) {
            VTR_LOG_WARN("Net %lu (%s) is marked as global in the netlist, but is non-global in the .route file\n", size_t(inet), net_list.net_name(inet).c_str());
        }

        name = format_name(name);

        if (0 != net_list.net_name(inet).compare(name)) {
            vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                      "Net name %s for net number %lu specified in the routing file does not match given %s. Running without flat routing; if this file was created with flat routing, re-run vpr with the --flat_routing option",
                      name.c_str(), size_t(inet), net_list.net_name(inet).c_str());
        }

        process_nodes(net_list, fp, inet, filename, lineno);
    }
    input_tokens.clear();
    return;
}

static void process_nodes(const Netlist<>& net_list, std::ifstream& fp, ClusterNetId inet, const char* filename, int& lineno) {
    /* Not a global net. Goes through every node and add it into trace.head*/
    auto& device_ctx = g_vpr_ctx.mutable_device();
    const auto& rr_graph = device_ctx.rr_graph;
    auto& route_ctx = g_vpr_ctx.mutable_routing();
    const auto& grid_blocks = g_vpr_ctx.placement().grid_blocks();

    t_trace* head_ptr = nullptr;
    t_trace* tptr = nullptr;

    /*remember the position of the last line in order to go back*/
    std::streampos oldpos = fp.tellg();
    int inode, layer_num, x, y, layer_num2, x2, y2, ptc, switch_id, net_pin_index, offset;
    std::string prev_type;
    int node_count = 0;
    std::string input;
    std::vector<std::string> tokens;
    RRNodeId prev_node(-1);

    /*Walk through every line that begins with Node:*/
    while (std::getline(fp, input)) {
        ++lineno;

        tokens.clear();
        tokens = vtr::split(input);

        if (tokens.empty()) {
            continue; /*Skip blank lines*/
        } else if (tokens[0][0] == '#') {
            continue; /*Skip commented lines*/
        } else if (tokens[0] == "Net") {
            /*End of the nodes list,
             *  return by moving the position of next char of input stream to be before net*/
            fp.seekg(oldpos);
            break;
        } else if (input == "\n\nUsed in local cluster only, reserved one CLB pin\n\n") {
            if (net_list.net_sinks(inet).size() != 0) {
                vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                          "Net %d should be used in local cluster only, reserved one CLB pin");
            }
            break;
        } else if (tokens[0] == "Node:") {
            /*An actual line, go through each node and add it to the route tree*/
            inode = atoi(tokens[1].c_str());

            /*First node needs to be source. It is isolated to correctly set heap head.*/
            if (node_count == 0 && tokens[2] != "SOURCE") {
                vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                          "First node in routing has to be a source type");
            }

            /*Check node types if match rr graph*/
            if (tokens[2] != rr_graph.node_type_string(RRNodeId(inode))) {
                vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                          "Node %d has a type that does not match the RR graph", inode);
            }

            format_coordinates(layer_num, x, y, tokens[3], inet, filename, lineno);
            auto rr_node = RRNodeId(inode);

            if (tokens[4] == "to") {
                format_coordinates(layer_num2, x2, y2, tokens[5], inet, filename, lineno);
                if (rr_graph.node_xlow(rr_node) != x || rr_graph.node_xhigh(rr_node) != x2 || rr_graph.node_yhigh(rr_node) != y2 || rr_graph.node_ylow(rr_node) != y) {
                    vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                              "The coordinates of node %d does not match the rr graph", inode);
                }
                offset = 2;

                /* Check for connectivity, this throws an exception when a dangling net is encountered in the routing file */
                bool legal_node = check_rr_graph_connectivity(prev_node, rr_node);
                if (!legal_node) {
                    vpr_throw(VPR_ERROR_ROUTE, filename, lineno, "Dangling branch at net %lu, nodes %d -> %d: %s", inet, prev_node, inode, input.c_str());
                }
                prev_node = rr_node;
            } else {
                if (rr_graph.node_xlow(rr_node) != x || rr_graph.node_xhigh(rr_node) != x || rr_graph.node_yhigh(rr_node) != y || rr_graph.node_ylow(rr_node) != y) {
                    vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                              "The coordinates of node %d does not match the rr graph", inode);
                }
                offset = 0;

                bool legal_node = check_rr_graph_connectivity(prev_node, rr_node);
                prev_node = rr_node;
                if (!legal_node) {
                    vpr_throw(VPR_ERROR_ROUTE, filename, lineno, "Dangling branch at net %lu, nodes %d -> %d: %s", inet, prev_node, inode, input.c_str());
                }
                prev_node = rr_node;
            }

            /* Verify types and ptc*/
            if (tokens[2] == "SOURCE" || tokens[2] == "SINK" || tokens[2] == "OPIN" || tokens[2] == "IPIN") {
                const auto& type = device_ctx.grid.get_physical_type({x, y, layer_num});
                if (tokens[4 + offset] == "Pad:" && !is_io_type(type)) {
                    vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                              "Node %d is of the wrong type", inode);
                }
            } else if (tokens[2] == "CHANX" || tokens[2] == "CHANY") {
                if (tokens[4 + offset] != "Track:") {
                    vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                              "A %s node have to have track info", tokens[2].c_str());
                }
            }

            ptc = atoi(tokens[5 + offset].c_str());
            if (rr_graph.node_ptc_num(RRNodeId(inode)) != ptc) {
                vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                          "The ptc num of node %d does not match the rr graph, Running without flat routing; "
                          "if this file was created with flat routing, re-run vpr with the --flat_routing option",
                          inode);
            }

            /*Process switches and pb pin info if it is ipin or opin type*/
            if (tokens[6 + offset] != "Switch:") {
                /*This is an opin or ipin, process its pin nums*/
                auto type = device_ctx.grid.get_physical_type({x, y, layer_num});
                if (!is_io_type(type) && (tokens[2] == "IPIN" || tokens[2] == "OPIN")) {
                    int pin_num = rr_graph.node_pin_num(RRNodeId(inode));
                    int width_offset = device_ctx.grid.get_width_offset({x, y, layer_num});
                    int height_offset = device_ctx.grid.get_height_offset({x, y, layer_num});
                    auto physical_tile = device_ctx.grid.get_physical_type({x, y, layer_num});
                    auto [sub_tile, sub_tile_rel_cap] = get_sub_tile_from_pin_physical_num(physical_tile, pin_num);
                    int sub_tile_offset = sub_tile->capacity.low + sub_tile_rel_cap;

                    ClusterBlockId iblock = grid_blocks.block_at_location({x - width_offset, y - height_offset, sub_tile_offset, layer_num});
                    VTR_ASSERT(iblock);

                    const t_pb_graph_pin* pb_pin;

                    if (is_pin_on_tile(physical_tile, pin_num)) {
                        pb_pin = get_pb_graph_node_pin_from_block_pin(iblock, pin_num);
                    } else {
                        pb_pin = get_pb_pin_from_pin_physical_num(physical_tile, pin_num);
                    }

                    t_pb_type* pb_type = pb_pin->parent_node->pb_type;

                    std::string pb_name, port_name;
                    int pb_pin_num;

                    format_pin_info(pb_name, port_name, pb_pin_num, tokens[6 + offset]);

                    if (pb_name != pb_type->name || port_name != pb_pin->port->name || pb_pin_num != pb_pin->pin_number) {
                        vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                                  "%d node does not have correct pins", inode);
                    }
                } else {
                    vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                              "%d node does not have correct pins", inode);
                }
                switch_id = atoi(tokens[8 + offset].c_str());
            } else {
                switch_id = atoi(tokens[7 + offset].c_str());
            }

            /* Process net pin index for sinks                                   *
             * If you have an old .route file, it may not have this information  *
             * Please check your .route file to see if it contains Net_pin_index *
             * information for sinks. If not, plrase re-generate the routing.    */
            if (tokens[2] == "SINK") {
                if (tokens[8 + offset] == "Net_pin_index:") {
                    net_pin_index = atoi(tokens[9 + offset].c_str());
                } else {
                    vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                              "%d (sink) node does not have net pin index. If you are using an old .route file without this information, please re-generate the routing.", inode);
                }
            } else {
                net_pin_index = OPEN; // net pin index is invalid for non-SINKs
            }

            /* Allocate and load correct values to trace.head */
            if (node_count == 0) {
                head_ptr = alloc_trace_data();
                head_ptr->index = inode;
                head_ptr->net_pin_index = net_pin_index;
                head_ptr->iswitch = switch_id;
                head_ptr->next = nullptr;
                tptr = head_ptr;
                node_count++;
            } else {
                tptr->next = alloc_trace_data();
                tptr = tptr->next;
                tptr->index = inode;
                tptr->net_pin_index = net_pin_index;
                tptr->iswitch = switch_id;
                tptr->next = nullptr;
                node_count++;
            }
        }
        /* stores last line so can easily go back to read */
        oldpos = fp.tellg();
    }

    /* Convert to route_tree after reading */
    VTR_ASSERT(validate_traceback(head_ptr));
    route_ctx.route_trees[inet] = TracebackCompat::traceback_to_route_tree(head_ptr);
    free_traceback(head_ptr);
}

/**
 * @brief This function goes through all the blocks in a global net and verify
 *        it with the clustered netlist and the placement
 */
static void process_global_blocks(const Netlist<>& net_list, std::ifstream& fp, ClusterNetId inet, const char* filename, int& lineno, bool is_flat) {
    std::string block, bnum_str;
    int layer_num, x, y;
    std::vector<std::string> tokens;

    std::streampos oldpos = fp.tellg();
    /*Walk through every block line*/
    while (std::getline(fp, block)) {
        ++lineno;
        tokens.clear();
        tokens = vtr::split(block);

        if (tokens.empty()) {
            continue; /*Skip blank lines*/
        } else if (tokens[0][0] == '#') {
            continue; /*Skip commented lines*/
        } else if (tokens[0] != "Block") {
            /*End of blocks, go back to reading nets*/
            fp.seekg(oldpos);
            return;
        } else {
            format_coordinates(layer_num, x, y, tokens[4], inet, filename, lineno);

            /*remove ()*/
            bnum_str = format_name(tokens[2]);
            /*remove #*/
            bnum_str.erase(bnum_str.begin());
            ParentBlockId bnum(atoi(bnum_str.c_str()));

            /*Check for name, coordinate, and pins*/
            if (0 != net_list.block_name(bnum).compare(tokens[1])) {
                vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                          "Block %s for block number %lu specified in the routing file does not match given %s",
                          tokens[1].c_str(), size_t(bnum), net_list.block_name(bnum).c_str());
            }

            auto block_loc = get_block_loc(bnum, is_flat);

            if (block_loc.loc.x != x || block_loc.loc.y != y) {
                vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                          "The placement coordinates (%d,%d,%d) of %zu block does not match given (%d,%d,%d)",
                          x, y, layer_num, size_t(bnum),
                          block_loc.loc.x, block_loc.loc.y, block_loc.loc.layer);
            }

            auto pin_class = get_class_range_for_block(bnum, is_flat);
            if (pin_class.low > atoi(tokens[7].c_str()) || pin_class.high < atoi(tokens[7].c_str())) {
                vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                          "The pin class %d of %lu net does not match given ",
                          atoi(tokens[7].c_str()), size_t(inet));
            }
        }
        oldpos = fp.tellg();
    }
}

///@brief Parse coordinates in the form of (x,y) into correct x and y values
static void format_coordinates(int& layer_num, int& x, int& y, std::string coord, ClusterNetId net, const char* filename, const int lineno) {
    coord = format_name(coord);

    std::stringstream coord_stream(coord);
    std::vector<int> coords;
    int tmp_coord;
    while (coord_stream >> tmp_coord) {
        coords.push_back(tmp_coord);
        coord_stream.ignore(1, ',');
    }
    if (coords.size() != 2 && coords.size() != 3) {
        vpr_throw(VPR_ERROR_ROUTE, filename, lineno,
                  "Net %lu has coordinates that is not in the form (layer_num,x,y)", size_t(net));
    }

    if (coords.size() == 2) {
        layer_num = 0;
        x = coords[0];
        y = coords[1];
    } else {
        VTR_ASSERT(coords.size() == 3);
        x = coords[0];
        y = coords[1];
        layer_num = coords[2];
    }
}

/**
 * @brief Parse the pin info in the form of pb_name.port_name[pb_pin_num]
 *        into its appropriate variables
 */
static void format_pin_info(std::string& pb_name, std::string& port_name, int& pb_pin_num, std::string input) {
    std::stringstream pb_info(input);
    std::getline(pb_info, pb_name, '.');
    std::getline(pb_info, port_name, '[');
    pb_info >> pb_pin_num;
    if (!pb_info) {
        VPR_FATAL_ERROR(VPR_ERROR_ROUTE,
                        "Format of this pin info %s is incorrect",
                        input.c_str());
    }
}

///@brief Return actual name by extracting it out of the form of (name)
static std::string format_name(std::string name) {
    if (name.length() > 2) {
        name.erase(name.begin());
        name.erase(name.end() - 1);
        return name;
    } else {
        VPR_FATAL_ERROR(VPR_ERROR_ROUTE,
                        "%s should be enclosed by parenthesis",
                        name.c_str());
        return nullptr;
    }
}

/* Check for a discontinuity in the trace looking at the rr_graph
 * This is a check to ensure illegal dangling branches are caught before the program moves further
 * @returns false if there is a discontinuity */
static bool check_rr_graph_connectivity(RRNodeId prev_node, RRNodeId node) {
    // Check if its the first node of the series
    if (prev_node == RRNodeId(-1)) return true;

    // Check if the nodes are the same, which is illegal
    if (prev_node == node) return false;

    auto& device_ctx = g_vpr_ctx.device();
    const auto& rr_graph = device_ctx.rr_graph;
    // If it's starting a new sub branch this is ok
    if (rr_graph.node_type(prev_node) == SINK) return true;

    for (RREdgeId edge : rr_graph.edge_range(prev_node)) {
        //If the sink node is reachable by previous node return true
        if (rr_graph.rr_nodes().edge_sink_node(edge) == node) {
            return true;
        }

        // If there are any non-configurable branches return true
        short edge_switch = rr_graph.rr_nodes().edge_switch(edge);
        if (!(rr_graph.rr_switch_inf(RRSwitchId(edge_switch)).configurable())) return true;
    }

    // If it's part of a non configurable node list, return true
    if (rr_graph.num_non_configurable_edges(node) != 0) {
        return true;
    }

    return false;
}

void print_route(const Netlist<>& net_list,
                 FILE* fp,
                 bool is_flat) {
    const auto& grid_blocks = g_vpr_ctx.placement().grid_blocks();
    auto& device_ctx = g_vpr_ctx.device();
    const auto& rr_graph = device_ctx.rr_graph;
    auto& route_ctx = g_vpr_ctx.mutable_routing();

    if (route_ctx.route_trees.empty())
        return; //Only if routing exists

    for (auto net_id : net_list.nets()) {
        if (!net_list.net_is_ignored(net_id)) {
            fprintf(fp, "\n\nNet %zu (%s)\n\n", size_t(net_id), net_list.net_name(net_id).c_str());
            if (net_list.net_sinks(net_id).size() == false) {
                fprintf(fp, "\n\nUsed in local cluster only, reserved one CLB pin\n\n");
            } else {
                if (!route_ctx.route_trees[net_id])
                    continue;

                t_trace* head = TracebackCompat::traceback_from_route_tree(route_ctx.route_trees[net_id].value());
                t_trace* tptr = head;

                while (tptr != nullptr) {
                    RRNodeId inode = RRNodeId(tptr->index);
                    t_rr_type rr_type = rr_graph.node_type(inode);
                    int ilow = rr_graph.node_xlow(inode);
                    int jlow = rr_graph.node_ylow(inode);
                    int layer_num = rr_graph.node_layer(inode);

                    fprintf(fp, "Node:\t%zu\t%6s (%d,%d,%d) ", size_t(inode),
                            rr_graph.node_type_string(inode), ilow, jlow, layer_num);

                    if ((ilow != rr_graph.node_xhigh(inode))
                        || (jlow != rr_graph.node_yhigh(inode)))
                        fprintf(fp, "to (%d,%d,%d) ", rr_graph.node_xhigh(inode),
                                rr_graph.node_yhigh(inode), layer_num);

                    switch (rr_type) {
                        case IPIN:
                        case OPIN:
                            if (is_io_type(device_ctx.grid.get_physical_type({ilow, jlow, layer_num}))) {
                                fprintf(fp, " Pad: ");
                            } else { /* IO Pad. */
                                fprintf(fp, " Pin: ");
                            }
                            break;

                        case CHANX:
                        case CHANY:
                            fprintf(fp, " Track: ");
                            break;

                        case SOURCE:
                        case SINK:
                            if (is_io_type(device_ctx.grid.get_physical_type({ilow, jlow, layer_num}))) {
                                fprintf(fp, " Pad: ");
                            } else { /* IO Pad. */
                                fprintf(fp, " Class: ");
                            }
                            break;

                        default:
                            VPR_FATAL_ERROR(VPR_ERROR_ROUTE,
                                            "in print_route: Unexpected traceback element type: %d (%s).\n",
                                            rr_type, rr_graph.node_type_string(inode));
                            break;
                    }

                    fprintf(fp, "%d  ", rr_graph.node_ptc_num(inode));

                    auto physical_tile = device_ctx.grid.get_physical_type({ilow, jlow, layer_num});
                    if (!is_io_type(physical_tile) && (rr_type == IPIN || rr_type == OPIN)) {
                        int pin_num = rr_graph.node_pin_num(inode);
                        int xoffset = device_ctx.grid.get_width_offset({ilow, jlow, layer_num});
                        int yoffset = device_ctx.grid.get_height_offset({ilow, jlow, layer_num});
                        auto [sub_tile, sub_tile_rel_cap] = get_sub_tile_from_pin_physical_num(physical_tile, pin_num);
                        int sub_tile_offset = sub_tile->capacity.low + sub_tile_rel_cap;

                        ClusterBlockId iblock = grid_blocks.block_at_location({ilow - xoffset, jlow - yoffset,
                                                                               sub_tile_offset, layer_num});
                        VTR_ASSERT(iblock);
                        const t_pb_graph_pin* pb_pin;
                        if (is_pin_on_tile(physical_tile, pin_num)) {
                            pb_pin = get_pb_graph_node_pin_from_block_pin(iblock, pin_num);
                        } else {
                            pb_pin = get_pb_pin_from_pin_physical_num(physical_tile, pin_num);
                        }
                        const t_pb_type* pb_type = pb_pin->parent_node->pb_type;
                        fprintf(fp, " %s.%s[%d] ", pb_type->name, pb_pin->port->name, pb_pin->pin_number);
                    }

                    /* Uncomment line below if you're debugging and want to see the switch types *
                     * used in the routing.                                                      */
                    fprintf(fp, "Switch: %d", int(tptr->iswitch));

                    //Save net pin index for sinks
                    if (rr_type == SINK) {
                        fprintf(fp, " Net_pin_index: %d", tptr->net_pin_index);
                    }

                    fprintf(fp, "\n");

                    tptr = tptr->next;
                }

                free_traceback(head);
            }
        } else { /* Global net.  Never routed. */
            fprintf(fp, "\n\nNet %zu (%s): global net connecting:\n\n", size_t(net_id),
                    net_list.net_name(net_id).c_str());

            for (auto pin_id : net_list.net_pins(net_id)) {
                ParentBlockId block_id = net_list.pin_block(pin_id);
                int iclass = get_block_pin_class_num(block_id, pin_id, is_flat);
                t_block_loc blk_loc;
                blk_loc = get_block_loc(block_id, is_flat);
                fprintf(fp, "Block %s (#%zu) at (%d,%d,%d), Pin class %d.\n",
                        net_list.block_name(block_id).c_str(),
                        size_t(block_id),
                        blk_loc.loc.x,
                        blk_loc.loc.y,
                        blk_loc.loc.layer,
                        iclass);
            }
        }
    }
}

/* Prints out the routing to file route_file.  */
void print_route(const Netlist<>& net_list,
                 const char* placement_file,
                 const char* route_file,
                 bool is_flat) {
    FILE* fp;

    fp = fopen(route_file, "w");

    auto& place_ctx = g_vpr_ctx.placement();
    auto& device_ctx = g_vpr_ctx.device();
    auto& route_ctx = g_vpr_ctx.mutable_routing();

    fprintf(fp, "Placement_File: %s Placement_ID: %s\n", placement_file, place_ctx.placement_id.c_str());

    fprintf(fp, "Array size: %zu x %zu logic blocks.\n", device_ctx.grid.width(), device_ctx.grid.height());
    fprintf(fp, "\nRouting:");

    print_route(net_list, fp, is_flat);

    fclose(fp);

    //Save the digest of the route file
    route_ctx.routing_id = vtr::secure_digest_file(route_file);
}
