//
// Copyright 2016 Pixar
//
// Licensed under the terms set forth in the LICENSE.txt file available at
// https://openusd.org/license.
//
#ifndef PXR_IMAGING_HD_ST_RESOURCE_BINDER_H
#define PXR_IMAGING_HD_ST_RESOURCE_BINDER_H

#include "pxr/pxr.h"
#include "pxr/imaging/hdSt/api.h"
#include "pxr/imaging/hdSt/binding.h"
#include "pxr/imaging/hd/version.h"
#include "pxr/imaging/hgi/capabilities.h"
#include "pxr/imaging/hgi/handle.h"
#include "pxr/base/tf/token.h"
#include "pxr/base/tf/stl.h"
#include "pxr/base/tf/staticTokens.h"

#include <memory>

PXR_NAMESPACE_OPEN_SCOPE


class HdStDrawItem;
struct HgiResourceBindingsDesc;

using HdStBufferResourceSharedPtr = 
    std::shared_ptr<class HdStBufferResource>;
using HdStBufferArrayRangeSharedPtr =
    std::shared_ptr<class HdStBufferArrayRange>;

using HdStShaderCodeSharedPtr = std::shared_ptr<class HdStShaderCode>;
using HdStShaderCodeSharedPtrVector = std::vector<HdStShaderCodeSharedPtr>;
using HdStBindingRequestVector = std::vector<class HdStBindingRequest>;

using HgiTextureHandle = HgiHandle<class HgiTexture>;
using HgiSamplerHandle = HgiHandle<class HgiSampler>;
using HgiShaderProgramHandle = HgiHandle<class HgiShaderProgram>;

/// Suffixes appended to material param names for a binding name.
///
#define HDST_RESOURCE_BINDING_SUFFIX_TOKENS     \
    ((fallback, "_fallback"))                   \
    ((samplingTransform, "_samplingTransform")) \
    ((layout, "_layout"))                       \
    ((texture, "_texture"))                     \
    ((valid, "_valid"))

TF_DECLARE_PUBLIC_TOKENS(HdSt_ResourceBindingSuffixTokens,
                         HDST_RESOURCE_BINDING_SUFFIX_TOKENS);

/// \class HdSt_ResourceBinder
/// 
/// A helper class to maintain all vertex/buffer/uniform binding points to be
/// used for both codegen time and rendering time.
///
/// Storm uses 6 different types of coherent buffers.
///
/// 1. Constant buffer
///   constant primvars, which is uniform for all instances/elements/vertices.
///     ex. transform, object color
//    [SSBO, BindlessUniform]
///
/// 2. Instance buffer
///   instance primvars, one-per-instance.
///     ex. translate/scale/rotate, instanceIndices
//    [SSBO, BindlessUniform]
///
/// 3. Element buffer
///   element primvars. one-per-element (face, line).
///     ex. face color
///   [SSBO]
///
/// 4. Vertex buffer
///   vertex primvars. one-per-vertex.
///     ex. positions, normals, vertex color
///   [VertexAttribute]
///
/// 5. Index buffer
///   points/triangles/quads/lines/patches indices.
///     ex. indices, primitive param.
///   [IndexAttribute, SSBO]
///
/// 6. DrawIndex buffer
///   draw command data. one-per-drawitem (gl_DrawID equivalent)
///     ex. drawing coordinate, instance counts
///   [VertexAttribute]
///
///
///
/// For instance index indirection, three bindings are needed:
///
///    +-----------------------------------------+
///    |  instance indices buffer resource       | <-- <arrayBinding>
///    +-----------------------------------------+
///    |* culled instance indices buffer resource| <-- <culledArrayBinding>
///    +-----------------------------------------+  (bindless uniform
///                  ^   ^      ^                      or SSBO)
/// DrawCalls +---+  |   |      |
///          0|   |---   |      |
///           +---+      |      |
///          1|   |-------      |
///           +---+             |
///          2|   |--------------
///           +---+
///             ^
///             --- <baseBinding>
///                  (immediate:uniform, indirect:vertex attrib)
///
/// (*) GPU frustum culling shader shuffles instance indices into
///     culled indices buffer.
///
///
/// HdSt_ResourceBinder also takes custom bindings.
///
/// Custom bindings are used to manage bindable resources for
/// glsl shader code which is not itself generated by codegen.
///
/// For each custom binding, codegen will emit a binding definition
/// that can be used as the value of a glsl \a binding or
/// \a location layout qualifier.
///
/// e.g. Adding a custom binding of 2 for "paramsBuffer", will
/// cause codegen to emit the definition:
/// \code
/// #define paramsBuffer_Binding 2
/// \endcode
/// which can be used in a custom glsl resource declaration as:
/// \code
/// layout (binding = paramsBuffer_Binding) buffer ParamsBuffer { ... };
/// \endcode
///
class HdSt_ResourceBinder {
public:
    /// binding metadata for codegen
    class MetaData {
    public:
        MetaData() : instancerNumLevels(0) {}

        typedef size_t ID;
        /// Returns the hash value of this metadata.
        HDST_API
        ID ComputeHash() const;

        // -------------------------------------------------------------------
        // for a primvar in interleaved buffer array (Constant, ShaderData)
        struct StructEntry {
            StructEntry(TfToken const &name,
                        TfToken const &dataType,
                        int offset, int arraySize,
                        bool concatenateNames=false)
                : name(name)
                , dataType(dataType)
                , offset(offset)
                , arraySize(arraySize)
                , concatenateNames(concatenateNames)
            {}

            TfToken name;
            TfToken dataType;
            int offset;
            int arraySize;
            bool concatenateNames;

            bool operator < (StructEntry const &other) const {
                return offset < other.offset;
            }
        };
        struct StructBlock {
            StructBlock(TfToken const &name, int arraySize = 1)
                : blockName(name)
                , arraySize(arraySize) {}
            TfToken blockName;
            std::vector<StructEntry> entries;
            int arraySize;
        };
        using StructBlockBinding = std::map<HdStBinding, StructBlock>;

        // -------------------------------------------------------------------
        // for a primvar in non-interleaved buffer array (Vertex, Element, ...)
        struct Primvar {
            Primvar() {}
            Primvar(TfToken const &name, TfToken const &dataType)
                : name(name), dataType(dataType) {}
            TfToken name;
            TfToken dataType;
        };
        using PrimvarBinding = std::map<HdStBinding, Primvar>;

        // -------------------------------------------------------------------
        // for a face-varying primvar in non-interleaved buffer array
        struct FvarPrimvar : Primvar {
            FvarPrimvar() : channel(0) {}
            FvarPrimvar(TfToken const &name, TfToken const &dataType, 
                        int channel)
                : Primvar(name, dataType), channel(channel) {}
            int channel;
        };
        using FvarPrimvarBinding = std::map<HdStBinding, FvarPrimvar>;

        // -------------------------------------------------------------------
        // for instance primvars
        struct NestedPrimvar {
            NestedPrimvar() {}
            NestedPrimvar(TfToken const &name, TfToken const &dataType,
                        int level)
                : name(name), dataType(dataType), level(level) {}
            TfToken name;
            TfToken dataType;
            int level;
        };
        using NestedPrimvarBinding = std::map<HdStBinding, NestedPrimvar>;

        // -------------------------------------------------------------------
        // for shader parameter accessors
        struct ShaderParameterAccessor {
             ShaderParameterAccessor() {}
             ShaderParameterAccessor(TfToken const &name,
                                     TfToken const &dataType,
                                     std::string const &swizzle=std::string(),
                                     TfTokenVector const &inPrimvars=TfTokenVector(),
                                     bool const isPremultiplied=false,
                                     bool const processTextureFallbackValue=false,
                                     size_t const arrayOfTexturesSize=0)
                 : name(name), dataType(dataType), swizzle(swizzle),
                  inPrimvars(inPrimvars), isPremultiplied(isPremultiplied),
                  processTextureFallbackValue(processTextureFallbackValue),
                  arrayOfTexturesSize(arrayOfTexturesSize) {}
             TfToken name;        // e.g. Kd
             TfToken dataType;    // e.g. vec4
             std::string swizzle; // e.g. xyzw
             TfTokenVector inPrimvars; // for primvar renaming and texture
                                       // coordinates,
             bool isPremultiplied; // indicates if texture parameter has been 
                                   // pre-multiplied by alpha on the CPU
             bool processTextureFallbackValue; // use NAME_fallback from shader
                                               // bar if texture is not valid
                                               // (determineed from bool
                                               // NAME_valid or bindless
                                               // handle), only supported for
                                               // material shader and for uv
                                               // and field textures.
            size_t arrayOfTexturesSize; // If the shaderParameterAccessor is 
                                     // associated with an HdStBinding of type 
                                     // ARRAY_OF_TEXTURE_2D or 
                                     // BINDLESS_ARRAY_OF_TEXTURE_2D, this will 
                                     // indicate the size of the array.
        };
        using ShaderParameterBinding =
                std::map<HdStBinding, ShaderParameterAccessor>;

        // -------------------------------------------------------------------
        // for specific buffer array (drawing coordinate, instance indices)
        struct BindingDeclaration {
            BindingDeclaration() {}
            BindingDeclaration(TfToken const &name,
                               TfToken const &dataType,
                               HdStBinding binding,
                               bool isWritable = false)
                : name(name)
                , dataType(dataType)
                , binding(binding)
                , isWritable(isWritable) { }

            TfToken name;
            TfToken dataType;
            HdStBinding binding;
            bool isWritable;
        };

        // -------------------------------------------------------------------
        // for accessing drawingCoord directly from a buffer
        struct DrawingCoordBufferBinding {
            DrawingCoordBufferBinding()
                : bufferName()
                , offset(0)
                , stride(0) { }
            DrawingCoordBufferBinding(TfToken const &bufferName,
                                      uint32_t offset,
                                      uint32_t stride)
                : bufferName(bufferName)
                , offset(offset)
                , stride(stride) { }
            TfToken bufferName;
            uint32_t offset;
            uint32_t stride;
        };
        DrawingCoordBufferBinding drawingCoordBufferBinding;

        // -------------------------------------------------------------------

        StructBlockBinding constantData;
        StructBlockBinding shaderData;
        StructBlockBinding topologyVisibilityData;
        PrimvarBinding elementData;
        PrimvarBinding vertexData;
        PrimvarBinding varyingData;
        FvarPrimvarBinding fvarData;
        PrimvarBinding computeReadWriteData;
        PrimvarBinding computeReadOnlyData;
        NestedPrimvarBinding instanceData;
        int instancerNumLevels;

        ShaderParameterBinding shaderParameterBinding;

        BindingDeclaration drawingCoord0Binding;
        BindingDeclaration drawingCoord1Binding;
        BindingDeclaration drawingCoord2Binding;
        BindingDeclaration drawingCoordIBinding;
        BindingDeclaration instanceIndexArrayBinding;
        BindingDeclaration culledInstanceIndexArrayBinding;
        BindingDeclaration instanceIndexBaseBinding;
        BindingDeclaration primitiveParamBinding;
        BindingDeclaration tessFactorsBinding;
        BindingDeclaration edgeIndexBinding;
        BindingDeclaration coarseFaceIndexBinding;
        BindingDeclaration indexBufferBinding;
        std::vector<BindingDeclaration> fvarPatchParamBindings;
        std::vector<BindingDeclaration> fvarIndicesBindings;

        StructBlockBinding customInterleavedBindings;
        std::vector<BindingDeclaration> customBindings;
    };

    /// Constructor.
    HDST_API
    HdSt_ResourceBinder();

    /// Assign all binding points used in drawitem and custom bindings.
    /// Returns metadata to be used for codegen.
    HDST_API
    void ResolveBindings(HdStDrawItem const *drawItem,
                         HdStShaderCodeSharedPtrVector const &shaders,
                         MetaData *metaDataOut,
                         MetaData::DrawingCoordBufferBinding const &dcBinding,
                         bool instanceDraw,
                         HdStBindingRequestVector const &customBindings,
                         HgiCapabilities const *capabilities);

    /// Assign all binding points used in computation.
    /// Returns metadata to be used for codegen.
    HDST_API
    void ResolveComputeBindings(HdBufferSpecVector const &readWriteBufferSpecs,
                                HdBufferSpecVector const &readOnlyBufferSpecs,
                                HdStShaderCodeSharedPtrVector const &shaders,
                                MetaData *metaDataOut,
                                HgiCapabilities const *capabilities);

    ////////////////////////////////////////////////////////////
    // Hgi Binding
    ////////////////////////////////////////////////////////////

    HDST_API
    void GetBufferBindingDesc(
                HgiResourceBindingsDesc * bindingsDesc,
                TfToken const & name,
                HdStBufferResourceSharedPtr const & resource,
                int offset,
                int level = -1,
                int numElements = 1) const;

    HDST_API
    void GetBufferArrayBindingDesc(
                HgiResourceBindingsDesc * bindingsDesc,
                HdStBufferArrayRangeSharedPtr const & bar) const;

    HDST_API
    void GetInterleavedBufferArrayBindingDesc(
                HgiResourceBindingsDesc *bindingsDesc,
                HdStBufferArrayRangeSharedPtr const & bar,
                TfToken const & name) const;

    HDST_API
    void GetInstanceBufferArrayBindingDesc(
                HgiResourceBindingsDesc * bindingsDesc,
                HdStBufferArrayRangeSharedPtr const & bar,
                int level) const;
    
    HDST_API
    void GetBindingRequestBindingDesc(
                HgiResourceBindingsDesc * bindingsDesc,
                HdStBindingRequest const & req) const;

    HDST_API
    void GetTextureBindingDesc(
                HgiResourceBindingsDesc * bindingsDesc,
                TfToken const & name,
                HgiSamplerHandle const & texelSampler,
                HgiTextureHandle const & texelTexture) const;

    HDST_API
    void GetTextureBindingDescs(
                HgiResourceBindingsDesc * bindingsDesc,
                TfToken const & name,
                std::vector<HgiSamplerHandle> const & texelSamplers,
                std::vector<HgiTextureHandle> const & texelTextures) const;

    HDST_API
    void GetTextureWithLayoutBindingDesc(
                HgiResourceBindingsDesc * bindingsDesc,
                TfToken const & name,
                HgiSamplerHandle const & texelSampler,
                HgiTextureHandle const & texelTexture,
                HgiSamplerHandle const & layoutSampler,
                HgiTextureHandle const & layoutTexture) const;
    
    HDST_API
    void
    GetTextureWithLayoutBindingDescs(
                HgiResourceBindingsDesc * bindingsDesc,
                TfToken const & name,
                std::vector<HgiSamplerHandle> const & texelSamplers,
                std::vector<HgiTextureHandle> const & texelTextures,
                std::vector<HgiSamplerHandle> const & layoutSamplers,
                std::vector<HgiTextureHandle> const & layoutTextures) const; 

    ////////////////////////////////////////////////////////////
    // GL Binding
    ////////////////////////////////////////////////////////////

    HDST_API
    void Bind(HdStBindingRequest const& req) const;
    HDST_API
    void Unbind(HdStBindingRequest const& req) const;

    /// bind/unbind BufferArray
    HDST_API
    void BindBufferArray(HdStBufferArrayRangeSharedPtr const &bar) const;
    HDST_API
    void UnbindBufferArray(HdStBufferArrayRangeSharedPtr const &bar) const;

    /// bind/unbind interleaved constant buffer
    HDST_API
    void BindConstantBuffer(
        HdStBufferArrayRangeSharedPtr const & constantBar) const;
    HDST_API
    void UnbindConstantBuffer(
        HdStBufferArrayRangeSharedPtr const &constantBar) const;

    /// bind/unbind interleaved buffer
    HDST_API
    void BindInterleavedBuffer(
        HdStBufferArrayRangeSharedPtr const & constantBar,
        TfToken const &name) const;
    HDST_API
    void UnbindInterleavedBuffer(
        HdStBufferArrayRangeSharedPtr const &constantBar,
        TfToken const &name) const;

    /// bind/unbind nested instance BufferArray
    HDST_API
    void BindInstanceBufferArray(
        HdStBufferArrayRangeSharedPtr const &bar, int level) const;
    HDST_API
    void UnbindInstanceBufferArray(
        HdStBufferArrayRangeSharedPtr const &bar, int level) const;

    /// piecewise buffer binding utility
    /// (to be used for frustum culling, draw indirect result)
    HDST_API
    void BindBuffer(TfToken const &name,
                    HdStBufferResourceSharedPtr const &resource) const;
    HDST_API
    void BindBuffer(TfToken const &name,
                    HdStBufferResourceSharedPtr const &resource,
                    int offset, int level=-1, int numElements=1) const;
    HDST_API
    void UnbindBuffer(TfToken const &name,
                      HdStBufferResourceSharedPtr const &resource,
                      int level=-1) const;

    /// bind(update) a standalone uniform (unsigned int)
    HDST_API
    void BindUniformui(TfToken const &name, int count,
                       const unsigned int *value) const;

    /// bind a standalone uniform (signed int, ivec2, ivec3, ivec4)
    HDST_API
    void BindUniformi(TfToken const &name, int count, const int *value) const;

    /// bind a standalone uniform array (int[N])
    HDST_API
    void BindUniformArrayi(TfToken const &name, int count, const int *value) const;

    /// bind a standalone uniform (float, vec2, vec3, vec4, mat4)
    HDST_API
    void BindUniformf(TfToken const &name, int count, const float *value) const;

    ////////////////////////////////////////////////////////////
    // Binding Queries
    ////////////////////////////////////////////////////////////

    /// Returns whether a binding exists.
    bool HasBinding(TfToken const &name, int level=-1) const {
        return _bindingMap.find(NameAndLevel(name, level)) != _bindingMap.end();
    }

    /// Returns binding point.
    HdStBinding GetBinding(TfToken const &name, int level=-1) const {
        HdStBinding binding;
        TfMapLookup(_bindingMap, NameAndLevel(name, level), &binding);
        return binding;
    }

    /// Returns the bindless handle for \p textureHandle using \p samplerHandle
    HDST_API
    static uint64_t GetSamplerBindlessHandle(
        HgiSamplerHandle const &samplerHandle,
        HgiTextureHandle const &textureHandle);

    /// Returns the bindless handle for \p textureHandle w/o separate sampler
    HDST_API
    static uint64_t GetTextureBindlessHandle(
        HgiTextureHandle const &textureHandle);

    /// Binds the sampler and texture for \p name
    /// Does nothing if the named resource is a bindless resource.
    HDST_API
    void BindTexture(const TfToken &name,
                     HgiSamplerHandle const &samplerHandle,
                     HgiTextureHandle const &textureHandle,
                     const bool bind) const;
    HDST_API
    void BindTextures(const TfToken &name,
                      std::vector<HgiSamplerHandle> const &samplerHandles,
                      std::vector<HgiTextureHandle> const &textureHandles,
                      const bool bind) const;

    /// Binds the sampler and texture for \p name along with an additional
    /// layout texture as needed for Ptex or UDIM textures.
    /// Does nothing if the named resource is a bindless resource.
    HDST_API
    void BindTextureWithLayout(TfToken const &name,
                               HgiSamplerHandle const &texelSampler,
                               HgiTextureHandle const &texelTexture,
                               HgiSamplerHandle const &layoutSampler,
                               HgiTextureHandle const &layoutTexture,
                               const bool bind) const;
    HDST_API
    void BindTexturesWithLayout(
        TfToken const &name,
        std::vector<HgiSamplerHandle> const &texelSamplers,
        std::vector<HgiTextureHandle> const &texelTextures,
        std::vector<HgiSamplerHandle> const &layoutSamplers,
        std::vector<HgiTextureHandle> const &layoutTextures,
        const bool bind) const;

private:
    // for batch execution
    struct NameAndLevel {
        NameAndLevel(TfToken const &n, int lv=-1) :
            name(n), level(lv) {}
        TfToken name;
        int level;

        bool operator < (NameAndLevel const &other) const {
            return name  < other.name ||
                  (name == other.name && level < other.level);
        }
    };
    using _BindingMap = std::map<NameAndLevel, HdStBinding>;
    _BindingMap _bindingMap;
};


PXR_NAMESPACE_CLOSE_SCOPE

#endif  // PXR_IMAGING_HD_ST_RESOURCE_BINDER_H
