/**
 * Copyright 2013-2023 Software Radio Systems Limited
 *
 * This file is part of srsRAN.
 *
 * srsRAN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * srsRAN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * A copy of the GNU Affero General Public License can be found in
 * the LICENSE file in the top-level directory of this distribution
 * and at http://www.gnu.org/licenses/.
 *
 */

#include "srsran/asn1/s1ap.h"
#include <sstream>

using namespace asn1;
using namespace asn1::s1ap;

/*******************************************************************************
 *                                Struct Methods
 ******************************************************************************/

// PrivateIE-ID ::= CHOICE
void private_ie_id_c::set(types::options e)
{
  type_ = e;
}
uint32_t& private_ie_id_c::set_local()
{
  set(types::local);
  return c;
}
void private_ie_id_c::set_global()
{
  set(types::global);
}
void private_ie_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::local:
      j.write_int("local", c);
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
  }
  j.end_obj();
}
SRSASN_CODE private_ie_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::local:
      HANDLE_CODE(pack_integer(bref, c, (uint32_t)0u, (uint32_t)65535u, false, true));
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE private_ie_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::local:
      HANDLE_CODE(unpack_integer(c, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* private_ie_id_c::types_opts::to_string() const
{
  static const char* options[] = {"local", "global"};
  return convert_enum_idx(options, 2, value, "private_ie_id_c::types");
}

// PrivateIE-Field{S1AP-PRIVATE-IES : IEsSetParam} ::= SEQUENCE{{S1AP-PRIVATE-IES}}
template <class ies_set_paramT_>
SRSASN_CODE private_ie_field_s<ies_set_paramT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(id.pack(bref));
  HANDLE_CODE(crit.pack(bref));
  HANDLE_CODE(value.pack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
SRSASN_CODE private_ie_field_s<ies_set_paramT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(id.unpack(bref));
  HANDLE_CODE(crit.unpack(bref));
  HANDLE_CODE(value.unpack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
void private_ie_field_s<ies_set_paramT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("id");
  id.to_json(j);
  j.write_str("criticality", crit.to_string());
  j.end_obj();
}

// ProtocolIE-FieldPair{S1AP-PROTOCOL-IES-PAIR : IEsSetParam} ::= SEQUENCE{{S1AP-PROTOCOL-IES-PAIR}}
template <class ies_set_paramT_>
SRSASN_CODE protocol_ie_field_pair_s<ies_set_paramT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, id, (uint32_t)0u, (uint32_t)65535u, false, true));
  warn_assert(first_crit != ies_set_paramT_::get_first_crit(id), __func__, __LINE__);
  HANDLE_CODE(first_crit.pack(bref));
  HANDLE_CODE(first_value.pack(bref));
  warn_assert(second_crit != ies_set_paramT_::get_second_crit(id), __func__, __LINE__);
  HANDLE_CODE(second_crit.pack(bref));
  HANDLE_CODE(second_value.pack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
SRSASN_CODE protocol_ie_field_pair_s<ies_set_paramT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(first_crit.unpack(bref));
  first_value = ies_set_paramT_::get_first_value(id);
  HANDLE_CODE(first_value.unpack(bref));
  HANDLE_CODE(second_crit.unpack(bref));
  second_value = ies_set_paramT_::get_second_value(id);
  HANDLE_CODE(second_value.unpack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
void protocol_ie_field_pair_s<ies_set_paramT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("id", id);
  j.write_str("firstCriticality", first_crit.to_string());
  j.write_str("secondCriticality", second_crit.to_string());
  j.end_obj();
}
template <class ies_set_paramT_>
bool protocol_ie_field_pair_s<ies_set_paramT_>::load_info_obj(const uint32_t& id_)
{
  if (not ies_set_paramT_::is_id_valid(id_)) {
    return false;
  }
  id           = id_;
  first_crit   = ies_set_paramT_::get_first_crit(id);
  first_value  = ies_set_paramT_::get_first_value(id);
  second_crit  = ies_set_paramT_::get_second_crit(id);
  second_value = ies_set_paramT_::get_second_value(id);
  return first_value.type().value != ies_set_paramT_::first_value_c::types_opts::nulltype and
         second_value.type().value != ies_set_paramT_::second_value_c::types_opts::nulltype;
}

// ActivatedCellsList-Item ::= SEQUENCE
SRSASN_CODE activ_cells_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(cell_id.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE activ_cells_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(cell_id.unpack(bref));

  return SRSASN_SUCCESS;
}
void activ_cells_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-ID", cell_id.to_string());
  j.end_obj();
}

// GUMMEI ::= SEQUENCE
SRSASN_CODE gummei_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(mme_group_id.pack(bref));
  HANDLE_CODE(mme_code.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE gummei_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(mme_group_id.unpack(bref));
  HANDLE_CODE(mme_code.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void gummei_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMN-Identity", plmn_id.to_string());
  j.write_str("mME-Group-ID", mme_group_id.to_string());
  j.write_str("mME-Code", mme_code.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Additional-GUTI ::= SEQUENCE
SRSASN_CODE add_guti_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(gummei.pack(bref));
  HANDLE_CODE(m_tmsi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE add_guti_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(gummei.unpack(bref));
  HANDLE_CODE(m_tmsi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void add_guti_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("gUMMEI");
  gummei.to_json(j);
  j.write_str("m-TMSI", m_tmsi.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Pre-emptionCapability ::= ENUMERATED
const char* pre_emption_cap_opts::to_string() const
{
  static const char* options[] = {"shall-not-trigger-pre-emption", "may-trigger-pre-emption"};
  return convert_enum_idx(options, 2, value, "pre_emption_cap_e");
}

// Pre-emptionVulnerability ::= ENUMERATED
const char* pre_emption_vulnerability_opts::to_string() const
{
  static const char* options[] = {"not-pre-emptable", "pre-emptable"};
  return convert_enum_idx(options, 2, value, "pre_emption_vulnerability_e");
}

// AllocationAndRetentionPriority ::= SEQUENCE
SRSASN_CODE alloc_and_retention_prio_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, prio_level, (uint8_t)0u, (uint8_t)15u, false, true));
  HANDLE_CODE(pre_emption_cap.pack(bref));
  HANDLE_CODE(pre_emption_vulnerability.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE alloc_and_retention_prio_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(prio_level, bref, (uint8_t)0u, (uint8_t)15u, false, true));
  HANDLE_CODE(pre_emption_cap.unpack(bref));
  HANDLE_CODE(pre_emption_vulnerability.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void alloc_and_retention_prio_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("priorityLevel", prio_level);
  j.write_str("pre-emptionCapability", pre_emption_cap.to_string());
  j.write_str("pre-emptionVulnerability", pre_emption_vulnerability.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EUTRAN-CGI ::= SEQUENCE
SRSASN_CODE eutran_cgi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(cell_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE eutran_cgi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(cell_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void eutran_cgi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("cell-ID", cell_id.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAI ::= SEQUENCE
SRSASN_CODE tai_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(tac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(tac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("tAC", tac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellBasedMDT ::= SEQUENCE
SRSASN_CODE cell_based_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, cell_id_listfor_mdt, 1, 32, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_based_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(cell_id_listfor_mdt, bref, 1, 32, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_based_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("cellIdListforMDT");
  for (const auto& e1 : cell_id_listfor_mdt) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TABasedMDT ::= SEQUENCE
SRSASN_CODE ta_based_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, talistfor_mdt, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ta_based_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(talistfor_mdt, bref, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ta_based_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("tAListforMDT");
  for (const auto& e1 : talistfor_mdt) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAIBasedMDT ::= SEQUENCE
SRSASN_CODE tai_based_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, tai_listfor_mdt, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_based_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(tai_listfor_mdt, bref, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_based_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("tAIListforMDT");
  for (const auto& e1 : tai_listfor_mdt) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AreaScopeOfMDT ::= CHOICE
void area_scope_of_mdt_c::destroy_()
{
  switch (type_) {
    case types::cell_based:
      c.destroy<cell_based_mdt_s>();
      break;
    case types::tabased:
      c.destroy<ta_based_mdt_s>();
      break;
    case types::tai_based:
      c.destroy<tai_based_mdt_s>();
      break;
    default:
      break;
  }
}
void area_scope_of_mdt_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_based:
      c.init<cell_based_mdt_s>();
      break;
    case types::tabased:
      c.init<ta_based_mdt_s>();
      break;
    case types::plmn_wide:
      break;
    case types::tai_based:
      c.init<tai_based_mdt_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
  }
}
area_scope_of_mdt_c::area_scope_of_mdt_c(const area_scope_of_mdt_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_based:
      c.init(other.c.get<cell_based_mdt_s>());
      break;
    case types::tabased:
      c.init(other.c.get<ta_based_mdt_s>());
      break;
    case types::plmn_wide:
      break;
    case types::tai_based:
      c.init(other.c.get<tai_based_mdt_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
  }
}
area_scope_of_mdt_c& area_scope_of_mdt_c::operator=(const area_scope_of_mdt_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_based:
      c.set(other.c.get<cell_based_mdt_s>());
      break;
    case types::tabased:
      c.set(other.c.get<ta_based_mdt_s>());
      break;
    case types::plmn_wide:
      break;
    case types::tai_based:
      c.set(other.c.get<tai_based_mdt_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
  }

  return *this;
}
cell_based_mdt_s& area_scope_of_mdt_c::set_cell_based()
{
  set(types::cell_based);
  return c.get<cell_based_mdt_s>();
}
ta_based_mdt_s& area_scope_of_mdt_c::set_tabased()
{
  set(types::tabased);
  return c.get<ta_based_mdt_s>();
}
void area_scope_of_mdt_c::set_plmn_wide()
{
  set(types::plmn_wide);
}
tai_based_mdt_s& area_scope_of_mdt_c::set_tai_based()
{
  set(types::tai_based);
  return c.get<tai_based_mdt_s>();
}
void area_scope_of_mdt_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_based:
      j.write_fieldname("cellBased");
      c.get<cell_based_mdt_s>().to_json(j);
      break;
    case types::tabased:
      j.write_fieldname("tABased");
      c.get<ta_based_mdt_s>().to_json(j);
      break;
    case types::plmn_wide:
      break;
    case types::tai_based:
      j.write_fieldname("tAIBased");
      c.get<tai_based_mdt_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
  }
  j.end_obj();
}
SRSASN_CODE area_scope_of_mdt_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_based:
      HANDLE_CODE(c.get<cell_based_mdt_s>().pack(bref));
      break;
    case types::tabased:
      HANDLE_CODE(c.get<ta_based_mdt_s>().pack(bref));
      break;
    case types::plmn_wide:
      break;
    case types::tai_based: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<tai_based_mdt_s>().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE area_scope_of_mdt_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_based:
      HANDLE_CODE(c.get<cell_based_mdt_s>().unpack(bref));
      break;
    case types::tabased:
      HANDLE_CODE(c.get<ta_based_mdt_s>().unpack(bref));
      break;
    case types::plmn_wide:
      break;
    case types::tai_based: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<tai_based_mdt_s>().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* area_scope_of_mdt_c::types_opts::to_string() const
{
  static const char* options[] = {"cellBased", "tABased", "pLMNWide", "tAIBased"};
  return convert_enum_idx(options, 4, value, "area_scope_of_mdt_c::types");
}

// CellBasedQMC ::= SEQUENCE
SRSASN_CODE cell_based_qmc_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, cell_id_listfor_qmc, 1, 32, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_based_qmc_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(cell_id_listfor_qmc, bref, 1, 32, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_based_qmc_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("cellIdListforQMC");
  for (const auto& e1 : cell_id_listfor_qmc) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PLMNAreaBasedQMC ::= SEQUENCE
SRSASN_CODE plmn_area_based_qmc_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, plmn_listfor_qmc, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_area_based_qmc_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(plmn_listfor_qmc, bref, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void plmn_area_based_qmc_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("plmnListforQMC");
  for (const auto& e1 : plmn_listfor_qmc) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TABasedQMC ::= SEQUENCE
SRSASN_CODE ta_based_qmc_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, talistfor_qmc, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ta_based_qmc_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(talistfor_qmc, bref, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ta_based_qmc_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("tAListforQMC");
  for (const auto& e1 : talistfor_qmc) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAIBasedQMC ::= SEQUENCE
SRSASN_CODE tai_based_qmc_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, tai_listfor_qmc, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_based_qmc_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(tai_listfor_qmc, bref, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_based_qmc_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("tAIListforQMC");
  for (const auto& e1 : tai_listfor_qmc) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AreaScopeOfQMC ::= CHOICE
void area_scope_of_qmc_c::destroy_()
{
  switch (type_) {
    case types::cell_based:
      c.destroy<cell_based_qmc_s>();
      break;
    case types::tabased:
      c.destroy<ta_based_qmc_s>();
      break;
    case types::tai_based:
      c.destroy<tai_based_qmc_s>();
      break;
    case types::plmn_area_based:
      c.destroy<plmn_area_based_qmc_s>();
      break;
    default:
      break;
  }
}
void area_scope_of_qmc_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_based:
      c.init<cell_based_qmc_s>();
      break;
    case types::tabased:
      c.init<ta_based_qmc_s>();
      break;
    case types::tai_based:
      c.init<tai_based_qmc_s>();
      break;
    case types::plmn_area_based:
      c.init<plmn_area_based_qmc_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_qmc_c");
  }
}
area_scope_of_qmc_c::area_scope_of_qmc_c(const area_scope_of_qmc_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_based:
      c.init(other.c.get<cell_based_qmc_s>());
      break;
    case types::tabased:
      c.init(other.c.get<ta_based_qmc_s>());
      break;
    case types::tai_based:
      c.init(other.c.get<tai_based_qmc_s>());
      break;
    case types::plmn_area_based:
      c.init(other.c.get<plmn_area_based_qmc_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_qmc_c");
  }
}
area_scope_of_qmc_c& area_scope_of_qmc_c::operator=(const area_scope_of_qmc_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_based:
      c.set(other.c.get<cell_based_qmc_s>());
      break;
    case types::tabased:
      c.set(other.c.get<ta_based_qmc_s>());
      break;
    case types::tai_based:
      c.set(other.c.get<tai_based_qmc_s>());
      break;
    case types::plmn_area_based:
      c.set(other.c.get<plmn_area_based_qmc_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_qmc_c");
  }

  return *this;
}
cell_based_qmc_s& area_scope_of_qmc_c::set_cell_based()
{
  set(types::cell_based);
  return c.get<cell_based_qmc_s>();
}
ta_based_qmc_s& area_scope_of_qmc_c::set_tabased()
{
  set(types::tabased);
  return c.get<ta_based_qmc_s>();
}
tai_based_qmc_s& area_scope_of_qmc_c::set_tai_based()
{
  set(types::tai_based);
  return c.get<tai_based_qmc_s>();
}
plmn_area_based_qmc_s& area_scope_of_qmc_c::set_plmn_area_based()
{
  set(types::plmn_area_based);
  return c.get<plmn_area_based_qmc_s>();
}
void area_scope_of_qmc_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_based:
      j.write_fieldname("cellBased");
      c.get<cell_based_qmc_s>().to_json(j);
      break;
    case types::tabased:
      j.write_fieldname("tABased");
      c.get<ta_based_qmc_s>().to_json(j);
      break;
    case types::tai_based:
      j.write_fieldname("tAIBased");
      c.get<tai_based_qmc_s>().to_json(j);
      break;
    case types::plmn_area_based:
      j.write_fieldname("pLMNAreaBased");
      c.get<plmn_area_based_qmc_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_qmc_c");
  }
  j.end_obj();
}
SRSASN_CODE area_scope_of_qmc_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_based:
      HANDLE_CODE(c.get<cell_based_qmc_s>().pack(bref));
      break;
    case types::tabased:
      HANDLE_CODE(c.get<ta_based_qmc_s>().pack(bref));
      break;
    case types::tai_based:
      HANDLE_CODE(c.get<tai_based_qmc_s>().pack(bref));
      break;
    case types::plmn_area_based:
      HANDLE_CODE(c.get<plmn_area_based_qmc_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_qmc_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE area_scope_of_qmc_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_based:
      HANDLE_CODE(c.get<cell_based_qmc_s>().unpack(bref));
      break;
    case types::tabased:
      HANDLE_CODE(c.get<ta_based_qmc_s>().unpack(bref));
      break;
    case types::tai_based:
      HANDLE_CODE(c.get<tai_based_qmc_s>().unpack(bref));
      break;
    case types::plmn_area_based:
      HANDLE_CODE(c.get<plmn_area_based_qmc_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_qmc_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* area_scope_of_qmc_c::types_opts::to_string() const
{
  static const char* options[] = {"cellBased", "tABased", "tAIBased", "pLMNAreaBased"};
  return convert_enum_idx(options, 4, value, "area_scope_of_qmc_c::types");
}

// CellIdentifierAndCELevelForCECapableUEs ::= SEQUENCE
SRSASN_CODE cell_id_and_ce_level_for_ce_capable_ues_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_cell_id.pack(bref));
  HANDLE_CODE(celevel.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_and_ce_level_for_ce_capable_ues_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_cell_id.unpack(bref));
  HANDLE_CODE(celevel.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_and_ce_level_for_ce_capable_ues_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-Cell-ID");
  global_cell_id.to_json(j);
  j.write_str("cELevel", celevel.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssistanceDataForCECapableUEs ::= SEQUENCE
SRSASN_CODE assist_data_for_ce_capable_ues_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cell_id_and_ce_level_for_ce_capable_ues.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE assist_data_for_ce_capable_ues_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cell_id_and_ce_level_for_ce_capable_ues.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void assist_data_for_ce_capable_ues_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cellIdentifierAndCELevelForCECapableUEs");
  cell_id_and_ce_level_for_ce_capable_ues.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RecommendedCellItem ::= SEQUENCE
SRSASN_CODE recommended_cell_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(time_stayed_in_cell_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutran_cgi.pack(bref));
  if (time_stayed_in_cell_present) {
    HANDLE_CODE(pack_integer(bref, time_stayed_in_cell, (uint16_t)0u, (uint16_t)4095u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_cell_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(time_stayed_in_cell_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutran_cgi.unpack(bref));
  if (time_stayed_in_cell_present) {
    HANDLE_CODE(unpack_integer(time_stayed_in_cell, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_cell_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRAN-CGI");
  eutran_cgi.to_json(j);
  if (time_stayed_in_cell_present) {
    j.write_int("timeStayedInCell", time_stayed_in_cell);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RecommendedCellItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t recommended_cell_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {214};
  return map_enum_number(options, 1, idx, "id");
}
bool recommended_cell_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 214 == id;
}
crit_e recommended_cell_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 214) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
recommended_cell_item_ies_o::value_c recommended_cell_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 214) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e recommended_cell_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 214) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void recommended_cell_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("RecommendedCellItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE recommended_cell_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_cell_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* recommended_cell_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"RecommendedCellItem"};
  return convert_enum_idx(options, 1, value, "recommended_cell_item_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<recommended_cell_item_ies_o>;

// NextPagingAreaScope ::= ENUMERATED
const char* next_paging_area_scope_opts::to_string() const
{
  static const char* options[] = {"same", "changed"};
  return convert_enum_idx(options, 2, value, "next_paging_area_scope_e");
}

// RecommendedCellsForPaging ::= SEQUENCE
SRSASN_CODE recommended_cells_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, recommended_cell_list, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_cells_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(recommended_cell_list, bref, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_cells_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("recommendedCellList");
  for (const auto& e1 : recommended_cell_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssistanceDataForRecommendedCells ::= SEQUENCE
SRSASN_CODE assist_data_for_recommended_cells_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE assist_data_for_recommended_cells_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void assist_data_for_recommended_cells_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("recommendedCellsForPaging");
  recommended_cells_for_paging.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PagingAttemptInformation ::= SEQUENCE
SRSASN_CODE paging_attempt_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(next_paging_area_scope_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, paging_attempt_count, (uint8_t)1u, (uint8_t)16u, true, true));
  HANDLE_CODE(pack_integer(bref, intended_nof_paging_attempts, (uint8_t)1u, (uint8_t)16u, true, true));
  if (next_paging_area_scope_present) {
    HANDLE_CODE(next_paging_area_scope.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE paging_attempt_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(next_paging_area_scope_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(paging_attempt_count, bref, (uint8_t)1u, (uint8_t)16u, true, true));
  HANDLE_CODE(unpack_integer(intended_nof_paging_attempts, bref, (uint8_t)1u, (uint8_t)16u, true, true));
  if (next_paging_area_scope_present) {
    HANDLE_CODE(next_paging_area_scope.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void paging_attempt_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pagingAttemptCount", paging_attempt_count);
  j.write_int("intendedNumberOfPagingAttempts", intended_nof_paging_attempts);
  if (next_paging_area_scope_present) {
    j.write_str("nextPagingAreaScope", next_paging_area_scope.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssistanceDataForPaging ::= SEQUENCE
SRSASN_CODE assist_data_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(assist_data_for_recommended_cells_present, 1));
  HANDLE_CODE(bref.pack(assist_data_for_ce_capable_ues_present, 1));
  HANDLE_CODE(bref.pack(paging_attempt_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (assist_data_for_recommended_cells_present) {
    HANDLE_CODE(assist_data_for_recommended_cells.pack(bref));
  }
  if (assist_data_for_ce_capable_ues_present) {
    HANDLE_CODE(assist_data_for_ce_capable_ues.pack(bref));
  }
  if (paging_attempt_info_present) {
    HANDLE_CODE(paging_attempt_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE assist_data_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(assist_data_for_recommended_cells_present, 1));
  HANDLE_CODE(bref.unpack(assist_data_for_ce_capable_ues_present, 1));
  HANDLE_CODE(bref.unpack(paging_attempt_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (assist_data_for_recommended_cells_present) {
    HANDLE_CODE(assist_data_for_recommended_cells.unpack(bref));
  }
  if (assist_data_for_ce_capable_ues_present) {
    HANDLE_CODE(assist_data_for_ce_capable_ues.unpack(bref));
  }
  if (paging_attempt_info_present) {
    HANDLE_CODE(paging_attempt_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void assist_data_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (assist_data_for_recommended_cells_present) {
    j.write_fieldname("assistanceDataForRecommendedCells");
    assist_data_for_recommended_cells.to_json(j);
  }
  if (assist_data_for_ce_capable_ues_present) {
    j.write_fieldname("assistanceDataForCECapableUEs");
    assist_data_for_ce_capable_ues.to_json(j);
  }
  if (paging_attempt_info_present) {
    j.write_fieldname("pagingAttemptInformation");
    paging_attempt_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// COUNTValueExtended ::= SEQUENCE
SRSASN_CODE count_value_extended_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdcp_sn_extended, (uint16_t)0u, (uint16_t)32767u, false, true));
  HANDLE_CODE(pack_integer(bref, hfn_modified, (uint32_t)0u, (uint32_t)131071u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE count_value_extended_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdcp_sn_extended, bref, (uint16_t)0u, (uint16_t)32767u, false, true));
  HANDLE_CODE(unpack_integer(hfn_modified, bref, (uint32_t)0u, (uint32_t)131071u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void count_value_extended_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDCP-SNExtended", pdcp_sn_extended);
  j.write_int("hFNModified", hfn_modified);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// COUNTvaluePDCP-SNlength18 ::= SEQUENCE
SRSASN_CODE coun_tvalue_pdcp_snlen18_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdcp_snlen18, (uint32_t)0u, (uint32_t)262143u, false, true));
  HANDLE_CODE(pack_integer(bref, hfnfor_pdcp_snlen18, (uint16_t)0u, (uint16_t)16383u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE coun_tvalue_pdcp_snlen18_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdcp_snlen18, bref, (uint32_t)0u, (uint32_t)262143u, false, true));
  HANDLE_CODE(unpack_integer(hfnfor_pdcp_snlen18, bref, (uint16_t)0u, (uint16_t)16383u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void coun_tvalue_pdcp_snlen18_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDCP-SNlength18", pdcp_snlen18);
  j.write_int("hFNforPDCP-SNlength18", hfnfor_pdcp_snlen18);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Bearers-SubjectToStatusTransfer-ItemExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t bearers_subject_to_status_transfer_item_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {179, 180, 181, 217, 218, 219};
  return map_enum_number(options, 6, idx, "id");
}
bool bearers_subject_to_status_transfer_item_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {179, 180, 181, 217, 218, 219};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e bearers_subject_to_status_transfer_item_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 179:
      return crit_e::ignore;
    case 180:
      return crit_e::ignore;
    case 181:
      return crit_e::ignore;
    case 217:
      return crit_e::ignore;
    case 218:
      return crit_e::ignore;
    case 219:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c
bearers_subject_to_status_transfer_item_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 179:
      ret.set(ext_c::types::ulcount_value_extended);
      break;
    case 180:
      ret.set(ext_c::types::dlcount_value_extended);
      break;
    case 181:
      ret.set(ext_c::types::receive_status_of_ulpdcpsdus_extended);
      break;
    case 217:
      ret.set(ext_c::types::ulcount_value_pdcp_snlen18);
      break;
    case 218:
      ret.set(ext_c::types::dlcount_value_pdcp_snlen18);
      break;
    case 219:
      ret.set(ext_c::types::receive_status_of_ulpdcpsdus_pdcp_snlen18);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e bearers_subject_to_status_transfer_item_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 179:
      return presence_e::optional;
    case 180:
      return presence_e::optional;
    case 181:
      return presence_e::optional;
    case 217:
      return presence_e::optional;
    case 218:
      return presence_e::optional;
    case 219:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::ulcount_value_extended:
      c.destroy<count_value_extended_s>();
      break;
    case types::dlcount_value_extended:
      c.destroy<count_value_extended_s>();
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      c.destroy<bounded_bitstring<1, 16384, false, true> >();
      break;
    case types::ulcount_value_pdcp_snlen18:
      c.destroy<coun_tvalue_pdcp_snlen18_s>();
      break;
    case types::dlcount_value_pdcp_snlen18:
      c.destroy<coun_tvalue_pdcp_snlen18_s>();
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      c.destroy<bounded_bitstring<1, 131072, false, true> >();
      break;
    default:
      break;
  }
}
void bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ulcount_value_extended:
      c.init<count_value_extended_s>();
      break;
    case types::dlcount_value_extended:
      c.init<count_value_extended_s>();
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      c.init<bounded_bitstring<1, 16384, false, true> >();
      break;
    case types::ulcount_value_pdcp_snlen18:
      c.init<coun_tvalue_pdcp_snlen18_s>();
      break;
    case types::dlcount_value_pdcp_snlen18:
      c.init<coun_tvalue_pdcp_snlen18_s>();
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      c.init<bounded_bitstring<1, 131072, false, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
  }
}
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ext_c(
    const bearers_subject_to_status_transfer_item_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ulcount_value_extended:
      c.init(other.c.get<count_value_extended_s>());
      break;
    case types::dlcount_value_extended:
      c.init(other.c.get<count_value_extended_s>());
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      c.init(other.c.get<bounded_bitstring<1, 16384, false, true> >());
      break;
    case types::ulcount_value_pdcp_snlen18:
      c.init(other.c.get<coun_tvalue_pdcp_snlen18_s>());
      break;
    case types::dlcount_value_pdcp_snlen18:
      c.init(other.c.get<coun_tvalue_pdcp_snlen18_s>());
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      c.init(other.c.get<bounded_bitstring<1, 131072, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
  }
}
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::operator=(
    const bearers_subject_to_status_transfer_item_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ulcount_value_extended:
      c.set(other.c.get<count_value_extended_s>());
      break;
    case types::dlcount_value_extended:
      c.set(other.c.get<count_value_extended_s>());
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      c.set(other.c.get<bounded_bitstring<1, 16384, false, true> >());
      break;
    case types::ulcount_value_pdcp_snlen18:
      c.set(other.c.get<coun_tvalue_pdcp_snlen18_s>());
      break;
    case types::dlcount_value_pdcp_snlen18:
      c.set(other.c.get<coun_tvalue_pdcp_snlen18_s>());
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      c.set(other.c.get<bounded_bitstring<1, 131072, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
  }

  return *this;
}
count_value_extended_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ulcount_value_extended()
{
  assert_choice_type(types::ulcount_value_extended, type_, "Extension");
  return c.get<count_value_extended_s>();
}
count_value_extended_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::dlcount_value_extended()
{
  assert_choice_type(types::dlcount_value_extended, type_, "Extension");
  return c.get<count_value_extended_s>();
}
bounded_bitstring<1, 16384, false, true>&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::receive_status_of_ulpdcpsdus_extended()
{
  assert_choice_type(types::receive_status_of_ulpdcpsdus_extended, type_, "Extension");
  return c.get<bounded_bitstring<1, 16384, false, true> >();
}
coun_tvalue_pdcp_snlen18_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ulcount_value_pdcp_snlen18()
{
  assert_choice_type(types::ulcount_value_pdcp_snlen18, type_, "Extension");
  return c.get<coun_tvalue_pdcp_snlen18_s>();
}
coun_tvalue_pdcp_snlen18_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::dlcount_value_pdcp_snlen18()
{
  assert_choice_type(types::dlcount_value_pdcp_snlen18, type_, "Extension");
  return c.get<coun_tvalue_pdcp_snlen18_s>();
}
bounded_bitstring<1, 131072, false, true>&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::receive_status_of_ulpdcpsdus_pdcp_snlen18()
{
  assert_choice_type(types::receive_status_of_ulpdcpsdus_pdcp_snlen18, type_, "Extension");
  return c.get<bounded_bitstring<1, 131072, false, true> >();
}
const count_value_extended_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ulcount_value_extended() const
{
  assert_choice_type(types::ulcount_value_extended, type_, "Extension");
  return c.get<count_value_extended_s>();
}
const count_value_extended_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::dlcount_value_extended() const
{
  assert_choice_type(types::dlcount_value_extended, type_, "Extension");
  return c.get<count_value_extended_s>();
}
const bounded_bitstring<1, 16384, false, true>&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::receive_status_of_ulpdcpsdus_extended() const
{
  assert_choice_type(types::receive_status_of_ulpdcpsdus_extended, type_, "Extension");
  return c.get<bounded_bitstring<1, 16384, false, true> >();
}
const coun_tvalue_pdcp_snlen18_s&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ulcount_value_pdcp_snlen18() const
{
  assert_choice_type(types::ulcount_value_pdcp_snlen18, type_, "Extension");
  return c.get<coun_tvalue_pdcp_snlen18_s>();
}
const coun_tvalue_pdcp_snlen18_s&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::dlcount_value_pdcp_snlen18() const
{
  assert_choice_type(types::dlcount_value_pdcp_snlen18, type_, "Extension");
  return c.get<coun_tvalue_pdcp_snlen18_s>();
}
const bounded_bitstring<1, 131072, false, true>&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::receive_status_of_ulpdcpsdus_pdcp_snlen18() const
{
  assert_choice_type(types::receive_status_of_ulpdcpsdus_pdcp_snlen18, type_, "Extension");
  return c.get<bounded_bitstring<1, 131072, false, true> >();
}
void bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ulcount_value_extended:
      j.write_fieldname("COUNTValueExtended");
      c.get<count_value_extended_s>().to_json(j);
      break;
    case types::dlcount_value_extended:
      j.write_fieldname("COUNTValueExtended");
      c.get<count_value_extended_s>().to_json(j);
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 16384, false, true> >().to_string());
      break;
    case types::ulcount_value_pdcp_snlen18:
      j.write_fieldname("COUNTvaluePDCP-SNlength18");
      c.get<coun_tvalue_pdcp_snlen18_s>().to_json(j);
      break;
    case types::dlcount_value_pdcp_snlen18:
      j.write_fieldname("COUNTvaluePDCP-SNlength18");
      c.get<coun_tvalue_pdcp_snlen18_s>().to_json(j);
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 131072, false, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ulcount_value_extended:
      HANDLE_CODE(c.get<count_value_extended_s>().pack(bref));
      break;
    case types::dlcount_value_extended:
      HANDLE_CODE(c.get<count_value_extended_s>().pack(bref));
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      HANDLE_CODE((c.get<bounded_bitstring<1, 16384, false, true> >().pack(bref)));
      break;
    case types::ulcount_value_pdcp_snlen18:
      HANDLE_CODE(c.get<coun_tvalue_pdcp_snlen18_s>().pack(bref));
      break;
    case types::dlcount_value_pdcp_snlen18:
      HANDLE_CODE(c.get<coun_tvalue_pdcp_snlen18_s>().pack(bref));
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      HANDLE_CODE((c.get<bounded_bitstring<1, 131072, false, true> >().pack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ulcount_value_extended:
      HANDLE_CODE(c.get<count_value_extended_s>().unpack(bref));
      break;
    case types::dlcount_value_extended:
      HANDLE_CODE(c.get<count_value_extended_s>().unpack(bref));
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      HANDLE_CODE((c.get<bounded_bitstring<1, 16384, false, true> >().unpack(bref)));
      break;
    case types::ulcount_value_pdcp_snlen18:
      HANDLE_CODE(c.get<coun_tvalue_pdcp_snlen18_s>().unpack(bref));
      break;
    case types::dlcount_value_pdcp_snlen18:
      HANDLE_CODE(c.get<coun_tvalue_pdcp_snlen18_s>().unpack(bref));
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      HANDLE_CODE((c.get<bounded_bitstring<1, 131072, false, true> >().unpack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"COUNTValueExtended",
                                  "COUNTValueExtended",
                                  "BIT STRING",
                                  "COUNTvaluePDCP-SNlength18",
                                  "COUNTvaluePDCP-SNlength18",
                                  "BIT STRING"};
  return convert_enum_idx(options, 6, value, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::types");
}

// COUNTvalue ::= SEQUENCE
SRSASN_CODE coun_tvalue_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdcp_sn, (uint16_t)0u, (uint16_t)4095u, false, true));
  HANDLE_CODE(pack_integer(bref, hfn, (uint32_t)0u, (uint32_t)1048575u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE coun_tvalue_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdcp_sn, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  HANDLE_CODE(unpack_integer(hfn, bref, (uint32_t)0u, (uint32_t)1048575u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void coun_tvalue_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDCP-SN", pdcp_sn);
  j.write_int("hFN", hfn);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::protocol_ext_field_s<bearers_subject_to_status_transfer_item_ext_ies_o>;

bearers_subject_to_status_transfer_item_ext_ies_container::bearers_subject_to_status_transfer_item_ext_ies_container() :
  ulcount_value_extended(179, crit_e::ignore),
  dlcount_value_extended(180, crit_e::ignore),
  receive_status_of_ulpdcpsdus_extended(181, crit_e::ignore),
  ulcount_value_pdcp_snlen18(217, crit_e::ignore),
  dlcount_value_pdcp_snlen18(218, crit_e::ignore),
  receive_status_of_ulpdcpsdus_pdcp_snlen18(219, crit_e::ignore)
{}
SRSASN_CODE bearers_subject_to_status_transfer_item_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += ulcount_value_extended_present ? 1 : 0;
  nof_ies += dlcount_value_extended_present ? 1 : 0;
  nof_ies += receive_status_of_ulpdcpsdus_extended_present ? 1 : 0;
  nof_ies += ulcount_value_pdcp_snlen18_present ? 1 : 0;
  nof_ies += dlcount_value_pdcp_snlen18_present ? 1 : 0;
  nof_ies += receive_status_of_ulpdcpsdus_pdcp_snlen18_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (ulcount_value_extended_present) {
    HANDLE_CODE(ulcount_value_extended.pack(bref));
  }
  if (dlcount_value_extended_present) {
    HANDLE_CODE(dlcount_value_extended.pack(bref));
  }
  if (receive_status_of_ulpdcpsdus_extended_present) {
    HANDLE_CODE(receive_status_of_ulpdcpsdus_extended.pack(bref));
  }
  if (ulcount_value_pdcp_snlen18_present) {
    HANDLE_CODE(ulcount_value_pdcp_snlen18.pack(bref));
  }
  if (dlcount_value_pdcp_snlen18_present) {
    HANDLE_CODE(dlcount_value_pdcp_snlen18.pack(bref));
  }
  if (receive_status_of_ulpdcpsdus_pdcp_snlen18_present) {
    HANDLE_CODE(receive_status_of_ulpdcpsdus_pdcp_snlen18.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 179: {
        ulcount_value_extended_present = true;
        ulcount_value_extended.id      = id;
        HANDLE_CODE(ulcount_value_extended.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ulcount_value_extended.ext.unpack(bref));
        break;
      }
      case 180: {
        dlcount_value_extended_present = true;
        dlcount_value_extended.id      = id;
        HANDLE_CODE(dlcount_value_extended.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(dlcount_value_extended.ext.unpack(bref));
        break;
      }
      case 181: {
        receive_status_of_ulpdcpsdus_extended_present = true;
        receive_status_of_ulpdcpsdus_extended.id      = id;
        HANDLE_CODE(receive_status_of_ulpdcpsdus_extended.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(receive_status_of_ulpdcpsdus_extended.ext.unpack(bref));
        break;
      }
      case 217: {
        ulcount_value_pdcp_snlen18_present = true;
        ulcount_value_pdcp_snlen18.id      = id;
        HANDLE_CODE(ulcount_value_pdcp_snlen18.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ulcount_value_pdcp_snlen18.ext.unpack(bref));
        break;
      }
      case 218: {
        dlcount_value_pdcp_snlen18_present = true;
        dlcount_value_pdcp_snlen18.id      = id;
        HANDLE_CODE(dlcount_value_pdcp_snlen18.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(dlcount_value_pdcp_snlen18.ext.unpack(bref));
        break;
      }
      case 219: {
        receive_status_of_ulpdcpsdus_pdcp_snlen18_present = true;
        receive_status_of_ulpdcpsdus_pdcp_snlen18.id      = id;
        HANDLE_CODE(receive_status_of_ulpdcpsdus_pdcp_snlen18.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(receive_status_of_ulpdcpsdus_pdcp_snlen18.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void bearers_subject_to_status_transfer_item_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (ulcount_value_extended_present) {
    j.write_fieldname("");
    ulcount_value_extended.to_json(j);
  }
  if (dlcount_value_extended_present) {
    j.write_fieldname("");
    dlcount_value_extended.to_json(j);
  }
  if (receive_status_of_ulpdcpsdus_extended_present) {
    j.write_fieldname("");
    receive_status_of_ulpdcpsdus_extended.to_json(j);
  }
  if (ulcount_value_pdcp_snlen18_present) {
    j.write_fieldname("");
    ulcount_value_pdcp_snlen18.to_json(j);
  }
  if (dlcount_value_pdcp_snlen18_present) {
    j.write_fieldname("");
    dlcount_value_pdcp_snlen18.to_json(j);
  }
  if (receive_status_of_ulpdcpsdus_pdcp_snlen18_present) {
    j.write_fieldname("");
    receive_status_of_ulpdcpsdus_pdcp_snlen18.to_json(j);
  }
  j.end_obj();
}

// Bearers-SubjectToStatusTransfer-Item ::= SEQUENCE
SRSASN_CODE bearers_subject_to_status_transfer_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(receive_statusof_ulpdcpsdus_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(ul_coun_tvalue.pack(bref));
  HANDLE_CODE(dl_coun_tvalue.pack(bref));
  if (receive_statusof_ulpdcpsdus_present) {
    HANDLE_CODE(receive_statusof_ulpdcpsdus.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE bearers_subject_to_status_transfer_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(receive_statusof_ulpdcpsdus_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(ul_coun_tvalue.unpack(bref));
  HANDLE_CODE(dl_coun_tvalue.unpack(bref));
  if (receive_statusof_ulpdcpsdus_present) {
    HANDLE_CODE(receive_statusof_ulpdcpsdus.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void bearers_subject_to_status_transfer_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("uL-COUNTvalue");
  ul_coun_tvalue.to_json(j);
  j.write_fieldname("dL-COUNTvalue");
  dl_coun_tvalue.to_json(j);
  if (receive_statusof_ulpdcpsdus_present) {
    j.write_str("receiveStatusofULPDCPSDUs", receive_statusof_ulpdcpsdus.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Bearers-SubjectToStatusTransfer-ItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t bearers_subject_to_status_transfer_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {89};
  return map_enum_number(options, 1, idx, "id");
}
bool bearers_subject_to_status_transfer_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 89 == id;
}
crit_e bearers_subject_to_status_transfer_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 89) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
bearers_subject_to_status_transfer_item_ies_o::value_c
bearers_subject_to_status_transfer_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 89) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e bearers_subject_to_status_transfer_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 89) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void bearers_subject_to_status_transfer_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("Bearers-SubjectToStatusTransfer-Item");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* bearers_subject_to_status_transfer_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Bearers-SubjectToStatusTransfer-Item"};
  return convert_enum_idx(options, 1, value, "bearers_subject_to_status_transfer_item_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<bearers_subject_to_status_transfer_item_ies_o>;

// BluetoothMeasConfig ::= ENUMERATED
const char* bluetooth_meas_cfg_opts::to_string() const
{
  static const char* options[] = {"setup"};
  return convert_enum_idx(options, 1, value, "bluetooth_meas_cfg_e");
}

// BluetoothMeasurementConfiguration ::= SEQUENCE
SRSASN_CODE bluetooth_meas_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(bluetooth_meas_cfg_name_list_present, 1));
  HANDLE_CODE(bref.pack(bt_rssi_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(bluetooth_meas_cfg.pack(bref));
  if (bluetooth_meas_cfg_name_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, bluetooth_meas_cfg_name_list, 1, 4, true));
  }
  if (bt_rssi_present) {
    HANDLE_CODE(bt_rssi.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE bluetooth_meas_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(bluetooth_meas_cfg_name_list_present, 1));
  HANDLE_CODE(bref.unpack(bt_rssi_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(bluetooth_meas_cfg.unpack(bref));
  if (bluetooth_meas_cfg_name_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(bluetooth_meas_cfg_name_list, bref, 1, 4, true));
  }
  if (bt_rssi_present) {
    HANDLE_CODE(bt_rssi.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void bluetooth_meas_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("bluetoothMeasConfig", "setup");
  if (bluetooth_meas_cfg_name_list_present) {
    j.start_array("bluetoothMeasConfigNameList");
    for (const auto& e1 : bluetooth_meas_cfg_name_list) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (bt_rssi_present) {
    j.write_str("bt-rssi", "true");
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

const char* bluetooth_meas_cfg_s::bt_rssi_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "bluetooth_meas_cfg_s::bt_rssi_e_");
}

// CancelledCellinEAI-Item ::= SEQUENCE
SRSASN_CODE cancelled_cellin_eai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cancelled_cellin_eai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cancelled_cellin_eai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CancelledCellinTAI-Item ::= SEQUENCE
SRSASN_CODE cancelled_cellin_tai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cancelled_cellin_tai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cancelled_cellin_tai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellID-Cancelled-Item ::= SEQUENCE
SRSASN_CODE cell_id_cancelled_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_cancelled_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_cancelled_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyAreaID-Cancelled-Item ::= SEQUENCE
SRSASN_CODE emergency_area_id_cancelled_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, cancelled_cellin_eai, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_area_id_cancelled_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(cancelled_cellin_eai, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_area_id_cancelled_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyAreaID", emergency_area_id.to_string());
  j.start_array("cancelledCellinEAI");
  for (const auto& e1 : cancelled_cellin_eai) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAI-Cancelled-Item ::= SEQUENCE
SRSASN_CODE tai_cancelled_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, cancelled_cellin_tai, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_cancelled_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(cancelled_cellin_tai, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_cancelled_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  j.start_array("cancelledCellinTAI");
  for (const auto& e1 : cancelled_cellin_tai) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// BroadcastCancelledAreaList ::= CHOICE
void broadcast_cancelled_area_list_c::destroy_()
{
  switch (type_) {
    case types::cell_id_cancelled:
      c.destroy<cell_id_cancelled_l>();
      break;
    case types::tai_cancelled:
      c.destroy<tai_cancelled_l>();
      break;
    case types::emergency_area_id_cancelled:
      c.destroy<emergency_area_id_cancelled_l>();
      break;
    default:
      break;
  }
}
void broadcast_cancelled_area_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_cancelled:
      c.init<cell_id_cancelled_l>();
      break;
    case types::tai_cancelled:
      c.init<tai_cancelled_l>();
      break;
    case types::emergency_area_id_cancelled:
      c.init<emergency_area_id_cancelled_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
}
broadcast_cancelled_area_list_c::broadcast_cancelled_area_list_c(const broadcast_cancelled_area_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_cancelled:
      c.init(other.c.get<cell_id_cancelled_l>());
      break;
    case types::tai_cancelled:
      c.init(other.c.get<tai_cancelled_l>());
      break;
    case types::emergency_area_id_cancelled:
      c.init(other.c.get<emergency_area_id_cancelled_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
}
broadcast_cancelled_area_list_c&
broadcast_cancelled_area_list_c::operator=(const broadcast_cancelled_area_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_id_cancelled:
      c.set(other.c.get<cell_id_cancelled_l>());
      break;
    case types::tai_cancelled:
      c.set(other.c.get<tai_cancelled_l>());
      break;
    case types::emergency_area_id_cancelled:
      c.set(other.c.get<emergency_area_id_cancelled_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }

  return *this;
}
cell_id_cancelled_l& broadcast_cancelled_area_list_c::set_cell_id_cancelled()
{
  set(types::cell_id_cancelled);
  return c.get<cell_id_cancelled_l>();
}
tai_cancelled_l& broadcast_cancelled_area_list_c::set_tai_cancelled()
{
  set(types::tai_cancelled);
  return c.get<tai_cancelled_l>();
}
emergency_area_id_cancelled_l& broadcast_cancelled_area_list_c::set_emergency_area_id_cancelled()
{
  set(types::emergency_area_id_cancelled);
  return c.get<emergency_area_id_cancelled_l>();
}
void broadcast_cancelled_area_list_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_id_cancelled:
      j.start_array("cellID-Cancelled");
      for (const auto& e1 : c.get<cell_id_cancelled_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_cancelled:
      j.start_array("tAI-Cancelled");
      for (const auto& e1 : c.get<tai_cancelled_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_cancelled:
      j.start_array("emergencyAreaID-Cancelled");
      for (const auto& e1 : c.get<emergency_area_id_cancelled_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
  j.end_obj();
}
SRSASN_CODE broadcast_cancelled_area_list_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_id_cancelled:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cell_id_cancelled_l>(), 1, 65535, true));
      break;
    case types::tai_cancelled:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_cancelled_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_cancelled:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_cancelled_l>(), 1, 65535, true));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE broadcast_cancelled_area_list_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_id_cancelled:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cell_id_cancelled_l>(), bref, 1, 65535, true));
      break;
    case types::tai_cancelled:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_cancelled_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_cancelled:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_cancelled_l>(), bref, 1, 65535, true));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* broadcast_cancelled_area_list_c::types_opts::to_string() const
{
  static const char* options[] = {"cellID-Cancelled", "tAI-Cancelled", "emergencyAreaID-Cancelled"};
  return convert_enum_idx(options, 3, value, "broadcast_cancelled_area_list_c::types");
}

// CompletedCellinEAI-Item ::= SEQUENCE
SRSASN_CODE completed_cellin_eai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE completed_cellin_eai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void completed_cellin_eai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CompletedCellinTAI-Item ::= SEQUENCE
SRSASN_CODE completed_cellin_tai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE completed_cellin_tai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void completed_cellin_tai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellID-Broadcast-Item ::= SEQUENCE
SRSASN_CODE cell_id_broadcast_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_broadcast_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_broadcast_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyAreaID-Broadcast-Item ::= SEQUENCE
SRSASN_CODE emergency_area_id_broadcast_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, completed_cellin_eai, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_area_id_broadcast_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(completed_cellin_eai, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_area_id_broadcast_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyAreaID", emergency_area_id.to_string());
  j.start_array("completedCellinEAI");
  for (const auto& e1 : completed_cellin_eai) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAI-Broadcast-Item ::= SEQUENCE
SRSASN_CODE tai_broadcast_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, completed_cellin_tai, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_broadcast_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(completed_cellin_tai, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_broadcast_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  j.start_array("completedCellinTAI");
  for (const auto& e1 : completed_cellin_tai) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// BroadcastCompletedAreaList ::= CHOICE
void broadcast_completed_area_list_c::destroy_()
{
  switch (type_) {
    case types::cell_id_broadcast:
      c.destroy<cell_id_broadcast_l>();
      break;
    case types::tai_broadcast:
      c.destroy<tai_broadcast_l>();
      break;
    case types::emergency_area_id_broadcast:
      c.destroy<emergency_area_id_broadcast_l>();
      break;
    default:
      break;
  }
}
void broadcast_completed_area_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_broadcast:
      c.init<cell_id_broadcast_l>();
      break;
    case types::tai_broadcast:
      c.init<tai_broadcast_l>();
      break;
    case types::emergency_area_id_broadcast:
      c.init<emergency_area_id_broadcast_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
}
broadcast_completed_area_list_c::broadcast_completed_area_list_c(const broadcast_completed_area_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_broadcast:
      c.init(other.c.get<cell_id_broadcast_l>());
      break;
    case types::tai_broadcast:
      c.init(other.c.get<tai_broadcast_l>());
      break;
    case types::emergency_area_id_broadcast:
      c.init(other.c.get<emergency_area_id_broadcast_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
}
broadcast_completed_area_list_c&
broadcast_completed_area_list_c::operator=(const broadcast_completed_area_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_id_broadcast:
      c.set(other.c.get<cell_id_broadcast_l>());
      break;
    case types::tai_broadcast:
      c.set(other.c.get<tai_broadcast_l>());
      break;
    case types::emergency_area_id_broadcast:
      c.set(other.c.get<emergency_area_id_broadcast_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }

  return *this;
}
cell_id_broadcast_l& broadcast_completed_area_list_c::set_cell_id_broadcast()
{
  set(types::cell_id_broadcast);
  return c.get<cell_id_broadcast_l>();
}
tai_broadcast_l& broadcast_completed_area_list_c::set_tai_broadcast()
{
  set(types::tai_broadcast);
  return c.get<tai_broadcast_l>();
}
emergency_area_id_broadcast_l& broadcast_completed_area_list_c::set_emergency_area_id_broadcast()
{
  set(types::emergency_area_id_broadcast);
  return c.get<emergency_area_id_broadcast_l>();
}
void broadcast_completed_area_list_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_id_broadcast:
      j.start_array("cellID-Broadcast");
      for (const auto& e1 : c.get<cell_id_broadcast_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_broadcast:
      j.start_array("tAI-Broadcast");
      for (const auto& e1 : c.get<tai_broadcast_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_broadcast:
      j.start_array("emergencyAreaID-Broadcast");
      for (const auto& e1 : c.get<emergency_area_id_broadcast_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
  j.end_obj();
}
SRSASN_CODE broadcast_completed_area_list_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_id_broadcast:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cell_id_broadcast_l>(), 1, 65535, true));
      break;
    case types::tai_broadcast:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_broadcast_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_broadcast:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_broadcast_l>(), 1, 65535, true));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE broadcast_completed_area_list_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_id_broadcast:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cell_id_broadcast_l>(), bref, 1, 65535, true));
      break;
    case types::tai_broadcast:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_broadcast_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_broadcast:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_broadcast_l>(), bref, 1, 65535, true));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* broadcast_completed_area_list_c::types_opts::to_string() const
{
  static const char* options[] = {"cellID-Broadcast", "tAI-Broadcast", "emergencyAreaID-Broadcast"};
  return convert_enum_idx(options, 3, value, "broadcast_completed_area_list_c::types");
}

// CGI ::= SEQUENCE
SRSASN_CODE cgi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(rac_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(lac.pack(bref));
  HANDLE_CODE(ci.pack(bref));
  if (rac_present) {
    HANDLE_CODE(rac.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cgi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(rac_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(lac.unpack(bref));
  HANDLE_CODE(ci.unpack(bref));
  if (rac_present) {
    HANDLE_CODE(rac.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cgi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("lAC", lac.to_string());
  j.write_str("cI", ci.to_string());
  if (rac_present) {
    j.write_str("rAC", rac.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CNType ::= ENUMERATED
const char* cn_type_opts::to_string() const
{
  static const char* options[] = {"fiveGCForbidden", "epc-Forbiddden"};
  return convert_enum_idx(options, 2, value, "cn_type_e");
}
uint8_t cn_type_opts::to_number() const
{
  static const uint8_t options[] = {5};
  return map_enum_number(options, 1, value, "cn_type_e");
}

// CNTypeRestrictions-Item ::= SEQUENCE
SRSASN_CODE cn_type_restricts_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(cntype.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cn_type_restricts_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(cntype.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cn_type_restricts_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMN-Identity", plmn_id.to_string());
  j.write_str("cNType", cntype.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CSG-IdList-Item ::= SEQUENCE
SRSASN_CODE csg_id_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(csg_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csg_id_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(csg_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csg_id_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cSG-Id", csg_id.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CSGMembershipStatus ::= ENUMERATED
const char* csg_membership_status_opts::to_string() const
{
  static const char* options[] = {"member", "not-member"};
  return convert_enum_idx(options, 2, value, "csg_membership_status_e");
}

// CellAccessMode ::= ENUMERATED
const char* cell_access_mode_opts::to_string() const
{
  static const char* options[] = {"hybrid"};
  return convert_enum_idx(options, 1, value, "cell_access_mode_e");
}

// CSGMembershipInfo ::= SEQUENCE
SRSASN_CODE csg_membership_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cell_access_mode_present, 1));
  HANDLE_CODE(bref.pack(plm_nid_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(csg_membership_status.pack(bref));
  HANDLE_CODE(csg_id.pack(bref));
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.pack(bref));
  }
  if (plm_nid_present) {
    HANDLE_CODE(plm_nid.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csg_membership_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cell_access_mode_present, 1));
  HANDLE_CODE(bref.unpack(plm_nid_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(csg_membership_status.unpack(bref));
  HANDLE_CODE(csg_id.unpack(bref));
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.unpack(bref));
  }
  if (plm_nid_present) {
    HANDLE_CODE(plm_nid.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csg_membership_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cSGMembershipStatus", csg_membership_status.to_string());
  j.write_str("cSG-Id", csg_id.to_string());
  if (cell_access_mode_present) {
    j.write_str("cellAccessMode", "hybrid");
  }
  if (plm_nid_present) {
    j.write_str("pLMNidentity", plm_nid.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// IRAT-Cell-ID ::= CHOICE
void irat_cell_id_c::destroy_()
{
  switch (type_) {
    case types::eutran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::utran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::geran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ehrpd:
      c.destroy<fixed_octstring<16, true> >();
      break;
    default:
      break;
  }
}
void irat_cell_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::eutran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::utran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::geran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ehrpd:
      c.init<fixed_octstring<16, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
  }
}
irat_cell_id_c::irat_cell_id_c(const irat_cell_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::eutran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::utran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::geran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ehrpd:
      c.init(other.c.get<fixed_octstring<16, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
  }
}
irat_cell_id_c& irat_cell_id_c::operator=(const irat_cell_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::eutran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::utran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::geran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ehrpd:
      c.set(other.c.get<fixed_octstring<16, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
  }

  return *this;
}
unbounded_octstring<true>& irat_cell_id_c::set_eutran()
{
  set(types::eutran);
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& irat_cell_id_c::set_utran()
{
  set(types::utran);
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& irat_cell_id_c::set_geran()
{
  set(types::geran);
  return c.get<unbounded_octstring<true> >();
}
fixed_octstring<16, true>& irat_cell_id_c::set_ehrpd()
{
  set(types::ehrpd);
  return c.get<fixed_octstring<16, true> >();
}
void irat_cell_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::eutran:
      j.write_str("eUTRAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::utran:
      j.write_str("uTRAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::geran:
      j.write_str("gERAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ehrpd:
      j.write_str("eHRPD", c.get<fixed_octstring<16, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
  }
  j.end_obj();
}
SRSASN_CODE irat_cell_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::eutran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::geran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ehrpd: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_octstring<16, true> >().pack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE irat_cell_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::eutran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::geran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ehrpd: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_octstring<16, true> >().unpack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* irat_cell_id_c::types_opts::to_string() const
{
  static const char* options[] = {"eUTRAN", "uTRAN", "gERAN", "eHRPD"};
  return convert_enum_idx(options, 4, value, "irat_cell_id_c::types");
}

// CandidatePCI ::= SEQUENCE
SRSASN_CODE candidate_pci_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, pci, (uint16_t)0u, (uint16_t)503u, false, true));
  HANDLE_CODE(earfcn.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE candidate_pci_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(pci, bref, (uint16_t)0u, (uint16_t)503u, false, true));
  HANDLE_CODE(earfcn.unpack(bref));

  return SRSASN_SUCCESS;
}
void candidate_pci_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pCI", pci);
  j.write_str("eARFCN", earfcn.to_string());
  j.end_obj();
}

// CauseMisc ::= ENUMERATED
const char* cause_misc_opts::to_string() const
{
  static const char* options[] = {"control-processing-overload",
                                  "not-enough-user-plane-processing-resources",
                                  "hardware-failure",
                                  "om-intervention",
                                  "unspecified",
                                  "unknown-PLMN"};
  return convert_enum_idx(options, 6, value, "cause_misc_e");
}

// CauseNas ::= ENUMERATED
const char* cause_nas_opts::to_string() const
{
  static const char* options[] = {
      "normal-release", "authentication-failure", "detach", "unspecified", "csg-subscription-expiry"};
  return convert_enum_idx(options, 5, value, "cause_nas_e");
}

// CauseProtocol ::= ENUMERATED
const char* cause_protocol_opts::to_string() const
{
  static const char* options[] = {"transfer-syntax-error",
                                  "abstract-syntax-error-reject",
                                  "abstract-syntax-error-ignore-and-notify",
                                  "message-not-compatible-with-receiver-state",
                                  "semantic-error",
                                  "abstract-syntax-error-falsely-constructed-message",
                                  "unspecified"};
  return convert_enum_idx(options, 7, value, "cause_protocol_e");
}

// CauseRadioNetwork ::= ENUMERATED
const char* cause_radio_network_opts::to_string() const
{
  static const char* options[] = {"unspecified",
                                  "tx2relocoverall-expiry",
                                  "successful-handover",
                                  "release-due-to-eutran-generated-reason",
                                  "handover-cancelled",
                                  "partial-handover",
                                  "ho-failure-in-target-EPC-eNB-or-target-system",
                                  "ho-target-not-allowed",
                                  "tS1relocoverall-expiry",
                                  "tS1relocprep-expiry",
                                  "cell-not-available",
                                  "unknown-targetID",
                                  "no-radio-resources-available-in-target-cell",
                                  "unknown-mme-ue-s1ap-id",
                                  "unknown-enb-ue-s1ap-id",
                                  "unknown-pair-ue-s1ap-id",
                                  "handover-desirable-for-radio-reason",
                                  "time-critical-handover",
                                  "resource-optimisation-handover",
                                  "reduce-load-in-serving-cell",
                                  "user-inactivity",
                                  "radio-connection-with-ue-lost",
                                  "load-balancing-tau-required",
                                  "cs-fallback-triggered",
                                  "ue-not-available-for-ps-service",
                                  "radio-resources-not-available",
                                  "failure-in-radio-interface-procedure",
                                  "invalid-qos-combination",
                                  "interrat-redirection",
                                  "interaction-with-other-procedure",
                                  "unknown-E-RAB-ID",
                                  "multiple-E-RAB-ID-instances",
                                  "encryption-and-or-integrity-protection-algorithms-not-supported",
                                  "s1-intra-system-handover-triggered",
                                  "s1-inter-system-handover-triggered",
                                  "x2-handover-triggered",
                                  "redirection-towards-1xRTT",
                                  "not-supported-QCI-value",
                                  "invalid-CSG-Id",
                                  "release-due-to-pre-emption"};
  return convert_enum_idx(options, 40, value, "cause_radio_network_e");
}

// CauseTransport ::= ENUMERATED
const char* cause_transport_opts::to_string() const
{
  static const char* options[] = {"transport-resource-unavailable", "unspecified"};
  return convert_enum_idx(options, 2, value, "cause_transport_e");
}

// Cause ::= CHOICE
void cause_c::destroy_() {}
void cause_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
cause_c::cause_c(const cause_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::radio_network:
      c.init(other.c.get<cause_radio_network_e>());
      break;
    case types::transport:
      c.init(other.c.get<cause_transport_e>());
      break;
    case types::nas:
      c.init(other.c.get<cause_nas_e>());
      break;
    case types::protocol:
      c.init(other.c.get<cause_protocol_e>());
      break;
    case types::misc:
      c.init(other.c.get<cause_misc_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }
}
cause_c& cause_c::operator=(const cause_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::radio_network:
      c.set(other.c.get<cause_radio_network_e>());
      break;
    case types::transport:
      c.set(other.c.get<cause_transport_e>());
      break;
    case types::nas:
      c.set(other.c.get<cause_nas_e>());
      break;
    case types::protocol:
      c.set(other.c.get<cause_protocol_e>());
      break;
    case types::misc:
      c.set(other.c.get<cause_misc_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }

  return *this;
}
cause_radio_network_e& cause_c::set_radio_network()
{
  set(types::radio_network);
  return c.get<cause_radio_network_e>();
}
cause_transport_e& cause_c::set_transport()
{
  set(types::transport);
  return c.get<cause_transport_e>();
}
cause_nas_e& cause_c::set_nas()
{
  set(types::nas);
  return c.get<cause_nas_e>();
}
cause_protocol_e& cause_c::set_protocol()
{
  set(types::protocol);
  return c.get<cause_protocol_e>();
}
cause_misc_e& cause_c::set_misc()
{
  set(types::misc);
  return c.get<cause_misc_e>();
}
void cause_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::radio_network:
      j.write_str("radioNetwork", c.get<cause_radio_network_e>().to_string());
      break;
    case types::transport:
      j.write_str("transport", c.get<cause_transport_e>().to_string());
      break;
    case types::nas:
      j.write_str("nas", c.get<cause_nas_e>().to_string());
      break;
    case types::protocol:
      j.write_str("protocol", c.get<cause_protocol_e>().to_string());
      break;
    case types::misc:
      j.write_str("misc", c.get<cause_misc_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }
  j.end_obj();
}
SRSASN_CODE cause_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::radio_network:
      HANDLE_CODE(c.get<cause_radio_network_e>().pack(bref));
      break;
    case types::transport:
      HANDLE_CODE(c.get<cause_transport_e>().pack(bref));
      break;
    case types::nas:
      HANDLE_CODE(c.get<cause_nas_e>().pack(bref));
      break;
    case types::protocol:
      HANDLE_CODE(c.get<cause_protocol_e>().pack(bref));
      break;
    case types::misc:
      HANDLE_CODE(c.get<cause_misc_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cause_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::radio_network:
      HANDLE_CODE(c.get<cause_radio_network_e>().unpack(bref));
      break;
    case types::transport:
      HANDLE_CODE(c.get<cause_transport_e>().unpack(bref));
      break;
    case types::nas:
      HANDLE_CODE(c.get<cause_nas_e>().unpack(bref));
      break;
    case types::protocol:
      HANDLE_CODE(c.get<cause_protocol_e>().unpack(bref));
      break;
    case types::misc:
      HANDLE_CODE(c.get<cause_misc_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* cause_c::types_opts::to_string() const
{
  static const char* options[] = {"radioNetwork", "transport", "nas", "protocol", "misc"};
  return convert_enum_idx(options, 5, value, "cause_c::types");
}
uint8_t cause_c::types_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "cause_c::types");
}

// Cdma2000OneXSRVCCInfo ::= SEQUENCE
SRSASN_CODE cdma2000_one_xsrvcc_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cdma2000_one_xmeid.pack(bref));
  HANDLE_CODE(cdma2000_one_xmsi.pack(bref));
  HANDLE_CODE(cdma2000_one_xpilot.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cdma2000_one_xsrvcc_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cdma2000_one_xmeid.unpack(bref));
  HANDLE_CODE(cdma2000_one_xmsi.unpack(bref));
  HANDLE_CODE(cdma2000_one_xpilot.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cdma2000_one_xsrvcc_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cdma2000OneXMEID", cdma2000_one_xmeid.to_string());
  j.write_str("cdma2000OneXMSI", cdma2000_one_xmsi.to_string());
  j.write_str("cdma2000OneXPilot", cdma2000_one_xpilot.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellsToActivateList-Item ::= SEQUENCE
SRSASN_CODE cells_to_activ_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(cell_id.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cells_to_activ_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(cell_id.unpack(bref));

  return SRSASN_SUCCESS;
}
void cells_to_activ_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-ID", cell_id.to_string());
  j.end_obj();
}

// CellActivationRequest ::= SEQUENCE
SRSASN_CODE cell_activation_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(minimum_activation_time_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, cells_to_activ_list, 1, 256, true));
  if (minimum_activation_time_present) {
    HANDLE_CODE(pack_integer(bref, minimum_activation_time, (uint8_t)1u, (uint8_t)60u, false, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_activation_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(minimum_activation_time_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(cells_to_activ_list, bref, 1, 256, true));
  if (minimum_activation_time_present) {
    HANDLE_CODE(unpack_integer(minimum_activation_time, bref, (uint8_t)1u, (uint8_t)60u, false, true));
  }

  return SRSASN_SUCCESS;
}
void cell_activation_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("cellsToActivateList");
  for (const auto& e1 : cells_to_activ_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (minimum_activation_time_present) {
    j.write_int("minimumActivationTime", minimum_activation_time);
  }
  j.end_obj();
}

// CellActivationResponse ::= SEQUENCE
SRSASN_CODE cell_activation_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, activ_cells_list, 0, 256, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_activation_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(activ_cells_list, bref, 0, 256, true));

  return SRSASN_SUCCESS;
}
void cell_activation_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("activatedCellsList");
  for (const auto& e1 : activ_cells_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// EHRPDCompositeAvailableCapacity ::= SEQUENCE
SRSASN_CODE ehrpd_composite_available_capacity_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, ehrpd_sector_capacity_class_value, (uint8_t)1u, (uint8_t)100u, true, true));
  HANDLE_CODE(pack_integer(bref, ehrpd_capacity_value, (uint8_t)0u, (uint8_t)100u, false, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ehrpd_composite_available_capacity_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(ehrpd_sector_capacity_class_value, bref, (uint8_t)1u, (uint8_t)100u, true, true));
  HANDLE_CODE(unpack_integer(ehrpd_capacity_value, bref, (uint8_t)0u, (uint8_t)100u, false, true));

  return SRSASN_SUCCESS;
}
void ehrpd_composite_available_capacity_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("eHRPDSectorCapacityClassValue", ehrpd_sector_capacity_class_value);
  j.write_int("eHRPDCapacityValue", ehrpd_capacity_value);
  j.end_obj();
}

// EHRPDSectorLoadReportingResponse ::= SEQUENCE
SRSASN_CODE ehrpd_sector_load_report_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(dl_ehrpd_composite_available_capacity.pack(bref));
  HANDLE_CODE(ul_ehrpd_composite_available_capacity.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ehrpd_sector_load_report_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(dl_ehrpd_composite_available_capacity.unpack(bref));
  HANDLE_CODE(ul_ehrpd_composite_available_capacity.unpack(bref));

  return SRSASN_SUCCESS;
}
void ehrpd_sector_load_report_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("dL-EHRPD-CompositeAvailableCapacity");
  dl_ehrpd_composite_available_capacity.to_json(j);
  j.write_fieldname("uL-EHRPD-CompositeAvailableCapacity");
  ul_ehrpd_composite_available_capacity.to_json(j);
  j.end_obj();
}

// EUTRANcellLoadReportingResponse ::= SEQUENCE
SRSASN_CODE eutra_ncell_load_report_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(composite_available_capacity_group.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE eutra_ncell_load_report_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(composite_available_capacity_group.unpack(bref));

  return SRSASN_SUCCESS;
}
void eutra_ncell_load_report_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("compositeAvailableCapacityGroup", composite_available_capacity_group.to_string());
  j.end_obj();
}

// CellLoadReportingResponse ::= CHOICE
void cell_load_report_resp_c::destroy_()
{
  switch (type_) {
    case types::eutran:
      c.destroy<eutra_ncell_load_report_resp_s>();
      break;
    case types::utran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::geran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ehrpd:
      c.destroy<ehrpd_sector_load_report_resp_s>();
      break;
    default:
      break;
  }
}
void cell_load_report_resp_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::eutran:
      c.init<eutra_ncell_load_report_resp_s>();
      break;
    case types::utran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::geran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ehrpd:
      c.init<ehrpd_sector_load_report_resp_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
  }
}
cell_load_report_resp_c::cell_load_report_resp_c(const cell_load_report_resp_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::eutran:
      c.init(other.c.get<eutra_ncell_load_report_resp_s>());
      break;
    case types::utran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::geran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ehrpd:
      c.init(other.c.get<ehrpd_sector_load_report_resp_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
  }
}
cell_load_report_resp_c& cell_load_report_resp_c::operator=(const cell_load_report_resp_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::eutran:
      c.set(other.c.get<eutra_ncell_load_report_resp_s>());
      break;
    case types::utran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::geran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ehrpd:
      c.set(other.c.get<ehrpd_sector_load_report_resp_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
  }

  return *this;
}
eutra_ncell_load_report_resp_s& cell_load_report_resp_c::set_eutran()
{
  set(types::eutran);
  return c.get<eutra_ncell_load_report_resp_s>();
}
unbounded_octstring<true>& cell_load_report_resp_c::set_utran()
{
  set(types::utran);
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& cell_load_report_resp_c::set_geran()
{
  set(types::geran);
  return c.get<unbounded_octstring<true> >();
}
ehrpd_sector_load_report_resp_s& cell_load_report_resp_c::set_ehrpd()
{
  set(types::ehrpd);
  return c.get<ehrpd_sector_load_report_resp_s>();
}
void cell_load_report_resp_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::eutran:
      j.write_fieldname("eUTRAN");
      c.get<eutra_ncell_load_report_resp_s>().to_json(j);
      break;
    case types::utran:
      j.write_str("uTRAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::geran:
      j.write_str("gERAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ehrpd:
      j.write_fieldname("eHRPD");
      c.get<ehrpd_sector_load_report_resp_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
  }
  j.end_obj();
}
SRSASN_CODE cell_load_report_resp_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::eutran:
      HANDLE_CODE(c.get<eutra_ncell_load_report_resp_s>().pack(bref));
      break;
    case types::utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::geran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ehrpd: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<ehrpd_sector_load_report_resp_s>().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_load_report_resp_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::eutran:
      HANDLE_CODE(c.get<eutra_ncell_load_report_resp_s>().unpack(bref));
      break;
    case types::utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::geran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ehrpd: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<ehrpd_sector_load_report_resp_s>().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* cell_load_report_resp_c::types_opts::to_string() const
{
  static const char* options[] = {"eUTRAN", "uTRAN", "gERAN", "eHRPD"};
  return convert_enum_idx(options, 4, value, "cell_load_report_resp_c::types");
}

// NotifyFlag ::= ENUMERATED
const char* notify_flag_opts::to_string() const
{
  static const char* options[] = {"activated", "deactivated"};
  return convert_enum_idx(options, 2, value, "notify_flag_e");
}

// NotificationCellList-Item ::= SEQUENCE
SRSASN_CODE notif_cell_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(cell_id.pack(bref));
  HANDLE_CODE(notify_flag.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE notif_cell_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(cell_id.unpack(bref));
  HANDLE_CODE(notify_flag.unpack(bref));

  return SRSASN_SUCCESS;
}
void notif_cell_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-ID", cell_id.to_string());
  j.write_str("notifyFlag", notify_flag.to_string());
  j.end_obj();
}

// CellStateIndication ::= SEQUENCE
SRSASN_CODE cell_state_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, notif_cell_list, 1, 256, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_state_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(notif_cell_list, bref, 1, 256, true));

  return SRSASN_SUCCESS;
}
void cell_state_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("notificationCellList");
  for (const auto& e1 : notif_cell_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// PrivacyIndicator ::= ENUMERATED
const char* privacy_ind_opts::to_string() const
{
  static const char* options[] = {"immediate-MDT", "logged-MDT"};
  return convert_enum_idx(options, 2, value, "privacy_ind_e");
}

// CellTrafficTraceIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t cell_traffic_trace_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 86, 100, 131, 166};
  return map_enum_number(options, 6, idx, "id");
}
bool cell_traffic_trace_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 86, 100, 131, 166};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e cell_traffic_trace_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 86:
      return crit_e::ignore;
    case 100:
      return crit_e::ignore;
    case 131:
      return crit_e::ignore;
    case 166:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
cell_traffic_trace_ies_o::value_c cell_traffic_trace_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 86:
      ret.set(value_c::types::e_utran_trace_id);
      break;
    case 100:
      ret.set(value_c::types::eutran_cgi);
      break;
    case 131:
      ret.set(value_c::types::trace_collection_entity_ip_address);
      break;
    case 166:
      ret.set(value_c::types::privacy_ind);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e cell_traffic_trace_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 86:
      return presence_e::mandatory;
    case 100:
      return presence_e::mandatory;
    case 131:
      return presence_e::mandatory;
    case 166:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void cell_traffic_trace_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::e_utran_trace_id:
      c.destroy<fixed_octstring<8, true> >();
      break;
    case types::eutran_cgi:
      c.destroy<eutran_cgi_s>();
      break;
    case types::trace_collection_entity_ip_address:
      c.destroy<bounded_bitstring<1, 160, true, true> >();
      break;
    default:
      break;
  }
}
void cell_traffic_trace_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::e_utran_trace_id:
      c.init<fixed_octstring<8, true> >();
      break;
    case types::eutran_cgi:
      c.init<eutran_cgi_s>();
      break;
    case types::trace_collection_entity_ip_address:
      c.init<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::privacy_ind:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
}
cell_traffic_trace_ies_o::value_c::value_c(const cell_traffic_trace_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      c.init(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::eutran_cgi:
      c.init(other.c.get<eutran_cgi_s>());
      break;
    case types::trace_collection_entity_ip_address:
      c.init(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::privacy_ind:
      c.init(other.c.get<privacy_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
}
cell_traffic_trace_ies_o::value_c&
cell_traffic_trace_ies_o::value_c::operator=(const cell_traffic_trace_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      c.set(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::eutran_cgi:
      c.set(other.c.get<eutran_cgi_s>());
      break;
    case types::trace_collection_entity_ip_address:
      c.set(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::privacy_ind:
      c.set(other.c.get<privacy_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }

  return *this;
}
uint64_t& cell_traffic_trace_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& cell_traffic_trace_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
fixed_octstring<8, true>& cell_traffic_trace_ies_o::value_c::e_utran_trace_id()
{
  assert_choice_type(types::e_utran_trace_id, type_, "Value");
  return c.get<fixed_octstring<8, true> >();
}
eutran_cgi_s& cell_traffic_trace_ies_o::value_c::eutran_cgi()
{
  assert_choice_type(types::eutran_cgi, type_, "Value");
  return c.get<eutran_cgi_s>();
}
bounded_bitstring<1, 160, true, true>& cell_traffic_trace_ies_o::value_c::trace_collection_entity_ip_address()
{
  assert_choice_type(types::trace_collection_entity_ip_address, type_, "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
privacy_ind_e& cell_traffic_trace_ies_o::value_c::privacy_ind()
{
  assert_choice_type(types::privacy_ind, type_, "Value");
  return c.get<privacy_ind_e>();
}
const uint64_t& cell_traffic_trace_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& cell_traffic_trace_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const fixed_octstring<8, true>& cell_traffic_trace_ies_o::value_c::e_utran_trace_id() const
{
  assert_choice_type(types::e_utran_trace_id, type_, "Value");
  return c.get<fixed_octstring<8, true> >();
}
const eutran_cgi_s& cell_traffic_trace_ies_o::value_c::eutran_cgi() const
{
  assert_choice_type(types::eutran_cgi, type_, "Value");
  return c.get<eutran_cgi_s>();
}
const bounded_bitstring<1, 160, true, true>&
cell_traffic_trace_ies_o::value_c::trace_collection_entity_ip_address() const
{
  assert_choice_type(types::trace_collection_entity_ip_address, type_, "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
const privacy_ind_e& cell_traffic_trace_ies_o::value_c::privacy_ind() const
{
  assert_choice_type(types::privacy_ind, type_, "Value");
  return c.get<privacy_ind_e>();
}
void cell_traffic_trace_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<8, true> >().to_string());
      break;
    case types::eutran_cgi:
      j.write_fieldname("EUTRAN-CGI");
      c.get<eutran_cgi_s>().to_json(j);
      break;
    case types::trace_collection_entity_ip_address:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 160, true, true> >().to_string());
      break;
    case types::privacy_ind:
      j.write_str("PrivacyIndicator", c.get<privacy_ind_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE cell_traffic_trace_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::e_utran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().pack(bref)));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().pack(bref));
      break;
    case types::trace_collection_entity_ip_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().pack(bref)));
      break;
    case types::privacy_ind:
      HANDLE_CODE(c.get<privacy_ind_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_traffic_trace_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::e_utran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().unpack(bref)));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().unpack(bref));
      break;
    case types::trace_collection_entity_ip_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().unpack(bref)));
      break;
    case types::privacy_ind:
      HANDLE_CODE(c.get<privacy_ind_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* cell_traffic_trace_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "OCTET STRING",
                                  "EUTRAN-CGI",
                                  "BIT STRING",
                                  "PrivacyIndicator"};
  return convert_enum_idx(options, 6, value, "cell_traffic_trace_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<cell_traffic_trace_ies_o>;

cell_traffic_trace_ies_container::cell_traffic_trace_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  e_utran_trace_id(86, crit_e::ignore),
  eutran_cgi(100, crit_e::ignore),
  trace_collection_entity_ip_address(131, crit_e::ignore),
  privacy_ind(166, crit_e::ignore)
{}
SRSASN_CODE cell_traffic_trace_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 5;
  nof_ies += privacy_ind_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(e_utran_trace_id.pack(bref));
  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.pack(bref));
  if (privacy_ind_present) {
    HANDLE_CODE(privacy_ind.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_traffic_trace_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 5;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 86: {
        nof_mandatory_ies--;
        e_utran_trace_id.id = id;
        HANDLE_CODE(e_utran_trace_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(e_utran_trace_id.value.unpack(bref));
        break;
      }
      case 100: {
        nof_mandatory_ies--;
        eutran_cgi.id = id;
        HANDLE_CODE(eutran_cgi.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(eutran_cgi.value.unpack(bref));
        break;
      }
      case 131: {
        nof_mandatory_ies--;
        trace_collection_entity_ip_address.id = id;
        HANDLE_CODE(trace_collection_entity_ip_address.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(trace_collection_entity_ip_address.value.unpack(bref));
        break;
      }
      case 166: {
        privacy_ind_present = true;
        privacy_ind.id      = id;
        HANDLE_CODE(privacy_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(privacy_ind.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void cell_traffic_trace_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  e_utran_trace_id.to_json(j);
  j.write_fieldname("");
  eutran_cgi.to_json(j);
  j.write_fieldname("");
  trace_collection_entity_ip_address.to_json(j);
  if (privacy_ind_present) {
    j.write_fieldname("");
    privacy_ind.to_json(j);
  }
  j.end_obj();
}

// Cell-Size ::= ENUMERATED
const char* cell_size_opts::to_string() const
{
  static const char* options[] = {"verysmall", "small", "medium", "large"};
  return convert_enum_idx(options, 4, value, "cell_size_e");
}

// CellType ::= SEQUENCE
SRSASN_CODE cell_type_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cell_size.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_type_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cell_size.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_type_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-Size", cell_size.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RAT-Type ::= ENUMERATED
const char* rat_type_opts::to_string() const
{
  static const char* options[] = {"nbiot"};
  return convert_enum_idx(options, 1, value, "rat_type_e");
}

// SupportedTAs-Item-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t supported_tas_item_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {232};
  return map_enum_number(options, 1, idx, "id");
}
bool supported_tas_item_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 232 == id;
}
crit_e supported_tas_item_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 232) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
supported_tas_item_ext_ies_o::ext_c supported_tas_item_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 232) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e supported_tas_item_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 232) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void supported_tas_item_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("RAT-Type", "nbiot");
  j.end_obj();
}
SRSASN_CODE supported_tas_item_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE supported_tas_item_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* supported_tas_item_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"RAT-Type"};
  return convert_enum_idx(options, 1, value, "supported_tas_item_ext_ies_o::ext_c::types");
}

// SupportedTAs-Item ::= SEQUENCE
SRSASN_CODE supported_tas_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tac.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, broadcast_plmns, 1, 6, true));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE supported_tas_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tac.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(broadcast_plmns, bref, 1, 6, true));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void supported_tas_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("tAC", tac.to_string());
  j.start_array("broadcastPLMNs");
  for (const auto& e1 : broadcast_plmns) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// ConnectedengNBItem ::= SEQUENCE
SRSASN_CODE connectedeng_nb_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(en_g_nb_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, supported_tas, 1, 256, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE connectedeng_nb_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(en_g_nb_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(supported_tas, bref, 1, 256, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void connectedeng_nb_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("en-gNB-ID", en_g_nb_id.to_string());
  j.start_array("supportedTAs");
  for (const auto& e1 : supported_tas) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// GBR-QosInformation-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t gbr_qos_info_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {255, 256, 257, 258};
  return map_enum_number(options, 4, idx, "id");
}
bool gbr_qos_info_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {255, 256, 257, 258};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e gbr_qos_info_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 255:
      return crit_e::ignore;
    case 256:
      return crit_e::ignore;
    case 257:
      return crit_e::ignore;
    case 258:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
gbr_qos_info_ext_ies_o::ext_c gbr_qos_info_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 255:
      ret.set(ext_c::types::extended_erab_maximum_bitrate_dl);
      break;
    case 256:
      ret.set(ext_c::types::extended_erab_maximum_bitrate_ul);
      break;
    case 257:
      ret.set(ext_c::types::extended_erab_guaranteed_bitrate_dl);
      break;
    case 258:
      ret.set(ext_c::types::extended_erab_guaranteed_bitrate_ul);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e gbr_qos_info_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 255:
      return presence_e::optional;
    case 256:
      return presence_e::optional;
    case 257:
      return presence_e::optional;
    case 258:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void gbr_qos_info_ext_ies_o::ext_c::destroy_() {}
void gbr_qos_info_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
gbr_qos_info_ext_ies_o::ext_c::ext_c(const gbr_qos_info_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::extended_erab_maximum_bitrate_dl:
      c.init(other.c.get<uint64_t>());
      break;
    case types::extended_erab_maximum_bitrate_ul:
      c.init(other.c.get<uint64_t>());
      break;
    case types::extended_erab_guaranteed_bitrate_dl:
      c.init(other.c.get<uint64_t>());
      break;
    case types::extended_erab_guaranteed_bitrate_ul:
      c.init(other.c.get<uint64_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "gbr_qos_info_ext_ies_o::ext_c");
  }
}
gbr_qos_info_ext_ies_o::ext_c& gbr_qos_info_ext_ies_o::ext_c::operator=(const gbr_qos_info_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::extended_erab_maximum_bitrate_dl:
      c.set(other.c.get<uint64_t>());
      break;
    case types::extended_erab_maximum_bitrate_ul:
      c.set(other.c.get<uint64_t>());
      break;
    case types::extended_erab_guaranteed_bitrate_dl:
      c.set(other.c.get<uint64_t>());
      break;
    case types::extended_erab_guaranteed_bitrate_ul:
      c.set(other.c.get<uint64_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "gbr_qos_info_ext_ies_o::ext_c");
  }

  return *this;
}
uint64_t& gbr_qos_info_ext_ies_o::ext_c::extended_erab_maximum_bitrate_dl()
{
  assert_choice_type(types::extended_erab_maximum_bitrate_dl, type_, "Extension");
  return c.get<uint64_t>();
}
uint64_t& gbr_qos_info_ext_ies_o::ext_c::extended_erab_maximum_bitrate_ul()
{
  assert_choice_type(types::extended_erab_maximum_bitrate_ul, type_, "Extension");
  return c.get<uint64_t>();
}
uint64_t& gbr_qos_info_ext_ies_o::ext_c::extended_erab_guaranteed_bitrate_dl()
{
  assert_choice_type(types::extended_erab_guaranteed_bitrate_dl, type_, "Extension");
  return c.get<uint64_t>();
}
uint64_t& gbr_qos_info_ext_ies_o::ext_c::extended_erab_guaranteed_bitrate_ul()
{
  assert_choice_type(types::extended_erab_guaranteed_bitrate_ul, type_, "Extension");
  return c.get<uint64_t>();
}
const uint64_t& gbr_qos_info_ext_ies_o::ext_c::extended_erab_maximum_bitrate_dl() const
{
  assert_choice_type(types::extended_erab_maximum_bitrate_dl, type_, "Extension");
  return c.get<uint64_t>();
}
const uint64_t& gbr_qos_info_ext_ies_o::ext_c::extended_erab_maximum_bitrate_ul() const
{
  assert_choice_type(types::extended_erab_maximum_bitrate_ul, type_, "Extension");
  return c.get<uint64_t>();
}
const uint64_t& gbr_qos_info_ext_ies_o::ext_c::extended_erab_guaranteed_bitrate_dl() const
{
  assert_choice_type(types::extended_erab_guaranteed_bitrate_dl, type_, "Extension");
  return c.get<uint64_t>();
}
const uint64_t& gbr_qos_info_ext_ies_o::ext_c::extended_erab_guaranteed_bitrate_ul() const
{
  assert_choice_type(types::extended_erab_guaranteed_bitrate_ul, type_, "Extension");
  return c.get<uint64_t>();
}
void gbr_qos_info_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::extended_erab_maximum_bitrate_dl:
      j.write_int("INTEGER (10000000001..4000000000000,...)", c.get<uint64_t>());
      break;
    case types::extended_erab_maximum_bitrate_ul:
      j.write_int("INTEGER (10000000001..4000000000000,...)", c.get<uint64_t>());
      break;
    case types::extended_erab_guaranteed_bitrate_dl:
      j.write_int("INTEGER (10000000001..4000000000000,...)", c.get<uint64_t>());
      break;
    case types::extended_erab_guaranteed_bitrate_ul:
      j.write_int("INTEGER (10000000001..4000000000000,...)", c.get<uint64_t>());
      break;
    default:
      log_invalid_choice_id(type_, "gbr_qos_info_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE gbr_qos_info_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::extended_erab_maximum_bitrate_dl:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    case types::extended_erab_maximum_bitrate_ul:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    case types::extended_erab_guaranteed_bitrate_dl:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    case types::extended_erab_guaranteed_bitrate_ul:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    default:
      log_invalid_choice_id(type_, "gbr_qos_info_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE gbr_qos_info_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::extended_erab_maximum_bitrate_dl:
      HANDLE_CODE(
          unpack_integer(c.get<uint64_t>(), bref, (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    case types::extended_erab_maximum_bitrate_ul:
      HANDLE_CODE(
          unpack_integer(c.get<uint64_t>(), bref, (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    case types::extended_erab_guaranteed_bitrate_dl:
      HANDLE_CODE(
          unpack_integer(c.get<uint64_t>(), bref, (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    case types::extended_erab_guaranteed_bitrate_ul:
      HANDLE_CODE(
          unpack_integer(c.get<uint64_t>(), bref, (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    default:
      log_invalid_choice_id(type_, "gbr_qos_info_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* gbr_qos_info_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (10000000001..4000000000000,...)",
                                  "INTEGER (10000000001..4000000000000,...)",
                                  "INTEGER (10000000001..4000000000000,...)",
                                  "INTEGER (10000000001..4000000000000,...)"};
  return convert_enum_idx(options, 4, value, "gbr_qos_info_ext_ies_o::ext_c::types");
}

// E-RABQoSParameters-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_qos_params_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {273, 274};
  return map_enum_number(options, 2, idx, "id");
}
bool erab_qos_params_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {273, 274};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_qos_params_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 273:
      return crit_e::ignore;
    case 274:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_qos_params_ext_ies_o::ext_c erab_qos_params_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 273:
      ret.set(ext_c::types::dl_packet_loss_rate);
      break;
    case 274:
      ret.set(ext_c::types::ul_packet_loss_rate);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_qos_params_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 273:
      return presence_e::optional;
    case 274:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void erab_qos_params_ext_ies_o::ext_c::destroy_() {}
void erab_qos_params_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
erab_qos_params_ext_ies_o::ext_c::ext_c(const erab_qos_params_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::dl_packet_loss_rate:
      c.init(other.c.get<uint16_t>());
      break;
    case types::ul_packet_loss_rate:
      c.init(other.c.get<uint16_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_qos_params_ext_ies_o::ext_c");
  }
}
erab_qos_params_ext_ies_o::ext_c&
erab_qos_params_ext_ies_o::ext_c::operator=(const erab_qos_params_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::dl_packet_loss_rate:
      c.set(other.c.get<uint16_t>());
      break;
    case types::ul_packet_loss_rate:
      c.set(other.c.get<uint16_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_qos_params_ext_ies_o::ext_c");
  }

  return *this;
}
uint16_t& erab_qos_params_ext_ies_o::ext_c::dl_packet_loss_rate()
{
  assert_choice_type(types::dl_packet_loss_rate, type_, "Extension");
  return c.get<uint16_t>();
}
uint16_t& erab_qos_params_ext_ies_o::ext_c::ul_packet_loss_rate()
{
  assert_choice_type(types::ul_packet_loss_rate, type_, "Extension");
  return c.get<uint16_t>();
}
const uint16_t& erab_qos_params_ext_ies_o::ext_c::dl_packet_loss_rate() const
{
  assert_choice_type(types::dl_packet_loss_rate, type_, "Extension");
  return c.get<uint16_t>();
}
const uint16_t& erab_qos_params_ext_ies_o::ext_c::ul_packet_loss_rate() const
{
  assert_choice_type(types::ul_packet_loss_rate, type_, "Extension");
  return c.get<uint16_t>();
}
void erab_qos_params_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::dl_packet_loss_rate:
      j.write_int("INTEGER (0..1000)", c.get<uint16_t>());
      break;
    case types::ul_packet_loss_rate:
      j.write_int("INTEGER (0..1000)", c.get<uint16_t>());
      break;
    default:
      log_invalid_choice_id(type_, "erab_qos_params_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_qos_params_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::dl_packet_loss_rate:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)1000u, false, true));
      break;
    case types::ul_packet_loss_rate:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)1000u, false, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_qos_params_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_qos_params_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::dl_packet_loss_rate:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)1000u, false, true));
      break;
    case types::ul_packet_loss_rate:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)1000u, false, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_qos_params_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_qos_params_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1000)", "INTEGER (0..1000)"};
  return convert_enum_idx(options, 2, value, "erab_qos_params_ext_ies_o::ext_c::types");
}

template struct asn1::protocol_ext_field_s<gbr_qos_info_ext_ies_o>;

gbr_qos_info_ext_ies_container::gbr_qos_info_ext_ies_container() :
  extended_erab_maximum_bitrate_dl(255, crit_e::ignore),
  extended_erab_maximum_bitrate_ul(256, crit_e::ignore),
  extended_erab_guaranteed_bitrate_dl(257, crit_e::ignore),
  extended_erab_guaranteed_bitrate_ul(258, crit_e::ignore)
{}
SRSASN_CODE gbr_qos_info_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += extended_erab_maximum_bitrate_dl_present ? 1 : 0;
  nof_ies += extended_erab_maximum_bitrate_ul_present ? 1 : 0;
  nof_ies += extended_erab_guaranteed_bitrate_dl_present ? 1 : 0;
  nof_ies += extended_erab_guaranteed_bitrate_ul_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (extended_erab_maximum_bitrate_dl_present) {
    HANDLE_CODE(extended_erab_maximum_bitrate_dl.pack(bref));
  }
  if (extended_erab_maximum_bitrate_ul_present) {
    HANDLE_CODE(extended_erab_maximum_bitrate_ul.pack(bref));
  }
  if (extended_erab_guaranteed_bitrate_dl_present) {
    HANDLE_CODE(extended_erab_guaranteed_bitrate_dl.pack(bref));
  }
  if (extended_erab_guaranteed_bitrate_ul_present) {
    HANDLE_CODE(extended_erab_guaranteed_bitrate_ul.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE gbr_qos_info_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 255: {
        extended_erab_maximum_bitrate_dl_present = true;
        extended_erab_maximum_bitrate_dl.id      = id;
        HANDLE_CODE(extended_erab_maximum_bitrate_dl.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(extended_erab_maximum_bitrate_dl.ext.unpack(bref));
        break;
      }
      case 256: {
        extended_erab_maximum_bitrate_ul_present = true;
        extended_erab_maximum_bitrate_ul.id      = id;
        HANDLE_CODE(extended_erab_maximum_bitrate_ul.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(extended_erab_maximum_bitrate_ul.ext.unpack(bref));
        break;
      }
      case 257: {
        extended_erab_guaranteed_bitrate_dl_present = true;
        extended_erab_guaranteed_bitrate_dl.id      = id;
        HANDLE_CODE(extended_erab_guaranteed_bitrate_dl.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(extended_erab_guaranteed_bitrate_dl.ext.unpack(bref));
        break;
      }
      case 258: {
        extended_erab_guaranteed_bitrate_ul_present = true;
        extended_erab_guaranteed_bitrate_ul.id      = id;
        HANDLE_CODE(extended_erab_guaranteed_bitrate_ul.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(extended_erab_guaranteed_bitrate_ul.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void gbr_qos_info_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (extended_erab_maximum_bitrate_dl_present) {
    j.write_fieldname("");
    extended_erab_maximum_bitrate_dl.to_json(j);
  }
  if (extended_erab_maximum_bitrate_ul_present) {
    j.write_fieldname("");
    extended_erab_maximum_bitrate_ul.to_json(j);
  }
  if (extended_erab_guaranteed_bitrate_dl_present) {
    j.write_fieldname("");
    extended_erab_guaranteed_bitrate_dl.to_json(j);
  }
  if (extended_erab_guaranteed_bitrate_ul_present) {
    j.write_fieldname("");
    extended_erab_guaranteed_bitrate_ul.to_json(j);
  }
  j.end_obj();
}

// GBR-QosInformation ::= SEQUENCE
SRSASN_CODE gbr_qos_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_maximum_bitrate_dl, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(pack_integer(bref, erab_maximum_bitrate_ul, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(pack_integer(bref, erab_guaranteed_bitrate_dl, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(pack_integer(bref, erab_guaranteed_bitrate_ul, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE gbr_qos_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_maximum_bitrate_dl, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(unpack_integer(erab_maximum_bitrate_ul, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(unpack_integer(erab_guaranteed_bitrate_dl, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(unpack_integer(erab_guaranteed_bitrate_ul, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void gbr_qos_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-MaximumBitrateDL", erab_maximum_bitrate_dl);
  j.write_int("e-RAB-MaximumBitrateUL", erab_maximum_bitrate_ul);
  j.write_int("e-RAB-GuaranteedBitrateDL", erab_guaranteed_bitrate_dl);
  j.write_int("e-RAB-GuaranteedBitrateUL", erab_guaranteed_bitrate_ul);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ScheduledCommunicationTime ::= SEQUENCE
SRSASN_CODE sched_communication_time_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dayof_week_present, 1));
  HANDLE_CODE(bref.pack(timeof_day_start_present, 1));
  HANDLE_CODE(bref.pack(timeof_day_end_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (dayof_week_present) {
    HANDLE_CODE(dayof_week.pack(bref));
  }
  if (timeof_day_start_present) {
    HANDLE_CODE(pack_integer(bref, timeof_day_start, (uint32_t)0u, (uint32_t)86399u, true, true));
  }
  if (timeof_day_end_present) {
    HANDLE_CODE(pack_integer(bref, timeof_day_end, (uint32_t)0u, (uint32_t)86399u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sched_communication_time_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dayof_week_present, 1));
  HANDLE_CODE(bref.unpack(timeof_day_start_present, 1));
  HANDLE_CODE(bref.unpack(timeof_day_end_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (dayof_week_present) {
    HANDLE_CODE(dayof_week.unpack(bref));
  }
  if (timeof_day_start_present) {
    HANDLE_CODE(unpack_integer(timeof_day_start, bref, (uint32_t)0u, (uint32_t)86399u, true, true));
  }
  if (timeof_day_end_present) {
    HANDLE_CODE(unpack_integer(timeof_day_end, bref, (uint32_t)0u, (uint32_t)86399u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sched_communication_time_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (dayof_week_present) {
    j.write_str("dayofWeek", dayof_week.to_string());
  }
  if (timeof_day_start_present) {
    j.write_int("timeofDayStart", timeof_day_start);
  }
  if (timeof_day_end_present) {
    j.write_int("timeofDayEnd", timeof_day_end);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CE-ModeBRestricted ::= ENUMERATED
const char* ce_mode_brestricted_opts::to_string() const
{
  static const char* options[] = {"restricted", "not-restricted"};
  return convert_enum_idx(options, 2, value, "ce_mode_brestricted_e");
}

// DL-CP-SecurityInformation ::= SEQUENCE
SRSASN_CODE dl_cp_security_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(dl_nas_mac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_cp_security_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(dl_nas_mac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void dl_cp_security_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("dl-NAS-MAC", dl_nas_mac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::protocol_ext_field_s<erab_qos_params_ext_ies_o>;

erab_qos_params_ext_ies_container::erab_qos_params_ext_ies_container() :
  dl_packet_loss_rate(273, crit_e::ignore), ul_packet_loss_rate(274, crit_e::ignore)
{}
SRSASN_CODE erab_qos_params_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += dl_packet_loss_rate_present ? 1 : 0;
  nof_ies += ul_packet_loss_rate_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (dl_packet_loss_rate_present) {
    HANDLE_CODE(dl_packet_loss_rate.pack(bref));
  }
  if (ul_packet_loss_rate_present) {
    HANDLE_CODE(ul_packet_loss_rate.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_qos_params_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 273: {
        dl_packet_loss_rate_present = true;
        dl_packet_loss_rate.id      = id;
        HANDLE_CODE(dl_packet_loss_rate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(dl_packet_loss_rate.ext.unpack(bref));
        break;
      }
      case 274: {
        ul_packet_loss_rate_present = true;
        ul_packet_loss_rate.id      = id;
        HANDLE_CODE(ul_packet_loss_rate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ul_packet_loss_rate.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void erab_qos_params_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (dl_packet_loss_rate_present) {
    j.write_fieldname("");
    dl_packet_loss_rate.to_json(j);
  }
  if (ul_packet_loss_rate_present) {
    j.write_fieldname("");
    ul_packet_loss_rate.to_json(j);
  }
  j.end_obj();
}

// E-RABLevelQoSParameters ::= SEQUENCE
SRSASN_CODE erab_level_qos_params_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(gbr_qos_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, qci, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(alloc_retention_prio.pack(bref));
  if (gbr_qos_info_present) {
    HANDLE_CODE(gbr_qos_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_level_qos_params_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(gbr_qos_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(qci, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(alloc_retention_prio.unpack(bref));
  if (gbr_qos_info_present) {
    HANDLE_CODE(gbr_qos_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_level_qos_params_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("qCI", qci);
  j.write_fieldname("allocationRetentionPriority");
  alloc_retention_prio.to_json(j);
  if (gbr_qos_info_present) {
    j.write_fieldname("gbrQosInformation");
    gbr_qos_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EndIndication ::= ENUMERATED
const char* end_ind_opts::to_string() const
{
  static const char* options[] = {"no-further-data", "further-data-exists"};
  return convert_enum_idx(options, 2, value, "end_ind_e");
}

// EnhancedCoverageRestricted ::= ENUMERATED
const char* enhanced_coverage_restricted_opts::to_string() const
{
  static const char* options[] = {"restricted"};
  return convert_enum_idx(options, 1, value, "enhanced_coverage_restricted_e");
}

// Subscription-Based-UE-DifferentiationInfo ::= SEQUENCE
SRSASN_CODE subscription_based_ue_differentiation_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(periodic_communication_ind_present, 1));
  HANDLE_CODE(bref.pack(periodic_time_present, 1));
  HANDLE_CODE(bref.pack(sched_communication_time_present, 1));
  HANDLE_CODE(bref.pack(stationary_ind_present, 1));
  HANDLE_CODE(bref.pack(traffic_profile_present, 1));
  HANDLE_CODE(bref.pack(battery_ind_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (periodic_communication_ind_present) {
    HANDLE_CODE(periodic_communication_ind.pack(bref));
  }
  if (periodic_time_present) {
    HANDLE_CODE(pack_integer(bref, periodic_time, (uint16_t)1u, (uint16_t)3600u, true, true));
  }
  if (sched_communication_time_present) {
    HANDLE_CODE(sched_communication_time.pack(bref));
  }
  if (stationary_ind_present) {
    HANDLE_CODE(stationary_ind.pack(bref));
  }
  if (traffic_profile_present) {
    HANDLE_CODE(traffic_profile.pack(bref));
  }
  if (battery_ind_present) {
    HANDLE_CODE(battery_ind.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE subscription_based_ue_differentiation_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(periodic_communication_ind_present, 1));
  HANDLE_CODE(bref.unpack(periodic_time_present, 1));
  HANDLE_CODE(bref.unpack(sched_communication_time_present, 1));
  HANDLE_CODE(bref.unpack(stationary_ind_present, 1));
  HANDLE_CODE(bref.unpack(traffic_profile_present, 1));
  HANDLE_CODE(bref.unpack(battery_ind_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (periodic_communication_ind_present) {
    HANDLE_CODE(periodic_communication_ind.unpack(bref));
  }
  if (periodic_time_present) {
    HANDLE_CODE(unpack_integer(periodic_time, bref, (uint16_t)1u, (uint16_t)3600u, true, true));
  }
  if (sched_communication_time_present) {
    HANDLE_CODE(sched_communication_time.unpack(bref));
  }
  if (stationary_ind_present) {
    HANDLE_CODE(stationary_ind.unpack(bref));
  }
  if (traffic_profile_present) {
    HANDLE_CODE(traffic_profile.unpack(bref));
  }
  if (battery_ind_present) {
    HANDLE_CODE(battery_ind.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void subscription_based_ue_differentiation_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (periodic_communication_ind_present) {
    j.write_str("periodicCommunicationIndicator", periodic_communication_ind.to_string());
  }
  if (periodic_time_present) {
    j.write_int("periodicTime", periodic_time);
  }
  if (sched_communication_time_present) {
    j.write_fieldname("scheduledCommunicationTime");
    sched_communication_time.to_json(j);
  }
  if (stationary_ind_present) {
    j.write_str("stationaryIndication", stationary_ind.to_string());
  }
  if (traffic_profile_present) {
    j.write_str("trafficProfile", traffic_profile.to_string());
  }
  if (battery_ind_present) {
    j.write_str("batteryIndication", battery_ind.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

const char* subscription_based_ue_differentiation_info_s::periodic_communication_ind_opts::to_string() const
{
  static const char* options[] = {"periodically", "ondemand"};
  return convert_enum_idx(
      options, 2, value, "subscription_based_ue_differentiation_info_s::periodic_communication_ind_e_");
}

const char* subscription_based_ue_differentiation_info_s::stationary_ind_opts::to_string() const
{
  static const char* options[] = {"stationary", "mobile"};
  return convert_enum_idx(options, 2, value, "subscription_based_ue_differentiation_info_s::stationary_ind_e_");
}

const char* subscription_based_ue_differentiation_info_s::traffic_profile_opts::to_string() const
{
  static const char* options[] = {"single-packet", "dual-packets", "multiple-packets"};
  return convert_enum_idx(options, 3, value, "subscription_based_ue_differentiation_info_s::traffic_profile_e_");
}

const char* subscription_based_ue_differentiation_info_s::battery_ind_opts::to_string() const
{
  static const char* options[] = {
      "battery-powered", "battery-powered-not-rechargeable-or-replaceable", "not-battery-powered"};
  return convert_enum_idx(options, 3, value, "subscription_based_ue_differentiation_info_s::battery_ind_e_");
}

// ConnectionEstablishmentIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t conn_establishment_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 74, 251, 253, 271, 280, 278, 252};
  return map_enum_number(options, 9, idx, "id");
}
bool conn_establishment_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 74, 251, 253, 271, 280, 278, 252};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e conn_establishment_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 74:
      return crit_e::ignore;
    case 251:
      return crit_e::ignore;
    case 253:
      return crit_e::ignore;
    case 271:
      return crit_e::ignore;
    case 280:
      return crit_e::ignore;
    case 278:
      return crit_e::ignore;
    case 252:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
conn_establishment_ind_ies_o::value_c conn_establishment_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 74:
      ret.set(value_c::types::ue_radio_cap);
      break;
    case 251:
      ret.set(value_c::types::enhanced_coverage_restricted);
      break;
    case 253:
      ret.set(value_c::types::dl_cp_security_info);
      break;
    case 271:
      ret.set(value_c::types::ce_mode_brestricted);
      break;
    case 280:
      ret.set(value_c::types::end_ind);
      break;
    case 278:
      ret.set(value_c::types::subscription_based_ue_differentiation_info);
      break;
    case 252:
      ret.set(value_c::types::ue_level_qos_params);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e conn_establishment_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 74:
      return presence_e::optional;
    case 251:
      return presence_e::optional;
    case 253:
      return presence_e::optional;
    case 271:
      return presence_e::optional;
    case 280:
      return presence_e::optional;
    case 278:
      return presence_e::optional;
    case 252:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void conn_establishment_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ue_radio_cap:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::dl_cp_security_info:
      c.destroy<dl_cp_security_info_s>();
      break;
    case types::subscription_based_ue_differentiation_info:
      c.destroy<subscription_based_ue_differentiation_info_s>();
      break;
    case types::ue_level_qos_params:
      c.destroy<erab_level_qos_params_s>();
      break;
    default:
      break;
  }
}
void conn_establishment_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ue_radio_cap:
      c.init<unbounded_octstring<true> >();
      break;
    case types::enhanced_coverage_restricted:
      break;
    case types::dl_cp_security_info:
      c.init<dl_cp_security_info_s>();
      break;
    case types::ce_mode_brestricted:
      break;
    case types::end_ind:
      break;
    case types::subscription_based_ue_differentiation_info:
      c.init<subscription_based_ue_differentiation_info_s>();
      break;
    case types::ue_level_qos_params:
      c.init<erab_level_qos_params_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
  }
}
conn_establishment_ind_ies_o::value_c::value_c(const conn_establishment_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ue_radio_cap:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::enhanced_coverage_restricted:
      c.init(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::dl_cp_security_info:
      c.init(other.c.get<dl_cp_security_info_s>());
      break;
    case types::ce_mode_brestricted:
      c.init(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::end_ind:
      c.init(other.c.get<end_ind_e>());
      break;
    case types::subscription_based_ue_differentiation_info:
      c.init(other.c.get<subscription_based_ue_differentiation_info_s>());
      break;
    case types::ue_level_qos_params:
      c.init(other.c.get<erab_level_qos_params_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
  }
}
conn_establishment_ind_ies_o::value_c&
conn_establishment_ind_ies_o::value_c::operator=(const conn_establishment_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ue_radio_cap:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::enhanced_coverage_restricted:
      c.set(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::dl_cp_security_info:
      c.set(other.c.get<dl_cp_security_info_s>());
      break;
    case types::ce_mode_brestricted:
      c.set(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::end_ind:
      c.set(other.c.get<end_ind_e>());
      break;
    case types::subscription_based_ue_differentiation_info:
      c.set(other.c.get<subscription_based_ue_differentiation_info_s>());
      break;
    case types::ue_level_qos_params:
      c.set(other.c.get<erab_level_qos_params_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
  }

  return *this;
}
uint64_t& conn_establishment_ind_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& conn_establishment_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
unbounded_octstring<true>& conn_establishment_ind_ies_o::value_c::ue_radio_cap()
{
  assert_choice_type(types::ue_radio_cap, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
enhanced_coverage_restricted_e& conn_establishment_ind_ies_o::value_c::enhanced_coverage_restricted()
{
  assert_choice_type(types::enhanced_coverage_restricted, type_, "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
dl_cp_security_info_s& conn_establishment_ind_ies_o::value_c::dl_cp_security_info()
{
  assert_choice_type(types::dl_cp_security_info, type_, "Value");
  return c.get<dl_cp_security_info_s>();
}
ce_mode_brestricted_e& conn_establishment_ind_ies_o::value_c::ce_mode_brestricted()
{
  assert_choice_type(types::ce_mode_brestricted, type_, "Value");
  return c.get<ce_mode_brestricted_e>();
}
end_ind_e& conn_establishment_ind_ies_o::value_c::end_ind()
{
  assert_choice_type(types::end_ind, type_, "Value");
  return c.get<end_ind_e>();
}
subscription_based_ue_differentiation_info_s&
conn_establishment_ind_ies_o::value_c::subscription_based_ue_differentiation_info()
{
  assert_choice_type(types::subscription_based_ue_differentiation_info, type_, "Value");
  return c.get<subscription_based_ue_differentiation_info_s>();
}
erab_level_qos_params_s& conn_establishment_ind_ies_o::value_c::ue_level_qos_params()
{
  assert_choice_type(types::ue_level_qos_params, type_, "Value");
  return c.get<erab_level_qos_params_s>();
}
const uint64_t& conn_establishment_ind_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& conn_establishment_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const unbounded_octstring<true>& conn_establishment_ind_ies_o::value_c::ue_radio_cap() const
{
  assert_choice_type(types::ue_radio_cap, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const enhanced_coverage_restricted_e& conn_establishment_ind_ies_o::value_c::enhanced_coverage_restricted() const
{
  assert_choice_type(types::enhanced_coverage_restricted, type_, "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
const dl_cp_security_info_s& conn_establishment_ind_ies_o::value_c::dl_cp_security_info() const
{
  assert_choice_type(types::dl_cp_security_info, type_, "Value");
  return c.get<dl_cp_security_info_s>();
}
const ce_mode_brestricted_e& conn_establishment_ind_ies_o::value_c::ce_mode_brestricted() const
{
  assert_choice_type(types::ce_mode_brestricted, type_, "Value");
  return c.get<ce_mode_brestricted_e>();
}
const end_ind_e& conn_establishment_ind_ies_o::value_c::end_ind() const
{
  assert_choice_type(types::end_ind, type_, "Value");
  return c.get<end_ind_e>();
}
const subscription_based_ue_differentiation_info_s&
conn_establishment_ind_ies_o::value_c::subscription_based_ue_differentiation_info() const
{
  assert_choice_type(types::subscription_based_ue_differentiation_info, type_, "Value");
  return c.get<subscription_based_ue_differentiation_info_s>();
}
const erab_level_qos_params_s& conn_establishment_ind_ies_o::value_c::ue_level_qos_params() const
{
  assert_choice_type(types::ue_level_qos_params, type_, "Value");
  return c.get<erab_level_qos_params_s>();
}
void conn_establishment_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ue_radio_cap:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::enhanced_coverage_restricted:
      j.write_str("EnhancedCoverageRestricted", "restricted");
      break;
    case types::dl_cp_security_info:
      j.write_fieldname("DL-CP-SecurityInformation");
      c.get<dl_cp_security_info_s>().to_json(j);
      break;
    case types::ce_mode_brestricted:
      j.write_str("CE-ModeBRestricted", c.get<ce_mode_brestricted_e>().to_string());
      break;
    case types::end_ind:
      j.write_str("EndIndication", c.get<end_ind_e>().to_string());
      break;
    case types::subscription_based_ue_differentiation_info:
      j.write_fieldname("Subscription-Based-UE-DifferentiationInfo");
      c.get<subscription_based_ue_differentiation_info_s>().to_json(j);
      break;
    case types::ue_level_qos_params:
      j.write_fieldname("E-RABLevelQoSParameters");
      c.get<erab_level_qos_params_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE conn_establishment_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().pack(bref));
      break;
    case types::dl_cp_security_info:
      HANDLE_CODE(c.get<dl_cp_security_info_s>().pack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().pack(bref));
      break;
    case types::end_ind:
      HANDLE_CODE(c.get<end_ind_e>().pack(bref));
      break;
    case types::subscription_based_ue_differentiation_info:
      HANDLE_CODE(c.get<subscription_based_ue_differentiation_info_s>().pack(bref));
      break;
    case types::ue_level_qos_params:
      HANDLE_CODE(c.get<erab_level_qos_params_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE conn_establishment_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().unpack(bref));
      break;
    case types::dl_cp_security_info:
      HANDLE_CODE(c.get<dl_cp_security_info_s>().unpack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().unpack(bref));
      break;
    case types::end_ind:
      HANDLE_CODE(c.get<end_ind_e>().unpack(bref));
      break;
    case types::subscription_based_ue_differentiation_info:
      HANDLE_CODE(c.get<subscription_based_ue_differentiation_info_s>().unpack(bref));
      break;
    case types::ue_level_qos_params:
      HANDLE_CODE(c.get<erab_level_qos_params_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* conn_establishment_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "OCTET STRING",
                                  "EnhancedCoverageRestricted",
                                  "DL-CP-SecurityInformation",
                                  "CE-ModeBRestricted",
                                  "EndIndication",
                                  "Subscription-Based-UE-DifferentiationInfo",
                                  "E-RABLevelQoSParameters"};
  return convert_enum_idx(options, 9, value, "conn_establishment_ind_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<conn_establishment_ind_ies_o>;

conn_establishment_ind_ies_container::conn_establishment_ind_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  ue_radio_cap(74, crit_e::ignore),
  enhanced_coverage_restricted(251, crit_e::ignore),
  dl_cp_security_info(253, crit_e::ignore),
  ce_mode_brestricted(271, crit_e::ignore),
  end_ind(280, crit_e::ignore),
  subscription_based_ue_differentiation_info(278, crit_e::ignore),
  ue_level_qos_params(252, crit_e::ignore)
{}
SRSASN_CODE conn_establishment_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += ue_radio_cap_present ? 1 : 0;
  nof_ies += enhanced_coverage_restricted_present ? 1 : 0;
  nof_ies += dl_cp_security_info_present ? 1 : 0;
  nof_ies += ce_mode_brestricted_present ? 1 : 0;
  nof_ies += end_ind_present ? 1 : 0;
  nof_ies += subscription_based_ue_differentiation_info_present ? 1 : 0;
  nof_ies += ue_level_qos_params_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (ue_radio_cap_present) {
    HANDLE_CODE(ue_radio_cap.pack(bref));
  }
  if (enhanced_coverage_restricted_present) {
    HANDLE_CODE(enhanced_coverage_restricted.pack(bref));
  }
  if (dl_cp_security_info_present) {
    HANDLE_CODE(dl_cp_security_info.pack(bref));
  }
  if (ce_mode_brestricted_present) {
    HANDLE_CODE(ce_mode_brestricted.pack(bref));
  }
  if (end_ind_present) {
    HANDLE_CODE(end_ind.pack(bref));
  }
  if (subscription_based_ue_differentiation_info_present) {
    HANDLE_CODE(subscription_based_ue_differentiation_info.pack(bref));
  }
  if (ue_level_qos_params_present) {
    HANDLE_CODE(ue_level_qos_params.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE conn_establishment_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 74: {
        ue_radio_cap_present = true;
        ue_radio_cap.id      = id;
        HANDLE_CODE(ue_radio_cap.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_radio_cap.value.unpack(bref));
        break;
      }
      case 251: {
        enhanced_coverage_restricted_present = true;
        enhanced_coverage_restricted.id      = id;
        HANDLE_CODE(enhanced_coverage_restricted.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enhanced_coverage_restricted.value.unpack(bref));
        break;
      }
      case 253: {
        dl_cp_security_info_present = true;
        dl_cp_security_info.id      = id;
        HANDLE_CODE(dl_cp_security_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(dl_cp_security_info.value.unpack(bref));
        break;
      }
      case 271: {
        ce_mode_brestricted_present = true;
        ce_mode_brestricted.id      = id;
        HANDLE_CODE(ce_mode_brestricted.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ce_mode_brestricted.value.unpack(bref));
        break;
      }
      case 280: {
        end_ind_present = true;
        end_ind.id      = id;
        HANDLE_CODE(end_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(end_ind.value.unpack(bref));
        break;
      }
      case 278: {
        subscription_based_ue_differentiation_info_present = true;
        subscription_based_ue_differentiation_info.id      = id;
        HANDLE_CODE(subscription_based_ue_differentiation_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(subscription_based_ue_differentiation_info.value.unpack(bref));
        break;
      }
      case 252: {
        ue_level_qos_params_present = true;
        ue_level_qos_params.id      = id;
        HANDLE_CODE(ue_level_qos_params.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_level_qos_params.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void conn_establishment_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (ue_radio_cap_present) {
    j.write_fieldname("");
    ue_radio_cap.to_json(j);
  }
  if (enhanced_coverage_restricted_present) {
    j.write_fieldname("");
    enhanced_coverage_restricted.to_json(j);
  }
  if (dl_cp_security_info_present) {
    j.write_fieldname("");
    dl_cp_security_info.to_json(j);
  }
  if (ce_mode_brestricted_present) {
    j.write_fieldname("");
    ce_mode_brestricted.to_json(j);
  }
  if (end_ind_present) {
    j.write_fieldname("");
    end_ind.to_json(j);
  }
  if (subscription_based_ue_differentiation_info_present) {
    j.write_fieldname("");
    subscription_based_ue_differentiation_info.to_json(j);
  }
  if (ue_level_qos_params_present) {
    j.write_fieldname("");
    ue_level_qos_params.to_json(j);
  }
  j.end_obj();
}

// ENB-ID ::= CHOICE
void enb_id_c::destroy_()
{
  switch (type_) {
    case types::macro_enb_id:
      c.destroy<fixed_bitstring<20, false, true> >();
      break;
    case types::home_enb_id:
      c.destroy<fixed_bitstring<28, false, true> >();
      break;
    case types::short_macro_enb_id:
      c.destroy<fixed_bitstring<18, false, true> >();
      break;
    case types::long_macro_enb_id:
      c.destroy<fixed_bitstring<21, false, true> >();
      break;
    default:
      break;
  }
}
void enb_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::macro_enb_id:
      c.init<fixed_bitstring<20, false, true> >();
      break;
    case types::home_enb_id:
      c.init<fixed_bitstring<28, false, true> >();
      break;
    case types::short_macro_enb_id:
      c.init<fixed_bitstring<18, false, true> >();
      break;
    case types::long_macro_enb_id:
      c.init<fixed_bitstring<21, false, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
  }
}
enb_id_c::enb_id_c(const enb_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::macro_enb_id:
      c.init(other.c.get<fixed_bitstring<20, false, true> >());
      break;
    case types::home_enb_id:
      c.init(other.c.get<fixed_bitstring<28, false, true> >());
      break;
    case types::short_macro_enb_id:
      c.init(other.c.get<fixed_bitstring<18, false, true> >());
      break;
    case types::long_macro_enb_id:
      c.init(other.c.get<fixed_bitstring<21, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
  }
}
enb_id_c& enb_id_c::operator=(const enb_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::macro_enb_id:
      c.set(other.c.get<fixed_bitstring<20, false, true> >());
      break;
    case types::home_enb_id:
      c.set(other.c.get<fixed_bitstring<28, false, true> >());
      break;
    case types::short_macro_enb_id:
      c.set(other.c.get<fixed_bitstring<18, false, true> >());
      break;
    case types::long_macro_enb_id:
      c.set(other.c.get<fixed_bitstring<21, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
  }

  return *this;
}
fixed_bitstring<20, false, true>& enb_id_c::set_macro_enb_id()
{
  set(types::macro_enb_id);
  return c.get<fixed_bitstring<20, false, true> >();
}
fixed_bitstring<28, false, true>& enb_id_c::set_home_enb_id()
{
  set(types::home_enb_id);
  return c.get<fixed_bitstring<28, false, true> >();
}
fixed_bitstring<18, false, true>& enb_id_c::set_short_macro_enb_id()
{
  set(types::short_macro_enb_id);
  return c.get<fixed_bitstring<18, false, true> >();
}
fixed_bitstring<21, false, true>& enb_id_c::set_long_macro_enb_id()
{
  set(types::long_macro_enb_id);
  return c.get<fixed_bitstring<21, false, true> >();
}
void enb_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::macro_enb_id:
      j.write_str("macroENB-ID", c.get<fixed_bitstring<20, false, true> >().to_string());
      break;
    case types::home_enb_id:
      j.write_str("homeENB-ID", c.get<fixed_bitstring<28, false, true> >().to_string());
      break;
    case types::short_macro_enb_id:
      j.write_str("short-macroENB-ID", c.get<fixed_bitstring<18, false, true> >().to_string());
      break;
    case types::long_macro_enb_id:
      j.write_str("long-macroENB-ID", c.get<fixed_bitstring<21, false, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::macro_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<20, false, true> >().pack(bref)));
      break;
    case types::home_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<28, false, true> >().pack(bref)));
      break;
    case types::short_macro_enb_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_bitstring<18, false, true> >().pack(bref)));
    } break;
    case types::long_macro_enb_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_bitstring<21, false, true> >().pack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::macro_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<20, false, true> >().unpack(bref)));
      break;
    case types::home_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<28, false, true> >().unpack(bref)));
      break;
    case types::short_macro_enb_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_bitstring<18, false, true> >().unpack(bref)));
    } break;
    case types::long_macro_enb_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_bitstring<21, false, true> >().unpack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* enb_id_c::types_opts::to_string() const
{
  static const char* options[] = {"macroENB-ID", "homeENB-ID", "short-macroENB-ID", "long-macroENB-ID"};
  return convert_enum_idx(options, 4, value, "enb_id_c::types");
}

// GNB-Identity ::= CHOICE
void gnb_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("gNB-ID", c.to_string());
  j.end_obj();
}
SRSASN_CODE gnb_id_c::pack(bit_ref& bref) const
{
  pack_enum(bref, type());
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE gnb_id_c::unpack(cbit_ref& bref)
{
  types e;
  unpack_enum(e, bref);
  if (e != type()) {
    log_invalid_choice_id(e, "gnb_id_c");
    return SRSASN_ERROR_DECODE_FAIL;
  }
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* gnb_id_c::types_opts::to_string() const
{
  static const char* options[] = {"gNB-ID"};
  return convert_enum_idx(options, 1, value, "gnb_id_c::types");
}

// Global-ENB-ID ::= SEQUENCE
SRSASN_CODE global_enb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(enb_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE global_enb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(enb_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void global_enb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_fieldname("eNB-ID");
  enb_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Global-GNB-ID ::= SEQUENCE
SRSASN_CODE global_gnb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(gnb_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE global_gnb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(gnb_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void global_gnb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMN-Identity", plmn_id.to_string());
  j.write_fieldname("gNB-ID");
  gnb_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// GNB ::= SEQUENCE
SRSASN_CODE gnb_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_g_nb_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE gnb_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_g_nb_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void gnb_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-gNB-ID");
  global_g_nb_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// NG-eNB ::= SEQUENCE
SRSASN_CODE ng_enb_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_ng_enb_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ng_enb_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_ng_enb_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ng_enb_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-ng-eNB-ID");
  global_ng_enb_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Global-RAN-NODE-ID ::= CHOICE
void global_ran_node_id_c::destroy_()
{
  switch (type_) {
    case types::gnb:
      c.destroy<gnb_s>();
      break;
    case types::ng_enb:
      c.destroy<ng_enb_s>();
      break;
    default:
      break;
  }
}
void global_ran_node_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::gnb:
      c.init<gnb_s>();
      break;
    case types::ng_enb:
      c.init<ng_enb_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
  }
}
global_ran_node_id_c::global_ran_node_id_c(const global_ran_node_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::gnb:
      c.init(other.c.get<gnb_s>());
      break;
    case types::ng_enb:
      c.init(other.c.get<ng_enb_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
  }
}
global_ran_node_id_c& global_ran_node_id_c::operator=(const global_ran_node_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::gnb:
      c.set(other.c.get<gnb_s>());
      break;
    case types::ng_enb:
      c.set(other.c.get<ng_enb_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
  }

  return *this;
}
gnb_s& global_ran_node_id_c::set_gnb()
{
  set(types::gnb);
  return c.get<gnb_s>();
}
ng_enb_s& global_ran_node_id_c::set_ng_enb()
{
  set(types::ng_enb);
  return c.get<ng_enb_s>();
}
void global_ran_node_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::gnb:
      j.write_fieldname("gNB");
      c.get<gnb_s>().to_json(j);
      break;
    case types::ng_enb:
      j.write_fieldname("ng-eNB");
      c.get<ng_enb_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
  }
  j.end_obj();
}
SRSASN_CODE global_ran_node_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::gnb:
      HANDLE_CODE(c.get<gnb_s>().pack(bref));
      break;
    case types::ng_enb:
      HANDLE_CODE(c.get<ng_enb_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE global_ran_node_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::gnb:
      HANDLE_CODE(c.get<gnb_s>().unpack(bref));
      break;
    case types::ng_enb:
      HANDLE_CODE(c.get<ng_enb_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* global_ran_node_id_c::types_opts::to_string() const
{
  static const char* options[] = {"gNB", "ng-eNB"};
  return convert_enum_idx(options, 2, value, "global_ran_node_id_c::types");
}

// ContextatSource ::= SEQUENCE
SRSASN_CODE contextat_source_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(source_ng_ran_node_id.pack(bref));
  HANDLE_CODE(pack_integer(bref, ran_ue_ngap_id, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE contextat_source_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(source_ng_ran_node_id.unpack(bref));
  HANDLE_CODE(unpack_integer(ran_ue_ngap_id, bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void contextat_source_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("sourceNG-RAN-node-ID");
  source_ng_ran_node_id.to_json(j);
  j.write_int("rAN-UE-NGAP-ID", ran_ue_ngap_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TypeOfError ::= ENUMERATED
const char* type_of_error_opts::to_string() const
{
  static const char* options[] = {"not-understood", "missing"};
  return convert_enum_idx(options, 2, value, "type_of_error_e");
}

// CriticalityDiagnostics-IE-Item ::= SEQUENCE
SRSASN_CODE crit_diagnostics_ie_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(iecrit.pack(bref));
  HANDLE_CODE(pack_integer(bref, ie_id, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(type_of_error.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE crit_diagnostics_ie_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(iecrit.unpack(bref));
  HANDLE_CODE(unpack_integer(ie_id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(type_of_error.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void crit_diagnostics_ie_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("iECriticality", iecrit.to_string());
  j.write_int("iE-ID", ie_id);
  j.write_str("typeOfError", type_of_error.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TriggeringMessage ::= ENUMERATED
const char* trigger_msg_opts::to_string() const
{
  static const char* options[] = {"initiating-message", "successful-outcome", "unsuccessfull-outcome"};
  return convert_enum_idx(options, 3, value, "trigger_msg_e");
}

// CriticalityDiagnostics ::= SEQUENCE
SRSASN_CODE crit_diagnostics_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(proc_code_present, 1));
  HANDLE_CODE(bref.pack(trigger_msg_present, 1));
  HANDLE_CODE(bref.pack(proc_crit_present, 1));
  HANDLE_CODE(bref.pack(ies_crit_diagnostics_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (proc_code_present) {
    HANDLE_CODE(pack_integer(bref, proc_code, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  if (trigger_msg_present) {
    HANDLE_CODE(trigger_msg.pack(bref));
  }
  if (proc_crit_present) {
    HANDLE_CODE(proc_crit.pack(bref));
  }
  if (ies_crit_diagnostics_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ies_crit_diagnostics, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE crit_diagnostics_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(proc_code_present, 1));
  HANDLE_CODE(bref.unpack(trigger_msg_present, 1));
  HANDLE_CODE(bref.unpack(proc_crit_present, 1));
  HANDLE_CODE(bref.unpack(ies_crit_diagnostics_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (proc_code_present) {
    HANDLE_CODE(unpack_integer(proc_code, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  if (trigger_msg_present) {
    HANDLE_CODE(trigger_msg.unpack(bref));
  }
  if (proc_crit_present) {
    HANDLE_CODE(proc_crit.unpack(bref));
  }
  if (ies_crit_diagnostics_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ies_crit_diagnostics, bref, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void crit_diagnostics_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (proc_code_present) {
    j.write_int("procedureCode", proc_code);
  }
  if (trigger_msg_present) {
    j.write_str("triggeringMessage", trigger_msg.to_string());
  }
  if (proc_crit_present) {
    j.write_str("procedureCriticality", proc_crit.to_string());
  }
  if (ies_crit_diagnostics_present) {
    j.start_array("iEsCriticalityDiagnostics");
    for (const auto& e1 : ies_crit_diagnostics) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DeactivateTraceIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t deactiv_trace_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 86};
  return map_enum_number(options, 3, idx, "id");
}
bool deactiv_trace_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 86};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e deactiv_trace_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 86:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
deactiv_trace_ies_o::value_c deactiv_trace_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 86:
      ret.set(value_c::types::e_utran_trace_id);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e deactiv_trace_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 86:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void deactiv_trace_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::e_utran_trace_id:
      c.destroy<fixed_octstring<8, true> >();
      break;
    default:
      break;
  }
}
void deactiv_trace_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::e_utran_trace_id:
      c.init<fixed_octstring<8, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
}
deactiv_trace_ies_o::value_c::value_c(const deactiv_trace_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      c.init(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
}
deactiv_trace_ies_o::value_c& deactiv_trace_ies_o::value_c::operator=(const deactiv_trace_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      c.set(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }

  return *this;
}
uint64_t& deactiv_trace_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& deactiv_trace_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
fixed_octstring<8, true>& deactiv_trace_ies_o::value_c::e_utran_trace_id()
{
  assert_choice_type(types::e_utran_trace_id, type_, "Value");
  return c.get<fixed_octstring<8, true> >();
}
const uint64_t& deactiv_trace_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& deactiv_trace_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const fixed_octstring<8, true>& deactiv_trace_ies_o::value_c::e_utran_trace_id() const
{
  assert_choice_type(types::e_utran_trace_id, type_, "Value");
  return c.get<fixed_octstring<8, true> >();
}
void deactiv_trace_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<8, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE deactiv_trace_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::e_utran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().pack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE deactiv_trace_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::e_utran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().unpack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* deactiv_trace_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)", "INTEGER (0..16777215)", "OCTET STRING"};
  return convert_enum_idx(options, 3, value, "deactiv_trace_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<deactiv_trace_ies_o>;

deactiv_trace_ies_container::deactiv_trace_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject), enb_ue_s1ap_id(8, crit_e::reject), e_utran_trace_id(86, crit_e::ignore)
{}
SRSASN_CODE deactiv_trace_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(e_utran_trace_id.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE deactiv_trace_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 86: {
        nof_mandatory_ies--;
        e_utran_trace_id.id = id;
        HANDLE_CODE(e_utran_trace_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(e_utran_trace_id.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void deactiv_trace_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  e_utran_trace_id.to_json(j);
  j.end_obj();
}

// ForbiddenLAs-Item ::= SEQUENCE
SRSASN_CODE forbidden_las_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_lacs, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE forbidden_las_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(forbidden_lacs, bref, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void forbidden_las_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMN-Identity", plmn_id.to_string());
  j.start_array("forbiddenLACs");
  for (const auto& e1 : forbidden_lacs) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ForbiddenTAs-Item ::= SEQUENCE
SRSASN_CODE forbidden_tas_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_tacs, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE forbidden_tas_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(forbidden_tacs, bref, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void forbidden_tas_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMN-Identity", plmn_id.to_string());
  j.start_array("forbiddenTACs");
  for (const auto& e1 : forbidden_tacs) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// NRrestrictionin5GS ::= ENUMERATED
const char* nrrestrictin5_gs_opts::to_string() const
{
  static const char* options[] = {"nRrestrictedin5GS"};
  return convert_enum_idx(options, 1, value, "nrrestrictin5_gs_e");
}
uint8_t nrrestrictin5_gs_opts::to_number() const
{
  static const uint8_t options[] = {5};
  return map_enum_number(options, 1, value, "nrrestrictin5_gs_e");
}

// NRrestrictioninEPSasSecondaryRAT ::= ENUMERATED
const char* nrrestrictin_ep_sas_secondary_rat_opts::to_string() const
{
  static const char* options[] = {"nRrestrictedinEPSasSecondaryRAT"};
  return convert_enum_idx(options, 1, value, "nrrestrictin_ep_sas_secondary_rat_e");
}

// UnlicensedSpectrumRestriction ::= ENUMERATED
const char* unlicensed_spec_restrict_opts::to_string() const
{
  static const char* options[] = {"unlicensed-restricted"};
  return convert_enum_idx(options, 1, value, "unlicensed_spec_restrict_e");
}

// ForbiddenInterRATs ::= ENUMERATED
const char* forbidden_inter_rats_opts::to_string() const
{
  static const char* options[] = {"all", "geran", "utran", "cdma2000", "geranandutran", "cdma2000andutran"};
  return convert_enum_idx(options, 6, value, "forbidden_inter_rats_e");
}

// HandoverRestrictionList-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t ho_restrict_list_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {261, 270, 282, 287, 290};
  return map_enum_number(options, 5, idx, "id");
}
bool ho_restrict_list_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {261, 270, 282, 287, 290};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_restrict_list_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 261:
      return crit_e::ignore;
    case 270:
      return crit_e::ignore;
    case 282:
      return crit_e::ignore;
    case 287:
      return crit_e::ignore;
    case 290:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_restrict_list_ext_ies_o::ext_c ho_restrict_list_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 261:
      ret.set(ext_c::types::nrrestrictin_ep_sas_secondary_rat);
      break;
    case 270:
      ret.set(ext_c::types::unlicensed_spec_restrict);
      break;
    case 282:
      ret.set(ext_c::types::cn_type_restricts);
      break;
    case 287:
      ret.set(ext_c::types::nrrestrictin5_gs);
      break;
    case 290:
      ret.set(ext_c::types::last_ng_ranplmn_id);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_restrict_list_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 261:
      return presence_e::optional;
    case 270:
      return presence_e::optional;
    case 282:
      return presence_e::optional;
    case 287:
      return presence_e::optional;
    case 290:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void ho_restrict_list_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::cn_type_restricts:
      c.destroy<cn_type_restricts_l>();
      break;
    case types::last_ng_ranplmn_id:
      c.destroy<plm_nid>();
      break;
    default:
      break;
  }
}
void ho_restrict_list_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::nrrestrictin_ep_sas_secondary_rat:
      break;
    case types::unlicensed_spec_restrict:
      break;
    case types::cn_type_restricts:
      c.init<cn_type_restricts_l>();
      break;
    case types::nrrestrictin5_gs:
      break;
    case types::last_ng_ranplmn_id:
      c.init<plm_nid>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_restrict_list_ext_ies_o::ext_c");
  }
}
ho_restrict_list_ext_ies_o::ext_c::ext_c(const ho_restrict_list_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::nrrestrictin_ep_sas_secondary_rat:
      c.init(other.c.get<nrrestrictin_ep_sas_secondary_rat_e>());
      break;
    case types::unlicensed_spec_restrict:
      c.init(other.c.get<unlicensed_spec_restrict_e>());
      break;
    case types::cn_type_restricts:
      c.init(other.c.get<cn_type_restricts_l>());
      break;
    case types::nrrestrictin5_gs:
      c.init(other.c.get<nrrestrictin5_gs_e>());
      break;
    case types::last_ng_ranplmn_id:
      c.init(other.c.get<plm_nid>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_restrict_list_ext_ies_o::ext_c");
  }
}
ho_restrict_list_ext_ies_o::ext_c&
ho_restrict_list_ext_ies_o::ext_c::operator=(const ho_restrict_list_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::nrrestrictin_ep_sas_secondary_rat:
      c.set(other.c.get<nrrestrictin_ep_sas_secondary_rat_e>());
      break;
    case types::unlicensed_spec_restrict:
      c.set(other.c.get<unlicensed_spec_restrict_e>());
      break;
    case types::cn_type_restricts:
      c.set(other.c.get<cn_type_restricts_l>());
      break;
    case types::nrrestrictin5_gs:
      c.set(other.c.get<nrrestrictin5_gs_e>());
      break;
    case types::last_ng_ranplmn_id:
      c.set(other.c.get<plm_nid>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_restrict_list_ext_ies_o::ext_c");
  }

  return *this;
}
nrrestrictin_ep_sas_secondary_rat_e& ho_restrict_list_ext_ies_o::ext_c::nrrestrictin_ep_sas_secondary_rat()
{
  assert_choice_type(types::nrrestrictin_ep_sas_secondary_rat, type_, "Extension");
  return c.get<nrrestrictin_ep_sas_secondary_rat_e>();
}
unlicensed_spec_restrict_e& ho_restrict_list_ext_ies_o::ext_c::unlicensed_spec_restrict()
{
  assert_choice_type(types::unlicensed_spec_restrict, type_, "Extension");
  return c.get<unlicensed_spec_restrict_e>();
}
cn_type_restricts_l& ho_restrict_list_ext_ies_o::ext_c::cn_type_restricts()
{
  assert_choice_type(types::cn_type_restricts, type_, "Extension");
  return c.get<cn_type_restricts_l>();
}
nrrestrictin5_gs_e& ho_restrict_list_ext_ies_o::ext_c::nrrestrictin5_gs()
{
  assert_choice_type(types::nrrestrictin5_gs, type_, "Extension");
  return c.get<nrrestrictin5_gs_e>();
}
plm_nid& ho_restrict_list_ext_ies_o::ext_c::last_ng_ranplmn_id()
{
  assert_choice_type(types::last_ng_ranplmn_id, type_, "Extension");
  return c.get<plm_nid>();
}
const nrrestrictin_ep_sas_secondary_rat_e& ho_restrict_list_ext_ies_o::ext_c::nrrestrictin_ep_sas_secondary_rat() const
{
  assert_choice_type(types::nrrestrictin_ep_sas_secondary_rat, type_, "Extension");
  return c.get<nrrestrictin_ep_sas_secondary_rat_e>();
}
const unlicensed_spec_restrict_e& ho_restrict_list_ext_ies_o::ext_c::unlicensed_spec_restrict() const
{
  assert_choice_type(types::unlicensed_spec_restrict, type_, "Extension");
  return c.get<unlicensed_spec_restrict_e>();
}
const cn_type_restricts_l& ho_restrict_list_ext_ies_o::ext_c::cn_type_restricts() const
{
  assert_choice_type(types::cn_type_restricts, type_, "Extension");
  return c.get<cn_type_restricts_l>();
}
const nrrestrictin5_gs_e& ho_restrict_list_ext_ies_o::ext_c::nrrestrictin5_gs() const
{
  assert_choice_type(types::nrrestrictin5_gs, type_, "Extension");
  return c.get<nrrestrictin5_gs_e>();
}
const plm_nid& ho_restrict_list_ext_ies_o::ext_c::last_ng_ranplmn_id() const
{
  assert_choice_type(types::last_ng_ranplmn_id, type_, "Extension");
  return c.get<plm_nid>();
}
void ho_restrict_list_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::nrrestrictin_ep_sas_secondary_rat:
      j.write_str("NRrestrictioninEPSasSecondaryRAT", "nRrestrictedinEPSasSecondaryRAT");
      break;
    case types::unlicensed_spec_restrict:
      j.write_str("UnlicensedSpectrumRestriction", "unlicensed-restricted");
      break;
    case types::cn_type_restricts:
      j.start_array("CNTypeRestrictions");
      for (const auto& e1 : c.get<cn_type_restricts_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::nrrestrictin5_gs:
      j.write_str("NRrestrictionin5GS", "nRrestrictedin5GS");
      break;
    case types::last_ng_ranplmn_id:
      j.write_str("PLMNidentity", c.get<plm_nid>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "ho_restrict_list_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_restrict_list_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::nrrestrictin_ep_sas_secondary_rat:
      HANDLE_CODE(c.get<nrrestrictin_ep_sas_secondary_rat_e>().pack(bref));
      break;
    case types::unlicensed_spec_restrict:
      HANDLE_CODE(c.get<unlicensed_spec_restrict_e>().pack(bref));
      break;
    case types::cn_type_restricts:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cn_type_restricts_l>(), 1, 16, true));
      break;
    case types::nrrestrictin5_gs:
      HANDLE_CODE(c.get<nrrestrictin5_gs_e>().pack(bref));
      break;
    case types::last_ng_ranplmn_id:
      HANDLE_CODE(c.get<plm_nid>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_restrict_list_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_restrict_list_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::nrrestrictin_ep_sas_secondary_rat:
      HANDLE_CODE(c.get<nrrestrictin_ep_sas_secondary_rat_e>().unpack(bref));
      break;
    case types::unlicensed_spec_restrict:
      HANDLE_CODE(c.get<unlicensed_spec_restrict_e>().unpack(bref));
      break;
    case types::cn_type_restricts:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cn_type_restricts_l>(), bref, 1, 16, true));
      break;
    case types::nrrestrictin5_gs:
      HANDLE_CODE(c.get<nrrestrictin5_gs_e>().unpack(bref));
      break;
    case types::last_ng_ranplmn_id:
      HANDLE_CODE(c.get<plm_nid>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_restrict_list_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_restrict_list_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"NRrestrictioninEPSasSecondaryRAT",
                                  "UnlicensedSpectrumRestriction",
                                  "CNTypeRestrictions",
                                  "NRrestrictionin5GS",
                                  "PLMNidentity"};
  return convert_enum_idx(options, 5, value, "ho_restrict_list_ext_ies_o::ext_c::types");
}
uint8_t ho_restrict_list_ext_ies_o::ext_c::types_opts::to_number() const
{
  if (value == nrrestrictin5_gs) {
    return 5;
  }
  invalid_enum_number(value, "ho_restrict_list_ext_ies_o::ext_c::types");
  return 0;
}

// DLNASPDUDeliveryAckRequest ::= ENUMERATED
const char* dlnaspdu_delivery_ack_request_opts::to_string() const
{
  static const char* options[] = {"requested"};
  return convert_enum_idx(options, 1, value, "dlnaspdu_delivery_ack_request_e");
}

template struct asn1::protocol_ext_field_s<ho_restrict_list_ext_ies_o>;

ho_restrict_list_ext_ies_container::ho_restrict_list_ext_ies_container() :
  nrrestrictin_ep_sas_secondary_rat(261, crit_e::ignore),
  unlicensed_spec_restrict(270, crit_e::ignore),
  cn_type_restricts(282, crit_e::ignore),
  nrrestrictin5_gs(287, crit_e::ignore),
  last_ng_ranplmn_id(290, crit_e::ignore)
{}
SRSASN_CODE ho_restrict_list_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += nrrestrictin_ep_sas_secondary_rat_present ? 1 : 0;
  nof_ies += unlicensed_spec_restrict_present ? 1 : 0;
  nof_ies += cn_type_restricts_present ? 1 : 0;
  nof_ies += nrrestrictin5_gs_present ? 1 : 0;
  nof_ies += last_ng_ranplmn_id_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (nrrestrictin_ep_sas_secondary_rat_present) {
    HANDLE_CODE(nrrestrictin_ep_sas_secondary_rat.pack(bref));
  }
  if (unlicensed_spec_restrict_present) {
    HANDLE_CODE(unlicensed_spec_restrict.pack(bref));
  }
  if (cn_type_restricts_present) {
    HANDLE_CODE(cn_type_restricts.pack(bref));
  }
  if (nrrestrictin5_gs_present) {
    HANDLE_CODE(nrrestrictin5_gs.pack(bref));
  }
  if (last_ng_ranplmn_id_present) {
    HANDLE_CODE(last_ng_ranplmn_id.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_restrict_list_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 261: {
        nrrestrictin_ep_sas_secondary_rat_present = true;
        nrrestrictin_ep_sas_secondary_rat.id      = id;
        HANDLE_CODE(nrrestrictin_ep_sas_secondary_rat.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nrrestrictin_ep_sas_secondary_rat.ext.unpack(bref));
        break;
      }
      case 270: {
        unlicensed_spec_restrict_present = true;
        unlicensed_spec_restrict.id      = id;
        HANDLE_CODE(unlicensed_spec_restrict.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(unlicensed_spec_restrict.ext.unpack(bref));
        break;
      }
      case 282: {
        cn_type_restricts_present = true;
        cn_type_restricts.id      = id;
        HANDLE_CODE(cn_type_restricts.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cn_type_restricts.ext.unpack(bref));
        break;
      }
      case 287: {
        nrrestrictin5_gs_present = true;
        nrrestrictin5_gs.id      = id;
        HANDLE_CODE(nrrestrictin5_gs.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nrrestrictin5_gs.ext.unpack(bref));
        break;
      }
      case 290: {
        last_ng_ranplmn_id_present = true;
        last_ng_ranplmn_id.id      = id;
        HANDLE_CODE(last_ng_ranplmn_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(last_ng_ranplmn_id.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void ho_restrict_list_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (nrrestrictin_ep_sas_secondary_rat_present) {
    j.write_fieldname("");
    nrrestrictin_ep_sas_secondary_rat.to_json(j);
  }
  if (unlicensed_spec_restrict_present) {
    j.write_fieldname("");
    unlicensed_spec_restrict.to_json(j);
  }
  if (cn_type_restricts_present) {
    j.write_fieldname("");
    cn_type_restricts.to_json(j);
  }
  if (nrrestrictin5_gs_present) {
    j.write_fieldname("");
    nrrestrictin5_gs.to_json(j);
  }
  if (last_ng_ranplmn_id_present) {
    j.write_fieldname("");
    last_ng_ranplmn_id.to_json(j);
  }
  j.end_obj();
}

// HandoverRestrictionList ::= SEQUENCE
SRSASN_CODE ho_restrict_list_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(equivalent_plmns_present, 1));
  HANDLE_CODE(bref.pack(forbidden_tas_present, 1));
  HANDLE_CODE(bref.pack(forbidden_las_present, 1));
  HANDLE_CODE(bref.pack(forbidden_inter_rats_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(serving_plmn.pack(bref));
  if (equivalent_plmns_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, equivalent_plmns, 1, 15, true));
  }
  if (forbidden_tas_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_tas, 1, 16, true));
  }
  if (forbidden_las_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_las, 1, 16, true));
  }
  if (forbidden_inter_rats_present) {
    HANDLE_CODE(forbidden_inter_rats.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_restrict_list_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(equivalent_plmns_present, 1));
  HANDLE_CODE(bref.unpack(forbidden_tas_present, 1));
  HANDLE_CODE(bref.unpack(forbidden_las_present, 1));
  HANDLE_CODE(bref.unpack(forbidden_inter_rats_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(serving_plmn.unpack(bref));
  if (equivalent_plmns_present) {
    HANDLE_CODE(unpack_dyn_seq_of(equivalent_plmns, bref, 1, 15, true));
  }
  if (forbidden_tas_present) {
    HANDLE_CODE(unpack_dyn_seq_of(forbidden_tas, bref, 1, 16, true));
  }
  if (forbidden_las_present) {
    HANDLE_CODE(unpack_dyn_seq_of(forbidden_las, bref, 1, 16, true));
  }
  if (forbidden_inter_rats_present) {
    HANDLE_CODE(forbidden_inter_rats.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ho_restrict_list_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("servingPLMN", serving_plmn.to_string());
  if (equivalent_plmns_present) {
    j.start_array("equivalentPLMNs");
    for (const auto& e1 : equivalent_plmns) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (forbidden_tas_present) {
    j.start_array("forbiddenTAs");
    for (const auto& e1 : forbidden_tas) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (forbidden_las_present) {
    j.start_array("forbiddenLAs");
    for (const auto& e1 : forbidden_las) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (forbidden_inter_rats_present) {
    j.write_str("forbiddenInterRATs", forbidden_inter_rats.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// NRUESecurityCapabilities ::= SEQUENCE
SRSASN_CODE nrue_security_cap_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nrencryption_algorithms.pack(bref));
  HANDLE_CODE(nrintegrity_protection_algorithms.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE nrue_security_cap_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nrencryption_algorithms.unpack(bref));
  HANDLE_CODE(nrintegrity_protection_algorithms.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void nrue_security_cap_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("nRencryptionAlgorithms", nrencryption_algorithms.to_string());
  j.write_str("nRintegrityProtectionAlgorithms", nrintegrity_protection_algorithms.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PendingDataIndication ::= ENUMERATED
const char* pending_data_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "pending_data_ind_e");
}

// SRVCCOperationPossible ::= ENUMERATED
const char* srvcc_operation_possible_opts::to_string() const
{
  static const char* options[] = {"possible"};
  return convert_enum_idx(options, 1, value, "srvcc_operation_possible_e");
}

// UECapabilityInfoRequest ::= ENUMERATED
const char* ue_cap_info_request_opts::to_string() const
{
  static const char* options[] = {"requested"};
  return convert_enum_idx(options, 1, value, "ue_cap_info_request_e");
}

// DownlinkNASTransport-IEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t dl_nas_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 26, 41, 106, 124, 74, 249, 251, 269, 271, 275, 280, 283, 278, 299};
  return map_enum_number(options, 16, idx, "id");
}
bool dl_nas_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 26, 41, 106, 124, 74, 249, 251, 269, 271, 275, 280, 283, 278, 299};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_nas_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 26:
      return crit_e::reject;
    case 41:
      return crit_e::ignore;
    case 106:
      return crit_e::ignore;
    case 124:
      return crit_e::ignore;
    case 74:
      return crit_e::ignore;
    case 249:
      return crit_e::ignore;
    case 251:
      return crit_e::ignore;
    case 269:
      return crit_e::ignore;
    case 271:
      return crit_e::ignore;
    case 275:
      return crit_e::ignore;
    case 280:
      return crit_e::ignore;
    case 283:
      return crit_e::ignore;
    case 278:
      return crit_e::ignore;
    case 299:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_nas_transport_ies_o::value_c dl_nas_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 26:
      ret.set(value_c::types::nas_pdu);
      break;
    case 41:
      ret.set(value_c::types::ho_restrict_list);
      break;
    case 106:
      ret.set(value_c::types::subscriber_profile_idfor_rfp);
      break;
    case 124:
      ret.set(value_c::types::srvcc_operation_possible);
      break;
    case 74:
      ret.set(value_c::types::ue_radio_cap);
      break;
    case 249:
      ret.set(value_c::types::dlnaspdu_delivery_ack_request);
      break;
    case 251:
      ret.set(value_c::types::enhanced_coverage_restricted);
      break;
    case 269:
      ret.set(value_c::types::nrue_security_cap);
      break;
    case 271:
      ret.set(value_c::types::ce_mode_brestricted);
      break;
    case 275:
      ret.set(value_c::types::ue_cap_info_request);
      break;
    case 280:
      ret.set(value_c::types::end_ind);
      break;
    case 283:
      ret.set(value_c::types::pending_data_ind);
      break;
    case 278:
      ret.set(value_c::types::subscription_based_ue_differentiation_info);
      break;
    case 299:
      ret.set(value_c::types::add_rrm_prio_idx);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_nas_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 26:
      return presence_e::mandatory;
    case 41:
      return presence_e::optional;
    case 106:
      return presence_e::optional;
    case 124:
      return presence_e::optional;
    case 74:
      return presence_e::optional;
    case 249:
      return presence_e::optional;
    case 251:
      return presence_e::optional;
    case 269:
      return presence_e::optional;
    case 271:
      return presence_e::optional;
    case 275:
      return presence_e::optional;
    case 280:
      return presence_e::optional;
    case 283:
      return presence_e::optional;
    case 278:
      return presence_e::optional;
    case 299:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_nas_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ho_restrict_list:
      c.destroy<ho_restrict_list_s>();
      break;
    case types::ue_radio_cap:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::nrue_security_cap:
      c.destroy<nrue_security_cap_s>();
      break;
    case types::subscription_based_ue_differentiation_info:
      c.destroy<subscription_based_ue_differentiation_info_s>();
      break;
    case types::add_rrm_prio_idx:
      c.destroy<fixed_bitstring<32, false, true> >();
      break;
    default:
      break;
  }
}
void dl_nas_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ho_restrict_list:
      c.init<ho_restrict_list_s>();
      break;
    case types::subscriber_profile_idfor_rfp:
      break;
    case types::srvcc_operation_possible:
      break;
    case types::ue_radio_cap:
      c.init<unbounded_octstring<true> >();
      break;
    case types::dlnaspdu_delivery_ack_request:
      break;
    case types::enhanced_coverage_restricted:
      break;
    case types::nrue_security_cap:
      c.init<nrue_security_cap_s>();
      break;
    case types::ce_mode_brestricted:
      break;
    case types::ue_cap_info_request:
      break;
    case types::end_ind:
      break;
    case types::pending_data_ind:
      break;
    case types::subscription_based_ue_differentiation_info:
      c.init<subscription_based_ue_differentiation_info_s>();
      break;
    case types::add_rrm_prio_idx:
      c.init<fixed_bitstring<32, false, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
}
dl_nas_transport_ies_o::value_c::value_c(const dl_nas_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ho_restrict_list:
      c.init(other.c.get<ho_restrict_list_s>());
      break;
    case types::subscriber_profile_idfor_rfp:
      c.init(other.c.get<uint16_t>());
      break;
    case types::srvcc_operation_possible:
      c.init(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::ue_radio_cap:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::dlnaspdu_delivery_ack_request:
      c.init(other.c.get<dlnaspdu_delivery_ack_request_e>());
      break;
    case types::enhanced_coverage_restricted:
      c.init(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::nrue_security_cap:
      c.init(other.c.get<nrue_security_cap_s>());
      break;
    case types::ce_mode_brestricted:
      c.init(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::ue_cap_info_request:
      c.init(other.c.get<ue_cap_info_request_e>());
      break;
    case types::end_ind:
      c.init(other.c.get<end_ind_e>());
      break;
    case types::pending_data_ind:
      c.init(other.c.get<pending_data_ind_e>());
      break;
    case types::subscription_based_ue_differentiation_info:
      c.init(other.c.get<subscription_based_ue_differentiation_info_s>());
      break;
    case types::add_rrm_prio_idx:
      c.init(other.c.get<fixed_bitstring<32, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
}
dl_nas_transport_ies_o::value_c&
dl_nas_transport_ies_o::value_c::operator=(const dl_nas_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ho_restrict_list:
      c.set(other.c.get<ho_restrict_list_s>());
      break;
    case types::subscriber_profile_idfor_rfp:
      c.set(other.c.get<uint16_t>());
      break;
    case types::srvcc_operation_possible:
      c.set(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::ue_radio_cap:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::dlnaspdu_delivery_ack_request:
      c.set(other.c.get<dlnaspdu_delivery_ack_request_e>());
      break;
    case types::enhanced_coverage_restricted:
      c.set(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::nrue_security_cap:
      c.set(other.c.get<nrue_security_cap_s>());
      break;
    case types::ce_mode_brestricted:
      c.set(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::ue_cap_info_request:
      c.set(other.c.get<ue_cap_info_request_e>());
      break;
    case types::end_ind:
      c.set(other.c.get<end_ind_e>());
      break;
    case types::pending_data_ind:
      c.set(other.c.get<pending_data_ind_e>());
      break;
    case types::subscription_based_ue_differentiation_info:
      c.set(other.c.get<subscription_based_ue_differentiation_info_s>());
      break;
    case types::add_rrm_prio_idx:
      c.set(other.c.get<fixed_bitstring<32, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }

  return *this;
}
uint64_t& dl_nas_transport_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& dl_nas_transport_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::nas_pdu()
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
ho_restrict_list_s& dl_nas_transport_ies_o::value_c::ho_restrict_list()
{
  assert_choice_type(types::ho_restrict_list, type_, "Value");
  return c.get<ho_restrict_list_s>();
}
uint16_t& dl_nas_transport_ies_o::value_c::subscriber_profile_idfor_rfp()
{
  assert_choice_type(types::subscriber_profile_idfor_rfp, type_, "Value");
  return c.get<uint16_t>();
}
srvcc_operation_possible_e& dl_nas_transport_ies_o::value_c::srvcc_operation_possible()
{
  assert_choice_type(types::srvcc_operation_possible, type_, "Value");
  return c.get<srvcc_operation_possible_e>();
}
unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::ue_radio_cap()
{
  assert_choice_type(types::ue_radio_cap, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
dlnaspdu_delivery_ack_request_e& dl_nas_transport_ies_o::value_c::dlnaspdu_delivery_ack_request()
{
  assert_choice_type(types::dlnaspdu_delivery_ack_request, type_, "Value");
  return c.get<dlnaspdu_delivery_ack_request_e>();
}
enhanced_coverage_restricted_e& dl_nas_transport_ies_o::value_c::enhanced_coverage_restricted()
{
  assert_choice_type(types::enhanced_coverage_restricted, type_, "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
nrue_security_cap_s& dl_nas_transport_ies_o::value_c::nrue_security_cap()
{
  assert_choice_type(types::nrue_security_cap, type_, "Value");
  return c.get<nrue_security_cap_s>();
}
ce_mode_brestricted_e& dl_nas_transport_ies_o::value_c::ce_mode_brestricted()
{
  assert_choice_type(types::ce_mode_brestricted, type_, "Value");
  return c.get<ce_mode_brestricted_e>();
}
ue_cap_info_request_e& dl_nas_transport_ies_o::value_c::ue_cap_info_request()
{
  assert_choice_type(types::ue_cap_info_request, type_, "Value");
  return c.get<ue_cap_info_request_e>();
}
end_ind_e& dl_nas_transport_ies_o::value_c::end_ind()
{
  assert_choice_type(types::end_ind, type_, "Value");
  return c.get<end_ind_e>();
}
pending_data_ind_e& dl_nas_transport_ies_o::value_c::pending_data_ind()
{
  assert_choice_type(types::pending_data_ind, type_, "Value");
  return c.get<pending_data_ind_e>();
}
subscription_based_ue_differentiation_info_s&
dl_nas_transport_ies_o::value_c::subscription_based_ue_differentiation_info()
{
  assert_choice_type(types::subscription_based_ue_differentiation_info, type_, "Value");
  return c.get<subscription_based_ue_differentiation_info_s>();
}
fixed_bitstring<32, false, true>& dl_nas_transport_ies_o::value_c::add_rrm_prio_idx()
{
  assert_choice_type(types::add_rrm_prio_idx, type_, "Value");
  return c.get<fixed_bitstring<32, false, true> >();
}
const uint64_t& dl_nas_transport_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& dl_nas_transport_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::nas_pdu() const
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const ho_restrict_list_s& dl_nas_transport_ies_o::value_c::ho_restrict_list() const
{
  assert_choice_type(types::ho_restrict_list, type_, "Value");
  return c.get<ho_restrict_list_s>();
}
const uint16_t& dl_nas_transport_ies_o::value_c::subscriber_profile_idfor_rfp() const
{
  assert_choice_type(types::subscriber_profile_idfor_rfp, type_, "Value");
  return c.get<uint16_t>();
}
const srvcc_operation_possible_e& dl_nas_transport_ies_o::value_c::srvcc_operation_possible() const
{
  assert_choice_type(types::srvcc_operation_possible, type_, "Value");
  return c.get<srvcc_operation_possible_e>();
}
const unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::ue_radio_cap() const
{
  assert_choice_type(types::ue_radio_cap, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const dlnaspdu_delivery_ack_request_e& dl_nas_transport_ies_o::value_c::dlnaspdu_delivery_ack_request() const
{
  assert_choice_type(types::dlnaspdu_delivery_ack_request, type_, "Value");
  return c.get<dlnaspdu_delivery_ack_request_e>();
}
const enhanced_coverage_restricted_e& dl_nas_transport_ies_o::value_c::enhanced_coverage_restricted() const
{
  assert_choice_type(types::enhanced_coverage_restricted, type_, "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
const nrue_security_cap_s& dl_nas_transport_ies_o::value_c::nrue_security_cap() const
{
  assert_choice_type(types::nrue_security_cap, type_, "Value");
  return c.get<nrue_security_cap_s>();
}
const ce_mode_brestricted_e& dl_nas_transport_ies_o::value_c::ce_mode_brestricted() const
{
  assert_choice_type(types::ce_mode_brestricted, type_, "Value");
  return c.get<ce_mode_brestricted_e>();
}
const ue_cap_info_request_e& dl_nas_transport_ies_o::value_c::ue_cap_info_request() const
{
  assert_choice_type(types::ue_cap_info_request, type_, "Value");
  return c.get<ue_cap_info_request_e>();
}
const end_ind_e& dl_nas_transport_ies_o::value_c::end_ind() const
{
  assert_choice_type(types::end_ind, type_, "Value");
  return c.get<end_ind_e>();
}
const pending_data_ind_e& dl_nas_transport_ies_o::value_c::pending_data_ind() const
{
  assert_choice_type(types::pending_data_ind, type_, "Value");
  return c.get<pending_data_ind_e>();
}
const subscription_based_ue_differentiation_info_s&
dl_nas_transport_ies_o::value_c::subscription_based_ue_differentiation_info() const
{
  assert_choice_type(types::subscription_based_ue_differentiation_info, type_, "Value");
  return c.get<subscription_based_ue_differentiation_info_s>();
}
const fixed_bitstring<32, false, true>& dl_nas_transport_ies_o::value_c::add_rrm_prio_idx() const
{
  assert_choice_type(types::add_rrm_prio_idx, type_, "Value");
  return c.get<fixed_bitstring<32, false, true> >();
}
void dl_nas_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ho_restrict_list:
      j.write_fieldname("HandoverRestrictionList");
      c.get<ho_restrict_list_s>().to_json(j);
      break;
    case types::subscriber_profile_idfor_rfp:
      j.write_int("INTEGER (1..256)", c.get<uint16_t>());
      break;
    case types::srvcc_operation_possible:
      j.write_str("SRVCCOperationPossible", "possible");
      break;
    case types::ue_radio_cap:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::dlnaspdu_delivery_ack_request:
      j.write_str("DLNASPDUDeliveryAckRequest", "requested");
      break;
    case types::enhanced_coverage_restricted:
      j.write_str("EnhancedCoverageRestricted", "restricted");
      break;
    case types::nrue_security_cap:
      j.write_fieldname("NRUESecurityCapabilities");
      c.get<nrue_security_cap_s>().to_json(j);
      break;
    case types::ce_mode_brestricted:
      j.write_str("CE-ModeBRestricted", c.get<ce_mode_brestricted_e>().to_string());
      break;
    case types::ue_cap_info_request:
      j.write_str("UECapabilityInfoRequest", "requested");
      break;
    case types::end_ind:
      j.write_str("EndIndication", c.get<end_ind_e>().to_string());
      break;
    case types::pending_data_ind:
      j.write_str("PendingDataIndication", "true");
      break;
    case types::subscription_based_ue_differentiation_info:
      j.write_fieldname("Subscription-Based-UE-DifferentiationInfo");
      c.get<subscription_based_ue_differentiation_info_s>().to_json(j);
      break;
    case types::add_rrm_prio_idx:
      j.write_str("BIT STRING", c.get<fixed_bitstring<32, false, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_nas_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().pack(bref));
      break;
    case types::subscriber_profile_idfor_rfp:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().pack(bref));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::dlnaspdu_delivery_ack_request:
      HANDLE_CODE(c.get<dlnaspdu_delivery_ack_request_e>().pack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().pack(bref));
      break;
    case types::nrue_security_cap:
      HANDLE_CODE(c.get<nrue_security_cap_s>().pack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().pack(bref));
      break;
    case types::ue_cap_info_request:
      HANDLE_CODE(c.get<ue_cap_info_request_e>().pack(bref));
      break;
    case types::end_ind:
      HANDLE_CODE(c.get<end_ind_e>().pack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().pack(bref));
      break;
    case types::subscription_based_ue_differentiation_info:
      HANDLE_CODE(c.get<subscription_based_ue_differentiation_info_s>().pack(bref));
      break;
    case types::add_rrm_prio_idx:
      HANDLE_CODE((c.get<fixed_bitstring<32, false, true> >().pack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_nas_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().unpack(bref));
      break;
    case types::subscriber_profile_idfor_rfp:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().unpack(bref));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::dlnaspdu_delivery_ack_request:
      HANDLE_CODE(c.get<dlnaspdu_delivery_ack_request_e>().unpack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().unpack(bref));
      break;
    case types::nrue_security_cap:
      HANDLE_CODE(c.get<nrue_security_cap_s>().unpack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().unpack(bref));
      break;
    case types::ue_cap_info_request:
      HANDLE_CODE(c.get<ue_cap_info_request_e>().unpack(bref));
      break;
    case types::end_ind:
      HANDLE_CODE(c.get<end_ind_e>().unpack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().unpack(bref));
      break;
    case types::subscription_based_ue_differentiation_info:
      HANDLE_CODE(c.get<subscription_based_ue_differentiation_info_s>().unpack(bref));
      break;
    case types::add_rrm_prio_idx:
      HANDLE_CODE((c.get<fixed_bitstring<32, false, true> >().unpack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_nas_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "OCTET STRING",
                                  "HandoverRestrictionList",
                                  "INTEGER (1..256)",
                                  "SRVCCOperationPossible",
                                  "OCTET STRING",
                                  "DLNASPDUDeliveryAckRequest",
                                  "EnhancedCoverageRestricted",
                                  "NRUESecurityCapabilities",
                                  "CE-ModeBRestricted",
                                  "UECapabilityInfoRequest",
                                  "EndIndication",
                                  "PendingDataIndication",
                                  "Subscription-Based-UE-DifferentiationInfo",
                                  "BIT STRING"};
  return convert_enum_idx(options, 16, value, "dl_nas_transport_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_nas_transport_ies_o>;

dl_nas_transport_ies_container::dl_nas_transport_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  nas_pdu(26, crit_e::reject),
  ho_restrict_list(41, crit_e::ignore),
  subscriber_profile_idfor_rfp(106, crit_e::ignore),
  srvcc_operation_possible(124, crit_e::ignore),
  ue_radio_cap(74, crit_e::ignore),
  dlnaspdu_delivery_ack_request(249, crit_e::ignore),
  enhanced_coverage_restricted(251, crit_e::ignore),
  nrue_security_cap(269, crit_e::ignore),
  ce_mode_brestricted(271, crit_e::ignore),
  ue_cap_info_request(275, crit_e::ignore),
  end_ind(280, crit_e::ignore),
  pending_data_ind(283, crit_e::ignore),
  subscription_based_ue_differentiation_info(278, crit_e::ignore),
  add_rrm_prio_idx(299, crit_e::ignore)
{}
SRSASN_CODE dl_nas_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += ho_restrict_list_present ? 1 : 0;
  nof_ies += subscriber_profile_idfor_rfp_present ? 1 : 0;
  nof_ies += srvcc_operation_possible_present ? 1 : 0;
  nof_ies += ue_radio_cap_present ? 1 : 0;
  nof_ies += dlnaspdu_delivery_ack_request_present ? 1 : 0;
  nof_ies += enhanced_coverage_restricted_present ? 1 : 0;
  nof_ies += nrue_security_cap_present ? 1 : 0;
  nof_ies += ce_mode_brestricted_present ? 1 : 0;
  nof_ies += ue_cap_info_request_present ? 1 : 0;
  nof_ies += end_ind_present ? 1 : 0;
  nof_ies += pending_data_ind_present ? 1 : 0;
  nof_ies += subscription_based_ue_differentiation_info_present ? 1 : 0;
  nof_ies += add_rrm_prio_idx_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(nas_pdu.pack(bref));
  if (ho_restrict_list_present) {
    HANDLE_CODE(ho_restrict_list.pack(bref));
  }
  if (subscriber_profile_idfor_rfp_present) {
    HANDLE_CODE(subscriber_profile_idfor_rfp.pack(bref));
  }
  if (srvcc_operation_possible_present) {
    HANDLE_CODE(srvcc_operation_possible.pack(bref));
  }
  if (ue_radio_cap_present) {
    HANDLE_CODE(ue_radio_cap.pack(bref));
  }
  if (dlnaspdu_delivery_ack_request_present) {
    HANDLE_CODE(dlnaspdu_delivery_ack_request.pack(bref));
  }
  if (enhanced_coverage_restricted_present) {
    HANDLE_CODE(enhanced_coverage_restricted.pack(bref));
  }
  if (nrue_security_cap_present) {
    HANDLE_CODE(nrue_security_cap.pack(bref));
  }
  if (ce_mode_brestricted_present) {
    HANDLE_CODE(ce_mode_brestricted.pack(bref));
  }
  if (ue_cap_info_request_present) {
    HANDLE_CODE(ue_cap_info_request.pack(bref));
  }
  if (end_ind_present) {
    HANDLE_CODE(end_ind.pack(bref));
  }
  if (pending_data_ind_present) {
    HANDLE_CODE(pending_data_ind.pack(bref));
  }
  if (subscription_based_ue_differentiation_info_present) {
    HANDLE_CODE(subscription_based_ue_differentiation_info.pack(bref));
  }
  if (add_rrm_prio_idx_present) {
    HANDLE_CODE(add_rrm_prio_idx.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_nas_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 26: {
        nof_mandatory_ies--;
        nas_pdu.id = id;
        HANDLE_CODE(nas_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nas_pdu.value.unpack(bref));
        break;
      }
      case 41: {
        ho_restrict_list_present = true;
        ho_restrict_list.id      = id;
        HANDLE_CODE(ho_restrict_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ho_restrict_list.value.unpack(bref));
        break;
      }
      case 106: {
        subscriber_profile_idfor_rfp_present = true;
        subscriber_profile_idfor_rfp.id      = id;
        HANDLE_CODE(subscriber_profile_idfor_rfp.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(subscriber_profile_idfor_rfp.value.unpack(bref));
        break;
      }
      case 124: {
        srvcc_operation_possible_present = true;
        srvcc_operation_possible.id      = id;
        HANDLE_CODE(srvcc_operation_possible.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(srvcc_operation_possible.value.unpack(bref));
        break;
      }
      case 74: {
        ue_radio_cap_present = true;
        ue_radio_cap.id      = id;
        HANDLE_CODE(ue_radio_cap.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_radio_cap.value.unpack(bref));
        break;
      }
      case 249: {
        dlnaspdu_delivery_ack_request_present = true;
        dlnaspdu_delivery_ack_request.id      = id;
        HANDLE_CODE(dlnaspdu_delivery_ack_request.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(dlnaspdu_delivery_ack_request.value.unpack(bref));
        break;
      }
      case 251: {
        enhanced_coverage_restricted_present = true;
        enhanced_coverage_restricted.id      = id;
        HANDLE_CODE(enhanced_coverage_restricted.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enhanced_coverage_restricted.value.unpack(bref));
        break;
      }
      case 269: {
        nrue_security_cap_present = true;
        nrue_security_cap.id      = id;
        HANDLE_CODE(nrue_security_cap.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nrue_security_cap.value.unpack(bref));
        break;
      }
      case 271: {
        ce_mode_brestricted_present = true;
        ce_mode_brestricted.id      = id;
        HANDLE_CODE(ce_mode_brestricted.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ce_mode_brestricted.value.unpack(bref));
        break;
      }
      case 275: {
        ue_cap_info_request_present = true;
        ue_cap_info_request.id      = id;
        HANDLE_CODE(ue_cap_info_request.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_cap_info_request.value.unpack(bref));
        break;
      }
      case 280: {
        end_ind_present = true;
        end_ind.id      = id;
        HANDLE_CODE(end_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(end_ind.value.unpack(bref));
        break;
      }
      case 283: {
        pending_data_ind_present = true;
        pending_data_ind.id      = id;
        HANDLE_CODE(pending_data_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pending_data_ind.value.unpack(bref));
        break;
      }
      case 278: {
        subscription_based_ue_differentiation_info_present = true;
        subscription_based_ue_differentiation_info.id      = id;
        HANDLE_CODE(subscription_based_ue_differentiation_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(subscription_based_ue_differentiation_info.value.unpack(bref));
        break;
      }
      case 299: {
        add_rrm_prio_idx_present = true;
        add_rrm_prio_idx.id      = id;
        HANDLE_CODE(add_rrm_prio_idx.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(add_rrm_prio_idx.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_nas_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  nas_pdu.to_json(j);
  if (ho_restrict_list_present) {
    j.write_fieldname("");
    ho_restrict_list.to_json(j);
  }
  if (subscriber_profile_idfor_rfp_present) {
    j.write_fieldname("");
    subscriber_profile_idfor_rfp.to_json(j);
  }
  if (srvcc_operation_possible_present) {
    j.write_fieldname("");
    srvcc_operation_possible.to_json(j);
  }
  if (ue_radio_cap_present) {
    j.write_fieldname("");
    ue_radio_cap.to_json(j);
  }
  if (dlnaspdu_delivery_ack_request_present) {
    j.write_fieldname("");
    dlnaspdu_delivery_ack_request.to_json(j);
  }
  if (enhanced_coverage_restricted_present) {
    j.write_fieldname("");
    enhanced_coverage_restricted.to_json(j);
  }
  if (nrue_security_cap_present) {
    j.write_fieldname("");
    nrue_security_cap.to_json(j);
  }
  if (ce_mode_brestricted_present) {
    j.write_fieldname("");
    ce_mode_brestricted.to_json(j);
  }
  if (ue_cap_info_request_present) {
    j.write_fieldname("");
    ue_cap_info_request.to_json(j);
  }
  if (end_ind_present) {
    j.write_fieldname("");
    end_ind.to_json(j);
  }
  if (pending_data_ind_present) {
    j.write_fieldname("");
    pending_data_ind.to_json(j);
  }
  if (subscription_based_ue_differentiation_info_present) {
    j.write_fieldname("");
    subscription_based_ue_differentiation_info.to_json(j);
  }
  if (add_rrm_prio_idx_present) {
    j.write_fieldname("");
    add_rrm_prio_idx.to_json(j);
  }
  j.end_obj();
}

// DownlinkNonUEAssociatedLPPaTransport-IEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t dl_non_ueassociated_lp_pa_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {148, 147};
  return map_enum_number(options, 2, idx, "id");
}
bool dl_non_ueassociated_lp_pa_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {148, 147};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_non_ueassociated_lp_pa_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 148:
      return crit_e::reject;
    case 147:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_non_ueassociated_lp_pa_transport_ies_o::value_c
dl_non_ueassociated_lp_pa_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 148:
      ret.set(value_c::types::routing_id);
      break;
    case 147:
      ret.set(value_c::types::lp_pa_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_non_ueassociated_lp_pa_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 148:
      return presence_e::mandatory;
    case 147:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_non_ueassociated_lp_pa_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::lp_pa_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_non_ueassociated_lp_pa_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::routing_id:
      break;
    case types::lp_pa_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
  }
}
dl_non_ueassociated_lp_pa_transport_ies_o::value_c::value_c(
    const dl_non_ueassociated_lp_pa_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::routing_id:
      c.init(other.c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
  }
}
dl_non_ueassociated_lp_pa_transport_ies_o::value_c& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::operator=(
    const dl_non_ueassociated_lp_pa_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::routing_id:
      c.set(other.c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
  }

  return *this;
}
uint16_t& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::routing_id()
{
  assert_choice_type(types::routing_id, type_, "Value");
  return c.get<uint16_t>();
}
unbounded_octstring<true>& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::lp_pa_pdu()
{
  assert_choice_type(types::lp_pa_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint16_t& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::routing_id() const
{
  assert_choice_type(types::routing_id, type_, "Value");
  return c.get<uint16_t>();
}
const unbounded_octstring<true>& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::lp_pa_pdu() const
{
  assert_choice_type(types::lp_pa_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_non_ueassociated_lp_pa_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::routing_id:
      j.write_int("INTEGER (0..255)", c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::routing_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::lp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::routing_id:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::lp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_non_ueassociated_lp_pa_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..255)", "OCTET STRING"};
  return convert_enum_idx(options, 2, value, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c::types");
}
uint8_t dl_non_ueassociated_lp_pa_transport_ies_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {0};
  return map_enum_number(options, 1, value, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_non_ueassociated_lp_pa_transport_ies_o>;

dl_non_ueassociated_lp_pa_transport_ies_container::dl_non_ueassociated_lp_pa_transport_ies_container() :
  routing_id(148, crit_e::reject), lp_pa_pdu(147, crit_e::reject)
{}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(routing_id.pack(bref));
  HANDLE_CODE(lp_pa_pdu.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 148: {
        nof_mandatory_ies--;
        routing_id.id = id;
        HANDLE_CODE(routing_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(routing_id.value.unpack(bref));
        break;
      }
      case 147: {
        nof_mandatory_ies--;
        lp_pa_pdu.id = id;
        HANDLE_CODE(lp_pa_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(lp_pa_pdu.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_non_ueassociated_lp_pa_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  routing_id.to_json(j);
  j.write_fieldname("");
  lp_pa_pdu.to_json(j);
  j.end_obj();
}

// E-RABDataForwardingItem ::= SEQUENCE
SRSASN_CODE erab_data_forwarding_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dl_transport_layer_address_present, 1));
  HANDLE_CODE(bref.pack(dl_g_tp_teid_present, 1));
  HANDLE_CODE(bref.pack(ul_transport_layer_address_present, 1));
  HANDLE_CODE(bref.pack(ul_gtp_teid_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dl_transport_layer_address_present) {
    HANDLE_CODE(dl_transport_layer_address.pack(bref));
  }
  if (dl_g_tp_teid_present) {
    HANDLE_CODE(dl_g_tp_teid.pack(bref));
  }
  if (ul_transport_layer_address_present) {
    HANDLE_CODE(ul_transport_layer_address.pack(bref));
  }
  if (ul_gtp_teid_present) {
    HANDLE_CODE(ul_gtp_teid.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_data_forwarding_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dl_transport_layer_address_present, 1));
  HANDLE_CODE(bref.unpack(dl_g_tp_teid_present, 1));
  HANDLE_CODE(bref.unpack(ul_transport_layer_address_present, 1));
  HANDLE_CODE(bref.unpack(ul_gtp_teid_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dl_transport_layer_address_present) {
    HANDLE_CODE(dl_transport_layer_address.unpack(bref));
  }
  if (dl_g_tp_teid_present) {
    HANDLE_CODE(dl_g_tp_teid.unpack(bref));
  }
  if (ul_transport_layer_address_present) {
    HANDLE_CODE(ul_transport_layer_address.unpack(bref));
  }
  if (ul_gtp_teid_present) {
    HANDLE_CODE(ul_gtp_teid.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_data_forwarding_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (dl_transport_layer_address_present) {
    j.write_str("dL-transportLayerAddress", dl_transport_layer_address.to_string());
  }
  if (dl_g_tp_teid_present) {
    j.write_str("dL-gTP-TEID", dl_g_tp_teid.to_string());
  }
  if (ul_transport_layer_address_present) {
    j.write_str("uL-TransportLayerAddress", ul_transport_layer_address.to_string());
  }
  if (ul_gtp_teid_present) {
    j.write_str("uL-GTP-TEID", ul_gtp_teid.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABDataForwardingItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_data_forwarding_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {14};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_data_forwarding_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 14 == id;
}
crit_e erab_data_forwarding_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 14) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_data_forwarding_item_ies_o::value_c erab_data_forwarding_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 14) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_data_forwarding_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 14) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_data_forwarding_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABDataForwardingItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_data_forwarding_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_data_forwarding_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_data_forwarding_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABDataForwardingItem"};
  return convert_enum_idx(options, 1, value, "erab_data_forwarding_item_ies_o::value_c::types");
}

// Cdma2000HOStatus ::= ENUMERATED
const char* cdma2000_ho_status_opts::to_string() const
{
  static const char* options[] = {"hOSuccess", "hOFailure"};
  return convert_enum_idx(options, 2, value, "cdma2000_ho_status_e");
}

// Cdma2000RATType ::= ENUMERATED
const char* cdma2000_rat_type_opts::to_string() const
{
  static const char* options[] = {"hRPD", "onexRTT"};
  return convert_enum_idx(options, 2, value, "cdma2000_rat_type_e");
}
uint8_t cdma2000_rat_type_opts::to_number() const
{
  if (value == onex_rtt) {
    return 1;
  }
  invalid_enum_number(value, "cdma2000_rat_type_e");
  return 0;
}

// DownlinkS1cdma2000tunnellingIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t dl_s1cdma2000tunnelling_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 12, 83, 71, 70};
  return map_enum_number(options, 6, idx, "id");
}
bool dl_s1cdma2000tunnelling_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 12, 83, 71, 70};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_s1cdma2000tunnelling_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 12:
      return crit_e::ignore;
    case 83:
      return crit_e::ignore;
    case 71:
      return crit_e::reject;
    case 70:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_s1cdma2000tunnelling_ies_o::value_c dl_s1cdma2000tunnelling_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 12:
      ret.set(value_c::types::erab_subjectto_data_forwarding_list);
      break;
    case 83:
      ret.set(value_c::types::cdma2000_ho_status);
      break;
    case 71:
      ret.set(value_c::types::cdma2000_rat_type);
      break;
    case 70:
      ret.set(value_c::types::cdma2000_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_s1cdma2000tunnelling_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 12:
      return presence_e::optional;
    case 83:
      return presence_e::optional;
    case 71:
      return presence_e::mandatory;
    case 70:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_s1cdma2000tunnelling_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_subjectto_data_forwarding_list:
      c.destroy<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
      break;
    case types::cdma2000_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_s1cdma2000tunnelling_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.init<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
      break;
    case types::cdma2000_ho_status:
      break;
    case types::cdma2000_rat_type:
      break;
    case types::cdma2000_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
  }
}
dl_s1cdma2000tunnelling_ies_o::value_c::value_c(const dl_s1cdma2000tunnelling_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.init(other.c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >());
      break;
    case types::cdma2000_ho_status:
      c.init(other.c.get<cdma2000_ho_status_e>());
      break;
    case types::cdma2000_rat_type:
      c.init(other.c.get<cdma2000_rat_type_e>());
      break;
    case types::cdma2000_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
  }
}
dl_s1cdma2000tunnelling_ies_o::value_c&
dl_s1cdma2000tunnelling_ies_o::value_c::operator=(const dl_s1cdma2000tunnelling_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.set(other.c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >());
      break;
    case types::cdma2000_ho_status:
      c.set(other.c.get<cdma2000_ho_status_e>());
      break;
    case types::cdma2000_rat_type:
      c.set(other.c.get<cdma2000_rat_type_e>());
      break;
    case types::cdma2000_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
  }

  return *this;
}
uint64_t& dl_s1cdma2000tunnelling_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& dl_s1cdma2000tunnelling_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
erab_ie_container_list_l<erab_data_forwarding_item_ies_o>&
dl_s1cdma2000tunnelling_ies_o::value_c::erab_subjectto_data_forwarding_list()
{
  assert_choice_type(types::erab_subjectto_data_forwarding_list, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
}
cdma2000_ho_status_e& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_ho_status()
{
  assert_choice_type(types::cdma2000_ho_status, type_, "Value");
  return c.get<cdma2000_ho_status_e>();
}
cdma2000_rat_type_e& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_rat_type()
{
  assert_choice_type(types::cdma2000_rat_type, type_, "Value");
  return c.get<cdma2000_rat_type_e>();
}
unbounded_octstring<true>& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_pdu()
{
  assert_choice_type(types::cdma2000_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& dl_s1cdma2000tunnelling_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& dl_s1cdma2000tunnelling_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const erab_ie_container_list_l<erab_data_forwarding_item_ies_o>&
dl_s1cdma2000tunnelling_ies_o::value_c::erab_subjectto_data_forwarding_list() const
{
  assert_choice_type(types::erab_subjectto_data_forwarding_list, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
}
const cdma2000_ho_status_e& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_ho_status() const
{
  assert_choice_type(types::cdma2000_ho_status, type_, "Value");
  return c.get<cdma2000_ho_status_e>();
}
const cdma2000_rat_type_e& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_rat_type() const
{
  assert_choice_type(types::cdma2000_rat_type, type_, "Value");
  return c.get<cdma2000_rat_type_e>();
}
const unbounded_octstring<true>& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_pdu() const
{
  assert_choice_type(types::cdma2000_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_s1cdma2000tunnelling_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_subjectto_data_forwarding_list:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}");
      break;
    case types::cdma2000_ho_status:
      j.write_str("Cdma2000HOStatus", c.get<cdma2000_ho_status_e>().to_string());
      break;
    case types::cdma2000_rat_type:
      j.write_str("Cdma2000RATType", c.get<cdma2000_rat_type_e>().to_string());
      break;
    case types::cdma2000_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_s1cdma2000tunnelling_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_subjectto_data_forwarding_list:
      HANDLE_CODE(
          pack_dyn_seq_of(bref, c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >(), 1, 256, true));
      break;
    case types::cdma2000_ho_status:
      HANDLE_CODE(c.get<cdma2000_ho_status_e>().pack(bref));
      break;
    case types::cdma2000_rat_type:
      HANDLE_CODE(c.get<cdma2000_rat_type_e>().pack(bref));
      break;
    case types::cdma2000_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_s1cdma2000tunnelling_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_subjectto_data_forwarding_list:
      HANDLE_CODE(
          unpack_dyn_seq_of(c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >(), bref, 1, 256, true));
      break;
    case types::cdma2000_ho_status:
      HANDLE_CODE(c.get<cdma2000_ho_status_e>().unpack(bref));
      break;
    case types::cdma2000_rat_type:
      HANDLE_CODE(c.get<cdma2000_rat_type_e>().unpack(bref));
      break;
    case types::cdma2000_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_s1cdma2000tunnelling_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}",
                                  "Cdma2000HOStatus",
                                  "Cdma2000RATType",
                                  "OCTET STRING"};
  return convert_enum_idx(options, 6, value, "dl_s1cdma2000tunnelling_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_s1cdma2000tunnelling_ies_o>;

dl_s1cdma2000tunnelling_ies_container::dl_s1cdma2000tunnelling_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  erab_subjectto_data_forwarding_list(12, crit_e::ignore),
  cdma2000_ho_status(83, crit_e::ignore),
  cdma2000_rat_type(71, crit_e::reject),
  cdma2000_pdu(70, crit_e::reject)
{}
SRSASN_CODE dl_s1cdma2000tunnelling_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += erab_subjectto_data_forwarding_list_present ? 1 : 0;
  nof_ies += cdma2000_ho_status_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_subjectto_data_forwarding_list_present) {
    HANDLE_CODE(erab_subjectto_data_forwarding_list.pack(bref));
  }
  if (cdma2000_ho_status_present) {
    HANDLE_CODE(cdma2000_ho_status.pack(bref));
  }
  HANDLE_CODE(cdma2000_rat_type.pack(bref));
  HANDLE_CODE(cdma2000_pdu.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_s1cdma2000tunnelling_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 12: {
        erab_subjectto_data_forwarding_list_present = true;
        erab_subjectto_data_forwarding_list.id      = id;
        HANDLE_CODE(erab_subjectto_data_forwarding_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_subjectto_data_forwarding_list.value.unpack(bref));
        break;
      }
      case 83: {
        cdma2000_ho_status_present = true;
        cdma2000_ho_status.id      = id;
        HANDLE_CODE(cdma2000_ho_status.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cdma2000_ho_status.value.unpack(bref));
        break;
      }
      case 71: {
        nof_mandatory_ies--;
        cdma2000_rat_type.id = id;
        HANDLE_CODE(cdma2000_rat_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cdma2000_rat_type.value.unpack(bref));
        break;
      }
      case 70: {
        nof_mandatory_ies--;
        cdma2000_pdu.id = id;
        HANDLE_CODE(cdma2000_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cdma2000_pdu.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_s1cdma2000tunnelling_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_subjectto_data_forwarding_list_present) {
    j.write_fieldname("");
    erab_subjectto_data_forwarding_list.to_json(j);
  }
  if (cdma2000_ho_status_present) {
    j.write_fieldname("");
    cdma2000_ho_status.to_json(j);
  }
  j.write_fieldname("");
  cdma2000_rat_type.to_json(j);
  j.write_fieldname("");
  cdma2000_pdu.to_json(j);
  j.end_obj();
}

// DownlinkUEAssociatedLPPaTransport-IEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t dl_ueassociated_lp_pa_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 148, 147};
  return map_enum_number(options, 4, idx, "id");
}
bool dl_ueassociated_lp_pa_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 148, 147};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_ueassociated_lp_pa_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 148:
      return crit_e::reject;
    case 147:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_ueassociated_lp_pa_transport_ies_o::value_c dl_ueassociated_lp_pa_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 148:
      ret.set(value_c::types::routing_id);
      break;
    case 147:
      ret.set(value_c::types::lp_pa_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_ueassociated_lp_pa_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 148:
      return presence_e::mandatory;
    case 147:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_ueassociated_lp_pa_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::lp_pa_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_ueassociated_lp_pa_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::routing_id:
      break;
    case types::lp_pa_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
  }
}
dl_ueassociated_lp_pa_transport_ies_o::value_c::value_c(const dl_ueassociated_lp_pa_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::routing_id:
      c.init(other.c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
  }
}
dl_ueassociated_lp_pa_transport_ies_o::value_c&
dl_ueassociated_lp_pa_transport_ies_o::value_c::operator=(const dl_ueassociated_lp_pa_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::routing_id:
      c.set(other.c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
  }

  return *this;
}
uint64_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
uint16_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::routing_id()
{
  assert_choice_type(types::routing_id, type_, "Value");
  return c.get<uint16_t>();
}
unbounded_octstring<true>& dl_ueassociated_lp_pa_transport_ies_o::value_c::lp_pa_pdu()
{
  assert_choice_type(types::lp_pa_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const uint16_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::routing_id() const
{
  assert_choice_type(types::routing_id, type_, "Value");
  return c.get<uint16_t>();
}
const unbounded_octstring<true>& dl_ueassociated_lp_pa_transport_ies_o::value_c::lp_pa_pdu() const
{
  assert_choice_type(types::lp_pa_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_ueassociated_lp_pa_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::routing_id:
      j.write_int("INTEGER (0..255)", c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_ueassociated_lp_pa_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::routing_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::lp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ueassociated_lp_pa_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::routing_id:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::lp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_ueassociated_lp_pa_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "INTEGER (0..255)", "OCTET STRING"};
  return convert_enum_idx(options, 4, value, "dl_ueassociated_lp_pa_transport_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_ueassociated_lp_pa_transport_ies_o>;

dl_ueassociated_lp_pa_transport_ies_container::dl_ueassociated_lp_pa_transport_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  routing_id(148, crit_e::reject),
  lp_pa_pdu(147, crit_e::reject)
{}
SRSASN_CODE dl_ueassociated_lp_pa_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(routing_id.pack(bref));
  HANDLE_CODE(lp_pa_pdu.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ueassociated_lp_pa_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 148: {
        nof_mandatory_ies--;
        routing_id.id = id;
        HANDLE_CODE(routing_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(routing_id.value.unpack(bref));
        break;
      }
      case 147: {
        nof_mandatory_ies--;
        lp_pa_pdu.id = id;
        HANDLE_CODE(lp_pa_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(lp_pa_pdu.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_ueassociated_lp_pa_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  routing_id.to_json(j);
  j.write_fieldname("");
  lp_pa_pdu.to_json(j);
  j.end_obj();
}

// E-RABAdmittedItem ::= SEQUENCE
SRSASN_CODE erab_admitted_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dl_transport_layer_address_present, 1));
  HANDLE_CODE(bref.pack(dl_g_tp_teid_present, 1));
  HANDLE_CODE(bref.pack(ul_transport_layer_address_present, 1));
  HANDLE_CODE(bref.pack(ul_gtp_teid_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (dl_transport_layer_address_present) {
    HANDLE_CODE(dl_transport_layer_address.pack(bref));
  }
  if (dl_g_tp_teid_present) {
    HANDLE_CODE(dl_g_tp_teid.pack(bref));
  }
  if (ul_transport_layer_address_present) {
    HANDLE_CODE(ul_transport_layer_address.pack(bref));
  }
  if (ul_gtp_teid_present) {
    HANDLE_CODE(ul_gtp_teid.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_admitted_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dl_transport_layer_address_present, 1));
  HANDLE_CODE(bref.unpack(dl_g_tp_teid_present, 1));
  HANDLE_CODE(bref.unpack(ul_transport_layer_address_present, 1));
  HANDLE_CODE(bref.unpack(ul_gtp_teid_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (dl_transport_layer_address_present) {
    HANDLE_CODE(dl_transport_layer_address.unpack(bref));
  }
  if (dl_g_tp_teid_present) {
    HANDLE_CODE(dl_g_tp_teid.unpack(bref));
  }
  if (ul_transport_layer_address_present) {
    HANDLE_CODE(ul_transport_layer_address.unpack(bref));
  }
  if (ul_gtp_teid_present) {
    HANDLE_CODE(ul_gtp_teid.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_admitted_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (dl_transport_layer_address_present) {
    j.write_str("dL-transportLayerAddress", dl_transport_layer_address.to_string());
  }
  if (dl_g_tp_teid_present) {
    j.write_str("dL-gTP-TEID", dl_g_tp_teid.to_string());
  }
  if (ul_transport_layer_address_present) {
    j.write_str("uL-TransportLayerAddress", ul_transport_layer_address.to_string());
  }
  if (ul_gtp_teid_present) {
    j.write_str("uL-GTP-TEID", ul_gtp_teid.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABAdmittedItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_admitted_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {20};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_admitted_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 20 == id;
}
crit_e erab_admitted_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 20) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_admitted_item_ies_o::value_c erab_admitted_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 20) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_admitted_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 20) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_admitted_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABAdmittedItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_admitted_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_admitted_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_admitted_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABAdmittedItem"};
  return convert_enum_idx(options, 1, value, "erab_admitted_item_ies_o::value_c::types");
}

// E-RABFailedToResumeItemResumeReq ::= SEQUENCE
SRSASN_CODE erab_failed_to_resume_item_resume_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_resume_item_resume_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_failed_to_resume_item_resume_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABFailedToResumeItemResumeReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_failed_to_resume_item_resume_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {236};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_failed_to_resume_item_resume_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 236 == id;
}
crit_e erab_failed_to_resume_item_resume_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 236) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_failed_to_resume_item_resume_req_ies_o::value_c
erab_failed_to_resume_item_resume_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 236) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_failed_to_resume_item_resume_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 236) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_failed_to_resume_item_resume_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABFailedToResumeItemResumeReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_failed_to_resume_item_resume_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_resume_item_resume_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_failed_to_resume_item_resume_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABFailedToResumeItemResumeReq"};
  return convert_enum_idx(options, 1, value, "erab_failed_to_resume_item_resume_req_ies_o::value_c::types");
}

// E-RABFailedToResumeItemResumeRes ::= SEQUENCE
SRSASN_CODE erab_failed_to_resume_item_resume_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_resume_item_resume_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_failed_to_resume_item_resume_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABFailedToResumeItemResumeResIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_failed_to_resume_item_resume_res_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {238};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_failed_to_resume_item_resume_res_ies_o::is_id_valid(const uint32_t& id)
{
  return 238 == id;
}
crit_e erab_failed_to_resume_item_resume_res_ies_o::get_crit(const uint32_t& id)
{
  if (id == 238) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_failed_to_resume_item_resume_res_ies_o::value_c
erab_failed_to_resume_item_resume_res_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 238) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_failed_to_resume_item_resume_res_ies_o::get_presence(const uint32_t& id)
{
  if (id == 238) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_failed_to_resume_item_resume_res_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABFailedToResumeItemResumeRes");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_failed_to_resume_item_resume_res_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_resume_item_resume_res_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_failed_to_resume_item_resume_res_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABFailedToResumeItemResumeRes"};
  return convert_enum_idx(options, 1, value, "erab_failed_to_resume_item_resume_res_ies_o::value_c::types");
}

// E-RABFailedToSetupItemHOReqAck ::= SEQUENCE
SRSASN_CODE erab_failed_to_setup_item_ho_req_ack_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_setup_item_ho_req_ack_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_failed_to_setup_item_ho_req_ack_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABFailedtoSetupItemHOReqAckIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_failedto_setup_item_ho_req_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {21};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_failedto_setup_item_ho_req_ack_ies_o::is_id_valid(const uint32_t& id)
{
  return 21 == id;
}
crit_e erab_failedto_setup_item_ho_req_ack_ies_o::get_crit(const uint32_t& id)
{
  if (id == 21) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_failedto_setup_item_ho_req_ack_ies_o::value_c
erab_failedto_setup_item_ho_req_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 21) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_failedto_setup_item_ho_req_ack_ies_o::get_presence(const uint32_t& id)
{
  if (id == 21) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_failedto_setup_item_ho_req_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABFailedToSetupItemHOReqAck");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_failedto_setup_item_ho_req_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failedto_setup_item_ho_req_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_failedto_setup_item_ho_req_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABFailedToSetupItemHOReqAck"};
  return convert_enum_idx(options, 1, value, "erab_failedto_setup_item_ho_req_ack_ies_o::value_c::types");
}

// DL-Forwarding ::= ENUMERATED
const char* dl_forwarding_opts::to_string() const
{
  static const char* options[] = {"dL-Forwarding-proposed"};
  return convert_enum_idx(options, 1, value, "dl_forwarding_e");
}

// E-RABInformationListItem ::= SEQUENCE
SRSASN_CODE erab_info_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dl_forwarding_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dl_forwarding_present) {
    HANDLE_CODE(dl_forwarding.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_info_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dl_forwarding_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dl_forwarding_present) {
    HANDLE_CODE(dl_forwarding.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_info_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (dl_forwarding_present) {
    j.write_str("dL-Forwarding", "dL-Forwarding-proposed");
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABInformationListIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_info_list_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {78};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_info_list_ies_o::is_id_valid(const uint32_t& id)
{
  return 78 == id;
}
crit_e erab_info_list_ies_o::get_crit(const uint32_t& id)
{
  if (id == 78) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_info_list_ies_o::value_c erab_info_list_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 78) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_info_list_ies_o::get_presence(const uint32_t& id)
{
  if (id == 78) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_info_list_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABInformationListItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_info_list_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_info_list_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_info_list_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABInformationListItem"};
  return convert_enum_idx(options, 1, value, "erab_info_list_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_info_list_ies_o>;

// E-RABItem ::= SEQUENCE
SRSASN_CODE erab_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {35};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 35 == id;
}
crit_e erab_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 35) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_item_ies_o::value_c erab_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 35) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 35) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABItem"};
  return convert_enum_idx(options, 1, value, "erab_item_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_item_ies_o>;

// E-RABModifyItemBearerModConf ::= SEQUENCE
SRSASN_CODE erab_modify_item_bearer_mod_conf_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_item_bearer_mod_conf_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_modify_item_bearer_mod_conf_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyItemBearerModConfIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_modify_item_bearer_mod_conf_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {204};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_modify_item_bearer_mod_conf_ies_o::is_id_valid(const uint32_t& id)
{
  return 204 == id;
}
crit_e erab_modify_item_bearer_mod_conf_ies_o::get_crit(const uint32_t& id)
{
  if (id == 204) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_modify_item_bearer_mod_conf_ies_o::value_c erab_modify_item_bearer_mod_conf_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 204) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_modify_item_bearer_mod_conf_ies_o::get_presence(const uint32_t& id)
{
  if (id == 204) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_modify_item_bearer_mod_conf_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABModifyItemBearerModConf");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_modify_item_bearer_mod_conf_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_item_bearer_mod_conf_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_modify_item_bearer_mod_conf_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABModifyItemBearerModConf"};
  return convert_enum_idx(options, 1, value, "erab_modify_item_bearer_mod_conf_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_modify_item_bearer_mod_conf_ies_o>;

// E-RABModificationConfirmIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_mod_confirm_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 203, 205, 210, 58, 146};
  return map_enum_number(options, 7, idx, "id");
}
bool erab_mod_confirm_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 203, 205, 210, 58, 146};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_mod_confirm_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 203:
      return crit_e::ignore;
    case 205:
      return crit_e::ignore;
    case 210:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 146:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_mod_confirm_ies_o::value_c erab_mod_confirm_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 203:
      ret.set(value_c::types::erab_modify_list_bearer_mod_conf);
      break;
    case 205:
      ret.set(value_c::types::erab_failed_to_modify_list_bearer_mod_conf);
      break;
    case 210:
      ret.set(value_c::types::erab_to_be_released_list_bearer_mod_conf);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 146:
      ret.set(value_c::types::csg_membership_status);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_mod_confirm_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 203:
      return presence_e::optional;
    case 205:
      return presence_e::optional;
    case 210:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 146:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_mod_confirm_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_modify_list_bearer_mod_conf:
      c.destroy<erab_modify_list_bearer_mod_conf_l>();
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      c.destroy<erab_list_l>();
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void erab_mod_confirm_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_modify_list_bearer_mod_conf:
      c.init<erab_modify_list_bearer_mod_conf_l>();
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      c.init<erab_list_l>();
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::csg_membership_status:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
  }
}
erab_mod_confirm_ies_o::value_c::value_c(const erab_mod_confirm_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_conf:
      c.init(other.c.get<erab_modify_list_bearer_mod_conf_l>());
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::csg_membership_status:
      c.init(other.c.get<csg_membership_status_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
  }
}
erab_mod_confirm_ies_o::value_c&
erab_mod_confirm_ies_o::value_c::operator=(const erab_mod_confirm_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_conf:
      c.set(other.c.get<erab_modify_list_bearer_mod_conf_l>());
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::csg_membership_status:
      c.set(other.c.get<csg_membership_status_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_mod_confirm_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_mod_confirm_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
erab_modify_list_bearer_mod_conf_l& erab_mod_confirm_ies_o::value_c::erab_modify_list_bearer_mod_conf()
{
  assert_choice_type(types::erab_modify_list_bearer_mod_conf, type_, "Value");
  return c.get<erab_modify_list_bearer_mod_conf_l>();
}
erab_list_l& erab_mod_confirm_ies_o::value_c::erab_failed_to_modify_list_bearer_mod_conf()
{
  assert_choice_type(types::erab_failed_to_modify_list_bearer_mod_conf, type_, "Value");
  return c.get<erab_list_l>();
}
erab_list_l& erab_mod_confirm_ies_o::value_c::erab_to_be_released_list_bearer_mod_conf()
{
  assert_choice_type(types::erab_to_be_released_list_bearer_mod_conf, type_, "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& erab_mod_confirm_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
csg_membership_status_e& erab_mod_confirm_ies_o::value_c::csg_membership_status()
{
  assert_choice_type(types::csg_membership_status, type_, "Value");
  return c.get<csg_membership_status_e>();
}
const uint64_t& erab_mod_confirm_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_mod_confirm_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const erab_modify_list_bearer_mod_conf_l& erab_mod_confirm_ies_o::value_c::erab_modify_list_bearer_mod_conf() const
{
  assert_choice_type(types::erab_modify_list_bearer_mod_conf, type_, "Value");
  return c.get<erab_modify_list_bearer_mod_conf_l>();
}
const erab_list_l& erab_mod_confirm_ies_o::value_c::erab_failed_to_modify_list_bearer_mod_conf() const
{
  assert_choice_type(types::erab_failed_to_modify_list_bearer_mod_conf, type_, "Value");
  return c.get<erab_list_l>();
}
const erab_list_l& erab_mod_confirm_ies_o::value_c::erab_to_be_released_list_bearer_mod_conf() const
{
  assert_choice_type(types::erab_to_be_released_list_bearer_mod_conf, type_, "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& erab_mod_confirm_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const csg_membership_status_e& erab_mod_confirm_ies_o::value_c::csg_membership_status() const
{
  assert_choice_type(types::csg_membership_status, type_, "Value");
  return c.get<csg_membership_status_e>();
}
void erab_mod_confirm_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_conf:
      j.start_array("E-RABModifyListBearerModConf");
      for (const auto& e1 : c.get<erab_modify_list_bearer_mod_conf_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::csg_membership_status:
      j.write_str("CSGMembershipStatus", c.get<csg_membership_status_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_mod_confirm_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_modify_list_bearer_mod_conf:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_modify_list_bearer_mod_conf_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_confirm_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_modify_list_bearer_mod_conf:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_modify_list_bearer_mod_conf_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_mod_confirm_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABModifyListBearerModConf",
                                  "E-RABList",
                                  "E-RABList",
                                  "CriticalityDiagnostics",
                                  "CSGMembershipStatus"};
  return convert_enum_idx(options, 7, value, "erab_mod_confirm_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_mod_confirm_ies_o>;

erab_mod_confirm_ies_container::erab_mod_confirm_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_modify_list_bearer_mod_conf(203, crit_e::ignore),
  erab_failed_to_modify_list_bearer_mod_conf(205, crit_e::ignore),
  erab_to_be_released_list_bearer_mod_conf(210, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore),
  csg_membership_status(146, crit_e::ignore)
{}
SRSASN_CODE erab_mod_confirm_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += erab_modify_list_bearer_mod_conf_present ? 1 : 0;
  nof_ies += erab_failed_to_modify_list_bearer_mod_conf_present ? 1 : 0;
  nof_ies += erab_to_be_released_list_bearer_mod_conf_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  nof_ies += csg_membership_status_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_modify_list_bearer_mod_conf_present) {
    HANDLE_CODE(erab_modify_list_bearer_mod_conf.pack(bref));
  }
  if (erab_failed_to_modify_list_bearer_mod_conf_present) {
    HANDLE_CODE(erab_failed_to_modify_list_bearer_mod_conf.pack(bref));
  }
  if (erab_to_be_released_list_bearer_mod_conf_present) {
    HANDLE_CODE(erab_to_be_released_list_bearer_mod_conf.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (csg_membership_status_present) {
    HANDLE_CODE(csg_membership_status.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_confirm_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 203: {
        erab_modify_list_bearer_mod_conf_present = true;
        erab_modify_list_bearer_mod_conf.id      = id;
        HANDLE_CODE(erab_modify_list_bearer_mod_conf.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_modify_list_bearer_mod_conf.value.unpack(bref));
        break;
      }
      case 205: {
        erab_failed_to_modify_list_bearer_mod_conf_present = true;
        erab_failed_to_modify_list_bearer_mod_conf.id      = id;
        HANDLE_CODE(erab_failed_to_modify_list_bearer_mod_conf.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_failed_to_modify_list_bearer_mod_conf.value.unpack(bref));
        break;
      }
      case 210: {
        erab_to_be_released_list_bearer_mod_conf_present = true;
        erab_to_be_released_list_bearer_mod_conf.id      = id;
        HANDLE_CODE(erab_to_be_released_list_bearer_mod_conf.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_to_be_released_list_bearer_mod_conf.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      case 146: {
        csg_membership_status_present = true;
        csg_membership_status.id      = id;
        HANDLE_CODE(csg_membership_status.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(csg_membership_status.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_mod_confirm_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_modify_list_bearer_mod_conf_present) {
    j.write_fieldname("");
    erab_modify_list_bearer_mod_conf.to_json(j);
  }
  if (erab_failed_to_modify_list_bearer_mod_conf_present) {
    j.write_fieldname("");
    erab_failed_to_modify_list_bearer_mod_conf.to_json(j);
  }
  if (erab_to_be_released_list_bearer_mod_conf_present) {
    j.write_fieldname("");
    erab_to_be_released_list_bearer_mod_conf.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  if (csg_membership_status_present) {
    j.write_fieldname("");
    csg_membership_status.to_json(j);
  }
  j.end_obj();
}

// E-RABUsageReportItem ::= SEQUENCE
SRSASN_CODE erabusage_report_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(start_timestamp.pack(bref));
  HANDLE_CODE(end_timestamp.pack(bref));
  HANDLE_CODE(pack_integer(bref, usage_count_ul, (uint64_t)0u, (uint64_t)18446744073709551615u, false, true));
  HANDLE_CODE(pack_integer(bref, usage_count_dl, (uint64_t)0u, (uint64_t)18446744073709551615u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erabusage_report_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(start_timestamp.unpack(bref));
  HANDLE_CODE(end_timestamp.unpack(bref));
  HANDLE_CODE(unpack_integer(usage_count_ul, bref, (uint64_t)0u, (uint64_t)18446744073709551615u, false, true));
  HANDLE_CODE(unpack_integer(usage_count_dl, bref, (uint64_t)0u, (uint64_t)18446744073709551615u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erabusage_report_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("startTimestamp", start_timestamp.to_string());
  j.write_str("endTimestamp", end_timestamp.to_string());
  j.write_int("usageCountUL", usage_count_ul);
  j.write_int("usageCountDL", usage_count_dl);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABUsageReportItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erabusage_report_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {267};
  return map_enum_number(options, 1, idx, "id");
}
bool erabusage_report_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 267 == id;
}
crit_e erabusage_report_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 267) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erabusage_report_item_ies_o::value_c erabusage_report_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 267) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erabusage_report_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 267) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erabusage_report_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABUsageReportItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erabusage_report_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erabusage_report_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erabusage_report_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABUsageReportItem"};
  return convert_enum_idx(options, 1, value, "erabusage_report_item_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erabusage_report_item_ies_o>;

// NR-CGI ::= SEQUENCE
SRSASN_CODE nr_cgi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(nrcell_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE nr_cgi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(nrcell_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void nr_cgi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_str("nRCellIdentity", nrcell_id.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SecondaryRATType ::= ENUMERATED
const char* secondary_rat_type_opts::to_string() const
{
  static const char* options[] = {"nR", "unlicensed"};
  return convert_enum_idx(options, 2, value, "secondary_rat_type_e");
}

// E-RABNotToBeModifiedItemBearerModInd ::= SEQUENCE
SRSASN_CODE erab_not_to_be_modified_item_bearer_mod_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(dl_gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_not_to_be_modified_item_bearer_mod_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(dl_gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_not_to_be_modified_item_bearer_mod_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("dL-GTP-TEID", dl_gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeModifiedItemBearerModInd ::= SEQUENCE
SRSASN_CODE erab_to_be_modified_item_bearer_mod_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(dl_gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(dl_gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_modified_item_bearer_mod_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("dL-GTP-TEID", dl_gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PSCellInformation ::= SEQUENCE
SRSASN_CODE ps_cell_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ncgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ps_cell_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ncgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ps_cell_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nCGI");
  ncgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SecondaryRATDataUsageReportItem ::= SEQUENCE
SRSASN_CODE secondary_rat_data_usage_report_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(secondary_rat_type.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, erabusage_report_list, 1, 2, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE secondary_rat_data_usage_report_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(secondary_rat_type.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(erabusage_report_list, bref, 1, 2, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void secondary_rat_data_usage_report_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("secondaryRATType", secondary_rat_type.to_string());
  j.start_array("e-RABUsageReportList");
  for (const auto& e1 : erabusage_report_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABNotToBeModifiedItemBearerModIndIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_not_to_be_modified_item_bearer_mod_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {202};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_not_to_be_modified_item_bearer_mod_ind_ies_o::is_id_valid(const uint32_t& id)
{
  return 202 == id;
}
crit_e erab_not_to_be_modified_item_bearer_mod_ind_ies_o::get_crit(const uint32_t& id)
{
  if (id == 202) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c
erab_not_to_be_modified_item_bearer_mod_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 202) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_not_to_be_modified_item_bearer_mod_ind_ies_o::get_presence(const uint32_t& id)
{
  if (id == 202) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABNotToBeModifiedItemBearerModInd");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABNotToBeModifiedItemBearerModInd"};
  return convert_enum_idx(options, 1, value, "erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::types");
}

// E-RABToBeModifiedItemBearerModIndIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_modified_item_bearer_mod_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {200};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_modified_item_bearer_mod_ind_ies_o::is_id_valid(const uint32_t& id)
{
  return 200 == id;
}
crit_e erab_to_be_modified_item_bearer_mod_ind_ies_o::get_crit(const uint32_t& id)
{
  if (id == 200) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c
erab_to_be_modified_item_bearer_mod_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 200) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_modified_item_bearer_mod_ind_ies_o::get_presence(const uint32_t& id)
{
  if (id == 200) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeModifiedItemBearerModInd");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeModifiedItemBearerModInd"};
  return convert_enum_idx(options, 1, value, "erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::types");
}

// SecondaryRATDataUsageReportItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t secondary_rat_data_usage_report_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {265};
  return map_enum_number(options, 1, idx, "id");
}
bool secondary_rat_data_usage_report_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 265 == id;
}
crit_e secondary_rat_data_usage_report_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 265) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
secondary_rat_data_usage_report_item_ies_o::value_c
secondary_rat_data_usage_report_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 265) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e secondary_rat_data_usage_report_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 265) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void secondary_rat_data_usage_report_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("SecondaryRATDataUsageReportItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE secondary_rat_data_usage_report_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE secondary_rat_data_usage_report_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* secondary_rat_data_usage_report_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"SecondaryRATDataUsageReportItem"};
  return convert_enum_idx(options, 1, value, "secondary_rat_data_usage_report_item_ies_o::value_c::types");
}

// UserLocationInformation-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t user_location_info_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {288};
  return map_enum_number(options, 1, idx, "id");
}
bool user_location_info_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 288 == id;
}
crit_e user_location_info_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 288) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
user_location_info_ext_ies_o::ext_c user_location_info_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 288) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e user_location_info_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 288) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void user_location_info_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("PSCellInformation");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE user_location_info_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE user_location_info_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* user_location_info_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"PSCellInformation"};
  return convert_enum_idx(options, 1, value, "user_location_info_ext_ies_o::ext_c::types");
}

template struct asn1::protocol_ie_single_container_s<secondary_rat_data_usage_report_item_ies_o>;

// TunnelInformation ::= SEQUENCE
SRSASN_CODE tunnel_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(udp_port_num_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(transport_layer_address.pack(bref));
  if (udp_port_num_present) {
    HANDLE_CODE(udp_port_num.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tunnel_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(udp_port_num_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(transport_layer_address.unpack(bref));
  if (udp_port_num_present) {
    HANDLE_CODE(udp_port_num.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tunnel_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  if (udp_port_num_present) {
    j.write_str("uDP-Port-Number", udp_port_num.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UserLocationInformation ::= SEQUENCE
SRSASN_CODE user_location_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE user_location_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutran_cgi.unpack(bref));
  HANDLE_CODE(tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void user_location_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eutran-cgi");
  eutran_cgi.to_json(j);
  j.write_fieldname("tai");
  tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// E-RABModificationIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_mod_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 199, 201, 226, 176, 264, 189};
  return map_enum_number(options, 8, idx, "id");
}
bool erab_mod_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 199, 201, 226, 176, 264, 189};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_mod_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 199:
      return crit_e::reject;
    case 201:
      return crit_e::reject;
    case 226:
      return crit_e::reject;
    case 176:
      return crit_e::ignore;
    case 264:
      return crit_e::ignore;
    case 189:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_mod_ind_ies_o::value_c erab_mod_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 199:
      ret.set(value_c::types::erab_to_be_modified_list_bearer_mod_ind);
      break;
    case 201:
      ret.set(value_c::types::erab_not_to_be_modified_list_bearer_mod_ind);
      break;
    case 226:
      ret.set(value_c::types::csg_membership_info);
      break;
    case 176:
      ret.set(value_c::types::tunnel_info_for_bbf);
      break;
    case 264:
      ret.set(value_c::types::secondary_rat_data_usage_report_list);
      break;
    case 189:
      ret.set(value_c::types::user_location_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_mod_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 199:
      return presence_e::mandatory;
    case 201:
      return presence_e::optional;
    case 226:
      return presence_e::optional;
    case 176:
      return presence_e::optional;
    case 264:
      return presence_e::optional;
    case 189:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_mod_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_to_be_modified_list_bearer_mod_ind:
      c.destroy<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >();
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      c.destroy<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >();
      break;
    case types::csg_membership_info:
      c.destroy<csg_membership_info_s>();
      break;
    case types::tunnel_info_for_bbf:
      c.destroy<tunnel_info_s>();
      break;
    case types::secondary_rat_data_usage_report_list:
      c.destroy<secondary_rat_data_usage_report_list_l>();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_s>();
      break;
    default:
      break;
  }
}
void erab_mod_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      c.init<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >();
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      c.init<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >();
      break;
    case types::csg_membership_info:
      c.init<csg_membership_info_s>();
      break;
    case types::tunnel_info_for_bbf:
      c.init<tunnel_info_s>();
      break;
    case types::secondary_rat_data_usage_report_list:
      c.init<secondary_rat_data_usage_report_list_l>();
      break;
    case types::user_location_info:
      c.init<user_location_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
  }
}
erab_mod_ind_ies_o::value_c::value_c(const erab_mod_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      c.init(other.c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >());
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      c.init(other.c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >());
      break;
    case types::csg_membership_info:
      c.init(other.c.get<csg_membership_info_s>());
      break;
    case types::tunnel_info_for_bbf:
      c.init(other.c.get<tunnel_info_s>());
      break;
    case types::secondary_rat_data_usage_report_list:
      c.init(other.c.get<secondary_rat_data_usage_report_list_l>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
  }
}
erab_mod_ind_ies_o::value_c& erab_mod_ind_ies_o::value_c::operator=(const erab_mod_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      c.set(other.c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >());
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      c.set(other.c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >());
      break;
    case types::csg_membership_info:
      c.set(other.c.get<csg_membership_info_s>());
      break;
    case types::tunnel_info_for_bbf:
      c.set(other.c.get<tunnel_info_s>());
      break;
    case types::secondary_rat_data_usage_report_list:
      c.set(other.c.get<secondary_rat_data_usage_report_list_l>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_mod_ind_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_mod_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o>&
erab_mod_ind_ies_o::value_c::erab_to_be_modified_list_bearer_mod_ind()
{
  assert_choice_type(types::erab_to_be_modified_list_bearer_mod_ind, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >();
}
erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o>&
erab_mod_ind_ies_o::value_c::erab_not_to_be_modified_list_bearer_mod_ind()
{
  assert_choice_type(types::erab_not_to_be_modified_list_bearer_mod_ind, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >();
}
csg_membership_info_s& erab_mod_ind_ies_o::value_c::csg_membership_info()
{
  assert_choice_type(types::csg_membership_info, type_, "Value");
  return c.get<csg_membership_info_s>();
}
tunnel_info_s& erab_mod_ind_ies_o::value_c::tunnel_info_for_bbf()
{
  assert_choice_type(types::tunnel_info_for_bbf, type_, "Value");
  return c.get<tunnel_info_s>();
}
secondary_rat_data_usage_report_list_l& erab_mod_ind_ies_o::value_c::secondary_rat_data_usage_report_list()
{
  assert_choice_type(types::secondary_rat_data_usage_report_list, type_, "Value");
  return c.get<secondary_rat_data_usage_report_list_l>();
}
user_location_info_s& erab_mod_ind_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_s>();
}
const uint64_t& erab_mod_ind_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_mod_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o>&
erab_mod_ind_ies_o::value_c::erab_to_be_modified_list_bearer_mod_ind() const
{
  assert_choice_type(types::erab_to_be_modified_list_bearer_mod_ind, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >();
}
const erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o>&
erab_mod_ind_ies_o::value_c::erab_not_to_be_modified_list_bearer_mod_ind() const
{
  assert_choice_type(types::erab_not_to_be_modified_list_bearer_mod_ind, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >();
}
const csg_membership_info_s& erab_mod_ind_ies_o::value_c::csg_membership_info() const
{
  assert_choice_type(types::csg_membership_info, type_, "Value");
  return c.get<csg_membership_info_s>();
}
const tunnel_info_s& erab_mod_ind_ies_o::value_c::tunnel_info_for_bbf() const
{
  assert_choice_type(types::tunnel_info_for_bbf, type_, "Value");
  return c.get<tunnel_info_s>();
}
const secondary_rat_data_usage_report_list_l& erab_mod_ind_ies_o::value_c::secondary_rat_data_usage_report_list() const
{
  assert_choice_type(types::secondary_rat_data_usage_report_list, type_, "Value");
  return c.get<secondary_rat_data_usage_report_list_l>();
}
const user_location_info_s& erab_mod_ind_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_s>();
}
void erab_mod_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABToBeModifiedItemBearerModIndIEs}}");
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABNotToBeModifiedItemBearerModIndIEs}}");
      break;
    case types::csg_membership_info:
      j.write_fieldname("CSGMembershipInfo");
      c.get<csg_membership_info_s>().to_json(j);
      break;
    case types::tunnel_info_for_bbf:
      j.write_fieldname("TunnelInformation");
      c.get<tunnel_info_s>().to_json(j);
      break;
    case types::secondary_rat_data_usage_report_list:
      j.start_array("SecondaryRATDataUsageReportList");
      for (const auto& e1 : c.get<secondary_rat_data_usage_report_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_mod_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      HANDLE_CODE(pack_dyn_seq_of(
          bref, c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >(), 1, 256, true));
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      HANDLE_CODE(pack_dyn_seq_of(
          bref, c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >(), 1, 256, true));
      break;
    case types::csg_membership_info:
      HANDLE_CODE(c.get<csg_membership_info_s>().pack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().pack(bref));
      break;
    case types::secondary_rat_data_usage_report_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<secondary_rat_data_usage_report_list_l>(), 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      HANDLE_CODE(unpack_dyn_seq_of(
          c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >(), bref, 1, 256, true));
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      HANDLE_CODE(unpack_dyn_seq_of(
          c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >(), bref, 1, 256, true));
      break;
    case types::csg_membership_info:
      HANDLE_CODE(c.get<csg_membership_info_s>().unpack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().unpack(bref));
      break;
    case types::secondary_rat_data_usage_report_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<secondary_rat_data_usage_report_list_l>(), bref, 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_mod_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RAB-IE-ContainerList{{E-RABToBeModifiedItemBearerModIndIEs}}",
                                  "E-RAB-IE-ContainerList{{E-RABNotToBeModifiedItemBearerModIndIEs}}",
                                  "CSGMembershipInfo",
                                  "TunnelInformation",
                                  "SecondaryRATDataUsageReportList",
                                  "UserLocationInformation"};
  return convert_enum_idx(options, 8, value, "erab_mod_ind_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_mod_ind_ies_o>;

erab_mod_ind_ies_container::erab_mod_ind_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  erab_to_be_modified_list_bearer_mod_ind(199, crit_e::reject),
  erab_not_to_be_modified_list_bearer_mod_ind(201, crit_e::reject),
  csg_membership_info(226, crit_e::reject),
  tunnel_info_for_bbf(176, crit_e::ignore),
  secondary_rat_data_usage_report_list(264, crit_e::ignore),
  user_location_info(189, crit_e::ignore)
{}
SRSASN_CODE erab_mod_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += erab_not_to_be_modified_list_bearer_mod_ind_present ? 1 : 0;
  nof_ies += csg_membership_info_present ? 1 : 0;
  nof_ies += tunnel_info_for_bbf_present ? 1 : 0;
  nof_ies += secondary_rat_data_usage_report_list_present ? 1 : 0;
  nof_ies += user_location_info_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(erab_to_be_modified_list_bearer_mod_ind.pack(bref));
  if (erab_not_to_be_modified_list_bearer_mod_ind_present) {
    HANDLE_CODE(erab_not_to_be_modified_list_bearer_mod_ind.pack(bref));
  }
  if (csg_membership_info_present) {
    HANDLE_CODE(csg_membership_info.pack(bref));
  }
  if (tunnel_info_for_bbf_present) {
    HANDLE_CODE(tunnel_info_for_bbf.pack(bref));
  }
  if (secondary_rat_data_usage_report_list_present) {
    HANDLE_CODE(secondary_rat_data_usage_report_list.pack(bref));
  }
  if (user_location_info_present) {
    HANDLE_CODE(user_location_info.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 199: {
        nof_mandatory_ies--;
        erab_to_be_modified_list_bearer_mod_ind.id = id;
        HANDLE_CODE(erab_to_be_modified_list_bearer_mod_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_to_be_modified_list_bearer_mod_ind.value.unpack(bref));
        break;
      }
      case 201: {
        erab_not_to_be_modified_list_bearer_mod_ind_present = true;
        erab_not_to_be_modified_list_bearer_mod_ind.id      = id;
        HANDLE_CODE(erab_not_to_be_modified_list_bearer_mod_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_not_to_be_modified_list_bearer_mod_ind.value.unpack(bref));
        break;
      }
      case 226: {
        csg_membership_info_present = true;
        csg_membership_info.id      = id;
        HANDLE_CODE(csg_membership_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(csg_membership_info.value.unpack(bref));
        break;
      }
      case 176: {
        tunnel_info_for_bbf_present = true;
        tunnel_info_for_bbf.id      = id;
        HANDLE_CODE(tunnel_info_for_bbf.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(tunnel_info_for_bbf.value.unpack(bref));
        break;
      }
      case 264: {
        secondary_rat_data_usage_report_list_present = true;
        secondary_rat_data_usage_report_list.id      = id;
        HANDLE_CODE(secondary_rat_data_usage_report_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(secondary_rat_data_usage_report_list.value.unpack(bref));
        break;
      }
      case 189: {
        user_location_info_present = true;
        user_location_info.id      = id;
        HANDLE_CODE(user_location_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(user_location_info.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_mod_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  erab_to_be_modified_list_bearer_mod_ind.to_json(j);
  if (erab_not_to_be_modified_list_bearer_mod_ind_present) {
    j.write_fieldname("");
    erab_not_to_be_modified_list_bearer_mod_ind.to_json(j);
  }
  if (csg_membership_info_present) {
    j.write_fieldname("");
    csg_membership_info.to_json(j);
  }
  if (tunnel_info_for_bbf_present) {
    j.write_fieldname("");
    tunnel_info_for_bbf.to_json(j);
  }
  if (secondary_rat_data_usage_report_list_present) {
    j.write_fieldname("");
    secondary_rat_data_usage_report_list.to_json(j);
  }
  if (user_location_info_present) {
    j.write_fieldname("");
    user_location_info.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyItemBearerModRes ::= SEQUENCE
SRSASN_CODE erab_modify_item_bearer_mod_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_item_bearer_mod_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_modify_item_bearer_mod_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyItemBearerModResIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_modify_item_bearer_mod_res_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {37};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_modify_item_bearer_mod_res_ies_o::is_id_valid(const uint32_t& id)
{
  return 37 == id;
}
crit_e erab_modify_item_bearer_mod_res_ies_o::get_crit(const uint32_t& id)
{
  if (id == 37) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_modify_item_bearer_mod_res_ies_o::value_c erab_modify_item_bearer_mod_res_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 37) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_modify_item_bearer_mod_res_ies_o::get_presence(const uint32_t& id)
{
  if (id == 37) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_modify_item_bearer_mod_res_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABModifyItemBearerModRes");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_modify_item_bearer_mod_res_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_item_bearer_mod_res_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_modify_item_bearer_mod_res_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABModifyItemBearerModRes"};
  return convert_enum_idx(options, 1, value, "erab_modify_item_bearer_mod_res_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_modify_item_bearer_mod_res_ies_o>;

// TransportInformation ::= SEQUENCE
SRSASN_CODE transport_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(ul_gtp_teid.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE transport_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(ul_gtp_teid.unpack(bref));

  return SRSASN_SUCCESS;
}
void transport_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("uL-GTP-TEID", ul_gtp_teid.to_string());
  j.end_obj();
}

// E-RABToBeModifyItemBearerModReqExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_to_be_modify_item_bearer_mod_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {185};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_modify_item_bearer_mod_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 185 == id;
}
crit_e erab_to_be_modify_item_bearer_mod_req_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 185) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c
erab_to_be_modify_item_bearer_mod_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 185) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_modify_item_bearer_mod_req_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 185) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("TransportInformation");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"TransportInformation"};
  return convert_enum_idx(options, 1, value, "erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::types");
}

// E-RABToBeModifiedItemBearerModReq ::= SEQUENCE
SRSASN_CODE erab_to_be_modified_item_bearer_mod_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.pack(bref));
  HANDLE_CODE(nas_pdu.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.unpack(bref));
  HANDLE_CODE(nas_pdu.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_modified_item_bearer_mod_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("e-RABLevelQoSParameters");
  erab_level_qos_params.to_json(j);
  j.write_str("nAS-PDU", nas_pdu.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// E-RABToBeModifiedItemBearerModReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_modified_item_bearer_mod_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {36};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_modified_item_bearer_mod_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 36 == id;
}
crit_e erab_to_be_modified_item_bearer_mod_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 36) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_modified_item_bearer_mod_req_ies_o::value_c
erab_to_be_modified_item_bearer_mod_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 36) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_modified_item_bearer_mod_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 36) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeModifiedItemBearerModReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeModifiedItemBearerModReq"};
  return convert_enum_idx(options, 1, value, "erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::types");
}

// UEAggregate-MaximumBitrates-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t ue_aggregate_maximum_bitrates_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {259, 260};
  return map_enum_number(options, 2, idx, "id");
}
bool ue_aggregate_maximum_bitrates_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {259, 260};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ue_aggregate_maximum_bitrates_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 259:
      return crit_e::ignore;
    case 260:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ue_aggregate_maximum_bitrates_ext_ies_o::ext_c ue_aggregate_maximum_bitrates_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 259:
      ret.set(ext_c::types::extended_u_eaggregate_maximum_bit_rate_dl);
      break;
    case 260:
      ret.set(ext_c::types::extended_u_eaggregate_maximum_bit_rate_ul);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ue_aggregate_maximum_bitrates_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 259:
      return presence_e::optional;
    case 260:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::destroy_() {}
void ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::ext_c(const ue_aggregate_maximum_bitrates_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::extended_u_eaggregate_maximum_bit_rate_dl:
      c.init(other.c.get<uint64_t>());
      break;
    case types::extended_u_eaggregate_maximum_bit_rate_ul:
      c.init(other.c.get<uint64_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_aggregate_maximum_bitrates_ext_ies_o::ext_c");
  }
}
ue_aggregate_maximum_bitrates_ext_ies_o::ext_c&
ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::operator=(const ue_aggregate_maximum_bitrates_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::extended_u_eaggregate_maximum_bit_rate_dl:
      c.set(other.c.get<uint64_t>());
      break;
    case types::extended_u_eaggregate_maximum_bit_rate_ul:
      c.set(other.c.get<uint64_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_aggregate_maximum_bitrates_ext_ies_o::ext_c");
  }

  return *this;
}
uint64_t& ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::extended_u_eaggregate_maximum_bit_rate_dl()
{
  assert_choice_type(types::extended_u_eaggregate_maximum_bit_rate_dl, type_, "Extension");
  return c.get<uint64_t>();
}
uint64_t& ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::extended_u_eaggregate_maximum_bit_rate_ul()
{
  assert_choice_type(types::extended_u_eaggregate_maximum_bit_rate_ul, type_, "Extension");
  return c.get<uint64_t>();
}
const uint64_t& ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::extended_u_eaggregate_maximum_bit_rate_dl() const
{
  assert_choice_type(types::extended_u_eaggregate_maximum_bit_rate_dl, type_, "Extension");
  return c.get<uint64_t>();
}
const uint64_t& ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::extended_u_eaggregate_maximum_bit_rate_ul() const
{
  assert_choice_type(types::extended_u_eaggregate_maximum_bit_rate_ul, type_, "Extension");
  return c.get<uint64_t>();
}
void ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::extended_u_eaggregate_maximum_bit_rate_dl:
      j.write_int("INTEGER (10000000001..4000000000000,...)", c.get<uint64_t>());
      break;
    case types::extended_u_eaggregate_maximum_bit_rate_ul:
      j.write_int("INTEGER (10000000001..4000000000000,...)", c.get<uint64_t>());
      break;
    default:
      log_invalid_choice_id(type_, "ue_aggregate_maximum_bitrates_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::extended_u_eaggregate_maximum_bit_rate_dl:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    case types::extended_u_eaggregate_maximum_bit_rate_ul:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    default:
      log_invalid_choice_id(type_, "ue_aggregate_maximum_bitrates_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::extended_u_eaggregate_maximum_bit_rate_dl:
      HANDLE_CODE(
          unpack_integer(c.get<uint64_t>(), bref, (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    case types::extended_u_eaggregate_maximum_bit_rate_ul:
      HANDLE_CODE(
          unpack_integer(c.get<uint64_t>(), bref, (uint64_t)10000000001u, (uint64_t)4000000000000u, true, true));
      break;
    default:
      log_invalid_choice_id(type_, "ue_aggregate_maximum_bitrates_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (10000000001..4000000000000,...)",
                                  "INTEGER (10000000001..4000000000000,...)"};
  return convert_enum_idx(options, 2, value, "ue_aggregate_maximum_bitrates_ext_ies_o::ext_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_to_be_modified_item_bearer_mod_req_ies_o>;

// SecondaryRATDataUsageRequest ::= ENUMERATED
const char* secondary_rat_data_usage_request_opts::to_string() const
{
  static const char* options[] = {"requested"};
  return convert_enum_idx(options, 1, value, "secondary_rat_data_usage_request_e");
}

template struct asn1::protocol_ext_field_s<ue_aggregate_maximum_bitrates_ext_ies_o>;

ue_aggregate_maximum_bitrates_ext_ies_container::ue_aggregate_maximum_bitrates_ext_ies_container() :
  extended_u_eaggregate_maximum_bit_rate_dl(259, crit_e::ignore),
  extended_u_eaggregate_maximum_bit_rate_ul(260, crit_e::ignore)
{}
SRSASN_CODE ue_aggregate_maximum_bitrates_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += extended_u_eaggregate_maximum_bit_rate_dl_present ? 1 : 0;
  nof_ies += extended_u_eaggregate_maximum_bit_rate_ul_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (extended_u_eaggregate_maximum_bit_rate_dl_present) {
    HANDLE_CODE(extended_u_eaggregate_maximum_bit_rate_dl.pack(bref));
  }
  if (extended_u_eaggregate_maximum_bit_rate_ul_present) {
    HANDLE_CODE(extended_u_eaggregate_maximum_bit_rate_ul.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_aggregate_maximum_bitrates_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 259: {
        extended_u_eaggregate_maximum_bit_rate_dl_present = true;
        extended_u_eaggregate_maximum_bit_rate_dl.id      = id;
        HANDLE_CODE(extended_u_eaggregate_maximum_bit_rate_dl.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(extended_u_eaggregate_maximum_bit_rate_dl.ext.unpack(bref));
        break;
      }
      case 260: {
        extended_u_eaggregate_maximum_bit_rate_ul_present = true;
        extended_u_eaggregate_maximum_bit_rate_ul.id      = id;
        HANDLE_CODE(extended_u_eaggregate_maximum_bit_rate_ul.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(extended_u_eaggregate_maximum_bit_rate_ul.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void ue_aggregate_maximum_bitrates_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (extended_u_eaggregate_maximum_bit_rate_dl_present) {
    j.write_fieldname("");
    extended_u_eaggregate_maximum_bit_rate_dl.to_json(j);
  }
  if (extended_u_eaggregate_maximum_bit_rate_ul_present) {
    j.write_fieldname("");
    extended_u_eaggregate_maximum_bit_rate_ul.to_json(j);
  }
  j.end_obj();
}

// UEAggregateMaximumBitrate ::= SEQUENCE
SRSASN_CODE ue_aggregate_maximum_bitrate_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, ueaggregate_maximum_bit_rate_dl, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(pack_integer(bref, ueaggregate_maximum_bit_rate_ul, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_aggregate_maximum_bitrate_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(ueaggregate_maximum_bit_rate_dl, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(unpack_integer(ueaggregate_maximum_bit_rate_ul, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_aggregate_maximum_bitrate_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("uEaggregateMaximumBitRateDL", ueaggregate_maximum_bit_rate_dl);
  j.write_int("uEaggregateMaximumBitRateUL", ueaggregate_maximum_bit_rate_ul);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyRequestIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_modify_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 66, 30, 268};
  return map_enum_number(options, 5, idx, "id");
}
bool erab_modify_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 66, 30, 268};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_modify_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 66:
      return crit_e::reject;
    case 30:
      return crit_e::reject;
    case 268:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_modify_request_ies_o::value_c erab_modify_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 30:
      ret.set(value_c::types::erab_to_be_modified_list_bearer_mod_req);
      break;
    case 268:
      ret.set(value_c::types::secondary_rat_data_usage_request);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_modify_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 66:
      return presence_e::optional;
    case 30:
      return presence_e::mandatory;
    case 268:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_modify_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      c.destroy<erab_to_be_modified_list_bearer_mod_req_l>();
      break;
    default:
      break;
  }
}
void erab_modify_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      c.init<erab_to_be_modified_list_bearer_mod_req_l>();
      break;
    case types::secondary_rat_data_usage_request:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
  }
}
erab_modify_request_ies_o::value_c::value_c(const erab_modify_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      c.init(other.c.get<erab_to_be_modified_list_bearer_mod_req_l>());
      break;
    case types::secondary_rat_data_usage_request:
      c.init(other.c.get<secondary_rat_data_usage_request_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
  }
}
erab_modify_request_ies_o::value_c&
erab_modify_request_ies_o::value_c::operator=(const erab_modify_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      c.set(other.c.get<erab_to_be_modified_list_bearer_mod_req_l>());
      break;
    case types::secondary_rat_data_usage_request:
      c.set(other.c.get<secondary_rat_data_usage_request_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_modify_request_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_modify_request_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
ue_aggregate_maximum_bitrate_s& erab_modify_request_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type(types::ueaggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_to_be_modified_list_bearer_mod_req_l& erab_modify_request_ies_o::value_c::erab_to_be_modified_list_bearer_mod_req()
{
  assert_choice_type(types::erab_to_be_modified_list_bearer_mod_req, type_, "Value");
  return c.get<erab_to_be_modified_list_bearer_mod_req_l>();
}
secondary_rat_data_usage_request_e& erab_modify_request_ies_o::value_c::secondary_rat_data_usage_request()
{
  assert_choice_type(types::secondary_rat_data_usage_request, type_, "Value");
  return c.get<secondary_rat_data_usage_request_e>();
}
const uint64_t& erab_modify_request_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_modify_request_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const ue_aggregate_maximum_bitrate_s& erab_modify_request_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type(types::ueaggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_to_be_modified_list_bearer_mod_req_l&
erab_modify_request_ies_o::value_c::erab_to_be_modified_list_bearer_mod_req() const
{
  assert_choice_type(types::erab_to_be_modified_list_bearer_mod_req, type_, "Value");
  return c.get<erab_to_be_modified_list_bearer_mod_req_l>();
}
const secondary_rat_data_usage_request_e& erab_modify_request_ies_o::value_c::secondary_rat_data_usage_request() const
{
  assert_choice_type(types::secondary_rat_data_usage_request, type_, "Value");
  return c.get<secondary_rat_data_usage_request_e>();
}
void erab_modify_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      j.start_array("E-RABToBeModifiedListBearerModReq");
      for (const auto& e1 : c.get<erab_to_be_modified_list_bearer_mod_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::secondary_rat_data_usage_request:
      j.write_str("SecondaryRATDataUsageRequest", "requested");
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_modify_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_to_be_modified_list_bearer_mod_req_l>(), 1, 256, true));
      break;
    case types::secondary_rat_data_usage_request:
      HANDLE_CODE(c.get<secondary_rat_data_usage_request_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_to_be_modified_list_bearer_mod_req_l>(), bref, 1, 256, true));
      break;
    case types::secondary_rat_data_usage_request:
      HANDLE_CODE(c.get<secondary_rat_data_usage_request_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_modify_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "UEAggregateMaximumBitrate",
                                  "E-RABToBeModifiedListBearerModReq",
                                  "SecondaryRATDataUsageRequest"};
  return convert_enum_idx(options, 5, value, "erab_modify_request_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_modify_request_ies_o>;

erab_modify_request_ies_container::erab_modify_request_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_modified_list_bearer_mod_req(30, crit_e::reject),
  secondary_rat_data_usage_request(268, crit_e::ignore)
{}
SRSASN_CODE erab_modify_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += ueaggregate_maximum_bitrate_present ? 1 : 0;
  nof_ies += secondary_rat_data_usage_request_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (ueaggregate_maximum_bitrate_present) {
    HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  }
  HANDLE_CODE(erab_to_be_modified_list_bearer_mod_req.pack(bref));
  if (secondary_rat_data_usage_request_present) {
    HANDLE_CODE(secondary_rat_data_usage_request.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 66: {
        ueaggregate_maximum_bitrate_present = true;
        ueaggregate_maximum_bitrate.id      = id;
        HANDLE_CODE(ueaggregate_maximum_bitrate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ueaggregate_maximum_bitrate.value.unpack(bref));
        break;
      }
      case 30: {
        nof_mandatory_ies--;
        erab_to_be_modified_list_bearer_mod_req.id = id;
        HANDLE_CODE(erab_to_be_modified_list_bearer_mod_req.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_to_be_modified_list_bearer_mod_req.value.unpack(bref));
        break;
      }
      case 268: {
        secondary_rat_data_usage_request_present = true;
        secondary_rat_data_usage_request.id      = id;
        HANDLE_CODE(secondary_rat_data_usage_request.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(secondary_rat_data_usage_request.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_modify_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (ueaggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ueaggregate_maximum_bitrate.to_json(j);
  }
  j.write_fieldname("");
  erab_to_be_modified_list_bearer_mod_req.to_json(j);
  if (secondary_rat_data_usage_request_present) {
    j.write_fieldname("");
    secondary_rat_data_usage_request.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyResponseIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_modify_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 31, 32, 58, 264};
  return map_enum_number(options, 6, idx, "id");
}
bool erab_modify_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 31, 32, 58, 264};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_modify_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 31:
      return crit_e::ignore;
    case 32:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 264:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_modify_resp_ies_o::value_c erab_modify_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 31:
      ret.set(value_c::types::erab_modify_list_bearer_mod_res);
      break;
    case 32:
      ret.set(value_c::types::erab_failed_to_modify_list);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 264:
      ret.set(value_c::types::secondary_rat_data_usage_report_list);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_modify_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 31:
      return presence_e::optional;
    case 32:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 264:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_modify_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_modify_list_bearer_mod_res:
      c.destroy<erab_modify_list_bearer_mod_res_l>();
      break;
    case types::erab_failed_to_modify_list:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    case types::secondary_rat_data_usage_report_list:
      c.destroy<secondary_rat_data_usage_report_list_l>();
      break;
    default:
      break;
  }
}
void erab_modify_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_modify_list_bearer_mod_res:
      c.init<erab_modify_list_bearer_mod_res_l>();
      break;
    case types::erab_failed_to_modify_list:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::secondary_rat_data_usage_report_list:
      c.init<secondary_rat_data_usage_report_list_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
  }
}
erab_modify_resp_ies_o::value_c::value_c(const erab_modify_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_res:
      c.init(other.c.get<erab_modify_list_bearer_mod_res_l>());
      break;
    case types::erab_failed_to_modify_list:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::secondary_rat_data_usage_report_list:
      c.init(other.c.get<secondary_rat_data_usage_report_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
  }
}
erab_modify_resp_ies_o::value_c&
erab_modify_resp_ies_o::value_c::operator=(const erab_modify_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_res:
      c.set(other.c.get<erab_modify_list_bearer_mod_res_l>());
      break;
    case types::erab_failed_to_modify_list:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::secondary_rat_data_usage_report_list:
      c.set(other.c.get<secondary_rat_data_usage_report_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_modify_resp_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_modify_resp_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
erab_modify_list_bearer_mod_res_l& erab_modify_resp_ies_o::value_c::erab_modify_list_bearer_mod_res()
{
  assert_choice_type(types::erab_modify_list_bearer_mod_res, type_, "Value");
  return c.get<erab_modify_list_bearer_mod_res_l>();
}
erab_list_l& erab_modify_resp_ies_o::value_c::erab_failed_to_modify_list()
{
  assert_choice_type(types::erab_failed_to_modify_list, type_, "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& erab_modify_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
secondary_rat_data_usage_report_list_l& erab_modify_resp_ies_o::value_c::secondary_rat_data_usage_report_list()
{
  assert_choice_type(types::secondary_rat_data_usage_report_list, type_, "Value");
  return c.get<secondary_rat_data_usage_report_list_l>();
}
const uint64_t& erab_modify_resp_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_modify_resp_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const erab_modify_list_bearer_mod_res_l& erab_modify_resp_ies_o::value_c::erab_modify_list_bearer_mod_res() const
{
  assert_choice_type(types::erab_modify_list_bearer_mod_res, type_, "Value");
  return c.get<erab_modify_list_bearer_mod_res_l>();
}
const erab_list_l& erab_modify_resp_ies_o::value_c::erab_failed_to_modify_list() const
{
  assert_choice_type(types::erab_failed_to_modify_list, type_, "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& erab_modify_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const secondary_rat_data_usage_report_list_l&
erab_modify_resp_ies_o::value_c::secondary_rat_data_usage_report_list() const
{
  assert_choice_type(types::secondary_rat_data_usage_report_list, type_, "Value");
  return c.get<secondary_rat_data_usage_report_list_l>();
}
void erab_modify_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_res:
      j.start_array("E-RABModifyListBearerModRes");
      for (const auto& e1 : c.get<erab_modify_list_bearer_mod_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_modify_list:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::secondary_rat_data_usage_report_list:
      j.start_array("SecondaryRATDataUsageReportList");
      for (const auto& e1 : c.get<secondary_rat_data_usage_report_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_modify_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_modify_list_bearer_mod_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_modify_list_bearer_mod_res_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_modify_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::secondary_rat_data_usage_report_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<secondary_rat_data_usage_report_list_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_modify_list_bearer_mod_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_modify_list_bearer_mod_res_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_modify_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::secondary_rat_data_usage_report_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<secondary_rat_data_usage_report_list_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_modify_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABModifyListBearerModRes",
                                  "E-RABList",
                                  "CriticalityDiagnostics",
                                  "SecondaryRATDataUsageReportList"};
  return convert_enum_idx(options, 6, value, "erab_modify_resp_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_modify_resp_ies_o>;

erab_modify_resp_ies_container::erab_modify_resp_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_modify_list_bearer_mod_res(31, crit_e::ignore),
  erab_failed_to_modify_list(32, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore),
  secondary_rat_data_usage_report_list(264, crit_e::ignore)
{}
SRSASN_CODE erab_modify_resp_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += erab_modify_list_bearer_mod_res_present ? 1 : 0;
  nof_ies += erab_failed_to_modify_list_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  nof_ies += secondary_rat_data_usage_report_list_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_modify_list_bearer_mod_res_present) {
    HANDLE_CODE(erab_modify_list_bearer_mod_res.pack(bref));
  }
  if (erab_failed_to_modify_list_present) {
    HANDLE_CODE(erab_failed_to_modify_list.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (secondary_rat_data_usage_report_list_present) {
    HANDLE_CODE(secondary_rat_data_usage_report_list.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_resp_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 31: {
        erab_modify_list_bearer_mod_res_present = true;
        erab_modify_list_bearer_mod_res.id      = id;
        HANDLE_CODE(erab_modify_list_bearer_mod_res.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_modify_list_bearer_mod_res.value.unpack(bref));
        break;
      }
      case 32: {
        erab_failed_to_modify_list_present = true;
        erab_failed_to_modify_list.id      = id;
        HANDLE_CODE(erab_failed_to_modify_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_failed_to_modify_list.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      case 264: {
        secondary_rat_data_usage_report_list_present = true;
        secondary_rat_data_usage_report_list.id      = id;
        HANDLE_CODE(secondary_rat_data_usage_report_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(secondary_rat_data_usage_report_list.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_modify_resp_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_modify_list_bearer_mod_res_present) {
    j.write_fieldname("");
    erab_modify_list_bearer_mod_res.to_json(j);
  }
  if (erab_failed_to_modify_list_present) {
    j.write_fieldname("");
    erab_failed_to_modify_list.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  if (secondary_rat_data_usage_report_list_present) {
    j.write_fieldname("");
    secondary_rat_data_usage_report_list.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseCommandIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_release_cmd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 66, 33, 26};
  return map_enum_number(options, 5, idx, "id");
}
bool erab_release_cmd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 66, 33, 26};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_release_cmd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 66:
      return crit_e::reject;
    case 33:
      return crit_e::ignore;
    case 26:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_release_cmd_ies_o::value_c erab_release_cmd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 33:
      ret.set(value_c::types::erab_to_be_released_list);
      break;
    case 26:
      ret.set(value_c::types::nas_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_release_cmd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 66:
      return presence_e::optional;
    case 33:
      return presence_e::mandatory;
    case 26:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_release_cmd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_released_list:
      c.destroy<erab_list_l>();
      break;
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void erab_release_cmd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_released_list:
      c.init<erab_list_l>();
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
  }
}
erab_release_cmd_ies_o::value_c::value_c(const erab_release_cmd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_released_list:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
  }
}
erab_release_cmd_ies_o::value_c&
erab_release_cmd_ies_o::value_c::operator=(const erab_release_cmd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_released_list:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_release_cmd_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_release_cmd_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
ue_aggregate_maximum_bitrate_s& erab_release_cmd_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type(types::ueaggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_list_l& erab_release_cmd_ies_o::value_c::erab_to_be_released_list()
{
  assert_choice_type(types::erab_to_be_released_list, type_, "Value");
  return c.get<erab_list_l>();
}
unbounded_octstring<true>& erab_release_cmd_ies_o::value_c::nas_pdu()
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& erab_release_cmd_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_release_cmd_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const ue_aggregate_maximum_bitrate_s& erab_release_cmd_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type(types::ueaggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_list_l& erab_release_cmd_ies_o::value_c::erab_to_be_released_list() const
{
  assert_choice_type(types::erab_to_be_released_list, type_, "Value");
  return c.get<erab_list_l>();
}
const unbounded_octstring<true>& erab_release_cmd_ies_o::value_c::nas_pdu() const
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
void erab_release_cmd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_released_list:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_release_cmd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_released_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_cmd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_released_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_release_cmd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "UEAggregateMaximumBitrate", "E-RABList", "OCTET STRING"};
  return convert_enum_idx(options, 5, value, "erab_release_cmd_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_release_cmd_ies_o>;

erab_release_cmd_ies_container::erab_release_cmd_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_released_list(33, crit_e::ignore),
  nas_pdu(26, crit_e::ignore)
{}
SRSASN_CODE erab_release_cmd_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += ueaggregate_maximum_bitrate_present ? 1 : 0;
  nof_ies += nas_pdu_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (ueaggregate_maximum_bitrate_present) {
    HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  }
  HANDLE_CODE(erab_to_be_released_list.pack(bref));
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_cmd_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 66: {
        ueaggregate_maximum_bitrate_present = true;
        ueaggregate_maximum_bitrate.id      = id;
        HANDLE_CODE(ueaggregate_maximum_bitrate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ueaggregate_maximum_bitrate.value.unpack(bref));
        break;
      }
      case 33: {
        nof_mandatory_ies--;
        erab_to_be_released_list.id = id;
        HANDLE_CODE(erab_to_be_released_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_to_be_released_list.value.unpack(bref));
        break;
      }
      case 26: {
        nas_pdu_present = true;
        nas_pdu.id      = id;
        HANDLE_CODE(nas_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nas_pdu.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_release_cmd_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (ueaggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ueaggregate_maximum_bitrate.to_json(j);
  }
  j.write_fieldname("");
  erab_to_be_released_list.to_json(j);
  if (nas_pdu_present) {
    j.write_fieldname("");
    nas_pdu.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_release_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 110, 189, 264};
  return map_enum_number(options, 5, idx, "id");
}
bool erab_release_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 110, 189, 264};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_release_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 110:
      return crit_e::ignore;
    case 189:
      return crit_e::ignore;
    case 264:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_release_ind_ies_o::value_c erab_release_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 110:
      ret.set(value_c::types::erab_released_list);
      break;
    case 189:
      ret.set(value_c::types::user_location_info);
      break;
    case 264:
      ret.set(value_c::types::secondary_rat_data_usage_report_list);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_release_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 110:
      return presence_e::mandatory;
    case 189:
      return presence_e::optional;
    case 264:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_release_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_released_list:
      c.destroy<erab_list_l>();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_s>();
      break;
    case types::secondary_rat_data_usage_report_list:
      c.destroy<secondary_rat_data_usage_report_list_l>();
      break;
    default:
      break;
  }
}
void erab_release_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_released_list:
      c.init<erab_list_l>();
      break;
    case types::user_location_info:
      c.init<user_location_info_s>();
      break;
    case types::secondary_rat_data_usage_report_list:
      c.init<secondary_rat_data_usage_report_list_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
  }
}
erab_release_ind_ies_o::value_c::value_c(const erab_release_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_released_list:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_s>());
      break;
    case types::secondary_rat_data_usage_report_list:
      c.init(other.c.get<secondary_rat_data_usage_report_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
  }
}
erab_release_ind_ies_o::value_c&
erab_release_ind_ies_o::value_c::operator=(const erab_release_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_released_list:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_s>());
      break;
    case types::secondary_rat_data_usage_report_list:
      c.set(other.c.get<secondary_rat_data_usage_report_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_release_ind_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_release_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
erab_list_l& erab_release_ind_ies_o::value_c::erab_released_list()
{
  assert_choice_type(types::erab_released_list, type_, "Value");
  return c.get<erab_list_l>();
}
user_location_info_s& erab_release_ind_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_s>();
}
secondary_rat_data_usage_report_list_l& erab_release_ind_ies_o::value_c::secondary_rat_data_usage_report_list()
{
  assert_choice_type(types::secondary_rat_data_usage_report_list, type_, "Value");
  return c.get<secondary_rat_data_usage_report_list_l>();
}
const uint64_t& erab_release_ind_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_release_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const erab_list_l& erab_release_ind_ies_o::value_c::erab_released_list() const
{
  assert_choice_type(types::erab_released_list, type_, "Value");
  return c.get<erab_list_l>();
}
const user_location_info_s& erab_release_ind_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_s>();
}
const secondary_rat_data_usage_report_list_l&
erab_release_ind_ies_o::value_c::secondary_rat_data_usage_report_list() const
{
  assert_choice_type(types::secondary_rat_data_usage_report_list, type_, "Value");
  return c.get<secondary_rat_data_usage_report_list_l>();
}
void erab_release_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_released_list:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_s>().to_json(j);
      break;
    case types::secondary_rat_data_usage_report_list:
      j.start_array("SecondaryRATDataUsageReportList");
      for (const auto& e1 : c.get<secondary_rat_data_usage_report_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_release_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_released_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().pack(bref));
      break;
    case types::secondary_rat_data_usage_report_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<secondary_rat_data_usage_report_list_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_released_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().unpack(bref));
      break;
    case types::secondary_rat_data_usage_report_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<secondary_rat_data_usage_report_list_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_release_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABList",
                                  "UserLocationInformation",
                                  "SecondaryRATDataUsageReportList"};
  return convert_enum_idx(options, 5, value, "erab_release_ind_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_release_ind_ies_o>;

erab_release_ind_ies_container::erab_release_ind_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  erab_released_list(110, crit_e::ignore),
  user_location_info(189, crit_e::ignore),
  secondary_rat_data_usage_report_list(264, crit_e::ignore)
{}
SRSASN_CODE erab_release_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += user_location_info_present ? 1 : 0;
  nof_ies += secondary_rat_data_usage_report_list_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(erab_released_list.pack(bref));
  if (user_location_info_present) {
    HANDLE_CODE(user_location_info.pack(bref));
  }
  if (secondary_rat_data_usage_report_list_present) {
    HANDLE_CODE(secondary_rat_data_usage_report_list.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 110: {
        nof_mandatory_ies--;
        erab_released_list.id = id;
        HANDLE_CODE(erab_released_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_released_list.value.unpack(bref));
        break;
      }
      case 189: {
        user_location_info_present = true;
        user_location_info.id      = id;
        HANDLE_CODE(user_location_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(user_location_info.value.unpack(bref));
        break;
      }
      case 264: {
        secondary_rat_data_usage_report_list_present = true;
        secondary_rat_data_usage_report_list.id      = id;
        HANDLE_CODE(secondary_rat_data_usage_report_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(secondary_rat_data_usage_report_list.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_release_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  erab_released_list.to_json(j);
  if (user_location_info_present) {
    j.write_fieldname("");
    user_location_info.to_json(j);
  }
  if (secondary_rat_data_usage_report_list_present) {
    j.write_fieldname("");
    secondary_rat_data_usage_report_list.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseItemBearerRelComp ::= SEQUENCE
SRSASN_CODE erab_release_item_bearer_rel_comp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_item_bearer_rel_comp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_release_item_bearer_rel_comp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseItemBearerRelCompIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_release_item_bearer_rel_comp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {15};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_release_item_bearer_rel_comp_ies_o::is_id_valid(const uint32_t& id)
{
  return 15 == id;
}
crit_e erab_release_item_bearer_rel_comp_ies_o::get_crit(const uint32_t& id)
{
  if (id == 15) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_release_item_bearer_rel_comp_ies_o::value_c erab_release_item_bearer_rel_comp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 15) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_release_item_bearer_rel_comp_ies_o::get_presence(const uint32_t& id)
{
  if (id == 15) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_release_item_bearer_rel_comp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABReleaseItemBearerRelComp");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_release_item_bearer_rel_comp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_item_bearer_rel_comp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_release_item_bearer_rel_comp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABReleaseItemBearerRelComp"};
  return convert_enum_idx(options, 1, value, "erab_release_item_bearer_rel_comp_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_release_item_bearer_rel_comp_ies_o>;

// E-RABReleaseResponseIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_release_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 69, 34, 58, 189, 264};
  return map_enum_number(options, 7, idx, "id");
}
bool erab_release_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 69, 34, 58, 189, 264};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_release_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 69:
      return crit_e::ignore;
    case 34:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 189:
      return crit_e::ignore;
    case 264:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_release_resp_ies_o::value_c erab_release_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 69:
      ret.set(value_c::types::erab_release_list_bearer_rel_comp);
      break;
    case 34:
      ret.set(value_c::types::erab_failed_to_release_list);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 189:
      ret.set(value_c::types::user_location_info);
      break;
    case 264:
      ret.set(value_c::types::secondary_rat_data_usage_report_list);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_release_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 69:
      return presence_e::optional;
    case 34:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 189:
      return presence_e::optional;
    case 264:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_release_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_release_list_bearer_rel_comp:
      c.destroy<erab_release_list_bearer_rel_comp_l>();
      break;
    case types::erab_failed_to_release_list:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_s>();
      break;
    case types::secondary_rat_data_usage_report_list:
      c.destroy<secondary_rat_data_usage_report_list_l>();
      break;
    default:
      break;
  }
}
void erab_release_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_release_list_bearer_rel_comp:
      c.init<erab_release_list_bearer_rel_comp_l>();
      break;
    case types::erab_failed_to_release_list:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::user_location_info:
      c.init<user_location_info_s>();
      break;
    case types::secondary_rat_data_usage_report_list:
      c.init<secondary_rat_data_usage_report_list_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
  }
}
erab_release_resp_ies_o::value_c::value_c(const erab_release_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_release_list_bearer_rel_comp:
      c.init(other.c.get<erab_release_list_bearer_rel_comp_l>());
      break;
    case types::erab_failed_to_release_list:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_s>());
      break;
    case types::secondary_rat_data_usage_report_list:
      c.init(other.c.get<secondary_rat_data_usage_report_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
  }
}
erab_release_resp_ies_o::value_c&
erab_release_resp_ies_o::value_c::operator=(const erab_release_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_release_list_bearer_rel_comp:
      c.set(other.c.get<erab_release_list_bearer_rel_comp_l>());
      break;
    case types::erab_failed_to_release_list:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_s>());
      break;
    case types::secondary_rat_data_usage_report_list:
      c.set(other.c.get<secondary_rat_data_usage_report_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_release_resp_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_release_resp_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
erab_release_list_bearer_rel_comp_l& erab_release_resp_ies_o::value_c::erab_release_list_bearer_rel_comp()
{
  assert_choice_type(types::erab_release_list_bearer_rel_comp, type_, "Value");
  return c.get<erab_release_list_bearer_rel_comp_l>();
}
erab_list_l& erab_release_resp_ies_o::value_c::erab_failed_to_release_list()
{
  assert_choice_type(types::erab_failed_to_release_list, type_, "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& erab_release_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
user_location_info_s& erab_release_resp_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_s>();
}
secondary_rat_data_usage_report_list_l& erab_release_resp_ies_o::value_c::secondary_rat_data_usage_report_list()
{
  assert_choice_type(types::secondary_rat_data_usage_report_list, type_, "Value");
  return c.get<secondary_rat_data_usage_report_list_l>();
}
const uint64_t& erab_release_resp_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_release_resp_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const erab_release_list_bearer_rel_comp_l& erab_release_resp_ies_o::value_c::erab_release_list_bearer_rel_comp() const
{
  assert_choice_type(types::erab_release_list_bearer_rel_comp, type_, "Value");
  return c.get<erab_release_list_bearer_rel_comp_l>();
}
const erab_list_l& erab_release_resp_ies_o::value_c::erab_failed_to_release_list() const
{
  assert_choice_type(types::erab_failed_to_release_list, type_, "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& erab_release_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const user_location_info_s& erab_release_resp_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_s>();
}
const secondary_rat_data_usage_report_list_l&
erab_release_resp_ies_o::value_c::secondary_rat_data_usage_report_list() const
{
  assert_choice_type(types::secondary_rat_data_usage_report_list, type_, "Value");
  return c.get<secondary_rat_data_usage_report_list_l>();
}
void erab_release_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_release_list_bearer_rel_comp:
      j.start_array("E-RABReleaseListBearerRelComp");
      for (const auto& e1 : c.get<erab_release_list_bearer_rel_comp_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_release_list:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_s>().to_json(j);
      break;
    case types::secondary_rat_data_usage_report_list:
      j.start_array("SecondaryRATDataUsageReportList");
      for (const auto& e1 : c.get<secondary_rat_data_usage_report_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_release_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_release_list_bearer_rel_comp:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_release_list_bearer_rel_comp_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_release_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().pack(bref));
      break;
    case types::secondary_rat_data_usage_report_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<secondary_rat_data_usage_report_list_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_release_list_bearer_rel_comp:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_release_list_bearer_rel_comp_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_release_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().unpack(bref));
      break;
    case types::secondary_rat_data_usage_report_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<secondary_rat_data_usage_report_list_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_release_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABReleaseListBearerRelComp",
                                  "E-RABList",
                                  "CriticalityDiagnostics",
                                  "UserLocationInformation",
                                  "SecondaryRATDataUsageReportList"};
  return convert_enum_idx(options, 7, value, "erab_release_resp_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_release_resp_ies_o>;

erab_release_resp_ies_container::erab_release_resp_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_release_list_bearer_rel_comp(69, crit_e::ignore),
  erab_failed_to_release_list(34, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore),
  user_location_info(189, crit_e::ignore),
  secondary_rat_data_usage_report_list(264, crit_e::ignore)
{}
SRSASN_CODE erab_release_resp_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += erab_release_list_bearer_rel_comp_present ? 1 : 0;
  nof_ies += erab_failed_to_release_list_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  nof_ies += user_location_info_present ? 1 : 0;
  nof_ies += secondary_rat_data_usage_report_list_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_release_list_bearer_rel_comp_present) {
    HANDLE_CODE(erab_release_list_bearer_rel_comp.pack(bref));
  }
  if (erab_failed_to_release_list_present) {
    HANDLE_CODE(erab_failed_to_release_list.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (user_location_info_present) {
    HANDLE_CODE(user_location_info.pack(bref));
  }
  if (secondary_rat_data_usage_report_list_present) {
    HANDLE_CODE(secondary_rat_data_usage_report_list.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_resp_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 69: {
        erab_release_list_bearer_rel_comp_present = true;
        erab_release_list_bearer_rel_comp.id      = id;
        HANDLE_CODE(erab_release_list_bearer_rel_comp.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_release_list_bearer_rel_comp.value.unpack(bref));
        break;
      }
      case 34: {
        erab_failed_to_release_list_present = true;
        erab_failed_to_release_list.id      = id;
        HANDLE_CODE(erab_failed_to_release_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_failed_to_release_list.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      case 189: {
        user_location_info_present = true;
        user_location_info.id      = id;
        HANDLE_CODE(user_location_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(user_location_info.value.unpack(bref));
        break;
      }
      case 264: {
        secondary_rat_data_usage_report_list_present = true;
        secondary_rat_data_usage_report_list.id      = id;
        HANDLE_CODE(secondary_rat_data_usage_report_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(secondary_rat_data_usage_report_list.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_release_resp_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_release_list_bearer_rel_comp_present) {
    j.write_fieldname("");
    erab_release_list_bearer_rel_comp.to_json(j);
  }
  if (erab_failed_to_release_list_present) {
    j.write_fieldname("");
    erab_failed_to_release_list.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  if (user_location_info_present) {
    j.write_fieldname("");
    user_location_info.to_json(j);
  }
  if (secondary_rat_data_usage_report_list_present) {
    j.write_fieldname("");
    secondary_rat_data_usage_report_list.to_json(j);
  }
  j.end_obj();
}

// E-RABSetupItemBearerSURes ::= SEQUENCE
SRSASN_CODE erab_setup_item_bearer_su_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_item_bearer_su_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_setup_item_bearer_su_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABSetupItemBearerSUResIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_setup_item_bearer_su_res_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {39};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_setup_item_bearer_su_res_ies_o::is_id_valid(const uint32_t& id)
{
  return 39 == id;
}
crit_e erab_setup_item_bearer_su_res_ies_o::get_crit(const uint32_t& id)
{
  if (id == 39) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_setup_item_bearer_su_res_ies_o::value_c erab_setup_item_bearer_su_res_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 39) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_setup_item_bearer_su_res_ies_o::get_presence(const uint32_t& id)
{
  if (id == 39) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_setup_item_bearer_su_res_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABSetupItemBearerSURes");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_setup_item_bearer_su_res_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_item_bearer_su_res_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_setup_item_bearer_su_res_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABSetupItemBearerSURes"};
  return convert_enum_idx(options, 1, value, "erab_setup_item_bearer_su_res_ies_o::value_c::types");
}

// E-RABSetupItemCtxtSURes ::= SEQUENCE
SRSASN_CODE erab_setup_item_ctxt_su_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_item_ctxt_su_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_setup_item_ctxt_su_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABSetupItemCtxtSUResIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_setup_item_ctxt_su_res_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {50};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_setup_item_ctxt_su_res_ies_o::is_id_valid(const uint32_t& id)
{
  return 50 == id;
}
crit_e erab_setup_item_ctxt_su_res_ies_o::get_crit(const uint32_t& id)
{
  if (id == 50) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_setup_item_ctxt_su_res_ies_o::value_c erab_setup_item_ctxt_su_res_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 50) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_setup_item_ctxt_su_res_ies_o::get_presence(const uint32_t& id)
{
  if (id == 50) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_setup_item_ctxt_su_res_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABSetupItemCtxtSURes");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_setup_item_ctxt_su_res_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_item_ctxt_su_res_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_setup_item_ctxt_su_res_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABSetupItemCtxtSURes"};
  return convert_enum_idx(options, 1, value, "erab_setup_item_ctxt_su_res_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_setup_item_bearer_su_res_ies_o>;

template struct asn1::protocol_ie_single_container_s<erab_setup_item_ctxt_su_res_ies_o>;

// BearerType ::= ENUMERATED
const char* bearer_type_opts::to_string() const
{
  static const char* options[] = {"non-IP"};
  return convert_enum_idx(options, 1, value, "bearer_type_e");
}

// E-RABToBeSetupItemBearerSUReqExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_to_be_setup_item_bearer_su_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {156, 183, 233};
  return map_enum_number(options, 3, idx, "id");
}
bool erab_to_be_setup_item_bearer_su_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {156, 183, 233};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_to_be_setup_item_bearer_su_req_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 156:
      return crit_e::ignore;
    case 183:
      return crit_e::ignore;
    case 233:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c
erab_to_be_setup_item_bearer_su_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 156:
      ret.set(ext_c::types::correlation_id);
      break;
    case 183:
      ret.set(ext_c::types::sipto_correlation_id);
      break;
    case 233:
      ret.set(ext_c::types::bearer_type);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_bearer_su_req_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 156:
      return presence_e::optional;
    case 183:
      return presence_e::optional;
    case 233:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::correlation_id:
      c.destroy<fixed_octstring<4, true> >();
      break;
    case types::sipto_correlation_id:
      c.destroy<fixed_octstring<4, true> >();
      break;
    default:
      break;
  }
}
void erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::correlation_id:
      c.init<fixed_octstring<4, true> >();
      break;
    case types::sipto_correlation_id:
      c.init<fixed_octstring<4, true> >();
      break;
    case types::bearer_type:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::ext_c(
    const erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::correlation_id:
      c.init(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::sipto_correlation_id:
      c.init(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::bearer_type:
      c.init(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::operator=(
    const erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::correlation_id:
      c.set(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::sipto_correlation_id:
      c.set(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::bearer_type:
      c.set(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
  }

  return *this;
}
fixed_octstring<4, true>& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::correlation_id()
{
  assert_choice_type(types::correlation_id, type_, "Extension");
  return c.get<fixed_octstring<4, true> >();
}
fixed_octstring<4, true>& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::sipto_correlation_id()
{
  assert_choice_type(types::sipto_correlation_id, type_, "Extension");
  return c.get<fixed_octstring<4, true> >();
}
bearer_type_e& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::bearer_type()
{
  assert_choice_type(types::bearer_type, type_, "Extension");
  return c.get<bearer_type_e>();
}
const fixed_octstring<4, true>& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::correlation_id() const
{
  assert_choice_type(types::correlation_id, type_, "Extension");
  return c.get<fixed_octstring<4, true> >();
}
const fixed_octstring<4, true>& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::sipto_correlation_id() const
{
  assert_choice_type(types::sipto_correlation_id, type_, "Extension");
  return c.get<fixed_octstring<4, true> >();
}
const bearer_type_e& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::bearer_type() const
{
  assert_choice_type(types::bearer_type, type_, "Extension");
  return c.get<bearer_type_e>();
}
void erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::correlation_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<4, true> >().to_string());
      break;
    case types::sipto_correlation_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<4, true> >().to_string());
      break;
    case types::bearer_type:
      j.write_str("BearerType", "non-IP");
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().pack(bref)));
      break;
    case types::sipto_correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().pack(bref)));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().unpack(bref)));
      break;
    case types::sipto_correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().unpack(bref)));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"OCTET STRING", "OCTET STRING", "BearerType"};
  return convert_enum_idx(options, 3, value, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::types");
}

template struct asn1::protocol_ext_field_s<erab_to_be_setup_item_bearer_su_req_ext_ies_o>;

erab_to_be_setup_item_bearer_su_req_ext_ies_container::erab_to_be_setup_item_bearer_su_req_ext_ies_container() :
  correlation_id(156, crit_e::ignore), sipto_correlation_id(183, crit_e::ignore), bearer_type(233, crit_e::reject)
{}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += correlation_id_present ? 1 : 0;
  nof_ies += sipto_correlation_id_present ? 1 : 0;
  nof_ies += bearer_type_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (correlation_id_present) {
    HANDLE_CODE(correlation_id.pack(bref));
  }
  if (sipto_correlation_id_present) {
    HANDLE_CODE(sipto_correlation_id.pack(bref));
  }
  if (bearer_type_present) {
    HANDLE_CODE(bearer_type.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 156: {
        correlation_id_present = true;
        correlation_id.id      = id;
        HANDLE_CODE(correlation_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(correlation_id.ext.unpack(bref));
        break;
      }
      case 183: {
        sipto_correlation_id_present = true;
        sipto_correlation_id.id      = id;
        HANDLE_CODE(sipto_correlation_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(sipto_correlation_id.ext.unpack(bref));
        break;
      }
      case 233: {
        bearer_type_present = true;
        bearer_type.id      = id;
        HANDLE_CODE(bearer_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(bearer_type.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_bearer_su_req_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (correlation_id_present) {
    j.write_fieldname("");
    correlation_id.to_json(j);
  }
  if (sipto_correlation_id_present) {
    j.write_fieldname("");
    sipto_correlation_id.to_json(j);
  }
  if (bearer_type_present) {
    j.write_fieldname("");
    bearer_type.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemBearerSUReq ::= SEQUENCE
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.pack(bref));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  HANDLE_CODE(nas_pdu.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.unpack(bref));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  HANDLE_CODE(nas_pdu.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_bearer_su_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("e-RABlevelQoSParameters");
  erab_level_qos_params.to_json(j);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  j.write_str("nAS-PDU", nas_pdu.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemBearerSUReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_setup_item_bearer_su_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {17};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_setup_item_bearer_su_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 17 == id;
}
crit_e erab_to_be_setup_item_bearer_su_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 17) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_setup_item_bearer_su_req_ies_o::value_c
erab_to_be_setup_item_bearer_su_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 17) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_bearer_su_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 17) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_setup_item_bearer_su_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSetupItemBearerSUReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_to_be_setup_item_bearer_su_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSetupItemBearerSUReq"};
  return convert_enum_idx(options, 1, value, "erab_to_be_setup_item_bearer_su_req_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_to_be_setup_item_bearer_su_req_ies_o>;

// E-RABSetupRequestIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_setup_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 66, 16};
  return map_enum_number(options, 4, idx, "id");
}
bool erab_setup_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 66, 16};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_setup_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 66:
      return crit_e::reject;
    case 16:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_setup_request_ies_o::value_c erab_setup_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 16:
      ret.set(value_c::types::erab_to_be_setup_list_bearer_su_req);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_setup_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 66:
      return presence_e::optional;
    case 16:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_setup_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      c.destroy<erab_to_be_setup_list_bearer_su_req_l>();
      break;
    default:
      break;
  }
}
void erab_setup_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      c.init<erab_to_be_setup_list_bearer_su_req_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
  }
}
erab_setup_request_ies_o::value_c::value_c(const erab_setup_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      c.init(other.c.get<erab_to_be_setup_list_bearer_su_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
  }
}
erab_setup_request_ies_o::value_c&
erab_setup_request_ies_o::value_c::operator=(const erab_setup_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      c.set(other.c.get<erab_to_be_setup_list_bearer_su_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_setup_request_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_setup_request_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
ue_aggregate_maximum_bitrate_s& erab_setup_request_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type(types::ueaggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_to_be_setup_list_bearer_su_req_l& erab_setup_request_ies_o::value_c::erab_to_be_setup_list_bearer_su_req()
{
  assert_choice_type(types::erab_to_be_setup_list_bearer_su_req, type_, "Value");
  return c.get<erab_to_be_setup_list_bearer_su_req_l>();
}
const uint64_t& erab_setup_request_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_setup_request_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const ue_aggregate_maximum_bitrate_s& erab_setup_request_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type(types::ueaggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_to_be_setup_list_bearer_su_req_l&
erab_setup_request_ies_o::value_c::erab_to_be_setup_list_bearer_su_req() const
{
  assert_choice_type(types::erab_to_be_setup_list_bearer_su_req, type_, "Value");
  return c.get<erab_to_be_setup_list_bearer_su_req_l>();
}
void erab_setup_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      j.start_array("E-RABToBeSetupListBearerSUReq");
      for (const auto& e1 : c.get<erab_to_be_setup_list_bearer_su_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_setup_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_to_be_setup_list_bearer_su_req_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_to_be_setup_list_bearer_su_req_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_setup_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "UEAggregateMaximumBitrate", "E-RABToBeSetupListBearerSUReq"};
  return convert_enum_idx(options, 4, value, "erab_setup_request_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_setup_request_ies_o>;

erab_setup_request_ies_container::erab_setup_request_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_setup_list_bearer_su_req(16, crit_e::reject)
{}
SRSASN_CODE erab_setup_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += ueaggregate_maximum_bitrate_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (ueaggregate_maximum_bitrate_present) {
    HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  }
  HANDLE_CODE(erab_to_be_setup_list_bearer_su_req.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 66: {
        ueaggregate_maximum_bitrate_present = true;
        ueaggregate_maximum_bitrate.id      = id;
        HANDLE_CODE(ueaggregate_maximum_bitrate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ueaggregate_maximum_bitrate.value.unpack(bref));
        break;
      }
      case 16: {
        nof_mandatory_ies--;
        erab_to_be_setup_list_bearer_su_req.id = id;
        HANDLE_CODE(erab_to_be_setup_list_bearer_su_req.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_to_be_setup_list_bearer_su_req.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_setup_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (ueaggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ueaggregate_maximum_bitrate.to_json(j);
  }
  j.write_fieldname("");
  erab_to_be_setup_list_bearer_su_req.to_json(j);
  j.end_obj();
}

// E-RABSetupResponseIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_setup_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 28, 29, 58};
  return map_enum_number(options, 5, idx, "id");
}
bool erab_setup_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 28, 29, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_setup_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 28:
      return crit_e::ignore;
    case 29:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_setup_resp_ies_o::value_c erab_setup_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 28:
      ret.set(value_c::types::erab_setup_list_bearer_su_res);
      break;
    case 29:
      ret.set(value_c::types::erab_failed_to_setup_list_bearer_su_res);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_setup_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 28:
      return presence_e::optional;
    case 29:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void erab_setup_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_setup_list_bearer_su_res:
      c.destroy<erab_setup_list_bearer_su_res_l>();
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void erab_setup_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_setup_list_bearer_su_res:
      c.init<erab_setup_list_bearer_su_res_l>();
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
  }
}
erab_setup_resp_ies_o::value_c::value_c(const erab_setup_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_setup_list_bearer_su_res:
      c.init(other.c.get<erab_setup_list_bearer_su_res_l>());
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
  }
}
erab_setup_resp_ies_o::value_c& erab_setup_resp_ies_o::value_c::operator=(const erab_setup_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_setup_list_bearer_su_res:
      c.set(other.c.get<erab_setup_list_bearer_su_res_l>());
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
  }

  return *this;
}
uint64_t& erab_setup_resp_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_setup_resp_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
erab_setup_list_bearer_su_res_l& erab_setup_resp_ies_o::value_c::erab_setup_list_bearer_su_res()
{
  assert_choice_type(types::erab_setup_list_bearer_su_res, type_, "Value");
  return c.get<erab_setup_list_bearer_su_res_l>();
}
erab_list_l& erab_setup_resp_ies_o::value_c::erab_failed_to_setup_list_bearer_su_res()
{
  assert_choice_type(types::erab_failed_to_setup_list_bearer_su_res, type_, "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& erab_setup_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& erab_setup_resp_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_setup_resp_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const erab_setup_list_bearer_su_res_l& erab_setup_resp_ies_o::value_c::erab_setup_list_bearer_su_res() const
{
  assert_choice_type(types::erab_setup_list_bearer_su_res, type_, "Value");
  return c.get<erab_setup_list_bearer_su_res_l>();
}
const erab_list_l& erab_setup_resp_ies_o::value_c::erab_failed_to_setup_list_bearer_su_res() const
{
  assert_choice_type(types::erab_failed_to_setup_list_bearer_su_res, type_, "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& erab_setup_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void erab_setup_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_setup_list_bearer_su_res:
      j.start_array("E-RABSetupListBearerSURes");
      for (const auto& e1 : c.get<erab_setup_list_bearer_su_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_setup_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_setup_list_bearer_su_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_setup_list_bearer_su_res_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_setup_list_bearer_su_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_setup_list_bearer_su_res_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_setup_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABSetupListBearerSURes",
                                  "E-RABList",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "erab_setup_resp_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<erab_setup_resp_ies_o>;

erab_setup_resp_ies_container::erab_setup_resp_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_setup_list_bearer_su_res(28, crit_e::ignore),
  erab_failed_to_setup_list_bearer_su_res(29, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE erab_setup_resp_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += erab_setup_list_bearer_su_res_present ? 1 : 0;
  nof_ies += erab_failed_to_setup_list_bearer_su_res_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_setup_list_bearer_su_res_present) {
    HANDLE_CODE(erab_setup_list_bearer_su_res.pack(bref));
  }
  if (erab_failed_to_setup_list_bearer_su_res_present) {
    HANDLE_CODE(erab_failed_to_setup_list_bearer_su_res.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_resp_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 28: {
        erab_setup_list_bearer_su_res_present = true;
        erab_setup_list_bearer_su_res.id      = id;
        HANDLE_CODE(erab_setup_list_bearer_su_res.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_setup_list_bearer_su_res.value.unpack(bref));
        break;
      }
      case 29: {
        erab_failed_to_setup_list_bearer_su_res_present = true;
        erab_failed_to_setup_list_bearer_su_res.id      = id;
        HANDLE_CODE(erab_failed_to_setup_list_bearer_su_res.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_failed_to_setup_list_bearer_su_res.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_setup_resp_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_setup_list_bearer_su_res_present) {
    j.write_fieldname("");
    erab_setup_list_bearer_su_res.to_json(j);
  }
  if (erab_failed_to_setup_list_bearer_su_res_present) {
    j.write_fieldname("");
    erab_failed_to_setup_list_bearer_su_res.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemCtxtSUReqExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_to_be_setup_item_ctxt_su_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {156, 183, 233};
  return map_enum_number(options, 3, idx, "id");
}
bool erab_to_be_setup_item_ctxt_su_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {156, 183, 233};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_to_be_setup_item_ctxt_su_req_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 156:
      return crit_e::ignore;
    case 183:
      return crit_e::ignore;
    case 233:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c
erab_to_be_setup_item_ctxt_su_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 156:
      ret.set(ext_c::types::correlation_id);
      break;
    case 183:
      ret.set(ext_c::types::sipto_correlation_id);
      break;
    case 233:
      ret.set(ext_c::types::bearer_type);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_ctxt_su_req_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 156:
      return presence_e::optional;
    case 183:
      return presence_e::optional;
    case 233:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::correlation_id:
      c.destroy<fixed_octstring<4, true> >();
      break;
    case types::sipto_correlation_id:
      c.destroy<fixed_octstring<4, true> >();
      break;
    default:
      break;
  }
}
void erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::correlation_id:
      c.init<fixed_octstring<4, true> >();
      break;
    case types::sipto_correlation_id:
      c.init<fixed_octstring<4, true> >();
      break;
    case types::bearer_type:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::ext_c(
    const erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::correlation_id:
      c.init(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::sipto_correlation_id:
      c.init(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::bearer_type:
      c.init(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::operator=(
    const erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::correlation_id:
      c.set(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::sipto_correlation_id:
      c.set(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::bearer_type:
      c.set(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
  }

  return *this;
}
fixed_octstring<4, true>& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::correlation_id()
{
  assert_choice_type(types::correlation_id, type_, "Extension");
  return c.get<fixed_octstring<4, true> >();
}
fixed_octstring<4, true>& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::sipto_correlation_id()
{
  assert_choice_type(types::sipto_correlation_id, type_, "Extension");
  return c.get<fixed_octstring<4, true> >();
}
bearer_type_e& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::bearer_type()
{
  assert_choice_type(types::bearer_type, type_, "Extension");
  return c.get<bearer_type_e>();
}
const fixed_octstring<4, true>& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::correlation_id() const
{
  assert_choice_type(types::correlation_id, type_, "Extension");
  return c.get<fixed_octstring<4, true> >();
}
const fixed_octstring<4, true>& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::sipto_correlation_id() const
{
  assert_choice_type(types::sipto_correlation_id, type_, "Extension");
  return c.get<fixed_octstring<4, true> >();
}
const bearer_type_e& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::bearer_type() const
{
  assert_choice_type(types::bearer_type, type_, "Extension");
  return c.get<bearer_type_e>();
}
void erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::correlation_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<4, true> >().to_string());
      break;
    case types::sipto_correlation_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<4, true> >().to_string());
      break;
    case types::bearer_type:
      j.write_str("BearerType", "non-IP");
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().pack(bref)));
      break;
    case types::sipto_correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().pack(bref)));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().unpack(bref)));
      break;
    case types::sipto_correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().unpack(bref)));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"OCTET STRING", "OCTET STRING", "BearerType"};
  return convert_enum_idx(options, 3, value, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::types");
}

template struct asn1::protocol_ext_field_s<erab_to_be_setup_item_ctxt_su_req_ext_ies_o>;

erab_to_be_setup_item_ctxt_su_req_ext_ies_container::erab_to_be_setup_item_ctxt_su_req_ext_ies_container() :
  correlation_id(156, crit_e::ignore), sipto_correlation_id(183, crit_e::ignore), bearer_type(233, crit_e::reject)
{}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += correlation_id_present ? 1 : 0;
  nof_ies += sipto_correlation_id_present ? 1 : 0;
  nof_ies += bearer_type_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (correlation_id_present) {
    HANDLE_CODE(correlation_id.pack(bref));
  }
  if (sipto_correlation_id_present) {
    HANDLE_CODE(sipto_correlation_id.pack(bref));
  }
  if (bearer_type_present) {
    HANDLE_CODE(bearer_type.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 156: {
        correlation_id_present = true;
        correlation_id.id      = id;
        HANDLE_CODE(correlation_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(correlation_id.ext.unpack(bref));
        break;
      }
      case 183: {
        sipto_correlation_id_present = true;
        sipto_correlation_id.id      = id;
        HANDLE_CODE(sipto_correlation_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(sipto_correlation_id.ext.unpack(bref));
        break;
      }
      case 233: {
        bearer_type_present = true;
        bearer_type.id      = id;
        HANDLE_CODE(bearer_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(bearer_type.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_ctxt_su_req_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (correlation_id_present) {
    j.write_fieldname("");
    correlation_id.to_json(j);
  }
  if (sipto_correlation_id_present) {
    j.write_fieldname("");
    sipto_correlation_id.to_json(j);
  }
  if (bearer_type_present) {
    j.write_fieldname("");
    bearer_type.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemCtxtSUReq ::= SEQUENCE
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(nas_pdu_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.pack(bref));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(nas_pdu_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.unpack(bref));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_ctxt_su_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("e-RABlevelQoSParameters");
  erab_level_qos_params.to_json(j);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (nas_pdu_present) {
    j.write_str("nAS-PDU", nas_pdu.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemCtxtSUReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_setup_item_ctxt_su_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {52};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_setup_item_ctxt_su_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 52 == id;
}
crit_e erab_to_be_setup_item_ctxt_su_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 52) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_setup_item_ctxt_su_req_ies_o::value_c erab_to_be_setup_item_ctxt_su_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 52) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_ctxt_su_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 52) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSetupItemCtxtSUReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSetupItemCtxtSUReq"};
  return convert_enum_idx(options, 1, value, "erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::types");
}

// Data-Forwarding-Not-Possible ::= ENUMERATED
const char* data_forwarding_not_possible_opts::to_string() const
{
  static const char* options[] = {"data-Forwarding-not-Possible"};
  return convert_enum_idx(options, 1, value, "data_forwarding_not_possible_e");
}

// E-RABToBeSetupItemHOReq-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_to_be_setup_item_ho_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {143, 233};
  return map_enum_number(options, 2, idx, "id");
}
bool erab_to_be_setup_item_ho_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {143, 233};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_to_be_setup_item_ho_req_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 143:
      return crit_e::ignore;
    case 233:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c erab_to_be_setup_item_ho_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 143:
      ret.set(ext_c::types::data_forwarding_not_possible);
      break;
    case 233:
      ret.set(ext_c::types::bearer_type);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_ho_req_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 143:
      return presence_e::optional;
    case 233:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::destroy_() {}
void erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::ext_c(const erab_to_be_setup_item_ho_req_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::data_forwarding_not_possible:
      c.init(other.c.get<data_forwarding_not_possible_e>());
      break;
    case types::bearer_type:
      c.init(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c&
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::operator=(const erab_to_be_setup_item_ho_req_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::data_forwarding_not_possible:
      c.set(other.c.get<data_forwarding_not_possible_e>());
      break;
    case types::bearer_type:
      c.set(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
  }

  return *this;
}
data_forwarding_not_possible_e& erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::data_forwarding_not_possible()
{
  assert_choice_type(types::data_forwarding_not_possible, type_, "Extension");
  return c.get<data_forwarding_not_possible_e>();
}
bearer_type_e& erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::bearer_type()
{
  assert_choice_type(types::bearer_type, type_, "Extension");
  return c.get<bearer_type_e>();
}
const data_forwarding_not_possible_e&
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::data_forwarding_not_possible() const
{
  assert_choice_type(types::data_forwarding_not_possible, type_, "Extension");
  return c.get<data_forwarding_not_possible_e>();
}
const bearer_type_e& erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::bearer_type() const
{
  assert_choice_type(types::bearer_type, type_, "Extension");
  return c.get<bearer_type_e>();
}
void erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::data_forwarding_not_possible:
      j.write_str("Data-Forwarding-Not-Possible", "data-Forwarding-not-Possible");
      break;
    case types::bearer_type:
      j.write_str("BearerType", "non-IP");
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::data_forwarding_not_possible:
      HANDLE_CODE(c.get<data_forwarding_not_possible_e>().pack(bref));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::data_forwarding_not_possible:
      HANDLE_CODE(c.get<data_forwarding_not_possible_e>().unpack(bref));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"Data-Forwarding-Not-Possible", "BearerType"};
  return convert_enum_idx(options, 2, value, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::types");
}

template struct asn1::protocol_ext_field_s<erab_to_be_setup_item_ho_req_ext_ies_o>;

erab_to_be_setup_item_ho_req_ext_ies_container::erab_to_be_setup_item_ho_req_ext_ies_container() :
  data_forwarding_not_possible(143, crit_e::ignore), bearer_type(233, crit_e::reject)
{}
SRSASN_CODE erab_to_be_setup_item_ho_req_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += data_forwarding_not_possible_present ? 1 : 0;
  nof_ies += bearer_type_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (data_forwarding_not_possible_present) {
    HANDLE_CODE(data_forwarding_not_possible.pack(bref));
  }
  if (bearer_type_present) {
    HANDLE_CODE(bearer_type.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 143: {
        data_forwarding_not_possible_present = true;
        data_forwarding_not_possible.id      = id;
        HANDLE_CODE(data_forwarding_not_possible.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(data_forwarding_not_possible.ext.unpack(bref));
        break;
      }
      case 233: {
        bearer_type_present = true;
        bearer_type.id      = id;
        HANDLE_CODE(bearer_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(bearer_type.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_ho_req_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (data_forwarding_not_possible_present) {
    j.write_fieldname("");
    data_forwarding_not_possible.to_json(j);
  }
  if (bearer_type_present) {
    j.write_fieldname("");
    bearer_type.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemHOReq ::= SEQUENCE
SRSASN_CODE erab_to_be_setup_item_ho_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  HANDLE_CODE(erab_level_qos_params.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ho_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  HANDLE_CODE(erab_level_qos_params.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_ho_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  j.write_fieldname("e-RABlevelQosParameters");
  erab_level_qos_params.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemHOReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_setup_item_ho_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {27};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_setup_item_ho_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 27 == id;
}
crit_e erab_to_be_setup_item_ho_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 27) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_setup_item_ho_req_ies_o::value_c erab_to_be_setup_item_ho_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 27) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_ho_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 27) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_setup_item_ho_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSetupItemHOReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_to_be_setup_item_ho_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSetupItemHOReq"};
  return convert_enum_idx(options, 1, value, "erab_to_be_setup_item_ho_req_ies_o::value_c::types");
}

template struct asn1::protocol_ie_single_container_s<erab_to_be_setup_item_ctxt_su_req_ies_o>;

// E-RABToBeSwitchedDLItem ::= SEQUENCE
SRSASN_CODE erab_to_be_switched_dl_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_switched_dl_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_switched_dl_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSwitchedDLItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_switched_dl_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {23};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_switched_dl_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 23 == id;
}
crit_e erab_to_be_switched_dl_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 23) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_switched_dl_item_ies_o::value_c erab_to_be_switched_dl_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 23) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_switched_dl_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 23) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_switched_dl_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSwitchedDLItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_switched_dl_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_switched_dl_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_to_be_switched_dl_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSwitchedDLItem"};
  return convert_enum_idx(options, 1, value, "erab_to_be_switched_dl_item_ies_o::value_c::types");
}

// E-RABToBeSwitchedULItem ::= SEQUENCE
SRSASN_CODE erab_to_be_switched_ul_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_switched_ul_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_switched_ul_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSwitchedULItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_switched_ul_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {94};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_switched_ul_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 94 == id;
}
crit_e erab_to_be_switched_ul_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 94) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_switched_ul_item_ies_o::value_c erab_to_be_switched_ul_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 94) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_switched_ul_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 94) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_switched_ul_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSwitchedULItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_switched_ul_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_switched_ul_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* erab_to_be_switched_ul_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSwitchedULItem"};
  return convert_enum_idx(options, 1, value, "erab_to_be_switched_ul_item_ies_o::value_c::types");
}

// EHRPDMultiSectorLoadReportingResponseItem ::= SEQUENCE
SRSASN_CODE ehrpd_multi_sector_load_report_resp_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(ehrpd_sector_id.pack(bref));
  HANDLE_CODE(ehrpd_sector_load_report_resp.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ehrpd_multi_sector_load_report_resp_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(ehrpd_sector_id.unpack(bref));
  HANDLE_CODE(ehrpd_sector_load_report_resp.unpack(bref));

  return SRSASN_SUCCESS;
}
void ehrpd_multi_sector_load_report_resp_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("eHRPD-Sector-ID", ehrpd_sector_id.to_string());
  j.write_fieldname("eHRPDSectorLoadReportingResponse");
  ehrpd_sector_load_report_resp.to_json(j);
  j.end_obj();
}

// ENBX2ExtTLA ::= SEQUENCE
SRSASN_CODE enbx2_ext_tla_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ipsec_tla_present, 1));
  HANDLE_CODE(bref.pack(gtptl_aa_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (ipsec_tla_present) {
    HANDLE_CODE(ipsec_tla.pack(bref));
  }
  if (gtptl_aa_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, gtptl_aa, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enbx2_ext_tla_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ipsec_tla_present, 1));
  HANDLE_CODE(bref.unpack(gtptl_aa_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (ipsec_tla_present) {
    HANDLE_CODE(ipsec_tla.unpack(bref));
  }
  if (gtptl_aa_present) {
    HANDLE_CODE(unpack_dyn_seq_of(gtptl_aa, bref, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void enbx2_ext_tla_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ipsec_tla_present) {
    j.write_str("iPsecTLA", ipsec_tla.to_string());
  }
  if (gtptl_aa_present) {
    j.start_array("gTPTLAa");
    for (const auto& e1 : gtptl_aa) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MutingAvailabilityIndication ::= ENUMERATED
const char* muting_availability_ind_opts::to_string() const
{
  static const char* options[] = {"available", "unavailable"};
  return convert_enum_idx(options, 2, value, "muting_availability_ind_e");
}

// RLFReportInformation ::= SEQUENCE
SRSASN_CODE rlf_report_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ue_rlf_report_container_for_extended_bands_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ue_rlf_report_container.pack(bref));
  if (ue_rlf_report_container_for_extended_bands_present) {
    HANDLE_CODE(ue_rlf_report_container_for_extended_bands.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE rlf_report_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ue_rlf_report_container_for_extended_bands_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ue_rlf_report_container.unpack(bref));
  if (ue_rlf_report_container_for_extended_bands_present) {
    HANDLE_CODE(ue_rlf_report_container_for_extended_bands.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void rlf_report_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("uE-RLF-Report-Container", ue_rlf_report_container.to_string());
  if (ue_rlf_report_container_for_extended_bands_present) {
    j.write_str("uE-RLF-Report-Container-for-extended-bands", ue_rlf_report_container_for_extended_bands.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SynchronisationStatus ::= ENUMERATED
const char* synchronisation_status_opts::to_string() const
{
  static const char* options[] = {"synchronous", "asynchronous"};
  return convert_enum_idx(options, 2, value, "synchronisation_status_e");
}

// TimeSynchronisationInfo-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t time_synchronisation_info_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {207};
  return map_enum_number(options, 1, idx, "id");
}
bool time_synchronisation_info_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 207 == id;
}
crit_e time_synchronisation_info_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 207) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
time_synchronisation_info_ext_ies_o::ext_c time_synchronisation_info_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 207) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e time_synchronisation_info_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 207) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void time_synchronisation_info_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("MutingAvailabilityIndication", c.to_string());
  j.end_obj();
}
SRSASN_CODE time_synchronisation_info_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE time_synchronisation_info_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* time_synchronisation_info_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"MutingAvailabilityIndication"};
  return convert_enum_idx(options, 1, value, "time_synchronisation_info_ext_ies_o::ext_c::types");
}

// Global-en-gNB-ID ::= SEQUENCE
SRSASN_CODE global_en_g_nb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(en_g_nb_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE global_en_g_nb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(en_g_nb_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void global_en_g_nb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("en-gNB-ID", en_g_nb_id.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MutingPatternInformation ::= SEQUENCE
SRSASN_CODE muting_pattern_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(muting_pattern_offset_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(muting_pattern_period.pack(bref));
  if (muting_pattern_offset_present) {
    HANDLE_CODE(pack_integer(bref, muting_pattern_offset, (uint16_t)0u, (uint16_t)10239u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE muting_pattern_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(muting_pattern_offset_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(muting_pattern_period.unpack(bref));
  if (muting_pattern_offset_present) {
    HANDLE_CODE(unpack_integer(muting_pattern_offset, bref, (uint16_t)0u, (uint16_t)10239u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void muting_pattern_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("muting-pattern-period", muting_pattern_period.to_string());
  if (muting_pattern_offset_present) {
    j.write_int("muting-pattern-offset", muting_pattern_offset);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

const char* muting_pattern_info_s::muting_pattern_period_opts::to_string() const
{
  static const char* options[] = {"ms0", "ms1280", "ms2560", "ms5120", "ms10240"};
  return convert_enum_idx(options, 5, value, "muting_pattern_info_s::muting_pattern_period_e_");
}
uint16_t muting_pattern_info_s::muting_pattern_period_opts::to_number() const
{
  static const uint16_t options[] = {0, 1280, 2560, 5120, 10240};
  return map_enum_number(options, 5, value, "muting_pattern_info_s::muting_pattern_period_e_");
}

// SONInformationReport ::= CHOICE
void son_info_report_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("rLFReportInformation");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE son_info_report_c::pack(bit_ref& bref) const
{
  pack_enum(bref, type());
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_report_c::unpack(cbit_ref& bref)
{
  types e;
  unpack_enum(e, bref);
  if (e != type()) {
    log_invalid_choice_id(e, "son_info_report_c");
    return SRSASN_ERROR_DECODE_FAIL;
  }
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* son_info_report_c::types_opts::to_string() const
{
  static const char* options[] = {"rLFReportInformation"};
  return convert_enum_idx(options, 1, value, "son_info_report_c::types");
}

// TimeSynchronisationInfo ::= SEQUENCE
SRSASN_CODE time_synchronisation_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, stratum_level, (uint8_t)0u, (uint8_t)3u, true, true));
  HANDLE_CODE(synchronisation_status.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE time_synchronisation_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(stratum_level, bref, (uint8_t)0u, (uint8_t)3u, true, true));
  HANDLE_CODE(synchronisation_status.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void time_synchronisation_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("stratumLevel", stratum_level);
  j.write_str("synchronisationStatus", synchronisation_status.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// X2TNLConfigurationInfo-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t x2_tnl_cfg_info_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {153, 193};
  return map_enum_number(options, 2, idx, "id");
}
bool x2_tnl_cfg_info_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {153, 193};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e x2_tnl_cfg_info_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 153:
      return crit_e::ignore;
    case 193:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
x2_tnl_cfg_info_ext_ies_o::ext_c x2_tnl_cfg_info_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 153:
      ret.set(ext_c::types::enbx2_extended_transport_layer_addresses);
      break;
    case 193:
      ret.set(ext_c::types::enb_indirect_x2_transport_layer_addresses);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e x2_tnl_cfg_info_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 153:
      return presence_e::optional;
    case 193:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void x2_tnl_cfg_info_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      c.destroy<enbx2_ext_tlas_l>();
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      c.destroy<enb_indirect_x2_transport_layer_addresses_l>();
      break;
    default:
      break;
  }
}
void x2_tnl_cfg_info_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      c.init<enbx2_ext_tlas_l>();
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      c.init<enb_indirect_x2_transport_layer_addresses_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
  }
}
x2_tnl_cfg_info_ext_ies_o::ext_c::ext_c(const x2_tnl_cfg_info_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      c.init(other.c.get<enbx2_ext_tlas_l>());
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      c.init(other.c.get<enb_indirect_x2_transport_layer_addresses_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
  }
}
x2_tnl_cfg_info_ext_ies_o::ext_c&
x2_tnl_cfg_info_ext_ies_o::ext_c::operator=(const x2_tnl_cfg_info_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      c.set(other.c.get<enbx2_ext_tlas_l>());
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      c.set(other.c.get<enb_indirect_x2_transport_layer_addresses_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
  }

  return *this;
}
enbx2_ext_tlas_l& x2_tnl_cfg_info_ext_ies_o::ext_c::enbx2_extended_transport_layer_addresses()
{
  assert_choice_type(types::enbx2_extended_transport_layer_addresses, type_, "Extension");
  return c.get<enbx2_ext_tlas_l>();
}
enb_indirect_x2_transport_layer_addresses_l&
x2_tnl_cfg_info_ext_ies_o::ext_c::enb_indirect_x2_transport_layer_addresses()
{
  assert_choice_type(types::enb_indirect_x2_transport_layer_addresses, type_, "Extension");
  return c.get<enb_indirect_x2_transport_layer_addresses_l>();
}
const enbx2_ext_tlas_l& x2_tnl_cfg_info_ext_ies_o::ext_c::enbx2_extended_transport_layer_addresses() const
{
  assert_choice_type(types::enbx2_extended_transport_layer_addresses, type_, "Extension");
  return c.get<enbx2_ext_tlas_l>();
}
const enb_indirect_x2_transport_layer_addresses_l&
x2_tnl_cfg_info_ext_ies_o::ext_c::enb_indirect_x2_transport_layer_addresses() const
{
  assert_choice_type(types::enb_indirect_x2_transport_layer_addresses, type_, "Extension");
  return c.get<enb_indirect_x2_transport_layer_addresses_l>();
}
void x2_tnl_cfg_info_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      j.start_array("ENBX2ExtTLAs");
      for (const auto& e1 : c.get<enbx2_ext_tlas_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      j.start_array("ENBIndirectX2TransportLayerAddresses");
      for (const auto& e1 : c.get<enb_indirect_x2_transport_layer_addresses_l>()) {
        j.write_str(e1.to_string());
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE x2_tnl_cfg_info_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<enbx2_ext_tlas_l>(), 1, 16, true));
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<enb_indirect_x2_transport_layer_addresses_l>(), 1, 2, true));
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE x2_tnl_cfg_info_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<enbx2_ext_tlas_l>(), bref, 1, 16, true));
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<enb_indirect_x2_transport_layer_addresses_l>(), bref, 1, 2, true));
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* x2_tnl_cfg_info_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"ENBX2ExtTLAs", "ENBIndirectX2TransportLayerAddresses"};
  return convert_enum_idx(options, 2, value, "x2_tnl_cfg_info_ext_ies_o::ext_c::types");
}

// EN-DCSONeNBIdentification ::= SEQUENCE
SRSASN_CODE en_dcso_nenb_identif_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(globalenbid.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE en_dcso_nenb_identif_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(globalenbid.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void en_dcso_nenb_identif_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("globaleNBID");
  globalenbid.to_json(j);
  j.write_fieldname("selectedTAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EN-DCSONengNBIdentification ::= SEQUENCE
SRSASN_CODE en_dcso_neng_nb_identif_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(globaleng_nbid.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE en_dcso_neng_nb_identif_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(globaleng_nbid.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void en_dcso_neng_nb_identif_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("globalengNBID");
  globaleng_nbid.to_json(j);
  j.write_fieldname("selectedTAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// FiveGSTAI ::= SEQUENCE
SRSASN_CODE five_gstai_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(five_gstac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE five_gstai_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(five_gstac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void five_gstai_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("fiveGSTAC", five_gstac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SONInformation-ExtensionIE ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t son_info_ext_ie_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {206};
  return map_enum_number(options, 1, idx, "id");
}
bool son_info_ext_ie_o::is_id_valid(const uint32_t& id)
{
  return 206 == id;
}
crit_e son_info_ext_ie_o::get_crit(const uint32_t& id)
{
  if (id == 206) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
son_info_ext_ie_o::value_c son_info_ext_ie_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 206) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e son_info_ext_ie_o::get_presence(const uint32_t& id)
{
  if (id == 206) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void son_info_ext_ie_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("SONInformationReport");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE son_info_ext_ie_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_ext_ie_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* son_info_ext_ie_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"SONInformationReport"};
  return convert_enum_idx(options, 1, value, "son_info_ext_ie_o::value_c::types");
}

// SONInformationReply-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t son_info_reply_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {149};
  return map_enum_number(options, 1, idx, "id");
}
bool son_info_reply_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 149 == id;
}
crit_e son_info_reply_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 149) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
son_info_reply_ext_ies_o::ext_c son_info_reply_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 149) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e son_info_reply_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 149) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void son_info_reply_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("TimeSynchronisationInfo");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE son_info_reply_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_reply_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* son_info_reply_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"TimeSynchronisationInfo"};
  return convert_enum_idx(options, 1, value, "son_info_reply_ext_ies_o::ext_c::types");
}

template struct asn1::protocol_ext_field_s<x2_tnl_cfg_info_ext_ies_o>;

x2_tnl_cfg_info_ext_ies_container::x2_tnl_cfg_info_ext_ies_container() :
  enbx2_extended_transport_layer_addresses(153, crit_e::ignore),
  enb_indirect_x2_transport_layer_addresses(193, crit_e::ignore)
{}
SRSASN_CODE x2_tnl_cfg_info_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += enbx2_extended_transport_layer_addresses_present ? 1 : 0;
  nof_ies += enb_indirect_x2_transport_layer_addresses_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (enbx2_extended_transport_layer_addresses_present) {
    HANDLE_CODE(enbx2_extended_transport_layer_addresses.pack(bref));
  }
  if (enb_indirect_x2_transport_layer_addresses_present) {
    HANDLE_CODE(enb_indirect_x2_transport_layer_addresses.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE x2_tnl_cfg_info_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 153: {
        enbx2_extended_transport_layer_addresses_present = true;
        enbx2_extended_transport_layer_addresses.id      = id;
        HANDLE_CODE(enbx2_extended_transport_layer_addresses.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enbx2_extended_transport_layer_addresses.ext.unpack(bref));
        break;
      }
      case 193: {
        enb_indirect_x2_transport_layer_addresses_present = true;
        enb_indirect_x2_transport_layer_addresses.id      = id;
        HANDLE_CODE(enb_indirect_x2_transport_layer_addresses.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_indirect_x2_transport_layer_addresses.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void x2_tnl_cfg_info_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (enbx2_extended_transport_layer_addresses_present) {
    j.write_fieldname("");
    enbx2_extended_transport_layer_addresses.to_json(j);
  }
  if (enb_indirect_x2_transport_layer_addresses_present) {
    j.write_fieldname("");
    enb_indirect_x2_transport_layer_addresses.to_json(j);
  }
  j.end_obj();
}

// X2TNLConfigurationInfo ::= SEQUENCE
SRSASN_CODE x2_tnl_cfg_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, enbx2_transport_layer_addresses, 1, 2, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE x2_tnl_cfg_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(enbx2_transport_layer_addresses, bref, 1, 2, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void x2_tnl_cfg_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("eNBX2TransportLayerAddresses");
  for (const auto& e1 : enbx2_transport_layer_addresses) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EN-DCTransferTypeReply ::= SEQUENCE
SRSASN_CODE en_dc_transfer_type_reply_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(sourceeng_nb.pack(bref));
  HANDLE_CODE(targetenb.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE en_dc_transfer_type_reply_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(sourceeng_nb.unpack(bref));
  HANDLE_CODE(targetenb.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void en_dc_transfer_type_reply_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("sourceengNB");
  sourceeng_nb.to_json(j);
  j.write_fieldname("targeteNB");
  targetenb.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EN-DCTransferTypeRequest ::= SEQUENCE
SRSASN_CODE en_dc_transfer_type_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(targetenb_present, 1));
  HANDLE_CODE(bref.pack(associated_tai_present, 1));
  HANDLE_CODE(bref.pack(broadcast5_gstai_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(sourceenb.pack(bref));
  HANDLE_CODE(targeteng_nb.pack(bref));
  if (targetenb_present) {
    HANDLE_CODE(targetenb.pack(bref));
  }
  if (associated_tai_present) {
    HANDLE_CODE(associated_tai.pack(bref));
  }
  if (broadcast5_gstai_present) {
    HANDLE_CODE(broadcast5_gstai.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE en_dc_transfer_type_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(targetenb_present, 1));
  HANDLE_CODE(bref.unpack(associated_tai_present, 1));
  HANDLE_CODE(bref.unpack(broadcast5_gstai_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(sourceenb.unpack(bref));
  HANDLE_CODE(targeteng_nb.unpack(bref));
  if (targetenb_present) {
    HANDLE_CODE(targetenb.unpack(bref));
  }
  if (associated_tai_present) {
    HANDLE_CODE(associated_tai.unpack(bref));
  }
  if (broadcast5_gstai_present) {
    HANDLE_CODE(broadcast5_gstai.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void en_dc_transfer_type_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("sourceeNB");
  sourceenb.to_json(j);
  j.write_fieldname("targetengNB");
  targeteng_nb.to_json(j);
  if (targetenb_present) {
    j.write_fieldname("targeteNB");
    targetenb.to_json(j);
  }
  if (associated_tai_present) {
    j.write_fieldname("associatedTAI");
    associated_tai.to_json(j);
  }
  if (broadcast5_gstai_present) {
    j.write_fieldname("broadcast5GSTAI");
    broadcast5_gstai.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::protocol_ie_single_container_s<son_info_ext_ie_o>;

// SONInformationReply ::= SEQUENCE
SRSASN_CODE son_info_reply_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(x2_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (x2_tnl_cfg_info_present) {
    HANDLE_CODE(x2_tnl_cfg_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_reply_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(x2_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (x2_tnl_cfg_info_present) {
    HANDLE_CODE(x2_tnl_cfg_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void son_info_reply_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (x2_tnl_cfg_info_present) {
    j.write_fieldname("x2TNLConfigurationInfo");
    x2_tnl_cfg_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// SONInformationRequest ::= ENUMERATED
const char* son_info_request_opts::to_string() const
{
  static const char* options[] = {
      "x2TNL-Configuration-Info", "time-Synchronisation-Info", "activate-Muting", "deactivate-Muting"};
  return convert_enum_idx(options, 4, value, "son_info_request_e");
}
uint8_t son_info_request_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "son_info_request_e");
}

// EN-DCSONTransferType ::= CHOICE
void en_dcson_transfer_type_c::destroy_()
{
  switch (type_) {
    case types::request:
      c.destroy<en_dc_transfer_type_request_s>();
      break;
    case types::reply:
      c.destroy<en_dc_transfer_type_reply_s>();
      break;
    default:
      break;
  }
}
void en_dcson_transfer_type_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::request:
      c.init<en_dc_transfer_type_request_s>();
      break;
    case types::reply:
      c.init<en_dc_transfer_type_reply_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "en_dcson_transfer_type_c");
  }
}
en_dcson_transfer_type_c::en_dcson_transfer_type_c(const en_dcson_transfer_type_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::request:
      c.init(other.c.get<en_dc_transfer_type_request_s>());
      break;
    case types::reply:
      c.init(other.c.get<en_dc_transfer_type_reply_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "en_dcson_transfer_type_c");
  }
}
en_dcson_transfer_type_c& en_dcson_transfer_type_c::operator=(const en_dcson_transfer_type_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::request:
      c.set(other.c.get<en_dc_transfer_type_request_s>());
      break;
    case types::reply:
      c.set(other.c.get<en_dc_transfer_type_reply_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "en_dcson_transfer_type_c");
  }

  return *this;
}
en_dc_transfer_type_request_s& en_dcson_transfer_type_c::set_request()
{
  set(types::request);
  return c.get<en_dc_transfer_type_request_s>();
}
en_dc_transfer_type_reply_s& en_dcson_transfer_type_c::set_reply()
{
  set(types::reply);
  return c.get<en_dc_transfer_type_reply_s>();
}
void en_dcson_transfer_type_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::request:
      j.write_fieldname("request");
      c.get<en_dc_transfer_type_request_s>().to_json(j);
      break;
    case types::reply:
      j.write_fieldname("reply");
      c.get<en_dc_transfer_type_reply_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "en_dcson_transfer_type_c");
  }
  j.end_obj();
}
SRSASN_CODE en_dcson_transfer_type_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::request:
      HANDLE_CODE(c.get<en_dc_transfer_type_request_s>().pack(bref));
      break;
    case types::reply:
      HANDLE_CODE(c.get<en_dc_transfer_type_reply_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "en_dcson_transfer_type_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE en_dcson_transfer_type_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::request:
      HANDLE_CODE(c.get<en_dc_transfer_type_request_s>().unpack(bref));
      break;
    case types::reply:
      HANDLE_CODE(c.get<en_dc_transfer_type_reply_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "en_dcson_transfer_type_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* en_dcson_transfer_type_c::types_opts::to_string() const
{
  static const char* options[] = {"request", "reply"};
  return convert_enum_idx(options, 2, value, "en_dcson_transfer_type_c::types");
}

// SONInformation ::= CHOICE
void son_info_c::destroy_()
{
  switch (type_) {
    case types::son_info_reply:
      c.destroy<son_info_reply_s>();
      break;
    case types::son_info_ext:
      c.destroy<protocol_ie_single_container_s<son_info_ext_ie_o> >();
      break;
    default:
      break;
  }
}
void son_info_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::son_info_request:
      break;
    case types::son_info_reply:
      c.init<son_info_reply_s>();
      break;
    case types::son_info_ext:
      c.init<protocol_ie_single_container_s<son_info_ext_ie_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
}
son_info_c::son_info_c(const son_info_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::son_info_request:
      c.init(other.c.get<son_info_request_e>());
      break;
    case types::son_info_reply:
      c.init(other.c.get<son_info_reply_s>());
      break;
    case types::son_info_ext:
      c.init(other.c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
}
son_info_c& son_info_c::operator=(const son_info_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::son_info_request:
      c.set(other.c.get<son_info_request_e>());
      break;
    case types::son_info_reply:
      c.set(other.c.get<son_info_reply_s>());
      break;
    case types::son_info_ext:
      c.set(other.c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }

  return *this;
}
son_info_request_e& son_info_c::set_son_info_request()
{
  set(types::son_info_request);
  return c.get<son_info_request_e>();
}
son_info_reply_s& son_info_c::set_son_info_reply()
{
  set(types::son_info_reply);
  return c.get<son_info_reply_s>();
}
protocol_ie_single_container_s<son_info_ext_ie_o>& son_info_c::set_son_info_ext()
{
  set(types::son_info_ext);
  return c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >();
}
void son_info_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::son_info_request:
      j.write_str("sONInformationRequest", c.get<son_info_request_e>().to_string());
      break;
    case types::son_info_reply:
      j.write_fieldname("sONInformationReply");
      c.get<son_info_reply_s>().to_json(j);
      break;
    case types::son_info_ext:
      j.write_fieldname("sONInformation-Extension");
      c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
  j.end_obj();
}
SRSASN_CODE son_info_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::son_info_request:
      HANDLE_CODE(c.get<son_info_request_e>().pack(bref));
      break;
    case types::son_info_reply:
      HANDLE_CODE(c.get<son_info_reply_s>().pack(bref));
      break;
    case types::son_info_ext: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::son_info_request:
      HANDLE_CODE(c.get<son_info_request_e>().unpack(bref));
      break;
    case types::son_info_reply:
      HANDLE_CODE(c.get<son_info_reply_s>().unpack(bref));
      break;
    case types::son_info_ext: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* son_info_c::types_opts::to_string() const
{
  static const char* options[] = {"sONInformationRequest", "sONInformationReply", "sONInformation-Extension"};
  return convert_enum_idx(options, 3, value, "son_info_c::types");
}

// EN-DCSONConfigurationTransfer ::= SEQUENCE
SRSASN_CODE en_dcson_cfg_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(x2_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(transfertype.pack(bref));
  HANDLE_CODE(son_info.pack(bref));
  if (x2_tnl_cfg_info_present) {
    HANDLE_CODE(x2_tnl_cfg_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE en_dcson_cfg_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(x2_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(transfertype.unpack(bref));
  HANDLE_CODE(son_info.unpack(bref));
  if (x2_tnl_cfg_info_present) {
    HANDLE_CODE(x2_tnl_cfg_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void en_dcson_cfg_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("transfertype");
  transfertype.to_json(j);
  j.write_fieldname("sONInformation");
  son_info.to_json(j);
  if (x2_tnl_cfg_info_present) {
    j.write_fieldname("x2TNLConfigInfo");
    x2_tnl_cfg_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ENB-StatusTransfer-TransparentContainer ::= SEQUENCE
SRSASN_CODE enb_status_transfer_transparent_container_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, bearers_subject_to_status_transfer_list, 1, 256, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_status_transfer_transparent_container_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(bearers_subject_to_status_transfer_list, bref, 1, 256, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void enb_status_transfer_transparent_container_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("bearers-SubjectToStatusTransferList");
  for (const auto& e1 : bearers_subject_to_status_transfer_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// S-TMSI ::= SEQUENCE
SRSASN_CODE s_tmsi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(mmec.pack(bref));
  HANDLE_CODE(m_tmsi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE s_tmsi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(mmec.unpack(bref));
  HANDLE_CODE(m_tmsi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void s_tmsi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("mMEC", mmec.to_string());
  j.write_str("m-TMSI", m_tmsi.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UL-CP-SecurityInformation ::= SEQUENCE
SRSASN_CODE ul_cp_security_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ul_nas_mac.pack(bref));
  HANDLE_CODE(ul_nas_count.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ul_cp_security_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ul_nas_mac.unpack(bref));
  HANDLE_CODE(ul_nas_count.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ul_cp_security_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("ul-NAS-MAC", ul_nas_mac.to_string());
  j.write_str("ul-NAS-Count", ul_nas_count.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ENBCPRelocationIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enbcp_relocation_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {8, 96, 100, 67, 254};
  return map_enum_number(options, 5, idx, "id");
}
bool enbcp_relocation_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {8, 96, 100, 67, 254};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enbcp_relocation_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 8:
      return crit_e::reject;
    case 96:
      return crit_e::reject;
    case 100:
      return crit_e::ignore;
    case 67:
      return crit_e::ignore;
    case 254:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enbcp_relocation_ind_ies_o::value_c enbcp_relocation_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 96:
      ret.set(value_c::types::s_tmsi);
      break;
    case 100:
      ret.set(value_c::types::eutran_cgi);
      break;
    case 67:
      ret.set(value_c::types::tai);
      break;
    case 254:
      ret.set(value_c::types::ul_cp_security_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enbcp_relocation_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 8:
      return presence_e::mandatory;
    case 96:
      return presence_e::mandatory;
    case 100:
      return presence_e::mandatory;
    case 67:
      return presence_e::mandatory;
    case 254:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void enbcp_relocation_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::s_tmsi:
      c.destroy<s_tmsi_s>();
      break;
    case types::eutran_cgi:
      c.destroy<eutran_cgi_s>();
      break;
    case types::tai:
      c.destroy<tai_s>();
      break;
    case types::ul_cp_security_info:
      c.destroy<ul_cp_security_info_s>();
      break;
    default:
      break;
  }
}
void enbcp_relocation_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::enb_ue_s1ap_id:
      break;
    case types::s_tmsi:
      c.init<s_tmsi_s>();
      break;
    case types::eutran_cgi:
      c.init<eutran_cgi_s>();
      break;
    case types::tai:
      c.init<tai_s>();
      break;
    case types::ul_cp_security_info:
      c.init<ul_cp_security_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
  }
}
enbcp_relocation_ind_ies_o::value_c::value_c(const enbcp_relocation_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::s_tmsi:
      c.init(other.c.get<s_tmsi_s>());
      break;
    case types::eutran_cgi:
      c.init(other.c.get<eutran_cgi_s>());
      break;
    case types::tai:
      c.init(other.c.get<tai_s>());
      break;
    case types::ul_cp_security_info:
      c.init(other.c.get<ul_cp_security_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
  }
}
enbcp_relocation_ind_ies_o::value_c&
enbcp_relocation_ind_ies_o::value_c::operator=(const enbcp_relocation_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::s_tmsi:
      c.set(other.c.get<s_tmsi_s>());
      break;
    case types::eutran_cgi:
      c.set(other.c.get<eutran_cgi_s>());
      break;
    case types::tai:
      c.set(other.c.get<tai_s>());
      break;
    case types::ul_cp_security_info:
      c.set(other.c.get<ul_cp_security_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
  }

  return *this;
}
uint32_t& enbcp_relocation_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
s_tmsi_s& enbcp_relocation_ind_ies_o::value_c::s_tmsi()
{
  assert_choice_type(types::s_tmsi, type_, "Value");
  return c.get<s_tmsi_s>();
}
eutran_cgi_s& enbcp_relocation_ind_ies_o::value_c::eutran_cgi()
{
  assert_choice_type(types::eutran_cgi, type_, "Value");
  return c.get<eutran_cgi_s>();
}
tai_s& enbcp_relocation_ind_ies_o::value_c::tai()
{
  assert_choice_type(types::tai, type_, "Value");
  return c.get<tai_s>();
}
ul_cp_security_info_s& enbcp_relocation_ind_ies_o::value_c::ul_cp_security_info()
{
  assert_choice_type(types::ul_cp_security_info, type_, "Value");
  return c.get<ul_cp_security_info_s>();
}
const uint32_t& enbcp_relocation_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const s_tmsi_s& enbcp_relocation_ind_ies_o::value_c::s_tmsi() const
{
  assert_choice_type(types::s_tmsi, type_, "Value");
  return c.get<s_tmsi_s>();
}
const eutran_cgi_s& enbcp_relocation_ind_ies_o::value_c::eutran_cgi() const
{
  assert_choice_type(types::eutran_cgi, type_, "Value");
  return c.get<eutran_cgi_s>();
}
const tai_s& enbcp_relocation_ind_ies_o::value_c::tai() const
{
  assert_choice_type(types::tai, type_, "Value");
  return c.get<tai_s>();
}
const ul_cp_security_info_s& enbcp_relocation_ind_ies_o::value_c::ul_cp_security_info() const
{
  assert_choice_type(types::ul_cp_security_info, type_, "Value");
  return c.get<ul_cp_security_info_s>();
}
void enbcp_relocation_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::s_tmsi:
      j.write_fieldname("S-TMSI");
      c.get<s_tmsi_s>().to_json(j);
      break;
    case types::eutran_cgi:
      j.write_fieldname("EUTRAN-CGI");
      c.get<eutran_cgi_s>().to_json(j);
      break;
    case types::tai:
      j.write_fieldname("TAI");
      c.get<tai_s>().to_json(j);
      break;
    case types::ul_cp_security_info:
      j.write_fieldname("UL-CP-SecurityInformation");
      c.get<ul_cp_security_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enbcp_relocation_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().pack(bref));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().pack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().pack(bref));
      break;
    case types::ul_cp_security_info:
      HANDLE_CODE(c.get<ul_cp_security_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enbcp_relocation_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().unpack(bref));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().unpack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().unpack(bref));
      break;
    case types::ul_cp_security_info:
      HANDLE_CODE(c.get<ul_cp_security_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* enbcp_relocation_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..16777215)", "S-TMSI", "EUTRAN-CGI", "TAI", "UL-CP-SecurityInformation"};
  return convert_enum_idx(options, 5, value, "enbcp_relocation_ind_ies_o::value_c::types");
}
uint8_t enbcp_relocation_ind_ies_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {0};
  return map_enum_number(options, 1, value, "enbcp_relocation_ind_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<enbcp_relocation_ind_ies_o>;

enbcp_relocation_ind_ies_container::enbcp_relocation_ind_ies_container() :
  enb_ue_s1ap_id(8, crit_e::reject),
  s_tmsi(96, crit_e::reject),
  eutran_cgi(100, crit_e::ignore),
  tai(67, crit_e::ignore),
  ul_cp_security_info(254, crit_e::reject)
{}
SRSASN_CODE enbcp_relocation_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 5;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(s_tmsi.pack(bref));
  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(ul_cp_security_info.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enbcp_relocation_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 5;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 96: {
        nof_mandatory_ies--;
        s_tmsi.id = id;
        HANDLE_CODE(s_tmsi.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(s_tmsi.value.unpack(bref));
        break;
      }
      case 100: {
        nof_mandatory_ies--;
        eutran_cgi.id = id;
        HANDLE_CODE(eutran_cgi.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(eutran_cgi.value.unpack(bref));
        break;
      }
      case 67: {
        nof_mandatory_ies--;
        tai.id = id;
        HANDLE_CODE(tai.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(tai.value.unpack(bref));
        break;
      }
      case 254: {
        nof_mandatory_ies--;
        ul_cp_security_info.id = id;
        HANDLE_CODE(ul_cp_security_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ul_cp_security_info.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void enbcp_relocation_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  s_tmsi.to_json(j);
  j.write_fieldname("");
  eutran_cgi.to_json(j);
  j.write_fieldname("");
  tai.to_json(j);
  j.write_fieldname("");
  ul_cp_security_info.to_json(j);
  j.end_obj();
}

// ListeningSubframePattern ::= SEQUENCE
SRSASN_CODE listening_sf_pattern_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pattern_period.pack(bref));
  HANDLE_CODE(pack_integer(bref, pattern_offset, (uint16_t)0u, (uint16_t)10239u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE listening_sf_pattern_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(pattern_period.unpack(bref));
  HANDLE_CODE(unpack_integer(pattern_offset, bref, (uint16_t)0u, (uint16_t)10239u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void listening_sf_pattern_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pattern-period", pattern_period.to_string());
  j.write_int("pattern-offset", pattern_offset);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

const char* listening_sf_pattern_s::pattern_period_opts::to_string() const
{
  static const char* options[] = {"ms1280", "ms2560", "ms5120", "ms10240"};
  return convert_enum_idx(options, 4, value, "listening_sf_pattern_s::pattern_period_e_");
}
uint16_t listening_sf_pattern_s::pattern_period_opts::to_number() const
{
  static const uint16_t options[] = {1280, 2560, 5120, 10240};
  return map_enum_number(options, 4, value, "listening_sf_pattern_s::pattern_period_e_");
}

// SynchronisationInformation ::= SEQUENCE
SRSASN_CODE synchronisation_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(source_stratum_level_present, 1));
  HANDLE_CODE(bref.pack(listening_sf_pattern_present, 1));
  HANDLE_CODE(bref.pack(aggressore_cgi_list_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (source_stratum_level_present) {
    HANDLE_CODE(pack_integer(bref, source_stratum_level, (uint8_t)0u, (uint8_t)3u, true, true));
  }
  if (listening_sf_pattern_present) {
    HANDLE_CODE(listening_sf_pattern.pack(bref));
  }
  if (aggressore_cgi_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, aggressore_cgi_list, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE synchronisation_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(source_stratum_level_present, 1));
  HANDLE_CODE(bref.unpack(listening_sf_pattern_present, 1));
  HANDLE_CODE(bref.unpack(aggressore_cgi_list_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (source_stratum_level_present) {
    HANDLE_CODE(unpack_integer(source_stratum_level, bref, (uint8_t)0u, (uint8_t)3u, true, true));
  }
  if (listening_sf_pattern_present) {
    HANDLE_CODE(listening_sf_pattern.unpack(bref));
  }
  if (aggressore_cgi_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(aggressore_cgi_list, bref, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void synchronisation_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (source_stratum_level_present) {
    j.write_int("sourceStratumLevel", source_stratum_level);
  }
  if (listening_sf_pattern_present) {
    j.write_fieldname("listeningSubframePattern");
    listening_sf_pattern.to_json(j);
  }
  if (aggressore_cgi_list_present) {
    j.start_array("aggressoreCGI-List");
    for (const auto& e1 : aggressore_cgi_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SONConfigurationTransfer-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t son_cfg_transfer_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {152, 209};
  return map_enum_number(options, 2, idx, "id");
}
bool son_cfg_transfer_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {152, 209};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e son_cfg_transfer_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 152:
      return crit_e::ignore;
    case 209:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
son_cfg_transfer_ext_ies_o::ext_c son_cfg_transfer_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 152:
      ret.set(ext_c::types::x2_tnl_cfg_info);
      break;
    case 209:
      ret.set(ext_c::types::synchronisation_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e son_cfg_transfer_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 152:
      return presence_e::conditional;
    case 209:
      return presence_e::conditional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void son_cfg_transfer_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::x2_tnl_cfg_info:
      c.destroy<x2_tnl_cfg_info_s>();
      break;
    case types::synchronisation_info:
      c.destroy<synchronisation_info_s>();
      break;
    default:
      break;
  }
}
void son_cfg_transfer_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::x2_tnl_cfg_info:
      c.init<x2_tnl_cfg_info_s>();
      break;
    case types::synchronisation_info:
      c.init<synchronisation_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
  }
}
son_cfg_transfer_ext_ies_o::ext_c::ext_c(const son_cfg_transfer_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::x2_tnl_cfg_info:
      c.init(other.c.get<x2_tnl_cfg_info_s>());
      break;
    case types::synchronisation_info:
      c.init(other.c.get<synchronisation_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
  }
}
son_cfg_transfer_ext_ies_o::ext_c&
son_cfg_transfer_ext_ies_o::ext_c::operator=(const son_cfg_transfer_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::x2_tnl_cfg_info:
      c.set(other.c.get<x2_tnl_cfg_info_s>());
      break;
    case types::synchronisation_info:
      c.set(other.c.get<synchronisation_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
  }

  return *this;
}
x2_tnl_cfg_info_s& son_cfg_transfer_ext_ies_o::ext_c::x2_tnl_cfg_info()
{
  assert_choice_type(types::x2_tnl_cfg_info, type_, "Extension");
  return c.get<x2_tnl_cfg_info_s>();
}
synchronisation_info_s& son_cfg_transfer_ext_ies_o::ext_c::synchronisation_info()
{
  assert_choice_type(types::synchronisation_info, type_, "Extension");
  return c.get<synchronisation_info_s>();
}
const x2_tnl_cfg_info_s& son_cfg_transfer_ext_ies_o::ext_c::x2_tnl_cfg_info() const
{
  assert_choice_type(types::x2_tnl_cfg_info, type_, "Extension");
  return c.get<x2_tnl_cfg_info_s>();
}
const synchronisation_info_s& son_cfg_transfer_ext_ies_o::ext_c::synchronisation_info() const
{
  assert_choice_type(types::synchronisation_info, type_, "Extension");
  return c.get<synchronisation_info_s>();
}
void son_cfg_transfer_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::x2_tnl_cfg_info:
      j.write_fieldname("X2TNLConfigurationInfo");
      c.get<x2_tnl_cfg_info_s>().to_json(j);
      break;
    case types::synchronisation_info:
      j.write_fieldname("SynchronisationInformation");
      c.get<synchronisation_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE son_cfg_transfer_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::x2_tnl_cfg_info:
      HANDLE_CODE(c.get<x2_tnl_cfg_info_s>().pack(bref));
      break;
    case types::synchronisation_info:
      HANDLE_CODE(c.get<synchronisation_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_cfg_transfer_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::x2_tnl_cfg_info:
      HANDLE_CODE(c.get<x2_tnl_cfg_info_s>().unpack(bref));
      break;
    case types::synchronisation_info:
      HANDLE_CODE(c.get<synchronisation_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* son_cfg_transfer_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"X2TNLConfigurationInfo", "SynchronisationInformation"};
  return convert_enum_idx(options, 2, value, "son_cfg_transfer_ext_ies_o::ext_c::types");
}
uint8_t son_cfg_transfer_ext_ies_o::ext_c::types_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "son_cfg_transfer_ext_ies_o::ext_c::types");
}

// SourceeNB-ID ::= SEQUENCE
SRSASN_CODE sourceenb_id_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sourceenb_id_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sourceenb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-ENB-ID");
  global_enb_id.to_json(j);
  j.write_fieldname("selected-TAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TargeteNB-ID ::= SEQUENCE
SRSASN_CODE targetenb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE targetenb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void targetenb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-ENB-ID");
  global_enb_id.to_json(j);
  j.write_fieldname("selected-TAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::protocol_ext_field_s<son_cfg_transfer_ext_ies_o>;

son_cfg_transfer_ext_ies_container::son_cfg_transfer_ext_ies_container() :
  x2_tnl_cfg_info(152, crit_e::ignore), synchronisation_info(209, crit_e::ignore)
{}
SRSASN_CODE son_cfg_transfer_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += x2_tnl_cfg_info_present ? 1 : 0;
  nof_ies += synchronisation_info_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (x2_tnl_cfg_info_present) {
    HANDLE_CODE(x2_tnl_cfg_info.pack(bref));
  }
  if (synchronisation_info_present) {
    HANDLE_CODE(synchronisation_info.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE son_cfg_transfer_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 152: {
        x2_tnl_cfg_info_present = true;
        x2_tnl_cfg_info.id      = id;
        HANDLE_CODE(x2_tnl_cfg_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(x2_tnl_cfg_info.ext.unpack(bref));
        break;
      }
      case 209: {
        synchronisation_info_present = true;
        synchronisation_info.id      = id;
        HANDLE_CODE(synchronisation_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(synchronisation_info.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void son_cfg_transfer_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (x2_tnl_cfg_info_present) {
    j.write_fieldname("");
    x2_tnl_cfg_info.to_json(j);
  }
  if (synchronisation_info_present) {
    j.write_fieldname("");
    synchronisation_info.to_json(j);
  }
  j.end_obj();
}

// SONConfigurationTransfer ::= SEQUENCE
SRSASN_CODE son_cfg_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(targetenb_id.pack(bref));
  HANDLE_CODE(sourceenb_id.pack(bref));
  HANDLE_CODE(son_info.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE son_cfg_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(targetenb_id.unpack(bref));
  HANDLE_CODE(sourceenb_id.unpack(bref));
  HANDLE_CODE(son_info.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void son_cfg_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("targeteNB-ID");
  targetenb_id.to_json(j);
  j.write_fieldname("sourceeNB-ID");
  sourceenb_id.to_json(j);
  j.write_fieldname("sONInformation");
  son_info.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ENBConfigurationTransferIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_cfg_transfer_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {129, 294};
  return map_enum_number(options, 2, idx, "id");
}
bool enb_cfg_transfer_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {129, 294};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enb_cfg_transfer_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 129:
      return crit_e::ignore;
    case 294:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enb_cfg_transfer_ies_o::value_c enb_cfg_transfer_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 129:
      ret.set(value_c::types::son_cfg_transfer_ect);
      break;
    case 294:
      ret.set(value_c::types::en_dcson_cfg_transfer_ect);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_cfg_transfer_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 129:
      return presence_e::optional;
    case 294:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void enb_cfg_transfer_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::son_cfg_transfer_ect:
      c.destroy<son_cfg_transfer_s>();
      break;
    case types::en_dcson_cfg_transfer_ect:
      c.destroy<en_dcson_cfg_transfer_s>();
      break;
    default:
      break;
  }
}
void enb_cfg_transfer_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::son_cfg_transfer_ect:
      c.init<son_cfg_transfer_s>();
      break;
    case types::en_dcson_cfg_transfer_ect:
      c.init<en_dcson_cfg_transfer_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_transfer_ies_o::value_c");
  }
}
enb_cfg_transfer_ies_o::value_c::value_c(const enb_cfg_transfer_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::son_cfg_transfer_ect:
      c.init(other.c.get<son_cfg_transfer_s>());
      break;
    case types::en_dcson_cfg_transfer_ect:
      c.init(other.c.get<en_dcson_cfg_transfer_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_transfer_ies_o::value_c");
  }
}
enb_cfg_transfer_ies_o::value_c&
enb_cfg_transfer_ies_o::value_c::operator=(const enb_cfg_transfer_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::son_cfg_transfer_ect:
      c.set(other.c.get<son_cfg_transfer_s>());
      break;
    case types::en_dcson_cfg_transfer_ect:
      c.set(other.c.get<en_dcson_cfg_transfer_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_transfer_ies_o::value_c");
  }

  return *this;
}
son_cfg_transfer_s& enb_cfg_transfer_ies_o::value_c::son_cfg_transfer_ect()
{
  assert_choice_type(types::son_cfg_transfer_ect, type_, "Value");
  return c.get<son_cfg_transfer_s>();
}
en_dcson_cfg_transfer_s& enb_cfg_transfer_ies_o::value_c::en_dcson_cfg_transfer_ect()
{
  assert_choice_type(types::en_dcson_cfg_transfer_ect, type_, "Value");
  return c.get<en_dcson_cfg_transfer_s>();
}
const son_cfg_transfer_s& enb_cfg_transfer_ies_o::value_c::son_cfg_transfer_ect() const
{
  assert_choice_type(types::son_cfg_transfer_ect, type_, "Value");
  return c.get<son_cfg_transfer_s>();
}
const en_dcson_cfg_transfer_s& enb_cfg_transfer_ies_o::value_c::en_dcson_cfg_transfer_ect() const
{
  assert_choice_type(types::en_dcson_cfg_transfer_ect, type_, "Value");
  return c.get<en_dcson_cfg_transfer_s>();
}
void enb_cfg_transfer_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::son_cfg_transfer_ect:
      j.write_fieldname("SONConfigurationTransfer");
      c.get<son_cfg_transfer_s>().to_json(j);
      break;
    case types::en_dcson_cfg_transfer_ect:
      j.write_fieldname("EN-DCSONConfigurationTransfer");
      c.get<en_dcson_cfg_transfer_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_transfer_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_cfg_transfer_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::son_cfg_transfer_ect:
      HANDLE_CODE(c.get<son_cfg_transfer_s>().pack(bref));
      break;
    case types::en_dcson_cfg_transfer_ect:
      HANDLE_CODE(c.get<en_dcson_cfg_transfer_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_transfer_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_transfer_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::son_cfg_transfer_ect:
      HANDLE_CODE(c.get<son_cfg_transfer_s>().unpack(bref));
      break;
    case types::en_dcson_cfg_transfer_ect:
      HANDLE_CODE(c.get<en_dcson_cfg_transfer_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_transfer_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* enb_cfg_transfer_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"SONConfigurationTransfer", "EN-DCSONConfigurationTransfer"};
  return convert_enum_idx(options, 2, value, "enb_cfg_transfer_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<enb_cfg_transfer_ies_o>;

enb_cfg_transfer_ies_container::enb_cfg_transfer_ies_container() :
  son_cfg_transfer_ect(129, crit_e::ignore), en_dcson_cfg_transfer_ect(294, crit_e::ignore)
{}
SRSASN_CODE enb_cfg_transfer_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += son_cfg_transfer_ect_present ? 1 : 0;
  nof_ies += en_dcson_cfg_transfer_ect_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (son_cfg_transfer_ect_present) {
    HANDLE_CODE(son_cfg_transfer_ect.pack(bref));
  }
  if (en_dcson_cfg_transfer_ect_present) {
    HANDLE_CODE(en_dcson_cfg_transfer_ect.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_transfer_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 129: {
        son_cfg_transfer_ect_present = true;
        son_cfg_transfer_ect.id      = id;
        HANDLE_CODE(son_cfg_transfer_ect.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(son_cfg_transfer_ect.value.unpack(bref));
        break;
      }
      case 294: {
        en_dcson_cfg_transfer_ect_present = true;
        en_dcson_cfg_transfer_ect.id      = id;
        HANDLE_CODE(en_dcson_cfg_transfer_ect.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(en_dcson_cfg_transfer_ect.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void enb_cfg_transfer_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (son_cfg_transfer_ect_present) {
    j.write_fieldname("");
    son_cfg_transfer_ect.to_json(j);
  }
  if (en_dcson_cfg_transfer_ect_present) {
    j.write_fieldname("");
    en_dcson_cfg_transfer_ect.to_json(j);
  }
  j.end_obj();
}

// NB-IoT-DefaultPagingDRX ::= ENUMERATED
const char* nb_io_t_default_paging_drx_opts::to_string() const
{
  static const char* options[] = {"v128", "v256", "v512", "v1024"};
  return convert_enum_idx(options, 4, value, "nb_io_t_default_paging_drx_e");
}
uint16_t nb_io_t_default_paging_drx_opts::to_number() const
{
  static const uint16_t options[] = {128, 256, 512, 1024};
  return map_enum_number(options, 4, value, "nb_io_t_default_paging_drx_e");
}

// PagingDRX ::= ENUMERATED
const char* paging_drx_opts::to_string() const
{
  static const char* options[] = {"v32", "v64", "v128", "v256"};
  return convert_enum_idx(options, 4, value, "paging_drx_e");
}
uint16_t paging_drx_opts::to_number() const
{
  static const uint16_t options[] = {32, 64, 128, 256};
  return map_enum_number(options, 4, value, "paging_drx_e");
}

// ENBConfigurationUpdateIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_cfg_upd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {60, 64, 128, 137, 234, 292, 293};
  return map_enum_number(options, 7, idx, "id");
}
bool enb_cfg_upd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {60, 64, 128, 137, 234, 292, 293};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enb_cfg_upd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 60:
      return crit_e::ignore;
    case 64:
      return crit_e::reject;
    case 128:
      return crit_e::reject;
    case 137:
      return crit_e::ignore;
    case 234:
      return crit_e::ignore;
    case 292:
      return crit_e::ignore;
    case 293:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enb_cfg_upd_ies_o::value_c enb_cfg_upd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 60:
      ret.set(value_c::types::enbname);
      break;
    case 64:
      ret.set(value_c::types::supported_tas);
      break;
    case 128:
      ret.set(value_c::types::csg_id_list);
      break;
    case 137:
      ret.set(value_c::types::default_paging_drx);
      break;
    case 234:
      ret.set(value_c::types::nb_io_t_default_paging_drx);
      break;
    case 292:
      ret.set(value_c::types::connectedeng_nb_to_add_list);
      break;
    case 293:
      ret.set(value_c::types::connectedeng_nb_to_rem_list);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_cfg_upd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 60:
      return presence_e::optional;
    case 64:
      return presence_e::optional;
    case 128:
      return presence_e::optional;
    case 137:
      return presence_e::optional;
    case 234:
      return presence_e::optional;
    case 292:
      return presence_e::optional;
    case 293:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void enb_cfg_upd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::enbname:
      c.destroy<printable_string<1, 150, true, true> >();
      break;
    case types::supported_tas:
      c.destroy<supported_tas_l>();
      break;
    case types::csg_id_list:
      c.destroy<csg_id_list_l>();
      break;
    case types::connectedeng_nb_to_add_list:
      c.destroy<connectedeng_nb_list_l>();
      break;
    case types::connectedeng_nb_to_rem_list:
      c.destroy<connectedeng_nb_list_l>();
      break;
    default:
      break;
  }
}
void enb_cfg_upd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::enbname:
      c.init<printable_string<1, 150, true, true> >();
      break;
    case types::supported_tas:
      c.init<supported_tas_l>();
      break;
    case types::csg_id_list:
      c.init<csg_id_list_l>();
      break;
    case types::default_paging_drx:
      break;
    case types::nb_io_t_default_paging_drx:
      break;
    case types::connectedeng_nb_to_add_list:
      c.init<connectedeng_nb_list_l>();
      break;
    case types::connectedeng_nb_to_rem_list:
      c.init<connectedeng_nb_list_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
  }
}
enb_cfg_upd_ies_o::value_c::value_c(const enb_cfg_upd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::enbname:
      c.init(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::supported_tas:
      c.init(other.c.get<supported_tas_l>());
      break;
    case types::csg_id_list:
      c.init(other.c.get<csg_id_list_l>());
      break;
    case types::default_paging_drx:
      c.init(other.c.get<paging_drx_e>());
      break;
    case types::nb_io_t_default_paging_drx:
      c.init(other.c.get<nb_io_t_default_paging_drx_e>());
      break;
    case types::connectedeng_nb_to_add_list:
      c.init(other.c.get<connectedeng_nb_list_l>());
      break;
    case types::connectedeng_nb_to_rem_list:
      c.init(other.c.get<connectedeng_nb_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
  }
}
enb_cfg_upd_ies_o::value_c& enb_cfg_upd_ies_o::value_c::operator=(const enb_cfg_upd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::enbname:
      c.set(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::supported_tas:
      c.set(other.c.get<supported_tas_l>());
      break;
    case types::csg_id_list:
      c.set(other.c.get<csg_id_list_l>());
      break;
    case types::default_paging_drx:
      c.set(other.c.get<paging_drx_e>());
      break;
    case types::nb_io_t_default_paging_drx:
      c.set(other.c.get<nb_io_t_default_paging_drx_e>());
      break;
    case types::connectedeng_nb_to_add_list:
      c.set(other.c.get<connectedeng_nb_list_l>());
      break;
    case types::connectedeng_nb_to_rem_list:
      c.set(other.c.get<connectedeng_nb_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
  }

  return *this;
}
printable_string<1, 150, true, true>& enb_cfg_upd_ies_o::value_c::enbname()
{
  assert_choice_type(types::enbname, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
supported_tas_l& enb_cfg_upd_ies_o::value_c::supported_tas()
{
  assert_choice_type(types::supported_tas, type_, "Value");
  return c.get<supported_tas_l>();
}
csg_id_list_l& enb_cfg_upd_ies_o::value_c::csg_id_list()
{
  assert_choice_type(types::csg_id_list, type_, "Value");
  return c.get<csg_id_list_l>();
}
paging_drx_e& enb_cfg_upd_ies_o::value_c::default_paging_drx()
{
  assert_choice_type(types::default_paging_drx, type_, "Value");
  return c.get<paging_drx_e>();
}
nb_io_t_default_paging_drx_e& enb_cfg_upd_ies_o::value_c::nb_io_t_default_paging_drx()
{
  assert_choice_type(types::nb_io_t_default_paging_drx, type_, "Value");
  return c.get<nb_io_t_default_paging_drx_e>();
}
connectedeng_nb_list_l& enb_cfg_upd_ies_o::value_c::connectedeng_nb_to_add_list()
{
  assert_choice_type(types::connectedeng_nb_to_add_list, type_, "Value");
  return c.get<connectedeng_nb_list_l>();
}
connectedeng_nb_list_l& enb_cfg_upd_ies_o::value_c::connectedeng_nb_to_rem_list()
{
  assert_choice_type(types::connectedeng_nb_to_rem_list, type_, "Value");
  return c.get<connectedeng_nb_list_l>();
}
const printable_string<1, 150, true, true>& enb_cfg_upd_ies_o::value_c::enbname() const
{
  assert_choice_type(types::enbname, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
const supported_tas_l& enb_cfg_upd_ies_o::value_c::supported_tas() const
{
  assert_choice_type(types::supported_tas, type_, "Value");
  return c.get<supported_tas_l>();
}
const csg_id_list_l& enb_cfg_upd_ies_o::value_c::csg_id_list() const
{
  assert_choice_type(types::csg_id_list, type_, "Value");
  return c.get<csg_id_list_l>();
}
const paging_drx_e& enb_cfg_upd_ies_o::value_c::default_paging_drx() const
{
  assert_choice_type(types::default_paging_drx, type_, "Value");
  return c.get<paging_drx_e>();
}
const nb_io_t_default_paging_drx_e& enb_cfg_upd_ies_o::value_c::nb_io_t_default_paging_drx() const
{
  assert_choice_type(types::nb_io_t_default_paging_drx, type_, "Value");
  return c.get<nb_io_t_default_paging_drx_e>();
}
const connectedeng_nb_list_l& enb_cfg_upd_ies_o::value_c::connectedeng_nb_to_add_list() const
{
  assert_choice_type(types::connectedeng_nb_to_add_list, type_, "Value");
  return c.get<connectedeng_nb_list_l>();
}
const connectedeng_nb_list_l& enb_cfg_upd_ies_o::value_c::connectedeng_nb_to_rem_list() const
{
  assert_choice_type(types::connectedeng_nb_to_rem_list, type_, "Value");
  return c.get<connectedeng_nb_list_l>();
}
void enb_cfg_upd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::enbname:
      j.write_str("PrintableString", c.get<printable_string<1, 150, true, true> >().to_string());
      break;
    case types::supported_tas:
      j.start_array("SupportedTAs");
      for (const auto& e1 : c.get<supported_tas_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::csg_id_list:
      j.start_array("CSG-IdList");
      for (const auto& e1 : c.get<csg_id_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::default_paging_drx:
      j.write_str("PagingDRX", c.get<paging_drx_e>().to_string());
      break;
    case types::nb_io_t_default_paging_drx:
      j.write_str("NB-IoT-DefaultPagingDRX", c.get<nb_io_t_default_paging_drx_e>().to_string());
      break;
    case types::connectedeng_nb_to_add_list:
      j.start_array("ConnectedengNBList");
      for (const auto& e1 : c.get<connectedeng_nb_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::connectedeng_nb_to_rem_list:
      j.start_array("ConnectedengNBList");
      for (const auto& e1 : c.get<connectedeng_nb_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_cfg_upd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enbname:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().pack(bref)));
      break;
    case types::supported_tas:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<supported_tas_l>(), 1, 256, true));
      break;
    case types::csg_id_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<csg_id_list_l>(), 1, 256, true));
      break;
    case types::default_paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().pack(bref));
      break;
    case types::nb_io_t_default_paging_drx:
      HANDLE_CODE(c.get<nb_io_t_default_paging_drx_e>().pack(bref));
      break;
    case types::connectedeng_nb_to_add_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<connectedeng_nb_list_l>(), 1, 256, true));
      break;
    case types::connectedeng_nb_to_rem_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<connectedeng_nb_list_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enbname:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().unpack(bref)));
      break;
    case types::supported_tas:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<supported_tas_l>(), bref, 1, 256, true));
      break;
    case types::csg_id_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<csg_id_list_l>(), bref, 1, 256, true));
      break;
    case types::default_paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().unpack(bref));
      break;
    case types::nb_io_t_default_paging_drx:
      HANDLE_CODE(c.get<nb_io_t_default_paging_drx_e>().unpack(bref));
      break;
    case types::connectedeng_nb_to_add_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<connectedeng_nb_list_l>(), bref, 1, 256, true));
      break;
    case types::connectedeng_nb_to_rem_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<connectedeng_nb_list_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* enb_cfg_upd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"PrintableString",
                                  "SupportedTAs",
                                  "CSG-IdList",
                                  "PagingDRX",
                                  "NB-IoT-DefaultPagingDRX",
                                  "ConnectedengNBList",
                                  "ConnectedengNBList"};
  return convert_enum_idx(options, 7, value, "enb_cfg_upd_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<enb_cfg_upd_ies_o>;

enb_cfg_upd_ies_container::enb_cfg_upd_ies_container() :
  enbname(60, crit_e::ignore),
  supported_tas(64, crit_e::reject),
  csg_id_list(128, crit_e::reject),
  default_paging_drx(137, crit_e::ignore),
  nb_io_t_default_paging_drx(234, crit_e::ignore),
  connectedeng_nb_to_add_list(292, crit_e::ignore),
  connectedeng_nb_to_rem_list(293, crit_e::ignore)
{}
SRSASN_CODE enb_cfg_upd_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += enbname_present ? 1 : 0;
  nof_ies += supported_tas_present ? 1 : 0;
  nof_ies += csg_id_list_present ? 1 : 0;
  nof_ies += default_paging_drx_present ? 1 : 0;
  nof_ies += nb_io_t_default_paging_drx_present ? 1 : 0;
  nof_ies += connectedeng_nb_to_add_list_present ? 1 : 0;
  nof_ies += connectedeng_nb_to_rem_list_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (enbname_present) {
    HANDLE_CODE(enbname.pack(bref));
  }
  if (supported_tas_present) {
    HANDLE_CODE(supported_tas.pack(bref));
  }
  if (csg_id_list_present) {
    HANDLE_CODE(csg_id_list.pack(bref));
  }
  if (default_paging_drx_present) {
    HANDLE_CODE(default_paging_drx.pack(bref));
  }
  if (nb_io_t_default_paging_drx_present) {
    HANDLE_CODE(nb_io_t_default_paging_drx.pack(bref));
  }
  if (connectedeng_nb_to_add_list_present) {
    HANDLE_CODE(connectedeng_nb_to_add_list.pack(bref));
  }
  if (connectedeng_nb_to_rem_list_present) {
    HANDLE_CODE(connectedeng_nb_to_rem_list.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 60: {
        enbname_present = true;
        enbname.id      = id;
        HANDLE_CODE(enbname.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enbname.value.unpack(bref));
        break;
      }
      case 64: {
        supported_tas_present = true;
        supported_tas.id      = id;
        HANDLE_CODE(supported_tas.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(supported_tas.value.unpack(bref));
        break;
      }
      case 128: {
        csg_id_list_present = true;
        csg_id_list.id      = id;
        HANDLE_CODE(csg_id_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(csg_id_list.value.unpack(bref));
        break;
      }
      case 137: {
        default_paging_drx_present = true;
        default_paging_drx.id      = id;
        HANDLE_CODE(default_paging_drx.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(default_paging_drx.value.unpack(bref));
        break;
      }
      case 234: {
        nb_io_t_default_paging_drx_present = true;
        nb_io_t_default_paging_drx.id      = id;
        HANDLE_CODE(nb_io_t_default_paging_drx.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nb_io_t_default_paging_drx.value.unpack(bref));
        break;
      }
      case 292: {
        connectedeng_nb_to_add_list_present = true;
        connectedeng_nb_to_add_list.id      = id;
        HANDLE_CODE(connectedeng_nb_to_add_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(connectedeng_nb_to_add_list.value.unpack(bref));
        break;
      }
      case 293: {
        connectedeng_nb_to_rem_list_present = true;
        connectedeng_nb_to_rem_list.id      = id;
        HANDLE_CODE(connectedeng_nb_to_rem_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(connectedeng_nb_to_rem_list.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void enb_cfg_upd_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (enbname_present) {
    j.write_fieldname("");
    enbname.to_json(j);
  }
  if (supported_tas_present) {
    j.write_fieldname("");
    supported_tas.to_json(j);
  }
  if (csg_id_list_present) {
    j.write_fieldname("");
    csg_id_list.to_json(j);
  }
  if (default_paging_drx_present) {
    j.write_fieldname("");
    default_paging_drx.to_json(j);
  }
  if (nb_io_t_default_paging_drx_present) {
    j.write_fieldname("");
    nb_io_t_default_paging_drx.to_json(j);
  }
  if (connectedeng_nb_to_add_list_present) {
    j.write_fieldname("");
    connectedeng_nb_to_add_list.to_json(j);
  }
  if (connectedeng_nb_to_rem_list_present) {
    j.write_fieldname("");
    connectedeng_nb_to_rem_list.to_json(j);
  }
  j.end_obj();
}

// ENBConfigurationUpdateAcknowledgeIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_cfg_upd_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {58};
  return map_enum_number(options, 1, idx, "id");
}
bool enb_cfg_upd_ack_ies_o::is_id_valid(const uint32_t& id)
{
  return 58 == id;
}
crit_e enb_cfg_upd_ack_ies_o::get_crit(const uint32_t& id)
{
  if (id == 58) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
enb_cfg_upd_ack_ies_o::value_c enb_cfg_upd_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 58) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_cfg_upd_ack_ies_o::get_presence(const uint32_t& id)
{
  if (id == 58) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void enb_cfg_upd_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("CriticalityDiagnostics");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE enb_cfg_upd_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* enb_cfg_upd_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"CriticalityDiagnostics"};
  return convert_enum_idx(options, 1, value, "enb_cfg_upd_ack_ies_o::value_c::types");
}

// TimeToWait ::= ENUMERATED
const char* time_to_wait_opts::to_string() const
{
  static const char* options[] = {"v1s", "v2s", "v5s", "v10s", "v20s", "v60s"};
  return convert_enum_idx(options, 6, value, "time_to_wait_e");
}
uint8_t time_to_wait_opts::to_number() const
{
  static const uint8_t options[] = {1, 2, 5, 10, 20, 60};
  return map_enum_number(options, 6, value, "time_to_wait_e");
}

// ENBConfigurationUpdateFailureIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_cfg_upd_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {2, 65, 58};
  return map_enum_number(options, 3, idx, "id");
}
bool enb_cfg_upd_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {2, 65, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enb_cfg_upd_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 2:
      return crit_e::ignore;
    case 65:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enb_cfg_upd_fail_ies_o::value_c enb_cfg_upd_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 65:
      ret.set(value_c::types::time_to_wait);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_cfg_upd_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 2:
      return presence_e::mandatory;
    case 65:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void enb_cfg_upd_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void enb_cfg_upd_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cause:
      c.init<cause_c>();
      break;
    case types::time_to_wait:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
  }
}
enb_cfg_upd_fail_ies_o::value_c::value_c(const enb_cfg_upd_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.init(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
  }
}
enb_cfg_upd_fail_ies_o::value_c&
enb_cfg_upd_fail_ies_o::value_c::operator=(const enb_cfg_upd_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.set(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
  }

  return *this;
}
cause_c& enb_cfg_upd_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
time_to_wait_e& enb_cfg_upd_fail_ies_o::value_c::time_to_wait()
{
  assert_choice_type(types::time_to_wait, type_, "Value");
  return c.get<time_to_wait_e>();
}
crit_diagnostics_s& enb_cfg_upd_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const cause_c& enb_cfg_upd_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const time_to_wait_e& enb_cfg_upd_fail_ies_o::value_c::time_to_wait() const
{
  assert_choice_type(types::time_to_wait, type_, "Value");
  return c.get<time_to_wait_e>();
}
const crit_diagnostics_s& enb_cfg_upd_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void enb_cfg_upd_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::time_to_wait:
      j.write_str("TimeToWait", c.get<time_to_wait_e>().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_cfg_upd_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* enb_cfg_upd_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Cause", "TimeToWait", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "enb_cfg_upd_fail_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<enb_cfg_upd_fail_ies_o>;

enb_cfg_upd_fail_ies_container::enb_cfg_upd_fail_ies_container() :
  cause(2, crit_e::ignore), time_to_wait(65, crit_e::ignore), crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE enb_cfg_upd_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 1;
  nof_ies += time_to_wait_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(cause.pack(bref));
  if (time_to_wait_present) {
    HANDLE_CODE(time_to_wait.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 1;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 2: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 65: {
        time_to_wait_present = true;
        time_to_wait.id      = id;
        HANDLE_CODE(time_to_wait.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(time_to_wait.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void enb_cfg_upd_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  cause.to_json(j);
  if (time_to_wait_present) {
    j.write_fieldname("");
    time_to_wait.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// LAI ::= SEQUENCE
SRSASN_CODE lai_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(lac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE lai_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(lac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void lai_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("lAC", lac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// GERAN-Cell-ID ::= SEQUENCE
SRSASN_CODE geran_cell_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(lai.pack(bref));
  HANDLE_CODE(rac.pack(bref));
  HANDLE_CODE(ci.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE geran_cell_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(lai.unpack(bref));
  HANDLE_CODE(rac.unpack(bref));
  HANDLE_CODE(ci.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void geran_cell_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("lAI");
  lai.to_json(j);
  j.write_str("rAC", rac.to_string());
  j.write_str("cI", ci.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TargetRNC-ID ::= SEQUENCE
SRSASN_CODE target_rnc_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(rac_present, 1));
  HANDLE_CODE(bref.pack(extended_rnc_id_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(lai.pack(bref));
  if (rac_present) {
    HANDLE_CODE(rac.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, rnc_id, (uint16_t)0u, (uint16_t)4095u, false, true));
  if (extended_rnc_id_present) {
    HANDLE_CODE(pack_integer(bref, extended_rnc_id, (uint32_t)4096u, (uint32_t)65535u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE target_rnc_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(rac_present, 1));
  HANDLE_CODE(bref.unpack(extended_rnc_id_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(lai.unpack(bref));
  if (rac_present) {
    HANDLE_CODE(rac.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(rnc_id, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  if (extended_rnc_id_present) {
    HANDLE_CODE(unpack_integer(extended_rnc_id, bref, (uint32_t)4096u, (uint32_t)65535u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void target_rnc_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("lAI");
  lai.to_json(j);
  if (rac_present) {
    j.write_str("rAC", rac.to_string());
  }
  j.write_int("rNC-ID", rnc_id);
  if (extended_rnc_id_present) {
    j.write_int("extendedRNC-ID", extended_rnc_id);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RIMRoutingAddress ::= CHOICE
void rim_routing_address_c::destroy_()
{
  switch (type_) {
    case types::geran_cell_id:
      c.destroy<geran_cell_id_s>();
      break;
    case types::target_rnc_id:
      c.destroy<target_rnc_id_s>();
      break;
    case types::ehrpd_sector_id:
      c.destroy<fixed_octstring<16, true> >();
      break;
    default:
      break;
  }
}
void rim_routing_address_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::geran_cell_id:
      c.init<geran_cell_id_s>();
      break;
    case types::target_rnc_id:
      c.init<target_rnc_id_s>();
      break;
    case types::ehrpd_sector_id:
      c.init<fixed_octstring<16, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
  }
}
rim_routing_address_c::rim_routing_address_c(const rim_routing_address_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::geran_cell_id:
      c.init(other.c.get<geran_cell_id_s>());
      break;
    case types::target_rnc_id:
      c.init(other.c.get<target_rnc_id_s>());
      break;
    case types::ehrpd_sector_id:
      c.init(other.c.get<fixed_octstring<16, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
  }
}
rim_routing_address_c& rim_routing_address_c::operator=(const rim_routing_address_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::geran_cell_id:
      c.set(other.c.get<geran_cell_id_s>());
      break;
    case types::target_rnc_id:
      c.set(other.c.get<target_rnc_id_s>());
      break;
    case types::ehrpd_sector_id:
      c.set(other.c.get<fixed_octstring<16, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
  }

  return *this;
}
geran_cell_id_s& rim_routing_address_c::set_geran_cell_id()
{
  set(types::geran_cell_id);
  return c.get<geran_cell_id_s>();
}
target_rnc_id_s& rim_routing_address_c::set_target_rnc_id()
{
  set(types::target_rnc_id);
  return c.get<target_rnc_id_s>();
}
fixed_octstring<16, true>& rim_routing_address_c::set_ehrpd_sector_id()
{
  set(types::ehrpd_sector_id);
  return c.get<fixed_octstring<16, true> >();
}
void rim_routing_address_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::geran_cell_id:
      j.write_fieldname("gERAN-Cell-ID");
      c.get<geran_cell_id_s>().to_json(j);
      break;
    case types::target_rnc_id:
      j.write_fieldname("targetRNC-ID");
      c.get<target_rnc_id_s>().to_json(j);
      break;
    case types::ehrpd_sector_id:
      j.write_str("eHRPD-Sector-ID", c.get<fixed_octstring<16, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
  }
  j.end_obj();
}
SRSASN_CODE rim_routing_address_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::geran_cell_id:
      HANDLE_CODE(c.get<geran_cell_id_s>().pack(bref));
      break;
    case types::target_rnc_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<target_rnc_id_s>().pack(bref));
    } break;
    case types::ehrpd_sector_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_octstring<16, true> >().pack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE rim_routing_address_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::geran_cell_id:
      HANDLE_CODE(c.get<geran_cell_id_s>().unpack(bref));
      break;
    case types::target_rnc_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<target_rnc_id_s>().unpack(bref));
    } break;
    case types::ehrpd_sector_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_octstring<16, true> >().unpack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* rim_routing_address_c::types_opts::to_string() const
{
  static const char* options[] = {"gERAN-Cell-ID", "targetRNC-ID", "eHRPD-Sector-ID"};
  return convert_enum_idx(options, 3, value, "rim_routing_address_c::types");
}

// RIMTransfer ::= SEQUENCE
SRSASN_CODE rim_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(rim_routing_address_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(rim_info.pack(bref));
  if (rim_routing_address_present) {
    HANDLE_CODE(rim_routing_address.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE rim_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(rim_routing_address_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(rim_info.unpack(bref));
  if (rim_routing_address_present) {
    HANDLE_CODE(rim_routing_address.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void rim_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("rIMInformation", rim_info.to_string());
  if (rim_routing_address_present) {
    j.write_fieldname("rIMRoutingAddress");
    rim_routing_address.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Inter-SystemInformationTransferType ::= CHOICE
void inter_sys_info_transfer_type_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("rIMTransfer");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE inter_sys_info_transfer_type_c::pack(bit_ref& bref) const
{
  pack_enum(bref, type());
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_sys_info_transfer_type_c::unpack(cbit_ref& bref)
{
  types e;
  unpack_enum(e, bref);
  if (e != type()) {
    log_invalid_choice_id(e, "inter_sys_info_transfer_type_c");
    return SRSASN_ERROR_DECODE_FAIL;
  }
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* inter_sys_info_transfer_type_c::types_opts::to_string() const
{
  static const char* options[] = {"rIMTransfer"};
  return convert_enum_idx(options, 1, value, "inter_sys_info_transfer_type_c::types");
}

// ENBDirectInformationTransferIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_direct_info_transfer_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {121};
  return map_enum_number(options, 1, idx, "id");
}
bool enb_direct_info_transfer_ies_o::is_id_valid(const uint32_t& id)
{
  return 121 == id;
}
crit_e enb_direct_info_transfer_ies_o::get_crit(const uint32_t& id)
{
  if (id == 121) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
enb_direct_info_transfer_ies_o::value_c enb_direct_info_transfer_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 121) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_direct_info_transfer_ies_o::get_presence(const uint32_t& id)
{
  if (id == 121) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void enb_direct_info_transfer_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("Inter-SystemInformationTransferType");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE enb_direct_info_transfer_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_direct_info_transfer_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* enb_direct_info_transfer_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Inter-SystemInformationTransferType"};
  return convert_enum_idx(options, 1, value, "enb_direct_info_transfer_ies_o::value_c::types");
}

// ENBStatusTransferIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_status_transfer_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 90};
  return map_enum_number(options, 3, idx, "id");
}
bool enb_status_transfer_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 90};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enb_status_transfer_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 90:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enb_status_transfer_ies_o::value_c enb_status_transfer_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 90:
      ret.set(value_c::types::enb_status_transfer_transparent_container);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_status_transfer_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 90:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void enb_status_transfer_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::enb_status_transfer_transparent_container:
      c.destroy<enb_status_transfer_transparent_container_s>();
      break;
    default:
      break;
  }
}
void enb_status_transfer_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::enb_status_transfer_transparent_container:
      c.init<enb_status_transfer_transparent_container_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
  }
}
enb_status_transfer_ies_o::value_c::value_c(const enb_status_transfer_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::enb_status_transfer_transparent_container:
      c.init(other.c.get<enb_status_transfer_transparent_container_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
  }
}
enb_status_transfer_ies_o::value_c&
enb_status_transfer_ies_o::value_c::operator=(const enb_status_transfer_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::enb_status_transfer_transparent_container:
      c.set(other.c.get<enb_status_transfer_transparent_container_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
  }

  return *this;
}
uint64_t& enb_status_transfer_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& enb_status_transfer_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
enb_status_transfer_transparent_container_s&
enb_status_transfer_ies_o::value_c::enb_status_transfer_transparent_container()
{
  assert_choice_type(types::enb_status_transfer_transparent_container, type_, "Value");
  return c.get<enb_status_transfer_transparent_container_s>();
}
const uint64_t& enb_status_transfer_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& enb_status_transfer_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const enb_status_transfer_transparent_container_s&
enb_status_transfer_ies_o::value_c::enb_status_transfer_transparent_container() const
{
  assert_choice_type(types::enb_status_transfer_transparent_container, type_, "Value");
  return c.get<enb_status_transfer_transparent_container_s>();
}
void enb_status_transfer_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::enb_status_transfer_transparent_container:
      j.write_fieldname("ENB-StatusTransfer-TransparentContainer");
      c.get<enb_status_transfer_transparent_container_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_status_transfer_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::enb_status_transfer_transparent_container:
      HANDLE_CODE(c.get<enb_status_transfer_transparent_container_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_status_transfer_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::enb_status_transfer_transparent_container:
      HANDLE_CODE(c.get<enb_status_transfer_transparent_container_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* enb_status_transfer_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "ENB-StatusTransfer-TransparentContainer"};
  return convert_enum_idx(options, 3, value, "enb_status_transfer_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<enb_status_transfer_ies_o>;

enb_status_transfer_ies_container::enb_status_transfer_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  enb_status_transfer_transparent_container(90, crit_e::reject)
{}
SRSASN_CODE enb_status_transfer_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_status_transfer_transparent_container.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_status_transfer_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 90: {
        nof_mandatory_ies--;
        enb_status_transfer_transparent_container.id = id;
        HANDLE_CODE(enb_status_transfer_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_status_transfer_transparent_container.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void enb_status_transfer_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_status_transfer_transparent_container.to_json(j);
  j.end_obj();
}

// EUTRANResponse ::= SEQUENCE
SRSASN_CODE eutran_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(cell_id.pack(bref));
  HANDLE_CODE(eutra_ncell_load_report_resp.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE eutran_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(cell_id.unpack(bref));
  HANDLE_CODE(eutra_ncell_load_report_resp.unpack(bref));

  return SRSASN_SUCCESS;
}
void eutran_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-ID", cell_id.to_string());
  j.write_fieldname("eUTRANcellLoadReportingResponse");
  eutra_ncell_load_report_resp.to_json(j);
  j.end_obj();
}

// ErrorIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t error_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 2, 58, 96};
  return map_enum_number(options, 5, idx, "id");
}
bool error_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 2, 58, 96};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e error_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 2:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 96:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
error_ind_ies_o::value_c error_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 96:
      ret.set(value_c::types::s_tmsi);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e error_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::optional;
    case 8:
      return presence_e::optional;
    case 2:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 96:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void error_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    case types::s_tmsi:
      c.destroy<s_tmsi_s>();
      break;
    default:
      break;
  }
}
void error_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::s_tmsi:
      c.init<s_tmsi_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
}
error_ind_ies_o::value_c::value_c(const error_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::s_tmsi:
      c.init(other.c.get<s_tmsi_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
}
error_ind_ies_o::value_c& error_ind_ies_o::value_c::operator=(const error_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::s_tmsi:
      c.set(other.c.get<s_tmsi_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }

  return *this;
}
uint64_t& error_ind_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& error_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
cause_c& error_ind_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& error_ind_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
s_tmsi_s& error_ind_ies_o::value_c::s_tmsi()
{
  assert_choice_type(types::s_tmsi, type_, "Value");
  return c.get<s_tmsi_s>();
}
const uint64_t& error_ind_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& error_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const cause_c& error_ind_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& error_ind_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const s_tmsi_s& error_ind_ies_o::value_c::s_tmsi() const
{
  assert_choice_type(types::s_tmsi, type_, "Value");
  return c.get<s_tmsi_s>();
}
void error_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::s_tmsi:
      j.write_fieldname("S-TMSI");
      c.get<s_tmsi_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE error_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE error_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* error_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "Cause", "CriticalityDiagnostics", "S-TMSI"};
  return convert_enum_idx(options, 5, value, "error_ind_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<error_ind_ies_o>;

error_ind_ies_container::error_ind_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  cause(2, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore),
  s_tmsi(96, crit_e::ignore)
{}
SRSASN_CODE error_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += mme_ue_s1ap_id_present ? 1 : 0;
  nof_ies += enb_ue_s1ap_id_present ? 1 : 0;
  nof_ies += cause_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  nof_ies += s_tmsi_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (mme_ue_s1ap_id_present) {
    HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  }
  if (enb_ue_s1ap_id_present) {
    HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  }
  if (cause_present) {
    HANDLE_CODE(cause.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (s_tmsi_present) {
    HANDLE_CODE(s_tmsi.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE error_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        mme_ue_s1ap_id_present = true;
        mme_ue_s1ap_id.id      = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        enb_ue_s1ap_id_present = true;
        enb_ue_s1ap_id.id      = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 2: {
        cause_present = true;
        cause.id      = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      case 96: {
        s_tmsi_present = true;
        s_tmsi.id      = id;
        HANDLE_CODE(s_tmsi.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(s_tmsi.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void error_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (mme_ue_s1ap_id_present) {
    j.write_fieldname("");
    mme_ue_s1ap_id.to_json(j);
  }
  if (enb_ue_s1ap_id_present) {
    j.write_fieldname("");
    enb_ue_s1ap_id.to_json(j);
  }
  if (cause_present) {
    j.write_fieldname("");
    cause.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  if (s_tmsi_present) {
    j.write_fieldname("");
    s_tmsi.to_json(j);
  }
  j.end_obj();
}

// NumberOfMeasurementReportingLevels ::= ENUMERATED
const char* nof_meas_report_levels_opts::to_string() const
{
  static const char* options[] = {"rl2", "rl3", "rl4", "rl5", "rl10"};
  return convert_enum_idx(options, 5, value, "nof_meas_report_levels_e");
}
uint8_t nof_meas_report_levels_opts::to_number() const
{
  static const uint8_t options[] = {2, 3, 4, 5, 10};
  return map_enum_number(options, 5, value, "nof_meas_report_levels_e");
}

// EventTriggeredCellLoadReportingRequest ::= SEQUENCE
SRSASN_CODE event_triggered_cell_load_report_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(nof_meas_report_levels.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE event_triggered_cell_load_report_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(nof_meas_report_levels.unpack(bref));

  return SRSASN_SUCCESS;
}
void event_triggered_cell_load_report_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("numberOfMeasurementReportingLevels", nof_meas_report_levels.to_string());
  j.end_obj();
}

// OverloadFlag ::= ENUMERATED
const char* overload_flag_opts::to_string() const
{
  static const char* options[] = {"overload"};
  return convert_enum_idx(options, 1, value, "overload_flag_e");
}

// EventTriggeredCellLoadReportingResponse ::= SEQUENCE
SRSASN_CODE event_triggered_cell_load_report_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(overload_flag_present, 1));

  HANDLE_CODE(cell_load_report_resp.pack(bref));
  if (overload_flag_present) {
    HANDLE_CODE(overload_flag.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE event_triggered_cell_load_report_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(overload_flag_present, 1));

  HANDLE_CODE(cell_load_report_resp.unpack(bref));
  if (overload_flag_present) {
    HANDLE_CODE(overload_flag.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void event_triggered_cell_load_report_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cellLoadReportingResponse");
  cell_load_report_resp.to_json(j);
  if (overload_flag_present) {
    j.write_str("overloadFlag", "overload");
  }
  j.end_obj();
}

// SourceOfUEActivityBehaviourInformation ::= ENUMERATED
const char* source_of_ue_activity_behaviour_info_opts::to_string() const
{
  static const char* options[] = {"subscription-information", "statistics"};
  return convert_enum_idx(options, 2, value, "source_of_ue_activity_behaviour_info_e");
}

// ExpectedUEActivityBehaviour ::= SEQUENCE
SRSASN_CODE expected_ue_activity_behaviour_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(expected_activity_period_present, 1));
  HANDLE_CODE(bref.pack(expected_idle_period_present, 1));
  HANDLE_CODE(bref.pack(sourceof_ue_activity_behaviour_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (expected_activity_period_present) {
    HANDLE_CODE(pack_integer(bref, expected_activity_period, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (expected_idle_period_present) {
    HANDLE_CODE(pack_integer(bref, expected_idle_period, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (sourceof_ue_activity_behaviour_info_present) {
    HANDLE_CODE(sourceof_ue_activity_behaviour_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE expected_ue_activity_behaviour_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(expected_activity_period_present, 1));
  HANDLE_CODE(bref.unpack(expected_idle_period_present, 1));
  HANDLE_CODE(bref.unpack(sourceof_ue_activity_behaviour_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (expected_activity_period_present) {
    HANDLE_CODE(unpack_integer(expected_activity_period, bref, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (expected_idle_period_present) {
    HANDLE_CODE(unpack_integer(expected_idle_period, bref, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (sourceof_ue_activity_behaviour_info_present) {
    HANDLE_CODE(sourceof_ue_activity_behaviour_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void expected_ue_activity_behaviour_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (expected_activity_period_present) {
    j.write_int("expectedActivityPeriod", expected_activity_period);
  }
  if (expected_idle_period_present) {
    j.write_int("expectedIdlePeriod", expected_idle_period);
  }
  if (sourceof_ue_activity_behaviour_info_present) {
    j.write_str("sourceofUEActivityBehaviourInformation", sourceof_ue_activity_behaviour_info.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ExpectedHOInterval ::= ENUMERATED
const char* expected_ho_interv_opts::to_string() const
{
  static const char* options[] = {"sec15", "sec30", "sec60", "sec90", "sec120", "sec180", "long-time"};
  return convert_enum_idx(options, 7, value, "expected_ho_interv_e");
}
uint8_t expected_ho_interv_opts::to_number() const
{
  static const uint8_t options[] = {15, 30, 60, 90, 120, 180};
  return map_enum_number(options, 6, value, "expected_ho_interv_e");
}

// ExpectedUEBehaviour ::= SEQUENCE
SRSASN_CODE expected_ue_behaviour_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(expected_activity_present, 1));
  HANDLE_CODE(bref.pack(expected_ho_interv_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (expected_activity_present) {
    HANDLE_CODE(expected_activity.pack(bref));
  }
  if (expected_ho_interv_present) {
    HANDLE_CODE(expected_ho_interv.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE expected_ue_behaviour_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(expected_activity_present, 1));
  HANDLE_CODE(bref.unpack(expected_ho_interv_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (expected_activity_present) {
    HANDLE_CODE(expected_activity.unpack(bref));
  }
  if (expected_ho_interv_present) {
    HANDLE_CODE(expected_ho_interv.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void expected_ue_behaviour_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (expected_activity_present) {
    j.write_fieldname("expectedActivity");
    expected_activity.to_json(j);
  }
  if (expected_ho_interv_present) {
    j.write_str("expectedHOInterval", expected_ho_interv.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TooEarlyInterRATHOReportReportFromEUTRAN ::= SEQUENCE
SRSASN_CODE too_early_inter_ratho_report_report_from_eutran_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mob_info_present, 1));

  HANDLE_CODE(uerlf_report_container.pack(bref));
  if (mob_info_present) {
    HANDLE_CODE(mob_info.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE too_early_inter_ratho_report_report_from_eutran_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mob_info_present, 1));

  HANDLE_CODE(uerlf_report_container.unpack(bref));
  if (mob_info_present) {
    HANDLE_CODE(mob_info.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void too_early_inter_ratho_report_report_from_eutran_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("uERLFReportContainer", uerlf_report_container.to_string());
  if (mob_info_present) {
    j.write_str("mobilityInformation", mob_info.to_string());
  }
  j.end_obj();
}

// FailureEventReport ::= CHOICE
void fail_event_report_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tooEarlyInterRATHOReportFromEUTRAN");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE fail_event_report_c::pack(bit_ref& bref) const
{
  pack_enum(bref, type());
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE fail_event_report_c::unpack(cbit_ref& bref)
{
  types e;
  unpack_enum(e, bref);
  if (e != type()) {
    log_invalid_choice_id(e, "fail_event_report_c");
    return SRSASN_ERROR_DECODE_FAIL;
  }
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* fail_event_report_c::types_opts::to_string() const
{
  static const char* options[] = {"tooEarlyInterRATHOReportFromEUTRAN"};
  return convert_enum_idx(options, 1, value, "fail_event_report_c::types");
}

// HoReportType ::= ENUMERATED
const char* ho_report_type_opts::to_string() const
{
  static const char* options[] = {"unnecessaryhotoanotherrat", "earlyirathandover"};
  return convert_enum_idx(options, 2, value, "ho_report_type_e");
}

// HoType ::= ENUMERATED
const char* ho_type_opts::to_string() const
{
  static const char* options[] = {"ltetoutran", "ltetogeran"};
  return convert_enum_idx(options, 2, value, "ho_type_e");
}

// HOReport ::= SEQUENCE
SRSASN_CODE ho_report_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(ho_type.pack(bref));
  HANDLE_CODE(ho_report_type.pack(bref));
  HANDLE_CODE(hosource_id.pack(bref));
  HANDLE_CODE(ho_target_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, candidate_cell_list, 1, 16, true));

  if (ext) {
    HANDLE_CODE(bref.pack(candidate_pci_list.is_present(), 1));

    if (candidate_pci_list.is_present()) {
      HANDLE_CODE(pack_dyn_seq_of(bref, *candidate_pci_list, 1, 16, true));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_report_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(ho_type.unpack(bref));
  HANDLE_CODE(ho_report_type.unpack(bref));
  HANDLE_CODE(hosource_id.unpack(bref));
  HANDLE_CODE(ho_target_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(candidate_cell_list, bref, 1, 16, true));

  if (ext) {
    bool candidate_pci_list_present;
    HANDLE_CODE(bref.unpack(candidate_pci_list_present, 1));
    candidate_pci_list.set_present(candidate_pci_list_present);

    if (candidate_pci_list.is_present()) {
      HANDLE_CODE(unpack_dyn_seq_of(*candidate_pci_list, bref, 1, 16, true));
    }
  }
  return SRSASN_SUCCESS;
}
void ho_report_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("hoType", ho_type.to_string());
  j.write_str("hoReportType", ho_report_type.to_string());
  j.write_fieldname("hosourceID");
  hosource_id.to_json(j);
  j.write_fieldname("hoTargetID");
  ho_target_id.to_json(j);
  j.start_array("candidateCellList");
  for (const auto& e1 : candidate_cell_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ext) {
    if (candidate_pci_list.is_present()) {
      j.start_array("candidatePCIList");
      for (const auto& e1 : *candidate_pci_list) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// HandoverCancelIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_cancel_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 2};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_cancel_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 2};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cancel_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 2:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cancel_ies_o::value_c ho_cancel_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cancel_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_cancel_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    default:
      break;
  }
}
void ho_cancel_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
}
ho_cancel_ies_o::value_c::value_c(const ho_cancel_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
}
ho_cancel_ies_o::value_c& ho_cancel_ies_o::value_c::operator=(const ho_cancel_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_cancel_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_cancel_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
cause_c& ho_cancel_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const uint64_t& ho_cancel_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_cancel_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const cause_c& ho_cancel_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
void ho_cancel_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cancel_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_cancel_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)", "INTEGER (0..16777215)", "Cause"};
  return convert_enum_idx(options, 3, value, "ho_cancel_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_cancel_ies_o>;

ho_cancel_ies_container::ho_cancel_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject), enb_ue_s1ap_id(8, crit_e::reject), cause(2, crit_e::ignore)
{}
SRSASN_CODE ho_cancel_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 2: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cancel_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.end_obj();
}

// HandoverCancelAcknowledgeIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_cancel_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 58};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_cancel_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cancel_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cancel_ack_ies_o::value_c ho_cancel_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cancel_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_cancel_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_cancel_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
}
ho_cancel_ack_ies_o::value_c::value_c(const ho_cancel_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
}
ho_cancel_ack_ies_o::value_c& ho_cancel_ack_ies_o::value_c::operator=(const ho_cancel_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_cancel_ack_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_cancel_ack_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
crit_diagnostics_s& ho_cancel_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_cancel_ack_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_cancel_ack_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const crit_diagnostics_s& ho_cancel_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_cancel_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cancel_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_cancel_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)", "INTEGER (0..16777215)", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "ho_cancel_ack_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_cancel_ack_ies_o>;

ho_cancel_ack_ies_container::ho_cancel_ack_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore), enb_ue_s1ap_id(8, crit_e::ignore), crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE ho_cancel_ack_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ack_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cancel_ack_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// HandoverType ::= ENUMERATED
const char* handov_type_opts::to_string() const
{
  static const char* options[] = {
      "intralte", "ltetoutran", "ltetogeran", "utrantolte", "gerantolte", "eps-to-5gs", "fivegs-to-eps"};
  return convert_enum_idx(options, 7, value, "handov_type_e");
}
int8_t handov_type_opts::to_number() const
{
  switch (value) {
    case eps_to_minus5gs:
      return -5;
    case fivegs_to_eps:
      return 5;
    default:
      invalid_enum_number(value, "handov_type_e");
  }
  return 0;
}

// HandoverCommandIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_cmd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 1, 135, 12, 13, 123, 139, 58};
  return map_enum_number(options, 9, idx, "id");
}
bool ho_cmd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 1, 135, 12, 13, 123, 139, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cmd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 1:
      return crit_e::reject;
    case 135:
      return crit_e::reject;
    case 12:
      return crit_e::ignore;
    case 13:
      return crit_e::ignore;
    case 123:
      return crit_e::reject;
    case 139:
      return crit_e::reject;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cmd_ies_o::value_c ho_cmd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 1:
      ret.set(value_c::types::handov_type);
      break;
    case 135:
      ret.set(value_c::types::nas_security_paramsfrom_e_utran);
      break;
    case 12:
      ret.set(value_c::types::erab_subjectto_data_forwarding_list);
      break;
    case 13:
      ret.set(value_c::types::erab_to_release_list_ho_cmd);
      break;
    case 123:
      ret.set(value_c::types::target_to_source_transparent_container);
      break;
    case 139:
      ret.set(value_c::types::target_to_source_transparent_container_secondary);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cmd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 1:
      return presence_e::mandatory;
    case 135:
      return presence_e::conditional;
    case 12:
      return presence_e::optional;
    case 13:
      return presence_e::optional;
    case 123:
      return presence_e::mandatory;
    case 139:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_cmd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_security_paramsfrom_e_utran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.destroy<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
      break;
    case types::erab_to_release_list_ho_cmd:
      c.destroy<erab_list_l>();
      break;
    case types::target_to_source_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::target_to_source_transparent_container_secondary:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_cmd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::handov_type:
      break;
    case types::nas_security_paramsfrom_e_utran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.init<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
      break;
    case types::erab_to_release_list_ho_cmd:
      c.init<erab_list_l>();
      break;
    case types::target_to_source_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::target_to_source_transparent_container_secondary:
      c.init<unbounded_octstring<true> >();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
}
ho_cmd_ies_o::value_c::value_c(const ho_cmd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::nas_security_paramsfrom_e_utran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.init(other.c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >());
      break;
    case types::erab_to_release_list_ho_cmd:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::target_to_source_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::target_to_source_transparent_container_secondary:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
}
ho_cmd_ies_o::value_c& ho_cmd_ies_o::value_c::operator=(const ho_cmd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::nas_security_paramsfrom_e_utran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.set(other.c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >());
      break;
    case types::erab_to_release_list_ho_cmd:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::target_to_source_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::target_to_source_transparent_container_secondary:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_cmd_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_cmd_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
handov_type_e& ho_cmd_ies_o::value_c::handov_type()
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
unbounded_octstring<true>& ho_cmd_ies_o::value_c::nas_security_paramsfrom_e_utran()
{
  assert_choice_type(types::nas_security_paramsfrom_e_utran, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
erab_ie_container_list_l<erab_data_forwarding_item_ies_o>& ho_cmd_ies_o::value_c::erab_subjectto_data_forwarding_list()
{
  assert_choice_type(types::erab_subjectto_data_forwarding_list, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
}
erab_list_l& ho_cmd_ies_o::value_c::erab_to_release_list_ho_cmd()
{
  assert_choice_type(types::erab_to_release_list_ho_cmd, type_, "Value");
  return c.get<erab_list_l>();
}
unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container()
{
  assert_choice_type(types::target_to_source_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container_secondary()
{
  assert_choice_type(types::target_to_source_transparent_container_secondary, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
crit_diagnostics_s& ho_cmd_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_cmd_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_cmd_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const handov_type_e& ho_cmd_ies_o::value_c::handov_type() const
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
const unbounded_octstring<true>& ho_cmd_ies_o::value_c::nas_security_paramsfrom_e_utran() const
{
  assert_choice_type(types::nas_security_paramsfrom_e_utran, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const erab_ie_container_list_l<erab_data_forwarding_item_ies_o>&
ho_cmd_ies_o::value_c::erab_subjectto_data_forwarding_list() const
{
  assert_choice_type(types::erab_subjectto_data_forwarding_list, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
}
const erab_list_l& ho_cmd_ies_o::value_c::erab_to_release_list_ho_cmd() const
{
  assert_choice_type(types::erab_to_release_list_ho_cmd, type_, "Value");
  return c.get<erab_list_l>();
}
const unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container() const
{
  assert_choice_type(types::target_to_source_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container_secondary() const
{
  assert_choice_type(types::target_to_source_transparent_container_secondary, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const crit_diagnostics_s& ho_cmd_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_cmd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::nas_security_paramsfrom_e_utran:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::erab_subjectto_data_forwarding_list:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}");
      break;
    case types::erab_to_release_list_ho_cmd:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::target_to_source_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::target_to_source_transparent_container_secondary:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cmd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::nas_security_paramsfrom_e_utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::erab_subjectto_data_forwarding_list:
      HANDLE_CODE(
          pack_dyn_seq_of(bref, c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >(), 1, 256, true));
      break;
    case types::erab_to_release_list_ho_cmd:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::target_to_source_transparent_container_secondary:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::nas_security_paramsfrom_e_utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::erab_subjectto_data_forwarding_list:
      HANDLE_CODE(
          unpack_dyn_seq_of(c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >(), bref, 1, 256, true));
      break;
    case types::erab_to_release_list_ho_cmd:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::target_to_source_transparent_container_secondary:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_cmd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "HandoverType",
                                  "OCTET STRING",
                                  "E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}",
                                  "E-RABList",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 9, value, "ho_cmd_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_cmd_ies_o>;

ho_cmd_ies_container::ho_cmd_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  handov_type(1, crit_e::reject),
  nas_security_paramsfrom_e_utran(135, crit_e::reject),
  erab_subjectto_data_forwarding_list(12, crit_e::ignore),
  erab_to_release_list_ho_cmd(13, crit_e::ignore),
  target_to_source_transparent_container(123, crit_e::reject),
  target_to_source_transparent_container_secondary(139, crit_e::reject),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE ho_cmd_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += nas_security_paramsfrom_e_utran_present ? 1 : 0;
  nof_ies += erab_subjectto_data_forwarding_list_present ? 1 : 0;
  nof_ies += erab_to_release_list_ho_cmd_present ? 1 : 0;
  nof_ies += target_to_source_transparent_container_secondary_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  if (nas_security_paramsfrom_e_utran_present) {
    HANDLE_CODE(nas_security_paramsfrom_e_utran.pack(bref));
  }
  if (erab_subjectto_data_forwarding_list_present) {
    HANDLE_CODE(erab_subjectto_data_forwarding_list.pack(bref));
  }
  if (erab_to_release_list_ho_cmd_present) {
    HANDLE_CODE(erab_to_release_list_ho_cmd.pack(bref));
  }
  HANDLE_CODE(target_to_source_transparent_container.pack(bref));
  if (target_to_source_transparent_container_secondary_present) {
    HANDLE_CODE(target_to_source_transparent_container_secondary.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 1: {
        nof_mandatory_ies--;
        handov_type.id = id;
        HANDLE_CODE(handov_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(handov_type.value.unpack(bref));
        break;
      }
      case 135: {
        nas_security_paramsfrom_e_utran_present = true;
        nas_security_paramsfrom_e_utran.id      = id;
        HANDLE_CODE(nas_security_paramsfrom_e_utran.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nas_security_paramsfrom_e_utran.value.unpack(bref));
        break;
      }
      case 12: {
        erab_subjectto_data_forwarding_list_present = true;
        erab_subjectto_data_forwarding_list.id      = id;
        HANDLE_CODE(erab_subjectto_data_forwarding_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_subjectto_data_forwarding_list.value.unpack(bref));
        break;
      }
      case 13: {
        erab_to_release_list_ho_cmd_present = true;
        erab_to_release_list_ho_cmd.id      = id;
        HANDLE_CODE(erab_to_release_list_ho_cmd.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_to_release_list_ho_cmd.value.unpack(bref));
        break;
      }
      case 123: {
        nof_mandatory_ies--;
        target_to_source_transparent_container.id = id;
        HANDLE_CODE(target_to_source_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(target_to_source_transparent_container.value.unpack(bref));
        break;
      }
      case 139: {
        target_to_source_transparent_container_secondary_present = true;
        target_to_source_transparent_container_secondary.id      = id;
        HANDLE_CODE(target_to_source_transparent_container_secondary.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(target_to_source_transparent_container_secondary.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cmd_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  if (nas_security_paramsfrom_e_utran_present) {
    j.write_fieldname("");
    nas_security_paramsfrom_e_utran.to_json(j);
  }
  if (erab_subjectto_data_forwarding_list_present) {
    j.write_fieldname("");
    erab_subjectto_data_forwarding_list.to_json(j);
  }
  if (erab_to_release_list_ho_cmd_present) {
    j.write_fieldname("");
    erab_to_release_list_ho_cmd.to_json(j);
  }
  j.write_fieldname("");
  target_to_source_transparent_container.to_json(j);
  if (target_to_source_transparent_container_secondary_present) {
    j.write_fieldname("");
    target_to_source_transparent_container_secondary.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// HandoverFailureIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 2, 58};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 2, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 2:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_fail_ies_o::value_c ho_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
}
ho_fail_ies_o::value_c::value_c(const ho_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
}
ho_fail_ies_o::value_c& ho_fail_ies_o::value_c::operator=(const ho_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_fail_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
cause_c& ho_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& ho_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_fail_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const cause_c& ho_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& ho_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "ho_fail_ies_o::value_c::types");
}
uint8_t ho_fail_ies_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {0};
  return map_enum_number(options, 1, value, "ho_fail_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_fail_ies_o>;

ho_fail_ies_container::ho_fail_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore), cause(2, crit_e::ignore), crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE ho_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 2: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// HandoverNotifyIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_notify_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 100, 67, 176, 186, 288};
  return map_enum_number(options, 7, idx, "id");
}
bool ho_notify_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 100, 67, 176, 186, 288};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_notify_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 100:
      return crit_e::ignore;
    case 67:
      return crit_e::ignore;
    case 176:
      return crit_e::ignore;
    case 186:
      return crit_e::ignore;
    case 288:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_notify_ies_o::value_c ho_notify_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 100:
      ret.set(value_c::types::eutran_cgi);
      break;
    case 67:
      ret.set(value_c::types::tai);
      break;
    case 176:
      ret.set(value_c::types::tunnel_info_for_bbf);
      break;
    case 186:
      ret.set(value_c::types::lhn_id);
      break;
    case 288:
      ret.set(value_c::types::ps_cell_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_notify_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 100:
      return presence_e::mandatory;
    case 67:
      return presence_e::mandatory;
    case 176:
      return presence_e::optional;
    case 186:
      return presence_e::optional;
    case 288:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_notify_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::eutran_cgi:
      c.destroy<eutran_cgi_s>();
      break;
    case types::tai:
      c.destroy<tai_s>();
      break;
    case types::tunnel_info_for_bbf:
      c.destroy<tunnel_info_s>();
      break;
    case types::lhn_id:
      c.destroy<bounded_octstring<32, 256, true> >();
      break;
    case types::ps_cell_info:
      c.destroy<ps_cell_info_s>();
      break;
    default:
      break;
  }
}
void ho_notify_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::eutran_cgi:
      c.init<eutran_cgi_s>();
      break;
    case types::tai:
      c.init<tai_s>();
      break;
    case types::tunnel_info_for_bbf:
      c.init<tunnel_info_s>();
      break;
    case types::lhn_id:
      c.init<bounded_octstring<32, 256, true> >();
      break;
    case types::ps_cell_info:
      c.init<ps_cell_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
}
ho_notify_ies_o::value_c::value_c(const ho_notify_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::eutran_cgi:
      c.init(other.c.get<eutran_cgi_s>());
      break;
    case types::tai:
      c.init(other.c.get<tai_s>());
      break;
    case types::tunnel_info_for_bbf:
      c.init(other.c.get<tunnel_info_s>());
      break;
    case types::lhn_id:
      c.init(other.c.get<bounded_octstring<32, 256, true> >());
      break;
    case types::ps_cell_info:
      c.init(other.c.get<ps_cell_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
}
ho_notify_ies_o::value_c& ho_notify_ies_o::value_c::operator=(const ho_notify_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::eutran_cgi:
      c.set(other.c.get<eutran_cgi_s>());
      break;
    case types::tai:
      c.set(other.c.get<tai_s>());
      break;
    case types::tunnel_info_for_bbf:
      c.set(other.c.get<tunnel_info_s>());
      break;
    case types::lhn_id:
      c.set(other.c.get<bounded_octstring<32, 256, true> >());
      break;
    case types::ps_cell_info:
      c.set(other.c.get<ps_cell_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_notify_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_notify_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
eutran_cgi_s& ho_notify_ies_o::value_c::eutran_cgi()
{
  assert_choice_type(types::eutran_cgi, type_, "Value");
  return c.get<eutran_cgi_s>();
}
tai_s& ho_notify_ies_o::value_c::tai()
{
  assert_choice_type(types::tai, type_, "Value");
  return c.get<tai_s>();
}
tunnel_info_s& ho_notify_ies_o::value_c::tunnel_info_for_bbf()
{
  assert_choice_type(types::tunnel_info_for_bbf, type_, "Value");
  return c.get<tunnel_info_s>();
}
bounded_octstring<32, 256, true>& ho_notify_ies_o::value_c::lhn_id()
{
  assert_choice_type(types::lhn_id, type_, "Value");
  return c.get<bounded_octstring<32, 256, true> >();
}
ps_cell_info_s& ho_notify_ies_o::value_c::ps_cell_info()
{
  assert_choice_type(types::ps_cell_info, type_, "Value");
  return c.get<ps_cell_info_s>();
}
const uint64_t& ho_notify_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_notify_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const eutran_cgi_s& ho_notify_ies_o::value_c::eutran_cgi() const
{
  assert_choice_type(types::eutran_cgi, type_, "Value");
  return c.get<eutran_cgi_s>();
}
const tai_s& ho_notify_ies_o::value_c::tai() const
{
  assert_choice_type(types::tai, type_, "Value");
  return c.get<tai_s>();
}
const tunnel_info_s& ho_notify_ies_o::value_c::tunnel_info_for_bbf() const
{
  assert_choice_type(types::tunnel_info_for_bbf, type_, "Value");
  return c.get<tunnel_info_s>();
}
const bounded_octstring<32, 256, true>& ho_notify_ies_o::value_c::lhn_id() const
{
  assert_choice_type(types::lhn_id, type_, "Value");
  return c.get<bounded_octstring<32, 256, true> >();
}
const ps_cell_info_s& ho_notify_ies_o::value_c::ps_cell_info() const
{
  assert_choice_type(types::ps_cell_info, type_, "Value");
  return c.get<ps_cell_info_s>();
}
void ho_notify_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::eutran_cgi:
      j.write_fieldname("EUTRAN-CGI");
      c.get<eutran_cgi_s>().to_json(j);
      break;
    case types::tai:
      j.write_fieldname("TAI");
      c.get<tai_s>().to_json(j);
      break;
    case types::tunnel_info_for_bbf:
      j.write_fieldname("TunnelInformation");
      c.get<tunnel_info_s>().to_json(j);
      break;
    case types::lhn_id:
      j.write_str("OCTET STRING", c.get<bounded_octstring<32, 256, true> >().to_string());
      break;
    case types::ps_cell_info:
      j.write_fieldname("PSCellInformation");
      c.get<ps_cell_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_notify_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().pack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().pack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().pack(bref));
      break;
    case types::lhn_id:
      HANDLE_CODE((c.get<bounded_octstring<32, 256, true> >().pack(bref)));
      break;
    case types::ps_cell_info:
      HANDLE_CODE(c.get<ps_cell_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_notify_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().unpack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().unpack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().unpack(bref));
      break;
    case types::lhn_id:
      HANDLE_CODE((c.get<bounded_octstring<32, 256, true> >().unpack(bref)));
      break;
    case types::ps_cell_info:
      HANDLE_CODE(c.get<ps_cell_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_notify_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "EUTRAN-CGI",
                                  "TAI",
                                  "TunnelInformation",
                                  "OCTET STRING",
                                  "PSCellInformation"};
  return convert_enum_idx(options, 7, value, "ho_notify_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_notify_ies_o>;

ho_notify_ies_container::ho_notify_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  eutran_cgi(100, crit_e::ignore),
  tai(67, crit_e::ignore),
  tunnel_info_for_bbf(176, crit_e::ignore),
  lhn_id(186, crit_e::ignore),
  ps_cell_info(288, crit_e::ignore)
{}
SRSASN_CODE ho_notify_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += tunnel_info_for_bbf_present ? 1 : 0;
  nof_ies += lhn_id_present ? 1 : 0;
  nof_ies += ps_cell_info_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  if (tunnel_info_for_bbf_present) {
    HANDLE_CODE(tunnel_info_for_bbf.pack(bref));
  }
  if (lhn_id_present) {
    HANDLE_CODE(lhn_id.pack(bref));
  }
  if (ps_cell_info_present) {
    HANDLE_CODE(ps_cell_info.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_notify_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 100: {
        nof_mandatory_ies--;
        eutran_cgi.id = id;
        HANDLE_CODE(eutran_cgi.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(eutran_cgi.value.unpack(bref));
        break;
      }
      case 67: {
        nof_mandatory_ies--;
        tai.id = id;
        HANDLE_CODE(tai.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(tai.value.unpack(bref));
        break;
      }
      case 176: {
        tunnel_info_for_bbf_present = true;
        tunnel_info_for_bbf.id      = id;
        HANDLE_CODE(tunnel_info_for_bbf.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(tunnel_info_for_bbf.value.unpack(bref));
        break;
      }
      case 186: {
        lhn_id_present = true;
        lhn_id.id      = id;
        HANDLE_CODE(lhn_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(lhn_id.value.unpack(bref));
        break;
      }
      case 288: {
        ps_cell_info_present = true;
        ps_cell_info.id      = id;
        HANDLE_CODE(ps_cell_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ps_cell_info.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_notify_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  eutran_cgi.to_json(j);
  j.write_fieldname("");
  tai.to_json(j);
  if (tunnel_info_for_bbf_present) {
    j.write_fieldname("");
    tunnel_info_for_bbf.to_json(j);
  }
  if (lhn_id_present) {
    j.write_fieldname("");
    lhn_id.to_json(j);
  }
  if (ps_cell_info_present) {
    j.write_fieldname("");
    ps_cell_info.to_json(j);
  }
  j.end_obj();
}

// HandoverPreparationFailureIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_prep_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 2, 58};
  return map_enum_number(options, 4, idx, "id");
}
bool ho_prep_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 2, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_prep_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 2:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_prep_fail_ies_o::value_c ho_prep_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_prep_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_prep_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_prep_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
}
ho_prep_fail_ies_o::value_c::value_c(const ho_prep_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
}
ho_prep_fail_ies_o::value_c& ho_prep_fail_ies_o::value_c::operator=(const ho_prep_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_prep_fail_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_prep_fail_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
cause_c& ho_prep_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& ho_prep_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_prep_fail_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_prep_fail_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const cause_c& ho_prep_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& ho_prep_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_prep_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_prep_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_prep_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_prep_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 4, value, "ho_prep_fail_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_prep_fail_ies_o>;

ho_prep_fail_ies_container::ho_prep_fail_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  cause(2, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE ho_prep_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_prep_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 2: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_prep_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// MBSFN-ResultToLogInfo ::= SEQUENCE
SRSASN_CODE mbsfn_result_to_log_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mbsfn_area_id_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (mbsfn_area_id_present) {
    HANDLE_CODE(pack_integer(bref, mbsfn_area_id, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  HANDLE_CODE(pack_integer(bref, carrier_freq, (uint32_t)0u, (uint32_t)262143u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbsfn_result_to_log_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mbsfn_area_id_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (mbsfn_area_id_present) {
    HANDLE_CODE(unpack_integer(mbsfn_area_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  HANDLE_CODE(unpack_integer(carrier_freq, bref, (uint32_t)0u, (uint32_t)262143u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void mbsfn_result_to_log_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (mbsfn_area_id_present) {
    j.write_int("mBSFN-AreaId", mbsfn_area_id);
  }
  j.write_int("carrierFreq", carrier_freq);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Links-to-log ::= ENUMERATED
const char* links_to_log_opts::to_string() const
{
  static const char* options[] = {"uplink", "downlink", "both-uplink-and-downlink"};
  return convert_enum_idx(options, 3, value, "links_to_log_e");
}

// LoggingDuration ::= ENUMERATED
const char* logging_dur_opts::to_string() const
{
  static const char* options[] = {"m10", "m20", "m40", "m60", "m90", "m120"};
  return convert_enum_idx(options, 6, value, "logging_dur_e");
}
uint8_t logging_dur_opts::to_number() const
{
  static const uint8_t options[] = {10, 20, 40, 60, 90, 120};
  return map_enum_number(options, 6, value, "logging_dur_e");
}

// LoggingInterval ::= ENUMERATED
const char* logging_interv_opts::to_string() const
{
  static const char* options[] = {"ms128", "ms256", "ms512", "ms1024", "ms2048", "ms3072", "ms4096", "ms6144"};
  return convert_enum_idx(options, 8, value, "logging_interv_e");
}
uint16_t logging_interv_opts::to_number() const
{
  static const uint16_t options[] = {128, 256, 512, 1024, 2048, 3072, 4096, 6144};
  return map_enum_number(options, 8, value, "logging_interv_e");
}

// M3period ::= ENUMERATED
const char* m3period_opts::to_string() const
{
  static const char* options[] = {
      "ms100", "ms1000", "ms10000", "ms1024", "ms1280", "ms2048", "ms2560", "ms5120", "ms10240", "min1"};
  return convert_enum_idx(options, 10, value, "m3period_e");
}
uint16_t m3period_opts::to_number() const
{
  static const uint16_t options[] = {100, 1000, 10000, 1024, 1280, 2048, 2560, 5120, 10240, 1};
  return map_enum_number(options, 10, value, "m3period_e");
}

// M4period ::= ENUMERATED
const char* m4period_opts::to_string() const
{
  static const char* options[] = {"ms1024", "ms2048", "ms5120", "ms10240", "min1"};
  return convert_enum_idx(options, 5, value, "m4period_e");
}
uint16_t m4period_opts::to_number() const
{
  static const uint16_t options[] = {1024, 2048, 5120, 10240, 1};
  return map_enum_number(options, 5, value, "m4period_e");
}

// M5period ::= ENUMERATED
const char* m5period_opts::to_string() const
{
  static const char* options[] = {"ms1024", "ms2048", "ms5120", "ms10240", "min1"};
  return convert_enum_idx(options, 5, value, "m5period_e");
}
uint16_t m5period_opts::to_number() const
{
  static const uint16_t options[] = {1024, 2048, 5120, 10240, 1};
  return map_enum_number(options, 5, value, "m5period_e");
}

// M6delay-threshold ::= ENUMERATED
const char* m6delay_thres_opts::to_string() const
{
  static const char* options[] = {
      "ms30", "ms40", "ms50", "ms60", "ms70", "ms80", "ms90", "ms100", "ms150", "ms300", "ms500", "ms750"};
  return convert_enum_idx(options, 12, value, "m6delay_thres_e");
}
uint16_t m6delay_thres_opts::to_number() const
{
  static const uint16_t options[] = {30, 40, 50, 60, 70, 80, 90, 100, 150, 300, 500, 750};
  return map_enum_number(options, 12, value, "m6delay_thres_e");
}

// M6report-Interval ::= ENUMERATED
const char* m6report_interv_opts::to_string() const
{
  static const char* options[] = {"ms1024", "ms2048", "ms5120", "ms10240"};
  return convert_enum_idx(options, 4, value, "m6report_interv_e");
}
uint16_t m6report_interv_opts::to_number() const
{
  static const uint16_t options[] = {1024, 2048, 5120, 10240};
  return map_enum_number(options, 4, value, "m6report_interv_e");
}

// WLANMeasConfig ::= ENUMERATED
const char* wlan_meas_cfg_opts::to_string() const
{
  static const char* options[] = {"setup"};
  return convert_enum_idx(options, 1, value, "wlan_meas_cfg_e");
}

// LoggedMBSFNMDT ::= SEQUENCE
SRSASN_CODE logged_mbsfnmdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mbsfn_result_to_log_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(logging_interv.pack(bref));
  HANDLE_CODE(logging_dur.pack(bref));
  if (mbsfn_result_to_log_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, mbsfn_result_to_log, 1, 8, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE logged_mbsfnmdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mbsfn_result_to_log_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(logging_interv.unpack(bref));
  HANDLE_CODE(logging_dur.unpack(bref));
  if (mbsfn_result_to_log_present) {
    HANDLE_CODE(unpack_dyn_seq_of(mbsfn_result_to_log, bref, 1, 8, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void logged_mbsfnmdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("loggingInterval", logging_interv.to_string());
  j.write_str("loggingDuration", logging_dur.to_string());
  if (mbsfn_result_to_log_present) {
    j.start_array("mBSFN-ResultToLog");
    for (const auto& e1 : mbsfn_result_to_log) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M3Configuration ::= SEQUENCE
SRSASN_CODE m3_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(m3period.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m3_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(m3period.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m3_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("m3period", m3period.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M4Configuration ::= SEQUENCE
SRSASN_CODE m4_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(m4period.pack(bref));
  HANDLE_CODE(m4_links_to_log.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m4_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(m4period.unpack(bref));
  HANDLE_CODE(m4_links_to_log.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m4_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("m4period", m4period.to_string());
  j.write_str("m4-links-to-log", m4_links_to_log.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M5Configuration ::= SEQUENCE
SRSASN_CODE m5_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(m5period.pack(bref));
  HANDLE_CODE(m5_links_to_log.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m5_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(m5period.unpack(bref));
  HANDLE_CODE(m5_links_to_log.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m5_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("m5period", m5period.to_string());
  j.write_str("m5-links-to-log", m5_links_to_log.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M6Configuration ::= SEQUENCE
SRSASN_CODE m6_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(m6delay_thres_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(m6report_interv.pack(bref));
  if (m6delay_thres_present) {
    HANDLE_CODE(m6delay_thres.pack(bref));
  }
  HANDLE_CODE(m6_links_to_log.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m6_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(m6delay_thres_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(m6report_interv.unpack(bref));
  if (m6delay_thres_present) {
    HANDLE_CODE(m6delay_thres.unpack(bref));
  }
  HANDLE_CODE(m6_links_to_log.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m6_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("m6report-Interval", m6report_interv.to_string());
  if (m6delay_thres_present) {
    j.write_str("m6delay-threshold", m6delay_thres.to_string());
  }
  j.write_str("m6-links-to-log", m6_links_to_log.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M7Configuration ::= SEQUENCE
SRSASN_CODE m7_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, m7period, (uint8_t)1u, (uint8_t)60u, true, true));
  HANDLE_CODE(m7_links_to_log.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m7_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(m7period, bref, (uint8_t)1u, (uint8_t)60u, true, true));
  HANDLE_CODE(m7_links_to_log.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m7_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("m7period", m7period);
  j.write_str("m7-links-to-log", m7_links_to_log.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MeasurementThresholdA2 ::= CHOICE
void meas_thres_a2_c::destroy_() {}
void meas_thres_a2_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
meas_thres_a2_c::meas_thres_a2_c(const meas_thres_a2_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::thres_rsrp:
      c.init(other.c.get<uint8_t>());
      break;
    case types::thres_rsrq:
      c.init(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
  }
}
meas_thres_a2_c& meas_thres_a2_c::operator=(const meas_thres_a2_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::thres_rsrp:
      c.set(other.c.get<uint8_t>());
      break;
    case types::thres_rsrq:
      c.set(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
  }

  return *this;
}
uint8_t& meas_thres_a2_c::set_thres_rsrp()
{
  set(types::thres_rsrp);
  return c.get<uint8_t>();
}
uint8_t& meas_thres_a2_c::set_thres_rsrq()
{
  set(types::thres_rsrq);
  return c.get<uint8_t>();
}
void meas_thres_a2_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::thres_rsrp:
      j.write_int("threshold-RSRP", c.get<uint8_t>());
      break;
    case types::thres_rsrq:
      j.write_int("threshold-RSRQ", c.get<uint8_t>());
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
  }
  j.end_obj();
}
SRSASN_CODE meas_thres_a2_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::thres_rsrp:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)97u, false, true));
      break;
    case types::thres_rsrq:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)34u, false, true));
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE meas_thres_a2_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::thres_rsrp:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)97u, false, true));
      break;
    case types::thres_rsrq:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)34u, false, true));
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* meas_thres_a2_c::types_opts::to_string() const
{
  static const char* options[] = {"threshold-RSRP", "threshold-RSRQ"};
  return convert_enum_idx(options, 2, value, "meas_thres_a2_c::types");
}

// ReportAmountMDT ::= ENUMERATED
const char* report_amount_mdt_opts::to_string() const
{
  static const char* options[] = {"r1", "r2", "r4", "r8", "r16", "r32", "r64", "rinfinity"};
  return convert_enum_idx(options, 8, value, "report_amount_mdt_e");
}
int8_t report_amount_mdt_opts::to_number() const
{
  static const int8_t options[] = {1, 2, 4, 8, 16, 32, 64, -1};
  return map_enum_number(options, 8, value, "report_amount_mdt_e");
}

// ReportIntervalMDT ::= ENUMERATED
const char* report_interv_mdt_opts::to_string() const
{
  static const char* options[] = {"ms120",
                                  "ms240",
                                  "ms480",
                                  "ms640",
                                  "ms1024",
                                  "ms2048",
                                  "ms5120",
                                  "ms10240",
                                  "min1",
                                  "min6",
                                  "min12",
                                  "min30",
                                  "min60"};
  return convert_enum_idx(options, 13, value, "report_interv_mdt_e");
}
uint16_t report_interv_mdt_opts::to_number() const
{
  static const uint16_t options[] = {120, 240, 480, 640, 1024, 2048, 5120, 10240, 1, 6, 12, 30, 60};
  return map_enum_number(options, 13, value, "report_interv_mdt_e");
}

// WLANMeasurementConfiguration ::= SEQUENCE
SRSASN_CODE wlan_meas_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(wlan_meas_cfg_name_list_present, 1));
  HANDLE_CODE(bref.pack(wlan_rssi_present, 1));
  HANDLE_CODE(bref.pack(wlan_rtt_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(wlan_meas_cfg.pack(bref));
  if (wlan_meas_cfg_name_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, wlan_meas_cfg_name_list, 1, 4, true));
  }
  if (wlan_rssi_present) {
    HANDLE_CODE(wlan_rssi.pack(bref));
  }
  if (wlan_rtt_present) {
    HANDLE_CODE(wlan_rtt.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE wlan_meas_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(wlan_meas_cfg_name_list_present, 1));
  HANDLE_CODE(bref.unpack(wlan_rssi_present, 1));
  HANDLE_CODE(bref.unpack(wlan_rtt_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(wlan_meas_cfg.unpack(bref));
  if (wlan_meas_cfg_name_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(wlan_meas_cfg_name_list, bref, 1, 4, true));
  }
  if (wlan_rssi_present) {
    HANDLE_CODE(wlan_rssi.unpack(bref));
  }
  if (wlan_rtt_present) {
    HANDLE_CODE(wlan_rtt.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void wlan_meas_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("wlanMeasConfig", "setup");
  if (wlan_meas_cfg_name_list_present) {
    j.start_array("wlanMeasConfigNameList");
    for (const auto& e1 : wlan_meas_cfg_name_list) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (wlan_rssi_present) {
    j.write_str("wlan-rssi", "true");
  }
  if (wlan_rtt_present) {
    j.write_str("wlan-rtt", "true");
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

const char* wlan_meas_cfg_s::wlan_rssi_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "wlan_meas_cfg_s::wlan_rssi_e_");
}

const char* wlan_meas_cfg_s::wlan_rtt_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "wlan_meas_cfg_s::wlan_rtt_e_");
}

// ImmediateMDT-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t immediate_mdt_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {171, 172, 173, 174, 220, 221, 284, 285};
  return map_enum_number(options, 8, idx, "id");
}
bool immediate_mdt_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {171, 172, 173, 174, 220, 221, 284, 285};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e immediate_mdt_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 171:
      return crit_e::ignore;
    case 172:
      return crit_e::ignore;
    case 173:
      return crit_e::ignore;
    case 174:
      return crit_e::ignore;
    case 220:
      return crit_e::ignore;
    case 221:
      return crit_e::ignore;
    case 284:
      return crit_e::ignore;
    case 285:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
immediate_mdt_ext_ies_o::ext_c immediate_mdt_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 171:
      ret.set(ext_c::types::m3_cfg);
      break;
    case 172:
      ret.set(ext_c::types::m4_cfg);
      break;
    case 173:
      ret.set(ext_c::types::m5_cfg);
      break;
    case 174:
      ret.set(ext_c::types::mdt_location_info);
      break;
    case 220:
      ret.set(ext_c::types::m6_cfg);
      break;
    case 221:
      ret.set(ext_c::types::m7_cfg);
      break;
    case 284:
      ret.set(ext_c::types::bluetooth_meas_cfg);
      break;
    case 285:
      ret.set(ext_c::types::wlan_meas_cfg);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e immediate_mdt_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 171:
      return presence_e::conditional;
    case 172:
      return presence_e::conditional;
    case 173:
      return presence_e::conditional;
    case 174:
      return presence_e::optional;
    case 220:
      return presence_e::conditional;
    case 221:
      return presence_e::conditional;
    case 284:
      return presence_e::optional;
    case 285:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void immediate_mdt_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::m3_cfg:
      c.destroy<m3_cfg_s>();
      break;
    case types::m4_cfg:
      c.destroy<m4_cfg_s>();
      break;
    case types::m5_cfg:
      c.destroy<m5_cfg_s>();
      break;
    case types::mdt_location_info:
      c.destroy<fixed_bitstring<8, false, true> >();
      break;
    case types::m6_cfg:
      c.destroy<m6_cfg_s>();
      break;
    case types::m7_cfg:
      c.destroy<m7_cfg_s>();
      break;
    case types::bluetooth_meas_cfg:
      c.destroy<bluetooth_meas_cfg_s>();
      break;
    case types::wlan_meas_cfg:
      c.destroy<wlan_meas_cfg_s>();
      break;
    default:
      break;
  }
}
void immediate_mdt_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::m3_cfg:
      c.init<m3_cfg_s>();
      break;
    case types::m4_cfg:
      c.init<m4_cfg_s>();
      break;
    case types::m5_cfg:
      c.init<m5_cfg_s>();
      break;
    case types::mdt_location_info:
      c.init<fixed_bitstring<8, false, true> >();
      break;
    case types::m6_cfg:
      c.init<m6_cfg_s>();
      break;
    case types::m7_cfg:
      c.init<m7_cfg_s>();
      break;
    case types::bluetooth_meas_cfg:
      c.init<bluetooth_meas_cfg_s>();
      break;
    case types::wlan_meas_cfg:
      c.init<wlan_meas_cfg_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
  }
}
immediate_mdt_ext_ies_o::ext_c::ext_c(const immediate_mdt_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::m3_cfg:
      c.init(other.c.get<m3_cfg_s>());
      break;
    case types::m4_cfg:
      c.init(other.c.get<m4_cfg_s>());
      break;
    case types::m5_cfg:
      c.init(other.c.get<m5_cfg_s>());
      break;
    case types::mdt_location_info:
      c.init(other.c.get<fixed_bitstring<8, false, true> >());
      break;
    case types::m6_cfg:
      c.init(other.c.get<m6_cfg_s>());
      break;
    case types::m7_cfg:
      c.init(other.c.get<m7_cfg_s>());
      break;
    case types::bluetooth_meas_cfg:
      c.init(other.c.get<bluetooth_meas_cfg_s>());
      break;
    case types::wlan_meas_cfg:
      c.init(other.c.get<wlan_meas_cfg_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
  }
}
immediate_mdt_ext_ies_o::ext_c& immediate_mdt_ext_ies_o::ext_c::operator=(const immediate_mdt_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::m3_cfg:
      c.set(other.c.get<m3_cfg_s>());
      break;
    case types::m4_cfg:
      c.set(other.c.get<m4_cfg_s>());
      break;
    case types::m5_cfg:
      c.set(other.c.get<m5_cfg_s>());
      break;
    case types::mdt_location_info:
      c.set(other.c.get<fixed_bitstring<8, false, true> >());
      break;
    case types::m6_cfg:
      c.set(other.c.get<m6_cfg_s>());
      break;
    case types::m7_cfg:
      c.set(other.c.get<m7_cfg_s>());
      break;
    case types::bluetooth_meas_cfg:
      c.set(other.c.get<bluetooth_meas_cfg_s>());
      break;
    case types::wlan_meas_cfg:
      c.set(other.c.get<wlan_meas_cfg_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
  }

  return *this;
}
m3_cfg_s& immediate_mdt_ext_ies_o::ext_c::m3_cfg()
{
  assert_choice_type(types::m3_cfg, type_, "Extension");
  return c.get<m3_cfg_s>();
}
m4_cfg_s& immediate_mdt_ext_ies_o::ext_c::m4_cfg()
{
  assert_choice_type(types::m4_cfg, type_, "Extension");
  return c.get<m4_cfg_s>();
}
m5_cfg_s& immediate_mdt_ext_ies_o::ext_c::m5_cfg()
{
  assert_choice_type(types::m5_cfg, type_, "Extension");
  return c.get<m5_cfg_s>();
}
fixed_bitstring<8, false, true>& immediate_mdt_ext_ies_o::ext_c::mdt_location_info()
{
  assert_choice_type(types::mdt_location_info, type_, "Extension");
  return c.get<fixed_bitstring<8, false, true> >();
}
m6_cfg_s& immediate_mdt_ext_ies_o::ext_c::m6_cfg()
{
  assert_choice_type(types::m6_cfg, type_, "Extension");
  return c.get<m6_cfg_s>();
}
m7_cfg_s& immediate_mdt_ext_ies_o::ext_c::m7_cfg()
{
  assert_choice_type(types::m7_cfg, type_, "Extension");
  return c.get<m7_cfg_s>();
}
bluetooth_meas_cfg_s& immediate_mdt_ext_ies_o::ext_c::bluetooth_meas_cfg()
{
  assert_choice_type(types::bluetooth_meas_cfg, type_, "Extension");
  return c.get<bluetooth_meas_cfg_s>();
}
wlan_meas_cfg_s& immediate_mdt_ext_ies_o::ext_c::wlan_meas_cfg()
{
  assert_choice_type(types::wlan_meas_cfg, type_, "Extension");
  return c.get<wlan_meas_cfg_s>();
}
const m3_cfg_s& immediate_mdt_ext_ies_o::ext_c::m3_cfg() const
{
  assert_choice_type(types::m3_cfg, type_, "Extension");
  return c.get<m3_cfg_s>();
}
const m4_cfg_s& immediate_mdt_ext_ies_o::ext_c::m4_cfg() const
{
  assert_choice_type(types::m4_cfg, type_, "Extension");
  return c.get<m4_cfg_s>();
}
const m5_cfg_s& immediate_mdt_ext_ies_o::ext_c::m5_cfg() const
{
  assert_choice_type(types::m5_cfg, type_, "Extension");
  return c.get<m5_cfg_s>();
}
const fixed_bitstring<8, false, true>& immediate_mdt_ext_ies_o::ext_c::mdt_location_info() const
{
  assert_choice_type(types::mdt_location_info, type_, "Extension");
  return c.get<fixed_bitstring<8, false, true> >();
}
const m6_cfg_s& immediate_mdt_ext_ies_o::ext_c::m6_cfg() const
{
  assert_choice_type(types::m6_cfg, type_, "Extension");
  return c.get<m6_cfg_s>();
}
const m7_cfg_s& immediate_mdt_ext_ies_o::ext_c::m7_cfg() const
{
  assert_choice_type(types::m7_cfg, type_, "Extension");
  return c.get<m7_cfg_s>();
}
const bluetooth_meas_cfg_s& immediate_mdt_ext_ies_o::ext_c::bluetooth_meas_cfg() const
{
  assert_choice_type(types::bluetooth_meas_cfg, type_, "Extension");
  return c.get<bluetooth_meas_cfg_s>();
}
const wlan_meas_cfg_s& immediate_mdt_ext_ies_o::ext_c::wlan_meas_cfg() const
{
  assert_choice_type(types::wlan_meas_cfg, type_, "Extension");
  return c.get<wlan_meas_cfg_s>();
}
void immediate_mdt_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::m3_cfg:
      j.write_fieldname("M3Configuration");
      c.get<m3_cfg_s>().to_json(j);
      break;
    case types::m4_cfg:
      j.write_fieldname("M4Configuration");
      c.get<m4_cfg_s>().to_json(j);
      break;
    case types::m5_cfg:
      j.write_fieldname("M5Configuration");
      c.get<m5_cfg_s>().to_json(j);
      break;
    case types::mdt_location_info:
      j.write_str("BIT STRING", c.get<fixed_bitstring<8, false, true> >().to_string());
      break;
    case types::m6_cfg:
      j.write_fieldname("M6Configuration");
      c.get<m6_cfg_s>().to_json(j);
      break;
    case types::m7_cfg:
      j.write_fieldname("M7Configuration");
      c.get<m7_cfg_s>().to_json(j);
      break;
    case types::bluetooth_meas_cfg:
      j.write_fieldname("BluetoothMeasurementConfiguration");
      c.get<bluetooth_meas_cfg_s>().to_json(j);
      break;
    case types::wlan_meas_cfg:
      j.write_fieldname("WLANMeasurementConfiguration");
      c.get<wlan_meas_cfg_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE immediate_mdt_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::m3_cfg:
      HANDLE_CODE(c.get<m3_cfg_s>().pack(bref));
      break;
    case types::m4_cfg:
      HANDLE_CODE(c.get<m4_cfg_s>().pack(bref));
      break;
    case types::m5_cfg:
      HANDLE_CODE(c.get<m5_cfg_s>().pack(bref));
      break;
    case types::mdt_location_info:
      HANDLE_CODE((c.get<fixed_bitstring<8, false, true> >().pack(bref)));
      break;
    case types::m6_cfg:
      HANDLE_CODE(c.get<m6_cfg_s>().pack(bref));
      break;
    case types::m7_cfg:
      HANDLE_CODE(c.get<m7_cfg_s>().pack(bref));
      break;
    case types::bluetooth_meas_cfg:
      HANDLE_CODE(c.get<bluetooth_meas_cfg_s>().pack(bref));
      break;
    case types::wlan_meas_cfg:
      HANDLE_CODE(c.get<wlan_meas_cfg_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE immediate_mdt_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::m3_cfg:
      HANDLE_CODE(c.get<m3_cfg_s>().unpack(bref));
      break;
    case types::m4_cfg:
      HANDLE_CODE(c.get<m4_cfg_s>().unpack(bref));
      break;
    case types::m5_cfg:
      HANDLE_CODE(c.get<m5_cfg_s>().unpack(bref));
      break;
    case types::mdt_location_info:
      HANDLE_CODE((c.get<fixed_bitstring<8, false, true> >().unpack(bref)));
      break;
    case types::m6_cfg:
      HANDLE_CODE(c.get<m6_cfg_s>().unpack(bref));
      break;
    case types::m7_cfg:
      HANDLE_CODE(c.get<m7_cfg_s>().unpack(bref));
      break;
    case types::bluetooth_meas_cfg:
      HANDLE_CODE(c.get<bluetooth_meas_cfg_s>().unpack(bref));
      break;
    case types::wlan_meas_cfg:
      HANDLE_CODE(c.get<wlan_meas_cfg_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* immediate_mdt_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"M3Configuration",
                                  "M4Configuration",
                                  "M5Configuration",
                                  "BIT STRING",
                                  "M6Configuration",
                                  "M7Configuration",
                                  "BluetoothMeasurementConfiguration",
                                  "WLANMeasurementConfiguration"};
  return convert_enum_idx(options, 8, value, "immediate_mdt_ext_ies_o::ext_c::types");
}
uint8_t immediate_mdt_ext_ies_o::ext_c::types_opts::to_number() const
{
  switch (value) {
    case m3_cfg:
      return 3;
    case m4_cfg:
      return 4;
    case m5_cfg:
      return 5;
    case m6_cfg:
      return 6;
    case m7_cfg:
      return 7;
    default:
      invalid_enum_number(value, "immediate_mdt_ext_ies_o::ext_c::types");
  }
  return 0;
}

// LoggedMDT-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t logged_mdt_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {284, 285};
  return map_enum_number(options, 2, idx, "id");
}
bool logged_mdt_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {284, 285};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e logged_mdt_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 284:
      return crit_e::ignore;
    case 285:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
logged_mdt_ext_ies_o::ext_c logged_mdt_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 284:
      ret.set(ext_c::types::bluetooth_meas_cfg);
      break;
    case 285:
      ret.set(ext_c::types::wlan_meas_cfg);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e logged_mdt_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 284:
      return presence_e::optional;
    case 285:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void logged_mdt_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::bluetooth_meas_cfg:
      c.destroy<bluetooth_meas_cfg_s>();
      break;
    case types::wlan_meas_cfg:
      c.destroy<wlan_meas_cfg_s>();
      break;
    default:
      break;
  }
}
void logged_mdt_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::bluetooth_meas_cfg:
      c.init<bluetooth_meas_cfg_s>();
      break;
    case types::wlan_meas_cfg:
      c.init<wlan_meas_cfg_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "logged_mdt_ext_ies_o::ext_c");
  }
}
logged_mdt_ext_ies_o::ext_c::ext_c(const logged_mdt_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::bluetooth_meas_cfg:
      c.init(other.c.get<bluetooth_meas_cfg_s>());
      break;
    case types::wlan_meas_cfg:
      c.init(other.c.get<wlan_meas_cfg_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "logged_mdt_ext_ies_o::ext_c");
  }
}
logged_mdt_ext_ies_o::ext_c& logged_mdt_ext_ies_o::ext_c::operator=(const logged_mdt_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::bluetooth_meas_cfg:
      c.set(other.c.get<bluetooth_meas_cfg_s>());
      break;
    case types::wlan_meas_cfg:
      c.set(other.c.get<wlan_meas_cfg_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "logged_mdt_ext_ies_o::ext_c");
  }

  return *this;
}
bluetooth_meas_cfg_s& logged_mdt_ext_ies_o::ext_c::bluetooth_meas_cfg()
{
  assert_choice_type(types::bluetooth_meas_cfg, type_, "Extension");
  return c.get<bluetooth_meas_cfg_s>();
}
wlan_meas_cfg_s& logged_mdt_ext_ies_o::ext_c::wlan_meas_cfg()
{
  assert_choice_type(types::wlan_meas_cfg, type_, "Extension");
  return c.get<wlan_meas_cfg_s>();
}
const bluetooth_meas_cfg_s& logged_mdt_ext_ies_o::ext_c::bluetooth_meas_cfg() const
{
  assert_choice_type(types::bluetooth_meas_cfg, type_, "Extension");
  return c.get<bluetooth_meas_cfg_s>();
}
const wlan_meas_cfg_s& logged_mdt_ext_ies_o::ext_c::wlan_meas_cfg() const
{
  assert_choice_type(types::wlan_meas_cfg, type_, "Extension");
  return c.get<wlan_meas_cfg_s>();
}
void logged_mdt_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::bluetooth_meas_cfg:
      j.write_fieldname("BluetoothMeasurementConfiguration");
      c.get<bluetooth_meas_cfg_s>().to_json(j);
      break;
    case types::wlan_meas_cfg:
      j.write_fieldname("WLANMeasurementConfiguration");
      c.get<wlan_meas_cfg_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "logged_mdt_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE logged_mdt_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::bluetooth_meas_cfg:
      HANDLE_CODE(c.get<bluetooth_meas_cfg_s>().pack(bref));
      break;
    case types::wlan_meas_cfg:
      HANDLE_CODE(c.get<wlan_meas_cfg_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "logged_mdt_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE logged_mdt_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::bluetooth_meas_cfg:
      HANDLE_CODE(c.get<bluetooth_meas_cfg_s>().unpack(bref));
      break;
    case types::wlan_meas_cfg:
      HANDLE_CODE(c.get<wlan_meas_cfg_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "logged_mdt_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* logged_mdt_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"BluetoothMeasurementConfiguration", "WLANMeasurementConfiguration"};
  return convert_enum_idx(options, 2, value, "logged_mdt_ext_ies_o::ext_c::types");
}

// M1PeriodicReporting ::= SEQUENCE
SRSASN_CODE m1_periodic_report_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(report_interv.pack(bref));
  HANDLE_CODE(report_amount.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m1_periodic_report_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(report_interv.unpack(bref));
  HANDLE_CODE(report_amount.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m1_periodic_report_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("reportInterval", report_interv.to_string());
  j.write_str("reportAmount", report_amount.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M1ReportingTrigger ::= ENUMERATED
const char* m1_report_trigger_opts::to_string() const
{
  static const char* options[] = {"periodic", "a2eventtriggered", "a2eventtriggered-periodic"};
  return convert_enum_idx(options, 3, value, "m1_report_trigger_e");
}

// M1ThresholdEventA2 ::= SEQUENCE
SRSASN_CODE m1_thres_event_a2_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(meas_thres.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m1_thres_event_a2_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(meas_thres.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m1_thres_event_a2_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("measurementThreshold");
  meas_thres.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MDTMode-ExtensionIE ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t mdt_mode_ext_ie_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {197};
  return map_enum_number(options, 1, idx, "id");
}
bool mdt_mode_ext_ie_o::is_id_valid(const uint32_t& id)
{
  return 197 == id;
}
crit_e mdt_mode_ext_ie_o::get_crit(const uint32_t& id)
{
  if (id == 197) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
mdt_mode_ext_ie_o::value_c mdt_mode_ext_ie_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 197) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e mdt_mode_ext_ie_o::get_presence(const uint32_t& id)
{
  if (id == 197) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void mdt_mode_ext_ie_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("LoggedMBSFNMDT");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE mdt_mode_ext_ie_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE mdt_mode_ext_ie_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* mdt_mode_ext_ie_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"LoggedMBSFNMDT"};
  return convert_enum_idx(options, 1, value, "mdt_mode_ext_ie_o::value_c::types");
}

template struct asn1::protocol_ext_field_s<immediate_mdt_ext_ies_o>;

immediate_mdt_ext_ies_container::immediate_mdt_ext_ies_container() :
  m3_cfg(171, crit_e::ignore),
  m4_cfg(172, crit_e::ignore),
  m5_cfg(173, crit_e::ignore),
  mdt_location_info(174, crit_e::ignore),
  m6_cfg(220, crit_e::ignore),
  m7_cfg(221, crit_e::ignore),
  bluetooth_meas_cfg(284, crit_e::ignore),
  wlan_meas_cfg(285, crit_e::ignore)
{}
SRSASN_CODE immediate_mdt_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += m3_cfg_present ? 1 : 0;
  nof_ies += m4_cfg_present ? 1 : 0;
  nof_ies += m5_cfg_present ? 1 : 0;
  nof_ies += mdt_location_info_present ? 1 : 0;
  nof_ies += m6_cfg_present ? 1 : 0;
  nof_ies += m7_cfg_present ? 1 : 0;
  nof_ies += bluetooth_meas_cfg_present ? 1 : 0;
  nof_ies += wlan_meas_cfg_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (m3_cfg_present) {
    HANDLE_CODE(m3_cfg.pack(bref));
  }
  if (m4_cfg_present) {
    HANDLE_CODE(m4_cfg.pack(bref));
  }
  if (m5_cfg_present) {
    HANDLE_CODE(m5_cfg.pack(bref));
  }
  if (mdt_location_info_present) {
    HANDLE_CODE(mdt_location_info.pack(bref));
  }
  if (m6_cfg_present) {
    HANDLE_CODE(m6_cfg.pack(bref));
  }
  if (m7_cfg_present) {
    HANDLE_CODE(m7_cfg.pack(bref));
  }
  if (bluetooth_meas_cfg_present) {
    HANDLE_CODE(bluetooth_meas_cfg.pack(bref));
  }
  if (wlan_meas_cfg_present) {
    HANDLE_CODE(wlan_meas_cfg.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE immediate_mdt_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 171: {
        m3_cfg_present = true;
        m3_cfg.id      = id;
        HANDLE_CODE(m3_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(m3_cfg.ext.unpack(bref));
        break;
      }
      case 172: {
        m4_cfg_present = true;
        m4_cfg.id      = id;
        HANDLE_CODE(m4_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(m4_cfg.ext.unpack(bref));
        break;
      }
      case 173: {
        m5_cfg_present = true;
        m5_cfg.id      = id;
        HANDLE_CODE(m5_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(m5_cfg.ext.unpack(bref));
        break;
      }
      case 174: {
        mdt_location_info_present = true;
        mdt_location_info.id      = id;
        HANDLE_CODE(mdt_location_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mdt_location_info.ext.unpack(bref));
        break;
      }
      case 220: {
        m6_cfg_present = true;
        m6_cfg.id      = id;
        HANDLE_CODE(m6_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(m6_cfg.ext.unpack(bref));
        break;
      }
      case 221: {
        m7_cfg_present = true;
        m7_cfg.id      = id;
        HANDLE_CODE(m7_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(m7_cfg.ext.unpack(bref));
        break;
      }
      case 284: {
        bluetooth_meas_cfg_present = true;
        bluetooth_meas_cfg.id      = id;
        HANDLE_CODE(bluetooth_meas_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(bluetooth_meas_cfg.ext.unpack(bref));
        break;
      }
      case 285: {
        wlan_meas_cfg_present = true;
        wlan_meas_cfg.id      = id;
        HANDLE_CODE(wlan_meas_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(wlan_meas_cfg.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void immediate_mdt_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (m3_cfg_present) {
    j.write_fieldname("");
    m3_cfg.to_json(j);
  }
  if (m4_cfg_present) {
    j.write_fieldname("");
    m4_cfg.to_json(j);
  }
  if (m5_cfg_present) {
    j.write_fieldname("");
    m5_cfg.to_json(j);
  }
  if (mdt_location_info_present) {
    j.write_fieldname("");
    mdt_location_info.to_json(j);
  }
  if (m6_cfg_present) {
    j.write_fieldname("");
    m6_cfg.to_json(j);
  }
  if (m7_cfg_present) {
    j.write_fieldname("");
    m7_cfg.to_json(j);
  }
  if (bluetooth_meas_cfg_present) {
    j.write_fieldname("");
    bluetooth_meas_cfg.to_json(j);
  }
  if (wlan_meas_cfg_present) {
    j.write_fieldname("");
    wlan_meas_cfg.to_json(j);
  }
  j.end_obj();
}

// ImmediateMDT ::= SEQUENCE
SRSASN_CODE immediate_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(m1thresevent_a2_present, 1));
  HANDLE_CODE(bref.pack(m1periodic_report_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(meass_to_activ.pack(bref));
  HANDLE_CODE(m1report_trigger.pack(bref));
  if (m1thresevent_a2_present) {
    HANDLE_CODE(m1thresevent_a2.pack(bref));
  }
  if (m1periodic_report_present) {
    HANDLE_CODE(m1periodic_report.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE immediate_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(m1thresevent_a2_present, 1));
  HANDLE_CODE(bref.unpack(m1periodic_report_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(meass_to_activ.unpack(bref));
  HANDLE_CODE(m1report_trigger.unpack(bref));
  if (m1thresevent_a2_present) {
    HANDLE_CODE(m1thresevent_a2.unpack(bref));
  }
  if (m1periodic_report_present) {
    HANDLE_CODE(m1periodic_report.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void immediate_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("measurementsToActivate", meass_to_activ.to_string());
  j.write_str("m1reportingTrigger", m1report_trigger.to_string());
  if (m1thresevent_a2_present) {
    j.write_fieldname("m1thresholdeventA2");
    m1thresevent_a2.to_json(j);
  }
  if (m1periodic_report_present) {
    j.write_fieldname("m1periodicReporting");
    m1periodic_report.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::protocol_ext_field_s<logged_mdt_ext_ies_o>;

logged_mdt_ext_ies_container::logged_mdt_ext_ies_container() :
  bluetooth_meas_cfg(284, crit_e::ignore), wlan_meas_cfg(285, crit_e::ignore)
{}
SRSASN_CODE logged_mdt_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += bluetooth_meas_cfg_present ? 1 : 0;
  nof_ies += wlan_meas_cfg_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (bluetooth_meas_cfg_present) {
    HANDLE_CODE(bluetooth_meas_cfg.pack(bref));
  }
  if (wlan_meas_cfg_present) {
    HANDLE_CODE(wlan_meas_cfg.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE logged_mdt_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 284: {
        bluetooth_meas_cfg_present = true;
        bluetooth_meas_cfg.id      = id;
        HANDLE_CODE(bluetooth_meas_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(bluetooth_meas_cfg.ext.unpack(bref));
        break;
      }
      case 285: {
        wlan_meas_cfg_present = true;
        wlan_meas_cfg.id      = id;
        HANDLE_CODE(wlan_meas_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(wlan_meas_cfg.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void logged_mdt_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (bluetooth_meas_cfg_present) {
    j.write_fieldname("");
    bluetooth_meas_cfg.to_json(j);
  }
  if (wlan_meas_cfg_present) {
    j.write_fieldname("");
    wlan_meas_cfg.to_json(j);
  }
  j.end_obj();
}

// LoggedMDT ::= SEQUENCE
SRSASN_CODE logged_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(logging_interv.pack(bref));
  HANDLE_CODE(logging_dur.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE logged_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(logging_interv.unpack(bref));
  HANDLE_CODE(logging_dur.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void logged_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("loggingInterval", logging_interv.to_string());
  j.write_str("loggingDuration", logging_dur.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::protocol_ie_single_container_s<mdt_mode_ext_ie_o>;

// ServiceType ::= ENUMERATED
const char* service_type_opts::to_string() const
{
  static const char* options[] = {"qMC-for-streaming-service", "qMC-for-MTSI-service"};
  return convert_enum_idx(options, 2, value, "service_type_e");
}

// MDT-Activation ::= ENUMERATED
const char* mdt_activation_opts::to_string() const
{
  static const char* options[] = {
      "immediate-MDT-only", "immediate-MDT-and-Trace", "logged-MDT-only", "logged-MBSFN-MDT"};
  return convert_enum_idx(options, 4, value, "mdt_activation_e");
}

// MDT-Configuration-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t mdt_cfg_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {178};
  return map_enum_number(options, 1, idx, "id");
}
bool mdt_cfg_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 178 == id;
}
crit_e mdt_cfg_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 178) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
mdt_cfg_ext_ies_o::ext_c mdt_cfg_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 178) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e mdt_cfg_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 178) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void mdt_cfg_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("MDTPLMNList");
  for (const auto& e1 : c) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  j.end_obj();
}
SRSASN_CODE mdt_cfg_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(pack_dyn_seq_of(bref, c, 1, 16, true));
  return SRSASN_SUCCESS;
}
SRSASN_CODE mdt_cfg_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(unpack_dyn_seq_of(c, bref, 1, 16, true));
  return SRSASN_SUCCESS;
}

const char* mdt_cfg_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"MDTPLMNList"};
  return convert_enum_idx(options, 1, value, "mdt_cfg_ext_ies_o::ext_c::types");
}

// MDTMode ::= CHOICE
void mdt_mode_c::destroy_()
{
  switch (type_) {
    case types::immediate_mdt:
      c.destroy<immediate_mdt_s>();
      break;
    case types::logged_mdt:
      c.destroy<logged_mdt_s>();
      break;
    case types::mdt_mode_ext:
      c.destroy<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >();
      break;
    default:
      break;
  }
}
void mdt_mode_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::immediate_mdt:
      c.init<immediate_mdt_s>();
      break;
    case types::logged_mdt:
      c.init<logged_mdt_s>();
      break;
    case types::mdt_mode_ext:
      c.init<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
  }
}
mdt_mode_c::mdt_mode_c(const mdt_mode_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::immediate_mdt:
      c.init(other.c.get<immediate_mdt_s>());
      break;
    case types::logged_mdt:
      c.init(other.c.get<logged_mdt_s>());
      break;
    case types::mdt_mode_ext:
      c.init(other.c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
  }
}
mdt_mode_c& mdt_mode_c::operator=(const mdt_mode_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::immediate_mdt:
      c.set(other.c.get<immediate_mdt_s>());
      break;
    case types::logged_mdt:
      c.set(other.c.get<logged_mdt_s>());
      break;
    case types::mdt_mode_ext:
      c.set(other.c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
  }

  return *this;
}
immediate_mdt_s& mdt_mode_c::set_immediate_mdt()
{
  set(types::immediate_mdt);
  return c.get<immediate_mdt_s>();
}
logged_mdt_s& mdt_mode_c::set_logged_mdt()
{
  set(types::logged_mdt);
  return c.get<logged_mdt_s>();
}
protocol_ie_single_container_s<mdt_mode_ext_ie_o>& mdt_mode_c::set_mdt_mode_ext()
{
  set(types::mdt_mode_ext);
  return c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >();
}
void mdt_mode_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::immediate_mdt:
      j.write_fieldname("immediateMDT");
      c.get<immediate_mdt_s>().to_json(j);
      break;
    case types::logged_mdt:
      j.write_fieldname("loggedMDT");
      c.get<logged_mdt_s>().to_json(j);
      break;
    case types::mdt_mode_ext:
      j.write_fieldname("mDTMode-Extension");
      c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
  }
  j.end_obj();
}
SRSASN_CODE mdt_mode_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::immediate_mdt:
      HANDLE_CODE(c.get<immediate_mdt_s>().pack(bref));
      break;
    case types::logged_mdt:
      HANDLE_CODE(c.get<logged_mdt_s>().pack(bref));
      break;
    case types::mdt_mode_ext: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE mdt_mode_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::immediate_mdt:
      HANDLE_CODE(c.get<immediate_mdt_s>().unpack(bref));
      break;
    case types::logged_mdt:
      HANDLE_CODE(c.get<logged_mdt_s>().unpack(bref));
      break;
    case types::mdt_mode_ext: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* mdt_mode_c::types_opts::to_string() const
{
  static const char* options[] = {"immediateMDT", "loggedMDT", "mDTMode-Extension"};
  return convert_enum_idx(options, 3, value, "mdt_mode_c::types");
}

// UEAppLayerMeasConfig-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t ue_app_layer_meas_cfg_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {276};
  return map_enum_number(options, 1, idx, "id");
}
bool ue_app_layer_meas_cfg_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 276 == id;
}
crit_e ue_app_layer_meas_cfg_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 276) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
ue_app_layer_meas_cfg_ext_ies_o::ext_c ue_app_layer_meas_cfg_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 276) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ue_app_layer_meas_cfg_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 276) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void ue_app_layer_meas_cfg_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("ServiceType", c.to_string());
  j.end_obj();
}
SRSASN_CODE ue_app_layer_meas_cfg_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_app_layer_meas_cfg_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* ue_app_layer_meas_cfg_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"ServiceType"};
  return convert_enum_idx(options, 1, value, "ue_app_layer_meas_cfg_ext_ies_o::ext_c::types");
}

// MDT-Configuration ::= SEQUENCE
SRSASN_CODE mdt_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(mdt_activation.pack(bref));
  HANDLE_CODE(area_scope_of_mdt.pack(bref));
  HANDLE_CODE(mdt_mode.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE mdt_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(mdt_activation.unpack(bref));
  HANDLE_CODE(area_scope_of_mdt.unpack(bref));
  HANDLE_CODE(mdt_mode.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void mdt_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("mdt-Activation", mdt_activation.to_string());
  j.write_fieldname("areaScopeOfMDT");
  area_scope_of_mdt.to_json(j);
  j.write_fieldname("mDTMode");
  mdt_mode.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// ProSeUEtoNetworkRelaying ::= ENUMERATED
const char* pro_se_ueto_network_relaying_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "pro_se_ueto_network_relaying_e");
}

// RequestTypeAdditionalInfo ::= ENUMERATED
const char* request_type_add_info_opts::to_string() const
{
  static const char* options[] = {"includePSCell"};
  return convert_enum_idx(options, 1, value, "request_type_add_info_e");
}

// UEAppLayerMeasConfig ::= SEQUENCE
SRSASN_CODE ue_app_layer_meas_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(container_for_app_layer_meas_cfg.pack(bref));
  HANDLE_CODE(area_scope_of_qmc.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_app_layer_meas_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(container_for_app_layer_meas_cfg.unpack(bref));
  HANDLE_CODE(area_scope_of_qmc.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void ue_app_layer_meas_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("containerForAppLayerMeasConfig", container_for_app_layer_meas_cfg.to_string());
  j.write_fieldname("areaScopeOfQMC");
  area_scope_of_qmc.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// EventType ::= ENUMERATED
const char* event_type_opts::to_string() const
{
  static const char* options[] = {"direct", "change-of-serve-cell", "stop-change-of-serve-cell"};
  return convert_enum_idx(options, 3, value, "event_type_e");
}

// PedestrianUE ::= ENUMERATED
const char* pedestrian_ue_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "pedestrian_ue_e");
}

// ProSeAuthorized-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t pro_se_authorized_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {216};
  return map_enum_number(options, 1, idx, "id");
}
bool pro_se_authorized_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 216 == id;
}
crit_e pro_se_authorized_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 216) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
pro_se_authorized_ext_ies_o::ext_c pro_se_authorized_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 216) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pro_se_authorized_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 216) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void pro_se_authorized_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("ProSeUEtoNetworkRelaying", c.to_string());
  j.end_obj();
}
SRSASN_CODE pro_se_authorized_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE pro_se_authorized_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* pro_se_authorized_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"ProSeUEtoNetworkRelaying"};
  return convert_enum_idx(options, 1, value, "pro_se_authorized_ext_ies_o::ext_c::types");
}
uint8_t pro_se_authorized_ext_ies_o::ext_c::types_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "pro_se_authorized_ext_ies_o::ext_c::types");
}

// ProSeDirectCommunication ::= ENUMERATED
const char* pro_se_direct_communication_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "pro_se_direct_communication_e");
}

// ProSeDirectDiscovery ::= ENUMERATED
const char* pro_se_direct_discovery_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "pro_se_direct_discovery_e");
}

// ReportArea ::= ENUMERATED
const char* report_area_opts::to_string() const
{
  static const char* options[] = {"ecgi"};
  return convert_enum_idx(options, 1, value, "report_area_e");
}

// RequestType-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t request_type_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {298};
  return map_enum_number(options, 1, idx, "id");
}
bool request_type_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 298 == id;
}
crit_e request_type_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 298) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
request_type_ext_ies_o::ext_c request_type_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 298) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e request_type_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 298) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void request_type_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("RequestTypeAdditionalInfo", "includePSCell");
  j.end_obj();
}
SRSASN_CODE request_type_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE request_type_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* request_type_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"RequestTypeAdditionalInfo"};
  return convert_enum_idx(options, 1, value, "request_type_ext_ies_o::ext_c::types");
}

// TraceActivation-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t trace_activation_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {162, 262};
  return map_enum_number(options, 2, idx, "id");
}
bool trace_activation_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {162, 262};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e trace_activation_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 162:
      return crit_e::ignore;
    case 262:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
trace_activation_ext_ies_o::ext_c trace_activation_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 162:
      ret.set(ext_c::types::mdt_cfg);
      break;
    case 262:
      ret.set(ext_c::types::ue_app_layer_meas_cfg);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e trace_activation_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 162:
      return presence_e::optional;
    case 262:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
void trace_activation_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::mdt_cfg:
      c.destroy<mdt_cfg_s>();
      break;
    case types::ue_app_layer_meas_cfg:
      c.destroy<ue_app_layer_meas_cfg_s>();
      break;
    default:
      break;
  }
}
void trace_activation_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mdt_cfg:
      c.init<mdt_cfg_s>();
      break;
    case types::ue_app_layer_meas_cfg:
      c.init<ue_app_layer_meas_cfg_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "trace_activation_ext_ies_o::ext_c");
  }
}
trace_activation_ext_ies_o::ext_c::ext_c(const trace_activation_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mdt_cfg:
      c.init(other.c.get<mdt_cfg_s>());
      break;
    case types::ue_app_layer_meas_cfg:
      c.init(other.c.get<ue_app_layer_meas_cfg_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "trace_activation_ext_ies_o::ext_c");
  }
}
trace_activation_ext_ies_o::ext_c&
trace_activation_ext_ies_o::ext_c::operator=(const trace_activation_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mdt_cfg:
      c.set(other.c.get<mdt_cfg_s>());
      break;
    case types::ue_app_layer_meas_cfg:
      c.set(other.c.get<ue_app_layer_meas_cfg_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "trace_activation_ext_ies_o::ext_c");
  }

  return *this;
}
mdt_cfg_s& trace_activation_ext_ies_o::ext_c::mdt_cfg()
{
  assert_choice_type(types::mdt_cfg, type_, "Extension");
  return c.get<mdt_cfg_s>();
}
ue_app_layer_meas_cfg_s& trace_activation_ext_ies_o::ext_c::ue_app_layer_meas_cfg()
{
  assert_choice_type(types::ue_app_layer_meas_cfg, type_, "Extension");
  return c.get<ue_app_layer_meas_cfg_s>();
}
const mdt_cfg_s& trace_activation_ext_ies_o::ext_c::mdt_cfg() const
{
  assert_choice_type(types::mdt_cfg, type_, "Extension");
  return c.get<mdt_cfg_s>();
}
const ue_app_layer_meas_cfg_s& trace_activation_ext_ies_o::ext_c::ue_app_layer_meas_cfg() const
{
  assert_choice_type(types::ue_app_layer_meas_cfg, type_, "Extension");
  return c.get<ue_app_layer_meas_cfg_s>();
}
void trace_activation_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mdt_cfg:
      j.write_fieldname("MDT-Configuration");
      c.get<mdt_cfg_s>().to_json(j);
      break;
    case types::ue_app_layer_meas_cfg:
      j.write_fieldname("UEAppLayerMeasConfig");
      c.get<ue_app_layer_meas_cfg_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "trace_activation_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE trace_activation_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mdt_cfg:
      HANDLE_CODE(c.get<mdt_cfg_s>().pack(bref));
      break;
    case types::ue_app_layer_meas_cfg:
      HANDLE_CODE(c.get<ue_app_layer_meas_cfg_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "trace_activation_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE trace_activation_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mdt_cfg:
      HANDLE_CODE(c.get<mdt_cfg_s>().unpack(bref));
      break;
    case types::ue_app_layer_meas_cfg:
      HANDLE_CODE(c.get<ue_app_layer_meas_cfg_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "trace_activation_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* trace_activation_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"MDT-Configuration", "UEAppLayerMeasConfig"};
  return convert_enum_idx(options, 2, value, "trace_activation_ext_ies_o::ext_c::types");
}

// TraceDepth ::= ENUMERATED
const char* trace_depth_opts::to_string() const
{
  static const char* options[] = {"minimum",
                                  "medium",
                                  "maximum",
                                  "minimumWithoutVendorSpecificExtension",
                                  "mediumWithoutVendorSpecificExtension",
                                  "maximumWithoutVendorSpecificExtension"};
  return convert_enum_idx(options, 6, value, "trace_depth_e");
}

// VehicleUE ::= ENUMERATED
const char* vehicle_ue_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "vehicle_ue_e");
}

// AerialUEsubscriptionInformation ::= ENUMERATED
const char* aerial_uesubscription_info_opts::to_string() const
{
  static const char* options[] = {"allowed", "not-allowed"};
  return convert_enum_idx(options, 2, value, "aerial_uesubscription_info_e");
}

// ManagementBasedMDTAllowed ::= ENUMERATED
const char* management_based_mdt_allowed_opts::to_string() const
{
  static const char* options[] = {"allowed"};
  return convert_enum_idx(options, 1, value, "management_based_mdt_allowed_e");
}

// ProSeAuthorized ::= SEQUENCE
SRSASN_CODE pro_se_authorized_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(pro_se_direct_discovery_present, 1));
  HANDLE_CODE(bref.pack(pro_se_direct_communication_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (pro_se_direct_discovery_present) {
    HANDLE_CODE(pro_se_direct_discovery.pack(bref));
  }
  if (pro_se_direct_communication_present) {
    HANDLE_CODE(pro_se_direct_communication.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pro_se_authorized_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(pro_se_direct_discovery_present, 1));
  HANDLE_CODE(bref.unpack(pro_se_direct_communication_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (pro_se_direct_discovery_present) {
    HANDLE_CODE(pro_se_direct_discovery.unpack(bref));
  }
  if (pro_se_direct_communication_present) {
    HANDLE_CODE(pro_se_direct_communication.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void pro_se_authorized_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (pro_se_direct_discovery_present) {
    j.write_str("proSeDirectDiscovery", pro_se_direct_discovery.to_string());
  }
  if (pro_se_direct_communication_present) {
    j.write_str("proSeDirectCommunication", pro_se_direct_communication.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// RequestType ::= SEQUENCE
SRSASN_CODE request_type_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(event_type.pack(bref));
  HANDLE_CODE(report_area.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE request_type_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(event_type.unpack(bref));
  HANDLE_CODE(report_area.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void request_type_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("eventType", event_type.to_string());
  j.write_str("reportArea", "ecgi");
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// SecurityContext ::= SEQUENCE
SRSASN_CODE security_context_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, next_hop_chaining_count, (uint8_t)0u, (uint8_t)7u, false, true));
  HANDLE_CODE(next_hop_param.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE security_context_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(next_hop_chaining_count, bref, (uint8_t)0u, (uint8_t)7u, false, true));
  HANDLE_CODE(next_hop_param.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void security_context_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("nextHopChainingCount", next_hop_chaining_count);
  j.write_str("nextHopParameter", next_hop_param.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::protocol_ext_field_s<trace_activation_ext_ies_o>;

trace_activation_ext_ies_container::trace_activation_ext_ies_container() :
  mdt_cfg(162, crit_e::ignore), ue_app_layer_meas_cfg(262, crit_e::ignore)
{}
SRSASN_CODE trace_activation_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += mdt_cfg_present ? 1 : 0;
  nof_ies += ue_app_layer_meas_cfg_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (mdt_cfg_present) {
    HANDLE_CODE(mdt_cfg.pack(bref));
  }
  if (ue_app_layer_meas_cfg_present) {
    HANDLE_CODE(ue_app_layer_meas_cfg.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE trace_activation_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 162: {
        mdt_cfg_present = true;
        mdt_cfg.id      = id;
        HANDLE_CODE(mdt_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mdt_cfg.ext.unpack(bref));
        break;
      }
      case 262: {
        ue_app_layer_meas_cfg_present = true;
        ue_app_layer_meas_cfg.id      = id;
        HANDLE_CODE(ue_app_layer_meas_cfg.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_app_layer_meas_cfg.ext.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void trace_activation_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (mdt_cfg_present) {
    j.write_fieldname("");
    mdt_cfg.to_json(j);
  }
  if (ue_app_layer_meas_cfg_present) {
    j.write_fieldname("");
    ue_app_layer_meas_cfg.to_json(j);
  }
  j.end_obj();
}

// TraceActivation ::= SEQUENCE
SRSASN_CODE trace_activation_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(e_utran_trace_id.pack(bref));
  HANDLE_CODE(interfaces_to_trace.pack(bref));
  HANDLE_CODE(trace_depth.pack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE trace_activation_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(e_utran_trace_id.unpack(bref));
  HANDLE_CODE(interfaces_to_trace.unpack(bref));
  HANDLE_CODE(trace_depth.unpack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void trace_activation_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("e-UTRAN-Trace-ID", e_utran_trace_id.to_string());
  j.write_str("interfacesToTrace", interfaces_to_trace.to_string());
  j.write_str("traceDepth", trace_depth.to_string());
  j.write_str("traceCollectionEntityIPAddress", trace_collection_entity_ip_address.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UESecurityCapabilities ::= SEQUENCE
SRSASN_CODE ue_security_cap_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(encryption_algorithms.pack(bref));
  HANDLE_CODE(integrity_protection_algorithms.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_security_cap_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(encryption_algorithms.unpack(bref));
  HANDLE_CODE(integrity_protection_algorithms.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_security_cap_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("encryptionAlgorithms", encryption_algorithms.to_string());
  j.write_str("integrityProtectionAlgorithms", integrity_protection_algorithms.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UESidelinkAggregateMaximumBitrate ::= SEQUENCE
SRSASN_CODE ue_sidelink_aggregate_maximum_bitrate_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(
      pack_integer(bref, uesidelink_aggregate_maximum_bit_rate, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_sidelink_aggregate_maximum_bitrate_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(
      unpack_integer(uesidelink_aggregate_maximum_bit_rate, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_sidelink_aggregate_maximum_bitrate_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("uESidelinkAggregateMaximumBitRate", uesidelink_aggregate_maximum_bit_rate);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UEUserPlaneCIoTSupportIndicator ::= ENUMERATED
const char* ueuser_plane_cio_tsupport_ind_opts::to_string() const
{
  static const char* options[] = {"supported"};
  return convert_enum_idx(options, 1, value, "ueuser_plane_cio_tsupport_ind_e");
}

// V2XServicesAuthorized ::= SEQUENCE
SRSASN_CODE v2xservices_authorized_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(vehicle_ue_present, 1));
  HANDLE_CODE(bref.pack(pedestrian_ue_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (vehicle_ue_present) {
    HANDLE_CODE(vehicle_ue.pack(bref));
  }
  if (pedestrian_ue_present) {
    HANDLE_CODE(pedestrian_ue.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE v2xservices_authorized_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(vehicle_ue_present, 1));
  HANDLE_CODE(bref.unpack(pedestrian_ue_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (vehicle_ue_present) {
    HANDLE_CODE(vehicle_ue.unpack(bref));
  }
  if (pedestrian_ue_present) {
    HANDLE_CODE(pedestrian_ue.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void v2xservices_authorized_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (vehicle_ue_present) {
    j.write_str("vehicleUE", vehicle_ue.to_string());
  }
  if (pedestrian_ue_present) {
    j.write_str("pedestrianUE", pedestrian_ue.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverRequestIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0,   1,   2,   66,  53,  104, 107, 41,  25,  98,  124, 40,  136, 127, 146, 75,
                                     158, 165, 177, 192, 196, 195, 241, 240, 248, 251, 269, 271, 277, 283, 278, 299};
  return map_enum_number(options, 32, idx, "id");
}
bool ho_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0,   1,   2,   66,  53,  104, 107, 41,  25,  98,  124, 40,  136, 127, 146, 75,
                                     158, 165, 177, 192, 196, 195, 241, 240, 248, 251, 269, 271, 277, 283, 278, 299};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 1:
      return crit_e::reject;
    case 2:
      return crit_e::ignore;
    case 66:
      return crit_e::reject;
    case 53:
      return crit_e::reject;
    case 104:
      return crit_e::reject;
    case 107:
      return crit_e::reject;
    case 41:
      return crit_e::ignore;
    case 25:
      return crit_e::ignore;
    case 98:
      return crit_e::ignore;
    case 124:
      return crit_e::ignore;
    case 40:
      return crit_e::reject;
    case 136:
      return crit_e::reject;
    case 127:
      return crit_e::reject;
    case 146:
      return crit_e::ignore;
    case 75:
      return crit_e::ignore;
    case 158:
      return crit_e::ignore;
    case 165:
      return crit_e::ignore;
    case 177:
      return crit_e::ignore;
    case 192:
      return crit_e::ignore;
    case 196:
      return crit_e::ignore;
    case 195:
      return crit_e::ignore;
    case 241:
      return crit_e::ignore;
    case 240:
      return crit_e::ignore;
    case 248:
      return crit_e::ignore;
    case 251:
      return crit_e::ignore;
    case 269:
      return crit_e::ignore;
    case 271:
      return crit_e::ignore;
    case 277:
      return crit_e::ignore;
    case 283:
      return crit_e::ignore;
    case 278:
      return crit_e::ignore;
    case 299:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_request_ies_o::value_c ho_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 1:
      ret.set(value_c::types::handov_type);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 53:
      ret.set(value_c::types::erab_to_be_setup_list_ho_req);
      break;
    case 104:
      ret.set(value_c::types::source_to_target_transparent_container);
      break;
    case 107:
      ret.set(value_c::types::ue_security_cap);
      break;
    case 41:
      ret.set(value_c::types::ho_restrict_list);
      break;
    case 25:
      ret.set(value_c::types::trace_activation);
      break;
    case 98:
      ret.set(value_c::types::request_type);
      break;
    case 124:
      ret.set(value_c::types::srvcc_operation_possible);
      break;
    case 40:
      ret.set(value_c::types::security_context);
      break;
    case 136:
      ret.set(value_c::types::nas_security_paramsto_e_utran);
      break;
    case 127:
      ret.set(value_c::types::csg_id);
      break;
    case 146:
      ret.set(value_c::types::csg_membership_status);
      break;
    case 75:
      ret.set(value_c::types::gummei_id);
      break;
    case 158:
      ret.set(value_c::types::mme_ue_s1ap_id_minus2);
      break;
    case 165:
      ret.set(value_c::types::management_based_mdt_allowed);
      break;
    case 177:
      ret.set(value_c::types::management_based_mdtplmn_list);
      break;
    case 192:
      ret.set(value_c::types::masked_imeisv);
      break;
    case 196:
      ret.set(value_c::types::expected_ue_behaviour);
      break;
    case 195:
      ret.set(value_c::types::pro_se_authorized);
      break;
    case 241:
      ret.set(value_c::types::ueuser_plane_cio_tsupport_ind);
      break;
    case 240:
      ret.set(value_c::types::v2xservices_authorized);
      break;
    case 248:
      ret.set(value_c::types::ue_sidelink_aggregate_maximum_bitrate);
      break;
    case 251:
      ret.set(value_c::types::enhanced_coverage_restricted);
      break;
    case 269:
      ret.set(value_c::types::nrue_security_cap);
      break;
    case 271:
      ret.set(value_c::types::ce_mode_brestricted);
      break;
    case 277:
      ret.set(value_c::types::aerial_uesubscription_info);
      break;
    case 283:
      ret.set(value_c::types::pending_data_ind);
      break;
    case 278:
      ret.set(value_c::types::subscription_based_ue_differentiation_info);
      break;
    case 299:
      ret.set(value_c::types::add_rrm_prio_idx);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 1:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 66:
      return presence_e::mandatory;
    case 53:
      return presence_e::mandatory;
    case 104:
      return presence_e::mandatory;
    case 107:
      return presence_e::mandatory;
    case 41:
      return presence_e::optional;
    case 25:
      return presence_e::optional;
    case 98:
      return presence_e::optional;
    case 124:
      return presence_e::optional;
    case 40:
      return presence_e::mandatory;
    case 136:
      return presence_e::conditional;
    case 127:
      return presence_e::optional;
    case 146:
      return presence_e::optional;
    case 75:
      return presence_e::optional;
    case 158:
      return presence_e::optional;
    case 165:
      return presence_e::optional;
    case 177:
      return presence_e::optional;
    case 192:
      return presence_e::optional;
    case 196:
      return presence_e::optional;
    case 195:
      return presence_e::optional;
    case 241:
      return presence_e::optional;
    case 240:
      return presence_e::optional;
    case 248:
      return presence_e::optional;
    case 251:
      return presence_e::optional;
    case 269:
      return presence_e::optional;
    case 271:
      return presence_e::optional;
    case 277:
      return presence_e::optional;
    case 283:
      return presence_e::optional;
    case 278:
      return presence_e::optional;
    case 299:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_ho_req:
      c.destroy<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >();
      break;
    case types::source_to_target_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ue_security_cap:
      c.destroy<ue_security_cap_s>();
      break;
    case types::ho_restrict_list:
      c.destroy<ho_restrict_list_s>();
      break;
    case types::trace_activation:
      c.destroy<trace_activation_s>();
      break;
    case types::request_type:
      c.destroy<request_type_s>();
      break;
    case types::security_context:
      c.destroy<security_context_s>();
      break;
    case types::nas_security_paramsto_e_utran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.destroy<fixed_bitstring<27, false, true> >();
      break;
    case types::gummei_id:
      c.destroy<gummei_s>();
      break;
    case types::management_based_mdtplmn_list:
      c.destroy<mdtplmn_list_l>();
      break;
    case types::masked_imeisv:
      c.destroy<fixed_bitstring<64, false, true> >();
      break;
    case types::expected_ue_behaviour:
      c.destroy<expected_ue_behaviour_s>();
      break;
    case types::pro_se_authorized:
      c.destroy<pro_se_authorized_s>();
      break;
    case types::v2xservices_authorized:
      c.destroy<v2xservices_authorized_s>();
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.destroy<ue_sidelink_aggregate_maximum_bitrate_s>();
      break;
    case types::nrue_security_cap:
      c.destroy<nrue_security_cap_s>();
      break;
    case types::subscription_based_ue_differentiation_info:
      c.destroy<subscription_based_ue_differentiation_info_s>();
      break;
    case types::add_rrm_prio_idx:
      c.destroy<fixed_bitstring<32, false, true> >();
      break;
    default:
      break;
  }
}
void ho_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::handov_type:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_ho_req:
      c.init<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >();
      break;
    case types::source_to_target_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ue_security_cap:
      c.init<ue_security_cap_s>();
      break;
    case types::ho_restrict_list:
      c.init<ho_restrict_list_s>();
      break;
    case types::trace_activation:
      c.init<trace_activation_s>();
      break;
    case types::request_type:
      c.init<request_type_s>();
      break;
    case types::srvcc_operation_possible:
      break;
    case types::security_context:
      c.init<security_context_s>();
      break;
    case types::nas_security_paramsto_e_utran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.init<fixed_bitstring<27, false, true> >();
      break;
    case types::csg_membership_status:
      break;
    case types::gummei_id:
      c.init<gummei_s>();
      break;
    case types::mme_ue_s1ap_id_minus2:
      break;
    case types::management_based_mdt_allowed:
      break;
    case types::management_based_mdtplmn_list:
      c.init<mdtplmn_list_l>();
      break;
    case types::masked_imeisv:
      c.init<fixed_bitstring<64, false, true> >();
      break;
    case types::expected_ue_behaviour:
      c.init<expected_ue_behaviour_s>();
      break;
    case types::pro_se_authorized:
      c.init<pro_se_authorized_s>();
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      break;
    case types::v2xservices_authorized:
      c.init<v2xservices_authorized_s>();
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.init<ue_sidelink_aggregate_maximum_bitrate_s>();
      break;
    case types::enhanced_coverage_restricted:
      break;
    case types::nrue_security_cap:
      c.init<nrue_security_cap_s>();
      break;
    case types::ce_mode_brestricted:
      break;
    case types::aerial_uesubscription_info:
      break;
    case types::pending_data_ind:
      break;
    case types::subscription_based_ue_differentiation_info:
      c.init<subscription_based_ue_differentiation_info_s>();
      break;
    case types::add_rrm_prio_idx:
      c.init<fixed_bitstring<32, false, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
}
ho_request_ies_o::value_c::value_c(const ho_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_ho_req:
      c.init(other.c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >());
      break;
    case types::source_to_target_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ue_security_cap:
      c.init(other.c.get<ue_security_cap_s>());
      break;
    case types::ho_restrict_list:
      c.init(other.c.get<ho_restrict_list_s>());
      break;
    case types::trace_activation:
      c.init(other.c.get<trace_activation_s>());
      break;
    case types::request_type:
      c.init(other.c.get<request_type_s>());
      break;
    case types::srvcc_operation_possible:
      c.init(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::security_context:
      c.init(other.c.get<security_context_s>());
      break;
    case types::nas_security_paramsto_e_utran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.init(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::csg_membership_status:
      c.init(other.c.get<csg_membership_status_e>());
      break;
    case types::gummei_id:
      c.init(other.c.get<gummei_s>());
      break;
    case types::mme_ue_s1ap_id_minus2:
      c.init(other.c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      c.init(other.c.get<management_based_mdt_allowed_e>());
      break;
    case types::management_based_mdtplmn_list:
      c.init(other.c.get<mdtplmn_list_l>());
      break;
    case types::masked_imeisv:
      c.init(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::expected_ue_behaviour:
      c.init(other.c.get<expected_ue_behaviour_s>());
      break;
    case types::pro_se_authorized:
      c.init(other.c.get<pro_se_authorized_s>());
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      c.init(other.c.get<ueuser_plane_cio_tsupport_ind_e>());
      break;
    case types::v2xservices_authorized:
      c.init(other.c.get<v2xservices_authorized_s>());
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.init(other.c.get<ue_sidelink_aggregate_maximum_bitrate_s>());
      break;
    case types::enhanced_coverage_restricted:
      c.init(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::nrue_security_cap:
      c.init(other.c.get<nrue_security_cap_s>());
      break;
    case types::ce_mode_brestricted:
      c.init(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::aerial_uesubscription_info:
      c.init(other.c.get<aerial_uesubscription_info_e>());
      break;
    case types::pending_data_ind:
      c.init(other.c.get<pending_data_ind_e>());
      break;
    case types::subscription_based_ue_differentiation_info:
      c.init(other.c.get<subscription_based_ue_differentiation_info_s>());
      break;
    case types::add_rrm_prio_idx:
      c.init(other.c.get<fixed_bitstring<32, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
}
ho_request_ies_o::value_c& ho_request_ies_o::value_c::operator=(const ho_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_ho_req:
      c.set(other.c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >());
      break;
    case types::source_to_target_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ue_security_cap:
      c.set(other.c.get<ue_security_cap_s>());
      break;
    case types::ho_restrict_list:
      c.set(other.c.get<ho_restrict_list_s>());
      break;
    case types::trace_activation:
      c.set(other.c.get<trace_activation_s>());
      break;
    case types::request_type:
      c.set(other.c.get<request_type_s>());
      break;
    case types::srvcc_operation_possible:
      c.set(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::security_context:
      c.set(other.c.get<security_context_s>());
      break;
    case types::nas_security_paramsto_e_utran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.set(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::csg_membership_status:
      c.set(other.c.get<csg_membership_status_e>());
      break;
    case types::gummei_id:
      c.set(other.c.get<gummei_s>());
      break;
    case types::mme_ue_s1ap_id_minus2:
      c.set(other.c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      c.set(other.c.get<management_based_mdt_allowed_e>());
      break;
    case types::management_based_mdtplmn_list:
      c.set(other.c.get<mdtplmn_list_l>());
      break;
    case types::masked_imeisv:
      c.set(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::expected_ue_behaviour:
      c.set(other.c.get<expected_ue_behaviour_s>());
      break;
    case types::pro_se_authorized:
      c.set(other.c.get<pro_se_authorized_s>());
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      c.set(other.c.get<ueuser_plane_cio_tsupport_ind_e>());
      break;
    case types::v2xservices_authorized:
      c.set(other.c.get<v2xservices_authorized_s>());
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.set(other.c.get<ue_sidelink_aggregate_maximum_bitrate_s>());
      break;
    case types::enhanced_coverage_restricted:
      c.set(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::nrue_security_cap:
      c.set(other.c.get<nrue_security_cap_s>());
      break;
    case types::ce_mode_brestricted:
      c.set(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::aerial_uesubscription_info:
      c.set(other.c.get<aerial_uesubscription_info_e>());
      break;
    case types::pending_data_ind:
      c.set(other.c.get<pending_data_ind_e>());
      break;
    case types::subscription_based_ue_differentiation_info:
      c.set(other.c.get<subscription_based_ue_differentiation_info_s>());
      break;
    case types::add_rrm_prio_idx:
      c.set(other.c.get<fixed_bitstring<32, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_request_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
handov_type_e& ho_request_ies_o::value_c::handov_type()
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
cause_c& ho_request_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
ue_aggregate_maximum_bitrate_s& ho_request_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type(types::ueaggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o>& ho_request_ies_o::value_c::erab_to_be_setup_list_ho_req()
{
  assert_choice_type(types::erab_to_be_setup_list_ho_req, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >();
}
unbounded_octstring<true>& ho_request_ies_o::value_c::source_to_target_transparent_container()
{
  assert_choice_type(types::source_to_target_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
ue_security_cap_s& ho_request_ies_o::value_c::ue_security_cap()
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
ho_restrict_list_s& ho_request_ies_o::value_c::ho_restrict_list()
{
  assert_choice_type(types::ho_restrict_list, type_, "Value");
  return c.get<ho_restrict_list_s>();
}
trace_activation_s& ho_request_ies_o::value_c::trace_activation()
{
  assert_choice_type(types::trace_activation, type_, "Value");
  return c.get<trace_activation_s>();
}
request_type_s& ho_request_ies_o::value_c::request_type()
{
  assert_choice_type(types::request_type, type_, "Value");
  return c.get<request_type_s>();
}
srvcc_operation_possible_e& ho_request_ies_o::value_c::srvcc_operation_possible()
{
  assert_choice_type(types::srvcc_operation_possible, type_, "Value");
  return c.get<srvcc_operation_possible_e>();
}
security_context_s& ho_request_ies_o::value_c::security_context()
{
  assert_choice_type(types::security_context, type_, "Value");
  return c.get<security_context_s>();
}
unbounded_octstring<true>& ho_request_ies_o::value_c::nas_security_paramsto_e_utran()
{
  assert_choice_type(types::nas_security_paramsto_e_utran, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
fixed_bitstring<27, false, true>& ho_request_ies_o::value_c::csg_id()
{
  assert_choice_type(types::csg_id, type_, "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
csg_membership_status_e& ho_request_ies_o::value_c::csg_membership_status()
{
  assert_choice_type(types::csg_membership_status, type_, "Value");
  return c.get<csg_membership_status_e>();
}
gummei_s& ho_request_ies_o::value_c::gummei_id()
{
  assert_choice_type(types::gummei_id, type_, "Value");
  return c.get<gummei_s>();
}
uint64_t& ho_request_ies_o::value_c::mme_ue_s1ap_id_minus2()
{
  assert_choice_type(types::mme_ue_s1ap_id_minus2, type_, "Value");
  return c.get<uint64_t>();
}
management_based_mdt_allowed_e& ho_request_ies_o::value_c::management_based_mdt_allowed()
{
  assert_choice_type(types::management_based_mdt_allowed, type_, "Value");
  return c.get<management_based_mdt_allowed_e>();
}
mdtplmn_list_l& ho_request_ies_o::value_c::management_based_mdtplmn_list()
{
  assert_choice_type(types::management_based_mdtplmn_list, type_, "Value");
  return c.get<mdtplmn_list_l>();
}
fixed_bitstring<64, false, true>& ho_request_ies_o::value_c::masked_imeisv()
{
  assert_choice_type(types::masked_imeisv, type_, "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
expected_ue_behaviour_s& ho_request_ies_o::value_c::expected_ue_behaviour()
{
  assert_choice_type(types::expected_ue_behaviour, type_, "Value");
  return c.get<expected_ue_behaviour_s>();
}
pro_se_authorized_s& ho_request_ies_o::value_c::pro_se_authorized()
{
  assert_choice_type(types::pro_se_authorized, type_, "Value");
  return c.get<pro_se_authorized_s>();
}
ueuser_plane_cio_tsupport_ind_e& ho_request_ies_o::value_c::ueuser_plane_cio_tsupport_ind()
{
  assert_choice_type(types::ueuser_plane_cio_tsupport_ind, type_, "Value");
  return c.get<ueuser_plane_cio_tsupport_ind_e>();
}
v2xservices_authorized_s& ho_request_ies_o::value_c::v2xservices_authorized()
{
  assert_choice_type(types::v2xservices_authorized, type_, "Value");
  return c.get<v2xservices_authorized_s>();
}
ue_sidelink_aggregate_maximum_bitrate_s& ho_request_ies_o::value_c::ue_sidelink_aggregate_maximum_bitrate()
{
  assert_choice_type(types::ue_sidelink_aggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_sidelink_aggregate_maximum_bitrate_s>();
}
enhanced_coverage_restricted_e& ho_request_ies_o::value_c::enhanced_coverage_restricted()
{
  assert_choice_type(types::enhanced_coverage_restricted, type_, "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
nrue_security_cap_s& ho_request_ies_o::value_c::nrue_security_cap()
{
  assert_choice_type(types::nrue_security_cap, type_, "Value");
  return c.get<nrue_security_cap_s>();
}
ce_mode_brestricted_e& ho_request_ies_o::value_c::ce_mode_brestricted()
{
  assert_choice_type(types::ce_mode_brestricted, type_, "Value");
  return c.get<ce_mode_brestricted_e>();
}
aerial_uesubscription_info_e& ho_request_ies_o::value_c::aerial_uesubscription_info()
{
  assert_choice_type(types::aerial_uesubscription_info, type_, "Value");
  return c.get<aerial_uesubscription_info_e>();
}
pending_data_ind_e& ho_request_ies_o::value_c::pending_data_ind()
{
  assert_choice_type(types::pending_data_ind, type_, "Value");
  return c.get<pending_data_ind_e>();
}
subscription_based_ue_differentiation_info_s& ho_request_ies_o::value_c::subscription_based_ue_differentiation_info()
{
  assert_choice_type(types::subscription_based_ue_differentiation_info, type_, "Value");
  return c.get<subscription_based_ue_differentiation_info_s>();
}
fixed_bitstring<32, false, true>& ho_request_ies_o::value_c::add_rrm_prio_idx()
{
  assert_choice_type(types::add_rrm_prio_idx, type_, "Value");
  return c.get<fixed_bitstring<32, false, true> >();
}
const uint64_t& ho_request_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const handov_type_e& ho_request_ies_o::value_c::handov_type() const
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
const cause_c& ho_request_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const ue_aggregate_maximum_bitrate_s& ho_request_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type(types::ueaggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o>&
ho_request_ies_o::value_c::erab_to_be_setup_list_ho_req() const
{
  assert_choice_type(types::erab_to_be_setup_list_ho_req, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >();
}
const unbounded_octstring<true>& ho_request_ies_o::value_c::source_to_target_transparent_container() const
{
  assert_choice_type(types::source_to_target_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const ue_security_cap_s& ho_request_ies_o::value_c::ue_security_cap() const
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
const ho_restrict_list_s& ho_request_ies_o::value_c::ho_restrict_list() const
{
  assert_choice_type(types::ho_restrict_list, type_, "Value");
  return c.get<ho_restrict_list_s>();
}
const trace_activation_s& ho_request_ies_o::value_c::trace_activation() const
{
  assert_choice_type(types::trace_activation, type_, "Value");
  return c.get<trace_activation_s>();
}
const request_type_s& ho_request_ies_o::value_c::request_type() const
{
  assert_choice_type(types::request_type, type_, "Value");
  return c.get<request_type_s>();
}
const srvcc_operation_possible_e& ho_request_ies_o::value_c::srvcc_operation_possible() const
{
  assert_choice_type(types::srvcc_operation_possible, type_, "Value");
  return c.get<srvcc_operation_possible_e>();
}
const security_context_s& ho_request_ies_o::value_c::security_context() const
{
  assert_choice_type(types::security_context, type_, "Value");
  return c.get<security_context_s>();
}
const unbounded_octstring<true>& ho_request_ies_o::value_c::nas_security_paramsto_e_utran() const
{
  assert_choice_type(types::nas_security_paramsto_e_utran, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const fixed_bitstring<27, false, true>& ho_request_ies_o::value_c::csg_id() const
{
  assert_choice_type(types::csg_id, type_, "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
const csg_membership_status_e& ho_request_ies_o::value_c::csg_membership_status() const
{
  assert_choice_type(types::csg_membership_status, type_, "Value");
  return c.get<csg_membership_status_e>();
}
const gummei_s& ho_request_ies_o::value_c::gummei_id() const
{
  assert_choice_type(types::gummei_id, type_, "Value");
  return c.get<gummei_s>();
}
const uint64_t& ho_request_ies_o::value_c::mme_ue_s1ap_id_minus2() const
{
  assert_choice_type(types::mme_ue_s1ap_id_minus2, type_, "Value");
  return c.get<uint64_t>();
}
const management_based_mdt_allowed_e& ho_request_ies_o::value_c::management_based_mdt_allowed() const
{
  assert_choice_type(types::management_based_mdt_allowed, type_, "Value");
  return c.get<management_based_mdt_allowed_e>();
}
const mdtplmn_list_l& ho_request_ies_o::value_c::management_based_mdtplmn_list() const
{
  assert_choice_type(types::management_based_mdtplmn_list, type_, "Value");
  return c.get<mdtplmn_list_l>();
}
const fixed_bitstring<64, false, true>& ho_request_ies_o::value_c::masked_imeisv() const
{
  assert_choice_type(types::masked_imeisv, type_, "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
const expected_ue_behaviour_s& ho_request_ies_o::value_c::expected_ue_behaviour() const
{
  assert_choice_type(types::expected_ue_behaviour, type_, "Value");
  return c.get<expected_ue_behaviour_s>();
}
const pro_se_authorized_s& ho_request_ies_o::value_c::pro_se_authorized() const
{
  assert_choice_type(types::pro_se_authorized, type_, "Value");
  return c.get<pro_se_authorized_s>();
}
const ueuser_plane_cio_tsupport_ind_e& ho_request_ies_o::value_c::ueuser_plane_cio_tsupport_ind() const
{
  assert_choice_type(types::ueuser_plane_cio_tsupport_ind, type_, "Value");
  return c.get<ueuser_plane_cio_tsupport_ind_e>();
}
const v2xservices_authorized_s& ho_request_ies_o::value_c::v2xservices_authorized() const
{
  assert_choice_type(types::v2xservices_authorized, type_, "Value");
  return c.get<v2xservices_authorized_s>();
}
const ue_sidelink_aggregate_maximum_bitrate_s& ho_request_ies_o::value_c::ue_sidelink_aggregate_maximum_bitrate() const
{
  assert_choice_type(types::ue_sidelink_aggregate_maximum_bitrate, type_, "Value");
  return c.get<ue_sidelink_aggregate_maximum_bitrate_s>();
}
const enhanced_coverage_restricted_e& ho_request_ies_o::value_c::enhanced_coverage_restricted() const
{
  assert_choice_type(types::enhanced_coverage_restricted, type_, "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
const nrue_security_cap_s& ho_request_ies_o::value_c::nrue_security_cap() const
{
  assert_choice_type(types::nrue_security_cap, type_, "Value");
  return c.get<nrue_security_cap_s>();
}
const ce_mode_brestricted_e& ho_request_ies_o::value_c::ce_mode_brestricted() const
{
  assert_choice_type(types::ce_mode_brestricted, type_, "Value");
  return c.get<ce_mode_brestricted_e>();
}
const aerial_uesubscription_info_e& ho_request_ies_o::value_c::aerial_uesubscription_info() const
{
  assert_choice_type(types::aerial_uesubscription_info, type_, "Value");
  return c.get<aerial_uesubscription_info_e>();
}
const pending_data_ind_e& ho_request_ies_o::value_c::pending_data_ind() const
{
  assert_choice_type(types::pending_data_ind, type_, "Value");
  return c.get<pending_data_ind_e>();
}
const subscription_based_ue_differentiation_info_s&
ho_request_ies_o::value_c::subscription_based_ue_differentiation_info() const
{
  assert_choice_type(types::subscription_based_ue_differentiation_info, type_, "Value");
  return c.get<subscription_based_ue_differentiation_info_s>();
}
const fixed_bitstring<32, false, true>& ho_request_ies_o::value_c::add_rrm_prio_idx() const
{
  assert_choice_type(types::add_rrm_prio_idx, type_, "Value");
  return c.get<fixed_bitstring<32, false, true> >();
}
void ho_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_setup_list_ho_req:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABToBeSetupItemHOReqIEs}}");
      break;
    case types::source_to_target_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ue_security_cap:
      j.write_fieldname("UESecurityCapabilities");
      c.get<ue_security_cap_s>().to_json(j);
      break;
    case types::ho_restrict_list:
      j.write_fieldname("HandoverRestrictionList");
      c.get<ho_restrict_list_s>().to_json(j);
      break;
    case types::trace_activation:
      j.write_fieldname("TraceActivation");
      c.get<trace_activation_s>().to_json(j);
      break;
    case types::request_type:
      j.write_fieldname("RequestType");
      c.get<request_type_s>().to_json(j);
      break;
    case types::srvcc_operation_possible:
      j.write_str("SRVCCOperationPossible", "possible");
      break;
    case types::security_context:
      j.write_fieldname("SecurityContext");
      c.get<security_context_s>().to_json(j);
      break;
    case types::nas_security_paramsto_e_utran:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::csg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<27, false, true> >().to_string());
      break;
    case types::csg_membership_status:
      j.write_str("CSGMembershipStatus", c.get<csg_membership_status_e>().to_string());
      break;
    case types::gummei_id:
      j.write_fieldname("GUMMEI");
      c.get<gummei_s>().to_json(j);
      break;
    case types::mme_ue_s1ap_id_minus2:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      j.write_str("ManagementBasedMDTAllowed", "allowed");
      break;
    case types::management_based_mdtplmn_list:
      j.start_array("MDTPLMNList");
      for (const auto& e1 : c.get<mdtplmn_list_l>()) {
        j.write_str(e1.to_string());
      }
      j.end_array();
      break;
    case types::masked_imeisv:
      j.write_str("BIT STRING", c.get<fixed_bitstring<64, false, true> >().to_string());
      break;
    case types::expected_ue_behaviour:
      j.write_fieldname("ExpectedUEBehaviour");
      c.get<expected_ue_behaviour_s>().to_json(j);
      break;
    case types::pro_se_authorized:
      j.write_fieldname("ProSeAuthorized");
      c.get<pro_se_authorized_s>().to_json(j);
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      j.write_str("UEUserPlaneCIoTSupportIndicator", "supported");
      break;
    case types::v2xservices_authorized:
      j.write_fieldname("V2XServicesAuthorized");
      c.get<v2xservices_authorized_s>().to_json(j);
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      j.write_fieldname("UESidelinkAggregateMaximumBitrate");
      c.get<ue_sidelink_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::enhanced_coverage_restricted:
      j.write_str("EnhancedCoverageRestricted", "restricted");
      break;
    case types::nrue_security_cap:
      j.write_fieldname("NRUESecurityCapabilities");
      c.get<nrue_security_cap_s>().to_json(j);
      break;
    case types::ce_mode_brestricted:
      j.write_str("CE-ModeBRestricted", c.get<ce_mode_brestricted_e>().to_string());
      break;
    case types::aerial_uesubscription_info:
      j.write_str("AerialUEsubscriptionInformation", c.get<aerial_uesubscription_info_e>().to_string());
      break;
    case types::pending_data_ind:
      j.write_str("PendingDataIndication", "true");
      break;
    case types::subscription_based_ue_differentiation_info:
      j.write_fieldname("Subscription-Based-UE-DifferentiationInfo");
      c.get<subscription_based_ue_differentiation_info_s>().to_json(j);
      break;
    case types::add_rrm_prio_idx:
      j.write_str("BIT STRING", c.get<fixed_bitstring<32, false, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_setup_list_ho_req:
      HANDLE_CODE(
          pack_dyn_seq_of(bref, c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >(), 1, 256, true));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().pack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().pack(bref));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().pack(bref));
      break;
    case types::request_type:
      HANDLE_CODE(c.get<request_type_s>().pack(bref));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().pack(bref));
      break;
    case types::security_context:
      HANDLE_CODE(c.get<security_context_s>().pack(bref));
      break;
    case types::nas_security_paramsto_e_utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().pack(bref)));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().pack(bref));
      break;
    case types::gummei_id:
      HANDLE_CODE(c.get<gummei_s>().pack(bref));
      break;
    case types::mme_ue_s1ap_id_minus2:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::management_based_mdt_allowed:
      HANDLE_CODE(c.get<management_based_mdt_allowed_e>().pack(bref));
      break;
    case types::management_based_mdtplmn_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<mdtplmn_list_l>(), 1, 16, true));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().pack(bref)));
      break;
    case types::expected_ue_behaviour:
      HANDLE_CODE(c.get<expected_ue_behaviour_s>().pack(bref));
      break;
    case types::pro_se_authorized:
      HANDLE_CODE(c.get<pro_se_authorized_s>().pack(bref));
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      HANDLE_CODE(c.get<ueuser_plane_cio_tsupport_ind_e>().pack(bref));
      break;
    case types::v2xservices_authorized:
      HANDLE_CODE(c.get<v2xservices_authorized_s>().pack(bref));
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_sidelink_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().pack(bref));
      break;
    case types::nrue_security_cap:
      HANDLE_CODE(c.get<nrue_security_cap_s>().pack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().pack(bref));
      break;
    case types::aerial_uesubscription_info:
      HANDLE_CODE(c.get<aerial_uesubscription_info_e>().pack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().pack(bref));
      break;
    case types::subscription_based_ue_differentiation_info:
      HANDLE_CODE(c.get<subscription_based_ue_differentiation_info_s>().pack(bref));
      break;
    case types::add_rrm_prio_idx:
      HANDLE_CODE((c.get<fixed_bitstring<32, false, true> >().pack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_setup_list_ho_req:
      HANDLE_CODE(unpack_dyn_seq_of(
          c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >(), bref, 1, 256, true));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().unpack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().unpack(bref));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().unpack(bref));
      break;
    case types::request_type:
      HANDLE_CODE(c.get<request_type_s>().unpack(bref));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().unpack(bref));
      break;
    case types::security_context:
      HANDLE_CODE(c.get<security_context_s>().unpack(bref));
      break;
    case types::nas_security_paramsto_e_utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().unpack(bref)));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().unpack(bref));
      break;
    case types::gummei_id:
      HANDLE_CODE(c.get<gummei_s>().unpack(bref));
      break;
    case types::mme_ue_s1ap_id_minus2:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::management_based_mdt_allowed:
      HANDLE_CODE(c.get<management_based_mdt_allowed_e>().unpack(bref));
      break;
    case types::management_based_mdtplmn_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<mdtplmn_list_l>(), bref, 1, 16, true));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().unpack(bref)));
      break;
    case types::expected_ue_behaviour:
      HANDLE_CODE(c.get<expected_ue_behaviour_s>().unpack(bref));
      break;
    case types::pro_se_authorized:
      HANDLE_CODE(c.get<pro_se_authorized_s>().unpack(bref));
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      HANDLE_CODE(c.get<ueuser_plane_cio_tsupport_ind_e>().unpack(bref));
      break;
    case types::v2xservices_authorized:
      HANDLE_CODE(c.get<v2xservices_authorized_s>().unpack(bref));
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_sidelink_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().unpack(bref));
      break;
    case types::nrue_security_cap:
      HANDLE_CODE(c.get<nrue_security_cap_s>().unpack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().unpack(bref));
      break;
    case types::aerial_uesubscription_info:
      HANDLE_CODE(c.get<aerial_uesubscription_info_e>().unpack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().unpack(bref));
      break;
    case types::subscription_based_ue_differentiation_info:
      HANDLE_CODE(c.get<subscription_based_ue_differentiation_info_s>().unpack(bref));
      break;
    case types::add_rrm_prio_idx:
      HANDLE_CODE((c.get<fixed_bitstring<32, false, true> >().unpack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "HandoverType",
                                  "Cause",
                                  "UEAggregateMaximumBitrate",
                                  "E-RAB-IE-ContainerList{{E-RABToBeSetupItemHOReqIEs}}",
                                  "OCTET STRING",
                                  "UESecurityCapabilities",
                                  "HandoverRestrictionList",
                                  "TraceActivation",
                                  "RequestType",
                                  "SRVCCOperationPossible",
                                  "SecurityContext",
                                  "OCTET STRING",
                                  "BIT STRING",
                                  "CSGMembershipStatus",
                                  "GUMMEI",
                                  "INTEGER (0..4294967295)",
                                  "ManagementBasedMDTAllowed",
                                  "MDTPLMNList",
                                  "BIT STRING",
                                  "ExpectedUEBehaviour",
                                  "ProSeAuthorized",
                                  "UEUserPlaneCIoTSupportIndicator",
                                  "V2XServicesAuthorized",
                                  "UESidelinkAggregateMaximumBitrate",
                                  "EnhancedCoverageRestricted",
                                  "NRUESecurityCapabilities",
                                  "CE-ModeBRestricted",
                                  "AerialUEsubscriptionInformation",
                                  "PendingDataIndication",
                                  "Subscription-Based-UE-DifferentiationInfo",
                                  "BIT STRING"};
  return convert_enum_idx(options, 32, value, "ho_request_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_request_ies_o>;

ho_request_ies_container::ho_request_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  handov_type(1, crit_e::reject),
  cause(2, crit_e::ignore),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_setup_list_ho_req(53, crit_e::reject),
  source_to_target_transparent_container(104, crit_e::reject),
  ue_security_cap(107, crit_e::reject),
  ho_restrict_list(41, crit_e::ignore),
  trace_activation(25, crit_e::ignore),
  request_type(98, crit_e::ignore),
  srvcc_operation_possible(124, crit_e::ignore),
  security_context(40, crit_e::reject),
  nas_security_paramsto_e_utran(136, crit_e::reject),
  csg_id(127, crit_e::reject),
  csg_membership_status(146, crit_e::ignore),
  gummei_id(75, crit_e::ignore),
  mme_ue_s1ap_id_minus2(158, crit_e::ignore),
  management_based_mdt_allowed(165, crit_e::ignore),
  management_based_mdtplmn_list(177, crit_e::ignore),
  masked_imeisv(192, crit_e::ignore),
  expected_ue_behaviour(196, crit_e::ignore),
  pro_se_authorized(195, crit_e::ignore),
  ueuser_plane_cio_tsupport_ind(241, crit_e::ignore),
  v2xservices_authorized(240, crit_e::ignore),
  ue_sidelink_aggregate_maximum_bitrate(248, crit_e::ignore),
  enhanced_coverage_restricted(251, crit_e::ignore),
  nrue_security_cap(269, crit_e::ignore),
  ce_mode_brestricted(271, crit_e::ignore),
  aerial_uesubscription_info(277, crit_e::ignore),
  pending_data_ind(283, crit_e::ignore),
  subscription_based_ue_differentiation_info(278, crit_e::ignore),
  add_rrm_prio_idx(299, crit_e::ignore)
{}
SRSASN_CODE ho_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 8;
  nof_ies += ho_restrict_list_present ? 1 : 0;
  nof_ies += trace_activation_present ? 1 : 0;
  nof_ies += request_type_present ? 1 : 0;
  nof_ies += srvcc_operation_possible_present ? 1 : 0;
  nof_ies += nas_security_paramsto_e_utran_present ? 1 : 0;
  nof_ies += csg_id_present ? 1 : 0;
  nof_ies += csg_membership_status_present ? 1 : 0;
  nof_ies += gummei_id_present ? 1 : 0;
  nof_ies += mme_ue_s1ap_id_minus2_present ? 1 : 0;
  nof_ies += management_based_mdt_allowed_present ? 1 : 0;
  nof_ies += management_based_mdtplmn_list_present ? 1 : 0;
  nof_ies += masked_imeisv_present ? 1 : 0;
  nof_ies += expected_ue_behaviour_present ? 1 : 0;
  nof_ies += pro_se_authorized_present ? 1 : 0;
  nof_ies += ueuser_plane_cio_tsupport_ind_present ? 1 : 0;
  nof_ies += v2xservices_authorized_present ? 1 : 0;
  nof_ies += ue_sidelink_aggregate_maximum_bitrate_present ? 1 : 0;
  nof_ies += enhanced_coverage_restricted_present ? 1 : 0;
  nof_ies += nrue_security_cap_present ? 1 : 0;
  nof_ies += ce_mode_brestricted_present ? 1 : 0;
  nof_ies += aerial_uesubscription_info_present ? 1 : 0;
  nof_ies += pending_data_ind_present ? 1 : 0;
  nof_ies += subscription_based_ue_differentiation_info_present ? 1 : 0;
  nof_ies += add_rrm_prio_idx_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  HANDLE_CODE(erab_to_be_setup_list_ho_req.pack(bref));
  HANDLE_CODE(source_to_target_transparent_container.pack(bref));
  HANDLE_CODE(ue_security_cap.pack(bref));
  if (ho_restrict_list_present) {
    HANDLE_CODE(ho_restrict_list.pack(bref));
  }
  if (trace_activation_present) {
    HANDLE_CODE(trace_activation.pack(bref));
  }
  if (request_type_present) {
    HANDLE_CODE(request_type.pack(bref));
  }
  if (srvcc_operation_possible_present) {
    HANDLE_CODE(srvcc_operation_possible.pack(bref));
  }
  HANDLE_CODE(security_context.pack(bref));
  if (nas_security_paramsto_e_utran_present) {
    HANDLE_CODE(nas_security_paramsto_e_utran.pack(bref));
  }
  if (csg_id_present) {
    HANDLE_CODE(csg_id.pack(bref));
  }
  if (csg_membership_status_present) {
    HANDLE_CODE(csg_membership_status.pack(bref));
  }
  if (gummei_id_present) {
    HANDLE_CODE(gummei_id.pack(bref));
  }
  if (mme_ue_s1ap_id_minus2_present) {
    HANDLE_CODE(mme_ue_s1ap_id_minus2.pack(bref));
  }
  if (management_based_mdt_allowed_present) {
    HANDLE_CODE(management_based_mdt_allowed.pack(bref));
  }
  if (management_based_mdtplmn_list_present) {
    HANDLE_CODE(management_based_mdtplmn_list.pack(bref));
  }
  if (masked_imeisv_present) {
    HANDLE_CODE(masked_imeisv.pack(bref));
  }
  if (expected_ue_behaviour_present) {
    HANDLE_CODE(expected_ue_behaviour.pack(bref));
  }
  if (pro_se_authorized_present) {
    HANDLE_CODE(pro_se_authorized.pack(bref));
  }
  if (ueuser_plane_cio_tsupport_ind_present) {
    HANDLE_CODE(ueuser_plane_cio_tsupport_ind.pack(bref));
  }
  if (v2xservices_authorized_present) {
    HANDLE_CODE(v2xservices_authorized.pack(bref));
  }
  if (ue_sidelink_aggregate_maximum_bitrate_present) {
    HANDLE_CODE(ue_sidelink_aggregate_maximum_bitrate.pack(bref));
  }
  if (enhanced_coverage_restricted_present) {
    HANDLE_CODE(enhanced_coverage_restricted.pack(bref));
  }
  if (nrue_security_cap_present) {
    HANDLE_CODE(nrue_security_cap.pack(bref));
  }
  if (ce_mode_brestricted_present) {
    HANDLE_CODE(ce_mode_brestricted.pack(bref));
  }
  if (aerial_uesubscription_info_present) {
    HANDLE_CODE(aerial_uesubscription_info.pack(bref));
  }
  if (pending_data_ind_present) {
    HANDLE_CODE(pending_data_ind.pack(bref));
  }
  if (subscription_based_ue_differentiation_info_present) {
    HANDLE_CODE(subscription_based_ue_differentiation_info.pack(bref));
  }
  if (add_rrm_prio_idx_present) {
    HANDLE_CODE(add_rrm_prio_idx.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 8;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 1: {
        nof_mandatory_ies--;
        handov_type.id = id;
        HANDLE_CODE(handov_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(handov_type.value.unpack(bref));
        break;
      }
      case 2: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 66: {
        nof_mandatory_ies--;
        ueaggregate_maximum_bitrate.id = id;
        HANDLE_CODE(ueaggregate_maximum_bitrate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ueaggregate_maximum_bitrate.value.unpack(bref));
        break;
      }
      case 53: {
        nof_mandatory_ies--;
        erab_to_be_setup_list_ho_req.id = id;
        HANDLE_CODE(erab_to_be_setup_list_ho_req.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_to_be_setup_list_ho_req.value.unpack(bref));
        break;
      }
      case 104: {
        nof_mandatory_ies--;
        source_to_target_transparent_container.id = id;
        HANDLE_CODE(source_to_target_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(source_to_target_transparent_container.value.unpack(bref));
        break;
      }
      case 107: {
        nof_mandatory_ies--;
        ue_security_cap.id = id;
        HANDLE_CODE(ue_security_cap.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_security_cap.value.unpack(bref));
        break;
      }
      case 41: {
        ho_restrict_list_present = true;
        ho_restrict_list.id      = id;
        HANDLE_CODE(ho_restrict_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ho_restrict_list.value.unpack(bref));
        break;
      }
      case 25: {
        trace_activation_present = true;
        trace_activation.id      = id;
        HANDLE_CODE(trace_activation.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(trace_activation.value.unpack(bref));
        break;
      }
      case 98: {
        request_type_present = true;
        request_type.id      = id;
        HANDLE_CODE(request_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(request_type.value.unpack(bref));
        break;
      }
      case 124: {
        srvcc_operation_possible_present = true;
        srvcc_operation_possible.id      = id;
        HANDLE_CODE(srvcc_operation_possible.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(srvcc_operation_possible.value.unpack(bref));
        break;
      }
      case 40: {
        nof_mandatory_ies--;
        security_context.id = id;
        HANDLE_CODE(security_context.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(security_context.value.unpack(bref));
        break;
      }
      case 136: {
        nas_security_paramsto_e_utran_present = true;
        nas_security_paramsto_e_utran.id      = id;
        HANDLE_CODE(nas_security_paramsto_e_utran.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nas_security_paramsto_e_utran.value.unpack(bref));
        break;
      }
      case 127: {
        csg_id_present = true;
        csg_id.id      = id;
        HANDLE_CODE(csg_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(csg_id.value.unpack(bref));
        break;
      }
      case 146: {
        csg_membership_status_present = true;
        csg_membership_status.id      = id;
        HANDLE_CODE(csg_membership_status.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(csg_membership_status.value.unpack(bref));
        break;
      }
      case 75: {
        gummei_id_present = true;
        gummei_id.id      = id;
        HANDLE_CODE(gummei_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(gummei_id.value.unpack(bref));
        break;
      }
      case 158: {
        mme_ue_s1ap_id_minus2_present = true;
        mme_ue_s1ap_id_minus2.id      = id;
        HANDLE_CODE(mme_ue_s1ap_id_minus2.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id_minus2.value.unpack(bref));
        break;
      }
      case 165: {
        management_based_mdt_allowed_present = true;
        management_based_mdt_allowed.id      = id;
        HANDLE_CODE(management_based_mdt_allowed.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(management_based_mdt_allowed.value.unpack(bref));
        break;
      }
      case 177: {
        management_based_mdtplmn_list_present = true;
        management_based_mdtplmn_list.id      = id;
        HANDLE_CODE(management_based_mdtplmn_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(management_based_mdtplmn_list.value.unpack(bref));
        break;
      }
      case 192: {
        masked_imeisv_present = true;
        masked_imeisv.id      = id;
        HANDLE_CODE(masked_imeisv.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(masked_imeisv.value.unpack(bref));
        break;
      }
      case 196: {
        expected_ue_behaviour_present = true;
        expected_ue_behaviour.id      = id;
        HANDLE_CODE(expected_ue_behaviour.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(expected_ue_behaviour.value.unpack(bref));
        break;
      }
      case 195: {
        pro_se_authorized_present = true;
        pro_se_authorized.id      = id;
        HANDLE_CODE(pro_se_authorized.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pro_se_authorized.value.unpack(bref));
        break;
      }
      case 241: {
        ueuser_plane_cio_tsupport_ind_present = true;
        ueuser_plane_cio_tsupport_ind.id      = id;
        HANDLE_CODE(ueuser_plane_cio_tsupport_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ueuser_plane_cio_tsupport_ind.value.unpack(bref));
        break;
      }
      case 240: {
        v2xservices_authorized_present = true;
        v2xservices_authorized.id      = id;
        HANDLE_CODE(v2xservices_authorized.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(v2xservices_authorized.value.unpack(bref));
        break;
      }
      case 248: {
        ue_sidelink_aggregate_maximum_bitrate_present = true;
        ue_sidelink_aggregate_maximum_bitrate.id      = id;
        HANDLE_CODE(ue_sidelink_aggregate_maximum_bitrate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_sidelink_aggregate_maximum_bitrate.value.unpack(bref));
        break;
      }
      case 251: {
        enhanced_coverage_restricted_present = true;
        enhanced_coverage_restricted.id      = id;
        HANDLE_CODE(enhanced_coverage_restricted.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enhanced_coverage_restricted.value.unpack(bref));
        break;
      }
      case 269: {
        nrue_security_cap_present = true;
        nrue_security_cap.id      = id;
        HANDLE_CODE(nrue_security_cap.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nrue_security_cap.value.unpack(bref));
        break;
      }
      case 271: {
        ce_mode_brestricted_present = true;
        ce_mode_brestricted.id      = id;
        HANDLE_CODE(ce_mode_brestricted.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ce_mode_brestricted.value.unpack(bref));
        break;
      }
      case 277: {
        aerial_uesubscription_info_present = true;
        aerial_uesubscription_info.id      = id;
        HANDLE_CODE(aerial_uesubscription_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(aerial_uesubscription_info.value.unpack(bref));
        break;
      }
      case 283: {
        pending_data_ind_present = true;
        pending_data_ind.id      = id;
        HANDLE_CODE(pending_data_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pending_data_ind.value.unpack(bref));
        break;
      }
      case 278: {
        subscription_based_ue_differentiation_info_present = true;
        subscription_based_ue_differentiation_info.id      = id;
        HANDLE_CODE(subscription_based_ue_differentiation_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(subscription_based_ue_differentiation_info.value.unpack(bref));
        break;
      }
      case 299: {
        add_rrm_prio_idx_present = true;
        add_rrm_prio_idx.id      = id;
        HANDLE_CODE(add_rrm_prio_idx.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(add_rrm_prio_idx.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.write_fieldname("");
  ueaggregate_maximum_bitrate.to_json(j);
  j.write_fieldname("");
  erab_to_be_setup_list_ho_req.to_json(j);
  j.write_fieldname("");
  source_to_target_transparent_container.to_json(j);
  j.write_fieldname("");
  ue_security_cap.to_json(j);
  if (ho_restrict_list_present) {
    j.write_fieldname("");
    ho_restrict_list.to_json(j);
  }
  if (trace_activation_present) {
    j.write_fieldname("");
    trace_activation.to_json(j);
  }
  if (request_type_present) {
    j.write_fieldname("");
    request_type.to_json(j);
  }
  if (srvcc_operation_possible_present) {
    j.write_fieldname("");
    srvcc_operation_possible.to_json(j);
  }
  j.write_fieldname("");
  security_context.to_json(j);
  if (nas_security_paramsto_e_utran_present) {
    j.write_fieldname("");
    nas_security_paramsto_e_utran.to_json(j);
  }
  if (csg_id_present) {
    j.write_fieldname("");
    csg_id.to_json(j);
  }
  if (csg_membership_status_present) {
    j.write_fieldname("");
    csg_membership_status.to_json(j);
  }
  if (gummei_id_present) {
    j.write_fieldname("");
    gummei_id.to_json(j);
  }
  if (mme_ue_s1ap_id_minus2_present) {
    j.write_fieldname("");
    mme_ue_s1ap_id_minus2.to_json(j);
  }
  if (management_based_mdt_allowed_present) {
    j.write_fieldname("");
    management_based_mdt_allowed.to_json(j);
  }
  if (management_based_mdtplmn_list_present) {
    j.write_fieldname("");
    management_based_mdtplmn_list.to_json(j);
  }
  if (masked_imeisv_present) {
    j.write_fieldname("");
    masked_imeisv.to_json(j);
  }
  if (expected_ue_behaviour_present) {
    j.write_fieldname("");
    expected_ue_behaviour.to_json(j);
  }
  if (pro_se_authorized_present) {
    j.write_fieldname("");
    pro_se_authorized.to_json(j);
  }
  if (ueuser_plane_cio_tsupport_ind_present) {
    j.write_fieldname("");
    ueuser_plane_cio_tsupport_ind.to_json(j);
  }
  if (v2xservices_authorized_present) {
    j.write_fieldname("");
    v2xservices_authorized.to_json(j);
  }
  if (ue_sidelink_aggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ue_sidelink_aggregate_maximum_bitrate.to_json(j);
  }
  if (enhanced_coverage_restricted_present) {
    j.write_fieldname("");
    enhanced_coverage_restricted.to_json(j);
  }
  if (nrue_security_cap_present) {
    j.write_fieldname("");
    nrue_security_cap.to_json(j);
  }
  if (ce_mode_brestricted_present) {
    j.write_fieldname("");
    ce_mode_brestricted.to_json(j);
  }
  if (aerial_uesubscription_info_present) {
    j.write_fieldname("");
    aerial_uesubscription_info.to_json(j);
  }
  if (pending_data_ind_present) {
    j.write_fieldname("");
    pending_data_ind.to_json(j);
  }
  if (subscription_based_ue_differentiation_info_present) {
    j.write_fieldname("");
    subscription_based_ue_differentiation_info.to_json(j);
  }
  if (add_rrm_prio_idx_present) {
    j.write_fieldname("");
    add_rrm_prio_idx.to_json(j);
  }
  j.end_obj();
}

// CE-mode-B-SupportIndicator ::= ENUMERATED
const char* ce_mode_b_support_ind_opts::to_string() const
{
  static const char* options[] = {"supported"};
  return convert_enum_idx(options, 1, value, "ce_mode_b_support_ind_e");
}

// HandoverRequestAcknowledgeIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_request_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 18, 19, 123, 127, 58, 145, 242};
  return map_enum_number(options, 9, idx, "id");
}
bool ho_request_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 18, 19, 123, 127, 58, 145, 242};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_request_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 18:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    case 123:
      return crit_e::reject;
    case 127:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 145:
      return crit_e::ignore;
    case 242:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_request_ack_ies_o::value_c ho_request_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 18:
      ret.set(value_c::types::erab_admitted_list);
      break;
    case 19:
      ret.set(value_c::types::erab_failed_to_setup_list_ho_req_ack);
      break;
    case 123:
      ret.set(value_c::types::target_to_source_transparent_container);
      break;
    case 127:
      ret.set(value_c::types::csg_id);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 145:
      ret.set(value_c::types::cell_access_mode);
      break;
    case 242:
      ret.set(value_c::types::ce_mode_b_support_ind);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_request_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 18:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    case 123:
      return presence_e::mandatory;
    case 127:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 145:
      return presence_e::optional;
    case 242:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_request_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_admitted_list:
      c.destroy<erab_ie_container_list_l<erab_admitted_item_ies_o> >();
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      c.destroy<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >();
      break;
    case types::target_to_source_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.destroy<fixed_bitstring<27, false, true> >();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_request_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_admitted_list:
      c.init<erab_ie_container_list_l<erab_admitted_item_ies_o> >();
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      c.init<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >();
      break;
    case types::target_to_source_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.init<fixed_bitstring<27, false, true> >();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::cell_access_mode:
      break;
    case types::ce_mode_b_support_ind:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
}
ho_request_ack_ies_o::value_c::value_c(const ho_request_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_admitted_list:
      c.init(other.c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >());
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      c.init(other.c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >());
      break;
    case types::target_to_source_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.init(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::cell_access_mode:
      c.init(other.c.get<cell_access_mode_e>());
      break;
    case types::ce_mode_b_support_ind:
      c.init(other.c.get<ce_mode_b_support_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
}
ho_request_ack_ies_o::value_c& ho_request_ack_ies_o::value_c::operator=(const ho_request_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_admitted_list:
      c.set(other.c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >());
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      c.set(other.c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >());
      break;
    case types::target_to_source_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.set(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::cell_access_mode:
      c.set(other.c.get<cell_access_mode_e>());
      break;
    case types::ce_mode_b_support_ind:
      c.set(other.c.get<ce_mode_b_support_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_request_ack_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_request_ack_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
erab_ie_container_list_l<erab_admitted_item_ies_o>& ho_request_ack_ies_o::value_c::erab_admitted_list()
{
  assert_choice_type(types::erab_admitted_list, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >();
}
erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o>&
ho_request_ack_ies_o::value_c::erab_failed_to_setup_list_ho_req_ack()
{
  assert_choice_type(types::erab_failed_to_setup_list_ho_req_ack, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >();
}
unbounded_octstring<true>& ho_request_ack_ies_o::value_c::target_to_source_transparent_container()
{
  assert_choice_type(types::target_to_source_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
fixed_bitstring<27, false, true>& ho_request_ack_ies_o::value_c::csg_id()
{
  assert_choice_type(types::csg_id, type_, "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
crit_diagnostics_s& ho_request_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
cell_access_mode_e& ho_request_ack_ies_o::value_c::cell_access_mode()
{
  assert_choice_type(types::cell_access_mode, type_, "Value");
  return c.get<cell_access_mode_e>();
}
ce_mode_b_support_ind_e& ho_request_ack_ies_o::value_c::ce_mode_b_support_ind()
{
  assert_choice_type(types::ce_mode_b_support_ind, type_, "Value");
  return c.get<ce_mode_b_support_ind_e>();
}
const uint64_t& ho_request_ack_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_request_ack_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const erab_ie_container_list_l<erab_admitted_item_ies_o>& ho_request_ack_ies_o::value_c::erab_admitted_list() const
{
  assert_choice_type(types::erab_admitted_list, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >();
}
const erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o>&
ho_request_ack_ies_o::value_c::erab_failed_to_setup_list_ho_req_ack() const
{
  assert_choice_type(types::erab_failed_to_setup_list_ho_req_ack, type_, "Value");
  return c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >();
}
const unbounded_octstring<true>& ho_request_ack_ies_o::value_c::target_to_source_transparent_container() const
{
  assert_choice_type(types::target_to_source_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const fixed_bitstring<27, false, true>& ho_request_ack_ies_o::value_c::csg_id() const
{
  assert_choice_type(types::csg_id, type_, "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
const crit_diagnostics_s& ho_request_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const cell_access_mode_e& ho_request_ack_ies_o::value_c::cell_access_mode() const
{
  assert_choice_type(types::cell_access_mode, type_, "Value");
  return c.get<cell_access_mode_e>();
}
const ce_mode_b_support_ind_e& ho_request_ack_ies_o::value_c::ce_mode_b_support_ind() const
{
  assert_choice_type(types::ce_mode_b_support_ind, type_, "Value");
  return c.get<ce_mode_b_support_ind_e>();
}
void ho_request_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_admitted_list:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABAdmittedItemIEs}}");
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABFailedtoSetupItemHOReqAckIEs}}");
      break;
    case types::target_to_source_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::csg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<27, false, true> >().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::cell_access_mode:
      j.write_str("CellAccessMode", "hybrid");
      break;
    case types::ce_mode_b_support_ind:
      j.write_str("CE-mode-B-SupportIndicator", "supported");
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_request_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_admitted_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >(), 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      HANDLE_CODE(pack_dyn_seq_of(
          bref, c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >(), 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().pack(bref)));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().pack(bref));
      break;
    case types::ce_mode_b_support_ind:
      HANDLE_CODE(c.get<ce_mode_b_support_ind_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_admitted_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      HANDLE_CODE(unpack_dyn_seq_of(
          c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >(), bref, 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().unpack(bref)));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().unpack(bref));
      break;
    case types::ce_mode_b_support_ind:
      HANDLE_CODE(c.get<ce_mode_b_support_ind_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_request_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RAB-IE-ContainerList{{E-RABAdmittedItemIEs}}",
                                  "E-RAB-IE-ContainerList{{E-RABFailedtoSetupItemHOReqAckIEs}}",
                                  "OCTET STRING",
                                  "BIT STRING",
                                  "CriticalityDiagnostics",
                                  "CellAccessMode",
                                  "CE-mode-B-SupportIndicator"};
  return convert_enum_idx(options, 9, value, "ho_request_ack_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_request_ack_ies_o>;

ho_request_ack_ies_container::ho_request_ack_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_admitted_list(18, crit_e::ignore),
  erab_failed_to_setup_list_ho_req_ack(19, crit_e::ignore),
  target_to_source_transparent_container(123, crit_e::reject),
  csg_id(127, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore),
  cell_access_mode(145, crit_e::ignore),
  ce_mode_b_support_ind(242, crit_e::ignore)
{}
SRSASN_CODE ho_request_ack_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += erab_failed_to_setup_list_ho_req_ack_present ? 1 : 0;
  nof_ies += csg_id_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  nof_ies += cell_access_mode_present ? 1 : 0;
  nof_ies += ce_mode_b_support_ind_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(erab_admitted_list.pack(bref));
  if (erab_failed_to_setup_list_ho_req_ack_present) {
    HANDLE_CODE(erab_failed_to_setup_list_ho_req_ack.pack(bref));
  }
  HANDLE_CODE(target_to_source_transparent_container.pack(bref));
  if (csg_id_present) {
    HANDLE_CODE(csg_id.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.pack(bref));
  }
  if (ce_mode_b_support_ind_present) {
    HANDLE_CODE(ce_mode_b_support_ind.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 18: {
        nof_mandatory_ies--;
        erab_admitted_list.id = id;
        HANDLE_CODE(erab_admitted_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_admitted_list.value.unpack(bref));
        break;
      }
      case 19: {
        erab_failed_to_setup_list_ho_req_ack_present = true;
        erab_failed_to_setup_list_ho_req_ack.id      = id;
        HANDLE_CODE(erab_failed_to_setup_list_ho_req_ack.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(erab_failed_to_setup_list_ho_req_ack.value.unpack(bref));
        break;
      }
      case 123: {
        nof_mandatory_ies--;
        target_to_source_transparent_container.id = id;
        HANDLE_CODE(target_to_source_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(target_to_source_transparent_container.value.unpack(bref));
        break;
      }
      case 127: {
        csg_id_present = true;
        csg_id.id      = id;
        HANDLE_CODE(csg_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(csg_id.value.unpack(bref));
        break;
      }
      case 58: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      case 145: {
        cell_access_mode_present = true;
        cell_access_mode.id      = id;
        HANDLE_CODE(cell_access_mode.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cell_access_mode.value.unpack(bref));
        break;
      }
      case 242: {
        ce_mode_b_support_ind_present = true;
        ce_mode_b_support_ind.id      = id;
        HANDLE_CODE(ce_mode_b_support_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ce_mode_b_support_ind.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_request_ack_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  erab_admitted_list.to_json(j);
  if (erab_failed_to_setup_list_ho_req_ack_present) {
    j.write_fieldname("");
    erab_failed_to_setup_list_ho_req_ack.to_json(j);
  }
  j.write_fieldname("");
  target_to_source_transparent_container.to_json(j);
  if (csg_id_present) {
    j.write_fieldname("");
    csg_id.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  if (cell_access_mode_present) {
    j.write_fieldname("");
    cell_access_mode.to_json(j);
  }
  if (ce_mode_b_support_ind_present) {
    j.write_fieldname("");
    ce_mode_b_support_ind.to_json(j);
  }
  j.end_obj();
}

// TargetNgRanNode-ID ::= SEQUENCE
SRSASN_CODE target_ng_ran_node_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_ran_node_id.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE target_ng_ran_node_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_ran_node_id.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void target_ng_ran_node_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-RAN-NODE-ID");
  global_ran_node_id.to_json(j);
  j.write_fieldname("selected-TAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Direct-Forwarding-Path-Availability ::= ENUMERATED
const char* direct_forwarding_path_availability_opts::to_string() const
{
  static const char* options[] = {"directPathAvailable"};
  return convert_enum_idx(options, 1, value, "direct_forwarding_path_availability_e");
}

// PS-ServiceNotAvailable ::= ENUMERATED
const char* ps_service_not_available_opts::to_string() const
{
  static const char* options[] = {"ps-service-not-available"};
  return convert_enum_idx(options, 1, value, "ps_service_not_available_e");
}

// SRVCCHOIndication ::= ENUMERATED
const char* srvccho_ind_opts::to_string() const
{
  static const char* options[] = {"pSandCS", "cSonly"};
  return convert_enum_idx(options, 2, value, "srvccho_ind_e");
}

// TargetID ::= CHOICE
void target_id_c::destroy_()
{
  switch (type_) {
    case types::targetenb_id:
      c.destroy<targetenb_id_s>();
      break;
    case types::target_rnc_id:
      c.destroy<target_rnc_id_s>();
      break;
    case types::cgi:
      c.destroy<cgi_s>();
      break;
    case types::targetg_ng_ran_node_id:
      c.destroy<target_ng_ran_node_id_s>();
      break;
    default:
      break;
  }
}
void target_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::targetenb_id:
      c.init<targetenb_id_s>();
      break;
    case types::target_rnc_id:
      c.init<target_rnc_id_s>();
      break;
    case types::cgi:
      c.init<cgi_s>();
      break;
    case types::targetg_ng_ran_node_id:
      c.init<target_ng_ran_node_id_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
}
target_id_c::target_id_c(const target_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::targetenb_id:
      c.init(other.c.get<targetenb_id_s>());
      break;
    case types::target_rnc_id:
      c.init(other.c.get<target_rnc_id_s>());
      break;
    case types::cgi:
      c.init(other.c.get<cgi_s>());
      break;
    case types::targetg_ng_ran_node_id:
      c.init(other.c.get<target_ng_ran_node_id_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
}
target_id_c& target_id_c::operator=(const target_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::targetenb_id:
      c.set(other.c.get<targetenb_id_s>());
      break;
    case types::target_rnc_id:
      c.set(other.c.get<target_rnc_id_s>());
      break;
    case types::cgi:
      c.set(other.c.get<cgi_s>());
      break;
    case types::targetg_ng_ran_node_id:
      c.set(other.c.get<target_ng_ran_node_id_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }

  return *this;
}
targetenb_id_s& target_id_c::set_targetenb_id()
{
  set(types::targetenb_id);
  return c.get<targetenb_id_s>();
}
target_rnc_id_s& target_id_c::set_target_rnc_id()
{
  set(types::target_rnc_id);
  return c.get<target_rnc_id_s>();
}
cgi_s& target_id_c::set_cgi()
{
  set(types::cgi);
  return c.get<cgi_s>();
}
target_ng_ran_node_id_s& target_id_c::set_targetg_ng_ran_node_id()
{
  set(types::targetg_ng_ran_node_id);
  return c.get<target_ng_ran_node_id_s>();
}
void target_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::targetenb_id:
      j.write_fieldname("targeteNB-ID");
      c.get<targetenb_id_s>().to_json(j);
      break;
    case types::target_rnc_id:
      j.write_fieldname("targetRNC-ID");
      c.get<target_rnc_id_s>().to_json(j);
      break;
    case types::cgi:
      j.write_fieldname("cGI");
      c.get<cgi_s>().to_json(j);
      break;
    case types::targetg_ng_ran_node_id:
      j.write_fieldname("targetgNgRanNode-ID");
      c.get<target_ng_ran_node_id_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
  j.end_obj();
}
SRSASN_CODE target_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::targetenb_id:
      HANDLE_CODE(c.get<targetenb_id_s>().pack(bref));
      break;
    case types::target_rnc_id:
      HANDLE_CODE(c.get<target_rnc_id_s>().pack(bref));
      break;
    case types::cgi:
      HANDLE_CODE(c.get<cgi_s>().pack(bref));
      break;
    case types::targetg_ng_ran_node_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<target_ng_ran_node_id_s>().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE target_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::targetenb_id:
      HANDLE_CODE(c.get<targetenb_id_s>().unpack(bref));
      break;
    case types::target_rnc_id:
      HANDLE_CODE(c.get<target_rnc_id_s>().unpack(bref));
      break;
    case types::cgi:
      HANDLE_CODE(c.get<cgi_s>().unpack(bref));
      break;
    case types::targetg_ng_ran_node_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<target_ng_ran_node_id_s>().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* target_id_c::types_opts::to_string() const
{
  static const char* options[] = {"targeteNB-ID", "targetRNC-ID", "cGI", "targetgNgRanNode-ID"};
  return convert_enum_idx(options, 4, value, "target_id_c::types");
}

// HandoverRequiredIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_required_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 1, 2, 4, 79, 125, 104, 138, 132, 133, 127, 145, 150};
  return map_enum_number(options, 14, idx, "id");
}
bool ho_required_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 1, 2, 4, 79, 125, 104, 138, 132, 133, 127, 145, 150};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_required_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 1:
      return crit_e::reject;
    case 2:
      return crit_e::ignore;
    case 4:
      return crit_e::reject;
    case 79:
      return crit_e::ignore;
    case 125:
      return crit_e::reject;
    case 104:
      return crit_e::reject;
    case 138:
      return crit_e::reject;
    case 132:
      return crit_e::reject;
    case 133:
      return crit_e::ignore;
    case 127:
      return crit_e::reject;
    case 145:
      return crit_e::reject;
    case 150:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_required_ies_o::value_c ho_required_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 1:
      ret.set(value_c::types::handov_type);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 4:
      ret.set(value_c::types::target_id);
      break;
    case 79:
      ret.set(value_c::types::direct_forwarding_path_availability);
      break;
    case 125:
      ret.set(value_c::types::srvccho_ind);
      break;
    case 104:
      ret.set(value_c::types::source_to_target_transparent_container);
      break;
    case 138:
      ret.set(value_c::types::source_to_target_transparent_container_secondary);
      break;
    case 132:
      ret.set(value_c::types::ms_classmark2);
      break;
    case 133:
      ret.set(value_c::types::ms_classmark3);
      break;
    case 127:
      ret.set(value_c::types::csg_id);
      break;
    case 145:
      ret.set(value_c::types::cell_access_mode);
      break;
    case 150:
      ret.set(value_c::types::ps_service_not_available);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_required_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 1:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 4:
      return presence_e::mandatory;
    case 79:
      return presence_e::optional;
    case 125:
      return presence_e::optional;
    case 104:
      return presence_e::mandatory;
    case 138:
      return presence_e::optional;
    case 132:
      return presence_e::conditional;
    case 133:
      return presence_e::conditional;
    case 127:
      return presence_e::optional;
    case 145:
      return presence_e::optional;
    case 150:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_required_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::target_id:
      c.destroy<target_id_c>();
      break;
    case types::source_to_target_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::source_to_target_transparent_container_secondary:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ms_classmark2:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ms_classmark3:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.destroy<fixed_bitstring<27, false, true> >();
      break;
    default:
      break;
  }
}
void ho_required_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::handov_type:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::target_id:
      c.init<target_id_c>();
      break;
    case types::direct_forwarding_path_availability:
      break;
    case types::srvccho_ind:
      break;
    case types::source_to_target_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::source_to_target_transparent_container_secondary:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ms_classmark2:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ms_classmark3:
      c.init<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.init<fixed_bitstring<27, false, true> >();
      break;
    case types::cell_access_mode:
      break;
    case types::ps_service_not_available:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
}
ho_required_ies_o::value_c::value_c(const ho_required_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::target_id:
      c.init(other.c.get<target_id_c>());
      break;
    case types::direct_forwarding_path_availability:
      c.init(other.c.get<direct_forwarding_path_availability_e>());
      break;
    case types::srvccho_ind:
      c.init(other.c.get<srvccho_ind_e>());
      break;
    case types::source_to_target_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::source_to_target_transparent_container_secondary:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ms_classmark2:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ms_classmark3:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.init(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::cell_access_mode:
      c.init(other.c.get<cell_access_mode_e>());
      break;
    case types::ps_service_not_available:
      c.init(other.c.get<ps_service_not_available_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
}
ho_required_ies_o::value_c& ho_required_ies_o::value_c::operator=(const ho_required_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::target_id:
      c.set(other.c.get<target_id_c>());
      break;
    case types::direct_forwarding_path_availability:
      c.set(other.c.get<direct_forwarding_path_availability_e>());
      break;
    case types::srvccho_ind:
      c.set(other.c.get<srvccho_ind_e>());
      break;
    case types::source_to_target_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::source_to_target_transparent_container_secondary:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ms_classmark2:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ms_classmark3:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.set(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::cell_access_mode:
      c.set(other.c.get<cell_access_mode_e>());
      break;
    case types::ps_service_not_available:
      c.set(other.c.get<ps_service_not_available_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_required_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_required_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
handov_type_e& ho_required_ies_o::value_c::handov_type()
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
cause_c& ho_required_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
target_id_c& ho_required_ies_o::value_c::target_id()
{
  assert_choice_type(types::target_id, type_, "Value");
  return c.get<target_id_c>();
}
direct_forwarding_path_availability_e& ho_required_ies_o::value_c::direct_forwarding_path_availability()
{
  assert_choice_type(types::direct_forwarding_path_availability, type_, "Value");
  return c.get<direct_forwarding_path_availability_e>();
}
srvccho_ind_e& ho_required_ies_o::value_c::srvccho_ind()
{
  assert_choice_type(types::srvccho_ind, type_, "Value");
  return c.get<srvccho_ind_e>();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container()
{
  assert_choice_type(types::source_to_target_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container_secondary()
{
  assert_choice_type(types::source_to_target_transparent_container_secondary, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::ms_classmark2()
{
  assert_choice_type(types::ms_classmark2, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::ms_classmark3()
{
  assert_choice_type(types::ms_classmark3, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
fixed_bitstring<27, false, true>& ho_required_ies_o::value_c::csg_id()
{
  assert_choice_type(types::csg_id, type_, "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
cell_access_mode_e& ho_required_ies_o::value_c::cell_access_mode()
{
  assert_choice_type(types::cell_access_mode, type_, "Value");
  return c.get<cell_access_mode_e>();
}
ps_service_not_available_e& ho_required_ies_o::value_c::ps_service_not_available()
{
  assert_choice_type(types::ps_service_not_available, type_, "Value");
  return c.get<ps_service_not_available_e>();
}
const uint64_t& ho_required_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type(types::mme_ue_s1ap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_required_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type(types::enb_ue_s1ap_id, type_, "Value");
  return c.get<uint32_t>();
}
const handov_type_e& ho_required_ies_o::value_c::handov_type() const
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
const cause_c& ho_required_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const target_id_c& ho_required_ies_o::value_c::target_id() const
{
  assert_choice_type(types::target_id, type_, "Value");
  return c.get<target_id_c>();
}
const direct_forwarding_path_availability_e& ho_required_ies_o::value_c::direct_forwarding_path_availability() const
{
  assert_choice_type(types::direct_forwarding_path_availability, type_, "Value");
  return c.get<direct_forwarding_path_availability_e>();
}
const srvccho_ind_e& ho_required_ies_o::value_c::srvccho_ind() const
{
  assert_choice_type(types::srvccho_ind, type_, "Value");
  return c.get<srvccho_ind_e>();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container() const
{
  assert_choice_type(types::source_to_target_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container_secondary() const
{
  assert_choice_type(types::source_to_target_transparent_container_secondary, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::ms_classmark2() const
{
  assert_choice_type(types::ms_classmark2, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::ms_classmark3() const
{
  assert_choice_type(types::ms_classmark3, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const fixed_bitstring<27, false, true>& ho_required_ies_o::value_c::csg_id() const
{
  assert_choice_type(types::csg_id, type_, "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
const cell_access_mode_e& ho_required_ies_o::value_c::cell_access_mode() const
{
  assert_choice_type(types::cell_access_mode, type_, "Value");
  return c.get<cell_access_mode_e>();
}
const ps_service_not_available_e& ho_required_ies_o::value_c::ps_service_not_available() const
{
  assert_choice_type(types::ps_service_not_available, type_, "Value");
  return c.get<ps_service_not_available_e>();
}
void ho_required_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::target_id:
      j.write_fieldname("TargetID");
      c.get<target_id_c>().to_json(j);
      break;
    case types::direct_forwarding_path_availability:
      j.write_str("Direct-Forwarding-Path-Availability", "directPathAvailable");
      break;
    case types::srvccho_ind:
      j.write_str("SRVCCHOIndication", c.get<srvccho_ind_e>().to_string());
      break;
    case types::source_to_target_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::source_to_target_transparent_container_secondary:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ms_classmark2:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ms_classmark3:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::csg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<27, false, true> >().to_string());
      break;
    case types::cell_access_mode:
      j.write_str("CellAccessMode", "hybrid");
      break;
    case types::ps_service_not_available:
      j.write_str("PS-ServiceNotAvailable", "ps-service-not-available");
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_required_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::target_id:
      HANDLE_CODE(c.get<target_id_c>().pack(bref));
      break;
    case types::direct_forwarding_path_availability:
      HANDLE_CODE(c.get<direct_forwarding_path_availability_e>().pack(bref));
      break;
    case types::srvccho_ind:
      HANDLE_CODE(c.get<srvccho_ind_e>().pack(bref));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::source_to_target_transparent_container_secondary:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ms_classmark2:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ms_classmark3:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().pack(bref)));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().pack(bref));
      break;
    case types::ps_service_not_available:
      HANDLE_CODE(c.get<ps_service_not_available_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_required_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::target_id:
      HANDLE_CODE(c.get<target_id_c>().unpack(bref));
      break;
    case types::direct_forwarding_path_availability:
      HANDLE_CODE(c.get<direct_forwarding_path_availability_e>().unpack(bref));
      break;
    case types::srvccho_ind:
      HANDLE_CODE(c.get<srvccho_ind_e>().unpack(bref));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::source_to_target_transparent_container_secondary:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ms_classmark2:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ms_classmark3:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().unpack(bref)));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().unpack(bref));
      break;
    case types::ps_service_not_available:
      HANDLE_CODE(c.get<ps_service_not_available_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_required_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "HandoverType",
                                  "Cause",
                                  "TargetID",
                                  "Direct-Forwarding-Path-Availability",
                                  "SRVCCHOIndication",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "BIT STRING",
                                  "CellAccessMode",
                                  "PS-ServiceNotAvailable"};
  return convert_enum_idx(options, 14, value, "ho_required_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_required_ies_o>;

ho_required_ies_container::ho_required_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  handov_type(1, crit_e::reject),
  cause(2, crit_e::ignore),
  target_id(4, crit_e::reject),
  direct_forwarding_path_availability(79, crit_e::ignore),
  srvccho_ind(125, crit_e::reject),
  source_to_target_transparent_container(104, crit_e::reject),
  source_to_target_transparent_container_secondary(138, crit_e::reject),
  ms_classmark2(132, crit_e::reject),
  ms_classmark3(133, crit_e::ignore),
  csg_id(127, crit_e::reject),
  cell_access_mode(145, crit_e::reject),
  ps_service_not_available(150, crit_e::ignore)
{}
SRSASN_CODE ho_required_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 6;
  nof_ies += direct_forwarding_path_availability_present ? 1 : 0;
  nof_ies += srvccho_ind_present ? 1 : 0;
  nof_ies += source_to_target_transparent_container_secondary_present ? 1 : 0;
  nof_ies += ms_classmark2_present ? 1 : 0;
  nof_ies += ms_classmark3_present ? 1 : 0;
  nof_ies += csg_id_present ? 1 : 0;
  nof_ies += cell_access_mode_present ? 1 : 0;
  nof_ies += ps_service_not_available_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  HANDLE_CODE(target_id.pack(bref));
  if (direct_forwarding_path_availability_present) {
    HANDLE_CODE(direct_forwarding_path_availability.pack(bref));
  }
  if (srvccho_ind_present) {
    HANDLE_CODE(srvccho_ind.pack(bref));
  }
  HANDLE_CODE(source_to_target_transparent_container.pack(bref));
  if (source_to_target_transparent_container_secondary_present) {
    HANDLE_CODE(source_to_target_transparent_container_secondary.pack(bref));
  }
  if (ms_classmark2_present) {
    HANDLE_CODE(ms_classmark2.pack(bref));
  }
  if (ms_classmark3_present) {
    HANDLE_CODE(ms_classmark3.pack(bref));
  }
  if (csg_id_present) {
    HANDLE_CODE(csg_id.pack(bref));
  }
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.pack(bref));
  }
  if (ps_service_not_available_present) {
    HANDLE_CODE(ps_service_not_available.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_required_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 6;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 0: {
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id = id;
        HANDLE_CODE(mme_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mme_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 8: {
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id = id;
        HANDLE_CODE(enb_ue_s1ap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(enb_ue_s1ap_id.value.unpack(bref));
        break;
      }
      case 1: {
        nof_mandatory_ies--;
        handov_type.id = id;
        HANDLE_CODE(handov_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(handov_type.value.unpack(bref));
        break;
      }
      case 2: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 4: {
        nof_mandatory_ies--;
        target_id.id = id;
        HANDLE_CODE(target_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(target_id.value.unpack(bref));
        break;
      }
      case 79: {
        direct_forwarding_path_availability_present = true;
        direct_forwarding_path_availability.id      = id;
        HANDLE_CODE(direct_forwarding_path_availability.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(direct_forwarding_path_availability.value.unpack(bref));
        break;
      }
      case 125: {
        srvccho_ind_present = true;
        srvccho_ind.id      = id;
        HANDLE_CODE(srvccho_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(srvccho_ind.value.unpack(bref));
        break;
      }
      case 104: {
        nof_mandatory_ies--;
        source_to_target_transparent_container.id = id;
        HANDLE_CODE(source_to_target_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(source_to_target_transparent_container.value.unpack(bref));
        break;
      }
      case 138: {
        source_to_target_transparent_container_secondary_present = true;
        source_to_target_transparent_container_secondary.id      = id;
        HANDLE_CODE(source_to_target_transparent_container_secondary.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(source_to_target_transparent_container_secondary.value.unpack(bref));
        break;
      }
      case 132: {
        ms_classmark2_present = true;
        ms_classmark2.id      = id;
        HANDLE_CODE(ms_classmark2.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ms_classmark2.value.unpack(bref));
        break;
      }
      case 133: {
        ms_classmark3_present = true;
        ms_classmark3.id      = id;
        HANDLE_CODE(ms_classmark3.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ms_classmark3.value.unpack(bref));
        break;
      }
      case 127: {
        csg_id_present = true;
        csg_id.id      = id;
        HANDLE_CODE(csg_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(csg_id.value.unpack(bref));
        break;
      }
      case 145: {
        cell_access_mode_present = true;
        cell_access_mode.id      = id;
        HANDLE_CODE(cell_access_mode.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cell_access_mode.value.unpack(bref));
        break;
      }
      case 150: {
        ps_service_not_available_present = true;
        ps_service_not_available.id      = id;
        HANDLE_CODE(ps_service_not_available.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ps_service_not_available.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_required_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.write_fieldname("");
  target_id.to_json(j);
  if (direct_forwarding_path_availability_present) {
    j.write_fieldname("");
    direct_forwarding_path_availability.to_json(j);
  }
  if (srvccho_ind_present) {
    j.write_fieldname("");
    srvccho_ind.to_json(j);
  }
  j.write_fieldname("");
  source_to_target_transparent_container.to_json(j);
  if (source_to_target_transparent_container_secondary_present) {
    j.write_fieldname("");
    source_to_target_transparent_container_secondary.to_json(j);
  }
  if (ms_classmark2_present) {
    j.write_fieldname("");
    ms_classmark2.to_json(j);
  }
  if (ms_classmark3_present) {
    j.write_fieldname("");
    ms_classmark3.to_json(j);
  }
  if (csg_id_present) {
    j.write_fieldname("");
    csg_id.to_json(j);
  }
  if (cell_access_mode_present) {
    j.write_fieldname("");
   