const char csv_lua[] =
"-- csv.lua (internal file)\n"
"\n"
"local ffi = require('ffi')\n"
"local log = require('log')\n"
"\n"
"ffi.cdef[[\n"
"    typedef void (*csv_emit_row_t)(void *ctx);\n"
"    typedef void (*csv_emit_field_t)(void *ctx, const char *field, const char *end);\n"
"\n"
"    struct csv\n"
"    {\n"
"        void *emit_ctx;\n"
"        csv_emit_row_t emit_row;\n"
"        csv_emit_field_t emit_field;\n"
"        char delimiter;\n"
"        char quote_char;\n"
"\n"
"        char prev_symbol;\n"
"        int error_status;\n"
"        int ending_spaces;\n"
"\n"
"        void *(*realloc)(void*, size_t);\n"
"\n"
"        int state;\n"
"        char *buf;\n"
"        char *bufp;\n"
"        size_t buf_len;\n"
"    };\n"
"\n"
"    void csv_create(struct csv *csv);\n"
"    void csv_destroy(struct csv *csv);\n"
"    void csv_setopt(struct csv *csv, int opt, ...);\n"
"\n"
"    struct csv_iterator {\n"
"        struct csv *csv;\n"
"        const char *buf_begin;\n"
"        const char *buf_end;\n"
"        const char *field;\n"
"        size_t field_len;\n"
"    };\n"
"    void csv_iterator_create(struct csv_iterator *it, struct csv *csv);\n"
"    int csv_next(struct csv_iterator *);\n"
"    void csv_feed(struct csv_iterator *, const char *, size_t);\n"
"    size_t csv_escape_field(struct csv *csv, const char *field, size_t field_len, char *dst, size_t buf_size);\n"
"    enum {\n"
"        CSV_IT_OK,\n"
"        CSV_IT_EOL,\n"
"        CSV_IT_NEEDMORE,\n"
"        CSV_IT_EOF,\n"
"        CSV_IT_ERROR\n"
"    };\n"
"]]\n"
"\n"
"local iter = function(csvstate, i)\n"
"    local readable = csvstate[1]\n"
"    local csv_chunk_size = csvstate[2]\n"
"    local csv = csvstate[3]\n"
"    local it = csvstate[4]\n"
"    local tup = {}\n"
"    local st = ffi.C.csv_next(it)\n"
"    while st ~= ffi.C.CSV_IT_EOF do\n"
"        if st == ffi.C.CSV_IT_NEEDMORE then\n"
"            if readable then\n"
"                local buf = readable:read(csv_chunk_size)\n"
"                ffi.C.csv_feed(it, buf, string.len(buf))\n"
"                -- extend buf lifetime - csv_feed saves pointers\n"
"                csvstate[5] = buf\n"
"            else\n"
"                ffi.C.csv_feed(it, \"\", 0)\n"
"            end\n"
"        elseif st == ffi.C.CSV_IT_EOL then\n"
"            i = i + 1\n"
"            if i > 0 then\n"
"                return i, tup\n"
"            end\n"
"        elseif st == ffi.C.CSV_IT_OK then\n"
"            if i >= 0 then\n"
"                tup[#tup + 1] = ffi.string(it.field, it.field_len)\n"
"            end\n"
"        elseif st == ffi.C.CSV_IT_ERROR then\n"
"            log.warn(\"CSV file has errors\")\n"
"            break\n"
"        elseif st == ffi.C.CSV_IT_EOF then\n"
"            ffi.C.csv_destroy(csv)\n"
"            break\n"
"        end\n"
"        st = ffi.C.csv_next(it)\n"
"    end\n"
"end\n"
"\n"
"local module = {}\n"
"\n"
"--@brief parse csv string by string\n"
"--@param readable must be string or object with method read(num) returns string\n"
"--@param opts.chunk_size (default 4096). Parser will read by chunk_size symbols\n"
"--@param opts.delimiter (default ',').\n"
"--@param opts.quote_char (default '\"').\n"
"--@param opts.skip_head_lines (default 0). Skip header.\n"
"--@return iter function, iterator state\n"
"module.iterate = function(readable, opts)\n"
"    opts = opts or {}\n"
"    if type(readable) ~= \"string\" and type(readable.read) ~= \"function\" then\n"
"        error(\"Usage: load(string or object with method read(num)\" ..\n"
"              \"returns string)\")\n"
"    end\n"
"    if not opts.chunk_size then\n"
"        opts.chunk_size = 4096\n"
"    end\n"
"    if not opts.delimiter then\n"
"        opts.delimiter = ','\n"
"    end\n"
"    if not opts.quote_char then\n"
"        opts.quote_char = '\"'\n"
"    end\n"
"    if not opts.skip_head_lines then\n"
"        opts.skip_head_lines = 0\n"
"    end\n"
"    local str\n"
"    if type(readable) == \"string\" then\n"
"        str = readable\n"
"        readable = nil\n"
"    else\n"
"        str = readable:read(opts.chunk_size)\n"
"    end\n"
"\n"
"    if not str then --read not works\n"
"        error(\"Usage: load(string or object with method read(num)\" ..\n"
"              \"returns string)\")\n"
"    end\n"
"    local it = ffi.new('struct csv_iterator')\n"
"    local csv = ffi.new('struct csv')\n"
"    ffi.C.csv_create(csv)\n"
"    ffi.gc(csv, ffi.C.csv_destroy)\n"
"\n"
"    csv.delimiter = string.byte(opts.delimiter)\n"
"    csv.quote_char = string.byte(opts.quote_char)\n"
"    ffi.C.csv_iterator_create(it, csv)\n"
"    ffi.C.csv_feed(it, str, string.len(str))\n"
"\n"
"    -- csv_feed remembers the pointer;\n"
"    -- str included in csv state to make sure it lives long enough\n"
"    return iter, {readable, opts.chunk_size, csv, it, str}, -opts.skip_head_lines\n"
"end\n"
"\n"
"--@brief parse csv and make table\n"
"--@return table\n"
"module.load = function(readable, opts)\n"
"    opts = opts or {}\n"
"    local result = {}\n"
"    for i, tup in module.iterate(readable, opts) do\n"
"        result[i] = tup\n"
"    end\n"
"    return result\n"
"end\n"
"\n"
"--@brief dumps tuple or table as csv\n"
"--@param t is tuple or table\n"
"--@param writable must be object with method write(string) like file or socket\n"
"--@param opts.delimiter (default ',').\n"
"--@param opts.quote_char (default '\"').\n"
"--@return there is no writable it returns csv as string\n"
"module.dump = function(t, opts, writable)\n"
"    opts = opts or {}\n"
"    writable = writable or nil\n"
"    if not opts.delimiter then\n"
"        opts.delimiter = ','\n"
"    end\n"
"    if not opts.quote_char then\n"
"        opts.quote_char = '\"'\n"
"    end\n"
"\n"
"    if (type(writable) ~= \"nil\" and type(writable.write) ~= \"function\")\n"
"        or type(t) ~= \"table\" then\n"
"        error(\"Usage: dump(table[, opts, writable])\")\n"
"    end\n"
"    local csv = ffi.new('struct csv')\n"
"    ffi.C.csv_create(csv)\n"
"    ffi.gc(csv, ffi.C.csv_destroy)\n"
"    csv.delimiter = string.byte(opts.delimiter)\n"
"    csv.quote_char = string.byte(opts.quote_char)\n"
"\n"
"    local bufsz = 256\n"
"    local buf = csv.realloc(ffi.cast(ffi.typeof('void *'), 0), bufsz)\n"
"    if type(t[1]) ~= 'table' then\n"
"        t = {t}\n"
"    end\n"
"    local result_table\n"
"    if type(writable) == 'nil' then\n"
"        result_table = {}\n"
"    end\n"
"    for _, line in pairs(t) do\n"
"        local first = true\n"
"        local output_tuple = {}\n"
"        for _, field in pairs(line) do\n"
"            local strf = tostring(field)\n"
"            local buf_new_size = (strf:len() + 1) * 2\n"
"            if buf_new_size > bufsz then\n"
"                bufsz = buf_new_size\n"
"                buf = csv.realloc(buf, bufsz)\n"
"            end\n"
"            local len = ffi.C.csv_escape_field(csv, strf,\n"
"                string.len(strf), buf, bufsz)\n"
"            if first then\n"
"                first = false\n"
"            else\n"
"                output_tuple[#output_tuple + 1] = opts.delimiter\n"
"            end\n"
"            output_tuple[#output_tuple + 1] = ffi.string(buf, len)\n"
"        end\n"
"\n"
"        output_tuple[#output_tuple + 1] = '\\n'\n"
"        if result_table then\n"
"            result_table[#result_table + 1] = table.concat(output_tuple)\n"
"        else\n"
"            writable:write(table.concat(output_tuple))\n"
"        end\n"
"    end\n"
"    ffi.C.csv_destroy(csv)\n"
"    csv.realloc(buf, 0)\n"
"    if result_table then\n"
"        return table.concat(result_table)\n"
"    end\n"
"end\n"
"\n"
"return module\n"
""
;
